#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/data/device_tags.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/actions/ut/library/helper.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/telematics/client/library/client.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(DynamicalPriceRiding) {

    Y_UNIT_TEST(FreePrice) {
        InitGlobalLog2Console(TLOG_INFO);
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::CarMarkers);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TInstant now = Now();
        TInstantGuard ig(now - TDuration::Hours(12));

        TEnvironmentGenerator::TCar newCar = eGenerator.CreateCar();

        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(newCar.IMEI);
        NDrive::TTelematicsTestClient::TPtr client = emulator->GetClient();
        UNIT_ASSERT(configGenerator.WaitCar(newCar.Id));
        TString offerId;
        {
            THolder<TStandartOfferReport> offer(new TStandartOfferReport(BuildOfferPtr(200, 100, 102400), nullptr));
            offer->GetOfferPtrAs<IOffer>()->SetObjectId(newCar.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(now + TDuration::Minutes(5));
            offer->GetOffer()->SetChargableAccounts({ "card", "bonus" });
            offerId = offer->GetOffer()->GetOfferId();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({offer.Release()}));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT, TDuration::Seconds(5)));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT, TDuration::Seconds(5)));

            ig.Set(now - TDuration::Hours(5));
            UNIT_ASSERT(configGenerator.WaitPrice(7 * 60 * 200));
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("37.58369774280378 55.7355775356348 37.58367628513167 55.73420944806805 37.586530155523526 55.734173126502995 37.58655161319563 55.73569860301168", coords));

            UNIT_ASSERT(configGenerator.UpsertArea("free", USER_ROOT_DEFAULT, coords, {"no_price"}));

            const TGeoCoord freeCoord(37.58519977985214, 55.73509326235609);
            driver.RelocateCar(newCar.IMEI, freeCoord);
            UNIT_ASSERT(configGenerator.WaitLocation(newCar.Id, freeCoord));
            {
                const TInstant start = Now();
                while (true) {
                    auto builder = driveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
                    auto session = builder->GetLastObjectSession(newCar.Id);
                    if (MakeSet((*session->GetLastEvent())->GetObjectSnapshotAs<THistoryDeviceSnapshot>()->GetLocationTagsArray()).contains("no_price")) {
                        break;
                    }
                    UNIT_ASSERT(Now() - start < TDuration::Minutes(2));
                }
            }

            ig.Set(now - TDuration::Hours(4));
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().Update(Now()));
            UNIT_ASSERT(configGenerator.WaitPrice(7 * 60 * 200));

            const TGeoCoord paymentCoord(37.59519977985214, 55.73509326235609);
            driver.RelocateCar(newCar.IMEI, paymentCoord);
            UNIT_ASSERT(configGenerator.WaitLocation(newCar.Id, paymentCoord));
            {
                const TInstant start = Now();
                while (true) {
                    auto builder = driveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
                    auto session = builder->GetLastObjectSession(newCar.Id);
                    if (!MakeSet((*session->GetLastEvent())->GetObjectSnapshotAs<THistoryDeviceSnapshot>()->GetLocationTagsArray()).contains("no_price")) {
                        break;
                    }
                    UNIT_ASSERT(Now() - start < TDuration::Minutes(2));
                }
            }

            ig.Set(now - TDuration::Hours(1));
            UNIT_ASSERT(configGenerator.WaitPrice(10 * 60 * 200));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT, TDuration::Seconds(5)));
        }

    }


    Y_UNIT_TEST(FreePriceRemoveArea) {
        InitGlobalLog2Console(TLOG_INFO);
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::CarMarkers);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TInstant now = Now();
        TInstantGuard ig(now - TDuration::Hours(12));

        TEnvironmentGenerator::TCar newCar = eGenerator.CreateCar();

        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(newCar.IMEI);
        NDrive::TTelematicsTestClient::TPtr client = emulator->GetClient();
        UNIT_ASSERT(configGenerator.WaitCar(newCar.Id));
        TString offerId;
        {
            THolder<TStandartOfferReport> offer(new TStandartOfferReport(BuildOfferPtr(200, 100, 102400), nullptr));
            offer->GetOfferPtrAs<IOffer>()->SetObjectId(newCar.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(now + TDuration::Minutes(5));
            offer->GetOffer()->SetChargableAccounts({ "card", "bonus" });
            offerId = offer->GetOffer()->GetOfferId();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({offer.Release()}));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT, TDuration::Seconds(5)));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT, TDuration::Seconds(5)));

            ig.Set(now - TDuration::Hours(5));
            UNIT_ASSERT(configGenerator.WaitPrice(7 * 60 * 200));
            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("37.58369774280378 55.7355775356348 37.58367628513167 55.73420944806805 37.586530155523526 55.734173126502995 37.58655161319563 55.73569860301168", coords));

            UNIT_ASSERT(configGenerator.UpsertArea("free", USER_ROOT_DEFAULT, coords, {"no_price"}));

            const TGeoCoord freeCoord(37.58519977985214, 55.73509326235609);
            driver.RelocateCar(newCar.IMEI, freeCoord);
            UNIT_ASSERT(configGenerator.WaitLocation(newCar.Id, freeCoord));

            {
                const TInstant start = Now();
                while (true) {
                    auto builder = driveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
                    auto session = builder->GetLastObjectSession(newCar.Id);
                    if (MakeSet((*session->GetLastEvent())->GetObjectSnapshotAs<THistoryDeviceSnapshot>()->GetLocationTagsArray()).contains("no_price")) {
                        break;
                    }
                    UNIT_ASSERT(Now() - start < TDuration::Minutes(2));
                }
            }

            ig.Set(now - TDuration::Hours(4));

            UNIT_ASSERT(configGenerator.WaitPrice(7 * 60 * 200));

            UNIT_ASSERT(configGenerator.RemoveArea("free", USER_ROOT_DEFAULT));
            {
                const TInstant start = Now();
                while (true) {
                    auto builder = driveApi.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
                    auto session = builder->GetLastObjectSession(newCar.Id);
                    if (!MakeSet((*session->GetLastEvent())->GetObjectSnapshotAs<THistoryDeviceSnapshot>()->GetLocationTagsArray()).contains("no_price")) {
                        break;
                    }
                    UNIT_ASSERT(Now() - start < TDuration::Minutes(2));
                }
            }

            ig.Set(now - TDuration::Hours(1));
            UNIT_ASSERT(configGenerator.WaitPrice(10 * 60 * 200));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT, TDuration::Seconds(5)));
        }

    }

}
