#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/cars/car_model.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(EntitySuite) {

    Y_UNIT_TEST(CarModels) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        {
            NJson::TJsonValue models = gServer.GetCarModels({}, USER_ID_DEFAULT);
            INFO_LOG << models << Endl;
            UNIT_ASSERT_VALUES_EQUAL(models["car_models"].GetArray().size(), 0);
        }
        ui32 startModelsCount = 0;
        {
            NJson::TJsonValue models = gServer.GetCarModels({}, USER_ROOT_DEFAULT);
            INFO_LOG << models << Endl;
            startModelsCount = models["car_models"].GetArraySafe().size();
            UNIT_ASSERT(startModelsCount);
        }

        UNIT_ASSERT(gServer.ModifyCarModel("zhiguli", "Zhiguli", "AvtoVAZ", USER_ROOT_DEFAULT));
        {
            NJson::TJsonValue models = gServer.GetCarModels({}, USER_ROOT_DEFAULT);
            INFO_LOG << models << Endl;
            UNIT_ASSERT_VALUES_EQUAL(models["car_models"].GetArraySafe().size(), startModelsCount + 1);
        }
        TString specId;
        {
            NJson::TJsonValue id = gServer.AddCarModelSpecification("zhiguli", "acceleration", "12,2", 0, USER_ROOT_DEFAULT);
            INFO_LOG << id << Endl;
            UNIT_ASSERT(id["id"].GetString(&specId) && !!specId);
        }
        {
            NJson::TJsonValue id = gServer.AddCarModelSpecification("zhiguli", "acceleration", "12,3", 0, USER_ROOT_DEFAULT, specId);
            INFO_LOG << id << Endl;
            TString anotherSpecId;
            UNIT_ASSERT(id["id"].GetString(&anotherSpecId));
            UNIT_ASSERT(anotherSpecId);
            UNIT_ASSERT_VALUES_EQUAL(specId, anotherSpecId);
        }

        /*
            Now, even though there is a spec, related to 'zhiguli', nothing prevents us to remove this model
            along with all its' added specs.
        */
        UNIT_ASSERT(gServer.RemoveCarModels({ "zhiguli" }, USER_ROOT_DEFAULT));
        {
            NJson::TJsonValue models = gServer.GetCarModels({}, USER_ROOT_DEFAULT);
            INFO_LOG << models << Endl;
            UNIT_ASSERT_VALUES_EQUAL(models["car_models"].GetArraySafe().size(), startModelsCount);
        }

        {
            NJson::TJsonValue models = gServer.GetCarModels({ "zhiguli" }, USER_ROOT_DEFAULT);
            INFO_LOG << models << Endl;
            UNIT_ASSERT_VALUES_EQUAL(models["car_models"].GetArray().size(), 0);
        }

        UNIT_ASSERT(gServer.ModifyCarModel("zhiguli", "Zhiguli", "AvtoVAZ", USER_ROOT_DEFAULT));
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            eGenerator.CreateCar(session, "zhiguli");
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(!gServer.RemoveCarModels({ "zhiguli" }, USER_ROOT_DEFAULT));
        {
            NJson::TJsonValue models = gServer.GetCarModels({}, USER_ROOT_DEFAULT);
            INFO_LOG << models << Endl;
            UNIT_ASSERT_VALUES_EQUAL(models["car_models"].GetArraySafe().size(), startModelsCount + 1);
        }

        // Modify default tags for the model incorrectly, ensure that there's a 400
        {

            NJson::TJsonValue malformedTagDescription;
            malformedTagDescription["tag_name"] = "new_car";
            malformedTagDescription["comment"] = "new car uploaded through admin portal";

            NJson::TJsonValue malformedTagsDescription = NJson::JSON_ARRAY;
            malformedTagsDescription.AppendValue(malformedTagDescription);

            NJson::TJsonValue payload;
            payload["default_tags"] = malformedTagsDescription;

            UNIT_ASSERT(!gServer.ModifyCarModel("zhiguli", "Zhiguli", "AvtoVAZ", USER_ROOT_DEFAULT, payload));

            // Check that it wasn't reflected in the base
            auto model = driveApi.GetModelsData()->FetchInfo("zhiguli").GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(model.GetDefaultTags().size(), 0);
        }

        // Modify default tags correctly and ensure it went well
        {

            NJson::TJsonValue tagDescription;
            tagDescription["tag_name"] = "new_car";
            tagDescription["comment"] = "new car uploaded through admin portal";
            tagDescription["priority"] = 1000;

            NJson::TJsonValue tagsDescription = NJson::JSON_ARRAY;
            tagsDescription.AppendValue(tagDescription);

            NJson::TJsonValue payload;
            payload["default_tags"] = tagsDescription;

            UNIT_ASSERT(gServer.ModifyCarModel("zhiguli", "Zhiguli", "AvtoVAZ", USER_ROOT_DEFAULT, payload));

            // Check that it WAS reflected in the base
            auto model = driveApi.GetModelsData()->FetchInfo("zhiguli").GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(model.GetDefaultTags().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(model.GetDefaultTags()[0].TagName, "new_car");
            UNIT_ASSERT_VALUES_EQUAL(model.GetDefaultTags()[0].Priority, 1000);
            UNIT_ASSERT_VALUES_EQUAL(model.GetDefaultTags()[0].Comment, "new car uploaded through admin portal");
        }

        // Check that a model will parse its' properties correctly
        {
            TDriveModelData kaptur = driveApi.GetModelsData()->FetchInfo("renault_kaptur").GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(kaptur.GetSpecifications().GetSeatsNumber(), 5);
            UNIT_ASSERT_VALUES_EQUAL(kaptur.GetSpecifications().GetPower(), 114);
            UNIT_ASSERT_C(fabs(kaptur.GetSpecifications().GetAccelerationTime() - 13.8) < 1e-5, TStringBuilder() << fabs(kaptur.GetSpecifications().GetAccelerationTime() - 13.8));
            UNIT_ASSERT_VALUES_EQUAL(kaptur.GetSpecifications().GetFuelTankVolume(), 60);
            UNIT_ASSERT_VALUES_EQUAL(kaptur.GetSpecifications().GetTransmission(), NDriveModelSpecification::ETransmissionType::Variable);
            UNIT_ASSERT_VALUES_EQUAL(kaptur.GetSpecifications().GetDriveUnit(), NDriveModelSpecification::EDriveUnitType::Front);
        }
    }

    Y_UNIT_TEST(Settings) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const ISettings* settings = &server->GetSettings();
        UNIT_ASSERT(settings);
        UNIT_ASSERT(server->GetConfig().GetSettingsConfig()->GetPrefix());
        TString key = "never_forget";
        TString value1 = "your_towel";
        TString value2 = "my_towel";
        UNIT_ASSERT(settings->SetValue(key, value1, "douglas_adams"));
        {
            auto got = settings->GetValue<TString>(key, TInstant::Max());
            UNIT_ASSERT(got);
            UNIT_ASSERT_VALUES_EQUAL(*got, value1);
        }
        UNIT_ASSERT(settings->SetValue(key, value2, "douglas_adams"));
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
            auto got = settings->GetValue<TString>(key);
            UNIT_ASSERT(got);
            UNIT_ASSERT_VALUES_EQUAL(*got, value2);
        }
    }

    Y_UNIT_TEST(Users) {
        TTestEnvironment env;
        const auto userData = env.GetServer()->GetDriveAPI()->GetUsersData();
        UNIT_ASSERT(userData);
        const auto& userTable = *userData;
        auto session = userTable.BuildSession();
        auto phoneVerified = userTable.Fetch(session, NSQL::TQueryOptions()
            .SetGenericCondition("is_phone_verified", true)
        );
        UNIT_ASSERT_C(phoneVerified, session.GetStringReport());
        UNIT_ASSERT(!phoneVerified->empty());
        for (auto&& user : *phoneVerified) {
            UNIT_ASSERT(user.IsPhoneVerified());
        }

        auto newUser = userData->RegisterNewUser("unittest", "42", "forty-two", session);
        UNIT_ASSERT(newUser);

        auto phoneNotVerified = userTable.FetchInfo(session, NSQL::TQueryOptions()
            .SetGenericCondition("is_phone_verified", false)
        );
        UNIT_ASSERT_C(phoneNotVerified, session.GetStringReport());
        UNIT_ASSERT(phoneNotVerified.GetResultPtr(newUser->GetUserId()));
        for (auto&& [id, user] : phoneNotVerified) {
            UNIT_ASSERT(!user.IsPhoneVerified());
        }
    }
}
