#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper2.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/background/car_markers/config.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/device_snapshot/snapshot.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/actions/ut/library/helper.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/tags/tag.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>


#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/mime/detect/detectmime.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(DriveEvolutions) {
    Y_UNIT_TEST(OffersDiscounts) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        THolder<TDiscountOfferCorrector> actorMasterCard(new TDiscountOfferCorrector);
        actorMasterCard->SetName("mastercard_offer_corrector");
        actorMasterCard->SetDescription("mastercard");
        TString discount = R"(
{
    "discount": 0.1,
    "visible": true,
    "icon": "fakeurl",
    "tag_name": "old_state_reservation",
    "bins_list": [510000],
    "check_bins": true
}
        )";
        NJson::TJsonValue masterCardDiscountJson;
        UNIT_ASSERT(NJson::ReadJsonFastTree(discount, &masterCardDiscountJson));
        UNIT_ASSERT(actorMasterCard->DeserializeSpecialsFromJson(masterCardDiscountJson));

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetRolesManager()->GetActionsDB().ForceUpsert(actorMasterCard.Release(), USER_ROOT_DEFAULT, session));
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("mastercard_offer_corrector").SetRoleId("standart_access");
                UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
            UNIT_ASSERT(session.Commit());
        }

        eGenerator.GetBillingMock().SetReply(R"(
{
    "status": "success",
    "bound_payment_methods": [{
        "region_id": 225,
        "payment_method": "card",
        "expiration_month": "07",
        "binding_ts": "1536324485.656",
        "id": "card-x12345",
        "expired": false,
        "card_bank": "TINKOFF BANK",
        "system": "VISA",
        "account": "510000****0257",
        "expiration_year": "2021"
    }]
}
        )");

        auto car = eGenerator.CreateCar();
        const TString newIMEI = car.IMEI;
        const TString newCar = car.Id;

        UNIT_ASSERT(configGenerator.SetDefaultCard("card-x12345", USER_ID_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(newIMEI);
        UNIT_ASSERT(configGenerator.WaitCar(car.Id));
        TGeoCoord moscowCoord(37.58643489401084, 55.73761541977004);
        driver.RelocateCar(newIMEI, moscowCoord);
        UNIT_ASSERT(configGenerator.WaitLocation(newCar, moscowCoord));
        {
            NJson::TJsonValue report;
            TString offerId = configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report);
            UNIT_ASSERT(!!offerId);
            INFO_LOG << report.GetStringRobust() << Endl;
            auto&& discounts = report["offers"][0]["discounts"].GetArray();
            UNIT_ASSERT(discounts.size() > 0);
            double mastercard = 0.0;
            for (auto&& discount : discounts) {
                if (discount["title"].GetString() == "mastercard") {
                    mastercard = discount["discount"].GetDoubleSafe();
                }
            }
            UNIT_ASSERT(mastercard > 0);
        }

        eGenerator.GetBillingMock().SetReply(R"(
{
    "status": "success",
    "bound_payment_methods": [{
        "region_id": 225,
        "payment_method": "card",
        "expiration_month": "07",
        "binding_ts": "1536324485.656",
        "id": "card-x12",
        "expired": false,
        "card_bank": "TINKOFF BANK",
        "system": "VISA",
        "account": "511100****0257",
        "expiration_year": "2021"
    }]
}
        )");

        UNIT_ASSERT(configGenerator.SetDefaultCard("card-x12", USER_ID_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            NJson::TJsonValue report;
            TString offerId = configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report);
            UNIT_ASSERT(!!offerId);
            INFO_LOG << report.GetStringRobust() << Endl;
            auto&& discounts = report["offers"][0]["discounts"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(discounts.size(), 0);
        }
    }

    Y_UNIT_TEST(BuildOffersForDefaultCard) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        THolder<TDiscountOfferCorrector> actorMasterCard(new TDiscountOfferCorrector);
        actorMasterCard->SetName("mastercard_offer_corrector");
        actorMasterCard->SetDescription("mastercard");
        TString discount = R"(
            {
                "discount": 0.1,
                "visible": true,
                "icon": "fakeurl",
                "tag_name": "old_state_reservation",
                "bins_list": [510000],
                "check_bins": true
            }
        )";
        NJson::TJsonValue masterCardDiscountJson;
        UNIT_ASSERT(NJson::ReadJsonFastTree(discount, &masterCardDiscountJson));
        UNIT_ASSERT(actorMasterCard->DeserializeSpecialsFromJson(masterCardDiscountJson));

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetRolesManager()->GetActionsDB().ForceUpsert(actorMasterCard.Release(), USER_ROOT_DEFAULT, session));
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("mastercard_offer_corrector").SetRoleId("standart_access");
                UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
            UNIT_ASSERT(session.Commit());
        }

        eGenerator.GetBillingMock().SetReply(R"(
            {
                "status": "success",
                "bound_payment_methods": [{
                    "region_id": 225,
                    "payment_method": "card",
                    "expiration_month": "07",
                    "binding_ts": "1536324485.656",
                    "id": "card-x12345",
                    "expired": false,
                    "card_bank": "TINKOFF BANK",
                    "system": "VISA",
                    "account": "510000****0257",
                    "expiration_year": "2021"
                },
                {
                    "region_id": 225,
                    "payment_method": "card",
                    "expiration_month": "07",
                    "binding_ts": "1536324485.656",
                    "id": "card-x12",
                    "expired": false,
                    "card_bank": "TINKOFF BANK",
                    "system": "VISA",
                    "account": "511100****0257",
                    "expiration_year": "2021"
                }
            ]
            }
        )");

        auto car = eGenerator.CreateCar();
        const TString newIMEI = car.IMEI;
        const TString newCar = car.Id;

        UNIT_ASSERT(configGenerator.SetDefaultCard("card-x12345", USER_ID_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(newIMEI);
        UNIT_ASSERT(configGenerator.WaitCar(car.Id));
        TGeoCoord moscowCoord(37.58643489401084, 55.73761541977004);
        driver.RelocateCar(newIMEI, moscowCoord);
        UNIT_ASSERT(configGenerator.WaitLocation(newCar, moscowCoord));
        {
            NJson::TJsonValue report;
            TString offerId = configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report);
            UNIT_ASSERT(!!offerId);
            INFO_LOG << report.GetStringRobust() << Endl;
            auto&& discounts = report["offers"][0]["discounts"].GetArray();
            UNIT_ASSERT(discounts.size() > 0);
            double mastercard = 0.0;
            for (auto&& discount : discounts) {
                if (discount["title"].GetString() == "mastercard") {
                    mastercard = discount["discount"].GetDoubleSafe();
                }
            }
            UNIT_ASSERT(mastercard > 0);
        }

        {
            NJson::TJsonValue report;
            TString offerId = configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report, 0, {}, {}, nullptr, "", TInstant::Max(), "card-x12");
            UNIT_ASSERT(!!offerId);
            INFO_LOG << report.GetStringRobust() << Endl;
            auto&& discounts = report["offers"][0]["discounts"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(discounts.size(), 0);
        }

        UNIT_ASSERT(configGenerator.SetDefaultCard("card-x12", USER_ID_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            auto card = driveApi.GetBillingManager().GetDefaultCreditCard(USER_ID_DEFAULT, Now());
            UNIT_ASSERT(card);
            UNIT_ASSERT_STRINGS_EQUAL(*card, "card-x12");
        }

        {
            NJson::TJsonValue report;
            TString offerId = configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report);
            UNIT_ASSERT(!!offerId);
            INFO_LOG << report.GetStringRobust() << Endl;
            auto&& discounts = report["offers"][0]["discounts"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(discounts.size(), 0);
        }

        {
            NJson::TJsonValue report;
            TString offerId = configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report, 0, {}, {}, nullptr, "", TInstant::Max(), "card-x12345");
            UNIT_ASSERT(!!offerId);
            INFO_LOG << report.GetStringRobust() << Endl;
            auto&& discounts = report["offers"][0]["discounts"].GetArray();
            UNIT_ASSERT(discounts.size() > 0);
            double mastercard = 0.0;
            for (auto&& discount : discounts) {
                if (discount["title"].GetString() == "mastercard") {
                    mastercard = discount["discount"].GetDoubleSafe();
                }
            }
            UNIT_ASSERT(mastercard > 0);

            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            auto card = driveApi.GetBillingManager().GetDefaultCreditCard(USER_ID_DEFAULT, Now());
            UNIT_ASSERT(card);
            UNIT_ASSERT_STRINGS_EQUAL(*card, "card-x12345");
        }


    }

    Y_UNIT_TEST(FreeTimeOfferConstruction) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetSensorApiName({});
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("correction_free_time_1600_1800").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT_C(server->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }

        auto car = eGenerator.CreateCar();
        const TString newIMEI = car.IMEI;
        const TString newCar = car.Id;

        auto emulator = tmBuilder.BuildEmulator(newIMEI);
        UNIT_ASSERT(configGenerator.WaitSensor(newCar, "mileage"));

        {
            NJson::TJsonValue report;
            TInstantGuard ig(TInstant::Seconds(1547650800));
            const TString someOfferId = configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report, 0);
            UNIT_ASSERT(someOfferId);
            INFO_LOG << report << Endl;
            TString offerId;
            for (auto&& offer : report["offers"].GetArraySafe()) {
                const auto& type = offer["type"].GetStringSafe();
                if (type == "standart_offer") {
                    offerId = offer["offer_id"].GetStringSafe();
                }
            }
            UNIT_ASSERT(offerId);
            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            ig.Set(TInstant::Seconds(1547665200));
            UNIT_ASSERT(configGenerator.WaitPrice((2 * 60 - 20) * 400));
        }
    }

    Y_UNIT_TEST(PackOfferConstruction) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        auto car = eGenerator.CreateCar();
        const TString newIMEI = car.IMEI;
        const TString newCar = car.Id;

        auto emulator = tmBuilder.BuildEmulator(newIMEI);
        UNIT_ASSERT(configGenerator.WaitSensor(newCar, "mileage"));

        {
            NJson::TJsonValue report;
            configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report, 0);
            INFO_LOG << report << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["offers"].GetArraySafe().size(), 2);
            configGenerator.CreateOffer(newCar, USER_ID_DEFAULT, &report, 200);
            UNIT_ASSERT_VALUES_EQUAL(report["offers"].GetArraySafe().size(), 6);
            bool found = false;
            for (auto&& i : report["offers"].GetArraySafe()) {
                if (i["type"].GetStringSafe() == "pack_offer_builder") {
                    const TString offerId = i["offer_id"].GetString();
                    UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
                    found = true;
                    break;
                }
            }
            UNIT_ASSERT(found);
        }
    }

    Y_UNIT_TEST(CarsListSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);
        TGeoCoord moscowCoord(37.58643489401084, 55.73761541977004);
        emulator->GetContext().SetCurrentPosition(moscowCoord);
        UNIT_ASSERT(configGenerator.WaitLocation(OBJECT_ID_DEFAULT, moscowCoord));

        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        {
            NJson::TJsonValue carsReport = configGenerator.GetAdminCarsList(USER_ROOT_DEFAULT);
            INFO_LOG << carsReport.GetStringRobust() << Endl;
            UNIT_ASSERT(carsReport["cars"].IsArray());
            UNIT_ASSERT(carsReport["cars"].GetArraySafe().size() > 0);
            UNIT_ASSERT(carsReport["models"]["bmw_520i_w"].IsMap());
            const TString& first = carsReport["cars"].GetArraySafe().front()["id"].GetStringSafe();
            UNIT_ASSERT(configGenerator.WaitLocation(first));
        }
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsList({OBJECT_ID_DEFAULT}, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT(userSessionReport["cars"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"].GetArraySafe().size(), 0);
        }
        {
            NJson::TJsonValue userReport = configGenerator.GetUserCarsList({}, USER_ROOT_DEFAULT, {}, "&sort=rank");
            INFO_LOG << userReport.GetStringRobust() << Endl;
            const NJson::TJsonValue& cars = userReport["cars"];
            UNIT_ASSERT(cars.IsArray());
            UNIT_ASSERT(!cars.GetArraySafe().empty());

            TMap<TString, i64> ranks;
            const NJson::TJsonValue& models = userReport["models"];
            for (auto&&[modelCode, modelObject] : models.GetMapSafe()) {
                ranks[modelCode] = modelObject["rank"].GetInteger();
            }

            const auto& arr = cars.GetArraySafe();
            UNIT_ASSERT(std::is_sorted(arr.begin(), arr.end(), [&ranks](const NJson::TJsonValue& left, const NJson::TJsonValue& right) {
                auto lrank = -1 * ranks.at(left["model_id"].GetStringSafe());
                auto rrank = -1 * ranks.at(right["model_id"].GetStringSafe());
                auto lnumber = left["number"].GetStringSafe();
                auto rnumber = right["number"].GetStringSafe();
                return std::tie(lrank, lnumber) < std::tie(rrank, rnumber);
            }));
        }
        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        TString tagId1;
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 1);
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                if (i["tag"].GetString() == "simple1") {
                    tagId1 = i["tag_id"].GetString();
                    UNIT_ASSERT_VALUES_EQUAL(i["performer"].GetString(), "");
                }
                UNIT_ASSERT(i["tag"].GetString() != "simple2");
            }
        }
        UNIT_ASSERT(!!tagId1);
        UNIT_ASSERT(configGenerator.StartTag(tagId1, USER_ID_DEFAULT));
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                if (i["tag"].GetString() == "simple1") {
                    UNIT_ASSERT_VALUES_EQUAL(i["performer"].GetString(), USER_ID_DEFAULT);
                }
            }
        }
        UNIT_ASSERT(configGenerator.FinishTag(tagId1, USER_ID_DEFAULT, false));
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                if (i["tag"].GetString() == "simple1") {
                    UNIT_ASSERT_VALUES_EQUAL(i["performer"].GetString(), "");
                }
            }
        }
        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("simple2"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        TString tagId;
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                if (i["tag"] == "simple2") {
                    UNIT_ASSERT(!tagId);
                    tagId = i["tag_id"].GetString();
                }
            }
            UNIT_ASSERT(!!tagId);
        }
        UNIT_ASSERT_C(configGenerator.RemoveTag({ tagId }, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car), TStringBuilder() << tagId);
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                UNIT_ASSERT(i["tag"].GetString() != "simple2");
            }
        }
        configGenerator.GetServiceCarsDetails({}, USER_ID_DEFAULT);
        configGenerator.GetServiceCarsList({}, USER_ID_DEFAULT);
    }

    Y_UNIT_TEST(CarsPropertiesReport) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        TInstant now = Now();
        TInstantGuard ig(now);

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TDeviceAdditionalFeature::TDescription description;
            description.SetPublicIcon("").SetName("car_property").SetType("additional_feature_tag").SetDefaultPriority(0);
            UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TDeviceAdditionalFeature::TDescription(description), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        eGenerator.BuildEnvironment();

        auto car = eGenerator.CreateCar();

        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(configGenerator.WaitCar(car.Id));
        TGeoCoord moscowCoord(37.58643489401084, 55.73761541977004);
        driver.RelocateCar(car.IMEI, moscowCoord);
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id));

        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("simple1"), car.Id, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));

        {
            NJson::TJsonValue userSessionReport = configGenerator.GetUserCarsList({car.Id}, USER_ID_DEFAULT, "", "traits=ReportShortProps");
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), car.Number);

            UNIT_ASSERT_C(userSessionReport["filters"].GetArraySafe().size() >= 2, TStringBuilder() << userSessionReport << Endl);
            const ui32 carPropsCount = userSessionReport["sf"].GetArraySafe().size();
            INFO_LOG << userSessionReport["sf"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(carPropsCount, 0);

            UNIT_ASSERT(configGenerator.AddTag(new TDeviceAdditionalFeature("car_property"), car.Id, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));

            Sleep(TDuration::Seconds(1));

            userSessionReport = configGenerator.GetUserCarsList({car.Id}, USER_ID_DEFAULT, "", "traits=ReportShortProps");
            INFO_LOG << userSessionReport["sf"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["sf"].GetArray().size(), carPropsCount + 1);
        }
    }

    Y_UNIT_TEST(ServiceAppUploadPhotos) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const TInstant now = Now();

        TBlob image = TBlob::FromString(Base64Decode("iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAAAXNSR0IArs4c6QAAAIRlWElmTU0AKgAAAAgABQESAAMAAAABAAEAAAEaAAUAAAABAAAASgEbAAUAAAABAAAAUgEoAAMAAAABAAIAAIdpAAQAAAABAAAAWgAAAAAAAABIAAAAAQAAAEgAAAABAAOgAQADAAAAAQABAACgAgAEAAAAAQAAAAGgAwAEAAAAAQAAAAEAAAAAChjw/QAAAAlwSFlzAAALEwAACxMBAJqcGAAAAVlpVFh0WE1MOmNvbS5hZG9iZS54bXAAAAAAADx4OnhtcG1ldGEgeG1sbnM6eD0iYWRvYmU6bnM6bWV0YS8iIHg6eG1wdGs9IlhNUCBDb3JlIDUuNC4wIj4KICAgPHJkZjpSREYgeG1sbnM6cmRmPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5LzAyLzIyLXJkZi1zeW50YXgtbnMjIj4KICAgICAgPHJkZjpEZXNjcmlwdGlvbiByZGY6YWJvdXQ9IiIKICAgICAgICAgICAgeG1sbnM6dGlmZj0iaHR0cDovL25zLmFkb2JlLmNvbS90aWZmLzEuMC8iPgogICAgICAgICA8dGlmZjpPcmllbnRhdGlvbj4xPC90aWZmOk9yaWVudGF0aW9uPgogICAgICA8L3JkZjpEZXNjcmlwdGlvbj4KICAgPC9yZGY6UkRGPgo8L3g6eG1wbWV0YT4KTMInWQAAAA1JREFUCB1jmFyf+R8ABZ8Ce8TZVxMAAAAASUVORK5CYII="));

        TMimeDetector detector;
        detector.Detect(image.Data(), image.Size());
        UNIT_ASSERT_VALUES_EQUAL(detector.Mime(), MIME_IMAGE_PNG);

        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        TString tagId1;
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 1);
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                if (i["tag"].GetString() == "simple1") {
                    tagId1 = i["tag_id"].GetString();
                    UNIT_ASSERT_VALUES_EQUAL(i["performer"].GetString(), "");
                }
                UNIT_ASSERT(i["tag"].GetString() != "simple2");
            }
        }
        UNIT_ASSERT(!!tagId1);
        UNIT_ASSERT(configGenerator.StartTag(tagId1, USER_ID_DEFAULT));
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                if (i["tag"].GetString() == "simple1") {
                    UNIT_ASSERT_VALUES_EQUAL(i["performer"].GetString(), USER_ID_DEFAULT);
                }
            }
        }
        UNIT_ASSERT(configGenerator.RegisterPhotos(USER_ID_DEFAULT, "photo_id", tagId1, MD5::Calc(TStringBuf(image.AsCharPtr(), image.Size())), "test_marker"));

        const auto& deviceTagsManager = driveApi.GetTagsManager().GetDeviceTags();
        auto session = deviceTagsManager.BuildSession(true);
        auto optionalEvents = deviceTagsManager.GetEventsByObject(OBJECT_ID_DEFAULT, session, 0, now - TDuration::Minutes(1));
        UNIT_ASSERT(optionalEvents);
        bool hasSnapshot = false;
        for (auto&& event : *optionalEvents) {
            if (event->GetName() != "simple1") {
                continue;
            }
            if (event.GetHistoryAction() == EObjectHistoryAction::AddSnapshot) {
                hasSnapshot = true;
                const TImagesSnapshot* imagesSnapshot = event->template GetObjectSnapshotAs<TImagesSnapshot>();
                UNIT_ASSERT(!!imagesSnapshot);
                UNIT_ASSERT_VALUES_EQUAL(event.GetHistoryUserId(), USER_ID_DEFAULT);
                const auto& images = imagesSnapshot->GetImages();
                UNIT_ASSERT_VALUES_EQUAL(images.size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(images.at(0).Marker, "test_marker");
                UNIT_ASSERT_VALUES_EQUAL(images.at(0).ExternalId, "photo_id");
            }
        }
        UNIT_ASSERT(hasSnapshot);
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetServiceCarsDetails({ OBJECT_ID_DEFAULT }, USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["tags"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["cars"][0]["number"].GetString(), "б504вг209");
            for (auto&& i : userSessionReport["cars"][0]["tags"].GetArraySafe()) {
                if (i["tag"].GetString() == "simple1") {
                    UNIT_ASSERT_VALUES_EQUAL(i["performer"].GetString(), USER_ID_DEFAULT);
                }
            }
        }
        UNIT_ASSERT(configGenerator.FinishTag(tagId1, USER_ID_DEFAULT, false));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(configGenerator.UploadPhotos(USER_ID_DEFAULT, OBJECT_ID_DEFAULT, "photo_id", image));
    }

    Y_UNIT_TEST(PhoneVerification) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        //Set phone not verified
        auto table = driveApi.GetDatabase().GetTable("\"user\"");
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto user = driveApi.GetUsersData()->FetchInfo(USER_ID_NOPHONE).GetResult().begin()->second;
            user.SetPhoneVerified(false);
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(user, "unittest", session));
            UNIT_ASSERT(session.Commit());
            Sleep(TDuration::Seconds(5));
        }

        {
            NJson::TJsonValue report = configGenerator.GetCurrentSession(USER_ID_NOPHONE);
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT(report["user"]["details"]["setup"]["phone"]["verified"].IsBoolean());
            UNIT_ASSERT(!report["user"]["details"]["setup"]["phone"]["verified"].GetBoolean());
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto user = driveApi.GetUsersData()->FetchInfo(USER_ID_NOPHONE).GetResult().begin()->second;
            user.SetPhoneVerified(true);
            driveApi.GetUsersData()->UpdateUser(user, "unittest", session);
            UNIT_ASSERT(session.Commit());
            Sleep(TDuration::Seconds(5));
        }

        {
            NJson::TJsonValue report = configGenerator.GetCurrentSession(USER_ID_NOPHONE);
            INFO_LOG << report.GetStringRobust() << Endl;
            UNIT_ASSERT(report["user"]["details"]["setup"]["phone"]["verified"].IsBoolean());
            UNIT_ASSERT(report["user"]["details"]["setup"]["phone"]["verified"].GetBoolean());
        }
    }

    void TestCleaning(NDrive::TServerConfigGenerator& configGenerator, const NDrive::TServerGuard& server, const TString& tagId) {
        UNIT_ASSERT(configGenerator.StartTag(tagId, USER_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "cleaning", *server));
        TString offerId;
        {
            auto offer = BuildOfferPtr(200, 100, 102400);
            offer->SetObjectId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
            offer->SetChargableAccounts({ "card", "bonus" });
            offerId = offer->GetOfferId();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({ new TStandartOfferReport(offer, nullptr) }));
        }
        UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "old_state_reservation", *server));
        UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["current_performing"].GetString() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["segment"]["session"]["current_performing"].GetString(), "old_state_acceptance");
        }
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "old_state_acceptance", *server));
        UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "old_state_riding", *server));
        UNIT_ASSERT(configGenerator.EvolveTag("old_state_parking", USER_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "old_state_parking", *server));
        UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "old_state_riding", *server));
        UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "cleaning", *server));
        {
            NJson::TJsonValue userSessionReport = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            UNIT_ASSERT(!userSessionReport["segment"]["session"].Has("current_performing"));
        }
    }

    Y_UNIT_TEST(CleanProcessingFromAvailable) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);
        UNIT_ASSERT(configGenerator.WaitLocation(OBJECT_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitSensor(OBJECT_ID_DEFAULT, "mileage"));
        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("cleaning"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("cleaning"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        TVector<TDBTag> tags;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "cleaning" }, tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 2);
        }
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "available", *server));

        UNIT_ASSERT(configGenerator.AddTag(&(new TDeviceTagRecord("cleaning"))->SetTagPriority(1000), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car, EUniquePolicy::Rewrite));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "cleaning", *server));

        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("cleaning"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car, EUniquePolicy::Rewrite));
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "available", *server));

        TestCleaning(configGenerator, server, tags[0].GetTagId());
    }

    Y_UNIT_TEST(CleanProcessingFromService) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);
        UNIT_ASSERT(configGenerator.WaitLocation(OBJECT_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitSensor(OBJECT_ID_DEFAULT, "mileage"));
        {
            ITag::TPtr tagService = new TDeviceTagRecord("priority_simple1");
            tagService->SetTagPriority(10);
            UNIT_ASSERT(configGenerator.AddTag(tagService, OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
            UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "service", *server));
        }
        UNIT_ASSERT(configGenerator.AddTag(new TDeviceTagRecord("cleaning"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        TVector<TDBTag> tags;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "cleaning" }, tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }
        UNIT_ASSERT(configGenerator.WaitStatus(OBJECT_ID_DEFAULT, "service", *server));
        TestCleaning(configGenerator, server, tags[0].GetTagId());
    }

    Y_UNIT_TEST(DefaultTagPriorityUsage) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        {
            NJson::TJsonValue report = gServer.ListTags(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.IsNull() || (report.Has("records") && report["records"].GetArraySafe().empty()));
        }
        {
            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("priority_simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
            NJson::TJsonValue report = gServer.ListTags(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            INFO_LOG << report << Endl;
            UNIT_ASSERT_VALUES_EQUAL(report["records"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(report["records"].GetArraySafe()[0]["tag"].GetString(), "priority_simple1");
            UNIT_ASSERT_VALUES_EQUAL(report["records"].GetArraySafe()[0]["priority"].GetInteger(), 1000);
        }
    }

    Y_UNIT_TEST(InitMultiObjects) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT1, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));

        TSet<TString> tagIds;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> tags;
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "simple1" }, tags, session));
            UNIT_ASSERT(tags.size());
            TVector<TDBTag> tags1;
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT1, { "simple1" }, tags1, session));
            UNIT_ASSERT(tags1.size());
            tagIds.emplace(tags.front().GetTagId());
            tagIds.emplace(tags1.front().GetTagId());
        }

        TVector<TDBTag> tags;
        UNIT_ASSERT(gServer.StartTag(tagIds, USER_ID_DEFAULT));
        auto session = driveApi.BuildTx<NSQL::Writable>();
        UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestorePerformerTags({ "simple1" }, { USER_ID_DEFAULT }, tags, session));
        ui32 idx = 0;
        for (auto&& i : tags) {
            if (tagIds.contains(i.GetTagId())) {
                ++idx;
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(idx, 2);
    }

    Y_UNIT_TEST(TagsDifferentTypes) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        UNIT_ASSERT(gServer.AddTag(new TSimpleUserTag("user_simple1"), USER_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::User));
        UNIT_ASSERT(!gServer.AddTag(new TSimpleUserTag("user_simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(!gServer.AddTag(new TSimpleUserTag("user_simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::User));
        UNIT_ASSERT(!gServer.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::User));
        UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));

        {
            NJson::TJsonValue report = gServer.ListTags(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            NJson::TJsonValue::TArray arr;
            UNIT_ASSERT_C(report["records"].GetArray(&arr), TStringBuilder() << report << Endl);
            ui32 idx = 0;
            for (auto&& i : arr) {
                UNIT_ASSERT(i.IsMap());
                UNIT_ASSERT_VALUES_EQUAL(i["object_id"].GetString(), OBJECT_ID_DEFAULT);
                if (i["tag"].GetString() == "simple1") {
                    UNIT_ASSERT(gServer.RemoveTag({ i["tag_id"].GetString() }, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
                    ++idx;
                }
            }
            UNIT_ASSERT(idx);
        }
        {
            NJson::TJsonValue report = gServer.ListTags(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            if (report != NJson::JSON_NULL) {
                NJson::TJsonValue::TArray arr;
                UNIT_ASSERT(report["records"].GetArray(&arr));
                for (auto&& i : arr) {
                    UNIT_ASSERT(i.IsMap());
                    UNIT_ASSERT_VALUES_EQUAL(i["object_id"].GetString(), OBJECT_ID_DEFAULT);
                    UNIT_ASSERT(i["tag"].GetString() != "simple1");
                }
            }
        }

        {
            NJson::TJsonValue report = gServer.ListTags(USER_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::User);
            NJson::TJsonValue::TArray arr;
            UNIT_ASSERT(report["records"].GetArray(&arr));
            ui32 idx = 0;
            for (auto&& i : arr) {
                UNIT_ASSERT(i.IsMap());
                UNIT_ASSERT_VALUES_EQUAL(i["object_id"].GetString(), USER_ID_DEFAULT);
                if (i["tag"].GetString() == "user_simple1") {
                    UNIT_ASSERT(gServer.RemoveTag({ i["tag_id"].GetString() }, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::User));
                    ++idx;
                }
            }
            UNIT_ASSERT(idx);
        }
        {
            NJson::TJsonValue report = gServer.ListTags(USER_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::User);
            if (report != NJson::JSON_NULL) {
                NJson::TJsonValue::TArray arr;
                UNIT_ASSERT(report["records"].GetArray(&arr));
                for (auto&& i : arr) {
                    UNIT_ASSERT(i.IsMap());
                    UNIT_ASSERT_VALUES_EQUAL(i["object_id"].GetString(), USER_ID_DEFAULT);
                    UNIT_ASSERT(i["tag"].GetString() != "user_simple1");
                }
            }
        }
    }

    Y_UNIT_TEST(AddServiceTag) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(!!userPermissions);
        {
            ITag::TPtr tag = new TDeviceTagRecord("simple1");
            tag->SetTagPriority(10);
            UNIT_ASSERT(gServer.AddTag(tag, OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            UNIT_ASSERT(gServer.WaitStatus(OBJECT_ID_DEFAULT, "service", *server));
        }
        {
            ITag::TPtr tag = new TDeviceTagRecord("urgent_cleaning");
            tag->SetTagPriority(100);
            UNIT_ASSERT(gServer.AddTag(tag, OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            UNIT_ASSERT(gServer.WaitStatus(OBJECT_ID_DEFAULT, "cleaning", *server));
        }
        {
            ITag::TPtr tag = new TDeviceTagRecord("installation");
            tag->SetTagPriority(1000);
            UNIT_ASSERT(gServer.AddTag(tag, OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            UNIT_ASSERT(gServer.WaitStatus(OBJECT_ID_DEFAULT, "new", *server));
        }
        {
            ITag::TPtr tag = new TDeviceTagRecord("cleaning");
            tag->SetTagPriority(1000);
            UNIT_ASSERT(gServer.AddTag(tag, OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        }
        UNIT_ASSERT(gServer.WaitStatus(OBJECT_ID_DEFAULT, "new", *server));
        {
            TVector<TDBTag> tags;
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "cleaning" }, tags, session));
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            }
            gServer.StartTag(tags.front().GetTagId(), USER_ROOT_DEFAULT);
        }
        UNIT_ASSERT(gServer.WaitStatus(OBJECT_ID_DEFAULT, "cleaning", *server));

    }

    Y_UNIT_TEST(PerformForceAndFree) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ID_DEFAULT, TUserPermissionsFeatures());
        TUserPermissions::TPtr userPermissions2 = driveApi.GetUserPermissions(USER_ID_DEFAULT2, TUserPermissionsFeatures());
        UNIT_ASSERT(!!userPermissions);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(new TDeviceTagRecord("simple1"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(new TDeviceTagRecord("simple2"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));

            {
                TVector<TDBTag> tags;
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "simple1" }, tags, session));
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().InitPerformer({ tags.front().GetTagId() }, *userPermissions, &*server, session));
            }
            {
                TVector<TDBTag> tags;
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "simple1", "simple2" }, tags, session));
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 2);
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().InitPerformer(TSet<TString>({ tags.front().GetTagId(), tags.back().GetTagId() }), *userPermissions2, &*server, session));
            }
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(new TDeviceTagRecord("simple1"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(new TDeviceTagRecord("simple2"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));

            {
                TVector<TDBTag> tags;
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "simple2" }, tags, session));
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().InitPerformer({ tags.front().GetTagId() }, *userPermissions, &*server, session));
            }
            {
                TVector<TDBTag> tags;
                UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "simple1", "simple2" }, tags, session));
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 2);
                UNIT_ASSERT(!driveApi.GetTagsManager().GetDeviceTags().InitPerformer(TSet<TString>({ tags.front().GetTagId(), tags.back().GetTagId() }), *userPermissions2, &*server, session));
            }
        }
    }

    Y_UNIT_TEST(DropServiceTagWithReleaseCar) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(new TServiceTagRecord("DTP"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(new TServiceTagRecord("DTP"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            TVector<TDBTag> tags;
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "DTP" }, tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 2);
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(!emulator->GetContext().GetDoorsOpened());
            UNIT_ASSERT(gServer.StartTag(tags.front().GetTagId(), USER_ID_DEFAULT));
            UNIT_ASSERT(gServer.StartTag(tags.back().GetTagId(), USER_ID_DEFAULT));
            UNIT_ASSERT(gServer.CarRootControl(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, "OPEN_DOORS"));
            UNIT_ASSERT(emulator->GetContext().GetDoorsOpened());
            UNIT_ASSERT(gServer.FinishTag(tags.back().GetTagId(), USER_ID_DEFAULT));
            UNIT_ASSERT(emulator->GetContext().GetDoorsOpened());
            UNIT_ASSERT(gServer.FinishTag(tags.front().GetTagId(), USER_ID_DEFAULT));
            UNIT_ASSERT(!emulator->GetContext().GetDoorsOpened());
        }
    }

    Y_UNIT_TEST(EvolutionFullScenario) {
        InitGlobalLog2Console(TLOG_DEBUG);
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        {
            const TString coordsArea = "36.50571688 57.96862233 38.89798983 59.28069963 37.73069247 59.82167053 41.37266024 60.48976579 46.98667391 57.62840898 45.115325 53.61686361 38.14631985 51.43302596 35.14705227 52.77246007 31.93355129 55.45020355 31.26887844 55.89103786 36.50571688 57.96862233";
            TVector<TGeoCoord> area;
            UNIT_ASSERT(TGeoCoord::DeserializeVector(coordsArea, area));
            UNIT_ASSERT(configGenerator.UpsertArea("test_msk_area", USER_ROOT_DEFAULT, area, {}));

            THolder<TAreaInfoForUser> tagUserInfo(new TAreaInfoForUser("common_area_user_info"));
            tagUserInfo->MutableInfos().emplace("phone", "phone");
            tagUserInfo->MutableInfos().emplace("support_feedback_form", "form");
            tagUserInfo->MutableInfos().emplace("support_phone", "phone");
            tagUserInfo->MutableInfos().emplace("support_telegram", "phone");
            tagUserInfo->MutableInfos().emplace("support_whatsapp", "phone");
            UNIT_ASSERT(configGenerator.AddTag(tagUserInfo.Release(), "test_msk_area", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Area));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }

        auto car = eGenerator.CreateCar();

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        NDrive::TTelematicsTestClient::TPtr client = emulator->GetClient();
        UNIT_ASSERT(configGenerator.WaitCar(car.Id));
        TString offerId;
        {
            TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ID_DEFAULT, TUserPermissionsFeatures());
            {
                auto offer = BuildOfferPtr(200, 100, 102400);
                offer->SetObjectId(car.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetChargableAccounts({ "card", "bonus" });
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
            }
            {
                TVector<TDBTag> tags;
                TInstant now = Now();
                TInstantGuard ig(now - TDuration::Hours(12));
                {
                    UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
                    NDrive::TServerConfigGenerator::TSessionStateGuard sg(configGenerator, "book_offer");
                    NJson::TJsonValue currentSession = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                    UNIT_ASSERT(!currentSession["notification"].IsDefined());
                }
                {
                    NJson::TJsonValue userSessionReport = configGenerator.GetUserLastSession(USER_ID_DEFAULT);
                    INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                    UNIT_ASSERT(!userSessionReport["sessions"][0]["segment"]["session"]["specials"]["is_cancelled"].GetBoolean());
                }
                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_acceptance", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    client->SetCommandResponseError(true);
                    ig.Set(now - TDuration::Hours(12));
                    UNIT_ASSERT(!configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
                    client->SetCommandResponseError(false);
                }

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_acceptance", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    client->SetResponseDelay(TDuration::Seconds(20));
                    ig.Set(now - TDuration::Hours(12));
                    UNIT_ASSERT(!configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT, TDuration::Seconds(5)));
                    client->SetResponseDelay(TDuration::Zero());
                }

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_acceptance", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    ig.Set(now - TDuration::Hours(12));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
                    NDrive::TServerConfigGenerator::TSessionStateGuard sg(configGenerator, "old_state_acceptance");
                    NJson::TJsonValue currentSession = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                    UNIT_ASSERT(!currentSession["notification"].IsDefined());
                }

                INFO_LOG << "DrDoorOpened: " << emulator->GetContext().GetDrDoorOpened() << Endl;
                UNIT_ASSERT(emulator->GetContext().GetDoorsOpened());

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_parking", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
                    ig.Set(now - TDuration::Hours(11));
                    UNIT_ASSERT(!configGenerator.EvolveTag("old_state_parking", USER_ID_DEFAULT));
                }

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_riding", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    ig.Set(now - TDuration::Hours(10));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
                    NDrive::TServerConfigGenerator::TSessionStateGuard sg(configGenerator, "old_state_riding");
                    NJson::TJsonValue currentSession = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                    UNIT_ASSERT(currentSession["notification"].IsString());
                }

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_parking", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    ig.Set(now - TDuration::Hours(5));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_parking", USER_ID_DEFAULT));
                    NDrive::TServerConfigGenerator::TSessionStateGuard sg(configGenerator, "old_state_parking");
                    NJson::TJsonValue currentSession = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                    UNIT_ASSERT(currentSession["notification"].IsString());
                }

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_reservation", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    ig.Set(now - TDuration::Hours(4));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
                    NDrive::TServerConfigGenerator::TSessionStateGuard sg(configGenerator, "old_state_reservation");
                    NJson::TJsonValue currentSession = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                    UNIT_ASSERT(!currentSession["notification"].IsDefined());
                }
                {
                    NJson::TJsonValue userSessionReport = configGenerator.GetUserLastSession(USER_ID_DEFAULT);
                    INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                    INFO_LOG << userSessionReport["sessions"][0]["segment"]["session"]["specials"]["total_price"] << Endl;
                    UNIT_ASSERT_VALUES_EQUAL(78000, userSessionReport["sessions"][0]["segment"]["session"]["specials"]["total_price"].GetUInteger());
                    UNIT_ASSERT(!userSessionReport["sessions"][0]["segment"]["session"]["specials"]["is_cancelled"].GetBoolean());

                }
                {
                    NJson::TJsonValue userSessionReport = configGenerator.GetUserSessions(USER_ID_DEFAULT);
                    INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                    INFO_LOG << userSessionReport["sessions"][0]["segment"]["start"].GetUInteger() << " / " << (now - TDuration::Hours(12)).Seconds() << Endl;
                    INFO_LOG << userSessionReport["sessions"][0]["segment"]["finish"].GetUInteger() << " / " << (now - TDuration::Hours(4)).Seconds() << Endl;
                    UNIT_ASSERT_VALUES_EQUAL(userSessionReport["sessions"][0]["segment"]["start"].GetUInteger(), (now - TDuration::Hours(12)).Seconds());
                    UNIT_ASSERT_VALUES_EQUAL(userSessionReport["sessions"][0]["segment"]["finish"].GetUInteger(), (now - TDuration::Hours(4)).Seconds());

                }

                UNIT_ASSERT(!emulator->GetContext().GetDoorsOpened());

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT_C(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_reservation", tags, session), TStringBuilder() << session.GetStringReport() << Endl);
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
                }
            }

            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto optionalSession = driveApi.GetSessionManager().GetSession(offerId, tx);
            UNIT_ASSERT(optionalSession);
            auto bSession = *optionalSession;
            UNIT_ASSERT(bSession);

            const auto& compiledRidesManager = driveApi.GetMinimalCompiledRides();
            auto defaultTx = NDrive::TEntitySession();
            auto sessions = compiledRidesManager.Get<TMinimalCompiledRiding>({offerId}, tx, defaultTx);
            UNIT_ASSERT(sessions);
            UNIT_ASSERT_VALUES_EQUAL(sessions->size(), 1);

            TMessagesCollector errors;
            auto compiledSession = TBillingSession::BuildFromCompiled(server.Get(), sessions->front(), errors);
            UNIT_ASSERT(compiledSession);

            TBillingSession::TBillingCompilation bCompilationStart;
            bCompilationStart.SetUntil(bSession->GetStartTS());
            TBillingSession::TBillingCompilation bCompilationUntil;
            bCompilationUntil.SetUntil(bSession->GetLastTS());

            UNIT_ASSERT(bSession->FillCompilation(bCompilationStart));
            UNIT_ASSERT(bSession->FillCompilation(bCompilationUntil));

            //const i32 billingPrice = bCompilationUntil.GetBillingSumPrice() - bCompilationStart.GetBillingSumPrice();
            const i32 reportPrice = bCompilationUntil.GetReportSumPrice() - bCompilationStart.GetReportSumPrice();
            UNIT_ASSERT_VALUES_EQUAL(reportPrice, 78000);
        }
    }

    Y_UNIT_TEST(EvolutionFullScenarioWithBackground) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::RentPricing);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);
        UNIT_ASSERT(configGenerator.WaitLocation(OBJECT_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitSensor(OBJECT_ID_DEFAULT, "mileage"));
        {
            TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ID_DEFAULT, TUserPermissionsFeatures());
            TString offerId;
            {
                THolder<TStandartOffer> offer(new TStandartOffer);
                offer->SetObjectId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->MutableRiding().SetPrice(200);
                offer->MutableParking().SetPrice(100);
                offer->SetChargableAccounts({ "card", "bonus" });
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer.Release(), nullptr)}));
            }
            {
                TVector<TDBTag> tags;
                TInstant now = Now();
                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "old_state_reservation" }, tags, session));
                    UNIT_ASSERT(!!userPermissions);
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT(session.Commit());
                    TInstantGuard ig(now - TDuration::Hours(12));
                    UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
                }

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_acceptance", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    TInstantGuard ig(now - TDuration::Hours(12));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
                }

                TInstantGuard ig(now - TDuration::Hours(5));
                for (ui32 i = 0; i <= 100; ++i) {
                    NDrive::TServerConfigGenerator::TDisableLogging disableLogging(configGenerator);
                    NJson::TJsonValue userSessionReport = configGenerator.GetUserLastSession(USER_ID_DEFAULT);
                    INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                    INFO_LOG << userSessionReport["sessions"][0]["segment"]["session"]["specials"]["total_price"] << Endl;
                    if (42000 == userSessionReport["sessions"][0]["segment"]["session"]["specials"]["total_price"].GetUInteger()) {
                        break;
                    } else {
                        UNIT_ASSERT(i < 100);
                    }
                    Sleep(TDuration::Seconds(1));
                }

                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_reservation", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    TInstantGuard ig(now - TDuration::Hours(4));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
                }
            }

        }
    }

    Y_UNIT_TEST(EvolutionFullScenarioTagsInPoint) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        configGenerator.SetNeedBackground(EServerBackgrounds::RentPricing);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id));
        NDrive::TTelematicsTestClient::TPtr client = emulator->GetClient();
        {
            TString offerId;
            {
                auto offer = BuildOfferPtr(200, 100, 102400);
                offer->SetObjectId(car.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetChargableAccounts({ "card", "bonus" });
                {
                    TDiscount discount;
                    TDiscount::TDiscountDetails details;
                    details.SetTagName("old_state_parking").SetTagsInPoint({"!allow_drop_car"});
                    TTimeRestriction restriction;
                    restriction.SetTimeRestriction(15, 2350);
                    TTimeRestrictionsPool<TTimeRestriction> freeTimetable;
                    freeTimetable.Add(restriction);
                    details.SetFreeTimetable(freeTimetable);
                    discount.AddDetails(details);
                    offer->AddDiscount(discount);
                }
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
            }
            {
                TGeoCoord moscowCoord(37.58643389401084, 55.73761641977004);
                TGeoCoord iranCoord(55.73761641977004, 37.58643389401084);
                TVector<TDBTag> tags;
                TInstant now = TInstant::Seconds(1534726800);
                driver.RelocateCar(car.IMEI, moscowCoord);
                UNIT_ASSERT(configGenerator.WaitLocation(car.Id, moscowCoord));
                TInstantGuard ig(now - TDuration::Hours(12));
                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(car.Id, {"old_state_reservation"}, tags, session));
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT(session.Commit());
                    UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
                }

                {
                    ig.Set(now - TDuration::Hours(12));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
                }

                {
                    ig.Set(now - TDuration::Hours(12));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
                }
                driver.RelocateCar(car.IMEI, moscowCoord);
                UNIT_ASSERT(configGenerator.WaitLocation(car.Id, moscowCoord));

                {
                    ig.Set(now - TDuration::Hours(11));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_parking", USER_ID_DEFAULT));
                }
                UNIT_ASSERT(configGenerator.WaitPrice(12000));

                {
                    ig.Set(now - TDuration::Hours(10));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
                }
                UNIT_ASSERT(configGenerator.WaitPrice(12000 + 6000));
                driver.RelocateCar(car.IMEI, iranCoord);
                UNIT_ASSERT(configGenerator.WaitLocation(car.Id, iranCoord));

                {
                    ig.Set(now - TDuration::Hours(9));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_parking", USER_ID_DEFAULT));
                }
                UNIT_ASSERT(configGenerator.WaitPrice(12000 + 6000 + 12000));

                {
                    ig.Set(now - TDuration::Hours(8));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
                }
                UNIT_ASSERT(configGenerator.WaitPrice(12000 + 6000 + 12000));
                driver.RelocateCar(car.IMEI, moscowCoord);
                UNIT_ASSERT(configGenerator.WaitLocation(car.Id, moscowCoord));

                {
                    ig.Set(now - TDuration::Hours(7));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_parking", USER_ID_DEFAULT));
                }
                UNIT_ASSERT(configGenerator.WaitPrice(12000 + 6000 + 12000 + 12000));

                {
                    ig.Set(now - TDuration::Hours(6));
                    UNIT_ASSERT(configGenerator.WaitPrice(12000 + 6000 + 12000 + 12000 + 6000));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
                }
                UNIT_ASSERT(configGenerator.WaitPrice(12000 + 6000 + 12000 + 12000 + 6000));

            }

        }
    }

    Y_UNIT_TEST(EvolutionFullScenarioPack) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        configGenerator.SetNeedBackground(EServerBackgrounds::RentPricing | EServerBackgrounds::CarMarkers);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto tcc = MakeHolder<NDrive::TTelematicsClientContext>();
        tcc->SetCurrentPosition({ 37.587703, 55.733404 });
        tcc->SetOdometerKm(1000);
        tcc->TrySetEngineStarted(true);
        auto emulator = tmBuilder.BuildEmulator(car.IMEI, std::move(tcc));
        UNIT_ASSERT(configGenerator.WaitCar(car.Id));
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id));
        UNIT_ASSERT(configGenerator.WaitSensor(car.Id, "mileage", "1000"));
        {
            TString offerId;
            TInstant now = Now();
            TInstantGuard ig(now - TDuration::Hours(12));
            {
                THolder<TPackOffer> offer(new TPackOffer);
                offer->SetMileageLimit(70).SetPackPrice(45000).SetDuration(TDuration::Hours(11)).MutableRiding().SetPrice(700);
                offer->SetOverrunKm(800);
                offer->SetChargableAccounts({ "card", "bonus" });
                offer->SetObjectId(car.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(now + TDuration::Minutes(5));
                offer->SetDistancePushThreshold(65);
                offer->SetDurationPushThreshold(TDuration::Hours(10));
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TPackOfferReport(offer.Release(), nullptr)}));
            }
            {
                ig.Set(now - TDuration::Hours(12));
                TVector<TDBTag> tags;
                UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));

                UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
                UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));

                {
                    ig.Set(now - TDuration::Hours(11) - TDuration::Minutes(50));
                    for (ui32 i = 0; i <= 100; ++i) {
                        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(configGenerator);
                        NJson::TJsonValue userSessionReport = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                        INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                        INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
                        if (45000 == userSessionReport["segment"]["session"]["specials"]["total_price"].GetUInteger()) {
                            break;
                        } else {
                            UNIT_ASSERT(i < 100);
                        }
                        Sleep(TDuration::Seconds(1));
                    }
                }

                {
                    ig.Set(now - TDuration::Hours(1) - TDuration::Hours(5));
                    for (ui32 i = 0; i <= 100; ++i) {
                        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(configGenerator);
                        NJson::TJsonValue userSessionReport = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                        INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                        INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
                        if (45000 == userSessionReport["segment"]["session"]["specials"]["total_price"].GetUInteger()) {
                            break;
                        } else {
                            UNIT_ASSERT(i < 100);
                        }
                        Sleep(TDuration::Seconds(1));
                    }
                }
                tmBuilder.SetSensorDoubleValue(*emulator, CAN_ODOMETER_KM, 1025.0);
                tmBuilder.SetSensorDoubleValue(*emulator, CAN_ODOMETER_KM, 1050.0);
                tmBuilder.SetSensorDoubleValue(*emulator, CAN_ODOMETER_KM, 1075.0);
                tmBuilder.SetSensorDoubleValue(*emulator, CAN_ODOMETER_KM, 1100.0);
                tmBuilder.SetSensorDoubleValue(*emulator, CAN_ODOMETER_KM, 1120.0);

                {
                    for (ui32 i = 0; i <= 100; ++i) {
                        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(configGenerator);
                        NJson::TJsonValue userSessionReport = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                        INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                        INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
                        if (45000 + 800 * 50 == userSessionReport["segment"]["session"]["specials"]["total_price"].GetUInteger()) {
                            break;
                        } else {
                            UNIT_ASSERT(i < 100);
                        }
                        Sleep(TDuration::Seconds(1));
                    }
                }
                ig.Set(now - TDuration::Hours(2) + TDuration::Minutes(10));
                {
                    ui32 distanceThresholdPushSent = 0;
                    ui32 durationThresholdPushSent = 0;
                    for (ui32 i = 0; i <= 100; ++i) {
                        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(configGenerator);
                        NJson::TJsonValue userSessionReport = configGenerator.GetUserLastSession(USER_ID_DEFAULT);
                        INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                        NJson::TJsonValue currentOfferState = userSessionReport["sessions"][0]["segment"]["session"]["specials"]["current_offer_state"];
                        distanceThresholdPushSent = currentOfferState["distance_threshold_push_sent"].GetUIntegerRobust();
                        durationThresholdPushSent = currentOfferState["duration_threshold_push_sent"].GetUIntegerRobust();
                        if (distanceThresholdPushSent > 0 || durationThresholdPushSent > 0) {
                            break;
                        } else {
                            Sleep(TDuration::Seconds(1));
                        }
                    }
                    UNIT_ASSERT_VALUES_EQUAL(distanceThresholdPushSent, 0);
                    UNIT_ASSERT_VALUES_EQUAL(durationThresholdPushSent, 1);
                }
                ig.Set(now);
                {
                    for (ui32 i = 0; i <= 100; ++i) {
                        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(configGenerator);
                        NJson::TJsonValue userSessionReport = configGenerator.GetUserLastSession(USER_ID_DEFAULT);
                        INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                        INFO_LOG << userSessionReport["sessions"][0]["segment"]["session"]["specials"]["total_price"] << Endl;
                        if (45000 + 800 * 50 + 60 * 700 == userSessionReport["sessions"][0]["segment"]["session"]["specials"]["total_price"].GetUInteger()) {
                            break;
                        } else {
                            UNIT_ASSERT(i < 100);
                        }
                        Sleep(TDuration::Seconds(1));
                    }
                }
                const TDriveAPI& driveApi = *server->GetDriveAPI();
                {
                    ui32 distanceThresholdPushSent = 0;
                    ui32 durationThresholdPushSent = 0;
                    for (ui32 i = 0; i <= 100; ++i) {
                        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(configGenerator);
                        NJson::TJsonValue userSessionReport = configGenerator.GetUserLastSession(USER_ID_DEFAULT);
                        INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                        NJson::TJsonValue currentOfferState = userSessionReport["sessions"][0]["segment"]["session"]["specials"]["current_offer_state"];
                        distanceThresholdPushSent = currentOfferState["distance_threshold_push_sent"].GetUIntegerRobust();
                        durationThresholdPushSent = currentOfferState["duration_threshold_push_sent"].GetUIntegerRobust();
                        if (distanceThresholdPushSent > 0 || durationThresholdPushSent > 0) {
                            break;
                        } else {
                            Sleep(TDuration::Seconds(1));
                        }
                    }
                    UNIT_ASSERT_VALUES_EQUAL(distanceThresholdPushSent, 0);
                    UNIT_ASSERT_VALUES_EQUAL(durationThresholdPushSent, 1);
                }
                {
                    TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ID_DEFAULT, TUserPermissionsFeatures());
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RestoreEvolutionTagsByUser(*userPermissions, "old_state_reservation", tags, session));
                }
                {
                    UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(tags[0]->GetPerformer(), USER_ID_DEFAULT);
                    ig.Set(now - TDuration::Hours(2));
                    UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
                }
                {
                    auto session = driveApi.BuildTx<NSQL::ReadOnly>();
                    auto distancePushTag = driveApi.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, { "test_push" }, session);
                    UNIT_ASSERT(distancePushTag);
                    UNIT_ASSERT_VALUES_EQUAL(distancePushTag->size(), 0);
                    auto durationPushTag = driveApi.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, { "test_push_smile" }, session);
                    UNIT_ASSERT(durationPushTag);
                    UNIT_ASSERT_VALUES_EQUAL(durationPushTag->size(), 1);
                }
            }

        }
    }

    Y_UNIT_TEST(EvolutionCancelScenario) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        NDrive::NTest::TScript script(configGenerator, Now() - TDuration::Minutes(2));
        using namespace NDrive::NTest;
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(TGeoCoord(37.5675511, 55.7323499));
        script.Add<TCreateAndBookOffer>().SetOfferType("standart_offer");
        script.Add<TDrop>(TDuration::Minutes(1)).SetExpectOK(true);

        const auto sessionChecker = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
            UNIT_ASSERT(userSessionReport["segment"]["meta"]["finished"].IsBoolean());
            UNIT_ASSERT(userSessionReport["segment"]["meta"]["finished"].GetBoolean());
            UNIT_ASSERT(userSessionReport["segment"]["session"]["specials"]["is_cancelled"].IsBoolean());
            UNIT_ASSERT(userSessionReport["segment"]["session"]["specials"]["is_cancelled"].GetBoolean());
        };

        script.Add<TCheckCurrentSession>(sessionChecker);
        script.Add<TDrop>().SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(EvolutionCancelScenarioFail) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        NDrive::NTest::TScript script(configGenerator, Now() - TDuration::Minutes(2));
        using namespace NDrive::NTest;
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(TGeoCoord(10, 10));
        script.Add<TCreateAndBookOffer>().SetOfferType("standart_offer");
        script.Add<TCarEmulatorControl>().SetAction(TCarEmulatorControl::EAction::Drop);
        script.Add<TDrop>(TDuration::Minutes(1)).SetExpectOK(false);

        const auto sessionChecker = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
            UNIT_ASSERT(userSessionReport["segment"]["meta"]["finished"].IsBoolean());
            UNIT_ASSERT(!userSessionReport["segment"]["meta"]["finished"].GetBoolean());
            UNIT_ASSERT(userSessionReport["segment"]["session"]["specials"]["is_cancelled"].IsBoolean());
            UNIT_ASSERT(!userSessionReport["segment"]["session"]["specials"]["is_cancelled"].GetBoolean());
        };

        script.Add<TCheckCurrentSession>(TDuration::Minutes(1), sessionChecker);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(EvolutionNoImei) {
        TTestEnvironment env;
        env.Build();

        const auto& server = *env.GetServer();
        auto car = env.GetEnvironmentGenerator().CreateCar();
        {
            auto tx = server.GetDriveAPI()->BuildTx<NSQL::Writable>();
            auto object = server.GetDriveDatabase().GetCarManager().GetObject(car.Id);
            UNIT_ASSERT(object);
            object->SetIMEI(TString{});
            UNIT_ASSERT(server.GetDriveDatabase().GetCarManager().UpdateCar(*object, USER_ROOT_DEFAULT, tx));
            UNIT_ASSERT(tx.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto offerId = env->CreateOffer(car.Id, USER_ID_DEFAULT);
        UNIT_ASSERT(offerId);
        bool booked = env->BookOffer(offerId, USER_ID_DEFAULT);
        UNIT_ASSERT(booked);
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", USER_ID_DEFAULT, TDuration::Seconds(10), {}, {}, NJson::ReadJsonFastTree(R"(
            {
                "snapshot": {
                    "type": "device_snapshot"
                }
            }
        )")));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", USER_ID_DEFAULT, TDuration::Seconds(10), {}, {}, NJson::ReadJsonFastTree(R"(
            {
                "snapshot": {
                    "type": "device_snapshot",
                    "data": {
                        "sensors": [
                            {
                                "id": 2103,
                                "value": 42,
                                "timestamp": 1657259196
                            },
                            {
                                "id": 2107,
                                "value": 42,
                                "timestamp": 1657259196
                            }
                        ]
                    }
                }
            }
        )")));
        UNIT_ASSERT(env->EvolveTag("old_state_reservation", USER_ID_DEFAULT, TDuration::Seconds(10), {}, {}, NJson::ReadJsonFastTree(R"(
            {
                "snapshot": {
                    "type": "device_snapshot",
                    "data": {
                        "sensors": [
                            {
                                "id": 2103,
                                "value": 4242,
                                "timestamp": 1657259396
                            },
                            {
                                "id": 2107,
                                "value": 24,
                                "timestamp": 1657259196
                            }
                        ]
                    }
                }
            }
        )")));
        {
            const auto& manager = server.GetDriveDatabase().GetCompiledSessionManager();
            auto tx = manager.BuildTx<NSQL::ReadOnly>();
            auto ydbTx = NDrive::TEntitySession();
            auto optionalSessions = manager.Get<TFullCompiledRiding>({ offerId }, tx, ydbTx);
            UNIT_ASSERT(optionalSessions);
            UNIT_ASSERT_VALUES_EQUAL(optionalSessions->size(), 1);
            auto session = optionalSessions->at(0);
            auto diff = session.GetSnapshotsDiffPtr();
            UNIT_ASSERT(diff);
            UNIT_ASSERT(diff->HasStartFuelLevel());
            UNIT_ASSERT(diff->HasLastFuelLevel());
            UNIT_ASSERT(diff->HasStartMileage());
            UNIT_ASSERT(diff->HasLastMileage());
            UNIT_ASSERT(diff->HasMileage());
            UNIT_ASSERT_DOUBLES_EQUAL(diff->GetMileageRef(), 4200, 0.01);
            UNIT_ASSERT_VALUES_EQUAL(diff->GetStartFuelLevelRef(), 42);
            UNIT_ASSERT_VALUES_EQUAL(diff->GetLastFuelLevelRef(), 24);
        }
    }

    Y_UNIT_TEST(UserDebtPolicy) {
        InitGlobalLog2Console(TLOG_DEBUG);
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);
        NDrive::TTelematicsTestClient::TPtr client = emulator->GetClient();
        UNIT_ASSERT(configGenerator.WaitLocation(OBJECT_ID_DEFAULT));
        UNIT_ASSERT(configGenerator.WaitSensor(OBJECT_ID_DEFAULT, "mileage"));

        TString cardsReply = eGenerator.GetBillingMock().GetReply();
        const TString emptyCardsReply = "{\"status\": \"success\", \"bound_payment_methods\" : []}";

        TInstant now = Now();
        TInstantGuard ig(now - TDuration::Hours(12));
        {
            TString offerId;
            {
                auto offer = BuildOfferPtr(200, 100, 102400);
                offer->SetObjectId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetChargableAccounts({ "card", "bonus" });
                offer->SetDepositAmount(27182);
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
            }
            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            // Remove cards to fail deposit check.
            eGenerator.GetBillingMock().SetReply(emptyCardsReply);
            driveApi.GetBillingManager().WaitBillingCycle();
            // Run cycle to refresh debts cache
            driveApi.GetBillingManager().WaitBillingCycle();
            UNIT_ASSERT(!configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
            // Run cycle to refresh debts cache
            driveApi.GetBillingManager().WaitBillingCycle();
        }
        {
            TString offerId;
            {
                auto offer = BuildOfferPtr(200, 100, 102400);
                offer->SetChargableAccounts({ "card", "bonus" });
                offer->SetObjectId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetDepositAmount(0);
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
            }
            // Return cards back to book offer
            eGenerator.GetBillingMock().SetReply(cardsReply);
            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
            ig.Set(now - TDuration::Hours(11));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
        }
        // Remove cards to create debt.
        eGenerator.GetBillingMock().SetReply(emptyCardsReply);
        driveApi.GetBillingManager().WaitBillingCycle();
        // Run cycle to refresh debts cache
        driveApi.GetBillingManager().WaitBillingCycle();
        {
            TString offerId;
            {
                auto offer = BuildOfferPtr(200, 100, 102400);
                offer->SetChargableAccounts({ "card", "bonus" });
                offer->SetObjectId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetDepositAmount(0);
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
            }
            // Return cards back to book offer
            eGenerator.GetBillingMock().SetReply(cardsReply);
            UNIT_ASSERT(!configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
        }
    }
}
