#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

namespace NDrive::NTest {
    class TExternalChatBuffer {
        R_FIELD(TString, Provider);
        R_FIELD(TString, ChatId);
        R_FIELD(TInstant, ReadUntil);
        R_FIELD(TMaybe<bool>, Closed);
        R_FIELD(NJson::TJsonValue, Meta);
        R_FIELD(NJson::TJsonValue, Messages);
        R_FIELD(NJson::TJsonValue, Attachments);

    public:
        TExternalChatBuffer(const TString& provider, const TString& chatId, const NJson::TJsonValue meta, const TInstant start)
            : Provider(provider)
            , ChatId(chatId)
            , Meta(meta)
        {
            lastTimestamp = start;
            Messages = NJson::JSON_ARRAY;
            Attachments = NJson::JSON_ARRAY;
        }

        void AddMessage(const TString& text, ui32 type = 0, TInstant timestamp = TInstant::Zero(), ui32 messageId = 0, const TVector<TString>& attachmentIds = {}) {
            NJson::TJsonValue message = NJson::JSON_MAP;
            lastMessageId = messageId ? messageId : ++lastMessageId;
            message["id"] = ToString(lastMessageId);
            if (text) {
                message["text"] = text;
            }
            message["timestamp"] = timestamp ? timestamp.Seconds() : lastTimestamp.Seconds();
            lastTimestamp = lastTimestamp + TDuration::Minutes(1);
            message["type"] = type;
            if (attachmentIds.size()) {
                NJson::TJsonValue attachJson = NJson::JSON_ARRAY;
                for (auto&& a : attachmentIds) {
                    attachJson.AppendValue(std::move(a));
                }
                message["attachments_id"] = attachJson;
            }
            Messages.AppendValue(std::move(message));
        }

        void ClearMessages() {
            Messages = NJson::JSON_ARRAY;
        }

        NJson::TJsonValue MakeJson() {
            NJson::TJsonValue messagesJson = NJson::JSON_MAP;
            messagesJson["provider"] = Provider;
            messagesJson["provider_chat_id"] = ChatId;
            if (Messages.GetArray().size()) {
                messagesJson["messages"] = Messages;
            }
            messagesJson["provider_meta"] = Meta;
            if (ReadUntil) {
                messagesJson["read_messages_ts"] = ReadUntil.Seconds();
            }
            if (Closed.Defined()) {
                messagesJson["closed"] = *Closed ? 1 : 0;
            }
            return messagesJson;
        }

    private:
        ui32 lastMessageId = 0;
        TInstant lastTimestamp;
    };

    class TUpsertExternalChat: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TUpsertExternalChat> Registrator;

    private:
        R_FIELD(NJson::TJsonValue, Payload);

    public:
        using TBase::TBase;

        TUpsertExternalChat(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<api/staff/chats/external/add_messages>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: external_chat_message" << Endl;
            ss << "</api/staff/chats/external/add_messages>" << Endl;
            return ss.Str();
        }

    protected:

        void DoExecute(TRTContext& context) override {
            NJson::TJsonValue reply = SendRequest(context, "/api/staff/chats/external/add_messages", Payload);
            UNIT_ASSERT(GetExpectOK() == !reply.IsNull());
        }
    };

    class TForwardExternalMessages: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TForwardExternalMessages> Registrator;

    private:
        R_FIELD(TSet<TString>, MessageIds);
        R_FIELD(TString, NewChatType);
        R_FIELD(TString, Comment);
        R_FIELD(TString, TagName);
        R_FIELD(TString, TopicLink);

    public:
        using TBase::TBase;

        TForwardExternalMessages(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<api/staff/chats/external/forward>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: support_center_forward_external" << Endl;
            ss << "</api/staff/chats/external/forward>" << Endl;
            return ss.Str();
        }

    protected:
        void DoExecute(TRTContext& context) override {
            NJson::TJsonValue request = NJson::JSON_MAP;
            request["new_chat_type"] = NewChatType;
            if (Comment) {
                request["comment"] = Comment;
            }
            request["message_ids"] = NJson::JSON_ARRAY;
            auto messagesJson = context.GetConfigGenerator().GetChatMessages(context.GetUserId(), "support.test_test-rand_6838392", USER_ROOT_DEFAULT);
            for (auto&& message : messagesJson["messages"].GetArray()) {
                if (MessageIds.contains(message["external_id"].GetString())) {
                    request["message_ids"].AppendValue(message["id"].GetUInteger());
                }
            }
            request["user_id"] = context.GetUserId();
            request["chat_id"] = TopicLink;
            NJson::TJsonValue reply = SendRequest(context, "/api/staff/chats/external/forward", request);
            UNIT_ASSERT(GetExpectOK() == !reply.IsNull());
        }
    };

    TAPIAction::TFactory::TRegistrator<TUpsertExternalChat> TUpsertExternalChat::Registrator("TUpsertExternalChat");
    TAPIAction::TFactory::TRegistrator<TCheckChatUnreads> TCheckChatUnreads::Registrator("TCheckChatUnreads");
    TAPIAction::TFactory::TRegistrator<TSendMessage> TSendMessage::Registrator("TSendMessage");
    TAPIAction::TFactory::TRegistrator<TForwardExternalMessages> TForwardExternalMessages::Registrator("TForwardExternalMessages");

    Y_UNIT_TEST_SUITE(ExternalChats) {

        void CheckExternalChatTag(TRTContext& context, const TString& tagName, const TString& externalId, const TString& provider, const TString& meta, const TString& comment = "", const ui32 count = 1) {
            TVector<TDBTag> dbTags;
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreTags({ context.GetUserId() }, { tagName }, dbTags, session), "Could not restore tags");
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), count);
            if (count) {
                bool tagFound = false;
                for (auto&& tag : dbTags) {
                    tagFound = true;
                    auto tagImpl = tag.GetTagAs<TSupportChatTag>();
                    UNIT_ASSERT(tagImpl);
                    tagFound &= tagImpl->GetExternalChatInfo().GetId() == externalId;
                    tagFound &= tagImpl->GetExternalChatInfo().GetProvider() == provider;
                    if (meta) {
                        tagFound &= tagImpl->GetMeta().GetStringRobust() == meta;
                    }
                    if (comment) {
                        tagFound &= tagImpl->GetComment() == comment;
                    }
                    if (tagFound) {
                        break;
                    }
                }
                UNIT_ASSERT(tagFound);
            }
        }

        void RegisterExternalChatTag(NDrive::NTest::TScript& script) {
            NJson::TJsonValue descJson;
            descJson["type"] = "user_support_chat_tag";
            descJson["name"] = "support_tag_external";
            descJson["display_name"] = "External chat";
            descJson["comment"] = "";
            descJson["unique_policy"] = "no_unique";
            NJson::TJsonValue onClose = NJson::JSON_MAP;
            onClose["evolve_to_tag"] = "support_chat_lowpriority";
            onClose["move_to_node"] = "chat_completed_pre";
            descJson["meta"] = NJson::JSON_MAP;
            descJson["meta"].InsertValue("on_close_behaviour", onClose);
            script.Add<NDrive::NTest::TRegisterTag>(descJson);
        }

        Y_UNIT_TEST(CreateChat) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            NJson::TJsonValue metaJson = NJson::JSON_MAP;
            metaJson["chat_classification"] = "pshchphsh";

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("skulik-was-not-there", true, "active");
                if (!!userId) {
                    context.SetUserId(userId);
                }
            });

            script.Add<THasChat>().SetChatCount(0);
            script.Add<TCheckHasTag>().SetTagName("support_tag_external").SetExpectOK(false);

            auto chatBuffer = TExternalChatBuffer("test_test", "rand_1923821", metaJson, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("text 1", 0);

            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_1923821", "test_test", "{\"chat_classification\":\"pshchphsh\"}");
            });
            script.Add<THasChat>().SetChatCount(1);
            script.Add<TCheckChatMessages>().SetChatId("support.test_test-rand_1923821").SetMessageValues({{4, "text 1"}}).SetMessageCount(5);

            chatBuffer.ClearMessages();
            chatBuffer.AddMessage("last message", 0, ModelingNow() - TDuration::Hours(1));
            chatBuffer.AddMessage("a comment", 1);
            chatBuffer.AddMessage("text 3", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_1923821", "test_test", "{\"chat_classification\":\"pshchphsh\"}");
            });
            script.Add<THasChat>().SetChatCount(1);
            script.Add<TCheckChatMessages>().SetChatId("support.test_test-rand_1923821").SetMessageValues({{5, "a comment"}, {6, "text 3"}, {7, "last message"}}).SetMessageCount(8);

            metaJson = NJson::JSON_MAP;
            metaJson["chat_classification"] = "mmmmm";
            auto chatBuffer2 = TExternalChatBuffer("test_test", "rand2_654987", metaJson, Now() - TDuration::Days(1));
            chatBuffer2.AddMessage("this is the text", 0);

            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer2.MakeJson());
            script.Add<TCheckChatMessages>().SetMessageValues({{4, "this is the text"}}).SetChatId("support.test_test-rand2_654987").SetMessageCount(5);
            script.Add<THasChat>().SetChatCount(2);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand2_654987", "test_test", "{\"chat_classification\":\"mmmmm\"}", "", 2);
            });

            script.Execute();
        }

        Y_UNIT_TEST(MarkAsRead) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("skulik-was-not-there", true, "active");
                if (!!userId) {
                    context.SetUserId(userId);
                }
            });
            auto chatBuffer = TExternalChatBuffer("test_test", "rand_5288745", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("text 1", 0);
            chatBuffer.AddMessage("text 2", 0);
            chatBuffer.AddMessage("text 3", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();
            script.Add<THasChat>().SetChatCount(1);

            script.Add<TCheckChatUnreads>().SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                UNIT_ASSERT(response["chats"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(response["chats"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(response["chats"].GetArray()[0]["chat_id"], "support.test_test-rand_5288745");
                UNIT_ASSERT(response["chats"].GetArray()[0]["messages"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(response["chats"].GetArray()[0]["messages"].GetArray().size(), 7);
            });

            chatBuffer.ClearMessages();
            chatBuffer.SetReadUntil(ModelingNow());
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TCheckChatUnreads>().SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                UNIT_ASSERT(response["chats"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(response["chats"].GetArray().size(), 0);
            });

            script.Execute();
        }

        Y_UNIT_TEST(SaveCommentToTag) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("skulik-was-not-there", true, "active");
                if (!!userId) {
                    context.SetUserId(userId);
                }
            });
            auto chatBuffer = TExternalChatBuffer("test_test", "rand_88003352", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("chat info", 1);
            chatBuffer.AddMessage("this is the text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_88003352", "test_test", "", "chat info");
            });
            script.Add<TCheckChatMessages>().SetOperatorId(USER_ROOT_DEFAULT).SetMessageValues({{4, "chat info"}, {5, "this is the text"}}).SetChatId("support.test_test-rand_88003352").SetMessageCount(6);

            script.Execute();
        }

        Y_UNIT_TEST(MessageToDeferredChat) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("skulik-was-not-there", true, "active");
                if (!!userId) {
                    context.SetUserId(userId);
                }
            });

            auto chatBuffer = TExternalChatBuffer("test_test", "rand_684444411", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("this is the text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TPerformTag>().SetTagName("support_tag_external").SetActionUserId(USER_ROOT_DEFAULT);
            script.Add<TDeferChat>().SetTagName("support_tag_external").SetDeferUntil(TInstant::Now() + TDuration::Days(1));
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_684444411", "test_test", "", "", 0);
            });

            auto chatBuffer2 = TExternalChatBuffer("test_test", "rand_684444411", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer2.AddMessage("new text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer2.MakeJson());

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_684444411", "test_test", "", "");
            });
            script.Add<TCheckChatMessages>().SetOperatorId(USER_ROOT_DEFAULT).SetMessageValues({{4, "this is the text"}, {5, "new text"}}).SetChatId("support.test_test-rand_684444411").SetMessageCount(6);
            script.Add<TCheckHasTag>().SetTagName(TSupportChatTag::DeferredContainerName).SetExpectOK(false);

            script.Execute();
        }

        Y_UNIT_TEST(MessageFromOperator) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TSetScriptUser>().SetUserId(USER_ID_DEFAULT);
            auto chatBuffer = TExternalChatBuffer("test_test", "rand_8800212121", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("chat info", 1);
            chatBuffer.AddMessage("this is the text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());

            NJson::TJsonValue message = NJson::JSON_MAP;
            message["message"] = "operator message";
            message["is_external"] = true;
            message["external_chat_id"] = "rand_8800212121";

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto client = context.GetServer()->GetTaxiChatClient();
                UNIT_ASSERT(client);
                const TTestTaxiChatClient* testClient = dynamic_cast<const TTestTaxiChatClient*>(client);
                UNIT_ASSERT(testClient);
                NUtil::THttpReply reply;
                reply.SetCode(200);
                testClient->SetReply(reply);
            });
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TSendMessage>().SetChatId("support.test_test-rand_8800212121").SetRequestJson(message)
                .SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                    UNIT_ASSERT_VALUES_EQUAL(response["text"].GetString(), "operator message");
            }).SetMessageUserId(USER_ID_DEFAULT).SetUserId(USER_ROOT_DEFAULT);

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto messagesJson = context.GetConfigGenerator().GetChatMessages(context.GetUserId(), "support.test_test-rand_8800212121", USER_ROOT_DEFAULT);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"], "operator message");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["external_status"], "sent");
            });

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto client = context.GetServer()->GetTaxiChatClient();
                UNIT_ASSERT(client);
                const TTestTaxiChatClient* testClient = dynamic_cast<const TTestTaxiChatClient*>(client);
                UNIT_ASSERT(testClient);
                NUtil::THttpReply reply;
                reply.SetCode(500);
                testClient->SetReply(reply);
            });

            NJson::TJsonValue message2 = NJson::JSON_MAP;
            message2["message"] = "bad message";
            message2["is_external"] = true;
            message2["external_chat_id"] = "rand_8800212121";

            script.Add<TSendMessage>().SetChatId("support.test_test-rand_8800212121").SetRequestJson(message2)
                .SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                    UNIT_ASSERT_VALUES_EQUAL(response["error_details"]["special_info"]["text"].GetString(), "bad message");
                    UNIT_ASSERT(response["error_details"]["special_info"]["debug"].GetMap().size() > 0);
            }).SetUserId(USER_ROOT_DEFAULT).SetExpectOK(false);

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto messagesJson = context.GetConfigGenerator().GetChatMessages(context.GetUserId(), "support.test_test-rand_8800212121", USER_ROOT_DEFAULT);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"], "operator message");
            });

            script.Execute();
        }

        Y_UNIT_TEST(CloseChat) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TSetScriptUser>().SetUserId(USER_ROOT_DEFAULT);

            auto chatBuffer = TExternalChatBuffer("test_test", "rand_5365345883", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("this is the text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto client = context.GetServer()->GetTaxiChatClient();
                UNIT_ASSERT(client);
                const TTestTaxiChatClient* testClient = dynamic_cast<const TTestTaxiChatClient*>(client);
                UNIT_ASSERT(testClient);
                NUtil::THttpReply reply;
                reply.SetCode(200);
                testClient->SetReply(reply);
            });
            script.Add<TPerformTag>().SetTagName("support_tag_external").SetObjectId(USER_ROOT_DEFAULT).SetActionUserId(USER_ROOT_DEFAULT);
            script.Add<TCloseChat>().SetTagName("support_tag_external").SetUserId(USER_ROOT_DEFAULT).SetIsExternal(true);
            script.Add<TCheckChatMessages>().SetMessageValues({{4, "this is the text"}, {5, "Чат закрыт"}}).SetChatId("support.test_test-rand_5365345883").SetExpectedActionType("chat_closed").SetMessageCount(6);
            script.Add<TCheckHasTag>().SetTagName("support_chat_lowpriority").SetObjectId(USER_ROOT_DEFAULT).SetExpectOK(true);
            script.Add<TCheckHasTag>().SetTagName("support_tag_external").SetObjectId(USER_ROOT_DEFAULT).SetExpectOK(false);
            script.Add<TCheckTagPerformer>().SetTagName("support_chat_lowpriority").SetPerformer("");
            script.Execute();
        }

        Y_UNIT_TEST(CloseChatBadResponse) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TSetScriptUser>().SetUserId(USER_ROOT_DEFAULT);

            auto chatBuffer = TExternalChatBuffer("test_test", "rand_6878485211", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("this is the text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto client = context.GetServer()->GetTaxiChatClient();
                UNIT_ASSERT(client);
                const TTestTaxiChatClient* testClient = dynamic_cast<const TTestTaxiChatClient*>(client);
                UNIT_ASSERT(testClient);
                NUtil::THttpReply reply;
                reply.SetCode(500);
                testClient->SetReply(reply);
            });
            script.Add<TCloseChat>().SetTagName("support_tag_external").SetUserId(USER_ROOT_DEFAULT).SetIsExternal(true).SetExpectOK(false);

            script.Add<TCheckHasTag>().SetTagName("support_tag_external").SetObjectId(USER_ROOT_DEFAULT).SetExpectOK(true);
            script.Add<TCheckHasTag>().SetTagName("support_chat_lowpriority").SetObjectId(USER_ROOT_DEFAULT).SetExpectOK(false);
            script.Execute();
        }

        Y_UNIT_TEST(ForwardMessages) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("skulik-was-not-there", true, "active");
                if (!!userId) {
                    context.SetUserId(userId);
                }
            });

            auto chatBuffer = TExternalChatBuffer("test_test", "rand_6838392", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("text1", 0);
            chatBuffer.AddMessage("text2", 0);
            chatBuffer.AddMessage("text3", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto client = context.GetServer()->GetTaxiChatClient();
                UNIT_ASSERT(client);
                const TTestTaxiChatClient* testClient = dynamic_cast<const TTestTaxiChatClient*>(client);
                UNIT_ASSERT(testClient);
                NUtil::THttpReply reply;
                reply.SetCode(200);
                testClient->SetReply(reply);
            });

            script.Add<TForwardExternalMessages>().SetTopicLink("support.test_test-rand_6838392").SetNewChatType("ttttype").SetComment("a comment").SetMessageIds({"1", "3"});

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                context.GetServer()->GetChatRobot("support")->Refresh(Now());
                auto messagesJson = context.GetConfigGenerator().GetChatMessages(context.GetUserId(), "support.test_test-rand_6838392", USER_ROOT_DEFAULT);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"], "text1");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["external_status"], "forwarded");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["external_id"], "1");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["text"], "text2");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["external_status"], "sent");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["external_id"], "2");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"], "text3");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["external_status"], "forwarded");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["external_id"], "3");
            });

            script.Execute();
        }

        Y_UNIT_TEST(Attachments) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            NJson::TJsonValue metaJson = NJson::JSON_MAP;
            metaJson["chat_classification"] = "pshchphsh";

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("skulik-was-not-there", true, "active");
                if (!!userId) {
                    context.SetUserId(userId);
                }
            });

            auto chatBuffer = TExternalChatBuffer("test_test", "rand_196584321873", metaJson, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("text 1", 0, Now(), 0);

            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_196584321873", "test_test", "{\"chat_classification\":\"pshchphsh\"}");
            });

            chatBuffer = TExternalChatBuffer("test_test", "rand_196584321873", NJson::JSON_MAP, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("text 2", 0, Now(), 1, { "123456" });
            chatBuffer.AddMessage("", 0, Now(), 2, { "789000", "000123" });

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto client = context.GetServer()->GetTaxiChatClient();
                UNIT_ASSERT(client);
                const TTestTaxiChatClient* testClient = dynamic_cast<const TTestTaxiChatClient*>(client);
                UNIT_ASSERT(testClient);
                TMap<TString, ITaxiChatClient::TTaxiChatAttachment> attachments =
                {   {"000123", ITaxiChatClient::TTaxiChatAttachment().SetContent("11111111").SetContentType("image/jpeg").SetId("000123")},
                    {"123456", ITaxiChatClient::TTaxiChatAttachment().SetContent("22222222").SetContentType("image/jpeg").SetId("123456")},
                    {"789000", ITaxiChatClient::TTaxiChatAttachment().SetContent("33333333").SetContentType("video/mp4").SetId("789000")} };
                testClient->SetAttachmentsReply(attachments);
            });

            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                Sleep(TDuration::Seconds(3));
                context.GetServer()->GetChatRobot("support")->Refresh(Now());
                auto messagesJson = context.GetConfigGenerator().GetChatMessages(context.GetUserId(), "support.test_test-rand_196584321873", USER_ROOT_DEFAULT);
                INFO_LOG << "ExternalDebug: " << messagesJson.GetStringRobust();
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"], "text 1");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["text"], "text 2");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"], "123456");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["type"], "media_resources");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["external_id"], "1");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["text"], "000123,789000");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["type"], "media_resources");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["external_id"], "2");
                UNIT_ASSERT(messagesJson["messages"].GetArray()[6]["cached_images"].IsArray());
                UNIT_ASSERT(messagesJson["messages"].GetArray()[7]["cached_images"].IsArray());
                const TString image1 = TStringBuilder() << context.GetUserId() << "/000123";
                const TString image2 = TStringBuilder() << context.GetUserId() << "/123456";
                const TString image3 = TStringBuilder() << context.GetUserId() << "/789000";
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["cached_images"].GetArray()[0].GetString(), image2);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["cached_images"].GetArray()[0].GetString(), image1);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["cached_images"].GetArray()[1].GetString(), image3);
                UNIT_ASSERT(messagesJson["messages"].GetArray()[6]["content_types"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["content_types"].GetArray()[0].GetString(), "image/jpeg");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["content_types"].GetArray()[0].GetString(), "image/jpeg");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["content_types"].GetArray()[1].GetString(), "video/mp4");
            });
            script.Execute();
        }

        Y_UNIT_TEST(DeferChatWithReply) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TSetScriptUser>().SetUserId(USER_ROOT_DEFAULT);

            auto chatBuffer = TExternalChatBuffer("test_test", "rand_536534532332883", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("this is the text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto client = context.GetServer()->GetTaxiChatClient();
                UNIT_ASSERT(client);
                const TTestTaxiChatClient* testClient = dynamic_cast<const TTestTaxiChatClient*>(client);
                UNIT_ASSERT(testClient);
                NUtil::THttpReply reply;
                reply.SetCode(200);
                testClient->SetReply(reply);
            });
            script.Add<TPerformTag>().SetTagName("support_tag_external").SetObjectId(USER_ROOT_DEFAULT).SetActionUserId(USER_ROOT_DEFAULT);
            script.Add<TDeferChat>().SetMessage("some message").SetTagName("support_tag_external").SetUserId(USER_ROOT_DEFAULT);
            script.Add<TCheckHasTag>().SetTagName(TSupportChatTag::DeferredName).SetObjectId(USER_ROOT_DEFAULT).SetExpectOK(true);
            script.Add<TCheckHasTag>().SetTagName("support_tag_external").SetObjectId(USER_ROOT_DEFAULT).SetExpectOK(false);
            script.Add<TCheckHasTag>().SetTagName("support_chat_deferred").SetObjectId(USER_ROOT_DEFAULT).SetExpectOK(true);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto messagesJson = context.GetConfigGenerator().GetChatMessages(context.GetUserId(), "support.test_test-rand_536534532332883", USER_ROOT_DEFAULT);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"], "this is the text");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["external_status"], "sent");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["text"], "some message");
                UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["external_status"], "sent");
            });
            auto chatBuffer2 = TExternalChatBuffer("test_test", "rand_536534532332883", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer2.AddMessage("new text", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer2.MakeJson());

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_536534532332883", "test_test", "", "");
            });
            script.Add<TCheckChatMessages>().SetOperatorId(USER_ROOT_DEFAULT).SetMessageValues({{5, "some message"}, {6, "new text"}}).SetChatId("support.test_test-rand_536534532332883").SetMessageCount(7);
            script.Add<TCheckHasTag>().SetTagName("support_chat_deferred").SetExpectOK(false);

            script.Execute();
        }

        Y_UNIT_TEST(CloseFromHandler) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            RegisterExternalChatTag(script);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("skulik-was-not-there", true, "active");
                if (!!userId) {
                    context.SetUserId(userId);
                }
            });
            auto chatBuffer = TExternalChatBuffer("test_test", "rand_546756674", NJson::JSON_NULL, Now() - TDuration::Days(1));
            chatBuffer.AddMessage("text 1", 0);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());
            script.Add<NDrive::NTest::TDropCache>();
            script.Add<THasChat>().SetChatCount(1);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_546756674", "test_test", "", "", 1);
            });
            chatBuffer.ClearMessages();
            chatBuffer.SetClosed(true);
            script.Add<TUpsertExternalChat>().SetPayload(chatBuffer.MakeJson());

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                CheckExternalChatTag(context, "support_tag_external", "rand_546756674", "test_test", "", "", 0);
            });

            script.Execute();
        }
    }
}
