#include <drive/backend/ut/library/script.h>

#include <drive/backend/device_snapshot/manager.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NDrive::NTest {
    class TFixpointOfferChecker: public TAPIAction {
    public:
        using TChecker = std::function<void(const NJson::TJsonValue& response, TRTContext& context)>;

    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TFixpointOfferChecker> Registrator;
        TChecker Checker;
        TGeoCoord Source;
        TGeoCoord Destination;

    public:
        using TBase::TBase;

        TFixpointOfferChecker(const TInstant startInstant, const TChecker checker,
                              const TGeoCoord& source, const TGeoCoord& destination)
            : TBase(startInstant)
            , Checker(checker)
            , Source(source)
            , Destination(destination)
        {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<external/offers/fixpoint>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: external_offer" << Endl;
            ss << "    RTLineAPIName: drive_router" << Endl;
            ss << "</external/offers/fixpoint>" << Endl;
            return ss.Str();
        }

    protected:
        void DoExecute(TRTContext& context) override {
            const TString request =
                    TStringBuilder()
                    << "/external/offers/fixpoint?src=" << CGIEscapeRet(Source.ToString())
                    << "&dst=" << CGIEscapeRet(Destination.ToString());
            auto response = SendRequest(context, request);
            Checker(response, context);
        }
    };
    TAPIAction::TFactory::TRegistrator<TFixpointOfferChecker> TFixpointOfferChecker::Registrator("FixpointOfferChecker");

    bool CarHashUpdated(const TDevicesSnapshotManager& snapshotsManager,
                        const TGeoCoord& pos, const TString& carId) {
        TRect<TGeoCoord> rect(pos);
        rect.GrowDistance(5);

        auto ptr = snapshotsManager.GetGeoHash();
        UNIT_ASSERT_C(ptr, "Missing geo hash");

        bool found = false;
        auto actor = [&found, carId](const TDevicesSnapshotManager::THashedDevice& d) -> bool {
            if (d.CarId == carId) {
                found = true;
                return false;
            }
            return true;
        };

        ptr->FindObjects(rect, actor);
        return found;
    }

    Y_UNIT_TEST_SUITE(ExternalOffers) {
        Y_UNIT_TEST(Smoke) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set("external_offer.fetch_cars_distance", "1000");

            TEnvironmentGenerator::TCar testCar;
            const TGeoCoord testCarPos = TGeoCoord(37.413324, 55.851612);
            script.Add<TCommonChecker>([&testCar, testCarPos](NDrive::NTest::TRTContext& context) {
                auto car = context.GetEGenerator().CreateCar();
                context.SetCar(car);
                context.RelocateCar(testCarPos, car);
                testCar = car;
            });
            script.Add<TCreateCar>().SetPosition(TGeoCoord(37.456337, 55.830122));

            // Update GeoHash
            script.Add<TCommonChecker>([&testCar, testCarPos](NDrive::NTest::TRTContext& context) {
                const auto& snapshotManager = context.GetServer()->GetSnapshotsManager();
                UNIT_ASSERT(TValidationHelpers::WaitFor(
                        [&testCar, testCarPos, &snapshotManager]() {
                            return CarHashUpdated(snapshotManager, testCarPos, testCar.Id);
                        },
                        TDuration::Seconds(30)));
            });

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<TFixpointOfferChecker>(
                    [&testCar](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                        const TString respText = response.GetStringRobust();
                        UNIT_ASSERT_C(response["is_registred"].GetBoolean(), respText);
                        UNIT_ASSERT_EQUAL_C(response["offers"].GetArray().size(), 1, respText);
                        const auto offer = response.GetValueByPath("offers.[0]");
                        UNIT_ASSERT_C(offer, respText);
                        UNIT_ASSERT_C((*offer)["deeplink"].GetString().StartsWith(
                                              "yandexdrive://cars/" + testCar.Number),
                                      offer->GetStringRobust());
                    },
                    TGeoCoord(37.416966, 55.852356), TGeoCoord(37.547163, 55.781083));
            UNIT_ASSERT(script.Execute());
        }
    }
}  // namespace NDrive::NTest
