#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>

#include <drive/backend/background/car_markers/config.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/account_tags.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tag.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>


Y_UNIT_TEST_SUITE(FixPoint) {
    using namespace NDrive::NTest;
    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateArea>().SetId("test-deny-1").SetCoords(coordsDenyArea).SetAreaTags({"deny_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TSleepAction>().SetWaitingDuration(TDuration::Seconds(10));
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TRelocateCar>().SetPosition(TGeoCoord(37.57248636458738, 55.73354854993565));
        const auto sessionCheckerDenyDropCar = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json["notification"] << Endl;
            if (json.Has("notification") && json["notification"].GetStringSafe() == "ALLOW_PARKING") {
                return true;
            }
            return false;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerDenyDropCar).SetUserId(USER_ID_DEFAULT);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto calcer = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer["pack_price"] << Endl;
            UNIT_ASSERT(jsonOffer["pack_price"].GetUIntegerSafe() > 100);
            return jsonOffer["pack_price"].GetUIntegerSafe();
        };
        script.Add<TCheckPrice>(TDuration::Minutes(10), calcer).SetPrecision(100);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            const NJson::TJsonValue& billInfo = json["segment"]["session"]["specials"]["bill"];
            for(const auto& record : billInfo.GetArray()) {
                UNIT_ASSERT_UNEQUAL(record["type"].GetString(), TBillRecord::CashbackType);
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(SimpleWithParking) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TSetSetting>().Set("offers.switching.enabled", ToString(true));
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TCreateAndSwitchOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TPark>(TDuration::Zero());
        script.Add<TRide>(TDuration::Minutes(1));
        script.Add<TDrop>(TDuration::Minutes(2)).SetCarPosition(to);
        const auto calcer = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            UNIT_ASSERT(jsonOffer["pack_price"].GetUIntegerSafe() + jsonOffer["prices"]["parking"].GetUIntegerSafe() > 100);
            return jsonOffer["pack_price"].GetUIntegerSafe() + jsonOffer["prices"]["parking"].GetUIntegerSafe();
        };
        script.Add<TCheckPrice>(TDuration::Minutes(10), calcer).SetPrecision(100);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(SimpleWithOvertimeAndOverrun) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TSetMileage>().SetValue(10);
        script.Add<TSetSetting>().Set("offers.switching.enabled", ToString(true));
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TCreateAndSwitchOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TSetMileage>().SetValue(10000);
        script.AddMove<TDrop>(TDuration::Minutes(2000)).SetCarPosition(to);
        const auto calcer = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            const double overrunPrice = ((10000 - 10) - jsonOffer["mileage_limit"].GetUIntegerSafe()) * (jsonOffer["rerun_price_km"].GetUIntegerSafe());
            const double overtimePrice = 1.0 * (2000 * 60 - jsonOffer["duration"].GetUIntegerSafe()) / 60 * (jsonOffer["prices"]["riding"].GetUIntegerSafe());
            const double packPrice = jsonOffer["pack_price"].GetUIntegerSafe();
            INFO_LOG << packPrice + overrunPrice + overtimePrice << Endl;
            return jsonOffer["pack_price"].GetUIntegerSafe() + overrunPrice + overtimePrice;
        };
        script.Add<TCheckPrice>(calcer).SetPrecision(100);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(IncorrectPoint) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        TGeoCoord incorrectTo(37.6675511, 55.7323499);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(incorrectTo);

        const auto calcer = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            UNIT_ASSERT(jsonOffer["prices"]["riding"].GetUIntegerSafe() > 100);
            return jsonOffer["prices"]["riding"].GetUIntegerSafe() * 1.1 * 1;
        };
        script.Add<TCheckPrice>(TDuration::Minutes(10), calcer).SetPrecision(100);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(OfferTagActivation) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_additional").SetUserPosition(from).SetExpectOK(false);
        THolder<TTemporaryActionTag> tag(new TTemporaryActionTag("additional_offer_tag"));
        tag->SetAttempts(2);
        script.Add<TAddUserTag>(tag.Release(), USER_ID_DEFAULT);
        script.Add<TDropCache>();
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_additional").SetUserPosition(from).SetExpectOK(true);
        script.Add<TDrop>();
        script.Add<TDropCache>();
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_additional").SetUserPosition(from).SetExpectOK(true);
        script.Add<TDrop>();
        script.Add<TDropCache>();
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_additional").SetUserPosition(from).SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(OfferAccountTagActivation) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);

        TString accountName = "limited";
        const TString offerName = "fixpoint_additional_for_wallet";
        script.Add<TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<TSetSetting>().Set("billing.filter_accounts", "true");
        script.Add<TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<TCreateAccount>(accountName);
        script.Add<TDropCache>();
        script.Add<TLinkAccount>(accountName).SetUserId(USER_ID_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TCommonChecker>([&accountName](TRTContext& context) {
            auto session = context.GetDriveAPI().GetBillingManager().BuildSession(true);
            auto accounts = context.GetDriveAPI().GetBillingManager().GetAccountsManager().GetUserAccounts(USER_ID_DEFAULT, session);
            UNIT_ASSERT(accounts);
            auto accountId = 0;
            for (const auto& acc : *accounts) {
                if (acc->GetUniqueName() == accountName) {
                    accountId = acc->GetId();
                }
            }
            UNIT_ASSERT_EQUAL(accountId, 1);
        });

        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName(offerName).SetAccountName(accountName).SetUserPosition(from).SetExpectOK(false);

        TString tagName = "additional_offer_account_tag";
        THolder<TAccountTemporaryActionTag> tag(new TAccountTemporaryActionTag("additional_offer_account_tag"));
        tag->SetAttempts(2);
        script.Add<TAddTag>(tag.Release()).SetObjectId("1").SetEntityType(NEntityTagsManager::EEntityType::Account);
        script.Add<TCheckHasTag>().SetTagName(tagName).SetObjectId("1").SetEntityType(NEntityTagsManager::EEntityType::Account);
        script.Add<TDropCache>();
        script.Add<TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName(offerName).SetAccountName(accountName).SetUserPosition(from).SetExpectOK(true);
        script.Add<TDrop>();
        script.Add<TDropCache>();
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName(offerName).SetAccountName("card").SetUserPosition(from).SetExpectOK(false);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName(offerName).SetAccountName(accountName).SetUserPosition(from).SetExpectOK(true);
        script.Add<TDrop>();
        script.Add<TDropCache>();
        script.Add<TSleepAction>().SetWaitingDuration(TDuration::Seconds(2));
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName(offerName).SetAccountName(accountName).SetUserPosition(from).SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(PromoCodeActivation) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);

        THolder<TTemporaryActionTag> tag(new TTemporaryActionTag("additional_offer_tag"));
        tag->SetAttempts(1);
        tag->SetPromoCode(true);
        script.Add<TAddUserTag>(tag.Release(), USER_ID_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TCommonChecker>([](TRTContext& context) {
            auto additionalActions = context.GetDriveAPI().GetRolesManager()->GetUserAdditionalActions(USER_ID_DEFAULT, context.GetDriveAPI().GetTagsManager(), true);
            UNIT_ASSERT(additionalActions);
            UNIT_ASSERT_VALUES_EQUAL(additionalActions->end() - additionalActions->begin(), 1);
            UNIT_ASSERT(additionalActions->begin()->GetAction()->GetSourceContext().IsPromoCode());
        });

        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_additional").SetUserPosition(from).SetExpectOK(true);
        script.Add<TDrop>();
        script.Add<TDropCache>();

        script.Add<TCommonChecker>([](TRTContext& context) {
            auto additionalActions = context.GetDriveAPI().GetRolesManager()->GetUserAdditionalActions(USER_ID_DEFAULT, context.GetDriveAPI().GetTagsManager(), true);
            UNIT_ASSERT(additionalActions);
            UNIT_ASSERT_EQUAL(additionalActions->end(), additionalActions->begin());
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(PotentialDiscount) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);

        TInstant start = ModelingNow();
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);

        THolder<TTemporaryActionTag> tag(new TTemporaryActionTag("additional_offer_tag"));
        tag->SetAttempts(1);
        tag->SetSince(start + TDuration::Days(1));
        script.Add<TAddUserTag>(tag.Release(), USER_ID_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TCommonChecker>([](TRTContext& context) {
            {
                auto additionalActions = context.GetDriveAPI().GetRolesManager()->GetUserAdditionalActions(USER_ID_DEFAULT, context.GetDriveAPI().GetTagsManager(), true);
                UNIT_ASSERT(additionalActions);
                UNIT_ASSERT_VALUES_EQUAL(additionalActions->end() - additionalActions->begin(), 1);
            }
            {
                auto additionalActions = context.GetDriveAPI().GetRolesManager()->GetUserAdditionalActions(USER_ID_DEFAULT, context.GetDriveAPI().GetTagsManager(), false);
                UNIT_ASSERT(additionalActions);
                UNIT_ASSERT_EQUAL(additionalActions->end(), additionalActions->begin());
            }
        });

        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_additional").SetUserPosition(from).SetExpectOK(false);
        script.Add<TCreateAndBookOffer>(TDuration::Days(2)).SetUserDestination(to).SetOfferName("fixpoint_additional").SetUserPosition(from).SetExpectOK(true);
        script.Add<TDrop>();
        script.Add<TDropCache>();

        script.Add<TCommonChecker>([](TRTContext& context) {
            auto additionalActions = context.GetDriveAPI().GetRolesManager()->GetUserAdditionalActions(USER_ID_DEFAULT, context.GetDriveAPI().GetTagsManager(), true);
            UNIT_ASSERT(additionalActions);
            UNIT_ASSERT_EQUAL(additionalActions->end(), additionalActions->begin());
        });

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixOnlySimple) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.62643984041714, 55.740277103506685);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(true);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);

        THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
        TAreaDropPolicyBuilder odp;
        odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("standart")).SetFee(10000).SetPolicy(EDropOfferPolicy::FeesFix);
        doTag->MutableDropPolicies().emplace_back(odp);

        script.Add<TAddTag>(TDuration::Zero(), doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        script.Add<TDropCache>();
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto calcer = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            return jsonOffer["prices"]["riding"].GetUIntegerSafe() + 10000;
        };
        script.Add<TCheckPrice>(TDuration::Minutes(10), calcer).SetPrecision(100);

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TDrop>(TDuration::Minutes(11)).SetCarPosition(to);
        const auto calcer1 = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer["pack_price"] << Endl;
            UNIT_ASSERT(jsonOffer["pack_price"].GetUIntegerSafe() > 100);
            return jsonOffer["pack_price"].GetUIntegerSafe();
        };
        script.Add<TCheckPrice>(TDuration::Minutes(20), calcer1).SetPrecision(100);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixCustomArea) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.62643984041714, 55.740277103506685);
        TGeoCoord destination(36.971014582760446, 55.553576360594036);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsAreaWithCustom = "36.96899756158125 55.562454627281234 36.963075244076364 55.55746845153333 36.961830699093454 55.55381962762004 36.965092265255564 55.551703153837984 36.97182997430097 55.551435545553005 36.982344233639346 55.55581436018857 36.97886809075605 55.56028997538196 36.96899756158125 55.562454627281234";
        const TString coordsAreaCustom = "36.96916922295819 55.55827114713122 36.96934088433514 55.556446815136205 36.97401865685711 55.55681168834037 36.97380408013593 55.558611679682365 36.96916922295819 55.55827114713122";
        script.Add<TBuildEnv>().SetNeedTelematics(true);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateArea>().SetId("rebuild_fix_zone").SetCoords(coordsAreaWithCustom).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TCreateArea>().SetId("fix_zone_allow_custom").SetCoords(coordsAreaCustom).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();
        script.Add<TCreateAndBookOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(destination).SetUserPosition(from).SetExpectOK(false);

        THolder<TFixPointCorrectionTag> doTag(new TFixPointCorrectionTag(TFixPointCorrectionTag::TypeName));
        doTag->SetLinkPublicArea("fix_zone_allow_custom");
        doTag->SetLinkArea("fix_zone_allow_custom");

        script.Add<TAddTag>(TDuration::Zero(), doTag.Release()).SetObjectId("rebuild_fix_zone").SetEntityType(NEntityTagsManager::EEntityType::Area);

        script.Add<TCreateAndBookOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(destination).SetUserPosition(from).SetExpectOK(true);

        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            TVector<TGeoCoord> gc;
            if (!TGeoCoord::DeserializeVector(json["segment"]["session"]["specials"]["current_offer"]["finish_area"].GetString(), gc)) {
                return false;
            }
            return gc.size() == 5
                && gc.front().GetLengthTo(gc.back()) == 0
                && gc[0].GetLengthTo(TGeoCoord(36.96916922295819, 55.55827114713122)) < 1e-3
                && gc[1].GetLengthTo(TGeoCoord(36.96934088433514, 55.556446815136205)) < 1e-3
                && gc[2].GetLengthTo(TGeoCoord(36.97401865685711, 55.55681168834037)) < 1e-3
                && gc[3].GetLengthTo(TGeoCoord(36.97380408013593, 55.558611679682365)) < 1e-3;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixOnlyUserAlerts) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.62643984041714, 55.740277103506685);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(true);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
        TAreaDropPolicyBuilder odp;
        odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("standart")).SetFee(10000).SetPolicy(EDropOfferPolicy::FeesFix).SetAlertType("test_alert").SetAlertsCount(1).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
        doTag->MutableDropPolicies().emplace_back(odp);

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");
        script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        script.Add<TDropCache>();

        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(2)).SetCheckStatus(true).SetCarPosition(from);
        const auto calcer0 = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            return jsonOffer["prices"]["riding"].GetUIntegerSafe() * 2;
        };
        script.Add<TCheckPrice>(TDuration::Minutes(10), calcer0).SetPrecision(100);

        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TDrop>(TDuration::Minutes(11)).SetCheckStatus(true).SetCarPosition(to).SetAlertLandingId("landing_alert1");
        const auto calcer = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            return jsonOffer["prices"]["riding"].GetUIntegerSafe();
        };
        script.Add<TCheckPrice>(TDuration::Minutes(20), calcer).SetPrecision(100);

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TDrop>(TDuration::Minutes(21)).SetCheckStatus(true).SetCarPosition(to).SetAlertLandingId("landing_alert2");
        const auto calcer1 = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            return jsonOffer["prices"]["riding"].GetUIntegerSafe() + 10000;
        };
        script.Add<TCheckPrice>(TDuration::Minutes(30), calcer1).SetPrecision(100);

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TDrop>(TDuration::Minutes(31)).SetCheckStatus(true).SetCarPosition(to);
        const auto calcer2 = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            return jsonOffer["prices"]["riding"].GetUIntegerSafe() + 10000;
        };
        script.Add<TCheckPrice>(TDuration::Minutes(40), calcer2).SetPrecision(100);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixOnlyUserAlertsForce) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.62643984041714, 55.740277103506685);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
        TAreaDropPolicyBuilder odp;
        odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("standart")).SetFee(10000).SetPolicy(EDropOfferPolicy::FeesFix).SetAlertType("test_alert").SetAlertsCount(1).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
        doTag->MutableDropPolicies().emplace_back(odp);

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");
        script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        script.Add<TDropCache>();

        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetExpectOK(false);
        script.Add<TDrop>(TDuration::Minutes(1)).SetEvolutionMode(EEvolutionMode::Force).SetCarPosition(to);
        const auto calcer0 = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer << Endl;
            return jsonOffer["prices"]["riding"].GetUIntegerSafe() + 10000;
        };
        script.Add<TCheckPrice>(TDuration::Minutes(10), calcer0).SetPrecision(100);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixOnlyUserAlertsFeesMax) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        const TGeoCoord internalFrom(37.58610478125119, 55.735918953797125);
        const TGeoCoord potentiallyInternalFrom(37.58665731630826, 55.7359613270497);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");

        double priceBase = 0;
        const auto checkerFixPriceBase = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceBase = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(priceBase > 1e-2);
        };
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceBase).SetUserPosition(from);

        double priceBasePotentiallyInternal = 0;
        const auto checkerFixPriceBasePotentially = [&priceBasePotentiallyInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceBasePotentiallyInternal = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(priceBasePotentiallyInternal > 1e-2);
        };
        script.Add<TRelocateCar>().SetPosition(potentiallyInternalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceBasePotentially).SetUserPosition(potentiallyInternalFrom);

        double priceBaseInternal = 0;
        const auto checkerFixPriceBaseOneZone = [&priceBaseInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceBaseInternal = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(priceBaseInternal > 1e-2);
        };
        script.Add<TRelocateCar>().SetPosition(internalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceBaseOneZone).SetUserPosition(internalFrom);

        THolder<TAreaPotentialTag> apTag(new TAreaPotentialTag(TAreaPotentialTag::TypeName));
        apTag->MutablePotentialSchemeTestOnly().MutablePoints().emplace_back(TLinearScheme::TKVPoint(1000, 100));

        script.Add<TAddTag>(apTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        script.Add<TDropCache>();

        const auto checkerFixPricePotentialOneZone = [&priceBaseInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            INFO_LOG << priceBaseInternal << " / " << offerJson["pack_price"].GetUIntegerSafe() << Endl;
            UNIT_ASSERT(Abs(priceBaseInternal / offerJson["pack_price"].GetUIntegerSafe() - 1) < 7e-2);
        };
        script.Add<TRelocateCar>().SetPosition(internalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPricePotentialOneZone).SetUserPosition(internalFrom);

        const auto checkerFixPricePotential = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            INFO_LOG << priceBase << " / " << offerJson["pack_price"].GetUIntegerSafe() << Endl;
            UNIT_ASSERT(Abs(priceBase / offerJson["pack_price"].GetUIntegerSafe() - 0.5) < 7e-2);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPricePotential).SetUserPosition(from);

        const auto checkerFixPriceMaxOneZonePotentialNear = [&priceBasePotentiallyInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << " / " << priceBasePotentiallyInternal << Endl;
            UNIT_ASSERT(Abs(priceBasePotentiallyInternal / offerJson["pack_price"].GetUIntegerSafe() - 1) < 7e-2);
        };
        script.Add<TRelocateCar>().SetPosition(potentiallyInternalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMaxOneZonePotentialNear).SetUserPosition(potentiallyInternalFrom);

        THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
        TAreaDropPolicyBuilder odp;
        odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000000).SetPolicy(EDropOfferPolicy::FeesMax).SetAlertType("test_alert").SetAlertsCount(0).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
        doTag->MutableDropPolicies().emplace_back(odp);

        script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        script.Add<TDropCache>();

        const auto checkerFixPriceMax = [](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_VALUES_EQUAL(offerJson["pack_price"].GetUIntegerSafe(), 10000000);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMax).SetUserPosition(from);

        const auto checkerFixPriceMaxOneZone = [&priceBaseInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << " / " << priceBaseInternal << Endl;
            UNIT_ASSERT(Abs(priceBaseInternal / offerJson["pack_price"].GetUIntegerSafe() - 1) < 7e-2);
        };
        script.Add<TRelocateCar>().SetPosition(internalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMaxOneZone).SetUserPosition(internalFrom);

        const auto checkerFixPriceMaxOneZonePotentially = [&priceBasePotentiallyInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << " / " << priceBasePotentiallyInternal << Endl;
            UNIT_ASSERT(Abs(priceBasePotentiallyInternal / offerJson["pack_price"].GetUIntegerSafe() - 1) < 7e-2);
        };
        script.Add<TRelocateCar>().SetPosition(potentiallyInternalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMaxOneZonePotentially).SetUserPosition(potentiallyInternalFrom);

        script.Add<TRelocateCar>().SetPosition(potentiallyInternalFrom);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 400;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto sessionChecker1 = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 10000000;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker1);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(to);
        script.Add<TDrop>().SetCarPosition(to);
        const auto sessionChecker0 = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 0;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker0);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixOnlyUserAlertsFeesMaxWithAlert) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        const TGeoCoord internalFrom(37.58610478125119, 55.735918953797125);
        const TGeoCoord potentiallyInternalFrom(37.58665731630826, 55.7359613270497);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");

        THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
        TAreaDropPolicyBuilder odp;
        odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000000).SetPolicy(EDropOfferPolicy::FeesMax).SetAlertType("test_alert").SetAlertsCount(1).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
        doTag->MutableDropPolicies().emplace_back(odp);

        script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        script.Add<TDropCache>();

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetUserPosition(from);

        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(to).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixOnlyUserAlertsFeesMaxDelta) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        const TGeoCoord internalFrom(37.58610478125119, 55.735918953797125);
        const TGeoCoord potentiallyInternalFrom(37.58665731630826, 55.7359613270497);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);

        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);

        const TString coordsStartArea = "37.624336988549466 55.741366564563656 37.62461593828702 55.739756572431695 37.62843540392423 55.739647623186066 37.62839248858 55.74090657361817 37.624336988549466 55.741366564563656";
        script.Add<TCreateArea>().SetId("test-start-1").SetCoords(coordsStartArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");

        double priceBase = 0;
        const auto checkerFixPriceBase = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceBase = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(priceBase > 1e-2);
        };
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceBase).SetUserPosition(from);

        {
            THolder<TAreaPotentialTag> apTag(new TAreaPotentialTag(TAreaPotentialTag::TypeName));
            apTag->MutablePotentialSchemeTestOnly().MutablePoints().emplace_back(TLinearScheme::TKVPoint(1000, 100));
            script.Add<TAddTag>(apTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }

        {
            THolder<TAreaPotentialTag> apTag(new TAreaPotentialTag(TAreaPotentialTag::TypeName));
            apTag->MutablePotentialSchemeTestOnly().MutablePoints().emplace_back(TLinearScheme::TKVPoint(1000, 50));
            script.Add<TAddTag>(apTag.Release()).SetObjectId("test-start-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }
        script.Add<TDropCache>();

        const auto checkerFixPricePotential = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            INFO_LOG << priceBase << " / " << offerJson["pack_price"].GetUIntegerSafe() << Endl;
            UNIT_ASSERT(Abs(priceBase / offerJson["pack_price"].GetUIntegerSafe() - 1.0 / 1.5) < 7e-2);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPricePotential).SetUserPosition(from);

        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            TAreaDropPolicyBuilder odp;
            odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(30000).SetPolicy(EDropOfferPolicy::FeesMinute).SetAlertType("test_alert").SetAlertsCount(0).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
            doTag->MutableDropPolicies().emplace_back(odp);
            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }
        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            TAreaDropPolicyBuilder odp;
            odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000).SetPolicy(EDropOfferPolicy::FeesMinute).SetAlertType("test_alert").SetAlertsCount(0).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
            doTag->MutableDropPolicies().emplace_back(odp);
            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-start-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }
        script.Add<TDropCache>();

        const auto checkerFixPriceMax = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT(Abs(priceBase / offerJson["pack_price"].GetUIntegerSafe() - 10000.0 / (1.5 * (10000 + 20000))) < 7e-2);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMax).SetUserPosition(from);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ZoneFixOnlyUserAlertsFeesMaxIncorrectPoint) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        const TGeoCoord internalFrom(37.58610478125119, 55.735918953797125);
        const TGeoCoord potentiallyInternalFrom(37.58665731630826, 55.7359613270497);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        const TGeoCoord toIncorrectPoint(37.579159700615804, 55.73928097545437);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");

        THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
        TAreaDropPolicyBuilder odp;
        odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000000).SetPolicy(EDropOfferPolicy::FeesMax).SetAlertType("test_alert").SetAlertsCount(0).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
        doTag->MutableDropPolicies().emplace_back(odp);

        script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        script.Add<TDropCache>();

        const auto checkerFixPriceMax = [](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_VALUES_EQUAL(offerJson["pack_price"].GetUIntegerSafe(), 10000000);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMax).SetUserPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(toIncorrectPoint);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            const NJson::TJsonValue& billInfo = json["segment"]["session"]["specials"]["bill"];
            if (billInfo[0]["type"] != "old_state_riding") {
                return false;
            }
            if (billInfo[2]["type"] != "fee_drop_zone_max") {
                return false;
            }
            if (billInfo[2]["cost"].GetUIntegerSafe() + billInfo[0]["cost"].GetUIntegerSafe() != 10000000) {
                return false;
            }
            if (billInfo[3]["type"] != "total") {
                return false;
            }
            if (billInfo[3]["cost"].GetUIntegerSafe() != 10000000) {
                return false;
            }
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 10000000;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FixAreaCrossMinFeeZone) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        const TGeoCoord toIncorrectInternalPoint(37.57951748241974, 55.7389913690975);
        const TGeoCoord toIncorrectNearPoint(37.580178940040845, 55.739453479612344);
        const TGeoCoord toIncorrectFarPoint(37.58167024825437, 55.74029178035889);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            TAreaDropPolicyBuilder odp;
            odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000000).SetPolicy(EDropOfferPolicy::FeesMax).SetAlertsCount(0);
            doTag->MutableDropPolicies().emplace_back(odp);

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
            script.Add<TDropCache>();
        }

        const auto checkerFixPriceBaseIncorrectNear = [](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_VALUES_EQUAL(offerJson["pack_price"].GetUIntegerSafe(), 10000000);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(toIncorrectNearPoint).SetChecker(checkerFixPriceBaseIncorrectNear).SetUserPosition(from);

        ui32 basePrice = 0;
        const auto checkerFixPriceBase = [&basePrice](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            basePrice = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT_C(basePrice < 250000 && basePrice > 100, TStringBuilder() << basePrice << Endl);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(toIncorrectFarPoint).SetChecker(checkerFixPriceBase).SetUserPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(toIncorrectFarPoint).SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(toIncorrectInternalPoint);
        const auto sessionChecker1 = [&basePrice](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (basePrice == 0) {
                return false;
            }
            return Abs(1.0 * json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() / basePrice - 1) < 7e-2;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker1);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FlowControlPriceMinuteCorrection) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        const TGeoCoord internalFrom(37.58610478125119, 55.735918953797125);
        const TGeoCoord potentiallyInternalFrom(37.58665731630826, 55.7359613270497);

        const TString coordsStartArea = "37.62543984041714 55.739277103506685 37.62543984041714 55.741277103506685 37.62743984041714 55.741277103506685 37.62743984041714 55.739277103506685 37.62543984041714 55.739277103506685";

        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 400;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        double priceBase = 0;
        const auto checkerFixPriceBase = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceBase = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(priceBase > 1e-2);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceBase).SetUserPosition(from);

        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            TAreaDropPolicyBuilder odp;
            odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(5000).SetPolicy(EDropOfferPolicy::FeesMinute).SetAlertType("test_alert").SetAlertsCount(0).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
            doTag->MutableDropPolicies().emplace_back(odp);

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }

        {
            THolder<TDestinationDetectorTag> doTag(new TDestinationDetectorTag(TDestinationDetectorTag::TypeName));
            doTag->SetFixValue(100000);
            doTag->SetStartZoneAttributesFilter(TTagsFilter::BuildFromString("-incorrect_start_zone"));

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }
        script.Add<TDropCache>();

        const auto checkerFixPriceMax = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_C(Abs(priceBase / offerJson["pack_price"].GetUIntegerSafe() - 100.0 / 150) < 7e-2, TStringBuilder() << priceBase << Endl);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMax).SetUserPosition(from);

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto sessionChecker1 = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 5400;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker1);
        script.Add<TCreateArea>().SetId("test-fix-start-1").SetCoords(coordsStartArea).SetAreaTags({"incorrect_start_zone"}).SetActionUserId(USER_ROOT_DEFAULT);
        const auto baseChecker = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_C(Abs(1 - 1.0 * priceBase / offerJson["pack_price"].GetUIntegerSafe()) < 7e-2, TStringBuilder() << priceBase << Endl);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(baseChecker).SetUserPosition(from);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FlowControlPriceMinuteCorrectionInternal) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        const TGeoCoord internalFrom(37.58610478125119, 55.735918953797125);
        const TGeoCoord potentiallyInternalFrom(37.58665731630826, 55.7359613270497);

        const TString coordsStartArea = "37.62543984041714 55.739277103506685 37.62543984041714 55.741277103506685 37.62743984041714 55.741277103506685 37.62743984041714 55.739277103506685 37.62543984041714 55.739277103506685";

        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        NDrive::NTest::TScript script(configGenerator);
        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        script.Add<TAddLanding>("landing_alert1", "{\"text\": \"111\"}");
        script.Add<TAddLanding>("landing_alert2", "{\"text\": \"222\"}");

        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 400;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        double priceBase = 0;
        const auto checkerFixPriceBase = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceBase = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(priceBase > 1e-2);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceBase).SetUserPosition(from);

        double priceBaseInternal = 0;
        const auto checkerFixPriceBaseInternal = [&priceBaseInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceBaseInternal = offerJson["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(priceBaseInternal > 1e-2);
        };
        script.Add<TRelocateCar>().SetPosition(internalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceBaseInternal).SetUserPosition(internalFrom);

        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            {
                TAreaDropPolicyBuilder odp;
                odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(5000).SetPolicy(EDropOfferPolicy::FeesMinute).SetAlertType("test_alert").SetAlertsCount(0).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
                doTag->MutableDropPolicies().emplace_back(odp);
            }
            {
                TAreaDropPolicyBuilder odp;
                odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(1000).SetPolicy(EDropOfferPolicy::FeesMinute).SetAlertType("test_alert").SetAlertsCount(0).SetAlertLandingIds({"landing_alert1, landing_alert2, NO_LANDING"});
                odp.SetIsInternalFlowCorrector(true);
                doTag->MutableDropPolicies().emplace_back(odp);
            }

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }

        {
            THolder<TDestinationDetectorTag> doTag(new TDestinationDetectorTag(TDestinationDetectorTag::TypeName));
            doTag->SetFixValue(100000);
            doTag->SetStartZoneAttributesFilter(TTagsFilter::BuildFromString("-incorrect_start_zone"));

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
        }
        script.Add<TDropCache>();

        const auto checkerFixPriceMax = [&priceBase](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_C(Abs(priceBase / offerJson["pack_price"].GetUIntegerSafe() - 100.0 / 150) < 7e-2, TStringBuilder() << priceBase << Endl);
        };
        script.Add<TRelocateCar>().SetPosition(from);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMax).SetUserPosition(from);

        const auto checkerFixPriceMaxInternal = [&priceBaseInternal](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_C(Abs(priceBaseInternal / offerJson["pack_price"].GetUIntegerSafe() - 100.0 / 110) < 7e-2, TStringBuilder() << priceBaseInternal << Endl);
        };
        script.Add<TRelocateCar>().SetPosition(internalFrom);
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(to).SetChecker(checkerFixPriceMaxInternal).SetUserPosition(internalFrom);

        script.Add<TRelocateCar>().SetPosition(internalFrom);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(internalFrom);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.AddMove<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto sessionChecker1 = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            return json["segment"]["session"]["specials"]["total_price"].GetUIntegerSafe() == 1400;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker1);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FixWithHintsDifferentOfferIds) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.42643984041714, 55.740277103506685);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);

        const TString jsonHintsStr = R"(
                                {"work":{
                                    "name": "Работа",
                                    "coordinate" : [37.6408, 55.7367]
                                },
                                "home":{
                                    "name": "Дом",
                                    "coordinate" : [37.3931, 55.8713]
                                }})";

        script.Add<TCreateAndBookOffer>().SetOffersCount(3).SetWithHints(true).SetOfferName("fixpoint_offer_constructor").SetPostJson(NJson::ReadJsonFastTree(jsonHintsStr)).SetUserPosition(from);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(Inheritance) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        NJson::TJsonValue originalReport;
        auto destination = TGeoCoord{37.640373, 55.736713};
        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        UNIT_ASSERT(env->WaitCar(carId));
        auto offerId = env->CreateOffer(carId, userId, &originalReport, 0, destination, {}, nullptr, "fixpoint_offer_constructor");
        UNIT_ASSERT(offerId);
        auto previousOfferId = offerId;
        auto previousPrice = originalReport["offers"][0]["pack_price"].GetUIntegerSafe();
        for (size_t i = 0; i < 5; ++i) {
            auto newDestination = destination;
            newDestination.X += 0.0001;
            newDestination.Y += 0.0001;
            NJson::TJsonValue post;
            post["previous_offer_id"] = previousOfferId;
            auto report = env->Request(
                userId,
                "/api/yandex/offers/create",
                TStringBuilder() << "&car_id=" << carId << "&offer_name=fixpoint_offer_constructor&user_destination=" << newDestination.X << '+' << newDestination.Y,
                post
            );
            auto newOfferId = report["offers"][0]["offer_id"].GetString();
            auto newPrice = originalReport["offers"][0]["pack_price"].GetUIntegerSafe();
            UNIT_ASSERT(newOfferId);
            UNIT_ASSERT_VALUES_EQUAL(newPrice, previousPrice);
            previousOfferId = newOfferId;
            previousPrice = newPrice;
        }
    }

    Y_UNIT_TEST(Cashback) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TSleepAction>().SetWaitingDuration(TDuration::Seconds(10));
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor_with_cashback").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TCommonChecker>([](TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("fixpoint_offer_constructor_with_cashback").SetRoleId("fixpoint_access");
            UNIT_ASSERT(context.GetDriveAPI().GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session) && session.Commit());
        });
        script.Add<TRelocateCar>().SetPosition(TGeoCoord(37.57248636458738, 55.73354854993565));
        const auto sessionCheckerWithoutCashback = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            const NJson::TJsonValue& billInfo = json["segment"]["session"]["specials"]["bill"];
            for(const auto& record : billInfo.GetArray()) {
                UNIT_ASSERT_UNEQUAL(record["type"].GetString(), TBillRecord::CashbackType);
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerWithoutCashback);
        script.Add<TDrop>(TDuration::Minutes(1)).SetCarPosition(to);
        const auto calcer = [](const NJson::TJsonValue& jsonOffer) -> ui32 {
            INFO_LOG << jsonOffer["pack_price"] << Endl;
            UNIT_ASSERT(jsonOffer["pack_price"].GetUIntegerSafe() > 100);
            return jsonOffer["pack_price"].GetUIntegerSafe();
        };
        script.Add<TCheckPrice>(TDuration::Minutes(10), calcer).SetPrecision(100);
        const auto sessionChecker = [](const TRTContext& context, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            const NJson::TJsonValue& billInfo = json["segment"]["session"]["specials"]["bill"];
            bool found = false;
            for(const auto& record : billInfo.GetArray()) {
                if (record["type"].GetString() == TBillRecord::CashbackType) {
                    UNIT_ASSERT(!found);
                    found = true;
                    UNIT_ASSERT_EQUAL(record["cost"].GetUInteger(), json["segment"]["session"]["specials"]["total_price"].GetUInteger() * context.GetOfferJsonUnsafe()["cashback_percent"].GetUInteger() / 100);
                }
            }
            return found;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);
        UNIT_ASSERT(script.Execute());
    }
}
