#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/offers/actions/flexipack.h>

#include <library/cpp/testing/unittest/registar.h>

inline const TString SimplePriceOfferConstructorName = "simple_price_offer_constructor";
inline const TString SimpleFlexiblePackOfferName = "simple_flexible_pack_offer";

void RegisterPriceOfferConstructor(const NDrive::IServer& server) {
    auto action = MakeAtomicShared<TPriceOfferConstructor>();
    action->SetName(SimplePriceOfferConstructorName);
    action->SetDescription(SimplePriceOfferConstructorName);
    action->MutableMarket().SetPackDurationPrice({
        { TDuration::Hours(3).Seconds(), 12345 },
    });
    action->MutableMarket().SetPackMileagePrice({
        { 0, 0 },
        { 50, 54321 },
    });
    RegisterAction(server, action);
}

void RegisterFlexiblePackOfferBuilder(const NDrive::IServer& server) {
    auto tagsFilter = TTagsFilter::BuildFromString("?old_state_reservation,?old_state_riding,?old_state_parking");
    auto action = MakeAtomicShared<TFlexiblePackOfferBuilder>(new TConstantPriceConfig(1042), new TConstantPriceConfig(4242), tagsFilter);
    action->SetName(SimpleFlexiblePackOfferName);
    action->SetDescription(SimpleFlexiblePackOfferName);
    action->SetDetailedDescription("Very detailed description");
    action->SetZeroMileageDetailedDescription("No mileage included");
    action->SetChargableAccounts({ "card" });
    action->SetPriceOfferConstructor(SimplePriceOfferConstructorName);
    action->SetPriceSource(TFlexiblePackOfferBuilder::EPriceSource::ExternalOnly);
    RegisterAction(server, action, "pack_access");
}

Y_UNIT_TEST_SUITE(FlexiblePackSuite) {
    Y_UNIT_TEST(Simple) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterPriceOfferConstructor(*env.GetServer());
        RegisterFlexiblePackOfferBuilder(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto offerId = env->CreateOffer(carId, userId, nullptr, 0, {}, {}, nullptr, SimpleFlexiblePackOfferName);
        UNIT_ASSERT(offerId);
    }

    Y_UNIT_TEST(ZeroMileage) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterPriceOfferConstructor(*env.GetServer());
        RegisterFlexiblePackOfferBuilder(*env.GetServer());

        NJson::TJsonValue post;
        post["variables"]["mileage"] = 0;

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "car_id=" + carId + "&offer_name=" + SimpleFlexiblePackOfferName, post);
        UNIT_ASSERT(createdOffers.IsDefined());
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
    }

    Y_UNIT_TEST(Restore) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterPriceOfferConstructor(*env.GetServer());
        RegisterFlexiblePackOfferBuilder(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto offerId = env->CreateOffer(carId, userId, nullptr, 0, {}, {}, nullptr, SimpleFlexiblePackOfferName);
        UNIT_ASSERT(offerId);

        auto restored = env->Request(userId, "/api/yandex/offers/restore", "offer_id=" + offerId);
        UNIT_ASSERT(restored.IsDefined());
    }
}
