#include <drive/backend/fueling_manager/ut/library/constants.h>
#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/fueling.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/standart.h>

#include <drive/telematics/server/library/config.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/system/env.h>

Y_UNIT_TEST_SUITE(DriveFueling) {

    class TFuelingTestManager {
        using EmulatorPtr = TAtomicSharedPtr<NDrive::TCarEmulator>;
        using DriverPtr = TAtomicSharedPtr<NDrive::TCarDriver>;


    public:
        NDrive::TServerConfigGenerator ConfigGenerator;
        TTelematicServerBuilder TmBuilder;
        THolder<TEnvironmentGenerator> EnvironmentGenerator;

        void Init() {
            ConfigGenerator.SetSensorApiName({});
            ConfigGenerator.SetNeedBackground(EServerBackgrounds::FuelingService);
        }

        NDrive::TServerConfigGenerator& GetConfigGenerator() {
            return ConfigGenerator;
        }
        TServerConfigConstructorParams GetConfigParams() {
            return TServerConfigConstructorParams(ConfigGenerator.GetString().data());
        }

        void RunDriving(const TString& model, NDrive::TServerGuard& server, EmulatorPtr& emulator, DriverPtr& driver, TString& objectId) {
            const TDriveAPI& driveApi = *server->GetDriveAPI();
            EnvironmentGenerator = MakeHolder<TEnvironmentGenerator>(*server.Get());
            EnvironmentGenerator->BuildEnvironment();
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TEnvironmentGenerator::TCar newCar = EnvironmentGenerator->CreateCar(session, model);
            UNIT_ASSERT(session.Commit());
            objectId = newCar.Id;
            TmBuilder.Run();
            driver = new NDrive::TCarDriver(TmBuilder.GetAPI());
            emulator = TmBuilder.BuildEmulator(newCar.IMEI);
            UNIT_ASSERT(ConfigGenerator.WaitLocation(objectId));
            UNIT_ASSERT(ConfigGenerator.WaitSensor(objectId, "mileage"));
            SetFuelPersent(objectId, 50, emulator);
            TString offerId;
            {
                THolder<TStandartOfferReport> offer(new TStandartOfferReport(new TStandartOffer, nullptr));
                offer->GetOfferPtrAs<IOffer>()->SetObjectId(objectId).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->GetOffer()->SetChargableAccounts({ "card", "bonus" });
                offerId = offer->GetOffer()->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({offer.Release()}));
            }
            UNIT_ASSERT(ConfigGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            UNIT_ASSERT(ConfigGenerator.WaitStatus(objectId, "old_state_reservation", *server));
            UNIT_ASSERT(ConfigGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
            UNIT_ASSERT(ConfigGenerator.WaitStatus(objectId, "old_state_acceptance", *server));
            UNIT_ASSERT(ConfigGenerator.EvolveTag("old_state_riding", USER_ID_DEFAULT));
        }

        struct TStationInfo {
            TGeoCoord Coord;
            TString Name;
            ui32 ColumnsCount;
        };

        TStationInfo GetStationInfo(const TString& stationId, const TSet<EFuelType>& filter = {}) {
            NJson::TJsonValue report = ConfigGenerator.GetFuelingMap(USER_ID_DEFAULT);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.Has("map"));
            UNIT_ASSERT(report["map"].Has("stations"));
            TGeoCoord coord;
            TString stationName;
            ui32 columnsSize = 0;
            for (auto&& i : report["map"]["stations"].GetArray()) {
                if (i.Has("id") && i["id"] == stationId) {
                    INFO_LOG << i << Endl;
                    UNIT_ASSERT(i.Has("name"));
                    stationName = i["name"].GetStringSafe();
                    UNIT_ASSERT(coord.DeserializeFromString(i["location"].GetString()));
                    UNIT_ASSERT(i.Has("columns"));
                    if (!filter.empty()) {
                        for (auto&& c : i["columns"].GetArray()) {
                            for (auto&& ft : c["ft"].GetArray()) {
                                EFuelType val = EFuelType::Undefined;
                                if (NJson::ParseField(ft, NJson::Stringify(val)) && val != EFuelType::Undefined && filter.contains(val)) {
                                    ++columnsSize;
                                    break;
                                }
                            }
                        }
                        return {coord, stationName, columnsSize};
                    } else {
                        columnsSize = i["columns"].GetArray().size();
                    }
                    return {coord, stationName, columnsSize};
                }
            }
            UNIT_FAIL("Station not found: " + stationId);
            return {coord, stationName, columnsSize};
        }

        void SetFuelPersent(const TString& objectId, const ui32 persent, EmulatorPtr emulator) {
            emulator->GetContext().SetFuelPercent(persent);
            emulator->GetContext().TrySetEngineStarted(true);
            UNIT_ASSERT(ConfigGenerator.WaitSensor(objectId, "fuel_level", ToString(persent)));
        }

        void RelocateCar(const TGeoCoord& coord, const TString& objectId, EmulatorPtr emulator, DriverPtr driver) {
            driver->RelocateCar(emulator->GetIMEI(), coord);
            UNIT_ASSERT(ConfigGenerator.WaitLocation(objectId, coord));
        }

        EFuelingStatus GetFuelingStatus(const bool byTag = false, const TString cgi = "") {
            EFuelingStatus result;
            if (byTag) {
                auto report = ConfigGenerator.GetCurrentSession(USER_ID_DEFAULT);
                INFO_LOG << report << Endl;
                UNIT_ASSERT(report["user"].Has("fueling") && report["user"]["fueling"].Has("status"));
                UNIT_ASSERT(NJson::ParseField(report["user"]["fueling"]["status"], NJson::Stringify(result), true));
                return result;
            }
            auto report = ConfigGenerator.GetFuelingInfo(USER_ID_DEFAULT, cgi);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.Has("current_state") && report["current_state"].Has("status"));
            UNIT_ASSERT(NJson::ParseField(report["current_state"]["status"], NJson::Stringify(result), true));
            return result;
        }

        double GetTagLiters(NDrive::TServerGuard& server) {
            TVector<TDBTag> dbTags;
            const TDriveAPI& driveApi = *(server->GetDriveAPI());
            auto session = driveApi.BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, { TUserFuelingTag::GetTypeName() }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT(dbTags.front().Is<TUserFuelingTag>());
            INFO_LOG << dbTags.front().GetTagAs<TUserFuelingTag>()->GetPublicReport(nullptr, nullptr) << Endl;
            return dbTags.front().GetTagAs<TUserFuelingTag>()->GetLiters();
        }

        void StartFueling(const TStationInfo& stationInfo, const TString& columnId, const TString& objectId, EmulatorPtr emulator, const bool byTag = false, const bool tanker = false, const TString& patchName = "") {
            if (!tanker) {
                SetFuelPersent(objectId, 50, emulator);
            }
            auto report = ConfigGenerator.GetFuelingInfo(USER_ID_DEFAULT, patchName ? "tag_name=" + patchName : "");
            UNIT_ASSERT_VALUES_EQUAL(report["current_state"]["status"].GetStringSafe(), ToString(EFuelingStatus::ReadyForStart));
            UNIT_ASSERT_VALUES_EQUAL(report["current_state"]["station"]["name"].GetStringSafe(), stationInfo.Name);
            UNIT_ASSERT_VALUES_EQUAL(report["current_state"]["station"]["columns"].GetArraySafe().size(), stationInfo.ColumnsCount);
            report = ConfigGenerator.GetCurrentSession(USER_ID_DEFAULT);
            UNIT_ASSERT_C(!report["user"].Has("fueling"), TStringBuilder() << report << Endl);
            UNIT_ASSERT(ConfigGenerator.StartFueling(USER_ID_DEFAULT, columnId, tanker, patchName));
            UNIT_ASSERT_VALUES_UNEQUAL(GetFuelingStatus(byTag), EFuelingStatus::ReadyForStart);
            bool done = false;
            const static TSet<EFuelingStatus> waitStatuses = {
                EFuelingStatus::AcceptOrder,
                EFuelingStatus::WaitingRefueling,
                EFuelingStatus::PaymentInProgress,
                EFuelingStatus::Free,
                EFuelingStatus::Fueling};
            auto startTime = Now();
            do {
                if (!waitStatuses.contains(GetFuelingStatus(byTag))) {
                    done = true;
                    break;
                } else {
                    Sleep(TDuration::Seconds(5));
                }
            } while (Now() - startTime < TDuration::Minutes(2));
            UNIT_ASSERT(done);
        }

        std::pair<EFuelType, double> GetActualFuelData(NDrive::TServerGuard& server) {
            TVector<TDBTag> dbTags;
            const TDriveAPI& driveApi = *server->GetDriveAPI();
            auto session = driveApi.BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, { TUserFuelingTag::GetTypeName() }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT(dbTags.front().Is<TUserFuelingTag>());
            INFO_LOG << dbTags.front().GetTagAs<TUserFuelingTag>()->GetPublicReport(nullptr, nullptr) << Endl;
            UNIT_ASSERT(dbTags.front().GetTagAs<TUserFuelingTag>()->HasActualFuelType());
            UNIT_ASSERT(dbTags.front().GetTagAs<TUserFuelingTag>()->HasActualLiters());
            return {dbTags.front().GetTagAs<TUserFuelingTag>()->GetActualFuelTypeRef(), dbTags.front().GetTagAs<TUserFuelingTag>()->GetActualLitersRef()};
        }

        void UpdateExpectFuelData(NDrive::TServerGuard& server, const EFuelType type, const double liters, const EFuelingStatus status, const TMaybe<EFuelType> actualFuel = {}) {
            const TDriveAPI& driveApi = *server->GetDriveAPI();
            auto checkUpdate = [&]() {
                Sleep(TDuration::Seconds(10));
                TVector<TDBTag> tags;
                auto session = driveApi.BuildTx<NSQL::Writable>();
                UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, { TUserFuelingTag::GetTypeName() }, tags, session));
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                auto tag = tags.front().MutableTagAs<TUserFuelingTag>();
                const bool res = tag && tag->HasFuelType() && tag->GetFuelTypeRef() == type && tag->GetLiters() == liters && tag->GetCurrentState() == status;
                if (!res) {
                    WARNING_LOG << "Tag need to retry for update";
                }
                return res;
            };
            const auto stopAt = Now() + TDuration().Minutes(2);
            do {
                INFO_LOG << "Try to update tag data " << type << " " << liters << " " << status << Endl;
                TVector<TDBTag> dbTags;
                auto session = driveApi.BuildTx<NSQL::Writable>();
                UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, { TUserFuelingTag::GetTypeName() }, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
                auto newTag = dbTags.front().MutableTagAs<TUserFuelingTag>();
                UNIT_ASSERT_VALUES_UNEQUAL(newTag, nullptr);
                newTag->SetFuelType(type);
                newTag->SetLiters(liters);
                newTag->SetCurrentState(status);
                if (actualFuel) {
                    newTag->SetActualFuelType(actualFuel);
                }
                UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().UpdateTagData(dbTags.front(), USER_ID_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            } while (!checkUpdate() && Now() < stopAt);
        }

        bool WaitFuelStatus(const EFuelingStatus status) {
            const auto startTime = Now();
            do {
                if (GetFuelingStatus() == status) {
                    return true;
                }
                Sleep(TDuration::Seconds(5));
            } while (Now() - startTime < TDuration::Minutes(2));
            ERROR_LOG << "Last fuel status " << GetFuelingStatus() << Endl;
            return false;
        }

        bool WaitFreeColumn(NDrive::TServerGuard& server, const TString& station, const TString& column) {
            const auto& mgr = server->GetFuelingManager();
            const auto startTime = Now();
            TMaybe<TFuelingManager::TStatusInfo> info;
            do {
                TMessagesCollector errors;
                if (mgr->GetPostStatus(station, column, info, EFuelClientType::Default, errors) == EFuelingStatus::Free) {
                    return true;
                }
                Sleep(TDuration::Seconds(2));
            } while (Now() - startTime < TDuration::Minutes(2));
            return false;
        }

        bool CheckTankerFuelingAbility() {
            auto report = ConfigGenerator.GetCurrentSession(USER_ID_DEFAULT);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.Has("tanker_fueling_ability"));
            return report["tanker_fueling_ability"].GetBooleanSafe();
        }

        bool WaitTankerFuelingAbility(bool status = true) {
            const auto startTime = Now();
            do {
                if (CheckTankerFuelingAbility() == status) {
                    return true;
                }
                Sleep(TDuration::Seconds(5));
            } while (Now() - startTime < TDuration::Minutes(2));
            ERROR_LOG << "Last ability status " << CheckTankerFuelingAbility() << Endl;
            return false;
        }

        void AddTankerFuelingAbility(NDrive::TServerGuard& server, NDrive::TCarEmulator& emulator, const TString& objectId, const bool withSecondSensor = false) {
            const ui32 val = 100;
            emulator.GetContext().SetTankerFuelLevel(val);
            UNIT_ASSERT(ConfigGenerator.WaitSensorState(objectId, "aux_fuel_level_1", ToString(val)));
            if (withSecondSensor) {
                emulator.GetContext().SetTankerSecondFuelLevel(val);
                UNIT_ASSERT(ConfigGenerator.WaitSensorState(objectId, "aux_fuel_level_2", ToString(val)));
            }
            const IDriveTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager();
            const TDeviceTagsManager& deviceTagsManager = tagsManager.GetDeviceTags();
            auto tag = MakeAtomicShared<TTelematicsConfigurationTag>();
            tag->SetApplyOnAdd(true);
            NJson::TJsonValue table;
            table.AppendValue(NJson::TMapBuilder("calibrated", 0)("original", 0));
            table.AppendValue(NJson::TMapBuilder("calibrated", 356)("original", 4095));
            NJson::TJsonValue data;
            data["calibrators"]["aux_fuel_level_1"]["table"] = table;
            if (withSecondSensor) {
                data["calibrators"]["aux_fuel_level_2"]["table"] = table;
            }
            TMessagesCollector errors;
            UNIT_ASSERT_C(tag->SpecialDataFromJson(data, &errors), errors.GetStringReport());
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, objectId, server.Get(), session));
            UNIT_ASSERT(session.Commit());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            for (int i = 0; i < 90; ++i) {
                if (CheckTankerFuelingAbility()) {
                    break;
                }
                Sleep(TDuration::Seconds(1));
            }
        }

        void AddFuelPatchTag(NDrive::TServerGuard& server, const TString& objectId, const EFuelPatchTags selectedTag) {
            auto tx = Yensured(server->GetDriveAPI())->template BuildTx<NSQL::Writable>();
            auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(ToString(selectedTag));
            UNIT_ASSERT_C(!!tag, "fail to constrect: " + ToString(selectedTag));
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tag, "robot-frontend", objectId, server.Get(), tx) && tx.Commit(), "fail to execute command:" + tx.GetStringReport());
        }
    };

    Y_UNIT_TEST(Simple) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(DEFAULT_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        manager.SetFuelPersent(objectId, 100, emulator);
        UNIT_ASSERT(manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT, "", false).IsNull());
        manager.StartFueling(stationInfo, DEFAULT_PETROL_COLUMN, objectId, emulator);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::Completed);
        UNIT_ASSERT(manager.GetConfigGenerator().CancelFueling(USER_ID_DEFAULT));
        manager.SetFuelPersent(objectId, 50, emulator);
        UNIT_ASSERT(!manager.GetConfigGenerator().StartFueling(USER_ID_DEFAULT, DEFAULT_PETROL_COLUMN));
        auto report = manager.GetConfigGenerator().GetCurrentSession(USER_ID_DEFAULT);
        UNIT_ASSERT_C(!report["user"].Has("fueling"), TStringBuilder() << report << Endl);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::ReadyForStart);
        manager.RelocateCar(TGeoCoord(1, 1), objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
    }

    Y_UNIT_TEST(SimpleCancelOnFail) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.post_pay.allowed_retry", ToString(EFuelingStatus::StationCanceled), USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("skoda_octavia", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(DEFAULT_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        manager.StartFueling(stationInfo, CANCEL_PETROL_COLUMN, objectId, emulator);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::StationCanceled));
        {
            auto result = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT);
            UNIT_ASSERT_C(result.Has("current_state") && result["current_state"].Has("can_retry") && !result["current_state"]["can_retry"].GetBoolean(), result.GetStringRobust());
        }
        UNIT_ASSERT(manager.GetConfigGenerator().CancelFueling(USER_ID_DEFAULT));
        TInstant startTime = Now();
        auto report = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT);
        while (Now() - startTime < TDuration::Minutes(2) && (!report["current_state"]["status"].IsString() || report["current_state"]["status"].GetStringSafe() != ToString(EFuelingStatus::ReadyForStart))) {
            report = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT);
            INFO_LOG << report << Endl;
            Sleep(TDuration::Seconds(5));
        }
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::ReadyForStart);
    }

    Y_UNIT_TEST(StationFilter) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        configGenerator.SetNeedBackground(EServerBackgrounds::FuelingService);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        {
            NJson::TJsonValue report = configGenerator.GetFuelingMap(USER_ID_DEFAULT);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.Has("map"));
            UNIT_ASSERT(report["map"].Has("stations"));
            bool hasPostPay = false;
            for (auto&& i : report["map"]["stations"].GetArraySafe()) {
                if (i.Has("post_pay") && i["post_pay"].GetBooleanSafe()) {
                    hasPostPay = true;
                    break;
                }
            }
            UNIT_ASSERT(!hasPostPay);
        }
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            NJson::TJsonValue report = configGenerator.GetFuelingMap(USER_ID_DEFAULT);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.Has("map"));
            UNIT_ASSERT(report["map"].Has("stations"));
            bool hasPostPay = false;
            for (auto&& i : report["map"]["stations"].GetArraySafe()) {
                if (i.Has("post_pay") && i["post_pay"].GetBooleanSafe()) {
                    hasPostPay = true;
                    break;
                }
            }
            UNIT_ASSERT(hasPostPay);
        }
    }

    Y_UNIT_TEST(GeoStationFilter) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(DEFAULT_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        {
            NJson::TJsonValue report = manager.GetConfigGenerator().GetFuelingMap(USER_ID_DEFAULT);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.Has("map"));
            UNIT_ASSERT(report["map"].Has("stations"));
            UNIT_ASSERT(report["map"]["stations"].GetArray().size() > 2);
        }
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.station_visibility_meters", "500", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            NJson::TJsonValue report = manager.GetConfigGenerator().GetFuelingMap(USER_ID_DEFAULT);
            INFO_LOG << report << Endl;
            UNIT_ASSERT(report.Has("map"));
            UNIT_ASSERT(report["map"].Has("stations"));
            UNIT_ASSERT_VALUES_EQUAL(report["map"]["stations"].GetArray().size(), 2);
        }
    }

    using TEmulatorPtr = TAtomicSharedPtr<NDrive::TCarEmulator>;
    using TDrivePtr = TAtomicSharedPtr<NDrive::TCarDriver>;

    void SimplePostPayCheck(TFuelingTestManager& manager, TEmulatorPtr& emulator, TDrivePtr& driver, const TString& objectId) {
        auto report = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT, "filled_up=true");
        UNIT_ASSERT_VALUES_EQUAL_C(report["current_state"]["status"].GetStringSafe(), ToString(EFuelingStatus::AcceptOrder), TStringBuilder() << report << Endl);
        TInstant startTime = Now();
        do {
            report = manager.GetConfigGenerator().GetCurrentSession(USER_ID_DEFAULT);
            INFO_LOG << manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT) << Endl;
            INFO_LOG << report << Endl;
            Sleep(TDuration::Seconds(5));
        } while (report["user"]["fueling"]["status"].GetStringSafe() != ToString(EFuelingStatus::Completed) && Now() - startTime < TDuration::Minutes(2));
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(true), EFuelingStatus::Completed);
        UNIT_ASSERT(manager.GetConfigGenerator().CancelFueling(USER_ID_DEFAULT));
        manager.SetFuelPersent(objectId, 50, emulator);
        UNIT_ASSERT(!manager.GetConfigGenerator().StartFueling(USER_ID_DEFAULT, DEFAULT_POSTPAY_PETROL_COLUMN));
        report = manager.GetConfigGenerator().GetCurrentSession(USER_ID_DEFAULT);
        UNIT_ASSERT_C(!report["user"].Has("fueling"), TStringBuilder() << report << Endl);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::ReadyForStart);
        manager.RelocateCar(TGeoCoord(1, 1), objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::NotInStation);
    }

    Y_UNIT_TEST(PostPay) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN));
        manager.StartFueling(stationInfo, DEFAULT_POSTPAY_PETROL_COLUMN, objectId, emulator, true);
        auto [type, liters] = manager.GetActualFuelData(server);
        manager.UpdateExpectFuelData(server, type, liters, EFuelingStatus::FuelingCompleted);
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(PostPayFuelGroups) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        const TString settings = R"(
            [
                {
                    "request" : ["a95"],
                    "allow" : ["a92"]
                }
            ]
        )";
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.fuel_groups", settings, USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId); // column has only a92
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN));
        manager.StartFueling(stationInfo, DEFAULT_POSTPAY_PETROL_COLUMN, objectId, emulator, true);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        {
            auto [type, liters] = manager.GetActualFuelData(server);
            type = EFuelType::A95;
            manager.UpdateExpectFuelData(server, type, liters, EFuelingStatus::FuelingCompleted);
        }
        auto report = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT, "filled_up=true"); // start order with a92 for expected a95
        UNIT_ASSERT_VALUES_EQUAL_C(report["current_state"]["status"].GetStringSafe(), ToString(EFuelingStatus::AcceptOrder), TStringBuilder() << report << Endl);
        TInstant startTime = Now();
        do {
            report = manager.GetConfigGenerator().GetCurrentSession(USER_ID_DEFAULT);
            INFO_LOG << manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT) << Endl;
            INFO_LOG << report << Endl;
            Sleep(TDuration::Seconds(5));
        } while (report["user"]["fueling"]["status"].GetStringSafe() != ToString(EFuelingStatus::Completed) && Now() - startTime < TDuration::Minutes(2));
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::Completed));
    }

    Y_UNIT_TEST(PostPayCheck) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN));
        manager.StartFueling(stationInfo, DEFAULT_POSTPAY_PETROL_COLUMN, objectId, emulator, true);
        auto [type, liters] = manager.GetActualFuelData(server);
        manager.UpdateExpectFuelData(server, type, liters, EFuelingStatus::Fueling);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        const EFuelType wrongType = type == EFuelType::A92 ? EFuelType::Diesel : EFuelType::A92;
        manager.UpdateExpectFuelData(server, wrongType, liters, EFuelingStatus::Fueling);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::WrongFuelType));
        manager.UpdateExpectFuelData(server, type, liters - 10, EFuelingStatus::Fueling);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false, "filled_up=true"), EFuelingStatus::WrongFuelLiters);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::WrongFuelLiters);
        UNIT_ASSERT(manager.GetConfigGenerator().CancelFueling(USER_ID_DEFAULT));
        const TSet<EFuelingStatus> expectedStatuses = {EFuelingStatus::WaitCancel, EFuelingStatus::ReadyForStart};
        UNIT_ASSERT(expectedStatuses.contains(manager.GetFuelingStatus(false)));
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::ReadyForStart));
    }

    Y_UNIT_TEST(FuelGroups) {
        const TString settings = R"(
            [
                {
                    "request" : ["a92", "a92_premium"],
                    "allow" : ["a92", "a92_premium", "a95", "a95_premium"]
                },
                {
                    "request" : ["a95", "a95_premium"],
                    "allow" : ["a95", "a95_premium"]
                },
                {
                    "request" : ["diesel", "diesel_premium", "diesel_winter"],
                    "allow" : ["diesel", "diesel_premium", "diesel_winter"]
                }
            ]
        )";
        TFuelGroups fuelGroups;
        {
            NJson::TJsonValue json;
            UNIT_ASSERT(ReadJsonTree(settings, &json));
            auto& map = fuelGroups.MutableMap();
            map = TFuelGroups::DeserializeFuelMap(json);
            UNIT_ASSERT(map);
        }
        UNIT_ASSERT(fuelGroups.FuelTypeCmp(EFuelType::A100, EFuelType::A100));
        UNIT_ASSERT(!fuelGroups.FuelTypeCmp(EFuelType::A100, EFuelType::A100Premium));
        UNIT_ASSERT(fuelGroups.FuelTypeCmp(EFuelType::A92, EFuelType::A95Premium));
        UNIT_ASSERT(!fuelGroups.FuelTypeCmp(EFuelType::A92, EFuelType::Diesel));
        UNIT_ASSERT(fuelGroups.FuelTypeCmp(EFuelType::A95, EFuelType::A95Premium));
        UNIT_ASSERT(!fuelGroups.FuelTypeCmp(EFuelType::A95, EFuelType::A92));
        {
            NJson::TJsonValue json;
            UNIT_ASSERT(ReadJsonTree("[]", &json));
            auto& map = fuelGroups.MutableMap();
            map = TFuelGroups::DeserializeFuelMap(json);
            UNIT_ASSERT(map);
        }
        UNIT_ASSERT(fuelGroups.FuelTypeCmp(EFuelType::A100, EFuelType::A100));
        UNIT_ASSERT(!fuelGroups.FuelTypeCmp(EFuelType::A100, EFuelType::A100Premium));
        UNIT_ASSERT(!fuelGroups.FuelTypeCmp(EFuelType::A92, EFuelType::A95Premium));
    }

    Y_UNIT_TEST(CustomPostPayOrder) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        const TString controlLiters = "1000";
        const TString controlFuelType = ToString(EFuelType::A92);
        NDrive::TServerConfigGenerator::CustomOrder order = {USER_ID_DEFAULT, objectId, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN, controlLiters, controlFuelType};
        UNIT_ASSERT(manager.GetConfigGenerator().SetCustomOrder(order));
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::Completed));
    }

    Y_UNIT_TEST(TankerFueling) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(!manager.CheckTankerFuelingAbility());
        manager.AddTankerFuelingAbility(server, *emulator, objectId);
        UNIT_ASSERT(manager.CheckTankerFuelingAbility());
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN));
        manager.StartFueling(stationInfo, DEFAULT_POSTPAY_PETROL_COLUMN, objectId, emulator, true, true);
        UNIT_ASSERT(manager.GetTagLiters(server) > 200);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(PrePayWithUpperBound) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        const TString model = "kia_rio";
        manager.RunDriving(model, server, emulator, driver, objectId);
        ui32 litersLimit = 0;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto gModelsData = server->GetDriveAPI()->GetModelsData()->FetchInfo(model, session);
            UNIT_ASSERT_C(gModelsData, session.GetStringReport());
            UNIT_ASSERT(!gModelsData.GetResult().empty());
            auto fuelTankVolume = NDrive::GetFuelTankVolume(nullptr, gModelsData.GetResultPtr(model));
            INFO_LOG << "FuelTankVolume for " << model << " is " << fuelTankVolume << Endl;
            litersLimit = fuelTankVolume / 4;
            INFO_LOG << "Liters limit" << litersLimit << Endl;
            UNIT_ASSERT(litersLimit);
        }
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.pre_pay.liters_upper_bound", ToString(litersLimit), USER_ROOT_DEFAULT));
        auto stationInfo = manager.GetStationInfo(DEFAULT_PETROL_STATION);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        manager.StartFueling(stationInfo, DEFAULT_PETROL_COLUMN, objectId, emulator, /* byTag = */ false, /* tanker = */ false);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::Completed);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetTagLiters(server), litersLimit);
    }

    Y_UNIT_TEST(PostPayWithUpperBound) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        const TString model = "kia_rio";
        manager.RunDriving(model, server, emulator, driver, objectId);
        ui32 litersLimit = 0;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto gModelsData = server->GetDriveAPI()->GetModelsData()->FetchInfo(model, session);
            UNIT_ASSERT_C(gModelsData, session.GetStringReport());
            UNIT_ASSERT(!gModelsData.GetResult().empty());
            auto fuelTankVolume = NDrive::GetFuelTankVolume(nullptr, gModelsData.GetResultPtr(model));
            INFO_LOG << "FuelTankVolume for " << model << " is " << fuelTankVolume << Endl;
            litersLimit = fuelTankVolume / 4;
            INFO_LOG << "Liters limit" << litersLimit << Endl;
            UNIT_ASSERT(litersLimit);
        }
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.pre_pay.liters_upper_bound", ToString(litersLimit), USER_ROOT_DEFAULT));
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN));
        manager.StartFueling(stationInfo, DEFAULT_POSTPAY_PETROL_COLUMN, objectId, emulator, /* byTag = */ true, /* tanker = */ false);
        UNIT_ASSERT(manager.GetTagLiters(server) > litersLimit);
    }

    Y_UNIT_TEST(PostPayRetry) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        NJson::TJsonValue payload;
        payload["fuel_type"] = ToString(EFuelType::Diesel);
        UNIT_ASSERT(manager.GetConfigGenerator().ModifyCarModel("kia_rio", "Renault Kaptur", "Renault", USER_ROOT_DEFAULT, payload));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, CANCEL_PETROL_COLUMN));
        manager.StartFueling(stationInfo, CANCEL_PETROL_COLUMN, objectId, emulator, /* byTag = */ true);
        auto [type, liters] = manager.GetActualFuelData(server);
        type = EFuelType::Diesel;
        manager.UpdateExpectFuelData(server, type, liters, EFuelingStatus::FuelingCompleted, type);
        auto report = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT, "filled_up=true");
        UNIT_ASSERT_VALUES_EQUAL_C(report["current_state"]["status"].GetStringSafe(), ToString(EFuelingStatus::AcceptOrder), TStringBuilder() << report << Endl);
        TInstant startTime = Now();
        do {
            report = manager.GetConfigGenerator().GetCurrentSession(USER_ID_DEFAULT);
            INFO_LOG << manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT) << Endl;
            INFO_LOG << report << Endl;
            Sleep(TDuration::Seconds(5));
        } while (report["user"]["fueling"]["status"].GetStringSafe() != ToString(EFuelingStatus::Completed) && Now() - startTime < TDuration::Minutes(2));
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::StationCanceled));
        {
            auto result = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT);
            UNIT_ASSERT_C(result.Has("current_state") && result["current_state"].Has("can_retry") && !result["current_state"]["can_retry"].GetBoolean(), result.GetStringRobust());
            result = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT, "filled_up=true&retry=true", /* assert = */ false);
            UNIT_ASSERT(!result.IsDefined());
        }
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.post_pay.allowed_retry", ToString(EFuelingStatus::StationCanceled), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        {
            auto result = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT);
            UNIT_ASSERT_C(result.Has("current_state") && result["current_state"].Has("can_retry") && result["current_state"]["can_retry"].GetBoolean(), result.GetStringRobust());
        }
        report = manager.GetConfigGenerator().GetFuelingInfo(USER_ID_DEFAULT, "filled_up=true&retry=true");
        UNIT_ASSERT_VALUES_EQUAL_C(report["current_state"]["status"].GetStringSafe(), ToString(EFuelingStatus::AcceptOrder), TStringBuilder() << report << Endl);
    }

    Y_UNIT_TEST(LocationDistance) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(DEFAULT_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        auto coord = stationInfo.Coord;
        manager.RelocateCar(coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::ReadyForStart);
        coord.X -= coord.MakeDXFromDistance(75);
        manager.RelocateCar(coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.station_search_meters", "20", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::ReadyForStart);
        manager.RelocateCar(coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::ReadyForStart);
    }

    Y_UNIT_TEST(LocationDistanceTwoStations) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        auto stationInfo = manager.GetStationInfo(DEFAULT_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        auto coord = stationInfo.Coord;
        manager.RelocateCar(coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::ReadyForStart);
        coord.X += coord.MakeDXFromDistance(100);
        manager.RelocateCar(coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.station_search_meters", "100", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::ReadyForStart);
        manager.RelocateCar(coord, objectId, emulator, driver);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::MoreThanOneStation);
    }

    Y_UNIT_TEST(SimplePatchFuel) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        manager.AddFuelPatchTag(server, objectId, EFuelPatchTags::SimplePatch);
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, NTestDriveFueling::PetrolColumnFor100Premium));
        manager.StartFueling(stationInfo, NTestDriveFueling::PetrolColumnFor100Premium, objectId, emulator, /* byTag = */ true);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        {
            auto [type, _] = manager.GetActualFuelData(server);
            UNIT_ASSERT_VALUES_EQUAL(type, EFuelType::A100Premium);
        }
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(ModelFuelWithPatch) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        manager.AddFuelPatchTag(server, objectId, EFuelPatchTags::SimplePatch);
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN));
        manager.StartFueling(stationInfo, DEFAULT_POSTPAY_PETROL_COLUMN, objectId, emulator, /* byTag = */ true);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        {
            auto [type, _] = manager.GetActualFuelData(server);
            UNIT_ASSERT_VALUES_EQUAL(type, EFuelType::A92);
        }
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(OverrideModelFuel) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        manager.AddFuelPatchTag(server, objectId, EFuelPatchTags::OverrideModel);
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(false), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, NTestDriveFueling::PetrolColumnFor100Premium));
        manager.StartFueling(stationInfo, NTestDriveFueling::PetrolColumnFor100Premium, objectId, emulator, /* byTag = */ true);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        {
            auto [type, _] = manager.GetActualFuelData(server);
            UNIT_ASSERT_VALUES_EQUAL(type, EFuelType::A100Premium);
        }
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(SimpleTankerPatchFuel) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        manager.AddFuelPatchTag(server, objectId, EFuelPatchTags::SimplePatch);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(!manager.CheckTankerFuelingAbility());
        manager.AddTankerFuelingAbility(server, *emulator, objectId);
        UNIT_ASSERT(manager.CheckTankerFuelingAbility());
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, DEFAULT_POSTPAY_PETROL_COLUMN));
        manager.StartFueling(stationInfo, DEFAULT_POSTPAY_PETROL_COLUMN, objectId, emulator, /* byTag = */ true, /* tanker = */ true);
        UNIT_ASSERT(manager.GetTagLiters(server) > 200);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(OverrideModelFuelForTanker) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        manager.AddFuelPatchTag(server, objectId, EFuelPatchTags::OverrideModel);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(!manager.CheckTankerFuelingAbility());
        manager.AddTankerFuelingAbility(server, *emulator, objectId);
        UNIT_ASSERT(manager.CheckTankerFuelingAbility());
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION);
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, NTestDriveFueling::PetrolColumnFor100Premium));
        manager.StartFueling(stationInfo, NTestDriveFueling::PetrolColumnFor100Premium, objectId, emulator, /* byTag = */ true, /* tanker = */ true);
        UNIT_ASSERT(manager.GetTagLiters(server) > 200);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(SelectedSimplePatchForTanker) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        manager.AddFuelPatchTag(server, objectId, EFuelPatchTags::SimplePatch);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(!manager.CheckTankerFuelingAbility());
        manager.AddTankerFuelingAbility(server, *emulator, objectId);
        UNIT_ASSERT(manager.CheckTankerFuelingAbility());
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION, { EFuelType::A100Premium });
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, NTestDriveFueling::PetrolColumnFor100Premium));
        manager.StartFueling(stationInfo, NTestDriveFueling::PetrolColumnFor100Premium, objectId, emulator, /* byTag = */ true, /* tanker = */ true, ToString(EFuelPatchTags::SimplePatch));
        UNIT_ASSERT(manager.GetTagLiters(server) > 200);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(OverrideSensorForTanker) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.enable_post_pay", "true", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        manager.AddFuelPatchTag(server, objectId, EFuelPatchTags::Dut1Sensor);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(!manager.CheckTankerFuelingAbility());
        manager.AddTankerFuelingAbility(server, *emulator, objectId);
        UNIT_ASSERT(manager.CheckTankerFuelingAbility());
        auto stationInfo = manager.GetStationInfo(POST_PAY_PETROL_STATION, { EFuelType::A100Premium });
        UNIT_ASSERT_VALUES_EQUAL(manager.GetFuelingStatus(), EFuelingStatus::NotInStation);
        manager.RelocateCar(stationInfo.Coord, objectId, emulator, driver);
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, NTestDriveFueling::PetrolColumnFor100Premium));
        UNIT_ASSERT(!manager.GetConfigGenerator().StartFueling(USER_ID_DEFAULT, NTestDriveFueling::PetrolColumnFor100Premium, /* tanker = */ true));
        UNIT_ASSERT(manager.WaitFreeColumn(server, POST_PAY_PETROL_STATION, NTestDriveFueling::PetrolColumnFor100Premium));
        manager.StartFueling(stationInfo, NTestDriveFueling::PetrolColumnFor100Premium, objectId, emulator, /* byTag = */ true, /* tanker = */ true, ToString(EFuelPatchTags::Dut1Sensor));
        UNIT_ASSERT(manager.GetTagLiters(server) > 200);
        UNIT_ASSERT(manager.WaitFuelStatus(EFuelingStatus::FuelingCompleted));
        SimplePostPayCheck(manager, emulator, driver, objectId);
    }

    Y_UNIT_TEST(TankerFuelingAbility) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(!manager.CheckTankerFuelingAbility());
        manager.AddTankerFuelingAbility(server, *emulator, objectId);
        UNIT_ASSERT(manager.WaitTankerFuelingAbility());
        {
            const ui32 val = 5000;
            emulator->GetContext().SetTankerFuelLevel(val);
            UNIT_ASSERT(manager.GetConfigGenerator().WaitSensorState(objectId, "aux_fuel_level_1", ToString(val)));
        }
        UNIT_ASSERT(manager.WaitTankerFuelingAbility(/* status = */ false));
    }

    Y_UNIT_TEST(TankerSecondFuelingAbility) {
        TFuelingTestManager manager;
        manager.Init();
        NDrive::TServerConfig config(manager.GetConfigParams());
        NDrive::TServerGuard server(config);
        UNIT_ASSERT(server->GetSettings().SetValue("fueling.tanker.sensors", "1142", USER_ROOT_DEFAULT));
        TAtomicSharedPtr<NDrive::TCarEmulator> emulator;
        TAtomicSharedPtr<NDrive::TCarDriver> driver;
        TString objectId;
        manager.RunDriving("kia_rio", server, emulator, driver, objectId);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(!manager.CheckTankerFuelingAbility());
        manager.AddTankerFuelingAbility(server, *emulator, objectId, /* withSecondSensor = */ true);
        UNIT_ASSERT(manager.WaitTankerFuelingAbility());
        {
            const ui32 val = 5000;
            emulator->GetContext().SetTankerFuelLevel(val);
            UNIT_ASSERT(manager.GetConfigGenerator().WaitSensorState(objectId, "aux_fuel_level_1", ToString(val)));
        }
        UNIT_ASSERT(manager.WaitTankerFuelingAbility());
        {
            const ui32 val = 5000;
            emulator->GetContext().SetTankerSecondFuelLevel(val);
            UNIT_ASSERT(manager.GetConfigGenerator().WaitSensorState(objectId, "aux_fuel_level_2", ToString(val)));
        }
        UNIT_ASSERT(manager.WaitTankerFuelingAbility(/* status = */ false));
    }

}
