#include <drive/backend/ut/library/script.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/data/chargable.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(Futures) {
    using namespace NDrive::NTest;

    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        TNotificationsProcessor notificationsChecker;
        bool checked = false;
        const auto notificationsCheckerImpl = [&checked](const TString& uid, const TString& message) ->bool {
            if (uid == "4008954170") {
                UNIT_ASSERT(!checked);
                checked = true;
                UNIT_ASSERT((message == "offers.futures.success") || (message == "Машина приехала"));
            }
            return true;
        };
        notificationsChecker.SetChecker(notificationsCheckerImpl);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());

        script.Add<TCreateAndBookOffer>().SetIsFutures(true).SetWaitingDuration(TDuration::Minutes(1)).SetUserId(USER_ID_DEFAULT1);
        script.Add<TSleepAction>().SetWaitingDuration(TDuration::Seconds(30));
        script.Add<TDrop>();
        const auto sessionCheckerOld = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (!json.Has("car") || json["segment"]["meta"]["finished"].GetBooleanSafe()) {
                return true;
            }
            return false;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerOld).SetUserId(USER_ID_DEFAULT);
        const auto sessionCheckerNew = [&notificationsChecker, &checked](const TRTContext& context, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << checked << "/" << notificationsChecker.GetCounter() << Endl;
            INFO_LOG << json.GetStringRobust() << Endl;
            if (!json.Has("car") || !json["segment"]["meta"]["finished"].IsBoolean() || json["segment"]["meta"]["finished"].GetBooleanSafe()) {
                return false;
            }
            INFO_LOG << json["car"]["number"] << Endl;
            return (json["car"]["number"].GetStringRobust() == context.GetCar().Number);
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerNew).SetUserId(USER_ID_DEFAULT1);
        UNIT_ASSERT(script.Execute());
        UNIT_ASSERT_C(checked, TStringBuilder() << notificationsChecker.GetCounter() << Endl);
    }

    Y_UNIT_TEST(CheckCarRelocationInvariantPrice) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        TNotificationsProcessor notificationsChecker;
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());

        ui32 priceStart = 0;
        const auto checkerStart = [&priceStart](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            priceStart = offerJson["pack_price"].GetUIntegerSafe();
        };

        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(from).SetChecker(checkerStart).SetIsFutures(true).SetFuturesWaitingDuration(TDuration::Minutes(1)).SetUserPosition(to).SetUserId(USER_ID_DEFAULT1);
        script.Add<TRelocateCar>().SetPosition((from + to) * 0.5);
        const auto checkerSecond = [&priceStart](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            UNIT_ASSERT_C(Abs(1 - 1.0 * priceStart / offerJson["pack_price"].GetUIntegerSafe()) < 0.1, TStringBuilder() << priceStart << " / " << offerJson << Endl);
        };
        script.Add<TCreateAndCheckOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(from).SetChecker(checkerSecond).SetIsFutures(true).SetFuturesWaitingDuration(TDuration::Minutes(1)).SetUserPosition(to).SetUserId(USER_ID_DEFAULT1);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(DoubleRent) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());

        script.Add<TCreateAndBookOffer>().SetIsFutures(true).SetWaitingDuration(TDuration::Minutes(1)).SetUserId(USER_ID_DEFAULT1);
        script.Add<TCreateAndBookOffer>().SetIsFutures(true).SetUserId(USER_ID_DEFAULT2).SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(Visibility) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());

        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (!json["car"]["location"].IsDefined()) {
                return false;
            }
            if (!json["car"]["futures_location"].IsDefined()) {
                return false;
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker).SetState("futures").SetUserId(USER_ID_DEFAULT);

        script.Add<TCheckCarVisibility>().SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);

        script.Add<TCreateAndBookOffer>().SetIsFutures(true).SetWaitingDuration(TDuration::Minutes(1)).SetUserId(USER_ID_DEFAULT1);
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ClusterModification) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.63446698352293, 55.75342958500839);

        TGeoCoord c1(37.53764996692134, 55.887761921053105);
        TGeoCoord c2(37.53318677112058, 55.85513978695587);
        TGeoCoord c3(37.58365521594478, 55.852436127220514);
        TGeoCoord c4(37.59704480334714, 55.88660420992141);

        TGeoCoord center(37.57404217883542, 55.87579723358177);

        NDrive::NTest::TScript script(configGenerator);
        TString clusterName = "cluster_north";
        const ui16 clusterId = FnvHash<ui32>(clusterName);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateArea>().SetId(clusterName).SetCoords({c1, c2, c3, c4, c1}).SetType("cluster").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TCreateCar>().SetPosition(center);
        script.Add<TCheckCarVisibility>().SetVisibility(true).SetClusterId(clusterId).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetClusterId(clusterId + 1).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCreateArea>().SetId(clusterName).SetCoords({c1, c2, c3, c4, c1}).SetType("cluster_fake").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetWaitingDuration(TDuration::Seconds(10)).SetClusterId(clusterId).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetClusterId(clusterId + 1).SetActionUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(InCluster) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.63446698352293, 55.75342958500839);

        TGeoCoord c1(37.53764996692134, 55.887761921053105);
        TGeoCoord c2(37.53318677112058, 55.85513978695587);
        TGeoCoord c3(37.58365521594478, 55.852436127220514);
        TGeoCoord c4(37.59704480334714, 55.88660420992141);

        TGeoCoord center(37.57404217883542, 55.87579723358177);

        NDrive::NTest::TScript script(configGenerator);
        TString clusterName = "cluster_north";
        const ui16 clusterId = FnvHash<ui32>(clusterName);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateArea>().SetId(clusterName).SetCoords({c1, c2, c3, c4, c1}).SetType("cluster").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TCreateCar>().SetPosition(center);
        script.Add<TCheckCarVisibility>().SetVisibility(true).SetClusterId(clusterId).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCheckCarVisibility>().SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetClusterId(clusterId).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCreateAndBookOffer>().SetUserDestination(center).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetClusterId(clusterId).SetActionUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ExtCluster) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.63446698352293, 55.75342958500839);

        TGeoCoord c1(37.53764996692134, 55.887761921053105);
        TGeoCoord c2(37.53318677112058, 55.85513978695587);
        TGeoCoord c3(37.58365521594478, 55.852436127220514);
        TGeoCoord c4(37.59704480334714, 55.88660420992141);

        TGeoCoord center(37.52769360705806, 55.79754966325999);

        NDrive::NTest::TScript script(configGenerator);
        TString clusterName = "cluster_north";
        const ui16 clusterId = FnvHash<ui32>(clusterName);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateArea>().SetId(clusterName).SetCoords({c1, c2, c3, c4, c1}).SetType("cluster").SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TCreateCar>().SetPosition(center);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(center).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TCheckCarVisibility>().SetWaitingDuration(TDuration::Seconds(15)).SetVisibility(true).SetActionUserId(USER_ID_DEFAULT2);
        script.Add<TCheckCarVisibility>().SetWaitingDuration(TDuration::Seconds(15)).SetVisibility(false).SetClusterId(clusterId).SetActionUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(Expired) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TCreateAndBookOffer>().SetIsFutures(true).SetWaitingDuration(TDuration::Minutes(1)).SetUserId(USER_ID_DEFAULT1);
        const auto sessionCheckerFutures = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (json.Has("car") && json["car"].Has("futures_location") && json["futures_offer_failed"].GetBooleanSafe() == false) {
                return true;
            }
            return false;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerFutures).SetUserId(USER_ID_DEFAULT1);

        const auto sessionCheckerFuturesExpired = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (json.Has("car") && !json["car"].Has("futures_location") && json["futures_offer_failed"].GetBooleanSafe() == true) {
                return true;
            }
            return false;
        };
        script.Add<TCheckCurrentSessionWaiting>(TDuration::Hours(2)).SetChecker(sessionCheckerFuturesExpired).SetUserId(USER_ID_DEFAULT1);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ParkingFuturesLocation) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::SurgeConstructor | EServerBackgrounds::FuturesBooking);
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>().SetNeedTelematics(false);
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero());
        script.Add<TPark>(TDuration::Zero());
        script.Add<TCheckCarVisibility>().SetVisibility(false).SetActionUserId(USER_ID_DEFAULT1);
        script.Add<TRide>(TDuration::Zero());
        script.Add<TCreateAndBookOffer>().SetIsFutures(true).SetWaitingDuration(TDuration::Minutes(1)).SetUserId(USER_ID_DEFAULT1);
        script.Add<TPark>(TDuration::Zero());
        const auto sessionCheckerFutures = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)-> bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (json.Has("car") && json["car"].Has("futures_location") && json["futures_offer_failed"].GetBooleanSafe() == false) {
                return true;
            }
            return false;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionCheckerFutures).SetUserId(USER_ID_DEFAULT1);
        UNIT_ASSERT(script.Execute());
    }
}
