#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive/takeout.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/library/cpp/takeout/client.h>
#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(GDPR) {

    Y_UNIT_TEST(TakeoutSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::TakeoutRegular);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there");
        TString uid = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetUid();

        UNIT_ASSERT(configGenerator.TakeoutRequest(uid));
        auto requestsMap = driveApi.GetTakeoutRequests().FetchInfo().GetResult();
        auto requestId = TString();
        for (auto&& requestIt : requestsMap) {
            if (requestIt.second.GetUserId() == userId) {
                requestId = requestIt.second.GetId();
                break;
            }
        }
        UNIT_ASSERT(requestId);

        bool isProcessed = false;
        for (size_t i = 0; i < 20; ++i) {
            if (driveApi.GetTakeoutRequests().FetchInfo(requestId).GetResult().begin()->second.GetProcessingCompletedAt() != TInstant::Zero()) {
                isProcessed = true;
                break;
            }
            Sleep(TDuration::Seconds(1));
        }

        UNIT_ASSERT_C(isProcessed, TStringBuilder() << "not processed after 20 seconds of waiting");
    }

    Y_UNIT_TEST(TakeoutManyThings) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::TakeoutRegular);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        auto userId = eGenerator.CreateUser("skulik-was-there");
        TString uid = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetUid();

        // Upload photos
        {
            auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TAtomicSharedPtr<TUserDocumentPhotoUploadTestCallback> callback = new TUserDocumentPhotoUploadTestCallback();
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::PassportSelfie, "passport_selfie_photo", "passport_selfie_video", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::PassportSelfie, "passport_selfie_photo_2", "passport_selfie_video_2", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::PassportBiographical, "passport_biographical_photo", "passport_biographical_video", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::PassportRegistration, "passport_registration_photo", "passport_registration_video", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::LicenseFront, "license_front_photo", "license_front_video", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::LicenseFront, "license_front_photo_2", "license_front_video_2", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::LicenseBack, "license_back_photo", "license_back_video", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::LicenseBack, "license_back_photo_2", "license_back_video_2", callback, *server);
            Sleep(TDuration::Seconds(1));
            docPhotoManager.AddDocumentPhoto(userId, "registration", NUserDocument::EType::LicenseBack, "license_back_photo_3", "license_back_video_3", callback, *server);
            while (callback->GetTotalResponses() < 9) ;
            UNIT_ASSERT_VALUES_EQUAL(callback->GetTotalSuccesses(), 9);
        }

        {
            // bind card
            eGenerator.GetBillingMock().SetReply("{\"status\": \"success\", \"bound_payment_methods\" : ["
                                                "{"
                                                "\"region_id\": 225, \"payment_method\" : \"card\", \"expiration_month\" : \"07\", \"binding_ts\" : \"1536324485.656\","
                                                "\"id\" : \"card-x12345\", \"expired\" : false, \"card_bank\" : \"TINKOFF BANK\", \"system\" : \"VISA\","
                                                "\"account\" : \"510000****0257\", \"expiration_year\" : \"2021\""
                                                "}"
                                                "]}");

            UNIT_ASSERT(configGenerator.SetDefaultCard("card-x12345", userId));
            Sleep(TDuration::Seconds(3));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }

        UNIT_ASSERT(configGenerator.TakeoutRequest(uid, "job-id-123"));
        auto requestsMap = driveApi.GetTakeoutRequests().FetchInfo();
        auto requestId = TString();
        for (auto&& requestIt : requestsMap) {
            if (requestIt.second.GetUserId() == userId) {
                requestId = requestIt.second.GetId();
                break;
            }
        }
        UNIT_ASSERT(requestId);

        bool isProcessed = false;
        for (size_t i = 0; i < 20; ++i) {
            if (driveApi.GetTakeoutRequests().FetchInfo(requestId).begin()->second.GetProcessingCompletedAt() != TInstant::Zero()) {
                isProcessed = true;
                break;
            }
            Sleep(TDuration::Seconds(1));
        }

        UNIT_ASSERT_C(isProcessed, TStringBuilder() << "not processed after 20 seconds of waiting");

        // now check that the data from photos is correct
        TFakeTakeoutClient* client = dynamic_cast<TFakeTakeoutClient*>(&driveApi.GetTakeoutClient());
        auto archiveContents = client->GetArchiveContents("job-id-123");

        UNIT_ASSERT_VALUES_EQUAL(archiveContents.size(), 13);
        for (auto&& element : archiveContents) {
            auto name = element.first;
            auto content = element.second;
            INFO_LOG << "takeout archive item: " << name << " " << content << Endl;
            if (name.StartsWith("lb")) {
                UNIT_ASSERT(content.StartsWith("license_back_photo"));
            } else if (name.StartsWith("lf")) {
                UNIT_ASSERT(content.StartsWith("license_front_photo"));
            } else if (name.StartsWith("pr")) {
                UNIT_ASSERT(content.StartsWith("passport_registration_photo"));
            } else if (name.StartsWith("ps")) {
                UNIT_ASSERT(content.StartsWith("passport_selfie_photo"));
            } else if (name.StartsWith("pb")) {
                UNIT_ASSERT(content.StartsWith("passport_biographical_photo"));
            } else if (name == "paymethods.json") {
                NJson::TJsonValue json;
                UNIT_ASSERT(ReadJsonFastTree(content, &json));
                UNIT_ASSERT_VALUES_EQUAL(json.GetArray()[0]["pan"]["suffix"].GetString(), "0257");
                UNIT_ASSERT_VALUES_EQUAL(json.GetArray()[0]["pan"]["prefix"].GetString(), "510000");
                UNIT_ASSERT_VALUES_EQUAL(json.GetArray()[0]["paymethod_id"].GetString(), "card-x12345");
            }
        }
    }

    Y_UNIT_TEST(TakeoutNotDriveUser) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(EServerBackgrounds::TakeoutRegular);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TString uid = "134143214321489";

        UNIT_ASSERT(configGenerator.TakeoutRequest(uid));
        auto requestsMap = driveApi.GetTakeoutRequests().FetchInfo();
        TString requestId;
        for (auto&& requestIt : requestsMap) {
            if (requestIt.second.GetUserId() == "user-absent") {
                requestId = requestIt.second.GetId();
                break;
            }
        }
        UNIT_ASSERT(requestId);

        bool isProcessed = false;
        for (size_t i = 0; i < 20; ++i) {
            if (driveApi.GetTakeoutRequests().FetchInfo(requestId).begin()->second.GetProcessingCompletedAt() != TInstant::Zero()) {
                isProcessed = true;
                break;
            }
            Sleep(TDuration::Seconds(1));
        }

        UNIT_ASSERT_C(isProcessed, TStringBuilder() << "not processed after 20 seconds of waiting");

        TFakeTakeoutClient* client = dynamic_cast<TFakeTakeoutClient*>(&driveApi.GetTakeoutClient());
        auto archiveContents = client->GetArchiveContents("job-id-123");
        UNIT_ASSERT_VALUES_EQUAL(archiveContents.size(), 0);
    }
}
