#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/head/head_account.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>


Y_UNIT_TEST_SUITE(HeadTests) {
    Y_UNIT_TEST(SimpleSession) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const THeadAccountManager& headManager = driveApi.GetHeadAccountManager();

        TEnvironmentGenerator eGenerator(*server.Get());
        auto car = eGenerator.CreateCar();

        TInstant now = Now() - TDuration::Hours(4);
        TInstantGuard ig(now);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(headManager.FinishSession(car.Id, session));
            UNIT_ASSERT(headManager.UpdateCarHeadId(car.Id, HEAD_ID_DEFAULT, USER_ID_DEFAULT, session, server.Get()));
            UNIT_ASSERT(session.Commit());
        }
        TString headId;
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(headManager.CreateSession(USER_ID_DEFAULT, car.Id, true, session));
            headId = headManager.GetHeadIdByCar(car.Id);
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(headId, HEAD_ID_DEFAULT);
    }

    Y_UNIT_TEST(CheckFlagRatio) {
        InitGlobalLog2Console(TLOG_DEBUG);
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const ISettings& settings = server->GetSettings();
        TRolesConfig rolesCfg;
        TUserPermissionsFeatures userFeatures;
        userFeatures.SetIsYandexUser(true);
        TUserPermissions::TPtr perm = driveApi.GetRolesManager()->GetUserPermissions(USER_ID_DEFAULT, driveApi.GetTagsManager(), rolesCfg, userFeatures, Now());
        TInstant now = Now() - TDuration::Hours(12);
        TInstantGuard ig(now);
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(settings.SetValue("launch_head_app", "disable", USER_ID_DEFAULT));
            UNIT_ASSERT(settings.SetValue("user_settings.tag_name", "user_app_settings", USER_ID_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
            UNIT_ASSERT(!driveApi.CheckCrossloginFlag(*perm, *server, session));
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(settings.SetValue("launch_head_app", "yandex-2", USER_ID_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
            UNIT_ASSERT(!driveApi.CheckCrossloginFlag(*perm, *server, session));
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(settings.SetValue("launch_head_app", "yandex-10", USER_ID_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
            UNIT_ASSERT(driveApi.CheckCrossloginFlag(*perm, *server, session));
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(settings.SetValue("launch_head_app", "all", USER_ID_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
            UNIT_ASSERT(driveApi.CheckCrossloginFlag(*perm, *server, session));
        }
    }

    /*
    Y_UNIT_TEST(FullScenarioHeadAppFlag) {
        InitGlobalLog2Console(TLOG_DEBUG);
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const THeadAccountManager& headManager = driveApi.GetHeadAccountManager();
        const ISettings& settings = server->GetSettings();

        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        auto car = eGenerator.CreateCar();

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(headManager.FinishSession(car.Id, session));
            UNIT_ASSERT(headManager.UpdateCarHeadId(car.Id, HEAD_ID_DEFAULT, USER_ID_DEFAULT, session, server.Get()));
            UNIT_ASSERT(session.Commit());
        }

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        NDrive::TTelematicsTestClient::TPtr client = emulator->GetClient();
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id));
        UNIT_ASSERT(configGenerator.WaitSensor(car.Id, "mileage"));
        UNIT_ASSERT(settings.SetValue("user_settings.tag_name", "user_app_settings", USER_ID_DEFAULT));
        UNIT_ASSERT(settings.SetValue("launch_head_app", "disable", USER_ID_DEFAULT));
        {
            TString offerId;
            {
                THolder<TStandartOffer> offer(new TStandartOffer);
                offer->SetObjectId(car.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetChargableAccounts({ "card", "bonus" });
                offerId = offer->GetOfferId();
                UNIT_ASSERT(driveApi.StoreOffers(&*server, {new TStandartOfferReport(offer.Release(), nullptr)}));
            }
            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            {
                NJson::TJsonValue userSessionReport = configGenerator.GetUserLastSession(USER_ID_DEFAULT);
                INFO_LOG << userSessionReport.GetStringRobust() << Endl;
                UNIT_ASSERT(!userSessionReport["sessions"][0]["segment"]["session"]["specials"]["is_cancelled"].GetBoolean());
            }
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                UNIT_ASSERT(headManager.SetPassportKey(USER_ID_DEFAULT, car.Id, "aaaaaaaaaaaaaaaa", session));
                session.Commit();
            }
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                const TString userInfo = headManager.GetDriveUserId(HEAD_ID_DEFAULT, session);
                auto authInfo = headManager.GetPassportKey(HEAD_ID_DEFAULT, session);
                UNIT_ASSERT_VALUES_EQUAL(userInfo, USER_ID_DEFAULT);
                UNIT_ASSERT_VALUES_EQUAL(authInfo.Status, "unknown");
                UNIT_ASSERT_VALUES_EQUAL(authInfo.Key, "");
            }
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        UNIT_ASSERT(settings.SetValue("launch_head_app", "all", USER_ID_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        {
            TString offerId;
            {
                THolder<TStandartOffer> offer(new TStandartOffer);
                offer->SetObjectId(car.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetChargableAccounts({ "card", "bonus" });
                offerId = offer->GetOfferId();
                UNIT_ASSERT(driveApi.StoreOffers(&*server, {new TStandartOfferReport(offer.Release(), nullptr)}));
            }
            UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                UNIT_ASSERT(headManager.SetPassportKey(USER_ID_DEFAULT, car.Id, "aaaaaaaaaaaaaaaa", session));
                session.Commit();
            }
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                auto userInfo = headManager.GetDriveUserId(HEAD_ID_DEFAULT, session);
                auto authInfo = headManager.GetPassportKey(HEAD_ID_DEFAULT, session);
                UNIT_ASSERT_VALUES_EQUAL(userInfo, USER_ID_DEFAULT);
                UNIT_ASSERT_VALUES_EQUAL(authInfo.Status, "new");
                UNIT_ASSERT_VALUES_EQUAL(authInfo.Key, "aaaaaaaaaaaaaaaa");
            }
            UNIT_ASSERT(configGenerator.EvolveTag("old_state_reservation", USER_ID_DEFAULT));
        }
    }
    */
}
