#include <drive/backend/ut/library/drivematics.h>
#include <drive/backend/ut/library/signalq.h>

#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/drivematics/signals/signal_configurations.h>
#include <drive/backend/drivematics/signals/tags.h>
#include <drive/backend/processors/leasing/add_cars.h>
#include <drive/backend/processors/leasing/get_signals.h>
#include <drive/backend/processors/leasing/signalq_status_history.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/rt_background/common_alerts/process.h>
#include <drive/backend/signalq/signals/tag.h>

#include <library/cpp/iterator/enumerate.h>
#include <library/cpp/testing/unittest/registar.h>

#include <random>

Y_UNIT_TEST_SUITE(LeasingScenarios) {

    void GenerateLeasingStatsData(NDrivematics::TLeasingStatsTag& tag, std::mt19937& mt) {
        std::uniform_real_distribution<double> utilizationURD(0, 1);
        std::uniform_real_distribution<double> simpleURD(0, 100);
        const int daysCount = 300;
        TVector<NDrivematics::TDailyLeasingStats> stats;
        stats.reserve(daysCount);
        for (int i = 0; i < daysCount; ++i) {
            NDrivematics::TDailyLeasingStats dailyStats;
            dailyStats.SetDay(10000 + i);
            dailyStats.SetSupplyHours(simpleURD(mt));
            dailyStats.SetUtilization(utilizationURD(mt));
            dailyStats.SetGMV(simpleURD(mt));
            dailyStats.SetParkCommission(simpleURD(mt));
            dailyStats.SetChurn(simpleURD(mt));
            stats.push_back(dailyStats);
        }
        tag.SetStats(MakeSet(stats));
        tag.SetIssueDate(10000);
    }

    Y_UNIT_TEST(SingleCarMetrics) {
        std::mt19937 mt(12313);
        NDrivematics::TLeasingStatsTag tag;
        GenerateLeasingStatsData(tag, mt);
        tag.SetAttenuation(12);
        auto stats = MakeVector(tag.GetStats());
        {
            const int queryDaysCount = 20;
            auto aggregatedStats = tag.AggregateStats(queryDaysCount);
            UNIT_ASSERT(aggregatedStats);

            double expUtilization = 0;
            double expSupplyHours = 0;
            double expGMV = 0;
            double expParkCommission = 0;
            int count = 0;
            for (auto i = stats.size() - 1; i >= 0 && count < queryDaysCount; --i, ++count) {
                const auto& dailyStats = stats[i];
                expUtilization += dailyStats.GetUtilization();
                expSupplyHours += dailyStats.GetSupplyHours();
                expGMV += dailyStats.GetGMV();
                expParkCommission += dailyStats.GetParkCommission();
            }
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetCarsCount(), 1);
            UNIT_ASSERT_DOUBLES_EQUAL(aggregatedStats->GetSupplyHours(), expSupplyHours, 1e-6);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetUtilization(), static_cast<ui32>(expUtilization * 100 / queryDaysCount));
            UNIT_ASSERT_DOUBLES_EQUAL(aggregatedStats->GetGMV(), expGMV, 1e-6);
            UNIT_ASSERT_DOUBLES_EQUAL(aggregatedStats->GetParkCommission(), expParkCommission, 1e-6);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetInsufficientSupplyHoursCarsCount(), 0);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetFreshIssueDateCarsCount(), 0);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetWeight(), 1);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetAttenuationRef(), 12);
        }
        {
            ui32 since = 10100;
            ui32 until = 10199;
            auto aggregatedStats = tag.AggregateStats(since, until);
            UNIT_ASSERT(aggregatedStats);

            double expUtilization = 0;
            double expSupplyHours = 0;
            double expGMV = 0;
            double expParkCommission = 0;
            for (const auto& dailyStats : stats) {
                if (since <= dailyStats.GetDay() && dailyStats.GetDay() <= until) {
                    expUtilization += dailyStats.GetUtilization();
                    expSupplyHours += dailyStats.GetSupplyHours();
                    expGMV += dailyStats.GetGMV();
                    expParkCommission += dailyStats.GetParkCommission();
                }
            }
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetCarsCount(), 1);
            UNIT_ASSERT_DOUBLES_EQUAL(aggregatedStats->GetSupplyHours(), expSupplyHours, 1e-6);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetUtilization(), static_cast<ui32>(expUtilization * 100 / 100));
            UNIT_ASSERT_DOUBLES_EQUAL(aggregatedStats->GetGMV(), expGMV, 1e-6);
            UNIT_ASSERT_DOUBLES_EQUAL(aggregatedStats->GetParkCommission(), expParkCommission, 1e-6);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetInsufficientSupplyHoursCarsCount(), 0);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetFreshIssueDateCarsCount(), 0);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetWeight(), 1);
            UNIT_ASSERT_VALUES_EQUAL(aggregatedStats->GetAttenuationRef(), 12);
        }
    }

    Y_UNIT_TEST(PortfolioHandler) {
        std::mt19937 mt(123123);
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        const ui32 noSHThreshold = 300;
        UNIT_ASSERT(server.Get()->GetSettings().SetValue("leasing_cabinet.supply_hours_threshold", ToString(noSHThreshold), USER_ROOT_DEFAULT));
        eGenerator.BuildEnvironment();
        const TString userId = eGenerator.CreateUser("leasing_guy", /* withRoles= */ false, "active");
        NDrivematics::TCarInfo carInfo;
        carInfo.SetTaxiCompanyTin(100);
        carInfo.SetLeasingCompanyName("xxx");
        carInfo.SetManufacturer("renault");
        carInfo.SetModel("captur");

        {

            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto action = MakeHolder<TTagAction>("observe_portfolio");
                action->AddTagAction(TTagAction::ETagAction::ObserveObject);
                action->AddTagAction(TTagAction::ETagAction::Observe);
                action->SetTagName("leasing_company_tag_" + ToString(FnvHash<ui64>(carInfo.GetLeasingCompanyName())));
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto roleHeader = TDriveRoleHeader{"observe_portfolio_role"};
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesDB().Upsert(roleHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("observe_portfolio").SetRoleId("observe_portfolio_role");
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TUserRole userRole;
                userRole.SetRoleId("observe_portfolio_role").SetUserId(userId);
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TTagDescription::TPtr tagDescription = new TTagDescription();
                tagDescription->SetName(NDrivematics::FreshIssueDateTagName);
                tagDescription->SetType(TDeviceTagRecord::TypeName);
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
            }
        }

        TVector<NDrivematics::TLeasingStatsTag> statsTags;
        for (int i = 0; i < 50; ++i) {
            const TString number = "12312" + ToString(i);
            carInfo.SetVIN("AAAA" + ToString(i));
            carInfo.SetNumber(number);
            if (i >= 25) {
                carInfo.SetHasTelematics(true);
            }
            auto reply = cg.AddLeasingCar(USER_ROOT_DEFAULT, carInfo);
            UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
            auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(NDrivematics::TLeasingStatsTag::TypeName);
            GenerateLeasingStatsData(*Yensured(dynamic_cast<NDrivematics::TLeasingStatsTag*>(tag.Get())), mt);
            statsTags.push_back(*dynamic_cast<NDrivematics::TLeasingStatsTag *>(tag.Get()));
            UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
            if (i % 25 >= 10 && i % 25 < 20) {
                auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(NDrivematics::FreshIssueDateTagName);
                UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
            }
        }
        UNIT_ASSERT(cg.LeasingAddScore(USER_ROOT_DEFAULT, carInfo.GetLeasingCompanyName(), 3.5, 1.1));
        auto segment = std::pair<ui32, ui32>{10100, 10199};
        Sleep(TDuration::Seconds(5));
        NUtil::THttpReply reply;
        for (int it = 0; it < 5; ++it) {
            reply = cg.GetLeasingPortfolio(userId, segment);
            if (reply.Code() != HTTP_OK) {
                Sleep(TDuration::Seconds(5));
            } else {
                break;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
        NJson::TJsonValue response;
        NJson::ReadJsonFastTree(reply.Content(), &response, /* throwOnError */ true);
        NDrivematics::TAggregatedLeasingStats allCarsAggregatedStats;
        NDrivematics::TAggregatedLeasingStats telematicsAggregatedStats;
        for (auto [idxStart, idxEnd, statsPtr, key] : TVector<std::tuple<size_t, size_t, NDrivematics::TAggregatedLeasingStats *, TString>>{
                 {0, 50, &allCarsAggregatedStats, "all_cars"},
                 {25, 50, &telematicsAggregatedStats, "telematics_cars"}}) {
            auto &stats = *statsPtr;
            TVector<NDrivematics::TAggregatedLeasingStats> aggregatedPerCar;
            for (size_t i = idxStart; i < idxEnd; ++i) {
                // do not count cars with fresh issue date tag
                if (i % 25 >= 10 && i % 25 < 20) {
                    continue;
                }
                auto aggregated = statsTags[i].AggregateStats(segment.first, segment.second);
                UNIT_ASSERT(aggregated);
                aggregatedPerCar.push_back(*aggregated);
            }

            stats.SetCarsCount(aggregatedPerCar.size());
            {
                double sumSupplyHours = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetSupplyHours();
                });
                stats.SetSupplyHours(sumSupplyHours / aggregatedPerCar.size());
            }
            {
                ui32 sumUtilization = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0, [](ui32 val, const auto& obj) {
                    return val + obj.GetUtilization();
                });
                stats.SetUtilization(sumUtilization / aggregatedPerCar.size());
            }
            {
                double sumGMV = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetGMV();
                });
                stats.SetGMV(sumGMV / aggregatedPerCar.size());
            }
            {
                double sumParkCommission = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetParkCommission();
                });
                stats.SetParkCommission(sumParkCommission / aggregatedPerCar.size());
            }
            {
                ui32 noSHCarsCount = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0, [](ui32 val, const auto& obj) {
                    auto res = val;
                    if (obj.GetSupplyHours() < noSHThreshold) {
                        ++res;
                    }
                    return res;
                });
                stats.SetInsufficientSupplyHoursCarsCount(noSHCarsCount);
            }
            {
                ui32 freshIssueDateCarsCount = idxEnd - idxStart - aggregatedPerCar.size();
                stats.SetFreshIssueDateCarsCount(freshIssueDateCarsCount);
            }
            UNIT_ASSERT_VALUES_EQUAL_C(response["portfolio"][0][key]["value"].GetInteger(), stats.GetUtilization(), response.GetStringRobust());
            UNIT_ASSERT_DOUBLES_EQUAL(response["portfolio"][1][key]["value"].GetDouble(), stats.GetSupplyHours(), 1e-6);
            UNIT_ASSERT_VALUES_EQUAL(response["portfolio"][2][key]["value"].GetInteger(), stats.GetCarsCount());
            UNIT_ASSERT_DOUBLES_EQUAL(response["portfolio"][3][key]["value"].GetDouble(), stats.GetGMV(), 1e-6);
            UNIT_ASSERT_DOUBLES_EQUAL(response["portfolio"][4][key]["value"].GetDouble(), stats.GetParkCommission(), 1e-6);
            UNIT_ASSERT_VALUES_EQUAL(response["portfolio"][5][key]["value"].GetInteger(), stats.GetInsufficientSupplyHoursCarsCount());
            UNIT_ASSERT_VALUES_EQUAL(response["portfolio"][6][key]["value"].GetInteger(), stats.GetFreshIssueDateCarsCount());
            UNIT_ASSERT_DOUBLES_EQUAL(response["score"].GetDouble(), 3.5, 1e-6);
        }
    }

    Y_UNIT_TEST(TaxiCompaniesHandler) {
        std::mt19937 mt(123123);
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        const ui32 noSHThreshold = 300;
        UNIT_ASSERT(server.Get()->GetSettings().SetValue("leasing_cabinet.supply_hours_threshold", ToString(noSHThreshold), USER_ROOT_DEFAULT));
        eGenerator.BuildEnvironment();
        const TString userId = eGenerator.CreateUser("leasing_guy", /* withRoles= */ false, "active");
        NDrivematics::TCarInfo carInfo;
        carInfo.SetTaxiCompanyTin(100);
        carInfo.SetLeasingCompanyName("xxx");
        carInfo.SetManufacturer("renault");
        carInfo.SetModel("captur");

        {
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto action = MakeHolder<TTagAction>("observe_taxi_company");
                action->AddTagAction(TTagAction::ETagAction::ObserveObject);
                action->AddTagAction(TTagAction::ETagAction::Observe);
                action->SetTagName("taxi_company_tag_" + ToString(carInfo.GetTaxiCompanyTin()));
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto roleHeader = TDriveRoleHeader{"observe_taxi_company_role"};
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesDB().Upsert(roleHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("observe_taxi_company").SetRoleId("observe_taxi_company_role");
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TUserRole userRole;
                userRole.SetRoleId("observe_taxi_company_role").SetUserId(userId);
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TTagDescription::TPtr tagDescription = new TTagDescription();
                tagDescription->SetName(NDrivematics::FreshIssueDateTagName);
                tagDescription->SetType(TDeviceTagRecord::TypeName);
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
            }
        }

        TVector<NDrivematics::TLeasingStatsTag> statsTags;
        for (int i = 0; i < 50; ++i) {
            const TString number = "12312" + ToString(i);
            carInfo.SetVIN("AAAA" + ToString(i));
            carInfo.SetNumber(number);
            if (i >= 25) {
                carInfo.SetHasTelematics(true);
            }
            auto reply = cg.AddLeasingCar(USER_ROOT_DEFAULT, carInfo);
            UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
            auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(NDrivematics::TLeasingStatsTag::TypeName);
            GenerateLeasingStatsData(*Yensured(dynamic_cast<NDrivematics::TLeasingStatsTag*>(tag.Get())), mt);
            statsTags.push_back(*dynamic_cast<NDrivematics::TLeasingStatsTag*>(tag.Get()));
            UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
            if (i % 25 >= 10 && i % 25 < 20) {
                auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(NDrivematics::FreshIssueDateTagName);
                UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
            }
        }

        auto segment = std::pair<ui32, ui32>{10100, 10199};
        Sleep(TDuration::Seconds(5));
        NUtil::THttpReply reply;
        for (int it = 0; it < 5; ++it) {
            reply = cg.GetLeasingTaxiCompanies(userId, segment);
            if (reply.Code() != HTTP_OK) {
                Sleep(TDuration::Seconds(5));
            } else {
                break;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
        NJson::TJsonValue response;
        NJson::ReadJsonFastTree(reply.Content(), &response, /* throwOnError */ true);
        NDrivematics::TAggregatedLeasingStats allCarsAggregatedStats;
        NDrivematics::TAggregatedLeasingStats telematicsAggregatedStats;
        for (auto [idxStart, idxEnd, statsPtr, key] : TVector<std::tuple<size_t, size_t, NDrivematics::TAggregatedLeasingStats *, TStringBuf>>{
                 {0, 50, &allCarsAggregatedStats, "all_cars"},
                 {25, 50, &telematicsAggregatedStats, "telematics_cars"}}) {
            auto &stats = *statsPtr;
            TVector<NDrivematics::TAggregatedLeasingStats> aggregatedPerCar;
            for (size_t i = idxStart; i < idxEnd; ++i) {
                // do not count cars with fresh issue date tag
                if (i % 25 >= 10 && i % 25 < 20) {
                    continue;
                }
                auto aggregated = statsTags[i].AggregateStats(segment.first, segment.second);
                UNIT_ASSERT(aggregated);
                aggregatedPerCar.push_back(*aggregated);
            }

            stats.SetCarsCount(aggregatedPerCar.size());
            {
                double sumSupplyHours = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetSupplyHours();
                });
                stats.SetSupplyHours(sumSupplyHours / aggregatedPerCar.size());
            }
            {
                ui32 sumUtilization = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0, [](ui32 val, const auto& obj) {
                    return val + obj.GetUtilization();
                });
                stats.SetUtilization(sumUtilization / aggregatedPerCar.size());
            }
            {
                double sumGMV = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetGMV();
                });
                stats.SetGMV(sumGMV);
            }
            {
                double sumParkCommission = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetParkCommission();
                });
                stats.SetParkCommission(sumParkCommission);
            }
            ForEach(stats.GetFields(), [response, key = key](auto&& field) {
                if (!(field.GetTraits() & NDrivematics::ReportInTaxiCompanies)) {
                    UNIT_ASSERT(!response["parks"][0][key].Has(field.GetName()));
                    return;
                }
                if (field.GetName() != "cars_count" && field.GetName() != "utilization") {
                    UNIT_ASSERT_DOUBLES_EQUAL_C(response["parks"][0][key][field.GetName()]["value"].GetDouble(), field.Value, 1e-4, response.GetStringRobust());
                } else {
                    UNIT_ASSERT_VALUES_EQUAL_C(response["parks"][0][key][field.GetName()]["value"].GetInteger(), field.Value, response.GetStringRobust());
                }
            });
        }
    }

    Y_UNIT_TEST(TaxiCompaniesWeights) {
        std::mt19937 mt(123123);
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        const ui32 noSHThreshold = 300;
        UNIT_ASSERT(server.Get()->GetSettings().SetValue("leasing_cabinet.supply_hours_threshold", ToString(noSHThreshold), USER_ROOT_DEFAULT));
        eGenerator.BuildEnvironment();
        const TString userId = eGenerator.CreateUser("leasing_guy", /* withRoles= */ false, "active");
        NDrivematics::TCarInfo carInfo;
        carInfo.SetTaxiCompanyTin(100);
        carInfo.SetLeasingCompanyName("fake");
        carInfo.SetManufacturer("renault");
        carInfo.SetModel("captur");

        {
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto action = MakeHolder<TTagAction>("observe_taxi_company");
                action->AddTagAction(TTagAction::ETagAction::Observe);
                action->SetTagName("taxi_company_tag_" + ToString(carInfo.GetTaxiCompanyTin()));
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto roleHeader = TDriveRoleHeader{"observe_taxi_company_role"};
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesDB().Upsert(roleHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("observe_taxi_company").SetRoleId("observe_taxi_company_role");
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TUserRole userRole;
                userRole.SetRoleId("observe_taxi_company_role").SetUserId(userId);
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TTagDescription::TPtr tagDescription = new TTagDescription();
                tagDescription->SetName(NDrivematics::FreshIssueDateTagName);
                tagDescription->SetType(TDeviceTagRecord::TypeName);
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
            }
        }

        TVector<NDrivematics::TLeasingStatsTag> statsTags;
        for (int i = 0; i < 50; ++i) {
            const TString number = "12312" + ToString(i);
            carInfo.SetVIN("AAAA" + ToString(i));
            carInfo.SetNumber(number);
            if (i >= 25) {
                carInfo.SetHasTelematics(true);
            }
            auto reply = cg.AddLeasingCar(USER_ROOT_DEFAULT, carInfo);
            UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
            auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(NDrivematics::TLeasingStatsTag::TypeName);
            auto &leasingTag = *Yensured(dynamic_cast<NDrivematics::TLeasingStatsTag*>(tag.Get()));
            GenerateLeasingStatsData(leasingTag, mt);
            leasingTag.SetWeight(std::uniform_int_distribution{1, 10}(mt));
            if (std::uniform_int_distribution{0, 1}(mt)) {
                leasingTag.SetAttenuation(std::uniform_real_distribution{0.1, 0.9}(mt));
            }
            statsTags.push_back(leasingTag);
            UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
            if (i % 25 >= 10 && i % 25 < 20) {
                auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(NDrivematics::FreshIssueDateTagName);
                UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
            }
        }

        auto segment = std::pair<ui32, ui32>{10100, 10199};
        Sleep(TDuration::Seconds(5));
        NUtil::THttpReply reply;
        for (int it = 0; it < 5; ++it) {
            reply = cg.GetLeasingTaxiCompanies(userId, segment);
            if (reply.Code() != HTTP_OK) {
                Sleep(TDuration::Seconds(5));
            } else {
                break;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
        NJson::TJsonValue response;
        NJson::ReadJsonFastTree(reply.Content(), &response, /* throwOnError */ true);
        NDrivematics::TAggregatedLeasingStats allCarsAggregatedStats;
        NDrivematics::TAggregatedLeasingStats telematicsAggregatedStats;
        for (auto [idxStart, idxEnd, statsPtr, key] : TVector<std::tuple<size_t, size_t, NDrivematics::TAggregatedLeasingStats *, TStringBuf>>{
                 {0, 50, &allCarsAggregatedStats, "all_cars"},
                 {25, 50, &telematicsAggregatedStats, "telematics_cars"}}) {
            auto &stats = *statsPtr;
            TVector<NDrivematics::TAggregatedLeasingStats> aggregatedPerCar;
            ui32 weightsSum = 0;
            for (size_t i = idxStart; i < idxEnd; ++i) {
                // do not count cars with fresh issue date tag
                if (i % 25 >= 10 && i % 25 < 20) {
                    continue;
                }
                auto aggregated = statsTags[i].AggregateStats(segment.first, segment.second);
                UNIT_ASSERT(aggregated);
                weightsSum += aggregated->GetWeight();
                aggregatedPerCar.push_back(*aggregated);
            }

            stats.SetCarsCount(weightsSum);
            {
                double sumSupplyHours = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetSupplyHours() * obj.GetWeight() * obj.OptionalAttenuation().GetOrElse(1);
                });
                stats.SetSupplyHours(sumSupplyHours / weightsSum);
            }
            {
                ui32 sumUtilization = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0, [](ui32 val, const auto& obj) {
                    return val + static_cast<ui32>(obj.GetUtilization() * obj.GetWeight() * obj.OptionalAttenuation().GetOrElse(1));
                });
                stats.SetUtilization(sumUtilization / weightsSum);
            }
            {
                double sumGMV = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetGMV() * obj.GetWeight() * obj.OptionalAttenuation().GetOrElse(1);
                });
                stats.SetGMV(sumGMV);
            }
            {
                double sumParkCommission = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetParkCommission() * obj.GetWeight() * obj.OptionalAttenuation().GetOrElse(1);
                });
                stats.SetParkCommission(sumParkCommission);
            }
            ForEach(stats.GetFields(), [response, key = key](auto&& field) {
                if (!(field.GetTraits() & NDrivematics::ReportInTaxiCompanies)) {
                    UNIT_ASSERT(!response["parks"][0][key].Has(field.GetName()));
                    return;
                }
                if (field.GetName() != "cars_count" && field.GetName() != "utilization") {
                    UNIT_ASSERT_DOUBLES_EQUAL_C(response["parks"][0][key][field.GetName()]["value"].GetDouble(), field.Value, 1e-4, response.GetStringRobust());
                } else {
                    UNIT_ASSERT_VALUES_EQUAL_C(response["parks"][0][key][field.GetName()]["value"].GetInteger(), field.Value, response.GetStringRobust());
                }
            });
        }
    }

    Y_UNIT_TEST(PortfolioLeasingCompanyNoAccess) {
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        const ui32 noSHThreshold = 300;
        UNIT_ASSERT(server.Get()->GetSettings().SetValue("leasing_cabinet.supply_hours_threshold", ToString(noSHThreshold), USER_ROOT_DEFAULT));
        eGenerator.BuildEnvironment();
        const TString userId = eGenerator.CreateUser("leasing_guy", /* withRoles= */ false, "active");
        NDrivematics::TCarInfo carInfo;
        carInfo.SetTaxiCompanyTin(100);
        carInfo.SetLeasingCompanyName("xxx");
        carInfo.SetManufacturer("renault");
        carInfo.SetModel("captur");

        NUtil::THttpReply reply = cg.GetLeasingPortfolio(userId);
        UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_FORBIDDEN, reply.Content());
    }

    Y_UNIT_TEST(PortfolioHandlerNoScoreNoCars) {
        std::mt19937 mt(123123);
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        const ui32 noSHThreshold = 300;
        UNIT_ASSERT(server.Get()->GetSettings().SetValue("leasing_cabinet.supply_hours_threshold", ToString(noSHThreshold), USER_ROOT_DEFAULT));
        eGenerator.BuildEnvironment();
        const TString userId = eGenerator.CreateUser("leasing_guy", /* withRoles= */ false, "active");
        NDrivematics::TCarInfo carInfo;
        carInfo.SetTaxiCompanyTin(100);
        carInfo.SetLeasingCompanyName("fake");
        carInfo.SetManufacturer("renault");
        carInfo.SetModel("captur");

        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            {
                auto action = MakeHolder<TTagAction>("observe_portfolio");
                action->AddTagAction(TTagAction::ETagAction::ObserveObject);
                action->SetTagName("leasing_company_tag_" + ToString(FnvHash<ui64>(carInfo.GetLeasingCompanyName())));
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
            }
            {
                auto roleHeader = TDriveRoleHeader{"observe_portfolio_role"};
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesDB().Upsert(roleHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("observe_portfolio").SetRoleId("observe_portfolio_role");
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
            }
            {
                TUserRole userRole;
                userRole.SetRoleId("observe_portfolio_role").SetUserId(userId);
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            }
            {
                TTagDescription::TPtr tagDescription = new TTagDescription();
                tagDescription->SetName(NDrivematics::FreshIssueDateTagName);
                tagDescription->SetType(TDeviceTagRecord::TypeName);
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
            }
            UNIT_ASSERT(session.Commit());
        }

        for (int i = 0; i < 1; ++i) {
            const TString number = "12312" + ToString(i);
            carInfo.SetVIN("AAAA" + ToString(i));
            carInfo.SetNumber(number);
            auto reply = cg.AddLeasingCar(USER_ROOT_DEFAULT, carInfo);
            UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
        }

        auto segment = std::pair<ui32, ui32>{10100, 10199};
        Sleep(TDuration::Seconds(5));
        NUtil::THttpReply reply;
        for (int it = 0; it < 5; ++it) {
            reply = cg.GetLeasingPortfolio(userId, segment);
            if (reply.Code() != HTTP_OK) {
                Sleep(TDuration::Seconds(5));
            } else {
                break;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
        NJson::TJsonValue response;
        NJson::ReadJsonFastTree(reply.Content(), &response, /* throwOnError */ true);
        auto stats = NDrivematics::TAggregatedLeasingStats{};
        for (TString key : {"all_cars", "telematics_cars"}) {
            UNIT_ASSERT(response["portfolio"][0][key]["value"].IsInteger());
            UNIT_ASSERT_VALUES_EQUAL(response["portfolio"][0][key]["value"].GetInteger(), stats.GetUtilization());
            UNIT_ASSERT(response["portfolio"][1][key]["value"].IsDouble());
            UNIT_ASSERT_DOUBLES_EQUAL(response["portfolio"][1][key]["value"].GetDouble(), stats.GetSupplyHours(), 1e-6);
            UNIT_ASSERT(response["portfolio"][2][key]["value"].IsInteger());
            UNIT_ASSERT_VALUES_EQUAL(response["portfolio"][2][key]["value"].GetInteger(), stats.GetCarsCount());
            UNIT_ASSERT(response["portfolio"][3][key]["value"].IsDouble());
            UNIT_ASSERT_DOUBLES_EQUAL(response["portfolio"][3][key]["value"].GetDouble(), stats.GetGMV(), 1e-6);
            UNIT_ASSERT(response["portfolio"][4][key]["value"].IsDouble());
            UNIT_ASSERT_DOUBLES_EQUAL(response["portfolio"][4][key]["value"].GetDouble(), stats.GetParkCommission(), 1e-6);
            UNIT_ASSERT(response["portfolio"][5][key]["value"].IsInteger());
            UNIT_ASSERT_VALUES_EQUAL(response["portfolio"][5][key]["value"].GetInteger(), stats.GetInsufficientSupplyHoursCarsCount());
            UNIT_ASSERT(response["portfolio"][6][key]["value"].IsInteger());
            UNIT_ASSERT_VALUES_EQUAL(response["portfolio"][6][key]["value"].GetInteger(), stats.GetFreshIssueDateCarsCount());
            UNIT_ASSERT(!response.Has("score"));
        }
    }

    Y_UNIT_TEST(TaxiCompaniesVisibility) {
        std::mt19937 mt(123123);
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        const ui32 noSHThreshold = 300;
        UNIT_ASSERT(server.Get()->GetSettings().SetValue("leasing_cabinet.supply_hours_threshold", ToString(noSHThreshold), USER_ROOT_DEFAULT));
        eGenerator.BuildEnvironment();
        const TString userId = eGenerator.CreateUser("leasing_guy", /* withRoles= */ false, "active");
        NDrivematics::TCarInfo carInfo;
        carInfo.SetTaxiCompanyTin(100);
        carInfo.SetLeasingCompanyName("xxx");
        carInfo.SetManufacturer("renault");
        carInfo.SetModel("captur");

        {
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto action = MakeHolder<TTagAction>("observe_taxi_company");
                action->AddTagAction(TTagAction::ETagAction::Observe);
                action->SetTagName("taxi_company_tag_" + ToString(carInfo.GetTaxiCompanyTin()));
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto roleHeader = TDriveRoleHeader{"observe_taxi_company_role"};
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesDB().Upsert(roleHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("observe_taxi_company").SetRoleId("observe_taxi_company_role");
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TUserRole userRole;
                userRole.SetRoleId("observe_taxi_company_role").SetUserId(userId);
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TTagDescription::TPtr tagDescription = new TTagDescription();
                tagDescription->SetName(NDrivematics::FreshIssueDateTagName);
                tagDescription->SetType(TDeviceTagRecord::TypeName);
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
                UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
            }
        }

        TVector<NDrivematics::TLeasingStatsTag> statsTags;
        for (int i = 0; i < 10; ++i) {
            const TString number = "12312" + ToString(i);
            carInfo.SetVIN("AAAA" + ToString(i));
            carInfo.SetNumber(number);
            if (i >= 5) {
                carInfo.SetTaxiCompanyTin(50);
            }

            auto reply = cg.AddLeasingCar(USER_ROOT_DEFAULT, carInfo);
            UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
            auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(NDrivematics::TLeasingStatsTag::TypeName);
            GenerateLeasingStatsData(*Yensured(dynamic_cast<NDrivematics::TLeasingStatsTag*>(tag.Get())), mt);
            statsTags.push_back(*dynamic_cast<NDrivematics::TLeasingStatsTag*>(tag.Get()));
            UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
        }

        auto segment = std::pair<ui32, ui32>{10100, 10199};
        Sleep(TDuration::Seconds(5));
        NUtil::THttpReply reply;
        for (int it = 0; it < 5; ++it) {
            reply = cg.GetLeasingTaxiCompanies(userId, segment);
            if (reply.Code() != HTTP_OK) {
                Sleep(TDuration::Seconds(5));
            } else {
                break;
            }
        }

        UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
        NJson::TJsonValue response;
        NJson::ReadJsonFastTree(reply.Content(), &response, /* throwOnError */ true);
        NDrivematics::TAggregatedLeasingStats allCarsAggregatedStats;
        for (auto [idxStart, idxEnd, statsPtr, key] : TVector<std::tuple<size_t, size_t, NDrivematics::TAggregatedLeasingStats *, TStringBuf>>{
                 {0, 5, &allCarsAggregatedStats, "all_cars"},
             }) {
            auto &stats = *statsPtr;
            TVector<NDrivematics::TAggregatedLeasingStats> aggregatedPerCar;
            for (size_t i = idxStart; i < idxEnd; ++i) {
                // do not count cars with fresh issue date tag
                auto aggregated = statsTags[i].AggregateStats(segment.first, segment.second);
                UNIT_ASSERT(aggregated);
                aggregatedPerCar.push_back(*aggregated);
            }

            stats.SetCarsCount(aggregatedPerCar.size());
            {
                double sumSupplyHours = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetSupplyHours();
                });
                stats.SetSupplyHours(sumSupplyHours / aggregatedPerCar.size());
            }
            {
                ui32 sumUtilization = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0, [](ui32 val, const auto& obj) {
                    return val + obj.GetUtilization();
                });
                stats.SetUtilization(sumUtilization / aggregatedPerCar.size());
            }
            {
                double sumGMV = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetGMV();
                });
                stats.SetGMV(sumGMV);
            }
            {
                double sumParkCommission = std::accumulate(aggregatedPerCar.begin(), aggregatedPerCar.end(), 0.0, [](double val, const auto& obj) {
                    return val + obj.GetParkCommission();
                });
                stats.SetParkCommission(sumParkCommission);
            }
            ForEach(stats.GetFields(), [response, key = key](auto&& field) {
                if (!(field.GetTraits() & NDrivematics::ReportInTaxiCompanies)) {
                    UNIT_ASSERT(!response["parks"][0][key].Has(field.GetName()));
                    return;
                }
                if (field.GetName() != "cars_count" && field.GetName() != "utilization") {
                    UNIT_ASSERT_DOUBLES_EQUAL_C(response["parks"][0][key][field.GetName()]["value"].GetDouble(), field.Value, 1e-4, response.GetStringRobust());
                } else {
                    UNIT_ASSERT_VALUES_EQUAL_C(response["parks"][0][key][field.GetName()]["value"].GetInteger(), field.Value, response.GetStringRobust());
                }
            });
        }
    }

    Y_UNIT_TEST(Signals) {
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        TVector<NDrivematics::TSignalDescription> signalsDescriptions;
        signalsDescriptions.emplace_back().SetName("signal_tag1").SetDisplayName("Сигнальный тег 1").SetPriority(NDrivematics::ISignalConfiguration::ESignalPriority::Normal).SetIsEditable(false);
        signalsDescriptions.emplace_back().SetName("signal_tag2").SetDisplayName("Сигнальный тег 2").SetPriority(NDrivematics::ISignalConfiguration::ESignalPriority::Normal).SetIsEditable(false);

        UNIT_ASSERT(server.Get()->GetSettings().SetValue(NDrive::TGetSignalsProcessor::SignalsDescriptionSettingName, NJson::ToJson(NJson::TJsonValue{NJson::TMapBuilder("descriptions", NJson::ToJson(signalsDescriptions))}).GetStringRobust(), USER_ROOT_DEFAULT));
        eGenerator.BuildEnvironment();
        const TString userId = eGenerator.CreateUser("leasing_guy", /* withRoles= */ false, "active");
        NDrivematics::TCarInfo carInfo;
        carInfo.SetTaxiCompanyTin(100);
        carInfo.SetLeasingCompanyName("xxx");
        carInfo.SetManufacturer("renault");
        carInfo.SetModel("captur");
        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            for (const auto& signalDesc : signalsDescriptions) {
                TTagDescription::TPtr tagDescription = new TTagDescription();
                tagDescription->SetName(signalDesc.GetName());
                tagDescription->SetType(NDrivematics::TGenericSignalCarTag::Type());
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
            }
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto action = MakeHolder<TTagAction>("observe_taxi_company");
            action->AddTagAction(TTagAction::ETagAction::ObserveObject);
            action->SetTagName("taxi_company_tag_" + ToString(carInfo.GetTaxiCompanyTin()));
            UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto action = MakeHolder<TTagAction>("observe_signals");
            action->AddTagAction(TTagAction::ETagAction::Observe);
            action->SetTagName("$signal_tag(1|2)");
            UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto roleHeader = TDriveRoleHeader{"observe_taxi_company_and_signals_role"};
            UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesDB().Upsert(roleHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            for (const TString &action : {"observe_taxi_company", "observe_signals"}) {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId(action).SetRoleId("observe_taxi_company_and_signals_role");
                UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session), session.GetStringReport());
            }
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("observe_taxi_company_and_signals_role").SetUserId(userId);
            UNIT_ASSERT(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(AddCompanyTag(userId, *server));

        TMap<TString, TString> carNumberToSignalName;
        TVector<std::pair<TString, TString>> carSignals;
        for (int i = 0; i < 9; ++i) {
            const TString number = "12312" + ToString(i);
            carInfo.SetVIN("AAAA" + ToString(i));
            carInfo.SetNumber(number);
            carInfo.SetTaxiCompanyTin(carInfo.GetTaxiCompanyTin());
            auto reply = cg.AddLeasingCar(USER_ROOT_DEFAULT, carInfo);
            UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());
            const TString signalName = (i < 7) ? signalsDescriptions[0].GetName() : signalsDescriptions[1].GetName();
            carNumberToSignalName[number] = signalName;
            carSignals.emplace_back(number, signalName);
            auto tag = server.Get()->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(signalName);
            UNIT_ASSERT_C(tag, "cannot create tag " << signalName);
            UNIT_ASSERT(cg.AddCarTag(tag, number, USER_ROOT_DEFAULT));
        }
        std::reverse(carSignals.begin(), carSignals.end());

        TSet<TString> allSignalNames{signalsDescriptions[0].GetName(), signalsDescriptions[1].GetName()};
        const int pageSize = 4;
        auto check = [&](const TSet<TString>& signalNames) {
            TSet<TString> reportedSignalNames = allSignalNames;
            if (!signalNames.empty()) {
                reportedSignalNames = signalNames;
            }
            TVector<std::pair<TString, TString>> actualCarSignals;
            for (const auto &elem : carSignals) {
                if (reportedSignalNames.contains(elem.second)) {
                    actualCarSignals.push_back(elem);
                }
            }
            bool canGetMorePages = true;
            TMaybe<ui64> carsCursor;
            for (int e = 0; e < 3 && canGetMorePages; ++e) {
                int lft = e * pageSize;
                int rgt = std::min<int>(actualCarSignals.size(), (e + 1) * pageSize);
                auto reply = cg.GetSignals(userId, MakeVector(signalNames), pageSize, carsCursor);
                UNIT_ASSERT_VALUES_EQUAL_C(reply.Code(), HTTP_OK, reply.Content());

                NJson::TJsonValue response;
                NJson::ReadJsonTree(reply.Content(), &response, /* throwOnError */ true);

                UNIT_ASSERT_VALUES_EQUAL(response["signals_descriptions"].GetMap().size(), 2);
                for (const auto &[name, desc] : response["signals_descriptions"].GetMap()) {
                    UNIT_ASSERT(allSignalNames.contains(name));
                    for (auto&& field : {
                        "is_editable",
                        "is_enabled",
                        "is_permanent",
                        "name",
                        "priority",
                        "source",
                    }) {
                        if (name == signalsDescriptions[0].GetName()) {
                            UNIT_ASSERT_VALUES_EQUAL(desc[field], NJson::ToJson(signalsDescriptions[0])[field]);
                        } else {
                            UNIT_ASSERT_VALUES_EQUAL(desc[field], NJson::ToJson(signalsDescriptions[1])[field]);
                        }
                    }
                }

                size_t expectedSignalsCount = rgt - lft;

                UNIT_ASSERT_VALUES_EQUAL(response["signals"].GetArray().size(), expectedSignalsCount);
                for (const auto &[signalIdx, signal] : Enumerate(response["signals"].GetArray())) {
                    UNIT_ASSERT_VALUES_EQUAL(signal["linked_entities"].GetArray().size(), 1);
                    UNIT_ASSERT_VALUES_EQUAL(signal["linked_entities"][0]["entity_type"].GetString(), "car");
                    auto carNumber = signal["linked_entities"][0]["details"]["number"].GetString();
                    UNIT_ASSERT_VALUES_EQUAL(carNumber, actualCarSignals[lft + signalIdx].first);
                    auto it = carNumberToSignalName.find(carNumber);
                    UNIT_ASSERT(it != carNumberToSignalName.end());
                    UNIT_ASSERT_VALUES_EQUAL(signal["name"].GetString(), it->second);
                    UNIT_ASSERT(reportedSignalNames.contains(signal["name"].GetString()));
                }
                if (response["next_cars_cursor"].IsDefined()) {
                    carsCursor = response["next_cars_cursor"].GetInteger();
                }
                UNIT_ASSERT(response["can_get_more_pages"].IsDefined());
                canGetMorePages = response["can_get_more_pages"].GetBoolean();
            }
            UNIT_ASSERT(!canGetMorePages);
        };
        check({});
        check({ "signal_tag1" });
        check({ "signal_tag2" });
        check({ "signal_tag1", "signal_tag2" });
    }

    Y_UNIT_TEST(SignalConfiguration) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>().SetNeedTelematics(false);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);


        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());
        script.Add<NDrive::NTest::TRide>(TDuration::Zero());
        script.Add<NDrive::NTest::TInitSessionId>();
        script.Add<NDrive::NTest::TDrop>().SetExpectOK(true);

        auto maxEventId = script.GetDriveAPI().GetTagsManager().GetTraceTags().GetHistoryManager().GetLockedMaxEventId();
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            const auto sessionId = context.GetCurrentSessionIdUnsafe();
            auto tagDescription = MakeAtomicShared<TScoringTraceTag::TDescription>();
            tagDescription->SetName("signal_my_unique_id");
            tagDescription->SetType(TScoringTraceTag::TypeName);
            tagDescription->SetKind(IScoringBaseTag::EScoringKind::Speeding);
            tagDescription->SetSignalEnabled(true);
            const auto& tagsMeta = context.GetServer()->GetDriveAPI()->GetTagsManager().GetTagsMeta();
            UNIT_ASSERT_C(tagsMeta.RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
            tagDescription->SetName("speeding_trace_tag");
            UNIT_ASSERT_C(tagsMeta.RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
            Sleep(TDuration::Seconds(3));
            auto tag = MakeHolder<TScoringTraceTag>();
            tag->SetName("speeding_trace_tag");
            tag->SetValue(100);

            const auto& traceTagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager().GetTraceTags();
            session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(traceTagsManager.AddTag(tag.Release(), USER_ROOT_DEFAULT, sessionId, context.GetServer().Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        });
        script.Add<NDrive::NTest::TCommonChecker>([maxEventId](NDrive::NTest::TRTContext& context) {
            const auto sessionId = context.GetCurrentSessionIdUnsafe();
            const TString jsonStr = R"({
                "speed_threshold": 42,
                "type": "speeding_signal",
                "display_name": "Тестовый сигнал"
            })";
            NJson::TJsonValue jsonConfig;
            UNIT_ASSERT(ReadJsonFastTree(jsonStr, &jsonConfig));
            TMessagesCollector errors;
            auto signalConfig = NDrivematics::ISignalConfiguration::ConstructConfiguration(jsonConfig, &errors, *context.GetServer());
            UNIT_ASSERT_C(signalConfig, errors.GetStringReport());
            signalConfig->SetSignalId("my_unique_id");
            auto vector = signalConfig->ConstructBackgroundProcesses();
            UNIT_ASSERT(vector.size() == 1);
            auto robotStateImpl = MakeAtomicShared<TRTCommonAlertsState>();
            robotStateImpl->SetLastEventId(maxEventId);
            UNIT_ASSERT(vector.front().GetProcessSettings()->Execute(robotStateImpl, *context.GetServer().Get()));
            auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            const auto& traceTagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager().GetTraceTags();
            TTaggedObjectsSnapshot traceObjects;
            UNIT_ASSERT_C(traceTagsManager.RestoreObjects({ sessionId }, traceObjects, session), session.GetStringReport());
            auto* traceObject = traceObjects.Get(sessionId);
            UNIT_ASSERT(traceObject);
            auto tag = traceObject->GetTag("signal_my_unique_id");
            UNIT_ASSERT(tag);
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(TelematicsLagSignalConfiguration) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);
        configGenerator.SetSensorApiName({});
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        const auto& carTagsManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
        eg.BuildEnvironment();
        auto car = eg.CreateCar();
        TGeoCoord someCoord(12.58643489401084, 45.73761541977004);
        {
            auto pusher = server->GetSensorApi()->GetPusher();
            NDrive::TSensor sensor(VEGA_SPEED);
            sensor.Timestamp = Now() - TDuration::Hours(10);
            sensor.Since = Now() - TDuration::Hours(10);
            sensor.Value = 123.0;
            pusher->Push(car.IMEI, sensor);
            auto sensors = server->GetSensorApi()->GetSensor(car.IMEI, sensor.Id);
            UNIT_ASSERT(sensors.HasValue());
            UNIT_ASSERT_VALUES_EQUAL(std::get<double>(sensors.GetValue()->Value), 123);
        }
        {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName("owning_car_tag");
            tagDescription->SetType(TDeviceTagRecord::TypeName);
            const auto& tagsMeta = server->GetDriveAPI()->GetTagsManager().GetTagsMeta();
            auto tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(tagsMeta.RegisterTag(tagDescription, USER_ROOT_DEFAULT, tx), tx.GetStringReport());
            UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
            Sleep(TDuration::Seconds(3));
            tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            {
                auto tag = MakeHolder<TDeviceTagRecord>();
                tag->SetName("owning_car_tag");
                UNIT_ASSERT_C(carTagsManager.AddTag(tag.Release(), USER_ROOT_DEFAULT, car.Id, server.Get(), tx), tx.GetStringReport());
            }
            {
                auto tag = MakeHolder<NDrivematics::THasTelematicsTag>();
                tag->SetName(NDrivematics::THasTelematicsTag::TypeName);
                UNIT_ASSERT_C(carTagsManager.AddTag(tag.Release(), USER_ROOT_DEFAULT, car.Id, server.Get(), tx), tx.GetStringReport());
            }
            UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
        }
        auto tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

        const TString jsonStr = R"({
            "type": "telematics_lag_signal",
            "display_name": "Сигнал телематики",
            "activation_threshold": 6000
        })";
        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(jsonStr, &jsonConfig));
        TMessagesCollector errors;
        auto configuration = NDrivematics::ISignalConfiguration::ConstructConfiguration(jsonConfig, &errors, *server);
        configuration->SetSignalId("my_unique_id");
        configuration->SetCarsFilter(TCarsFilter{});
        configuration->MutableCarsFilterRef().SetIncludeTagsFilter(TTagsFilter::BuildFromString("owning_car_tag"));
        configuration->CreateTags(USER_ROOT_DEFAULT, *server, tx);
        UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
        // wait for snapshot manager to update
        Sleep(TDuration::Seconds(5));
        tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        auto vector = configuration->ConstructBackgroundProcesses();
        UNIT_ASSERT(vector.size() == 1);
        UNIT_ASSERT(vector.front().GetProcessSettings()->Execute(nullptr, *server));
        auto optionalTaggedCar = carTagsManager.RestoreObject(car.Id, tx);
        UNIT_ASSERT_C(optionalTaggedCar, tx.GetStringReport());
        auto tag = optionalTaggedCar->GetTag("signal_my_unique_id");
        UNIT_ASSERT(tag);
    }

    Y_UNIT_TEST(SignalqStatusHistory) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& carId = OBJECT_ID_DEFAULT;

        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, SIGNAL_DEVICE_SN_DEFAULT, true, USER_ROOT_DEFAULT).Has("error_details"));

        NJson::TJsonValue postData = NJson::JSON_MAP;
        postData["car_id"] = carId;
        const auto now = Now();
        NJson::TJsonValue period;
        period["from"] = (now - TDuration::Days(1)).Seconds();
        period["to"] = now.Seconds();
        postData["period"] = period;

        UNIT_ASSERT(env->SignalqStatusHistory(postData, USER_ROOT_DEFAULT).first == 200);

        period["from"] = (now + TDuration::Days(1)).Seconds();
        postData["period"] = std::move(period);

        UNIT_ASSERT(env->SignalqStatusHistory(postData, USER_ROOT_DEFAULT).first == 400);
    }

    Y_UNIT_TEST(SignalqSetResolutionFromDrivematics) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSignalqSessionMetadata(*env.GetServer().Get());

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& api = *Yensured(env.GetServer()->GetDriveAPI());
        const auto carId = TString{OBJECT_ID_DEFAULT};
        const auto userId = TString{USER_ID_DEFAULT};
        const TString signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;
        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_events"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& traceTagManager = api.GetTagsManager().GetTraceTags();
        const auto offer = env.GetServer()->GetDriveAPI()->GetCurrentCarOffer(carId);
        UNIT_ASSERT(!offer);
        NJson::TJsonValue postData = NJson::JSON_MAP;
        NJson::TJsonValue& events =  postData.InsertValue("events", NJson::JSON_ARRAY);
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "event_id_with_actual_session_gap", Now()));
        env->SignalqSignalCreate(postData, USER_ROOT_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        TString tagId;
        TString offerId;
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            const auto offer = env.GetServer()->GetDriveAPI()->GetCurrentCarOffer(carId);
            UNIT_ASSERT(offer);
            offerId = offer->GetOfferId();
            const auto taggedSession = traceTagManager.RestoreObject(offerId, tx);
            UNIT_ASSERT(taggedSession);
            const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();
            auto signalqTraceTag = tags.front().GetTagAs<TSignalqEventTraceTag>();
            UNIT_ASSERT(!signalqTraceTag->GetResolution());
        }
        env->EvolveTag("old_state_reservation", userId);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        UNIT_ASSERT(!env->SignalqTraceTagResolutionSet(tagId, "wrong_event", USER_ROOT_DEFAULT).Has("error_details"));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            const auto taggedSession = traceTagManager.RestoreObject(offerId, tx);
            UNIT_ASSERT(taggedSession);
            const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();
            auto signalqTraceTag = tags.front().GetTagAs<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(signalqTraceTag->GetResolution(), "wrong_event");
        }
    }
}
