#include "car_driver.h"

#include <drive/telematics/server/tasks/lite.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NDrive;

TTripPlanner::TTripPlanner(const TString& imei, const TGeoCoord& startingPosition, const TGeoCoord& destination, TInstant start, TInstant finish, TAtomicSharedPtr<NDrive::TTelematicsApi> api, TAtomicSharedPtr<NGraph::TRouter> router)
    : TGlobalScheduler::TScheduledItem<TTripPlanner>("driver", "driver:" + imei, start)
    , IMEI(imei)
    , StartingPosition(startingPosition)
    , Destination(destination)
    , Start(start)
    , Finish(finish)
    , Api(api)
    , Router(router)
{
}

void TTripPlanner::Process(void* /*threadSpecificResource*/) {
    THolder<TTripPlanner> cleanup(this);

    auto route = Router->GetRoute(StartingPosition, Destination);
    const double speed = route.Length / ((Finish - TInstant::Now()).Seconds());

    {
        //set speed
        auto handle = Api->SetParameter<double>(IMEI, VEGA_SPEED, 0, speed);
        while (!Api->Await(handle)) {
            Sleep(TDuration::MilliSeconds(100));
        }
        UNIT_ASSERT_VALUES_EQUAL(Api->GetStatus(handle), NDrive::TTelematicsApi::EStatus::Success);
    }

    const TGeoCoord destination = Destination;
    const TDuration taskTimeout = NDrive::DefaultApiTimeout;
    const TDuration dataTimeout = NDrive::DefaultDataLifetime;
    NDrive::TSendCommandDistributedData data = Api->BuildCommand(IMEI, NDrive::NVega::ECommandCode::MOVE_TO_COORD, taskTimeout, dataTimeout, taskTimeout);
    data.SetMoveToCoordArgument(destination.X, destination.Y);

    {
        auto handle = Api->Command(data);
        while (!Api->Await(handle)) {
            Sleep(TDuration::MilliSeconds(100));
        }
        UNIT_ASSERT_VALUES_EQUAL(Api->GetStatus(handle), NDrive::TTelematicsApi::EStatus::Success);
    }
}

THolder<TGlobalScheduler::IScheduledItem> TTripPlanner::GetNextScheduledItem(TInstant /*now*/) const {
    return nullptr;
}

TCarDriver::TCarDriver(TAtomicSharedPtr<NDrive::TTelematicsApi> api, TAtomicSharedPtr<NGraph::TRouter> router)
    : Api(api)
    , Router(router)
    , GlobalSchedulerRegistrator(Name())
{
}

TGeoCoord TCarDriver::GetCurrentPosition(const TString& imei) {
    TGeoCoord result;
    {
        // get lon
        auto handle = Api->GetParameter(imei, VEGA_LON);
        while (!Api->Await(handle)) {
            Sleep(TDuration::MilliSeconds(100));
        }
        UNIT_ASSERT_VALUES_EQUAL(Api->GetStatus(handle), NDrive::TTelematicsApi::EStatus::Success);
        result.X = Api->GetValue<double>(handle);
    }

    {
        // get lat
        auto handle = Api->GetParameter(imei, VEGA_LAT);
        while (!Api->Await(handle)) {
            Sleep(TDuration::MilliSeconds(100));
        }
        UNIT_ASSERT_VALUES_EQUAL(Api->GetStatus(handle), NDrive::TTelematicsApi::EStatus::Success);
        result.Y = Api->GetValue<double>(handle);
    }
    return result;
}

void TCarDriver::PlanTrip(const TString& imei, const TGeoCoord& startingPosition, const TGeoCoord& destination, TInstant start, TInstant finish) {
    Y_ENSURE(Router);
    Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TTripPlanner>(imei, startingPosition, destination, start, finish, Api, Router)));
}

void TCarDriver::RelocateCar(const TString& imei, const TGeoCoord& destination) {
    {
        // set lon
        auto handle = Api->SetParameter<double>(imei, VEGA_LON, 0, destination.X);
        while (!Api->Await(handle)) {
            Sleep(TDuration::MilliSeconds(100));
        }
        UNIT_ASSERT_VALUES_EQUAL(Api->GetStatus(handle), NDrive::TTelematicsApi::EStatus::Success);
    }

    {
        // set lat
        auto handle = Api->SetParameter<double>(imei, VEGA_LAT, 0, destination.Y);
        while (!Api->Await(handle)) {
            Sleep(TDuration::MilliSeconds(100));
        }
        UNIT_ASSERT_VALUES_EQUAL(Api->GetStatus(handle), NDrive::TTelematicsApi::EStatus::Success);
    }
}

TString TCarDriver::Name() {
    return "driver";
}
