#include "helper.h"

#include "script.h"

#include <drive/backend/actions/administrative.h>
#include <drive/backend/actions/evolution_policy.h>
#include <drive/backend/actions/flag.h>
#include <drive/backend/actions/info_access.h>
#include <drive/backend/billing/trust/charge_logic.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/data/area_tags.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/fueling.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/data/radar.h>
#include <drive/backend/database/config.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/offers/actions/distributing_block.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/flows_control.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/car_scanner/processor.h>
#include <drive/backend/processors/leasing/add_cars.h>
#include <drive/backend/processors/rt_background/processor.h>
#include <drive/backend/promo_codes/actions.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/rt_background/future_book/config.h>
#include <drive/backend/rt_background/radar/processor.h>
#include <drive/backend/rt_background/rt_factors/config.h>

#include <drive/telematics/server/pusher/local.h>
#include <drive/tests/library/database.h>

#include <library/cpp/json/json_writer.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/storage/postgres/config.h>
#include <rtline/util/types/uuid.h>

#include <util/random/random.h>
#include <util/stream/null.h>
#include <util/system/env.h>
#include <util/system/hostname.h>

#include <maps/libs/local_postgres/include/instance.h>

const TVector<TString> TDocumentsHelper::UserProfileFields = TVector<TString>({
    "first_name",
    "last_name",
    "pn",
    "phone",
    "username",
    "email"
});

const TVector<TString> TDocumentsHelper::PassportStringFields = TVector<TString>({
    "first_name",
    "last_name",
    "middle_name",
    "doc_value",
    "birth_place",
    "gender",
    "citizenship",
    "subdivision_code",
    "registration_region",
    "registration_area",
    "registration_locality",
    "registration_street",
    "registration_house",
    "registration_housing",
    "registration_apartment"
});

const TVector<TString> TDocumentsHelper::PassportDateFields = TVector<TString>({
    "birth_date",
    "issue_date"
});

const TVector<TString> TDocumentsHelper::DrivingLicenseStringFields = TVector<TString>({
    "first_name",
    "last_name",
    "middle_name",
    "number_front",
    "number_back",
    "categories",
    "prev_licence_number",
});

const TVector<TString> TDocumentsHelper::DrivingLicenseDateFields = TVector<TString>({
    "birth_date",
    "issue_date",
    "experience_from",
    "prev_licence_issue_date",
    "categories_b_valid_from_date",
    "categories_b_valid_to_date",
});

const TVector<std::tuple<TString, TString>> TestingCars = {
    { OBJECT_ID_DEFAULT, OBJECT_IMEI_DEFAULT },
    { OBJECT_ID_TM, OBJECT_IMEI_TM },
    { "fad989f8-43eb-473b-87fd-ae25056be9a1", "872637844" },
};

const TVector<TString> TestingUsers = {
    USER_ID_DEFAULT,
    USER_ID_DEFAULT1,
    USER_ID_DEFAULT2,
    USER_ID_TECH,
    USER_ROOT_DEFAULT,
};

NDrive::TServerGuard::TServerGuard(NDrive::TServerConfig& config)
    : Server(MakeHolder<TServer>(config))
{
    RegisterGlobalMessageProcessor(this);
    Server->Run();
}

NDrive::TServerGuard::~TServerGuard() {
    UnregisterGlobalMessageProcessor(this);
    Server->Stop(0);
}

void TEnvironmentGenerator::BuildAreaTags(NDrive::TEntitySession& session) {
    {
        TTagDescription description;
        description.SetName("common_area_user_info").SetType("area_user_info");
        UNIT_ASSERT_C(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session), session.GetStringReport());
    }
    {
        TStandRegistrationPromoTag::TDescription description;
        description.SetMarkupTagName("promo_markup").SetBonusAmount(100500).SetName("test_area_reg_promo").SetType("area_stand_registration_promo");
        UNIT_ASSERT_C(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TStandRegistrationPromoTag::TDescription(description), USER_ID_DEFAULT, session), session.GetStringReport());
    }
}

void TEnvironmentGenerator::BuildFilters(NDrive::TEntitySession& session) {
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("'old_state_reservation"));
        filter.SetDescription("old_state_reservation").SetStateId("old_state_reservation").SetPriority(100).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("'old_state_riding"));
        filter.SetDescription("old_state_riding").SetStateId("old_state_riding").SetPriority(200).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("'old_state_acceptance"));
        filter.SetDescription("old_state_acceptance").SetStateId("old_state_acceptance").SetPriority(300).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("'old_state_parking"));
        filter.SetDescription("old_state_parking").SetStateId("old_state_parking").SetPriority(400).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("?urgent_cleaning,'cleaning,?cleaning*&old_state_reservation"));
        filter.SetDescription("cleaning").SetStateId("cleaning").SetPriority(2000).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("&old_state_reservation*-new_car*-installation"));
        filter.SetDescription("service").SetStateId("service").SetPriority(10000).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("new_car,installation"));
        filter.SetDescription("new").SetStateId("new").SetPriority(15000).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("#?old_state_reservation"));
        filter.SetDescription("huge_mileage").SetStateId("huge_mileage").SetPriority(19000).SetFilter(tagsFilter);
        filter.SetSensorFilter(NJson::FromJson<NDrive::TSensorFilter>(NJson::ReadJsonFastTree(R"({
            "conditions": [
                {
                    "sensor": "mileage",
                    "value_range": [100000, null]
                }
            ]
        })")));
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
    {
        TStateFilter filter;
        TTagsFilter tagsFilter;
        UNIT_ASSERT(tagsFilter.DeserializeFromString("#?old_state_reservation"));
        filter.SetDescription("available").SetStateId("available").SetPriority(20000).SetFilter(tagsFilter);
        DriveAPI.GetStateFiltersDB()->Upsert(filter, USER_ROOT_DEFAULT, session);
    }
}

void TEnvironmentGenerator::BuildFilterActions(NDrive::TEntitySession& session) {
    THolder<TFilterAction> action = MakeHolder<TFilterAction>("filter_unique_car_scanner", "scanner_simple1");
    action->SetText("aaa").SetPriority(0);
    UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
}

void TEnvironmentGenerator::BuildPromoCodePermissions(NDrive::TEntitySession& session) {
    {
        auto description = MakeHolder<TFixedBillingTagDescription>();
        description->SetName("add_bonus_tag").SetType("fixed_sum_tag");
        description->SetAction(TBillingTagDescription::EAction::Credit);
        description->SetAmount(100);
        description->SetAvailableAccounts({"bonus"});
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(description.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        auto description = MakeHolder<TTagDescription>("promo_account", "simple_account_tag");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(description.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        auto action = MakeHolder<TUserTagProfitGenerator>();
        action->SetName(PROMO_CODE_GENERATOR);
        TPromoTagProfit profit;
        profit.SetTagName("add_bonus_tag");
        action->SetPromoProfit(profit);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        auto action = MakeHolder<TUserTagProfitGenerator>();
        action->SetName(PROMO_CODE_GENERATOR_ACCOUNT);
        TPromoTagProfit profit;
        profit.SetTagName("promo_account");
        action->SetPromoProfit(profit);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        auto action = MakeHolder<TPromoCodeUsageActionImpl>();
        action->SetName("tst_promo_usage");
        action->SetAvailableTypes({PROMO_CODE_GENERATOR, PROMO_CODE_GENERATOR_ACCOUNT});
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TDriveRoleHeader promoUserRole("promo_access");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(promoUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        auto action = MakeHolder<TUserTagProfitGenerator>();
        action->SetName(PROMO_CODE_GENERATOR_SECOND);
        TPromoTagProfit profit;
        profit.SetTagName("add_bonus_tag");
        action->SetPromoProfit(profit);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        auto action = MakeHolder<TPromoCodeUsageActionImpl>();
        action->SetName("tst_promo_usage_second");
        action->SetAvailableTypes({PROMO_CODE_GENERATOR_SECOND});
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TFlagsAction> action(new TFlagsAction("enable_referral_program"));
        action->MutableFlags().emplace("enable_referral", "1");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("referral_builder"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Modify);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Add);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::PromoCodes);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("tst_promo_usage").SetRoleId("promo_access");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        actionHeader.SetSlaveObjectId("tst_promo_usage_second");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        actionHeader.SetSlaveObjectId("referral_builder");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        actionHeader.SetSlaveObjectId("enable_referral_program");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("promo_access").SetUserId(USER_ID_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("promo_access").SetUserId(USER_ID_DEFAULT1);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("promo_access").SetUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
}

void TEnvironmentGenerator::BuildOfferBuildersPermissions(NDrive::TEntitySession& session) {
    UNIT_ASSERT(
        DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Unlink(TVector<TString>({"chasiki_test", "chasiki_test2", "pack_offer_autotest", "minutki_test", "pack_offer_10h"}), "default_user", USER_ROOT_DEFAULT, session) &&
        DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleRoles().Unlink(TVector<TString>({"chasiki_test", "chasiki_test2", "pack_offer_autotest", "minutki_test", "pack_offer_10h"}), "default_user", USER_ROOT_DEFAULT, session)
    );
    UNIT_ASSERT(
        DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Unlink(TVector<TString>({"pack_offer_10h"}), "pack_access", USER_ROOT_DEFAULT, session) &&
        DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleRoles().Unlink(TVector<TString>({"pack_offer_10h"}), "pack_access", USER_ROOT_DEFAULT, session)
    );
    {
        THolder<TDiscountOfferCorrector> corrector(new TDiscountOfferCorrector("correction_free_time_1600_1800"));
        corrector->SetVisible(true);
        TDiscount::TDiscountDetails details;
        details.SetTagName("old_state_reservation");
        TTimeRestrictionsPool<TTimeRestriction> restrictions;
        TTimeRestriction restriction;
        restriction.SetTimeRestriction(1600, 1800);
        restrictions.Add(restriction);
        details.SetFreeTimetable(restrictions);
        corrector->AddDiscount(details);
        {
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(corrector.Release(), USER_ROOT_DEFAULT, session));
            {
                TDriveRoleHeader lock2UserRole("correction_free_time_1600_1800");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("correction_free_time_1600_1800").SetRoleId("correction_free_time_1600_1800");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
        }
    }
    TTagsFilter tagsFilter;
    if (SurgeTag == "old_state_reservation") {
        tagsFilter = TTagsFilter::BuildFromString("?old_state_reservation,?old_state_riding,?old_state_parking");
    } else {
        tagsFilter = TTagsFilter::BuildFromString("?" + SurgeTag);
    }
    {
        THolder<TStandartOfferConstructor> actorStandartOffer(new TStandartOfferConstructor(new TConstantPriceConfig, new TConstantPriceConfig, tagsFilter));
        actorStandartOffer->SetName("standart_offer_constructor");
        actorStandartOffer->SetDescription("standart_offer_constructor");
        actorStandartOffer->SetGrouppingTags({"standart"});
        actorStandartOffer->SetSwitchOfferTagsList({"standart", "fix"});
        actorStandartOffer->SetDetailedDescription("abc dfee (resource:standart_offer_description) sasdasdasd (resource:offer_signature) finish");
        actorStandartOffer->SetChargableAccounts({ "bonus", "card" });
        actorStandartOffer->SetUseDefaultShortDescriptions(false);
        actorStandartOffer->SetShortDescription({"_FreeFullDurationReservationSeconds_"});
        actorStandartOffer->SetSurgeEnabled(true);
        {
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorStandartOffer.Release(), USER_ROOT_DEFAULT, session));
            {
                TDriveRoleHeader lock2UserRole("standart_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("standart_offer_constructor").SetRoleId("standart_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
        }
    }
    {
        auto techOffer = MakeHolder<TStandartOfferConstructor>(
            new TConstantPriceConfig,
            new TConstantPriceConfig,
            TTagsFilter::BuildFromString("old_state_reservation")
        );
        techOffer->SetName("tech_offer_constructor");
        techOffer->SetDescription("tech_offer_constructor");
        techOffer->SetGrouppingTags({"standart"});
        techOffer->SetSwitchOfferTagsList({"standart", "fix"});
        techOffer->SetDetailedDescription("tech offer");
        techOffer->SetChargableAccounts({ "bonus", "card" });
        techOffer->SetUseDefaultShortDescriptions(false);
        {
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(std::move(techOffer), USER_ROOT_DEFAULT, session));
            {
                TDriveRoleHeader lock2UserRole("tech_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("tech_offer_constructor").SetRoleId("tech_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
        }
    }
    {
        THolder<TPackOfferConstructor> actorOffer(new TPackOfferConstructor(new TConstantPriceConfig, new TConstantPriceConfig, tagsFilter));
        actorOffer->SetDuration(TDuration::Hours(3));
        actorOffer->SetDescription("pack_offer_constructor");
        actorOffer->SetPackPrice(100000);
        actorOffer->SetGrouppingTags({"pack"});
        actorOffer->SetSwitchOfferTagsList({"pack"});
        actorOffer->SetMileageLimit(70);
        actorOffer->SetAvailableForScanner(false);
        actorOffer->SetRerunPriceKM(800);
        actorOffer->SetChargableAccounts({"bonus", "card"});

        actorOffer->SetName("pack_offer_constructor");
        actorOffer->MutableClientVersions().emplace("TEST_Version", 100);
        actorOffer->SetSurgeEnabled(true);

        THolder<TPackOfferConstructor> actorOfferReturn(new TPackOfferConstructor(*actorOffer));
        actorOfferReturn->SetDescription("return_pack_offer_constructor");
        actorOfferReturn->SetName("return_pack_offer_constructor");
        actorOfferReturn->SetReturningDuration(TDuration::Hours(1));

        THolder<TPackOfferConstructor> actorOfferB2B(new TPackOfferConstructor(*actorOffer));
        actorOfferB2B->SetDescription("pack_offer_constructor_b2b");
        actorOfferB2B->SetName("pack_offer_constructor_b2b");
        actorOfferB2B->SetChargableAccounts({"bonus"});

        THolder<TPackOfferConstructor> actorOfferQuanted(new TPackOfferConstructor(*actorOffer));
        actorOfferQuanted->SetDescription("pack_offer_constructor_quanted");
        actorOfferQuanted->SetName("pack_offer_constructor_quanted");
        actorOfferQuanted->SetDuration(TDuration::Minutes(1));
        actorOfferQuanted->SetPackPriceQuantum(10000);
        actorOfferQuanted->SetPackPriceQuantumPeriod(TDuration::Seconds(10));
        {
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOffer.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOfferReturn.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOfferB2B.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(std::move(actorOfferQuanted), USER_ROOT_DEFAULT, session));
            {
                TDriveRoleHeader lock2UserRole("pack_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("pack_offer_constructor").SetRoleId("pack_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("return_pack_offer_constructor").SetRoleId("pack_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("pack_offer_constructor_b2b").SetRoleId("pack_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("pack_offer_constructor_quanted").SetRoleId("pack_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
        }
    }
    {
        THolder<TFixPointOfferConstructor> actorOffer(new TFixPointOfferConstructor(new TConstantPriceConfig(10000), new TConstantPriceConfig(20000), tagsFilter));
        actorOffer->SetFeesForIncorrectFinish(0.1);
        actorOffer->SetRerunPriceKM(800);
//        actorOffer->SetParkingInPack(true);
        actorOffer->SetAvailableForScanner(false);
        actorOffer->SetReturningDuration(TDuration::Days(100));
        actorOffer->SetDescription("fixpoint_offer_constructor");
        actorOffer->SetName("fixpoint_offer_constructor");
        actorOffer->SetGrouppingTags({"fix"});
        actorOffer->SetSwitchOfferTagsList({"fix"});
        actorOffer->SetChargableAccounts({ "bonus", "card"});
        THolder<TFixPointOfferConstructor> actorOfferCopy(new TFixPointOfferConstructor(*actorOffer));
        THolder<TFixPointOfferConstructor> actorOfferWithCashback(new TFixPointOfferConstructor(*actorOffer));
        {
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOffer.Release(), USER_ROOT_DEFAULT, session));
            {
                TDriveRoleHeader lock2UserRole("fixpoint_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("fixpoint_offer_constructor").SetRoleId("fixpoint_access");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
        }

        actorOfferCopy->SetName("fixpoint_offer_constructor_child");
        actorOfferCopy->SetActionParent("fixpoint_offer_constructor");
        actorOfferCopy->SetRerunPriceKM(900);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOfferCopy.Release(), USER_ROOT_DEFAULT, session));

        actorOfferWithCashback->SetName("fixpoint_offer_constructor_with_cashback");
        actorOfferWithCashback->SetDescription("fixpoint_offer_constructor_with_cashback");
        actorOfferWithCashback->SetActionParent("fixpoint_offer_constructor");
        actorOfferWithCashback->SetCashbackPercent(5);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOfferWithCashback.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TFlowsOfferCorrector> flowsCorrector(new TFlowsOfferCorrector());
        flowsCorrector->SetDescription("flows_corrector");
        flowsCorrector->SetName("flows_corrector");
        {
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(flowsCorrector.Release(), USER_ROOT_DEFAULT, session));
            {
                TDriveRoleHeader lock2UserRole("flows_corrector");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
            }
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("flows_corrector").SetRoleId("flows_corrector");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
        }
    }
    {
        THolder<TFixPointOfferConstructor> actorOffer(new TFixPointOfferConstructor(new TConstantPriceConfig(10000), new TConstantPriceConfig(20000), TTagsFilter::BuildFromString("?" + SurgeTag)));
        actorOffer->SetFeesForIncorrectFinish(0.1);
        actorOffer->SetGrouppingTags({"fix"});
        actorOffer->SetSwitchOfferTagsList({"fix"});
        actorOffer->SetAvailableForScanner(false);
        actorOffer->SetRerunPriceKM(800);
        actorOffer->SetName("fixpoint_additional");
        actorOffer->SetDescription("fixpoint_additional");
        actorOffer->SetChargableAccounts({ "bonus", "card" });

        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOffer.Release(), USER_ROOT_DEFAULT, session));

    }
    {
        THolder<TFixPointOfferConstructor> actorOffer(new TFixPointOfferConstructor(new TConstantPriceConfig(10000), new TConstantPriceConfig(20000), TTagsFilter::BuildFromString("?" + SurgeTag)));
        actorOffer->SetFeesForIncorrectFinish(0.1);
        actorOffer->SetGrouppingTags({});
        actorOffer->SetSwitchOfferTagsList({"fix"});
        actorOffer->SetAvailableForScanner(false);
        actorOffer->SetRerunPriceKM(800);
        actorOffer->SetName("fixpoint_additional_for_wallet");
        actorOffer->SetDescription("fixpoint_additional_for_wallet");
        actorOffer->SetChargableAccounts({});

        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(actorOffer.Release(), USER_ROOT_DEFAULT, session));

    }
    for (auto&& i : TestingUsers) {
        TUserRole userRole;
        userRole.SetRoleId("standart_access").SetUserId(i);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        userRole.SetRoleId("pack_access").SetUserId(i);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        userRole.SetRoleId("fixpoint_access").SetUserId(i);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        userRole.SetRoleId("flows_corrector").SetUserId(i);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
}

void TEnvironmentGenerator::BuildInfoAccessPermissions(NDrive::TEntitySession& session) {
    {
        {
            TDriveRoleHeader lock2UserRole("view_permissions_all");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
        }
        {
            THolder<TInfoAccessAction> action(new TInfoAccessAction("user_access_base_iacc"));
            action->MutableDeviceTraits() = NDeviceReport::EReportTraits::ReportCars;
            action->MutableUserTraits() = 0;
            action->MutableCarRegistryTraits() = 0;
            action->MutableMessageVisibilityTraits() = 0;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
        {
            THolder<TInfoAccessAction> action(new TInfoAccessAction("view_permissions_action"));
            action->MutableDeviceTraits() = Max<NDeviceReport::TReportTraits>();
            action->MutableUserTraits() = Max<NUserReport::TReportTraits>() - NUserReport::EReportTraits::ReportHideNoTags;
            action->MutableCarRegistryTraits() = Max<NAttachmentReport::TStructureReportTraits>();
            action->MutableMessageVisibilityTraits() = (ui32)NDrive::NChat::TMessage::EMessageTraits::StaffOnly;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
        {
            THolder<TTagAction> action(new TTagAction("view_users_by_any_tags"));
            action->AddTagAction(TTagAction::ETagAction::ObserveObject);
            action->SetTagName("$.*");
            action->SetTagAttributesFilter(TTagsFilter::BuildFromString("user_tag"));
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
        {
            TDriveRoleHeader defaultUserRole("user_access_base");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(defaultUserRole, USER_ROOT_DEFAULT, session));
            {
                TLinkedRoleActionHeader actionHeader;
                actionHeader.SetSlaveObjectId("user_access_base_iacc").SetRoleId("user_access_base");
                UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            }
        }

        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("view_permissions_action").SetRoleId("view_permissions_all");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("view_users_by_any_tags").SetRoleId("view_permissions_all");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("user_access_base").SetUserId(USER_ID_DEFAULT1);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("view_permissions_all").SetUserId(USER_ID_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("view_permissions_all").SetUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("view_permissions_all").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
}

void TEnvironmentGenerator::BuildLockPermissions(NDrive::TEntitySession& session) {
    {
        {
            TDriveRoleHeader lock2UserRole("lock2");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock2UserRole, USER_ROOT_DEFAULT, session));
        }
        {
            THolder<TLockAction> action(new TLockAction("lock2"));
            action->SetLockedResourcesLimit(2);
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("lock2").SetRoleId("lock2");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
    }
    {
        {
            TDriveRoleHeader lock3UserRole("lock3");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lock3UserRole, USER_ROOT_DEFAULT, session));
        }
        {
            THolder<TLockAction> action(new TLockAction("lock3"));
            action->SetLockedResourcesLimit(3);
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("lock3").SetRoleId("lock3");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
    }
    {
        {
            TDriveRoleHeader lockSupport("lock_support");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(lockSupport, USER_ROOT_DEFAULT, session));
        }
        {
            THolder<TLockAction> action(new TLockAction("lock_support"));
            action->SetLockedResourcesLimit(3);
            action->SetLockedTags("$user_problem_tag_minor|user_problem_tag_major|support_chat_2_line|support_chat_1_line|support_chat_dtp|support_chat_performed");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("lock_support").SetRoleId("lock_support");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("lock2").SetUserId(USER_ID_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("lock3").SetUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("lock3").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
}

void TEnvironmentGenerator::BuildRootPermissions(NDrive::TEntitySession& session) {
    {
        TDriveRoleHeader rootUserRole("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("delegate_car"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Delegation);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Car);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("adm_localization"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Modify);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Remove);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Localization);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("actions_zone"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Add);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Remove);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Zone);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("adm_rt_background"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Add);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Modify);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Remove);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::RTBackground);
        action->MutableRTBackgroundTagsFilter() = TTagsFilter::BuildFromString("__ALL__");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("adm_chat_managing"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Modify);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Send);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Remove);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Add);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatMessage);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatResource);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatsList);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatSticker);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatRobot);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::User);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::SupportRequestCategorization);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::SupportCategorizationTree);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("adm_chat_supervisor"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::SendForce);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatMessage);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatResource);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatsList);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("add_any_tag"));
        action->AddTagAction(TTagAction::ETagAction::Add);
        action->AddTagAction(TTagAction::ETagAction::ObserveObject);
        action->SetTagName("$.*");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("delegate_car").SetRoleId("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("adm_localization").SetRoleId("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("actions_zone").SetRoleId("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("adm_rt_background").SetRoleId("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("adm_chat_managing").SetRoleId("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("adm_chat_supervisor").SetRoleId("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("add_any_tag").SetRoleId("root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("root").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }

    {
        TDriveRoleHeader supportAssigneeUserRole("support_assignee");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(supportAssigneeUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("assignee_on_user_problems"));
        action->AddTagAction(TTagAction::ETagAction::AssigneeToPerform);
        action->AddTagAction(TTagAction::ETagAction::Perform);
        action->AddTagAction(TTagAction::ETagAction::DropPerform);
        action->AddTagAction(TTagAction::ETagAction::Observe);
        action->SetTagName("$user_problem_tag_major|user_problem_tag_minor|support_chat_2_line|support_chat_1_line|support_chat_dtp|support_chat_performed");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("assignee_on_user_problems").SetRoleId("support_assignee");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("adm_chat_managing").SetRoleId("support_assignee");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }

    {
        TDriveRoleHeader rootUserRole("admin");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("observe_objects"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Confirm);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::User);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Car);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("manage_chats"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Modify);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Send);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatMessage);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatResource);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::ChatsList);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("observe_objects").SetRoleId("admin");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("manage_chats").SetRoleId("admin");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("admin").SetUserId(USER_ID_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("admin").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
}

void TEnvironmentGenerator::BuildAdmPermissions(NDrive::TEntitySession& session) {
    {
        TDriveRoleHeader rootUserRole("adm_permissions_default");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        TDriveRoleHeader rootUserRole("adm_permissions_root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        TDriveRoleHeader rootUserRole("full_tag_access");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("full_tag_access"));
        for (auto&& i : GetEnumAllValues<TTagAction::ETagAction>()) {
            if (i != TTagAction::ETagAction::HideObject && i != TTagAction::ETagAction::HideSelfPerformObject && i != TTagAction::ETagAction::AssigneeToPerform) {
                action->AddTagAction(i);
            }
        }
        action->SetTagName("$.*");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("adm_actions_default"));
        for (auto&& i : GetEnumAllValues<TAdministrativeAction::EAction>()) {
            action->MutableActions().emplace(i);
        }
        for (auto&& i : GetEnumAllValues<TAdministrativeAction::EEntity>()) {
            action->MutableEntities().emplace(i);
        }
        action->SetActionTypes({"*"});
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("actions_proposition"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Propose);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Confirm);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Reject);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Control);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Action);
        action->SetActionTypes({"*"});
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("actions_proposition").SetRoleId("adm_permissions_default");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("full_tag_access").SetRoleId("full_tag_access");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("adm_actions_default").SetRoleId("adm_permissions_root");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("adm_permissions_root").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("full_tag_access").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("adm_permissions_default").SetUserId(USER_ID_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("adm_permissions_default").SetUserId(USER_ID_DEFAULT1);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
}

void TEnvironmentGenerator::BuildDefaultData(NDrive::TEntitySession& session) {
    {
        TTagDescription description;
        description.SetName("old_state_reservation").SetType("chargable_tag");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("old_state_acceptance").SetType("chargable_tag");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("old_state_riding").SetType("chargable_tag");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("old_state_parking").SetType("chargable_tag");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }

    {
        THolder<TTagEvolutionAction> action(new TTagEvolutionAction("reservation->acceptance"));
        action->SetTagNameFrom("old_state_reservation").SetTagNameTo("old_state_acceptance");
        if (NeedTelematics) {
            action->SetCommandCode(NDrive::NVega::ECommandCode::OPEN_DOORS);
        }
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagEvolutionAction> action(new TTagEvolutionAction("acceptance->riding"));
        action->SetTagNameFrom("old_state_acceptance").SetTagNameTo("old_state_riding");
        if (NeedTelematics) {
            action->SetCommandCode(NDrive::NVega::ECommandCode::YADRIVE_START_OF_LEASE);
        }
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagEvolutionAction> action(new TTagEvolutionAction("riding->parking"));
        action->SetTagNameFrom("old_state_riding").SetTagNameTo("old_state_parking");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagEvolutionAction> action(new TTagEvolutionAction("parking->riding"));
        action->SetTagNameFrom("old_state_parking").SetTagNameTo("old_state_riding");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagEvolutionAction> action(new TTagEvolutionAction("parking->parking"));
        action->SetTagNameFrom("old_state_parking").SetTagNameTo("old_state_parking");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagEvolutionAction> action(new TTagEvolutionAction("->drop"));
        action->SetTagNameFrom("$old_state_riding|old_state_reservation|old_state_acceptance|old_state_parking").SetTagNameTo("old_state_reservation");
        if (NeedTelematics) {
            action->SetCommandCode(NDrive::NVega::ECommandCode::YADRIVE_END_OF_LEASE);
        }
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("hidden_manual_transmission"));
        action->AddTagAction(TTagAction::ETagAction::HideObject);
        action->SetTagName("manual_gearbox");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TSelfieVerificationEvolutionPolicyAction> action(new TSelfieVerificationEvolutionPolicyAction("selfie_verification_action"));
        action->SetVerificationTagName("need_selfie_verification");
        action->SetVerificationLandingId("selfie_verification_required");
        action->SetRecheckDocumentsTagName("selfie_verification_upload");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TRideCompletionEvolutionPolicyAction> action(new TRideCompletionEvolutionPolicyAction("ride_completion_action"));
        action->SetFreeDelegationEnabled(true);
        action->SetFreeDelegationConfirmationLanding("suspended_rent_only");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("perform_reservation"));
        action->SetTagName("old_state_reservation").AddTagAction(TTagAction::ETagAction::Perform).AddTagAction(TTagAction::ETagAction::ObserveObject);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("observe_class_base"));
        action->SetTagName("class_base").AddTagAction(TTagAction::ETagAction::ObserveObject);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TFilterAction> action(new TFilterAction("filter_base", "simple2"));
        action->SetText("filter_base").SetIcon("icon1");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TFilterAction> action(new TFilterAction("filter_comfort", "simple1"));
        action->SetText("filter_comfort").SetIcon("icon1");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("drop_reservation"));
        action->SetTagName("old_state_reservation").AddTagAction(TTagAction::ETagAction::DropPerform);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("car_control"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Control);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Car);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("hide_deleted_users"));
        action->AddTagAction(TTagAction::ETagAction::HideObject);
        action->SetTagName("$user_deleted_finally*");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("delegation_processing"));
        action->SetTagName("$.*_delegation_tag")
            .AddTagAction(TTagAction::ETagAction::Perform)
            .AddTagAction(TTagAction::ETagAction::ObserveBusyObject)
            .AddTagAction(TTagAction::ETagAction::Add)
            .AddTagAction(TTagAction::ETagAction::DropPerform)
            .AddTagAction(TTagAction::ETagAction::Remove)
            .AddTagAction(TTagAction::ETagAction::RemovePerform)
            ;
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TTagAction> action(new TTagAction("feedback_tags_operation"));
        action->SetTagName("$ugc_.*").AddTagAction(TTagAction::ETagAction::Add);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<TFlagsAction> action(new TFlagsAction("flag_test"));
        action->MutableFlags().emplace("test", "1");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
    }

    {
        TDriveRoleHeader defaultUserRole("default_user");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(defaultUserRole, USER_ROOT_DEFAULT, session));

        TDriveRoleHeader hiddenManualTransmissionRole("hidden_manual_transmission");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(hiddenManualTransmissionRole, USER_ROOT_DEFAULT, session));

        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("hidden_manual_transmission").SetRoleId("hidden_manual_transmission");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("selfie_verification_action").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("ride_completion_action").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("feedback_tags_operation").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("filter_base").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("flag_test").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("delegation_processing").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("hide_deleted_users").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("filter_comfort").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("car_control").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("perform_reservation").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("observe_class_base").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("drop_reservation").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("reservation->acceptance").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("acceptance->riding").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("riding->parking").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("parking->riding").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("parking->parking").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("->drop").SetRoleId("default_user");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TUserRole userRole;
            userRole.SetRoleId("default_user").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            userRole.SetRoleId("default_user").SetUserId(USER_ROOT_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            userRole.SetRoleId("default_user").SetUserId(USER_ID_DEFAULT1);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            userRole.SetRoleId("default_user").SetUserId(USER_ID_DEFAULT2);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {"old_state_reservation", "old_state_acceptance", "old_state_riding", "old_state_parking", "transformation"}, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_TM, {"old_state_reservation", "old_state_acceptance", "old_state_riding", "old_state_parking", "transformation"}, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreTags({}, {"long_term_standards_tier_1", "long_term_standards_tier_2"}, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, {TRadarUserTag::TypeName}, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetUserTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, {TOperationTag::TypeName, TFixedSumTag::TypeName}, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetUserTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }

        auto defaultModel = "porsche_carrera";
        {
            auto fetchResult = DriveAPI.GetModelsDB().FetchInfo(defaultModel, session);
            auto model = fetchResult.GetResultPtr(defaultModel);
            if (!model) {
                TDriveModelData newModel;
                newModel.SetCode(defaultModel);
                newModel.SetName("911");
                UNIT_ASSERT(DriveAPI.GetModelsDB().Upsert(newModel, session));
            }
        }

        for (auto&& [carId, imei] : TestingCars) {
            auto fetchResult = DriveAPI.GetCarManager().FetchInfo(carId, session);
            auto car = fetchResult.GetResultPtr(carId);
            if (!car) {
                TDriveCarInfo newCar;
                newCar.SetId(carId);
                newCar.SetIMEI(imei);
                newCar.SetModel(defaultModel);
                newCar.SetNumber("num_" + imei);
                newCar.SetVin("vin_" + imei);
                UNIT_ASSERT(DriveAPI.GetCarManager().UpdateCar(newCar, USER_ROOT_DEFAULT, session, nullptr, true));
            }
        }

        UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().AddTag(new TChargableTag("old_state_reservation"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, &Server, session));
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().AddTag(new TChargableTag("old_state_reservation"), USER_ID_DEFAULT, OBJECT_ID_TM, &Server, session));
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().AddTag(new TChargableTag("old_state_reservation"), USER_ID_DEFAULT, "fad989f8-43eb-473b-87fd-ae25056be9a1", &Server, session));
    }
}

void TEnvironmentGenerator::BuildCleanerData(NDrive::TEntitySession& session) {
    {
        TTagDescription description;
        description.SetName("cleaning").SetType("simple_car_tag");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        {
            THolder<TTagAction> action(new TTagAction("cleaning_processing"));
            action->SetTagName("cleaning")
                .AddTagAction(TTagAction::ETagAction::Perform)
                .AddTagAction(TTagAction::ETagAction::Add)
                .AddTagAction(TTagAction::ETagAction::ForcePerform)
                .AddTagAction(TTagAction::ETagAction::DropPerform)
                .AddTagAction(TTagAction::ETagAction::Remove)
                .AddTagAction(TTagAction::ETagAction::RemovePerform)
                ;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            TDriveRoleHeader cleanerUserRole("cleaner");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(cleanerUserRole, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("cleaning_processing").SetRoleId("cleaner");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "cleaner" }, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TUserRole userRole;
            userRole.SetRoleId("cleaner").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));

            userRole.SetRoleId("cleaner").SetUserId(USER_ID_DEFAULT2);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        }

    }
}

void TEnvironmentGenerator::BuildAdditionalOfferData(NDrive::TEntitySession& session) {
    {
        THolder<ITemporaryActionTag::TDescription> description(new ITemporaryActionTag::TDescription);
        description->SetName("additional_offer_tag").SetType("temp_action");
        description->MutableActionIds().emplace("fixpoint_additional");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(description.Release(), USER_ID_DEFAULT, session));
    }
    {
        THolder<ITemporaryActionTag::TDescription> description(new ITemporaryActionTag::TDescription);
        description->SetName("additional_offer_account_tag").SetType("account_temp_action");
        description->MutableActionIds().emplace("fixpoint_additional_for_wallet");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(description.Release(), USER_ID_DEFAULT, session));
    }
    {
        {
            THolder<TTagAction> action(new TTagAction("additional_offer_tag_processing"));
            action->SetTagName("additional_offer_tag|additional_offer_account_tag")
                .AddTagAction(TTagAction::ETagAction::Perform)
                .AddTagAction(TTagAction::ETagAction::Add)
                .AddTagAction(TTagAction::ETagAction::ForcePerform)
                .AddTagAction(TTagAction::ETagAction::DropPerform)
                .AddTagAction(TTagAction::ETagAction::Remove)
                .AddTagAction(TTagAction::ETagAction::RemovePerform)
                ;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            TDriveRoleHeader cleanerUserRole("additional_offer_tag_processor");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(cleanerUserRole, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("additional_offer_tag_processing").SetRoleId("additional_offer_tag_processor");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {"additional_offer_tag_processor"}, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TUserRole userRole;
            userRole.SetRoleId("additional_offer_tag_processor").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));

            userRole.SetRoleId("additional_offer_tag_processor").SetUserId(USER_ID_DEFAULT2);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        }

    }
}

void TEnvironmentGenerator::BuildScannerData(NDrive::TEntitySession& session) {
    {
        {
            THolder<TTagAction> action(new TTagAction("scanner_processing"));
            action->SetTagName(TRadarUserTag::TypeName)
                .AddTagAction(TTagAction::ETagAction::Perform)
                .AddTagAction(TTagAction::ETagAction::Add)
                .AddTagAction(TTagAction::ETagAction::ForcePerform)
                .AddTagAction(TTagAction::ETagAction::DropPerform)
                .AddTagAction(TTagAction::ETagAction::Remove)
                .AddTagAction(TTagAction::ETagAction::RemovePerform)
                ;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            TDriveRoleHeader cleanerUserRole("scanner");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(cleanerUserRole, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("scanner_processing").SetRoleId("scanner");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_TM, {"scanner"}, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TUserRole userRole;
            userRole.SetRoleId("scanner").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));

            userRole.SetRoleId("scanner").SetUserId(USER_ID_DEFAULT2);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        }

    }
}

void TEnvironmentGenerator::BuildPrioritySimpleData(NDrive::TEntitySession& session, const i32 priority, const TString& prefix, const TString& baseClass = "simple_car_tag") {
    {
        TTagDescription description;
        description.SetName(prefix + "simple1").SetType(baseClass).SetDefaultPriority(priority).SetEnableSessions(true);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("class_base").SetType("simple_car_tag").SetDefaultPriority(0).SetEnableSessions(false);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName(prefix + "simple2").SetType(baseClass).SetDefaultPriority(priority);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        {
            THolder<TTagAction> action(new TTagAction(prefix + "simple1_processing"));
            action->SetLanding({ "landing1", "landing2", "landing3", "landing5" });
            action->SetTagName(prefix + "simple1")
                .AddTagAction(TTagAction::ETagAction::Add)
                .AddTagAction(TTagAction::ETagAction::Propose)
                .AddTagAction(TTagAction::ETagAction::Confirm)
                .AddTagAction(TTagAction::ETagAction::Reject)
                .AddTagAction(TTagAction::ETagAction::Perform)
                .AddTagAction(TTagAction::ETagAction::ForcePerform)
                .AddTagAction(TTagAction::ETagAction::Observe)
                .AddTagAction(TTagAction::ETagAction::ObserveObject)
                .AddTagAction(TTagAction::ETagAction::DropPerform)
                .AddTagAction(TTagAction::ETagAction::Remove)
                .AddTagAction(TTagAction::ETagAction::RemovePerform)
                ;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            THolder<TTagAction> action(new TTagAction(prefix + "simple2_processing"));
            action->SetTagName(prefix + "simple2")
                .AddTagAction(TTagAction::ETagAction::Add)
                .AddTagAction(TTagAction::ETagAction::Propose)
                .AddTagAction(TTagAction::ETagAction::Confirm)
                .AddTagAction(TTagAction::ETagAction::Reject)
                .AddTagAction(TTagAction::ETagAction::Perform)
                .AddTagAction(TTagAction::ETagAction::Observe)
                .AddTagAction(TTagAction::ETagAction::ObserveObject)
                .AddTagAction(TTagAction::ETagAction::DropPerform)
                .AddTagAction(TTagAction::ETagAction::Remove)
                .AddTagAction(TTagAction::ETagAction::RemovePerform)
                ;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            THolder<TTagAction> action(new TTagAction(prefix + "car_property_processing"));
            action->SetTagName(prefix + "car_property")
                .AddTagAction(TTagAction::ETagAction::Observe)
                .AddTagAction(TTagAction::ETagAction::ObserveObject)
                ;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            THolder<TTagAction> action(new TTagAction(prefix + "car_complaint_processing"));
            action->SetTagName(prefix + "car_complaint")
                .AddTagAction(TTagAction::ETagAction::UpdateOnPerform)
                ;
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }

        {
            TDriveRoleHeader cleanerUserRole(prefix + "simple");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(cleanerUserRole, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId(prefix + "simple1_processing").SetRoleId(prefix + "simple");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId(prefix + "simple2_processing").SetRoleId(prefix + "simple");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId(prefix + "car_property_processing").SetRoleId(prefix + "simple");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId(prefix + "car_complaint_processing").SetRoleId(prefix + "simple");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { prefix + "simple1", prefix + "simple2" }, tags, session));
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tags, USER_ROOT_DEFAULT, session, true));
        }
        {
            TUserRole userRole;
            userRole.SetRoleId(prefix + "simple").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));

            userRole.SetRoleId(prefix + "simple").SetUserId(USER_ID_DEFAULT2);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));

            userRole.SetRoleId(prefix + "simple").SetUserId(USER_ROOT_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        }

    }
}

TDocumentsVerificationConfig TEnvironmentGenerator::BuildDefaultDocumentsVerificationConfig() const {
    TDocumentsVerificationConfig result;
    result
        .SetRequestsPath("//tmp/yang-hi-pi")
        .SetResultsPath("//tmp/yang-requests")
        .SetGeo({"*"})
        .SetYtClusters({})
        .SetMaxPoolSize(10);
    return result;
}

bool TEnvironmentGenerator::CreateNewYangAssignments(const NDrive::IServer* server) const {
    if (!DriveAPI.HasDocumentPhotosManager()) {
        ERROR_LOG << "user photo manager undefined" << Endl;
        return false;
    }
    TDocumentsVerificationManager manager(BuildDefaultDocumentsVerificationConfig(), DriveAPI.GetDocumentPhotosManager());
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    if (!manager.CreateNewAssignments(TInstant::Zero(), server, session) || !session.Commit()) {
        ERROR_LOG << "unable to create yang assignments" << Endl;
        return false;
    }
    return true;
}

bool TEnvironmentGenerator::CreateNewSelfieYangAssignments(const NDrive::IServer* server) const {
    if (!DriveAPI.HasDocumentPhotosManager()) {
        ERROR_LOG << "user photo manager undefined" << Endl;
        return false;
    }
    TDocumentsVerificationManager manager(BuildDefaultDocumentsVerificationConfig().SetTaskType(TDocumentsVerificationConfig::ETaskType::FaceMatching), DriveAPI.GetDocumentPhotosManager());
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    if (!manager.CreateNewAssignments(TInstant::Zero(), server, session) || !session.Commit()) {
        ERROR_LOG << "unable to create yang assignments" << Endl;
        return false;
    }
    return true;
}

TEnvironmentGenerator::TCar TEnvironmentGenerator::CreateCar(const TString& model, const TString& vin) {
    for (ui32 i = 0; i < 10; ++i) {
        NDrive::TEntitySession session = DriveAPI.BuildTx<NSQL::Writable>();
        auto result = CreateCar(session, model, Default<TString>(), vin);
        if (session.Commit()) {
            UNIT_ASSERT(DriveAPI.GetCarsData()->RefreshCache(Now()));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>("car_tags_history");
            return result;
        }
        Sleep(TDuration::MilliSeconds(200));
        UNIT_ASSERT_C(i < 9, session.GetStringReport());
    }
    return {};
}

TEnvironmentGenerator::TCar TEnvironmentGenerator::CreateCar(NDrive::TEntitySession& session, const TString& model, const TString& carIdSalt, const TString& vin) {
    TCar result;
    {
        ui64 imei = 500000000000000 + RandomNumber<ui64>(100000000000000);

        NStorage::TTableRecord record;
        result.Number = "ttt" + ToString(RandomNumber<ui32>()) + "ccc";
        TString carId = "";
        if (!!carIdSalt) {
            TGUID guid;
            TString hostName = HostName();
            guid.dw[0] = FnvHash<ui32>(hostName.data(), hostName.size());
            guid.dw[1] = FnvHash<ui32>(carIdSalt.data(), carIdSalt.size());
            carId = GetGuidAsString(guid);
        }
        const TString id = DriveAPI.GetCarsData()->RegisterNewCar("unittest", result.Number, ToString(imei), model, session, carId, vin);
        UNIT_ASSERT(!!id);

        result.Id = id;
        result.IMEI = ToString(imei);
        result.Vin = vin;
    }
    UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().AddTag(new TChargableTag("old_state_reservation"), USER_ID_DEFAULT, result.Id, &Server, session));
    UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().AddTag(new TDeviceTagRecord("class_base"), USER_ID_DEFAULT, result.Id, &Server, session));
    if (SurgeTag != "old_state_reservation") {
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().AddTag(new TDeviceTagRecord(SurgeTag), USER_ID_DEFAULT, result.Id, &Server, session));
    }
    INFO_LOG << "Created car " << result.Id << " " << result.IMEI << Endl;
    return result;
}

void TEnvironmentGenerator::BuildTechData(NDrive::TEntitySession& session) {
    {
        TTagDescription description;
        description.SetName("DTP").SetType("car_service_tag");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }
    {
        {
            THolder<TTagAction> action(new TTagAction("perform_DTP"));
            action->SetTagName("DTP").AddTagAction(TTagAction::ETagAction::Perform);
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            THolder<TTagAction> action(new TTagAction("drop_DTP"));
            action->SetTagName("DTP").AddTagAction(TTagAction::ETagAction::DropPerform);
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
        {
            THolder<TTagAction> action(new TTagAction("remove_performed_DTP"));
            action->SetTagName("DTP").AddTagAction(TTagAction::ETagAction::RemovePerform);
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
    }
    {
        {
            TTagDescription description;
            description.SetName("fueling").SetType("simple_fueling_tag");
            UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
        }
        {
            THolder<TTagAction> action(new TTagAction("fueling_tag_actions"));
            action->SetTagName("fueling").AddTagAction(TTagAction::ETagAction::Perform).AddTagAction(TTagAction::ETagAction::RemovePerform);
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ID_DEFAULT, session));
        }
    }
    {
        {
            TDriveRoleHeader techUserRole("tech");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(techUserRole, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("remove_performed_DTP").SetRoleId("tech");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("perform_DTP").SetRoleId("tech");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("drop_DTP").SetRoleId("tech");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TUserRole userRole;
            userRole.SetRoleId("tech").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            userRole.SetRoleId("tech").SetUserId(USER_ID_DEFAULT2);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        }
        {
            TDriveRoleHeader rootUserRole("fueling");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
        }
        {
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("fueling_tag_actions").SetRoleId("fueling");
            UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        }
        {
            TUserRole userRole;
            userRole.SetRoleId("fueling").SetUserId(USER_ID_TECH);
            UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
        }
    }
}

void TEnvironmentGenerator::BuildLandingPermissions(NDrive::TEntitySession& session) {
    {
        TDriveRoleHeader rootUserRole("landing");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("landing"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Delegation);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Modify);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Remove);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Propose);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Confirm);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Reject);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Stories);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Landings);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("landing").SetRoleId("landing");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("landing").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("landing").SetUserId(USER_ID_DEFAULT2);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    TLandingUserTag::TDescription description;
    description.SetLandingId("landing_tag").SetName("landing_tag").SetType(TLandingUserTag::TypeName).SetDefaultPriority(0);
    UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TLandingUserTag::TDescription(description), USER_ROOT_DEFAULT, session));
}

void TEnvironmentGenerator::BuildCarModelsPermissions(NDrive::TEntitySession& session) {
    {
        TDriveRoleHeader rootUserRole("car_models");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesDB().Upsert(rootUserRole, USER_ROOT_DEFAULT, session));
    }
    {
        THolder<TAdministrativeAction> action(new TAdministrativeAction("car_models"));
        action->MutableActions().emplace(TAdministrativeAction::EAction::Remove);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Add);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Observe);
        action->MutableActions().emplace(TAdministrativeAction::EAction::Modify);
        action->MutableEntities().emplace(TAdministrativeAction::EEntity::Models);
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
    }
    {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("car_models").SetRoleId("car_models");
        UNIT_ASSERT(DriveAPI.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    {
        TUserRole userRole;
        userRole.SetRoleId("car_models").SetUserId(USER_ROOT_DEFAULT);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
}

TString TEnvironmentGenerator::CreateUser(const TString& login, const bool withRoles, const TString& status, const TString& phoneNumber, const bool phoneVerified) {
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    auto optionalUser = DriveAPI.GetUsersData()->RegisterNewUser("unittest", ToString(RandomNumber<ui64>(10000000000)), login, session, phoneNumber);
    UNIT_ASSERT(optionalUser);
    auto userId = optionalUser->GetUserId();
    auto fetchResult = DriveAPI.GetUsersData()->FetchInfo(userId, session).GetResult();
    UNIT_ASSERT_VALUES_EQUAL(fetchResult.size(), 1);
    auto userData = fetchResult.begin()->second;
    userData.SetNewAccount(false);
    userData.SetStatus(status);
    userData.SetPhoneVerified(phoneVerified);
    userData.SetEmail("abcd@efgh.ij");
    userData.SetEMailVerified(true);
    DriveAPI.GetUsersData()->UpdateUser(userData, "robot-frontend", session);
    if (!withRoles) {
        UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        return userId;
    }

    {
        TUserRole userRole;
        userRole.SetRoleId("scanner").SetUserId(userId);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }

    {
        TUserRole userRole;
        userRole.SetRoleId("view_permissions_all").SetUserId(userId);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }

    {
        TUserRole userRole;
        userRole.SetRoleId("default_user").SetUserId(userId);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }

    {
        TUserRole userRole;
        userRole.SetRoleId("standart_access").SetUserId(userId);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }

    {
        TUserRole userRole;
        userRole.SetRoleId("simple").SetUserId(userId);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }

    {
        TUserRole userRole;
        userRole.SetRoleId("priority_simple").SetUserId(userId);
        UNIT_ASSERT(DriveAPI.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }

    UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
    return userId;
}

TString TEnvironmentGenerator::CreateUserDocumentPhoto(const TString& userId, const NUserDocument::EType& photoType, const NUserDocument::EVerificationStatus status) {
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    TUserDocumentPhoto photoObj(userId, photoType);
    photoObj.SetVerificationStatus(status);
    UNIT_ASSERT(DriveAPI.HasDocumentPhotosManager());
    DriveAPI.GetDocumentPhotosManager().GetUserPhotosDB().Upsert(photoObj, session);
    UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
    return photoObj.GetId();
}

bool TEnvironmentGenerator::CompleteYangAssignment(const TString& assignmentId, const TString& photoStatuses, const TYangDocumentVerificationAssignment::EFraudStatus& overallStatus, bool isExp, const TVector<TString>& assignmentIds) {
    UNIT_ASSERT(DriveAPI.HasDocumentPhotosManager());
    const auto& assignmentDB = DriveAPI.GetDocumentPhotosManager().GetDocumentVerificationAssignments();
    auto fetchResult = assignmentDB.FetchInfo(assignmentId);
    auto assignmentPtr = fetchResult.GetResultPtr(assignmentId);
    if (!assignmentPtr) {
        return false;
    }

    auto assignment = *assignmentPtr;
    assignment.SetIsFraud(overallStatus);
    assignment.SetProcessedAt(Now());
    assignment.SetIsExperimental(isExp);
    assignment.SetAssignmentIds(assignmentIds);
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    if (!assignmentDB.Upsert(assignment, session) || !session.Commit()) {
        return false;
    }

    if (!SetPhotoStatus(assignment.GetLicenseBackId(), photoStatuses[0])) {
        return false;
    }
    if (!SetPhotoStatus(assignment.GetLicenseFrontId(), photoStatuses[1])) {
        return false;
    }
    if (!SetPhotoStatus(assignment.GetLicenseFrontId(), photoStatuses[1])) {
        return false;
    }
    if (!SetPhotoStatus(assignment.GetPassportBiographicalId(), photoStatuses[2])) {
        return false;
    }
    if (!SetPhotoStatus(assignment.GetPassportRegistrationId(), photoStatuses[3])) {
        return false;
    }
    if (!SetPhotoStatus(assignment.GetPassportSelfieId(), photoStatuses[4])) {
        return false;
    }
    return false;
}

bool TEnvironmentGenerator::SetPhotoStatus(const TString& photoId, const char status) {
    TString statusStr(1, status);
    NUserDocument::EVerificationStatus photoStatus;
    UNIT_ASSERT_C(TryFromString(statusStr, photoStatus), TStringBuilder() << statusStr);
    UNIT_ASSERT(DriveAPI.HasDocumentPhotosManager());
    const auto& userPhotoDB = DriveAPI.GetDocumentPhotosManager().GetUserPhotosDB();
    auto fetchResult = userPhotoDB.FetchInfo(photoId);
    auto photoPtr = fetchResult.GetResultPtr(photoId);
    if (!photoPtr) {
        return false;
    }
    auto photo = *photoPtr;
    photo.SetVerifiedAt(Now());
    photo.SetVerificationStatus(NUserDocument::EVerificationStatus(photoStatus));
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    if (!userPhotoDB.Upsert(photo, session) || !session.Commit()) {
        return false;
    }

    return true;
}

void TEnvironmentGenerator::BuildCarTags(NDrive::TEntitySession& session) {
    {
        TPatchFuelingTag::TDescription description;
        description.MutablePatchTypes().push_back(EFuelType::A100Premium);
        description.SetName(ToString(EFuelPatchTags::SimplePatch)).SetType(TPatchFuelingTag::GetTypeName()).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TPatchFuelingTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TPatchFuelingTag::TDescription description;
        description.SetOverrideModel(true).MutablePatchTypes().push_back(EFuelType::A100Premium);
        description.SetName(ToString(EFuelPatchTags::OverrideModel)).SetType(TPatchFuelingTag::GetTypeName()).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TPatchFuelingTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TPatchFuelingTag::TDescription description;
        description.SetSensor(NDrive::NVega::AuxFuelLevel<1>()).MutablePatchTypes().push_back(EFuelType::A100Premium);
        description.SetName(ToString(EFuelPatchTags::Dut1Sensor)).SetType(TPatchFuelingTag::GetTypeName()).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TPatchFuelingTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TPatchFuelingTag::TDescription description;
        description.SetSensor(NDrive::NVega::AuxFuelLevel<2>()).MutablePatchTypes().push_back(EFuelType::A100Premium);
        description.SetName(ToString(EFuelPatchTags::Dut2Sensor)).SetType(TPatchFuelingTag::GetTypeName()).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TPatchFuelingTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
}

void TEnvironmentGenerator::BuildUserTags(NDrive::TEntitySession& session) {
    {
        TTagDescription description;
        description.SetName("promo_markup").SetType(TUniqueUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("user_deleted_finally").SetType(TUniqueUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TRegistrationUserTag::TDescription description;
        description.SetNoCleanup(true).SetName("fast_registration_available").SetType(TRegistrationUserTag::TypeName);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TRegistrationUserTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TRegistrationUserTag::TDescription description;
        description.SetNoCleanup(true).SetName("user_registered_manually").SetType(TRegistrationUserTag::TypeName);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TRegistrationUserTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TRegistrationUserTag::TDescription description;
        description.SetNoCleanup(true).SetName("datasync_problems").SetType(TRegistrationUserTag::TypeName);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TRegistrationUserTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetType(TUserContainerTag::TypeName).SetName(TSupportChatTag::DeferredContainerName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportChatTag::TDescription description;
        description.SetOperatorEntranceMessage("ВХОДИТ ОПЕРАТОР [i]МАРИЯ[/i]").SetOperatorName("Мария").SetName("support_chat_2_line").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportChatTag::TDescription description;
        description.SetOperatorEntranceMessage("ВХОДИТ ОПЕРАТОР [i]ВЛАДИМИР[/i]").SetOperatorName("Владимир").SetName("support_chat_1_line").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportChatTag::TDescription description;
        description.SetOperatorEntranceMessage("ВХОДИТ ОПЕРАТОР [i]ВЛАДИМИР[/i]").SetOperatorName("Владимир").SetName("support_chat_dtp").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportChatTag::TDescription description;
        description.SetOperatorEntranceMessage("ВХОДИТ ОПЕРАТОР [i]ВИКОНТ[/i]").SetOperatorName("Виконт").SetName("support_chat_performed").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
        description.MutableOnDeferBehaviour().SetDropPerformer(false);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportChatTag::TDescription description;
        description.SetOperatorEntranceMessage("ВХОДИТ ОПЕРАТОР [i]ДИАНА[/i]").SetOperatorName("Диана").SetName("support_chat_lowpriority").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportChatTag::TDescription description;
        description.SetOperatorEntranceMessage("").SetName(TSupportChatTag::DeferredName).SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportChatTag::TDescription description;
        description.SetRedirectChatNode("leave_feedback_intro").SetOperatorEntranceMessage("").SetName(TSupportChatTag::FeedbackName).SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("same_person").SetType(TConnectionUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(1).SetName("user_problem_tag_minor").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(4).SetName("user_problem_tag_major").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportOutgoingCommunicationTag::TDescription description;
        description.SetName("test_outgoing_communication").SetType(TSupportOutgoingCommunicationTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportOutgoingCommunicationTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportOutgoingCommunicationTag::TDescription description;
        description.SetName("leasing_support_notify").SetType(TSupportOutgoingCommunicationTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportOutgoingCommunicationTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(10000).SetName("blocked_fraud_selfie_screencap").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(10000).SetName("blocked_fraud_selfie_another_p").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(10000).SetDropOnReentry(true).SetChatMessagesGroup("ban_old_license_intro").SetChatMessagesGroupPriority(1000).SetName("blocked_old_license").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(10000).SetDropOnReentry(true).SetChatMessagesGroup("ban_speed_asshole").SetChatMessagesGroupPriority(1000).SetBanChat("high_speed_ban").SetName("blocked_speed_asshole").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(10000).SetName("blocked_external_blacklist").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("yaccount_reg_date").SetType(TSimpleUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetType(TSimpleUserTag::TypeName).SetDefaultPriority(0).SetName("duty");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetType(TSimpleUserTag::TypeName).SetDefaultPriority(0).SetName("duty_end");
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetDropOnReentry(true).SetPoint(10000).SetName("blocked_for_resubmit").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(10000).SetChatMessagesGroup("ban_duplicate_passport").SetChatMessagesGroupPriority(1000).SetName("blocked_duplicate_namber_pass").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TUserProblemTag::TDescription description;
        description.SetPoint(10000).SetName("suspicious_from_external_blacklist").SetType(TUserProblemTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TUserProblemTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TRegistrationUserTag::TDescription description;
        description.SetChatItem("soon_passport_expires").SetChatRobot("expiring_passport_notify").SetName("soon_passport_expires").SetType(TRegistrationUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TRegistrationUserTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TRegistrationUserTag::TDescription description;
        description.SetChatItem("soon_license_expires").SetChatRobot("expiring_license_notify").SetName("soon_license_expires").SetType(TRegistrationUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TRegistrationUserTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("need_selfie_verification").SetType(TUniqueUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("selfie_verification_upload").SetType(TSimpleUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("delegation_finished").SetType(TUniqueUserTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TTagDescription description;
        description.SetName("registration_info").SetType(TUserDictionaryTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TSupportPhoneCallTag::TDescription description;
        description.SetUrlTemplate("test.com/[call_id]").SetName("cc_audiotele_incoming").SetType(TSupportPhoneCallTag::TypeName).SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportPhoneCallTag::TDescription(description), USER_ROOT_DEFAULT, session));
    }
    {
        TString tagDescr = R"(
            {"tag_flow":"","enable_report":false,"tag_flow_priority":0,"unique_policy":"rewrite","groupping_attributes":[],"unlimited_history":false,
                "fields":[
                    {"default":"","name":"referral_code","materialize_default_value":false,"type":"string","id":"referral_code"},
                    {"default":"","name":"custom_referral_code","materialize_default_value":false,"type":"string","id":"custom_referral_code"},
                    {"default":"false","name":"disable_cross_login","materialize_default_value":false,"type":"string","id":"disable_cross_login"},
                    {"default":"true","name":"notifications_push","materialize_default_value":false,"type":"string","id":"notifications_push"},
                    {"default":"true","name":"notifications_landing","materialize_default_value":false,"type":"string","id":"notifications_landing"},
                    {"default":"true","name":"notifications_email","materialize_default_value":false,"type":"string","id":"notifications_email"}
                ],
            "deprecated":false,"transferred_to_double":false,"enable_sessions":false})";
        NJson::TJsonValue descrJson = NJson::JSON_NULL;
        UNIT_ASSERT(NJson::ReadJsonTree(tagDescr, &descrJson));
        TDictionaryTagDescription appSettings;
        UNIT_ASSERT(appSettings.DoDeserializeMetaFromJson(descrJson));
        appSettings.SetName("user_app_settings").SetType(TUserDictionaryTag::TypeName);
        UNIT_ASSERT_C(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TDictionaryTagDescription(appSettings), USER_ROOT_DEFAULT, session), session.GetStringReport());
    }
}

void TEnvironmentGenerator::BuildRTBackgrounds(const IServerBase& server, const ui64 backgrounds) {
    auto session = DriveAPI.BuildTx<NSQL::Writable>();
    if (backgrounds & EServerBackgrounds::FuturesBooking) {
        THolder<TRTFuturesBook> fb(new TRTFuturesBook);
        fb->SetNotifierName("internal_notifier");
        fb->SetPeriod(TDuration::Seconds(1));
        fb->SetEnabled(true);
        TRTBackgroundProcessContainer container(fb.Release());
        container.SetName("futures_book");
        UNIT_ASSERT(server.GetRTBackgroundManager()->UpsertObject(container, USER_ROOT_DEFAULT, session));
    }
    if (backgrounds & EServerBackgrounds::Radar) {
        auto fb = MakeHolder<TRadarUserTagProcess>();
        fb->SetPeriod(TDuration::Seconds(1));
        fb->SetEnabled(true);
        TRTBackgroundProcessContainer container(fb.Release());
        container.SetName("radar_user_tag_process");
        UNIT_ASSERT(server.GetRTBackgroundManager()->UpsertObject(container, USER_ROOT_DEFAULT, session));
    }
    if (backgrounds & EServerBackgrounds::SurgeConstructor) {
        THolder<TRTCarFactors> fb(new TRTCarFactors);
        fb->SetPeriod(TDuration::Seconds(1));
        fb->SetEnabled(true);
        TRTBackgroundProcessContainer container(fb.Release());
        container.SetName("car_features");
        UNIT_ASSERT(server.GetRTBackgroundManager()->UpsertObject(container, USER_ROOT_DEFAULT, session));
    }
    UNIT_ASSERT(session.Commit());
}

namespace {
    const ui16 TankerMockPort = 33333;
}

void TEnvironmentGenerator::BuildEnvironment(const ui32 envTraits) {
    auto& billingConfig = DriveAPI.GetBillingManager().GetConfig();
    ui16 billingPort = 0;
    for (auto&& logic : billingConfig.GetLogicConfigs()) {
        if (logic->GetType() == NDrive::NBilling::EAccount::Trust) {
            auto impl = VerifyDynamicCast<TTrustLogicConfig*>(logic.Get());
            billingPort = impl->GetClientConfig().GetPort();
            break;
        }
    }
    BillingMock = NAPHelper::BuildServer(billingPort, 8, TDuration::Seconds(0), 200);
    BillingMock->SetReply(R"({
                                    "status": "success",
                                    "payment_status": "authorized",
                                    "purchase_token": "123",
                                    "amount": "10",
                                    "bound_payment_methods" : [
                                    {
                                      "region_id": 225,
                                      "payment_method" : "card",
                                      "expiration_month" : "07",
                                      "binding_ts" : "1536324485.656",
                                      "id" : "card-x12345",
                                      "expired" : false,
                                      "card_bank" : "TINKOFF BANK",
                                      "system" : "VISA",
                                      "account" : "510000****0257",
                                      "expiration_year" : "2021"
                                    },
                                    {
                                      "payment_method" : "yandex_account",
                                      "account" : "w/30b153cc-8e30-58e2-8d1a-1095bc49b915",
                                      "payment_system" : null,
                                      "currency" : "RUB",
                                      "id" : "yandex_account-w/30b153cc-8e30-58e2-8d1a-1095bc49b915",
                                      "balance" : "251.27"
                                    }
                                    ]})");
    BillingMock->SetNeedHeader(false);

    {
        auto database = DriveAPI.GetDatabasePtr();
        UNIT_ASSERT(database);
        NDrive::DropTable(*database, "drive_offers");
        NDrive::DropTable(*database, "tags_description_propositions");
        NDrive::CreateTable(*database, "drive_offers");
        NDrive::CreateTable(*database, "call_priority_user");
        NDrive::CreateTable(*database, "tags_description_propositions");
    }

    auto session = DriveAPI.BuildTx<NSQL::Writable>();

    {
        session->Exec("DELETE FROM billing_tasks where user_id='" + USER_ID_DEFAULT + "'");
        session->Exec("DELETE FROM billing_tasks where user_id='" + USER_ID_DEFAULT1 + "'");
        session->Exec("DELETE FROM billing_tasks where user_id='" + USER_ID_DEFAULT2 + "'");
        session->Exec("DELETE FROM billing_tasks where user_id='" + USER_ID_TECH + "'");
    }

    {
        session->Exec("DELETE FROM user_roles where user_id='" + TString(USER_ID_DEFAULT) + "'");
        session->Exec("DELETE FROM user_roles where user_id='" + TString(USER_ID_DEFAULT1) + "'");
        session->Exec("DELETE FROM user_roles where user_id='" + TString(USER_ID_DEFAULT2) + "'");
        session->Exec("DELETE FROM user_roles where user_id='" + TString(USER_ID_TECH) + "'");
        session->Exec("DELETE FROM drive_role_roles");
        session->Exec("DELETE FROM drive_role_actions");
    }

    {
        session->Exec("DELETE FROM car_tags WHERE object_id IN (SELECT id FROM \"car\" WHERE number LIKE 'ttt%ccc')");
        session->Exec("DELETE FROM head_app_sessions WHERE car_id IN (SELECT id FROM \"car\" WHERE number LIKE 'ttt%ccc')");
        session->Exec("DELETE FROM head_app_sessions_history WHERE car_id IN (SELECT id FROM \"car\" WHERE number LIKE 'ttt%ccc')");
        session->Exec("DELETE FROM insurance_tasks WHERE object_id IN (SELECT id FROM \"car\" WHERE number LIKE 'ttt%ccc')");
        session->Exec("DELETE FROM insurance_tasks_history WHERE object_id IN (SELECT id FROM \"car\" WHERE number LIKE 'ttt%ccc')");
        session->Exec("DELETE FROM \"car\" WHERE number LIKE 'ttt%ccc'");
        session->Exec("DELETE FROM area_tags");
    }

    {
        TVector<TDBTag> tagsAll;
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {}, tagsAll, session));
        UNIT_ASSERT_C(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tagsAll, USER_ROOT_DEFAULT, session, true), TStringBuilder() << session.GetStringReport() << Endl);
    }

    {
        TVector<TDBTag> tagsAll;
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT1, {}, tagsAll, session));
        UNIT_ASSERT_C(DriveAPI.GetTagsManager().GetDeviceTags().RemoveTagsSimple(tagsAll, USER_ROOT_DEFAULT, session, true), TStringBuilder() << session.GetStringReport() << Endl);
    }

    for (auto&& userId : TestingUsers) {
        auto fetchResult = DriveAPI.GetUserManager().FetchInfo(userId, session);
        auto user = fetchResult.GetResultPtr(userId);
        if (!user) {
            auto newUser = TDriveUserData();
            newUser.SetUserId(userId);
            newUser.SetStatus(NDrive::UserStatusActive);
            auto added = DriveAPI.GetUserManager().UpdateUser(newUser, userId, session);
            UNIT_ASSERT(added);
            INFO_LOG << "Created user " << userId << ": " << added->GetReport().GetStringRobust() << Endl;
        } else {
            INFO_LOG << "Found user " << userId << ": " << user->GetReport().GetStringRobust() << Endl;
        }
    }

    BuildAreaTags(session);
    BuildFilters(session);
    BuildUserTags(session);
    BuildCarTags(session);
    if (envTraits & (ui32)EEnvironmentFeatures::OfferBuilder) {
        BuildOfferBuildersPermissions(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::Filters) {
        BuildFilterActions(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::InfoAccess) {
        BuildInfoAccessPermissions(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::Lock) {
        BuildLockPermissions(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::Root) {
        BuildRootPermissions(session);
    }
    BuildAdmPermissions(session);
    if (envTraits & (ui32)EEnvironmentFeatures::Default) {
        BuildDefaultData(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::Cleaner) {
        BuildCleanerData(session);
    }
    BuildAdditionalOfferData(session);
    if (envTraits & (ui32)EEnvironmentFeatures::Tech) {
        BuildTechData(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::Scanner) {
        BuildScannerData(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::Landing) {
        BuildLandingPermissions(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::CarModels) {
        BuildCarModelsPermissions(session);
    }
    if (envTraits & (ui32)EEnvironmentFeatures::PromoCodes) {
        BuildPromoCodePermissions(session);
    }
    BuildPrioritySimpleData(session, 0, "force_stop_car_", "simple_car_tag");
    BuildPrioritySimpleData(session, 0, "user_", "simple_user_tag");
    BuildPrioritySimpleData(session, 1000, "priority_user_", "simple_user_tag");
    BuildPrioritySimpleData(session, 0, "", "simple_car_tag");
    BuildPrioritySimpleData(session, 1000, "priority_", "simple_car_tag");
    BuildPrioritySimpleData(session, 0, "scanner_", "simple_car_tag");
    BuildPrioritySimpleData(session, 0, "uniq_", "simple_car_tag");

    if (SurgeTag != "old_state_reservation") {
        TTagDescription description;
        description.SetName(SurgeTag).SetType("simple_car_tag").SetDefaultPriority(0);
        UNIT_ASSERT(DriveAPI.GetTagsManager().GetTagsMeta().RegisterTag(new TTagDescription(description), USER_ID_DEFAULT, session));
    }

    {
        auto localizations = Server.GetLocalizationAs<NLocalization::TLocalizationDB>();
        UNIT_ASSERT(localizations);
        auto addLocalization = [&] (const TString& key, const TString& value) {
            NLocalization::TResource record(key);
            record.MutableLocalizations().emplace_back("rus", value);
            if (auto prevResource = localizations->GetObject(key)) {
                record.SetRevision(prevResource->GetRevision());
            }
            UNIT_ASSERT(localizations->UpsertObject(record, USER_ROOT_DEFAULT, session));
        };
        addLocalization("flexible_pack_offer.duration.subtitle", "flexible_pack_offer overtime _OvertimePriceDiscount_ rubles per minute");
        addLocalization("flexible_pack_offer.mileage.subtitle", "flexible_pack_offer overrun _OverrunPriceDiscount_ rubles per mile");
        addLocalization("session.early_return.bill.title", "session.early_return.bill.title");
        addLocalization("error.incorrect_evolution.title", "error.incorrect_evolution.title");
        addLocalization("error.incorrect_evolution.message", "error.incorrect_evolution.message");
        addLocalization("chat.localization_test", "test42");
    }

    UNIT_ASSERT(session.Commit());
    {
        const auto& settings = NDrive::GetServer().GetSettings();
        TVector<TSetting> settingValues;
        settingValues.emplace_back("api.car_router.maps_router.enabled", "false");
        settingValues.emplace_back("document_photo_manager.check_photo_file", "false");
        settingValues.emplace_back("offers.switching.enabled", "true");
        settingValues.emplace_back("offers.store_fake", "true");
        settingValues.emplace_back("pack_offer_distance_threshold_push_template", "REMAININGDISTANCE left");
        settingValues.emplace_back("pack_offer_duration_threshold_push_template", "REMAININGDURATION left");
        settingValues.emplace_back("pack_offer.distance_threshold.push_tag_name", "test_push");
        settingValues.emplace_back("pack_offer.duration_threshold.push_tag_name", "test_push_smile");
        settingValues.emplace_back("mobile.notifications.allow_drop_off", "You can [color=#80FF00FF]drop[/color] the car here");
        settingValues.emplace_back("mobile.notifications.allow_parking", "You can [b]park[/b] the car here");
        settingValues.emplace_back("mobile.notifications.deny_riding", "You are outside of the allowed zone");
        settingValues.emplace_back("handlers.user_app/scanner/start.global_scanner_available", "true");
        settingValues.emplace_back("handlers.user_app/scanner/start.user_position", "37 55");
        settingValues.emplace_back("billing.no_funds_statuses", "not_enough_funds,authorization_reject");
        settingValues.emplace_back("billing.card_account_type", "89");
        settingValues.emplace_back("book.restore_chargable_session_tag", "true");
        settingValues.emplace_back("default_location", "37.632588 55.987779");
        settingValues.emplace_back("stub_router.enabled", "true");
        settingValues.emplace_back("router_api_offer_builder", "drive_router");
        settingValues.emplace_back("mobile.notifications.allow_drop_off", "ALLOW_DROP_CAR");
        settingValues.emplace_back("mobile.notifications.allow_parking", "ALLOW_PARKING");
        settingValues.emplace_back("mobile.notifications.deny_riding", "DENY_RIDING");
        settingValues.emplace_back("offers.fix_point.min_price", "0");
        settingValues.emplace_back("offers.fix_point.min_price_difference", "0");
        settingValues.emplace_back("offers.fix_point.available_area_only", "true");
        settingValues.emplace_back("offers.fix_point.threshold_haversine", "50");
        settingValues.emplace_back("offers.fix_point.threshold_distance", "50");
        settingValues.emplace_back("offers.long_term.feedback.enabled", "false");
        settingValues.emplace_back("offers.long_term.report_autohide_stats", "true");
        settingValues.emplace_back("offers.long_term.use_insurance_type_variable", "true");
        settingValues.emplace_back("offers.long_term.group_offers.enabled", "true");
        settingValues.emplace_back("bill.store_discount_info", "true");
        settingValues.emplace_back("user_settings.tag_name", "user_app_settings");
        settingValues.emplace_back("user_flags.tag_name", "user_app_flags");
        settingValues.emplace_back("telematics.client2", "true");
        settingValues.emplace_back("transportation.destination.enabled", "true");
        settingValues.emplace_back("evolution.delegation.free.finish_push_tag", "delegation_finished");
        settingValues.emplace_back("location_tags.store_history_list", "no_price,allow_drop_car");
        settingValues.emplace_back("model_promo.flag_action", "standart_offer_constructor");
        settingValues.emplace_back("handlers.api/yandex/offers/fixpoint.external_offer.enable_null_destination", "false");
        settingValues.emplace_back("handlers.api/yandex/offers/fixpoint.external_offer.offers_number", "5");
        settingValues.emplace_back("handlers.api/yandex/offers/fixpoint.sparcity.enabled", "true");
        settingValues.emplace_back("handlers.api/taxi/offers/standard.external_offer.offer_builder_type", "standart_offer_builder");
        settingValues.emplace_back("handlers.api/taxi/offers/standard.external_offer.offers_number", "5");
        settingValues.emplace_back("handlers.api/yandex/offers/book.multi_rent", "true");
        settingValues.emplace_back("handlers.api/leasing/signals/list.get_signals_process2", "true");
        settingValues.emplace_back("external_offer.onboarding_fallback_user_id", USER_ID_DEFAULT);
        settingValues.emplace_back("session.user_sessions.depth", "2d");
        settingValues.emplace_back("tags_manager.user.cache_lifetime", "0");
        settingValues.emplace_back("tags_manager.user.use_search2", "true");
        settingValues.emplace_back("user_devices.cache_lifetime", "0");
        settingValues.emplace_back("user_roles.cache_lifetime", "0");
        settingValues.emplace_back("user.cache_lifetime", "0");
        settingValues.emplace_back("user_permissions.ignore_actuality", "true");
        settingValues.emplace_back("offers.distributing_block.shows_period", "3");
        settingValues.emplace_back("offers.distributing_block.non_shows_period", "5");
        settingValues.emplace_back("offers.distributing_block.enable_shows_restriction", "true");
        settingValues.emplace_back("offers.long_term.use_availability_flag_for_autohide", "true");
        bool success = false;
        for (ui32 i = 0; i < 10; ++i) {
            if (settings.SetValues(settingValues, USER_ROOT_DEFAULT)) {
                success = true;
                break;
            }
        }
        UNIT_ASSERT(success);
    }
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_role_actions_history");
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_role_roles_history");
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("user_roles_history");
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("area_tags_history");
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("tags_description_standart_history");
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_actions_standart_history");
}

TDriveAPIConfigGenerator::TDriveAPIConfigGenerator(const TString& databaseType/* = "SQLite"*/, const TString& databasePath/* = {}*/)
    : DatabaseType(GetEnv("DB_TYPE", databaseType))
    , DatabasePath(GetEnv("DB_PATH", databasePath ? databasePath : BinaryPath("drive/tests/resources/extmaps-carsharing-testing/extmaps-carsharing-testing.sqlite")))
    , OffersStorageName("offers_manager")
    , PrivateDataClientType("private_data_storage")
    , BillingPort(PortManager.GetPort())
{
    if (DatabaseType == "LocalPostgres") {
        std::string dumpFile = BinaryPath("drive/tests/resources/extmaps-carsharing-testing-pg/extmaps-carsharing-testing.sql");
        LocalPostgres = MakeHolder<maps::local_postgres::Database>();
        LocalPostgres->executeSql("ALTER DATABASE " + LocalPostgres->dbname() + " SET search_path TO public");
        LocalPostgres->executeSql("CREATE EXTENSION IF NOT EXISTS pg_trgm");
        LocalPostgres->executeSql("CREATE EXTENSION IF NOT EXISTS \"uuid-ossp\"");
        LocalPostgres->executeSqlFile(dumpFile, maps::local_postgres::ErrorHandling::Throw);
        DatabaseType = "Postgres";
        PostgresConfigGenerator.SetHost(LocalPostgres->host().c_str());
        PostgresConfigGenerator.SetPort(LocalPostgres->port());
        PostgresConfigGenerator.SetDb(LocalPostgres->dbname().c_str());
        PostgresConfigGenerator.SetUser(LocalPostgres->user().c_str());
        PostgresConfigGenerator.SetPassword(LocalPostgres->password().c_str());
        PostgresConfigGenerator.SetCertificate({});
    }
    bool sqlite = false;
    if (DatabaseType == "SQLite") {
        Y_ENSURE(DatabasePath);
        TFsPath workDir = GetWorkPath();
        TFsPath copy = workDir / (DatabasePath.Basename() + "." + ::ToString(Seconds()));
        DatabasePath.CopyTo(copy, true);
        DatabaseCopy = copy;
        sqlite = true;
    }
    if (auto database = CreateDatabase()) {
        NDrive::ApplyMigrations(*database, sqlite);
    }
}

void TDriveAPIConfigGenerator::SetOffersStorageName(const TString& name) {
    OffersStorageName = name;
}

void TDriveAPIConfigGenerator::SetPrivateDataClientType(const TString& value) {
    PrivateDataClientType = value;
}

void TDriveAPIConfigGenerator::DatabaseToString(IOutputStream& os) const {
    os << "    Type: " << DatabaseType << Endl;
    os << "    Name: extmaps-carsharing-testing" << Endl;
    os << "    Path: " << DatabaseCopy << Endl;
    os << "    ConnectionString: " << PostgresConfigGenerator.GetTestingConnectionString() << Endl;
    os << "    AgeForKillConnection: 30s" << Endl;
    os << "    ActivateTimeout: 300s" << Endl;
    os << "    ActivationSleep: 10s" << Endl;
}

void TDriveAPIConfigGenerator::ToString(IOutputStream& os, bool isYtEnabled) const {
    TString majorPasswordPath = GetEnv("MAJOR_PASSWORD_PATH", "");
    TString startrekTokenPath = GetEnv("TESTING_STARTREK_TOKEN_PATH", "");

    os << "Host: " << "carsharing.yandex.net" << Endl;
    os << "Port: " << "80" << Endl;

    os << "GeoAreasFreshness: 0s" << Endl;
    os << "OffersStorageName: " << OffersStorageName << Endl;
    os << "DefaultRoles: default_user" << Endl;
    os << "RequestPolicyName: rtline" << Endl;
    os << "<DeviceTagsHistory>" << Endl;
    os << "    Deep: 1d" << Endl;
    os << "    MaxHistoryDeep : 1000d" << Endl;
    os << "    UseCache : false" << Endl;
    os << "    ChunkSize : " << 10 * 1000 * 1000 << Endl;
    os << "</DeviceTagsHistory>" << Endl;
    os << "UseTagsHistoryFilter: true" << Endl;
    os << "PrivateDataClientType: " << PrivateDataClientType << Endl;
    os << "NeedDatasyncQueueClient: true" << Endl;
    if (isYtEnabled) {
        os << "YtToken: 0" << Endl;
    }
    os << "<StateFilters>" << Endl;
    os << "MaxAgeFilters: 0" << Endl;
    os << "MaxAgeObjects: 0" << Endl;
    os << "</StateFilters>" << Endl;
    os << "<Billing>" << Endl;
    os << "<Logics>" << Endl;
    os << "Modules: wallet,bonus,yandex_account,card,yandex_cashback" << Endl;
    os << "<wallet></wallet>" << Endl;
    os << "<bonus></bonus>" << Endl;
    os << "<card>" << Endl;
    os << "    Host: localhost" << Endl;
    os << "    Port: " << BillingPort << Endl;
    os << "    Https: 0" << Endl;
    os << "    RequestTimeout: 10s" << Endl;
    os << "    <RequestConfig>" << Endl;
    os << "        MaxAttempts: 1" << Endl;
    os << "        TimeoutSendingms: 900" << Endl;
    os << "    </RequestConfig>" << Endl;
    os << "</card>" << Endl;
    os << "<yandex_cashback>" << Endl;
    os << "    Host: localhost" << Endl;
    os << "    Port: " << BillingPort << Endl;
    os << "    Https: 0" << Endl;
    os << "    RequestTimeout: 10s" << Endl;
    os << "    <RequestConfig>" << Endl;
    os << "        MaxAttempts: 1" << Endl;
    os << "        TimeoutSendingms: 900" << Endl;
    os << "    </RequestConfig>" << Endl;
    os << "</yandex_cashback>" << Endl;
    os << "<yandex_account>" << Endl;
    os << "    Host: localhost" << Endl;
    os << "    Port: " << BillingPort << Endl;
    os << "    Https: 0" << Endl;
    os << "    RequestTimeout: 10s" << Endl;
    os << "    <RequestConfig>" << Endl;
    os << "        MaxAttempts: 1" << Endl;
    os << "        TimeoutSendingms: 900" << Endl;
    os << "    </RequestConfig>" << Endl;
    os << "</yandex_account>" << Endl;
    os << "</Logics>" << Endl;
    os << "    BatchUpdates: false" << Endl;
    os << "    MinimalPayment: 500" << Endl;
    os << "    NotifierName: test_start" << Endl;
    os << "    RobotUserId: 94c9653e-9f55-49f2-af8a-5156342bd9c8" << Endl;
    os << "    ActiveQueue: tests" << Endl;
    os << "    UseDBJsonStatements: false" << Endl;
    os << "</Billing>" << Endl;
    os << "DBName: test-drive-db" << Endl;
    os << "NeedDatasyncQueueClient: true" << Endl;
    os << "<Major>" << Endl;
    os << "    Host: b2btest.ma.ru" << Endl;
    os << "    Https: 1" << Endl;
    os << "    Port: 443" << Endl;
    os << "    UserName: yandex_api" << Endl;
    os << "    PasswordPath: " << majorPasswordPath << Endl;
    os << "</Major>" << Endl;
    os << "<Datasync>" << Endl;
    os << "    <TVM>" << Endl;
    os << "        Source: 2000184" << Endl;  // this is a TEST CLIENT, so it only has access to passport-test
    os << "        Destination: 2000060" << Endl;
    os << "        Token: 2MOzJtXfbLvZ3ySzj3zmRw" << Endl;  // this is a TEST TOKEN, so it doesn't give access to ANY actual prod collections
    os << "    </TVM>" << Endl;
    os << "    <RequestConfig>" << Endl;
    os << "        GlobalTimeout: 20000" << Endl;
    os << "    </RequestConfig>" << Endl;
    os << "    Host: api-stable.dst.yandex.net" << Endl;
    os << "    Port: 8080" << Endl;
    os << "    BaseRoute: v1/personality/profile/persdata/" << Endl;
    os << "</Datasync>" << Endl;
    os << "<Cars>" << Endl;
    os << "    IsSearchEnabled: 1" << Endl;
    os << "    HeavyIndexEnabled: 1" << Endl;
    os << "</Cars>" << Endl;
    os << "<Users>" << Endl;
    os << "    IsSearchEnabled: 1" << Endl;
    os << "    <RegistrationGeo>" << Endl;
    os << "        <moscow>" << Endl;
    os << "            Lat: 55.75222" << Endl;
    os << "            Lon: 37.61556" << Endl;
    os << "            LocationName: moscow" << Endl;
    os << "        </moscow>" << Endl;
    os << "        <spb>" << Endl;
    os << "            Lat: 59.93863" << Endl;
    os << "            Lon: 30.31413" << Endl;
    os << "            LocationName: spb" << Endl;
    os << "        </spb>" << Endl;
    os << "        <kazan>" << Endl;
    os << "            Lat: 55.78874" << Endl;
    os << "            Lon: 49.12214" << Endl;
    os << "            LocationName: kazan" << Endl;
    os << "        </kazan>" << Endl;
    os << "        <sochi>" << Endl;
    os << "            Lat: 43.58528" << Endl;
    os << "            Lon: 39.72028" << Endl;
    os << "            LocationName: sochi" << Endl;
    os << "        </sochi>" << Endl;
    os << "    </RegistrationGeo>" << Endl;
    os << "</Users>" << Endl;
    os << "<MDS>" << Endl;
    os << "    Host: s3.mdst.yandex.net" << Endl;
    os << "    Port: 80" << Endl;
    os << "    AccessSecretKey: ZlWrwNF3Y+LscjssWtPaPoQIDoYdW2MoQT1Yil5s" << Endl;
    os << "    AccessKeyId: cGowKRsmFFyu7hTXHj2p" << Endl;
    os << "    RequestTimeout: 60s" << Endl;
    os << "    EventLog: " << Endl;
    os << "    <RequestConfig>" << Endl;
    os << "        MaxAttempts: 10" << Endl;
    os << "        TasksCheckIntervalms: 5000" << Endl;
    os << "        GlobalTimeout: 60000" << Endl;
    os << "        TimeoutSendingms: 10000" << Endl;
    os << "    </RequestConfig>" << Endl;
    os << "</MDS>" << Endl;
    os << "<ZoneDb>" << Endl;
    os << "   Enable: true" << Endl;
    os << "</ZoneDb>" << Endl;
    os << "<DocumentPhotos>" << Endl;
    os << "    BucketName: carsharing-documents" << Endl;
    os << "    ContentEncryptionKey: 8906DC3630DAD635A4497AB3FBB879E1D3FD239A69E72A19390D126F149F39FC" << Endl;  // testing key, prod is different
    os << "</DocumentPhotos>" << Endl;
    os << "<Search>" << Endl;
    os << "    IsEnabled: true" << Endl;
    os << "</Search>" << Endl;
    os << "<Startrek>" << Endl;
    os << "    Host: st-api.test.yandex-team.ru" << Endl;
    os << "    Port: 443" << Endl;
    os << "    IsHttps: true" << Endl;
    os << "    Account: robot-carsharing" << Endl;
    os << "    TokenPath: " << startrekTokenPath << Endl;
    os << "    RequestTimeout: 1s" << Endl;
    os << "    <RequestConfig>" << Endl;
    os << "        MaxAttempts: 1" << Endl;
    os << "    </RequestConfig>" << Endl;
    os << "</Startrek>" << Endl;
}

NStorage::IDatabase::TPtr TDriveAPIConfigGenerator::CreateDatabase() const {
    TStringStream ss;
    DatabaseToString(ss);
    THolder<TAnyYandexConfig> config(new TAnyYandexConfig);
    UNIT_ASSERT(config->ParseMemory(ss.Str()));

    NStorage::TDatabaseConfig databaseConfig;
    databaseConfig.Init(config->GetRootSection());
    return databaseConfig.ConstructDatabase();
}

THolder<TAnyYandexConfig> TDriveAPIConfigGenerator::GenerateConfig(bool isYtEnabled=false) const {
    TStringStream ss;
    ToString(ss, isYtEnabled);
    THolder<TAnyYandexConfig> config(new TAnyYandexConfig);
    UNIT_ASSERT(config->ParseMemory(ss.Str()));
    return config;
}

THolder<TDriveAPIConfig> TDriveAPIConfigGenerator::Generate() const {
    auto result = MakeHolder<TDriveAPIConfig>();
    result->Init(GenerateConfig()->GetRootSection(), nullptr);
    return result;
}

ui32 TDriveAPIConfigGenerator::GetBillingPort() const {
    return BillingPort;
}

TDriveAPIConfigGenerator::~TDriveAPIConfigGenerator() = default;

namespace {
    TMutex StreamMutex;
    THolder<IOutputStream> CanonizationStream;
    THolder<IOutputStream> DescriptionStream;
}

namespace {
    void Scan(TStringBuf key, NJson::TJsonValue& value, const std::function<void(TStringBuf, NJson::TJsonValue&)>& scanner) {
        if (value.IsMap()) {
            for (auto&& i : value.GetMapSafe()) {
                Scan(i.first, i.second, scanner);
            }
        } else if (value.IsArray()) {
            for (auto&& i : value.GetArraySafe()) {
                Scan({}, i, scanner);
            }
        } else {
            scanner(key, value);
        }
    }

    const TSet<TString> DurationFields = {
        "extension",
        "free_time",
        "free_reservation",
        "pack",
        "remaining_time",
        "riding_duration",
    };

    const TSet<TString> TimestampFields = {
        "session_acceptance",
        "session_acceptance_start",
        "session_acceptance_finish",
        "current_fuel_level_timestamp",
        "start_fuel_level_timestamp",
        "finish_fuel_level_timestamp",
        "Created",
        "Deadline",
        "RandomId",
        "Timestamp",
        "timestamp",
        "since",
        "location",
        "blackbox",
        "created",
        "heartbeat",
        "deadline",
        "server_time",
        "started",
        "finished",
        "start",
        "current",
        "expected_finish",
        "finish",
        "finish_instant",
        "start_instant",
        "available_since",
        "minimal_period",
        "maximal_since",
        "since",
        "until",
    };

    const TSet<TString> CoordinateFields = {
        "lat",
        "latitude",
        "lon",
        "longitude"
    };

    const TSet<TString> TypeFields = {
        "has_all_of",
        "is_engine_on",
        "number",
        "car_number",
        "status",
        "VariativeDetector",
        "host",
        "imei",
        "deeplink",
        "mileage_limit",
        "price",
        "pack_price",
        "pack_price_undiscounted",
        "deposit",
        "localized_riding_duration",
        "localized_price",
        "monthly_cost",
        "source_location",
    };

    constexpr i64 DurationReplacement = 42;
    constexpr i64 TimestampReplacement = 1000000042;
    const TString UUIDReplacement = "UUID";
    const TString BooleanTypeReplacement = "boolean_type";
    const TString IntegerTypeReplacement = "integer_type";
    const TString StringTypeReplacement = "string_type";
}

TString NDrive::TServerConfigGenerator::Canonize(const TString& value) const {
    TCgiParameters cgi;
    cgi.Scan(value);
    for (auto&& i : cgi) {
        const TString& k = i.first;
        TString& v = i.second;

        TGUID discarded;
        if (GetGuid(v, discarded) || GetUuid(v, discarded)) {
            StringTokens.emplace(v, UUIDReplacement);
            v = UUIDReplacement;
            continue;
        }

        ui64 timestamp;
        if (TryFromString(v, timestamp)) {
            if (timestamp > 1500000000 && timestamp < 2000000000) {
                IntegerTokens.emplace(timestamp, TimestampReplacement);
                v = ::ToString(TimestampReplacement);
            }
        }

        if (TypeFields.contains(k)) {
            v = StringTypeReplacement;
            continue;
        }
    }
    if (!cgi.empty()) {
        return cgi.Print();
    }

    TString result = value;
    for (auto&& i : StringTokens) {
        SubstGlobal(result, i.first, i.second);
    }
    for (auto&& i : IntegerTokens) {
        SubstGlobal(result, ::ToString(i.first), ::ToString(i.second));
    }
    return result;
}

TString NDrive::TServerConfigGenerator::Canonize(const NJson::TJsonValue& value) const {
    NJson::TJsonValue result = value;
    Scan({}, result, [this](TStringBuf key, NJson::TJsonValue& v) {
        if (v.IsString()) {
            const auto& value = v.GetStringSafe();
            TGUID discarded;
            if (GetUuid(value, discarded) || GetGuid(value, discarded)) {
                auto p = StringTokens.find(value);
                if (p == StringTokens.end()) {
                    p = StringTokens.emplace(value, UUIDReplacement).first;
                }
                v = p->second;
                return;
            }
            if (key.EndsWith("_hr") || key.StartsWith("hr_")) {
                auto p = StringTokens.find(value);
                if (p == StringTokens.end()) {
                    p = StringTokens.emplace(value, "HR").first;
                }
                v = p->second;
                return;
            }
            if (value.StartsWith("ttt") && value.EndsWith("ccc")) {
                auto p = StringTokens.find(value);
                if (p == StringTokens.end()) {
                    p = StringTokens.emplace(value, "autogenerated_car_number").first;
                }
                v = p->second;
                return;
            }
            if (TypeFields.contains(key)) {
                StringTokens.emplace(v.GetString(), StringTypeReplacement);
                v = StringTypeReplacement;
                return;
            }
            return;
        }
        if (v.IsInteger()) {
            if (TimestampFields.contains(key) || key.EndsWith("_updated_at") || key.EndsWith("_instant")) {
                i64 value = v.GetIntegerSafe();
                auto p = IntegerTokens.find(value);
                if (p == IntegerTokens.end()) {
                    p = IntegerTokens.emplace(value, TimestampReplacement).first;
                }
                v = p->second;
                return;
            }
            if (DurationFields.contains(key) || key.EndsWith("duration") || key.StartsWith("old_state_")) {
                i64 value = v.GetIntegerSafe();
                auto p = IntegerTokens.find(value);
                if (p == IntegerTokens.end()) {
                    p = IntegerTokens.emplace(value, DurationReplacement).first;
                }
                v = p->second;
                return;
            }
            if (TypeFields.contains(key)) {
                v = IntegerTypeReplacement;
                return;
            }
            return;
        }
        if (v.IsDouble()) {
            if (!CoordinateFields.contains(key)) {
                return;
            }

            double value = v.GetDoubleSafe();
            if (value > 30 && value < 40) {
                v = i64(37);
            } else if (value > 50 && value < 60) {
                v = i64(55);
            } else {
                v = static_cast<i64>(value);
            }
            return;
        }
        if (v.IsBoolean()) {
            if (TypeFields.contains(key)) {
                v = BooleanTypeReplacement;
                return;
            }
            return;
        }
    });
    return NJson::WriteJson(result, /*formatOutput=*/true, /*sortKeys=*/true);
}

TString NDrive::TServerConfigGenerator::Canonize(const NUtil::THttpReply& reply) const {
    TStringStream ss;
    if (const TString& error = reply.ErrorMessage()) {
        ss << "ERROR: " << Canonize(error) << Endl;
    }
    if (const TString& content = reply.Content()) {
        ss << "CONTENT:" << Endl;
        NJson::TJsonValue json;
        if (NJson::ReadJsonFastTree(content, &json)) {
            ss << Canonize(json) << Endl;
        } else {
            ss << Canonize(content) << Endl;
        }
    }
    return ss.Str();
}

TString NDrive::TServerConfigGenerator::Canonize(const NNeh::THttpRequest& request) const {
    TStringStream ss;
    ss << "REQUEST " << request.GetRequestType() << " " << request.GetUri() << "?" << Canonize(request.GetCgiData()) << Endl;
    for (auto&& [key, value] : request.GetHeaders()) {
        auto v = value;
        if (key == "Authorization") {
            auto p = StringTokens.find(value);
            if (p == StringTokens.end()) {
                p = StringTokens.emplace(value, "AuthorizationUser").first;
            }
            v = p->second;
        }
        ss << key << ": " << v << Endl;
    }
    if (!request.GetPostData().Empty()) {
        TString post(request.GetPostData().AsCharPtr(), request.GetPostData().Size());
        NJson::TJsonValue json;
        if (NJson::ReadJsonFastTree(post, &json)) {
            ss << Canonize(json) << Endl;
        } else {
            ss << Canonize(post) << Endl;
        }
    }
    return ss.Str();
}

IOutputStream& GetOutputStream(THolder<IOutputStream>& outputStream, const TString& filePath) {
    TGuard<TMutex> g(::StreamMutex);
    if (!outputStream) {
        const TString filename = GetEnv(filePath);
        if (!filename) {
            outputStream.Reset(new TNullOutput);
        }
        else {
            outputStream.Reset(new TOFStream(filename));
        }
    }
    return *outputStream;
}

IOutputStream& GetCanonizationStream() {
    return GetOutputStream(::CanonizationStream, "DumpPath");
}

IOutputStream& GetDescriptionStream() {
    return GetOutputStream(::DescriptionStream, "DescriptionPath");
}

bool NDrive::TServerConfigGenerator::WaitLocation(const TString& carId, TMaybe<TGeoCoord> coordinate /*= {}*/, TMaybe<TString> tag /*= {}*/) {
    for (ui32 i = 0; i < 100; ++i) {
        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(*this);
        NJson::TJsonValue report = GetCarsList("/admin_app/car/list", { carId }, USER_ROOT_DEFAULT, {}, nullptr, "traits=ReportLocationDetails");
        NJson::TJsonValue location = report["cars"][0]["location"];
        if (location.IsMap()) {
            TGeoCoord current = {
                    location["lon"].GetDoubleRobust(),
                    location["lat"].GetDoubleRobust()
            };
            NDrive::TLocationTags tags;
            for (auto&& tag : location["tags"].GetArraySafe()) {
                tags.insert(tag.GetStringSafe());
            }
            INFO_LOG << carId << ": location " << current.ToString() << Endl;
            bool coordinateMatched = coordinate ? coordinate->GetLengthTo(current) < 10 : true;
            bool tagMatched = tag ? tags.contains(*tag) : true;
            if (coordinateMatched && tagMatched) {
                return true;
            }
        } else {
            INFO_LOG << carId << ": no location: " << NJson::WriteJson(report) << Endl;
        }
        Sleep(TDuration::Seconds(1));
    }
    return false;
}

bool NDrive::TServerConfigGenerator::WaitSensor(const TString& carId, TStringBuf sensor, TStringBuf expected) {
    for (ui32 i = 0; i < 100; ++i) {
        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(*this);
        NJson::TJsonValue report = GetCarsList("/api/staff/car/info", { carId }, USER_ROOT_DEFAULT);
        NJson::TJsonValue value = report["telematics"][sensor];
        if (value.IsDefined()) {
            auto valueFloat = value.GetDouble();
            auto valueString = value.GetStringRobust();
            INFO_LOG << carId << ": sensor " << sensor << " " << valueString << Endl;
            if (expected) {
                if (valueString == expected) {
                    return true;
                } else {
                    double expectedFloat;
                    if (TryFromString(expected, expectedFloat) && std::abs(valueFloat - expectedFloat) < 0.001) {
                        return true;
                    }
                }
            } else {
                return true;
            }
        } else {
            INFO_LOG << carId << ": no sensor " << sensor << Endl;
        }
        Sleep(TDuration::Seconds(1));
    }
    return false;
}

bool NDrive::TServerConfigGenerator::WaitSensorState(const TString& carId, TStringBuf sensor, TStringBuf expected) {
    for (ui32 i = 0; i < 100; ++i) {
        NDrive::TServerConfigGenerator::TDisableLogging disableLogging(*this);
        NJson::TJsonValue report = NJson::JSON_NULL;
        {
            NNeh::THttpRequest request;
            request.SetUri("/api/staff/telematics/info").SetCgiData("car_id=" + carId);
            request.AddHeader("Authorization", USER_ROOT_DEFAULT);
            NUtil::THttpReply result = GetSendReply(request);
            if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &report)) {
                ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
                return false;
            }
        }
        if (!report["sensors"].IsArray()) {
            ERROR_LOG << "Wrong response " << report << Endl;
            return false;
        }
        auto& sensors = report["sensors"].GetArray();
        auto pre = [&sensor](const auto& val) { return val["name"] == sensor; };
        auto it = FindIf(sensors.begin(), sensors.end(), pre);
        if (it != sensors.end()) {
            auto& value = (*it)["value"];
            auto valueFloat = value.GetDouble();
            auto valueString = value.GetStringRobust();
            INFO_LOG << carId << ": sensor " << sensor << " " << valueString << Endl;
            if (valueString == expected) {
                return true;
            } else {
                double expectedFloat;
                if (TryFromString(expected, expectedFloat) && std::abs(valueFloat - expectedFloat) < 0.001) {
                    return true;
                }
            }
        } else {
            INFO_LOG << carId << ": no sensor " << sensor << Endl;
        }
        Sleep(TDuration::Seconds(1));
    }
    return false;
}

bool NDrive::TServerConfigGenerator::WaitStatus(const TString& carId, TStringBuf status, const NDrive::IServer& server, TDuration waitingDuration) {
    TInstant start = Now();
    TString statusCurrent;
    SendGlobalMessage<NDrive::TCacheRefreshMessage>("car_tags_history");
    while (Now() - start < waitingDuration) {
        auto objStatuses = server.GetDriveAPI()->GetStateFiltersDB()->GetObjectStates();
        auto it = objStatuses.find(carId);
        if (it != objStatuses.end()) {
            statusCurrent = it->second;
        } else {
            statusCurrent = "UNKNOWN";
        }
        if (statusCurrent != status) {
            Sleep(TDuration::MilliSeconds(200));

            auto snapshot = server.GetSnapshotsManager().GetSnapshot(carId);

            TVector<TDBTag> tags;
            TMaybe<TTaggedDevice> td = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(carId, TInstant::Zero());
            UNIT_ASSERT(td);
            const TString statusCache = server.GetDriveAPI()->GetStateFiltersDB()->CalcStatus(td->GetTags(), snapshot.GetSensors());
            td = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(carId, Now());
            const TString statusActual = server.GetDriveAPI()->GetStateFiltersDB()->CalcStatus(td->GetTags(), snapshot.GetSensors());
            INFO_LOG << statusCache << "/" << statusActual << " / " << status << " / " << statusCurrent << Endl;

            continue;
        }
        return true;
    }
    ERROR_LOG << "cannot wait status " << status << " / " << statusCurrent << " for " << carId << Endl;
    return false;
}

bool NDrive::TServerConfigGenerator::WaitCar(const TString& carId) {
    return WaitLocation(carId) && WaitSensor(carId, "mileage") && WaitSensor(carId, "fuel_level");
}

NDrive::TServerConfigGenerator::~TServerConfigGenerator() {
    TGuard<TMutex> g(::StreamMutex);
    if (::CanonizationStream) {
        ::CanonizationStream->Flush();
    }
    if (::DescriptionStream) {
        ::DescriptionStream->Flush();
    }
}

void NDrive::TServerConfigGenerator::SetOffersStorageName(const TString& name) {
    DriveAPIGenerator.SetOffersStorageName(name);
}

void NDrive::TServerConfigGenerator::SetPrivateDataClientType(const TString& value) {
    DriveAPIGenerator.SetPrivateDataClientType(value);
}

bool GetSchemeFromFile(const TString& path, NDrive::TScheme& scheme) {
    auto content = TIFStream(BinaryPath(path)).ReadAll();
    NJson::TJsonValue json;
    return NJson::ReadJsonTree(content, &json) && scheme.DeserializeFromJson(json);
}

bool TValidationManager::TProcessorTemplates::TProcessоrStateTemplates::Init(const TYandexConfig::Section* section) {
    UNIT_ASSERT(section);
    Name = section->Name;
    UNIT_ASSERT(Name);
    Description = section->GetDirectives().Value<TString>("StateDescription", Description);
    RequiredChecksCount = section->GetDirectives().Value<ui32>("RequiredChecksCount", RequiredChecksCount);
    TString descriptionFilePath = section->GetDirectives().Value<TString>("DescriptionScheme", "");
    if (descriptionFilePath) {
        NDrive::TScheme scheme;
        if (!GetSchemeFromFile(descriptionFilePath, scheme)) {
            ERROR_LOG << "Incorrect description scheme in file " << descriptionFilePath << Endl;
            return false;
        }
        DescriptionScheme = scheme;
        ValidationSchemes.push_back(scheme);
    }
    TVector<TString> validationFilePaths = SplitString(section->GetDirectives().Value<TString>("ValidationSchemes", ""), ",");
    for (const auto& path : validationFilePaths) {
        NDrive::TScheme scheme;
        if (!GetSchemeFromFile(path, scheme)) {
            ERROR_LOG << "Incorrect validation scheme in file " << path << Endl;
            return false;
        }
        ValidationSchemes.push_back(scheme);
    }
    return !ValidationSchemes.empty();
}

bool TValidationManager::TProcessorTemplates::TProcessоrStateTemplates::ValidateJson(const NJson::TJsonValue& json) const {
    INFO_LOG << "Check state " << Name << " : " << json.GetStringRobust() << Endl;
    for (const auto& scheme : ValidationSchemes) {
        if (!scheme.ValidateJson(json)) {
            return false;
        }
    }
    ++ChecksCount;
    return true;
}

bool TValidationManager::TProcessorTemplates::TProcessоrStateTemplates::DescribeJson(const NJson::TJsonValue& json, TString& description) const {
    if (!DescriptionScheme) {
        return false;
    }
    try {
        description = DescriptionScheme->MakeJsonDescription(json);
    } catch (const yexception& ex) {
        ERROR_LOG << ex.what() << Endl;
        description = ex.what();
        return false;
    }
    return true;
}

bool TValidationManager::TProcessorTemplates::TProcessоrStateTemplates::CheckValidationCount() const {
    return ChecksCount >= RequiredChecksCount;
}

bool TValidationManager::TProcessorTemplates::Init(const TYandexConfig::Section* section) {
    UNIT_ASSERT(section);
    ProcessorName = section->Name;
    for (const auto& state : section->GetAllChildren()) {
        if (!States[state.first].Init(state.second)) {
            return false;
        }
    }
    return true;
}

bool TValidationManager::TProcessorTemplates::ValidateJson(const TString& state, const NJson::TJsonValue& json, bool required) const {
    auto itState = States.find(state);
    if (itState == States.end()) {
        WARNING_LOG << "No validation schema for state " << state << " for processor " << ProcessorName << Endl;
        return !required;
    }
    return itState->second.ValidateJson(json);
}

bool TValidationManager::TProcessorTemplates::DescribeJson(const TString& state, const NJson::TJsonValue& json, TString& description) const {
    auto itState = States.find(state);
    if (itState == States.end() || !itState->second.DescribeJson(json, description)) {
        WARNING_LOG << "No description schema for state " << state << " for processor " << ProcessorName << Endl;
        return false;
    }
    return true;
}

bool TValidationManager::TProcessorTemplates::CheckValidationCount() const {
    for (const auto& state : States) {
        if (!state.second.CheckValidationCount()) {
            return false;
        }
    }
    return true;
}

bool TValidationManager::TProcessorTemplates::GetStateDescription(const TString& state, TString& description) const {
    auto itState = States.find(state);
    if (itState == States.end()) {
        WARNING_LOG << "No description schema for state: " << state << Endl;
        return false;
    }
    description = itState->second.GetStateDescription();
    return true;
}

bool TValidationManager::ValidateAndDescribeJson(TValidationManager::EProcessorAction action, const TString& uri, const TString& state, const NJson::TJsonValue& json, bool required) const {
    TStringBuf processorBuf = uri;
    while (processorBuf.SkipPrefix("/")) {}
    TString processor(processorBuf);

    DescribeJson(action, processor, state, json);
    return ValidateJson(action, processor, state, json, required);
}

bool TValidationManager::ValidateJson(TValidationManager::EProcessorAction action, const TString& uri, const TString& state, const NJson::TJsonValue& json, bool required) const {
    auto itAction = Templates.find(action);
    if (itAction == Templates.end()) {
        return false;
    }
    auto it = itAction->second.find(uri);
    if (it == itAction->second.end()) {
        WARNING_LOG << "No schemes for processor: " << uri << Endl;
        return !required;
    }
    return it->second.ValidateJson(state, json, required);
}

bool TValidationManager::DescribeJson(TValidationManager::EProcessorAction action, const TString& uri, const TString& state, const NJson::TJsonValue& json) const {
    auto itAction = Templates.find(action);
    if (itAction == Templates.end()) {
        return false;
    }
    auto it = itAction->second.find(uri);
    TString description;
    TString stateDescription;
    if (it == itAction->second.end() || !it->second.GetStateDescription(state, stateDescription) || !it->second.DescribeJson(state, json, description)) {
        WARNING_LOG << "Invalid description scheme for processor: " << uri << Endl;
        return false;
    }
    GetDescriptionStream() << action << " FOR " << uri << " IN STATE " << state << Endl;
    if (WikiLayout) {
        GetDescriptionStream() << "<{";
        if (!stateDescription) {
            GetDescriptionStream() << "link" << Endl;
        }
    }
    GetDescriptionStream() << stateDescription << Endl;
    if (WikiLayout) {
        GetDescriptionStream() << "%%" << Endl;
    }
    GetDescriptionStream()  << description;
    if (WikiLayout) {
        GetDescriptionStream() << "%%" << Endl << "}>";
    }
    GetDescriptionStream() << Endl << Endl;
    return true;
}

bool NDrive::TServerConfigGenerator::ValidateAndDescribeReply(const TString& uri, const NUtil::THttpReply& reply, bool required) const {
    NJson::TJsonValue json = NJson::TJsonValue::UNDEFINED;
    if (reply.ErrorMessage() || (!NJson::ReadJsonTree(reply.Content(), &json) && required)) {
        return false;
    }
    return ValidateAndDescribeJson(TValidationManager::EProcessorAction::Reply, uri, json, required);
}

bool NDrive::TServerConfigGenerator::ValidateAndDescribeRequest(const NNeh::THttpRequest& request, bool required) const {
    if (request.GetRequestType() == "GET" && !request.GetPostData().Empty()) {
        return false;
    }

    TStringBuf post(request.GetPostData().AsCharPtr(), request.GetPostData().Size());
    NJson::TJsonValue json;
    if (NJson::ReadJsonFastTree(post, &json)) {
        return ValidateAndDescribeJson(TValidationManager::EProcessorAction::Request, request.GetUri(), json, required);
    }
    return !required;
}

bool TValidationManager::AddStateTemplates(TValidationManager::EProcessorAction action, const TYandexConfig::TSectionsMap& rootSections) {
    auto localSection = rootSections.find(::ToString(action));
    auto& templates = Templates[action];
    if (localSection != rootSections.end()) {
        for (const auto& processor : localSection->second->GetAllChildren()) {
            TProcessorTemplates processorTemplates;
            if (!processorTemplates.Init(processor.second)) {
                return false;
            }
            templates.emplace(processor.first, processorTemplates);
        }
    }
    return true;
}

bool TValidationManager::Init(const TYandexConfig::Section* section) {
    WikiLayout = IsTrue(GetEnv("WikiLayout", "false"));
    UNIT_ASSERT(section);
    auto rootSections = section->GetAllChildren();
    for (const auto& action : GetEnumAllValues<TValidationManager::EProcessorAction>()) {
        if (!AddStateTemplates(action, rootSections)) {
            return false;
        }
    }
    return true;
}

TValidationManager::~TValidationManager() {
    for (const auto& action : Templates) {
        for (const auto& processor : action.second) {
            UNIT_ASSERT(processor.second.CheckValidationCount());
        }
    }
}


NDrive::TServerConfigGenerator::TServerConfigGenerator(const TString& databaseType, const TString& databasePath, const TString& sensorApi, const TString& sensorsDumpPath, const TString& stateTemplatesPath)
    : SensorApiName(sensorApi)
    , SensorDump(sensorsDumpPath ? sensorsDumpPath : BinaryPath("drive/tests/resources/sensors/drive_cache.json"))
    , StateTemplatesPath(stateTemplatesPath ? stateTemplatesPath : BinaryPath("drive/tests/resources/state_schemes/state_config.conf"))
    , DriveAPIGenerator(databaseType, databasePath)
{
#ifdef _win_
    ControllerPort = 8000;
    ServerPort = 16000;
#else
    ControllerPort = Singleton<TPortManager>()->GetPort();
    ServerPort = Singleton<TPortManager>()->GetPort();
#endif
    {
        TFileOutput fileOutput("./test_port");
        fileOutput << ServerPort;
    }
    MetaConfig.SetMaxAttempts(1).SetGlobalTimeout(TDuration::Seconds(60));
    NehAgent = MakeHolder<NNeh::THttpClient>(MetaConfig);
    NehAgent->RegisterSource("frontend", "localhost", ServerPort, MetaConfig);
    GetCanonizationStream();
    GetDescriptionStream();
    if (!SensorApiName && SensorDump) {
        auto content = TIFStream(SensorDump).ReadAll();
        auto serialized = NJson::ReadJsonFastTree(content);
        UNIT_ASSERT(TLocalSensorStorage::Instance()->TryFromJson(serialized));
    }
    if (StateTemplatesPath) {
        auto configStr = TIFStream(StateTemplatesPath).ReadAll();
        TAnyYandexConfig config;
        UNIT_ASSERT(config.ParseMemory(configStr.data()));
        UNIT_ASSERT(ValidationManager.Init(config.GetRootSection()));
    }
    StringTokens.emplace(USER_ID_NOPHONE, USER_ID_NOPHONE);
    StringTokens.emplace(USER_ID_DEFAULT, USER_ID_DEFAULT);
    StringTokens.emplace(USER_ID_TECH, USER_ID_TECH);
    StringTokens.emplace(USER_ID_BLOCKED, USER_ID_BLOCKED);
    StringTokens.emplace(USER_ID_DEFAULT2, USER_ID_DEFAULT2);
    StringTokens.emplace(USER_ID_DEFAULT1, USER_ID_DEFAULT1);
    StringTokens.emplace(USER_ROOT_DEFAULT, USER_ROOT_DEFAULT);

    if (IsTrue(GetEnv("UseMockTanker", "true"))) {
        TankerMock = MakeAtomicShared<TTankerEmulator>();
        TankerMock->Run(TankerMockPort);
    }
}

void NDrive::TServerConfigGenerator::ToString(IOutputStream& os) const {
    os << "<DaemonConfig>" << Endl;
    os << "    LoggerType : console" << Endl;
    os << "    LogLevel : " << LogLevel << Endl;
    os << "    LogRotation : false" << Endl;
    os << "    MetricsMaxAge : 10" << Endl;
    os << "    MetricsPrefix : Refresh_" << Endl;
    os << "    StdOut: current.stdout" << Endl;
    os << "    StdErr: current.stderr" << Endl;
    os << "    <Controller>" << Endl;
    os << "        ClientTimeout : 200" << Endl;
    os << "        ConfigsControl : false" << Endl;
    os << "        ConfigsRoot : ./" << Endl;
    os << "        Log : current-controller.log" << Endl;
    os << "        MaxConnections : 0" << Endl;
    os << "        MaxQueue : 0" << Endl;
    os << "        Port : " << ControllerPort << Endl;
    os << "        StartServer : 1" << Endl;
    os << "        StateRoot : ./" << Endl;
    os << "        Threads : 20" << Endl;
    os << "        <DMOptions>" << Endl;
    os << "            ConnectionTimeout : 100" << Endl;
    os << "            CType : prestable" << Endl;
    os << "            Enabled : 0" << Endl;
    os << "            Host : localhost" << Endl;
    os << "            InteractionTimeout : 60000" << Endl;
    os << "            Port : 11111" << Endl;
    os << "            Slot : asdasda:12312" << Endl;
    os << "            TimeoutS : 45" << Endl;
    os << "        </DMOptions>" << Endl;
    os << "    </Controller>" << Endl;
    os << "</DaemonConfig>" << Endl;

    os << "<Server>" << Endl;
    os << "    TransactionNamesEnabled: " << true << Endl;
    os << "    <Chat>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "        <MessagesHistoryConfig>" << Endl;
    os << "            Deep: 0" << Endl;
    os << "        </MessagesHistoryConfig>" << Endl;
    os << "    </Chat>" << Endl;
    os << "    <RadarGeohash>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "    </RadarGeohash>" << Endl;
    if (YdbEndpoint && YdbDatabase) {
        os << "    <YDB>" << Endl;
        os << "        DBName: testing-drive-ydb" << Endl;
        os << "    </YDB>"  << Endl;
    }
    os << "    <MapsRouter>" << Endl;
    os << "    </MapsRouter>" << Endl;
    os << "    <ParkingZonesManager>" << Endl;
    os << "        RTLineAPIName: surge" << Endl;
    os << "    </ParkingZonesManager>" << Endl;
    os << "    <UsersController>" << Endl;
    os << "        SurgeServiceName: surge" << Endl;
    os << "    </UsersController>" << Endl;

    os << "    <PromoCodesManager>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "    </PromoCodesManager>" << Endl;

    if (NeedBackground & EServerBackgrounds::SurgeConstructor) {
        os << "    <SurgeConstructor>" << Endl;
        os << "        RTLineAPIName: drive_cache_prestable" << Endl;
        os << "        CarRouterAPI: drive_router" << Endl;
        os << "        TagNames: " << SurgeTag << Endl;
        os << "    </SurgeConstructor>" << Endl;
    }
    if (NeedBackground & EServerBackgrounds::FuelingService) {
        os << "    <FuelingManager>" << Endl;
        if (IsTrue(GetEnv("UseMockTanker", "true"))) {
            os << "        Host: localhost" << Endl;
            os << "        Port: " << TankerMockPort << Endl;
            os << "        Freshness: 6h" << Endl;
        } else {
            os << "        Host: app.tst.tanker.yandex.net" << Endl;
            os << "        Port: 80" << Endl;
        }
        os << "        IsHttps: false" << Endl;
        os << "        ApiKey: l02cispnptoq4o4hmp7brs" << Endl;
        os << "        <RequestPolicy>" << Endl;
        os << "            GlobalTimeout: 10000" << Endl;
        os << "        </RequestPolicy>" << Endl;
        os << "    </FuelingManager>" << Endl;
    }
    os << "    <RequestPolicy>" << Endl;
    os << "        <rtline>" << Endl;
    os << "            GlobalTimeout: 200000" << Endl;
    os << "            ConnectTimeout: 100" << Endl;
    os << "            TasksCheckIntervalms: 5000" << Endl;
    os << "            MaxAttempts: 3" << Endl;
    os << "        </rtline>" << Endl;
    os << "        <rtline_indexing>" << Endl;
    os << "            GlobalTimeout: 10000" << Endl;
    os << "            ConnectTimeout: 100" << Endl;
    os << "            TasksCheckIntervalms: 3000" << Endl;
    os << "            MaxAttempts: 3" << Endl;
    os << "        </rtline_indexing>" << Endl;
    os << "    </RequestPolicy>" << Endl;
    os << "    <AuthModules>" << Endl;
    os << "        <fake>" << Endl;
    os << "            Type: fake" << Endl;
    os << "            DefaultUserId: " << USER_ID_DEFAULT << Endl;
    os << "        </fake>" << Endl;
    os << "        <yang>" << Endl;
    os << "            Type: yang" << Endl;
    os << "            <YangClientConfig>" << Endl;
    os << "                 Uri: https://yang.yandex-team.ru" << Endl;
    os << "                 AuthType: OAuth" << Endl;
    os << "                 AuthToken: o" << Endl;
    os << "                 PathPrefix: /api/v1/" << Endl;
    os << "            </YangClientConfig>" << Endl;
    os << "            CheckIsActivePair: false" << Endl;
    os << "            DefaultUserId: 1bda8d81-db1f-41a0-8dca-b50d60980d6f" << Endl;
    os << "        </yang>" << Endl;
    os << "        <mosgorpas>" << Endl;
    os << "            Type: external_access_token" << Endl;
    os << "        </mosgorpas>" << Endl;
    os << "    </AuthModules>" << Endl;
    os << "    <Datasync>" << Endl;
    os << "        Host: api-stable.dst.yandex.net" << Endl;
    os << "        Port: 8080" << Endl;
    os << "    </Datasync>" << Endl;
    os << "    <HttpServer>" << Endl;
    os << "        Port: " << ServerPort << Endl;
    os << "        Threads: 16" << Endl;
    os << "    </HttpServer>" << Endl;
    os << "    <RequestHandlers>" << Endl;
    os << "        <default>" << Endl;
    os << "            ThreadsCount: 16" << Endl;
    os << "        </default>" << Endl;
    os << "        Threads: 16" << Endl;
    os << "    </RequestHandlers>" << Endl;

    os << "    <LPMClient>" << Endl;
    os << "        Name: LPM" << Endl;
    os << "        ServiceToken: Fake" << Endl;
    os << "        Host: localhost:" << DriveAPIGenerator.GetBillingPort() << Endl;
    os << "        RequestTimeout: 10s" << Endl;
    os << "        <RequestConfig>" << Endl;
    os << "            MaxAttempts: 1" << Endl;
    os << "            TimeoutSendingms: 900" << Endl;
    os << "        </RequestConfig>" << Endl;
    os << "        <CacheStorages>";
    os << "            <redis-main>" << Endl;
    os << "                CacheType: Redis" << Endl;
    os << "                DBName: unknown-drive-redis" << Endl;
    os << "                RefreshInterval: 5s" << Endl;
    os << "                CacheId: redis1" << Endl;
    os << "            </redis-main>" << Endl;
    os << "            <local-main>" << Endl;
    os << "                CacheType: local" << Endl;
    os << "                CacheId: local1" << Endl;
    os << "            </local-main>" << Endl;
    os << "        </CacheStorages>";
    os << "     </LPMClient>" << Endl;

    os << "    <RTLineAPIs>" << Endl;
    os << "       <surge>" << Endl;
    os << "           Host: saas-searchproxy-maps-prestable.yandex.net" << Endl;
    os << "           Port: 17000" << Endl;
    os << "           ServiceName: drive" << Endl;
    os << "           RequestPolicyName: rtline" << Endl;
    os << "           IndexingHost: saas-indexerproxy-maps-prestable.yandex.net" << Endl;
    os << "           IndexingPort: 80" << Endl;
    os << "           IndexingToken: 0e7d0426f506f52e7598c1b83cd6d17e" << Endl;
    os << "           IndexingPolicyName: rtline_indexing" << Endl;
    os << "       </surge>" << Endl;
    os << "       <drive_router>" << Endl;
    os << "           Host: saas-searchproxy-maps.yandex.net" << Endl;
    os << "           Port: 17000" << Endl;
    os << "           ServiceName: drive_router" << Endl;
    os << "           RequestPolicyName: rtline" << Endl;
    os << "           IndexingHost: saas-indexerproxy-maps.yandex.net" << Endl;
    os << "           IndexingPort: 80" << Endl;
    os << "           IndexingToken: 9a1311dbe58168aa1517cd484c211ee8" << Endl;
    os << "           IndexingPolicyName: rtline_indexing" << Endl;
    os << "       </drive_router>" << Endl;

    os << "       <drive_cache_prestable>" << Endl;
    os << "           Host: saas-searchproxy-maps-prestable.yandex.net" << Endl;
    os << "           Port: 17000" << Endl;
    os << "           ServiceName: drive_cache" << Endl;
    os << "           RequestPolicyName: rtline" << Endl;
    os << "           BalancerTimeoutTable: 10ms 30ms 50ms 70ms 90ms 110ms 130ms 150ms" << Endl;
    os << "           IndexingHost: saas-indexerproxy-maps-prestable.yandex.net" << Endl;
    os << "           IndexingPort: 80" << Endl;
    os << "           IndexingToken: ac6ac7f7a6544f336202c0b058104374" << Endl;
    os << "           IndexingPolicyName: rtline_indexing" << Endl;
    os << "       </drive_cache_prestable>" << Endl;

    os << "       <user_events_log_history>" << Endl;
    os << "           Host: saas-searchproxy-maps.yandex.net" << Endl;
    os << "           Port: 17000" << Endl;
    os << "           ServiceName: drive_search" << Endl;
    os << "           RequestPolicyName: rtline" << Endl;
    os << "       </user_events_log_history>" << Endl;

    os << "    </RTLineAPIs>" << Endl;
    os << "    <SensorApi>" << Endl;
    os << "        Name: " << GetSensorApiName() << Endl;
    os << "    </SensorApi>" << Endl;
    auto geobasePath = TFsPath(GetWorkPath()) / "geobase" / "geodata6.bin";
    if (geobasePath.Exists()) {
    os << "    <Geobase>" << Endl;
    os << "        Path: " << geobasePath << Endl;
    os << "    </Geobase>" << Endl;
    }
    os << "    <GeoFeatures>" << Endl;
    os << "        Name: drive_cache_prestable" << Endl;
    os << "    </GeoFeatures>" << Endl;
    os << "    <UserEventsApi>" << Endl;
    os << "        RTLineApiName: drive_cache_prestable" << Endl;
    os << "        ResponseTimeout: 300ms" << Endl;
    os << "        HistoryRTLineApiName: user_events_log_history" << Endl;
    os << "        HistoryResponseTimeout: 2000ms" << Endl;
    os << "    </UserEventsApi>" << Endl;
    os << "    <GlobalAsyncDelivery>" << Endl;
    os << "        Enabled: true" << Endl;
    os << "    </GlobalAsyncDelivery>" << Endl;
    os << "    <ModelsStorage>" << Endl;
    os << "        Name: models_storage" << Endl;
    os << "    </ModelsStorage>" << Endl;
    os << "    <ExternalDatabases>" << Endl;
    os << "        <test-drive-db>" << Endl;
    DriveAPIGenerator.DatabaseToString(os);
    os << "        </test-drive-db>" << Endl;
    if (YdbDatabase && YdbEndpoint) {
        os << "        <testing-drive-ydb>" << Endl;
        os << "             Type: YDB" << Endl;
        os << "             Endpoint: " << *YdbEndpoint << Endl;
        os << "             Database: " << *YdbDatabase << Endl;
        os << "        </testing-drive-ydb>" << Endl;
    }
    os << "        <unknown-drive-redis>" << Endl;
    os << "            Type: Redis" << Endl;
    os << "            Endpoint: " << GetEnv("REDIS_HOSTNAME", "localhost") << Endl;
    os << "            Port: " << GetEnv("REDIS_PORT", "26379") << Endl;
    os << "            Token: " << GetEnv("REDIS_TOKEN", "") << Endl;
    os << "        </unknown-drive-redis>" << Endl;
    os << "    </ExternalDatabases>" << Endl;
    os << "    <Localization>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "        Freshness: 1s" << Endl;
    os << "    </Localization>" << Endl;
    os << "    <UserDevicesManager>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "        Type: fake" << Endl;
    os << "        <Headers>" << Endl;
    os << "            IDFA: new_idfa" << Endl;
    os << "        </Headers>" << Endl;
    os << "        <EventsRegistrator>" << Endl;
    os << "            Type: fake" << Endl;
    os << "            <Events>" << Endl;
    os << "                success_registration: 1" << Endl;
    os << "                success_first_trip: 1" << Endl;
    os << "            </Events>" << Endl;
    os << "        </EventsRegistrator>" << Endl;
    os << "    </UserDevicesManager>" << Endl;
    os << "    <NotifiersManager>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "        <History>" << Endl;
    os << "            Deep: 0" << Endl;
    os << "        </History>" << Endl;
    os << "         <Propositions>";
    os << "             DefaultSelfConfirmationPolicy: Accept" << Endl;
    os << "         </Propositions>";
    os << "    </NotifiersManager>" << Endl;
    os << "    <ExternalAccessTokensManager>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "    </ExternalAccessTokensManager>" << Endl;
    os << "    <RTBackgroundManager>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "        PingPeriod: 1s" << Endl;
    os << "    </RTBackgroundManager>" << Endl;
    os << "    <Settings>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "        Prefix: development/" << Endl;
    os << "        Freshness: 1s" << Endl;
    os << "    </Settings>" << Endl;
    os << "    <Databases>" << Endl;
    os << "        <alerts_states>" << Endl;
    os << "            Type: LOCAL" << Endl;
    os << "            <LOCAL>" << Endl;
    os << "                Root: alerts_states" << Endl;
    os << "            </LOCAL>" << Endl;
    os << "        </alerts_states>" << Endl;
    os << "        <models_storage>" << Endl;
    os << "            Type: LOCAL" << Endl;
    os << "            <LOCAL>" << Endl;
    os << "                Root: models_storage" << Endl;
    os << "            </LOCAL>" << Endl;
    os << "        </models_storage>" << Endl;
    os << "        <chats_states>" << Endl;
    os << "            Type: LOCAL" << Endl;
    os << "            <LOCAL>" << Endl;
    os << "                Root: chats_states" << Endl;
    os << "            </LOCAL>" << Endl;
    os << "        </chats_states>" << Endl;
    os << "        <private_data_storage>" << Endl;
    os << "            Type: LOCAL" << Endl;
    os << "            <LOCAL>" << Endl;
    os << "                Root: private_data_storage" << Endl;
    os << "            </LOCAL>" << Endl;
    os << "        </private_data_storage>" << Endl;
    os << "    </Databases>" << Endl;
    os << "    <Ciphers>" << Endl;
    os << "        <test_aes>" << Endl;
    os << "            Type: aes" << Endl;
    os << "            Key: 47BCE5C74F589F4867DBD57E9CA9F80820EC512F5551A1A790C6A77479AF6033" << Endl;
    os << "        </test_aes>" << Endl;
    os << "    </Ciphers>" << Endl;
    os << "    <DocumentsManager>" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "        SessionYtDataPath: //home" << Endl;
    os << "        CompiledRidesYtDataPath: //home" << Endl;
    os << "        PaymentsYtDataPath: //home" << Endl;
    os << "        TracksService: drive_graph" << Endl;
    os << "    </DocumentsManager>" << Endl;
    os << "    <DriveAPI>" << Endl;
    DriveAPIGenerator.ToString(os, IsYtEnabled);
    os << "    </DriveAPI>" << Endl;
    os << "    <SelfHttpRequester>" << Endl;
    os << "        Environments: test_local" << Endl;
    os << "        DelegatorAuthHeader: 8b33b36b-ca2a-4f16-9af4-dc1598f02ec4" << Endl;
    os << "        Https: 0" << Endl;
    os << "        RequestTimeout: 60s" << Endl;
    os << "        <RequestConfig>" << Endl;
    os << "            MaxAttempts: 1" << Endl;
    os << "        </RequestConfig>" << Endl;
    os << "    </SelfHttpRequester>" << Endl;
    os << "    <AccountEmailBinder>" << Endl;
    os << "        Consumer: 123" << Endl;
    os << "    </AccountEmailBinder>" << Endl;

    os << "<SupportCenterManager>" << Endl;
    os << "    DBName: test-drive-db" << Endl;
    os << "    <Categorizer>" << Endl;
    os << "        <HistoryConfig>" << Endl;
    os << "            Deep: 0" << Endl;
    os << "        </HistoryConfig>" << Endl;
    os << "    </Categorizer>" << Endl;
    os << "</SupportCenterManager>" << Endl;

    os << "<TaxiChatClient>" << Endl;
    os << "    ClientType: test" << Endl;
    os << "</TaxiChatClient>" << Endl;

    os << "<TaxiSignalqDrivematicsApiClient>" << Endl;
    os << "    ClientType: fake" << Endl;
    os << "</TaxiSignalqDrivematicsApiClient>" << Endl;

    os << "<UserDocumentsChecks>" << Endl;
    os << "    DBName: test-drive-db" << Endl;
    os << "</UserDocumentsChecks>" << Endl;

    os << "<ChatRobots>" << Endl;
    os << "    StateStorageDatabase: test-drive-db" << Endl;
    os << "    MediaMdsBucket: carsharing-chat" << Endl;
    os << "    <Robots>" << Endl;
    os << "        <registration>" << Endl;
    os << "            ChatId: registration" << Endl;
    os << "            Title: Регистрация" << Endl;
    os << "            BotClassName: TRegistrationChatBot" << Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_correct.json") << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </registration>" << Endl;
    os << "        <expiring_license_notify>" << Endl;
    os << "            ChatId: expiring_license_notify" << Endl;
    os << "            BotClassName: TRegistrationChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_correct.json") << Endl;
    os << "            Title: Скоро истекают права" << Endl;
    os << "            StartFromCleanChat: true" << Endl;
    os << "        </expiring_license_notify>"<< Endl;
    os << "        <expiring_passport_notify>" << Endl;
    os << "            ChatId: expiring_passport_notify" << Endl;
    os << "            BotClassName: TRegistrationChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_correct.json") << Endl;
    os << "            Title: Скоро истекает паспорт" << Endl;
    os << "            StartFromCleanChat: true" << Endl;
    os << "        </expiring_passport_notify>"<< Endl;
    os << "        <high_speed_ban>" << Endl;
    os << "            ChatId: high_speed_ban" << Endl;
    os << "            BotClassName: TRegistrationChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_correct.json") << Endl;
    os << "            Title: Бан чат" << Endl;
    os << "        </high_speed_ban>"<< Endl;
    os << "        <promo>" << Endl;
    os << "            ChatId: promo" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_promo.json") << Endl;
    os << "            Title: Акция" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </promo>"<< Endl;
    os << "        <referral>" << Endl;
    os << "            ChatId: referral" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_referral.json") << Endl;
    os << "            Title: Реферальная программа" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </referral>"<< Endl;
    os << "        <support>" << Endl;
    os << "            ChatId: support" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            RobotName: rbt" << Endl;
    os << "            RobotUserId: robot-frontend" << Endl;
    os << "        </support>"<< Endl;
    os << "        <support_corp_client>" << Endl;
    os << "            ChatId: support_corp_client" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_corp_client.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_corp_client>"<< Endl;
    os << "        <outgoing_communication>" << Endl;
    os << "            ChatId: outgoing_communication" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </outgoing_communication>"<< Endl;
    os << "        <support_modern>" << Endl;
    os << "            ChatId: support_modern" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support_modern.json") << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_modern>"<< Endl;
    os << "        <support_modern_v2>" << Endl;
    os << "            ChatId: support_modern_v2" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_modern_v2.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_modern_v2>"<< Endl;
    os << "        <support_suggest>" << Endl;
    os << "            ChatId: support_suggest" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support_suggest.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_suggest>"<< Endl;
    os << "        <support_context_map>" << Endl;
    os << "            ChatId: support_context_map" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_context_map.json") << Endl;
    os << "            Title: Conext maps" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_context_map>"<< Endl;
    os << "        <support_treelike>" << Endl;
    os << "            ChatId: support_treelike" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support_treelike.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_treelike>"<< Endl;
    os << "        <support_documents_checks>" << Endl;
    os << "            ChatId: support_documents_checks" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_documents_checks.json") << Endl;
    os << "            Title: Documents checks" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_documents_checks>"<< Endl;
    os << "        <support_math_checks>" << Endl;
    os << "            ChatId: support_math_checks" << Endl;
    os << "            PersistVisitedStates: true" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_math.json") << Endl;
    os << "            Title: Math checks" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "        </support_math_checks>"<< Endl;
    os << "        <support_no_create_on_read>" << Endl;
    os << "            ChatId: support_no_create_on_read" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            CreateOnRead: false" << Endl;
    os << "        </support_no_create_on_read>"<< Endl;
    os << "        <support_intro>" << Endl;
    os << "            ChatId: support_intro" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support_intro.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            IsStatic: true" << Endl;
    os << "        </support_intro>"<< Endl;
    os << "        <support_intro_new>" << Endl;
    os << "            ChatId: support_intro_new" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_support_modern.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            IsStatic: true" << Endl;
    os << "        </support_intro_new>"<< Endl;
    os << "        <support_condition>" << Endl;
    os << "            ChatId: support_condition" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_condition.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            IsStatic: true" << Endl;
    os << "        </support_condition>"<< Endl;
    os << "        <static_announcements>" << Endl;
    os << "            ChatId: static_announcements" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_announcements.json") << Endl;
    os << "            Title: Доска объявлений" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            IsStatic: true" << Endl;
    os << "        </static_announcements>"<< Endl;
    os << "        <support_variables>" << Endl;
    os << "            ChatId: support_variables" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_variables.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            IsStatic: false" << Endl;
    os << "        </support_variables>"<< Endl;
    os << "        <support_switch>" << Endl;
    os << "            ChatId: support_switch" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_switch.json") << Endl;
    os << "            Title: Поддержка" << Endl;
    os << "            StartFromCleanChat: false" << Endl;
    os << "            IsStatic: false" << Endl;
    os << "        </support_switch>"<< Endl;
    os << "        <test_introscreen>" << Endl;
    os << "            ChatId: test_introscreen" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_test_introscreen.json") << Endl;
    os << "            Title: Сурдж-зоны" << Endl;
    os << "        </test_introscreen>"<< Endl;
    os << "        <channel>" << Endl;
    os << "            ChatId: channel" << Endl;
    os << "            BotClassName: TChannelChatBot"<< Endl;
    os << "            StartFromCleanChat: true" << Endl;
    os << "        </channel>"<< Endl;
    os << "        <device_verify>" << Endl;
    os << "            ChatId: device_verify" << Endl;
    os << "            BotClassName: TSupportChatBot"<< Endl;
    os << "            ChatScriptFilename: " << JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_device_verify.json") << Endl;
    os << "            Title: Это точно вы?" << Endl;
    os << "        </device_verify>"<< Endl;
    os << "        <introscreens>" << Endl;
    os << "            ChatId: introscreens" << Endl;
    os << "            BotClassName: TChannelChatBot"<< Endl;
    os << "            StartFromCleanChat: true" << Endl;
    os << "        </introscreens>"<< Endl;
    os << "    </Robots>" << Endl;
    os << "</ChatRobots>" << Endl;
    os << "<UserRegistrationManager>" << Endl;
    os << "    ChatId: registration" << Endl;
    os << "    BanChatId: registration" << Endl;
    os << "    IsAccountCheckEnabled: true" << Endl;
    os << "    SensitiveDataHashesKey: 90d8eb23f8eb14ebd95c49edc81dfcc8" << Endl;  // prod key is different
    os << "    <AccountChecker>" << Endl;
    os << "        Host: blackbox.yandex.net" << Endl;
    os << "        Route: /blackbox" << Endl;
    os << "        MinAccountAge: 365d" << Endl;
    os << "        NoScoringCountries: RUS,UKR,BLR" << Endl;
    os << "        HighPrioritySuids: 15,19,24,47,48,52,53,64,91,96,110,117" << Endl;
    os << "        LowPrioritySuids: 31,54,85,97,98,114,116" << Endl;
    os << "    </AccountChecker>" << Endl;
    os << "</UserRegistrationManager>" << Endl;
    os << "    <DistributingBlockEventsStorage>" << Endl;
    os << "        Type: db" << Endl;
    os << "        DBName: test-drive-db" << Endl;
    os << "    </DistributingBlockEventsStorage>" << Endl;

    os << "    <Processors>" << Endl;
    GetApiConfigString(os);
    os << "    </Processors>" << Endl;

    os << "    <BackgroundProcesses>" << Endl;
    GetBackgroundConfigString(os);
    os << "    </BackgroundProcesses>" << Endl;

    os << "    <Notifications>" << Endl;
    os << "        <test_start>" << Endl;
    os << "            NotificationType: logs" << Endl;
    os << "            LogPriority: DEBUG" << Endl;
    os << "        </test_start>" << Endl;
    os << "        <internal_notifier>" << Endl;
    os << "            NotificationType: internal" << Endl;
    os << "        </internal_notifier>" << Endl;
    os << "    </Notifications>" << Endl;
    os << "    <TelematicsClient>" << Endl;
    os << "        Shards: localhost:321 localhost:" << TTelematicServerBuilder::GetHttpPort() << Endl;
    os << "        FetchHost: saas-searchproxy-maps-prestable.yandex.net" << Endl;
    os << "        FetchService: drive_cache" << Endl;
    os << "        PushHost: saas-indexerproxy-maps-prestable.yandex.net" << Endl;
    os << "        PushToken: ac6ac7f7a6544f336202c0b058104374" << Endl;
    os << "    </TelematicsClient>" << Endl;
    os << "    <YangClient>" << Endl;
    os << "        Uri: https://yang.yandex-team.ru" << Endl;
    os << "        AuthType: OAuth" << Endl;
    os << "        AuthToken: o" << Endl;
    os << "        PathPrefix: /api/v1/" << Endl;
    os << "    </YangClient>" << Endl;
    os << "    YangClientType: fake" << Endl;
    os << "</Server>" << Endl;
}

void NDrive::TServerConfigGenerator::GetApiConfigString(IOutputStream& os) const {
    TSet<TString> APIs;
    NDrive::NTest::TAPIAction::TFactory::GetRegisteredKeys(APIs);
    for (auto&& key : APIs) {
        TAtomicSharedPtr<NDrive::NTest::TAPIAction> action = NDrive::NTest::TAPIAction::TFactory::Construct(key);
        UNIT_ASSERT(action);
        os << action->GetProcessorConfiguration();
    }
    os << "        ProcessorDefaultTimeout: 60s" << Endl;
    os << "        <default_config:default>" << Endl;
    os << "            DumpEventLog: true" << Endl;
    os << "        </default_config:default>" << Endl;
    os << "        <api/staff/constants>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: constants/list" << Endl;
    os << "        </api/staff/constants>" << Endl;

    os << "        <api/staff/user/rootify>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_rootify" << Endl;
    os << "        </api/staff/user/rootify>" << Endl;

    os << "        <api/staff/user_tags/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_tags/add" << Endl;
    os << "        </api/staff/user_tags/add>" << Endl;
    os << "        <api/staff/user_tags/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_tags/remove" << Endl;
    os << "        </api/staff/user_tags/remove>" << Endl;
    os << "        <api/staff/user_tags/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_tags/list" << Endl;
    os << "        </api/staff/user_tags/list>" << Endl;
    os << "        <api/yandex/user/tag/update>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_tags/update" << Endl;
    os << "        </api/yandex/user/tag/update>" << Endl;
    os << "        <api/staff/users/tags_history>" << Endl;
    os << "            ProcessorType: user_tags_history" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/users/tags_history>" << Endl;
    os << "        <api/staff/users/info>" << Endl;
    os << "            ProcessorType: user_admin_portal_info" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/users/info>" << Endl;
    os << "        <api/staff/users/edit>" << Endl;
    os << "            ProcessorType: user_edit" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/users/edit>" << Endl;
    os << "        <api/staff/users/history>" << Endl;
    os << "            ProcessorType: user_modification_history" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/users/history>" << Endl;

    os << "        <api/staff/car_tags/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_tags/add" << Endl;
    os << "        </api/staff/car_tags/add>" << Endl;
    os << "        <api/staff/car_tags/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_tags/remove" << Endl;
    os << "        </api/staff/car_tags/remove>" << Endl;
    os << "        <api/staff/car_tags/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_tags/list" << Endl;
    os << "        </api/staff/car_tags/list>" << Endl;

    os << "        <api/staff/account_tags/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: account_tags/add" << Endl;
    os << "        </api/staff/account_tags/add>" << Endl;
    os << "        <api/staff/account_tags/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: account_tags/remove" << Endl;
    os << "        </api/staff/account_tags/remove>" << Endl;
    os << "        <api/staff/account_tags/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: account_tags/list" << Endl;
    os << "        </api/staff/account_tags/list>" << Endl;

    os << "        <api/staff/trace_tags/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: trace_tags/list" << Endl;
    os << "        </api/staff/trace_tags/list>" << Endl;

    os << "        <api/signalq/notify_support>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: signalq_notify_support" << Endl;
    os << "        </api/signalq/notify_support>" << Endl;

    os << "        <api/signalq/trace/tag/resolution/set>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: signalq_fleet_set_resolution" << Endl;
    os << "        </api/signalq/trace/tag/resolution/set>" << Endl;

    os << "        <api/staff/localization/upsert>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: localization_upsert" << Endl;
    os << "        </api/staff/localization/upsert>" << Endl;
    os << "        <api/staff/localization/info>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: localization_info" << Endl;
    os << "        </api/staff/localization/info>" << Endl;
    os << "        <api/staff/localization/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: localization_remove" << Endl;
    os << "        </api/staff/localization/remove>" << Endl;

    os << "        <api/yandex/tag/evolve>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: tag_evolve" << Endl;
    os << "            OverrideCgiPart: evolution_mode=default" << Endl;
    os << "        </api/yandex/tag/evolve>" << Endl;
    os << "        <api/yandex/tag/evolve_adm>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: tag_evolve" << Endl;
    os << "        </api/yandex/tag/evolve_adm>" << Endl;
    os << "        <api/yandex/offers/book>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: book" << Endl;
    os << "        </api/yandex/offers/book>" << Endl;
    os << "        <api/yandex/rental/book>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: rental/book" << Endl;
    os << "        </api/yandex/rental/book>" << Endl;
    os << "        <api/yandex/rental/start>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: rental/start" << Endl;
    os << "        </api/yandex/rental/start>" << Endl;
    os << "        <api/yandex/organization/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: organization/list" << Endl;
    os << "        </api/yandex/organization/list>" << Endl;
    os << "        <api/yandex/car/tag/details>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_tags/details" << Endl;
    os << "        </api/yandex/car/tag/details>" << Endl;
    os << "        <api/yandex/warnings/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: warning_screen" << Endl;
    os << "        </api/yandex/warnings/get>" << Endl;
    os << "        <api/yandex/sessions/history>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: client_history" << Endl;
    os << "            RequestTimeout: 10s" << Endl;
    os << "        </api/yandex/sessions/history>" << Endl;
    os << "        <api/yandex/user_tags/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_tags/remove" << Endl;
    os << "        </api/yandex/user_tags/remove>" << Endl;
    os << "        <api/yandex/offers/replace_car_prepare>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: replace_car_prepare" << Endl;
    os << "        </api/yandex/offers/replace_car_prepare>" << Endl;
    os << "        <api/yandex/offers/replace_car>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: replace_car" << Endl;
    os << "        </api/yandex/offers/replace_car>" << Endl;
    os << "        <api/yandex/offers/fixpoint>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: external_offer" << Endl;
    os << "            UseCustomOfferReport: false" << Endl;
    os << "        </api/yandex/offers/fixpoint>" << Endl;
    os << "        <api/yandex/offers/restore>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: restore_offer" << Endl;
    os << "        </api/yandex/offers/restore>" << Endl;
    os << "        <api/yandex/offers/timetable>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: offers/timetable" << Endl;
    os << "        </api/yandex/offers/timetable>" << Endl;
    os << "        <api/yandex/rental/timetable>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: offers/timetable" << Endl;
    os << "            IsRentalOffers: true" << Endl;
    os << "        </api/yandex/rental/timetable>" << Endl;
    os << "        <api/yandex/offers/update>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: offers/update" << Endl;
    os << "        </api/yandex/offers/update>" << Endl;
    os << "        <api/yandex/sessions/agreement>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: session_agreement" << Endl;
    os << "        </api/yandex/sessions/agreement>" << Endl;
    os << "        <api/yandex/car/tag/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_tags/add" << Endl;
    os << "        </api/yandex/car/tag/add>" << Endl;
    os << "        <api/leasing/photo/markup/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: get_marked_photos" << Endl;
    os << "        </api/leasing/photo/markup/get>" << Endl;
    os << "        <api/maps/offers/fixpoint>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: external_offer" << Endl;
    os << "        </api/maps/offers/fixpoint>" << Endl;
    os << "        <api/taxi/offers/standard>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: external_offer" << Endl;
    os << "        </api/taxi/offers/standard>" << Endl;
    os << "        <api/yandex/card>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: set_card" << Endl;
    os << "        </api/yandex/card>" << Endl;
    os << "        <api/staff/areas/info>" << Endl;
    os << "            ProcessorType: areas_info" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/areas/info>" << Endl;
    os << "        <api/staff/areas/remove>" << Endl;
    os << "            ProcessorType: areas_remove" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/areas/remove>" << Endl;
    os << "        <api/staff/areas/upsert>" << Endl;
    os << "            ProcessorType: areas_upsert" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/areas/upsert>" << Endl;
    os << "        <api/staff/area_tags/add>" << Endl;
    os << "            ProcessorType: area_tags/add" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/area_tags/add>" << Endl;
    os << "        <api/staff/area_tags/remove>" << Endl;
    os << "            ProcessorType: area_tags/remove" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/area_tags/remove>" << Endl;
    os << "        <api/staff/area_tags/history>" << Endl;
    os << "            ProcessorType: area_tags_history" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/area_tags/history>" << Endl;

    os << "        <b2b/organization/create>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: create_account" << Endl;
    os << "            IgnoreDeviceId: true" << Endl;
    os << "            OverrideCgiPart: name=b2b_drive&create_access_tag=organization_creator&balance=true" << Endl;
    os << "        </b2b/organization/create>" << Endl;
    os << "        <b2b/organization/update>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: update_account_data" << Endl;
    os << "        </b2b/organization/update>" << Endl;
    os << "        <b2b/accounts/activate>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: activate_account" << Endl;
    os << "            MaxItems: 1000" << Endl;
    os << "        </b2b/accounts/activate>" << Endl;
    os << "        <b2b/wallets/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: get_common_accounts" << Endl;
    os << "        </b2b/wallets/get>" << Endl;
    os << "        <b2b/accounts/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: list_accounts" << Endl;
    os << "        </b2b/accounts/get>" << Endl;
    os << "        <api/staff/billing/accounts/update>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: update_common_account" << Endl;
    os << "        </api/staff/billing/accounts/update>" << Endl;
    os << "        <b2b/sessions/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: account_session_list" << Endl;
    os << "        </b2b/sessions/list>" << Endl;
    os << "        <api/staff/compiled_bill/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: add_compiled_bill" << Endl;
    os << "        </api/staff/compiled_bill/add>" << Endl;

    os << "        <service_tags/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </service_tags/add>" << Endl;
    os << "        <service_tags/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </service_tags/list>" << Endl;
    os << "        <service_tags/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </service_tags/remove>" << Endl;

    os << "        <car/tags/propositions/propose>" << Endl;
    os << "            ProcessorType: tags/propositions/propose" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </car/tags/propositions/propose>" << Endl;
    os << "        <car/tags/propositions/confirm>" << Endl;
    os << "            ProcessorType: tags/propositions/confirm" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </car/tags/propositions/confirm>" << Endl;
    os << "        <car/tags/propositions/reject>" << Endl;
    os << "            ProcessorType: tags/propositions/reject" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </car/tags/propositions/reject>" << Endl;

    os << "        <api/staff/tag/description/list>" << Endl;
    os << "            ProcessorType: service_tags/list" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/tag/description/list>" << Endl;

    os << "        <api/staff/tag/description/propose>" << Endl;
    os << "            ProcessorType: tags_description_propose" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/tag/description/propose>" << Endl;
    os << "        <api/staff/tag/description/confirm>" << Endl;
    os << "            ProcessorType: tags_description_confirm" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/tag/description/confirm>" << Endl;
    os << "        <api/staff/tag/description/reject>" << Endl;
    os << "            ProcessorType: tags_description_reject" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/tag/description/reject>" << Endl;

    os << "        <service_app/servicing/start>" << Endl;
    os << "            ProcessorType: service_app_start" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </service_app/servicing/start>" << Endl;

    os << "        <api/yandex/car/actualization>" << Endl;
    os << "            ProcessorType: tags_actualization" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/yandex/car/actualization>" << Endl;

    os << "        <api/yandex/landing/accept>" << Endl;
    os << "            ProcessorType: landing_accept" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/yandex/landing/accept>" << Endl;
    os << "        <api/yandex/landing/info>" << Endl;
    os << "            ProcessorType: landing_info" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/yandex/landing/info>" << Endl;

    os << "        <service_app/search>" << Endl;
    os << "            ProcessorType: service_app_search" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </service_app/search>" << Endl;
    os << "        <service_app/servicing/finish>" << Endl;
    os << "            ProcessorType: service_app_finish" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </service_app/servicing/finish>" << Endl;

    os << "        <api/staff/car/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: service_app_list" << Endl;
    os << "            OverrideCgiPart: report=admin_app" << Endl;
    os << "            RateLimit: 100" << Endl;
    os << "        </api/staff/car/list>" << Endl;

    os << "        <crm/organization/application>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: create_lead" << Endl;
    os << "            OverrideCgiPart: tag_name=from_crm" << Endl;
    os << "        </crm/organization/application>" << Endl;

    os << "        <b2b/sessions/dedicated_fleet>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: dedicated_fleet_sessions" << Endl;
    os << "        </b2b/sessions/dedicated_fleet>" << Endl;
    os << "        <b2b/dedicated_fleet/session/drop>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: dedicated_fleet_drop_session" << Endl;
    os << "        </b2b/dedicated_fleet/session/drop>" << Endl;

    os << "        <api/staff/car/control>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_root_control" << Endl;
    os << "        </api/staff/car/control>" << Endl;
    os << "        <api/staff/car/info>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_admin_portal_info" << Endl;
    os << "        </api/staff/car/info>" << Endl;
    os << "        <api/staff/telematics/info>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: telematics_car_state" << Endl;
    os << "        </api/staff/telematics/info>" << Endl;
    os << "        <service_app/car/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: service_app_list" << Endl;
    os << "            OverrideCgiPart: report=service_app" << Endl;
    os << "            RateLimit: 100" << Endl;
    os << "        </service_app/car/list>" << Endl;
    os << "        <admin_app/car/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: service_app_list" << Endl;
    os << "            OverrideCgiPart: report=admin_app" << Endl;
    os << "            RateLimit: 100" << Endl;
    os << "        </admin_app/car/list>" << Endl;
    os << "        <service_app/cached/car/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: service_app_list" << Endl;
    os << "            OverrideCgiPart: report=service_app" << Endl;
    os << "            RateLimit: 100" << Endl;
    os << "        </service_app/cached/car/list>" << Endl;
    os << "        <service_app/cached/car/list/reserved>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: service_app_list" << Endl;
    os << "            OverrideCgiPart: report=service_app" << Endl;
    os << "            RateLimit: 100" << Endl;
    os << "        </service_app/cached/car/list/reserved>" << Endl;
    os << "        <service_app/car/details>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: service_app_list" << Endl;
    os << "            OverrideCgiPart: report=service_app_details" << Endl;
    os << "        </service_app/car/details>" << Endl;
    os << "        <user_app/car/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: service_app_list" << Endl;
    os << "            OverrideCgiPart: report=user_app" << Endl;
    os << "            RateLimit: 100" << Endl;
    os << "        </user_app/car/list>" << Endl;
    os << "        <api/yandex/user_sessions>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: sessions_history" << Endl;
    os << "        </api/yandex/user_sessions>" << Endl;
    os << "        <api/yandex/sessions/current>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: current_session" << Endl;
    os << "        </api/yandex/sessions/current>" << Endl;
    os << "        <api/yandex/sessions/drop>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: drop_session" << Endl;
    os << "        </api/yandex/sessions/drop>" << Endl;
    os << "        <api/yandex/sessions/sharing/book>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: shared_session_book" << Endl;
    os << "        </api/yandex/sessions/sharing/book>" << Endl;
    os << "        <api/yandex/sessions/sharing/invite>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: shared_session_invite" << Endl;
    os << "        </api/yandex/sessions/sharing/invite>" << Endl;
    os << "        <api/yandex/sessions/state/drop>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: entity_tags/remove" << Endl;
    os << "            OverrideCgiPart: entities=car,user&force=true" << Endl;
    os << "        </api/yandex/sessions/state/drop>" << Endl;

    os << "    <api/staff/promo/get>" << Endl;
    os << "        ProcessorType: promo/get" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </api/staff/promo/get>" << Endl;
    os << "    <api/staff/promo/generation>" << Endl;
    os << "        ProcessorType: promo/generation" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </api/staff/promo/generation>" << Endl;
    os << "    <api/staff/promo/remove>" << Endl;
    os << "        ProcessorType: promo/remove" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </api/staff/promo/remove>" << Endl;
    os << "    <api/staff/promo/check>" << Endl;
    os << "        ProcessorType: promo/check" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </api/staff/promo/check>" << Endl;

    os << "    <api/staff/promo/give_out>" << Endl;
    os << "        ProcessorType: promo/give_out" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </api/staff/promo/give_out>" << Endl;
    os << "    <user_app/promo/accept>" << Endl;
    os << "        ProcessorType: promo/accept" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </user_app/promo/accept>" << Endl;
    os << "    <api/staff/promo/user_history>" << Endl;
    os << "        ProcessorType: promo/user_history" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </api/staff/promo/user_history>" << Endl;

    os << "    <user_app/promo/create_referral_code>" << Endl;
    os << "        ProcessorType: promo/create_referral_code" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </user_app/promo/create_referral_code>" << Endl;

    os << "    <user_app/promo/referral>" << Endl;
    os << "        ProcessorType: promo/referral" << Endl;
    os << "        AuthModuleName: fake" << Endl;
    os << "    </user_app/promo/referral>" << Endl;

    os << "        <user_app/fueling/start>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: fueling_start" << Endl;
    os << "        </user_app/fueling/start>" << Endl;

    os << "        <user_app/fueling/map>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: fueling_map" << Endl;
    os << "        </user_app/fueling/map>" << Endl;

    os << "        <user_app/fueling/info>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: fueling_info" << Endl;
    os << "        </user_app/fueling/info>" << Endl;

    os << "        <user_app/fueling/cancel>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: fueling_cancel" << Endl;
    os << "        </user_app/fueling/cancel>" << Endl;

    os << "        <api/staff/support/fueling/custom/start>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_create_fueling_order" << Endl;
    os << "        </api/staff/support/fueling/custom/start>" << Endl;

    os << "        <user_app/sessions/history>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: client_history" << Endl;
    os << "            OverrideCgiPart: numdoc=10" << Endl;
    os << "        </user_app/sessions/history>" << Endl;

    os << "        <user_app/scanner/start>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: start_scanner" << Endl;
    os << "        </user_app/scanner/start>" << Endl;
    os << "        <user_app/scanner/stop>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: stop_scanner" << Endl;
    os << "        </user_app/scanner/stop>" << Endl;

    os << "        <api/service/photos/register>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: register_photo" << Endl;
    os << "        </api/service/photos/register>" << Endl;

    os << "        <api/service/photos/upload>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: upload_service_photo" << Endl;
    os << "        </api/service/photos/upload>" << Endl;

    os << "        <api/staff/landing/clear>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_landings_clear" << Endl;
    os << "        </api/staff/landing/clear>" << Endl;

    os << "        <api/staff/landing/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: landings_get" << Endl;
    os << "        </api/staff/landing/get>" << Endl;

    os << "        <api/staff/landing/upsert>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: landing_upsert" << Endl;
    os << "        </api/staff/landing/upsert>" << Endl;

    os << "        <api/staff/landing/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: landing_remove" << Endl;
    os << "        </api/staff/landing/remove>" << Endl;

    os << "        <api/staff/landing/propose>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: drive_landings_propose" << Endl;
    os << "        </api/staff/landing/propose>" << Endl;

    os << "        <api/staff/landing/confirm>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: drive_landings_confirm" << Endl;
    os << "        </api/staff/landing/confirm>" << Endl;

    os << "        <api/staff/landing/reject>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: drive_landings_reject" << Endl;
    os << "        </api/staff/landing/reject>" << Endl;

    os << "        <api/yandex/models/create>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_model/create" << Endl;
    os << "        </api/yandex/models/create>" << Endl;

    os << "        <api/yandex/models/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_model/list" << Endl;
    os << "        </api/yandex/models/list>" << Endl;

    os << "        <api/yandex/models/update>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_model/update" << Endl;
    os << "        </api/yandex/models/update>" << Endl;

    os << "        <api/yandex/models/delete>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_model/delete" << Endl;
    os << "        </api/yandex/models/delete>" << Endl;

    os << "        <api/yandex/tariffs/filters/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: tariffs/filters/list" << Endl;
    os << "        </api/yandex/tariffs/filters/list>" << Endl;

    os << "        <api/yandex/tariffs/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: tariffs/list" << Endl;
    os << "        </api/yandex/tariffs/list>" << Endl;

    os << "        <api/staff/model/info>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_models_info" << Endl;
    os << "        </api/staff/model/info>" << Endl;

    os << "        <api/staff/model/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_models_remove" << Endl;
    os << "        </api/staff/model/remove>" << Endl;

    os << "        <api/staff/model_spec/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_model_specification_add" << Endl;
    os << "        </api/staff/model_spec/add>" << Endl;

    os << "        <api/staff/model_spec/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_model_specification_remove" << Endl;
    os << "        </api/staff/model_spec/remove>" << Endl;

    os << "        <api/yandex/car/edit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_upsert" << Endl;
    os << "        </api/yandex/car/edit>" << Endl;

    os << "        <api/staff/car/history>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_modification_history" << Endl;
    os << "        </api/staff/car/history>" << Endl;

    os << "        <api/staff/major/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: current_major_queries" << Endl;
    os << "        </api/staff/major/get>" << Endl;

    os << "        <api/staff/major/cancel>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: repair_tag_cancel" << Endl;
    os << "        </api/staff/major/cancel>" << Endl;

    os << "        <api/staff/actions/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: manage_action/add" << Endl;
    os << "        </api/staff/actions/add>" << Endl;

    os << "        <api/staff/actions/confirm>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: manage_action/confirm" << Endl;
    os << "        </api/staff/actions/confirm>" << Endl;

    os << "        <api/staff/actions/propose>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: manage_action/propose" << Endl;
    os << "        </api/staff/actions/propose>" << Endl;

    os << "        <api/staff/actions/reject>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: manage_action/reject" << Endl;
    os << "        </api/staff/actions/reject>" << Endl;

    os << "        <api/staff/actions/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: manage_action/list" << Endl;
    os << "        </api/staff/actions/list>" << Endl;

    os << "        <api/staff/actions/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: manage_action/remove" << Endl;
    os << "        </api/staff/actions/remove>" << Endl;

    os << "        <api/staff/user_roles/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_roles/list" << Endl;
    os << "        </api/staff/user_roles/list>" << Endl;

    os << "        <api/staff/user_roles/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_roles/add" << Endl;
    os << "        </api/staff/user_roles/add>" << Endl;

    os << "        <api/staff/roles/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: manage_roles/add" << Endl;
    os << "        </api/staff/roles/add>" << Endl;

    os << "        <api/staff/my/roles/activation>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_role_activation" << Endl;
    os << "        </api/staff/my/roles/activation>" << Endl;

    os << "        <service_app/attachments/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            OverrideCgiPart: report=service_app" << Endl;
    os << "            ProcessorType: car_attached_hardware" << Endl;
    os << "        </service_app/attachments/list>" << Endl;

    os << "        <api/staff/car/attachments/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            OverrideCgiPart: report=all" << Endl;
    os << "            ProcessorType: car_attached_hardware" << Endl;
    os << "        </api/staff/car/attachments/list>" << Endl;

    os << "        <service_app/attachments/assign>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_assign_attachment" << Endl;
    os << "        </service_app/attachments/assign>" << Endl;

    os << "        <service_app/attachments/history>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_assignment_history" << Endl;
    os << "        </service_app/attachments/history>" << Endl;

    os << "        <api/staff/car/attachments/unassign>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_unassign_attachment" << Endl;
    os << "        </api/staff/car/attachments/unassign>" << Endl;

    os << "        <api/staff/car/batch-upload>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_registry_update" << Endl;
    os << "        </api/staff/car/batch-upload>" << Endl;

    os << "        <api/staff/car/insurance/batch-upload>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: car_insurance_upload" << Endl;
    os << "        </api/staff/car/insurance/batch-upload>" << Endl;

    os << "        <api/staff/notifiers/upsert>" << Endl;
    os << "            ProcessorType: notifiers_upsert" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/notifiers/upsert>" << Endl;
    os << "        <api/staff/notifiers/propose>" << Endl;
    os << "            ProcessorType: notifiers_propose" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/notifiers/propose>" << Endl;
    os << "        <api/staff/notifiers/propositions>" << Endl;
    os << "            ProcessorType: notifiers_propositions" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/notifiers/propositions>" << Endl;
    os << "        <api/staff/notifiers/confirm>" << Endl;
    os << "            ProcessorType: notifiers_confirm" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/notifiers/confirm>" << Endl;
    os << "        <api/staff/notifiers/info>" << Endl;
    os << "            ProcessorType: notifiers_info" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/notifiers/info>" << Endl;

    os << "        <api/staff/bg/upsert>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: rt_background_upsert" << Endl;
    os << "        </api/staff/bg/upsert>" << Endl;
    os << "        <api/staff/bg/info>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: rt_background_info" << Endl;
    os << "        </api/staff/bg/info>" << Endl;
    os << "        <api/staff/bg/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: rt_background_remove" << Endl;
    os << "        </api/staff/bg/remove>" << Endl;

    os << "        <api/yandex/chat/history>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_history" << Endl;
    os << "        </api/yandex/chat/history>" << Endl;
    os << "        <api/yandex/chat/action>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_action" << Endl;
    os << "        </api/yandex/chat/action>" << Endl;

    os << "        <api/yandex/chat/get_or_create>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_get_or_create" << Endl;
    os << "        </api/yandex/chat/get_or_create>" << Endl;

    os << "         <api/yandex/chat/get_suggest>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: robot_chat_get_suggest" << Endl;
    os << "         </api/yandex/chat/get_suggest>" << Endl;

    os << "        <api/staff/chat/message>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_action" << Endl;
    os << "        </api/staff/chat/message>" << Endl;

    os << "        <api/takeout/request>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: takeout_request" << Endl;
    os << "        </api/takeout/request>" << Endl;

    os << "        <user_app/verification/new_device/submit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: verification/new_device/submit" << Endl;
    os << "        </user_app/verification/new_device/submit>" << Endl;

    os << "        <user_app/verification/new_device/commit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: verification/new_device/commit" << Endl;
    os << "        </user_app/verification/new_device/commit>" << Endl;

    os << "        <api/yandex/chats/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_list" << Endl;
    os << "        </api/yandex/chats/list>" << Endl;

    os << "        <api/yandex/chats/reset>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_reset" << Endl;
    os << "        </api/yandex/chats/reset>" << Endl;

    os << "        <api/yandex/chats/read>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_read" << Endl;
    os << "        </api/yandex/chats/read>" << Endl;

    os << "        <api/yandex/chats/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_remove" << Endl;
    os << "        </api/yandex/chats/remove>" << Endl;

    os << "        <api/yandex/user/document-photo>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_resource" << Endl;
    os << "        </api/yandex/user/document-photo>" << Endl;

    os << "        <api/yandex/user/email>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_email_bind" << Endl;
    os << "        </api/yandex/user/email>" << Endl;

    os << "        <api/staff/tags/search>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_tag_search" << Endl;
    os << "        </api/staff/tags/search>" << Endl;

    os << "        <api/staff/user/register-force>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_force_register" << Endl;
    os << "        </api/staff/user/register-force>" << Endl;
    os << "        <api/staff/user/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_force_delete" << Endl;
    os << "        </api/staff/user/remove>" << Endl;

    os << "        <api/staff/blacklist-external/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_blacklist_add" << Endl;
    os << "        </api/staff/blacklist-external/add>" << Endl;

    os << "        <support_api/yang/assignment/data>" << Endl;
    os << "            AuthModuleName: yang" << Endl;
    os << "            ProcessorType: yang_user_textdata" << Endl;
    os << "        </support_api/yang/assignment/data>" << Endl;

    os << "        <support_api/yang/assignment/user_info>" << Endl;
    os << "            AuthModuleName: yang" << Endl;
    os << "            ProcessorType: yang_user_info" << Endl;
    os << "            RequestTimeout: 60s" << Endl;
    os << "        </support_api/yang/assignment/user_info>" << Endl;

    os << "        <support_api/yang/assignment/queue>" << Endl;
    os << "            AuthModuleName: yang" << Endl;
    os << "            ProcessorType: yang_user_textdata_queue" << Endl;
    os << "        </support_api/yang/assignment/queue>" << Endl;

    os << "        <support_api/yang/assignment/historical>" << Endl;
    os << "            AuthModuleName: yang" << Endl;
    os << "            ProcessorType: yang_user_textdata_history" << Endl;
    os << "        </support_api/yang/assignment/historical>" << Endl;

    os << "        <support_api/registration/bv>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_background_video" << Endl;
    os << "        </support_api/registration/bv>" << Endl;

    os << "        <support_api/chat/resource>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_resource" << Endl;
    os << "        </support_api/chat/resource>" << Endl;

    os << "        <support_api/chat/resource/preview>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_resource_preview" << Endl;
    os << "        </support_api/chat/resource/preview>" << Endl;

    os << "        <support_api/chat/flags/edit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_flags" << Endl;
    os << "        </support_api/chat/flags/edit>" << Endl;

    os << "        <api/staff/tag/operators>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: tag_operators_list" << Endl;
    os << "        </api/staff/tag/operators>" << Endl;

    os << "        <api/staff/chat/message/edit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_edit" << Endl;
    os << "        </api/staff/chat/message/edit>" << Endl;

    os << "        <api/staff/chat/message/rate>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_rate_message" << Endl;
    os << "        </api/staff/chat/message/rate>" << Endl;

    os << "        <api/staff/support/requests>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_requests" << Endl;
    os << "        </api/staff/support/requests>" << Endl;

    os << "        <api/staff/support/request/defer>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_defer_request" << Endl;
    os << "        </api/staff/support/request/defer>" << Endl;

    os << "        <api/staff/support/request/undefer>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_undefer_request" << Endl;
    os << "        </api/staff/support/request/undefer>" << Endl;

    os << "        <api/staff/chat/sticker/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_sticker_list" << Endl;
    os << "            IsAdminProcessor: 1" << Endl;
    os << "        </api/staff/chat/sticker/list>" << Endl;

    os << "        <api/yandex/chat/sticker/list>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_sticker_list" << Endl;
    os << "            IsAdminProcessor: 0" << Endl;
    os << "        </api/yandex/chat/sticker/list>" << Endl;

    os << "        <api/staff/chat/sticker/edit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_sticker_edit" << Endl;
    os << "        </api/staff/chat/sticker/edit>" << Endl;

    os << "        <api/support_staff/documents_checks/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: get_user_documents_check" << Endl;
    os << "        </api/support_staff/documents_checks/get>" << Endl;

    os << "        <api/support_staff/documents_checks/get_history>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: get_user_documents_check_history" << Endl;
    os << "        </api/support_staff/documents_checks/get_history>" << Endl;

    os << "        <api/staff/chat/sticker/history>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_sticker_history" << Endl;
    os << "        </api/staff/chat/sticker/history>" << Endl;

    os << "        <api/staff/chats/feed>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: robot_chat_admin_feed" << Endl;
    os << "        </api/staff/chats/feed>" << Endl;

    os << "        <api/staff/support_calls/users_count>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_users_count" << Endl;
    os << "        </api/staff/support_calls/users_count>" << Endl;

    os << "        <api/support_staff/categorization/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_get_categorization" << Endl;
    os << "        </api/support_staff/categorization/get>" << Endl;

    os << "        <api/support_staff/categorization/add>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_add_categorization" << Endl;
    os << "        </api/support_staff/categorization/add>" << Endl;

    os << "        <api/support_staff/categorization/remove>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_remove_categorization" << Endl;
    os << "        </api/support_staff/categorization/remove>" << Endl;

    os << "        <api/support_staff/categorizer/tree>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_categorization_tree" << Endl;
    os << "        </api/support_staff/categorizer/tree>" << Endl;

    os << "        <api/support_staff/categorizer/tree/edit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_categorization_tree_edit" << Endl;
    os << "        </api/support_staff/categorizer/tree/edit>" << Endl;

    os << "        <api/support_staff/telephony/event>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: support_center_call_event" << Endl;
    os << "        </api/support_staff/telephony/event>" << Endl;

    os << "        <api/yandex/user_document_photo/upload>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_document_photo_upload" << Endl;
    os << "        </api/yandex/user_document_photo/upload>" << Endl;

    os << "        <api/yandex/user/settings/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: get_personal_settings" << Endl;
    os << "        </api/yandex/user/settings/get>" << Endl;

    os << "        <support_api/registration/user/phone/submit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_phone_bind_submit" << Endl;
    os << "        </support_api/registration/user/phone/submit>" << Endl;

    os << "        <support_api/registration/user/phone/commit>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_phone_bind_commit" << Endl;
    os << "        </support_api/registration/user/phone/commit>" << Endl;

    os << "        <support_api/yang/selfie-assignment/data>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: yang_face_matching_results" << Endl;
    os << "        </support_api/yang/selfie-assignment/data>" << Endl;

    os << "        <api/yandex/delegation/p2p>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_p2p_delegation" << Endl;
    os << "        </api/yandex/delegation/p2p>" << Endl;

    os << "        <api/yandex/delegation/p2p/reject>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: user_p2p_delegation_reject" << Endl;
    os << "        </api/yandex/delegation/p2p/reject>" << Endl;

    os << "        <api/staff/external_access_token/upsert>" << Endl;
    os << "            ProcessorType: upsert_external_access_token" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/external_access_token/upsert>" << Endl;

    os << "        <api/staff/external_access_token/invalidate>" << Endl;
    os << "            ProcessorType: invalidate_external_access_token" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "        </api/staff/external_access_token/invalidate>)" << Endl;

    os << "        <api/mosgorpas/offers/create>" << Endl;
    os << "            AuthModuleName: mosgorpas" << Endl;
    os << "            ProcessorType: create_offers" << Endl;
    os << "            IsExternal: true" << Endl;
    os << "        </api/mosgorpas/offers/create>" << Endl;

    os << "        <service_app/route/get>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: get_service_route" << Endl;
    os << "        </service_app/route/get>" << Endl;

    os << "        <api/yandex/distributing_block/show>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: show_distributing_block" << Endl;
    os << "        </api/yandex/distributing_block/show>" << Endl;

    os << "        <api/yandex/distributing_block/hide>" << Endl;
    os << "            AuthModuleName: fake" << Endl;
    os << "            ProcessorType: hide_distributing_block" << Endl;
    os << "        </api/yandex/distributing_block/hide>" << Endl;

    os << "        <api/staff/user/find_by_phone>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: find_user_by_phone" << Endl;
    os << "        </api/staff/user/find_by_phone>" << Endl;

    os << "        <api/telematics/push>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: telematics_pusher" << Endl;
    os << "        </api/telematics/push>" << Endl;

    os << "        <api/signalq/sessions/start>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: signalq_session_starter" << Endl;
    os << "        </api/signalq/sessions/start>" << Endl;

    os << "        <api/signalq/signals/create>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: signalq_signal_creater" << Endl;
    os << "        </api/signalq/signals/create>" << Endl;

    os << "        <api/leasing/signalq/status-history/intervals/list>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: signalq_status_history" << Endl;
    os << "        </api/leasing/signalq/status-history/intervals/list>" << Endl;

    os << "        <api/leasing/signalq/trace/tag/resolution/set>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: signalq_drivematics_set_resolution" << Endl;
    os << "        </api/leasing/signalq/trace/tag/resolution/set>" << Endl;

    os << "         <api/leasing/cars/add>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: add_leasing_cars" << Endl;
    os << "         </api/leasing/cars/add>" << Endl;

    os << "         <api/leasing/zone/add>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: zone_add" << Endl;
    os << "         </api/leasing/zone/add>" << Endl;

    os << "         <api/leasing/zone/remove>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: zone_remove" << Endl;
    os << "         </api/leasing/zone/remove>" << Endl;

    os << "         <api/leasing/zone/get>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: zone_get" << Endl;
    os << "         </api/leasing/zone/get>" << Endl;

    os << "         <api/leasing/portfolio/get>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: leasing_portfolio" << Endl;
    os << "             RequestTimeout: 10s" << Endl;
    os << "         </api/leasing/portfolio/get>" << Endl;

    os << "         <api/leasing/taxi_companies/list>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: taxi_companies_list" << Endl;
    os << "             RequestTimeout: 10s" << Endl;
    os << "         </api/leasing/taxi_companies/list>" << Endl;

    os << "         <api/leasing/signals/list>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: get_signals" << Endl;
    os << "             RequestTimeout: 10s" << Endl;
    os << "         </api/leasing/signals/list>" << Endl;

    os << "         <api/leasing/score/update>" << Endl;
    os << "             AuthModuleName: fake" << Endl;
    os << "             ProcessorType: add_leasing_score" << Endl;
    os << "         </api/leasing/score/update>" << Endl;
}

void NDrive::TServerConfigGenerator::GetBackgroundConfigString(IOutputStream& os) const {
    os << "        <Processes>" << Endl;
    if (NeedBackground & EServerBackgrounds::RentPricing) {
        os << "            <rent_pricing_watcher>" << Endl;
        os << "                Type: rent_pricing_watcher" << Endl;
        os << "                NotifierName: test_start" << Endl;
        os << "                Period: 5s" << Endl;
        os << "                SyncStorageName: alerts_states" << Endl;
        os << "            </rent_pricing_watcher>" << Endl;
    }
    if (NeedBackground & EServerBackgrounds::DBService) {
        os << "            <db_service>" << Endl;
        os << "                Type: db_service" << Endl;
        os << "                NotifierName: test_start" << Endl;
        os << "                Period: 1m" << Endl;
        os << "                SyncStorageName: alerts_states" << Endl;
        os << "            </db_service>" << Endl;
        os << "            <db_regular_service>" << Endl;
        os << "                Type: db_regular_service" << Endl;
        os << "                NotifierName: test_start" << Endl;
        os << "                Period: 10s" << Endl;
        os << "                SyncStorageName: alerts_states" << Endl;
        os << "            </db_regular_service>" << Endl;
    }
    if (NeedBackground & EServerBackgrounds::FuelingService) {
        os << "            <user_fueling_control>" << Endl;
        os << "                Type: user_fueling" << Endl;
        os << "                TagName: user_fueling_tag" << Endl;
        os << "                Period: 5s" << Endl;
        os << "                SyncStorageName: alerts_states" << Endl;
        os << "            </user_fueling_control>" << Endl;
    }
    if (NeedBackground & EServerBackgrounds::InsuranceTasks) {
        os << "            <insurance_tasks_builder>" << Endl;
        os << "                Type: insurance_tasks_builder" << Endl;
        os << "                NotifierName: test_start" << Endl;
        os << "                Period: 5s" << Endl;
        os << "                SyncStorageName: alerts_states" << Endl;
        os << "            </insurance_tasks_builder>" << Endl;
    }
    if (NeedBackground & EServerBackgrounds::CarMarkers) {
        os << "            <markers>" << Endl;
        os << "                Type: car_markers" << Endl;
        os << "                CleanTagName: just_cleaned" << Endl;
        os << "                FullTankTagName: full_tank_marker" << Endl;
        os << "                RidingTagsCorrection: no_price" << Endl;
        os << "                SyncStorageName: alerts_states" << Endl;
        os << "                PackOfferPushNotifier: test_start" << Endl;
        os << "                Period: 10s" << Endl;
        os << "            </markers>" << Endl;
    }
    if (NeedBackground & EServerBackgrounds::TakeoutRegular) {
        os << "            <takeout_regular>" << Endl;
        os << "                Type: takeout_regular" << Endl;
        os << "                SyncStorageName: alerts_states" << Endl;
        os << "                TracksAPIName: " << "drive_cache_prestable" << Endl;
        os << "                Period: 2s" << Endl;
        os << "            </takeout_regular>" << Endl;
    }
    os << "        </Processes>" << Endl;
}

THolder<TAnyYandexConfig> NDrive::TServerConfigGenerator::Generate() {
    TStringStream ss;
    ToString(ss);
    THolder<TAnyYandexConfig> config(new TAnyYandexConfig);
    UNIT_ASSERT(config->ParseMemory(ss.Str()));
    return config;
}

TPostgresConfigGenerator::TPostgresConfigGenerator()
    : Certificate(GetEnv("DB_SSL_CERTIFICATE", "/home/ivanmorozov/drive_keys/allCAs.pem"))
{
}

TString TPostgresConfigGenerator::GetTestingConnectionString() const {
    TStringStream ss;
    ss << "host=" << Host << " ";
    ss << "port=" << Port << " ";
    ss << "dbname=" << Db << " ";
    ss << "user=" << User << " ";
    ss << "password=" << Password << " ";
    if (Certificate) {
        ss << "sslmode=verify-full ";
        ss << "sslrootcert=" << Certificate;
    }
    return ss.Str();
}

NRTLine::TPostgresStorageOptionsImpl TPostgresConfigGenerator::GetPostgresConfigTesting() {
    NRTLine::TPostgresStorageOptionsImpl result;
    // standard place for certificate: sslrootcert=~/.postgresql/root.crt
    result.SetSimpleConnectionString(GetTestingConnectionString());
    result.SetHardLimitPoolSize(1);

    return result;
}

TFakeHistoryContext::TFakeHistoryContext(NStorage::IDatabase::TPtr database)
    : Database(database ? database : TDriveAPIConfigGenerator().CreateDatabase())
    , HistoryContext(Database)
    , TagsMeta(new TTagsMeta(HistoryContext, THistoryConfig().SetPingPeriod(TDuration::Seconds(20))))
{
    TagsMeta->InitPropositions(TPropositionsManagerConfig());
}

NUtil::THttpReply NDrive::TServerConfigGenerator::GetSendReply(const NNeh::THttpRequest& request, const TInstant deadline, const TString& preLandingId) const {
    if (LoggingEnabled) {
        GetCanonizationStream() << Canonize(request) << Endl;
        UNIT_ASSERT(ValidateAndDescribeRequest(request, false));
    }
    NUtil::THttpReply result = NehAgent->SendMessageSync(request, deadline);
    if (LoggingEnabled) {
        GetCanonizationStream() << Canonize(result) << Endl;
        if (result.IsSuccessReply()) {
            UNIT_ASSERT(ValidateAndDescribeReply(request.GetUri(), result, false));
        }
    }

    TString landingId = preLandingId;
    bool shouldPatch = true;
    if (preLandingId.StartsWith("!dont_patch ")) {
        landingId = preLandingId.substr(12);
        shouldPatch = false;
    }

    if (landingId != "") {
        INFO_LOG << "Result data: " << result.Content() << Endl;
        UNIT_ASSERT(result.Code() != HTTP_OK);
        NJson::TJsonValue jsonReply;
        UNIT_ASSERT(NJson::ReadJsonFastTree(result.Content(), &jsonReply));
        UNIT_ASSERT_VALUES_EQUAL(jsonReply["error_details"]["special_info"]["landings"][0]["id"].GetString(), landingId);
        NNeh::THttpRequest requestAccept = request;
        requestAccept.AddCgiData("&user_choice=accept");

        if (shouldPatch) {
            auto jsonPatchStr = jsonReply["error_details"]["special_info"]["landings"][0]["payload_patch"];
            if (jsonPatchStr.IsDefined() && jsonReply["error_details"]["special_info"]["landings"][0]["payload_patch"].IsString()) {
                NJson::TJsonValue jsonPatch;
                UNIT_ASSERT(NJson::ReadJsonTree(jsonReply["error_details"]["special_info"]["landings"][0]["payload_patch"].GetString(), &jsonPatch, false));
                NJson::TJsonValue currentPayload;
                TMemoryInput inp(requestAccept.GetPostData().Data(), requestAccept.GetPostData().Size());
                if (NJson::ReadJsonTree(&inp, &currentPayload, false) && jsonPatch.IsMap()) {
                    currentPayload["payload_patch"] = jsonReply["error_details"]["special_info"]["landings"][0]["payload_patch"].GetString();
                    requestAccept.SetPostData(TBlob::FromString(currentPayload.GetStringRobust())).SetRequestType("POST");
                }
            }
        }

        return NDrive::TServerConfigGenerator::GetSendReply(requestAccept, deadline);
    } else {
        if (result.Code() != HTTP_OK) {
            ERROR_LOG << result.GetDebugReply() << Endl;
        }
    }
    return result;
}

NJson::TJsonValue NDrive::TServerConfigGenerator::Request(const TString& userId, const TString& uri, const TString& cgi, const NJson::TJsonValue& post, bool checkReplyStatus, TMap<TString, TString> headers) const {
    NNeh::THttpRequest request;
    request.SetUri(uri);
    if (cgi) {
        request.SetCgiData(cgi);
    }
    if (post.IsDefined()) {
        request.SetPostData(post.GetStringRobust());
    }
    request.AddHeader("Authorization", userId);
    request.AddHeader("Lat", 55.736713);
    request.AddHeader("Lon", 37.640373);
    for (const auto& [key, value]: headers) {
        request.AddHeader(key, value);
    }
    auto reply = GetSendReply(request);
    if (checkReplyStatus) {
        UNIT_ASSERT_C(reply.IsSuccessReply(), TStringBuilder() << reply.Code() << ' ' << reply.Content());
    }
    return NJson::ReadJsonFastTree(reply.Content());
}

bool NDrive::TServerConfigGenerator::SwitchOffer(const TString& newOfferId, const TString& userId, const TString* deviceId /*= nullptr*/) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/yandex/offers/switch");
    if (!!newOfferId) {
        request.SetCgiData("&offer_id=" + newOfferId);
    }
    request.AddHeader("Authorization", userId);
    if (deviceId) {
        request.AddHeader("DeviceId", *deviceId);
    }
    return CommonSendReply(request);
}

bool NDrive::TServerConfigGenerator::EvolveTag(const TString& tagTargetName, const TString& userId, const TDuration timeout /*= TDuration::Seconds(60)*/, const TString& landingId, const TMaybe<EEvolutionMode> evolutionMode, NJson::TJsonValue post, const TString& sessionId) const {
    NNeh::THttpRequest request;
    post["tag_name"] = tagTargetName;
    request.SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
    if (!evolutionMode) {
        request.SetUri("/api/yandex/tag/evolve");
        request.SetCgiData("timeout=" + ::ToString(timeout.MicroSeconds()) + (sessionId ? ("&session_id=" + sessionId) : ""));
    } else {
        request.SetUri("/api/yandex/tag/evolve_adm");
        request.SetCgiData("timeout=" + ::ToString(timeout.MicroSeconds()) + (evolutionMode ? ("&evolution_mode=" + ::ToString(*evolutionMode)) : ""));
    }
    request.AddHeader("Authorization", userId);
    return CommonSendReply(request, 2, TInstant::Max(), landingId);
}

bool NDrive::TServerConfigGenerator::BookOffer(const TString& offerId, const TString& userId, const TString* deviceId, const TString& payloadPatch) const {
    NJson::TJsonValue baseBody;
    if (payloadPatch) {
        baseBody["payload_patch"] = payloadPatch;
    }
    return BookOffer(offerId, userId, deviceId, baseBody);
}

bool NDrive::TServerConfigGenerator::BookOffer(const TString& offerId, const TString& userId, const TString* deviceId /*= nullptr*/, const NJson::TJsonValue& baseBody /*= NJson::JSON_NULL*/) const {
    NNeh::THttpRequest request;
    NJson::TJsonValue post;
    if (baseBody.IsDefined()) {
        post = baseBody;
    }
    post["offer_id"] = offerId;
    request.SetUri("/api/yandex/offers/book").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
    request.AddHeader("Authorization", userId);
    if (deviceId) {
        request.AddHeader("DeviceId", *deviceId);
    }
    return CommonSendReply(request);
}

bool NDrive::TServerConfigGenerator::DropSession(const TString& sessionId, const TString& userId) const {
    NJson::TJsonValue resultJson;
    NNeh::THttpRequest request;
    request.SetUri("/api/yandex/sessions/drop");
    request.AddHeader("Authorization", userId);
    NJson::TJsonValue post;
    post["session_id"] = sessionId;
    request.SetPostData(TBlob::FromString(post.GetStringRobust()));
    return CommonSendReply(request);
}

NJson::TJsonValue NDrive::TServerConfigGenerator::GetUserLastSession(const TString& userId) const {
    NJson::TJsonValue resultJson;
    NNeh::THttpRequest request;
    request.SetUri("/api/yandex/user_sessions").SetCgiData("numdoc=1&type=universal");
    request.AddHeader("Authorization", userId);
    NUtil::THttpReply result = GetSendReply(request);
    if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
        ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        return NJson::JSON_NULL;
    }
    return resultJson;
}

void NDrive::TServerConfigGenerator::CompleteInitialRegistrationChatSteps(const NDrive::IServer* server, const TString& userId, TEnvironmentGenerator& eGenerator, bool bindCard) {
    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
    }

    // Intro
    {
        auto messagesJson = GetChatMessages(userId, "registration");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
        {
            auto chatsJson = GetChatsList(userId, server);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 2);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["faq_url"].GetString(), "https://yandex.ru/support/drive/joining/joining-faq.html");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["support_url"].GetString(), "https://forms.yandex.ru/surveys/6149/");
        INFO_LOG << "zxzx before intro" << Endl;
        UNIT_ASSERT(CommitChatAction(userId, "registration", "", {}));
        INFO_LOG << "zxzx after before intro" << Endl;
        UNIT_ASSERT(!ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
    }

    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 6);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 3);
        {
            auto result = GetChatMessages(userId, "registration", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(result["messages"].GetArray().size(), 6);
            UNIT_ASSERT_VALUES_EQUAL(result["user_last_viewed"].GetInteger(), result["messages"].GetArray()[2]["id"].GetInteger());
        }
    }

    // Support email
    {
        auto messagesJson = GetChatMessages(userId, "registration");
        {
            auto chatsJson = GetChatsList(userId, server);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 6);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
        INFO_LOG << "zxzx before support email" << Endl;
        UNIT_ASSERT(CommitChatAction(userId, "registration", "", {}));
        INFO_LOG << "zxzx after support email" << Endl;
        UNIT_ASSERT(!ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
    }

    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 10);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 3);
    }

    // Agreements
    {
        auto messagesJson = GetChatMessages(userId, "registration");
        {
            auto chatsJson = GetChatsList(userId, server);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 10);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 10);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
        INFO_LOG << "zxzx before agreements" << Endl;
        UNIT_ASSERT(CommitChatAction(userId, "registration", "", {}));
        INFO_LOG << "zxzx after agreements" << Endl;
        UNIT_ASSERT(!ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
    }

    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 13);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 2);
    }

    // Driving license
    {
        auto messagesJson = GetChatMessages(userId, "registration");
        {
            auto chatsJson = GetChatsList(userId, server);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 13);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 13);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");
        INFO_LOG << "zxzx before license" << Endl;
        UNIT_ASSERT(CommitChatAction(userId, "registration", "", {GetValidImageString(), GetValidImageString()}));
        INFO_LOG << "zxzx after license" << Endl;
        UNIT_ASSERT(!ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
    }

    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 15);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 1);
    }

    // Passport
    {
        auto messagesJson = GetChatMessages(userId, "registration");
        {
            auto chatsJson = GetChatsList(userId, server);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 15);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 15);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport");
        UNIT_ASSERT(CommitChatAction(userId, "registration", "", {GetValidImageString(), GetValidImageString()}));
        UNIT_ASSERT(!ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
    }

    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 20);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 4);
    }

    // Passport selfie
    {
        auto messagesJson = GetChatMessages(userId, "registration");
        {
            auto chatsJson = GetChatsList(userId, server);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 20);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 20);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_selfie");

        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[15]["type"].GetString(), "user_document_photos");
        UNIT_ASSERT(!GetDocumentPhoto(messagesJson["messages"].GetArray()[15]["text"].GetString(), userId).empty());

        UNIT_ASSERT(CommitChatAction(userId, "registration", "", {GetValidImageString()}));
    }

    if (!bindCard) {
        return;
    }

    // Credit card
    {
        eGenerator.GetBillingMock().SetReply(R"({
                                    "status": "success",
                                    "bound_payment_methods" : [
                                    {
                                      "region_id": 225,
                                      "payment_method" : "card",
                                      "expiration_month" : "07",
                                      "binding_ts" : "1536324485.656",
                                      "id" : "card-x12345",
                                      "expired" : false,
                                      "card_bank" : "TINKOFF BANK",
                                      "system" : "VISA",
                                      "account" : "510000****0257",
                                      "expiration_year" : "2021"
                                    },
                                    {
                                      "payment_method" : "yandex_account",
                                      "account" : "w/30b153cc-8e30-58e2-8d1a-1095bc49b915",
                                      "payment_system" : null,
                                      "currency" : "RUB",
                                      "id" : "yandex_account-w/30b153cc-8e30-58e2-8d1a-1095bc49b915",
                                      "balance" : "251.27"
                                    }
                                    ]})");

        auto messagesJson = GetChatMessages(userId, "registration");
        {
            auto chatsJson = GetChatsList(userId, server);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 22);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[20]["type"].GetString(), "user_document_photo");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 22);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "credit_card");
        UNIT_ASSERT(CommitChatAction(userId, "registration", "card-x12345", {}));
    }

    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 25);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 2);
    }

    // Outro (wait for verification results)
    {
        auto messagesJson = GetChatMessages(userId, "registration");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 25);
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[22]["text"].GetString(), "510000****0257");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "context_buttons");
    }

    {
        auto chatsJson = GetChatsList(userId, server);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 25);
        UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
    }
}

NUtil::THttpReply NDrive::TServerConfigGenerator::UpsertExternalAccessToken(const TString& userId, const TString& token, const TVector<TString>& ipWhiteList, const TString& userActorId) const {
    NNeh::THttpRequest request;
    request.SetUri("api/staff/external_access_token/upsert");
    request.AddHeader("Authorization", userActorId);
    NJson::TJsonValue ipWhiteListJson(NJson::EJsonValueType::JSON_ARRAY);
    for(const auto& node: ipWhiteList) {
        ipWhiteListJson.AppendValue(node);
    }
    request.SetPostData(Sprintf("{\"user_id\":\"%s\",\"token\":\"%s\",\"ip_whitelist\":%s}", userId.c_str(), token.c_str(), ipWhiteListJson.GetStringRobust().c_str()));
    return GetSendReply(request);
}

bool NDrive::TServerConfigGenerator::InvalidateExternalAccessToken(const TString& token, const TString& userActorId) const {
    NNeh::THttpRequest request;
    request.SetUri("api/staff/external_access_token/invalidate");
    request.AddHeader("Authorization", userActorId);
    request.SetPostData(Sprintf("{\"token\":\"%s\"}", token.c_str()));
    return CommonSendReply(request);
}

NUtil::THttpReply NDrive::TServerConfigGenerator::CreateMosgorpasOffer(const TString& carId, const TString& token, const TString& ipAddress) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/mosgorpas/offers/create");
    UNIT_ASSERT(carId);
    request.SetCgiData("car_id=" + carId);
    request.AddHeader("Authorization", "Bearer " + token);
    request.AddHeader("X-Forwarded-For-Y", ipAddress);
    NUtil::THttpReply result = GetSendReply(request);
    NJson::TJsonValue resultReport = NJson::JSON_NULL;
    return result;
}

NUtil::THttpReply NDrive::TServerConfigGenerator::GetServiceRoute(const TString& userId, const TString& userActorId) const {
    NNeh::THttpRequest request;
    request.SetUri("/service_app/route/get");
    request.SetCgiData("user_id=" + userId);
    request.AddHeader("Authorization", userActorId);
    NUtil::THttpReply result = GetSendReply(request);
    return result;
}

NUtil::THttpReply NDrive::TServerConfigGenerator::ShowDistributingBlock(const TString& userId, const TString& distributingBlockId) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/yandex/distributing_block/show");
    request.AddHeader("Authorization", userId);
    NJson::TJsonValue post = NJson::TMapBuilder
        ("id", distributingBlockId);

    request.SetPostData(post.GetStringRobust());
    return GetSendReply(request);
}

NUtil::THttpReply NDrive::TServerConfigGenerator::HideDistributingBlock(const TString& userId, const TString& distributingBlockId) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/yandex/distributing_block/hide");
    request.AddHeader("Authorization", userId);
    NJson::TJsonValue post = NJson::TMapBuilder
        ("id", distributingBlockId);

    request.SetPostData(post.GetStringRobust());
    return GetSendReply(request);
}

bool NDrive::TServerConfigGenerator::AddCarTag(NDrive::ITag::TPtr tag, const TString& carNumber, const TString& userActorId, const EUniquePolicy policy) const {
    NNeh::THttpRequest request;
    NJson::TJsonValue tagInfo;
    tagInfo["tag_name"] = tag->GetName();
    if (tag->HasTagPriority()) {
        tagInfo["priority"] = tag->GetTagPriority(0);
    }
    tag->SerializeSpecialDataToJson(tagInfo);
    tagInfo.InsertValue("car_number", carNumber);
    request.SetUri("/api/staff/car_tags/add").SetPostData(tagInfo.GetStringRobust()).SetRequestType("POST");
    request.SetCgiData("unique_policy=" + ::ToString(policy));
    request.AddHeader("Authorization", userActorId);
    return CommonSendReply(request);
}

NUtil::THttpReply NDrive::TServerConfigGenerator::AddLeasingCar(const TString& userId, const NDrivematics::TCarInfo& carInfo) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/leasing/cars/add");
    request.AddHeader("Authorization", userId);
    NJson::TJsonValue post = NJson::TMapBuilder
        ("car", NJson::ToJson(carInfo));

    request.SetPostData(post.GetStringRobust());
    return GetSendReply(request);
}

NUtil::THttpReply NDrive::TServerConfigGenerator::GetLeasingPortfolio(const TString& userId, TMaybe<std::pair<ui32, ui32>> segment) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/leasing/portfolio/get");
    request.AddHeader("Authorization", userId);
    if (segment) {
        NJson::TJsonValue post =
            NJson::TMapBuilder
                ("dates", NJson::TMapBuilder
                    ("since", segment->first)
                    ("until", segment->second)
        );
        request.SetPostData(post.GetStringRobust());
    }
    return GetSendReply(request);
}

NUtil::THttpReply NDrive::TServerConfigGenerator::GetLeasingTaxiCompanies(const TString& userId, TMaybe<std::pair<ui32, ui32>> segment) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/leasing/taxi_companies/list");
    request.AddHeader("Authorization", userId);
    if (segment) {
        NJson::TJsonValue post =
            NJson::TMapBuilder
                ("dates", NJson::TMapBuilder
                    ("since", segment->first)
                    ("until", segment->second)
        );
        request.SetPostData(post.GetStringRobust());
    }
    return GetSendReply(request);
}

NUtil::THttpReply NDrive::TServerConfigGenerator::GetSignals(const TString& userId, const TVector<TString>& signalsExt, TMaybe<ui64> pageSize, TMaybe<ui64> carTagsHistoryCursor, TMaybe<ui64> sessionTagsHistoryCursor) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/leasing/signals/list");
    request.AddHeader("Authorization", userId);
    TStringBuilder cgiData;
    if (!signalsExt.empty()) {
        cgiData << "signals=" << JoinStrings(signalsExt, ",");
    }
    if (pageSize) {
        if (cgiData.size() > 1) {
            cgiData << "&";
        }
        cgiData << "page_size=" << *pageSize;
    }
    if (carTagsHistoryCursor) {
        if (cgiData.size() > 1) {
            cgiData << "&";
        }
        cgiData << "cars_cursor=" << *carTagsHistoryCursor;
    }
    if (sessionTagsHistoryCursor) {
        if (cgiData.size() > 1) {
            cgiData << "&";
        }
        cgiData << "sessions_cursor=" << *sessionTagsHistoryCursor;
    }
    if (cgiData.size() > 1) {
        request.SetCgiData(cgiData);
    }
    return GetSendReply(request);
}

bool NDrive::TServerConfigGenerator::LeasingAddScore(const TString& userId, const TString& leasingCompanyName, double score, TMaybe<double> telematicsScore) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/leasing/score/update");
    request.AddHeader("Authorization", userId);
    NJson::TJsonValue post =
        NJson::TMapBuilder("score", score)
                          ("leasing_company", leasingCompanyName);

    if (telematicsScore) {
        post["telematics_cars_score"] = *telematicsScore;
    }
    request.SetPostData(post.GetStringRobust());

    return CommonSendReply(request);
}

NUtil::THttpReply NDrive::TServerConfigGenerator::ZoneRequest(const TString& userId, const TString method, const NJson::TJsonValue& post, const TString& cgi) const {
    NNeh::THttpRequest request;
    request.SetUri("/api/leasing/zone/" + method);
    request.AddHeader("Authorization", userId);

    if (post.IsDefined()) {
        request.SetPostData(post.GetStringRobust());
    }
    if (cgi) {
        request.SetCgiData(cgi);
    }
    return GetSendReply(request);
}


NJson::TJsonValue NDrive::TServerConfigGenerator::SignalqSessionsStart(const TString& serialNumber, const TString& operatorId) {
    NNeh::THttpRequest request;

    NJson::TJsonValue postData;
    postData["serial_number"] = serialNumber;
    postData["unix_timestamp"] = Now().Seconds();

    request.SetUri("/api/signalq/sessions/start").SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
    request.AddHeader("Authorization", operatorId);
    NUtil::THttpReply result = GetSendReply(request);
    NJson::TJsonValue resultReport = NJson::JSON_NULL;
    if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
        ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
    }
    return resultReport;
}

NJson::TJsonValue NDrive::TServerConfigGenerator::SignalqCreateJsonEvent(const TString& serialNumber, const TString& eventType, const TString& eventId, const TInstant timestamp) {
    NJson::TJsonValue eventJson;
    eventJson["serial_number"] = serialNumber;
    auto event = NJson::TJsonMap ({
        {"id", eventId},
        {"at", timestamp.ToString()},
        {"type", eventType},
        {"s3_photo_path", "8a80d933bee748198c2ca63f71b166fa/media/emmc/internal/20220318/photo0/20220318_141154.jpg"},
        {"s3_external_photo_path", "8a80d933bee748198c2ca63f71b166fa/media/emmc/internal/20220318/photo0/20220318_141154.jpg"},
        {"s3_video_path", "8a80d933bee748198c2ca63f71b166fa/media/emmc/internal/20220318/photo0/20220318_141154.jpg"},
        {"s3_external_video_path", "8a80d933bee748198c2ca63f71b166fa/media/emmc/internal/20220318/video0/20220318_141154.mp4"},
        {"signalled", true}
    });
    event["gnss"] = NJson::TJsonMap ({
        {"lat", "55.8272"},
        {"lon", "49.0301"},
        {"speed_kmph", "0.0"},
        {"accuracy_m", "11.8638"},
        {"timestamp", timestamp.ToString()},
        {"direction_deg", "252.8"}
    });
    eventJson["event"] = event;
    return eventJson;
}

NJson::TJsonValue NDrive::TServerConfigGenerator::SignalqSignalCreate(const NJson::TJsonValue& postData, const TString& operatorId) {
    NNeh::THttpRequest request;
    request.SetUri("api/signalq/signals/create").SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
    request.AddHeader("Authorization", operatorId);
    NUtil::THttpReply result = GetSendReply(request);
    NJson::TJsonValue resultReport = NJson::JSON_NULL;
    if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
        ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
    }
    return resultReport;
}

NJson::TJsonValue NDrive::TServerConfigGenerator::SignalqTraceTagResolutionSet(const TString& tagId, const TString& resolution, const TString& operatorId) {
    NNeh::THttpRequest request;

    NJson::TJsonValue postData;
    postData["tag_id"] = tagId;
    postData["resolution"] = resolution;

    request.SetUri("api/leasing/signalq/trace/tag/resolution/set").SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
    request.AddHeader("Authorization", operatorId);
    NUtil::THttpReply result = GetSendReply(request);
    NJson::TJsonValue resultReport = NJson::JSON_NULL;
    if ((result.Code() != 200 && result.Code() != 400 && result.Code() != 404) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
        ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
    }
    return resultReport;
}

NJson::TJsonValue NDrive::TServerConfigGenerator::SignalqTraceTagResolutionSet(const TString& serialNumber, const TString& eventId, const TInstant eventAt, const TString& resolution, const TString& operatorId) {
    NNeh::THttpRequest request;

    NJson::TJsonValue postData;
    postData["serial_number"] = serialNumber;
    postData["event_id"] = eventId;
    postData["event_at"] = eventAt.Seconds();
    postData["resolution"] = resolution;

    request.SetUri("api/signalq/trace/tag/resolution/set").SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
    request.AddHeader("Authorization", operatorId);
    NUtil::THttpReply result = GetSendReply(request);
    NJson::TJsonValue resultReport = NJson::JSON_NULL;
    if ((result.Code() != 200 && result.Code() != 400 && result.Code() != 404) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
        ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
    }
    return resultReport;
}

NJson::TJsonValue NDrive::TServerConfigGenerator::SignalqNotifySupport(const NJson::TJsonValue& postData, const TString& operatorId) {
    NNeh::THttpRequest request;
    request.SetUri("api/signalq/notify_support").SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
    request.AddHeader("Authorization", operatorId);
    NUtil::THttpReply result = GetSendReply(request);
    NJson::TJsonValue resultReport = NJson::JSON_NULL;
    if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
        ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
    }
    return resultReport;
}

std::pair<ui32, NJson::TJsonValue> NDrive::TServerConfigGenerator:: SignalqStatusHistory(const NJson::TJsonValue& postData, const TString& userId) {
    NNeh::THttpRequest request;
    request.SetUri("api/leasing/signalq/status-history/intervals/list").SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
    request.AddHeader("Authorization", userId);
    NUtil::THttpReply result = GetSendReply(request);
    NJson::TJsonValue resultReport = NJson::JSON_NULL;
    NJson::ReadJsonFastTree(result.Content(), &resultReport);
    return {result.Code(), resultReport};
}

TString TTestNodeResolver::GetNextNode(const IChatUserContext::TPtr /*context*/, const NDrive::NChat::TMessageEvents& /*messages*/) const {
    const auto compare = [](const std::pair<i32, TString>& lhs, const std::pair<i32, TString>& rhs) {
        return lhs.first > rhs.first;
    };
    auto classificationResults = MockClassificationResults.empty() ? GetDefaultClassificationResults() : MockClassificationResults;
    Sort(classificationResults.begin(), classificationResults.end(), compare);
    for (auto&& [conf, id] : classificationResults) {
        auto resolveParameters = TryFindResolveParameters(id, conf, false);
        if (resolveParameters) {
            return resolveParameters->GetNodeId();
        }
    }
    return GetDefaultNode();
}

NThreading::TFuture<TString> TTestNodeResolver::GetNextNodeFuture(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const {
    return NThreading::MakeFuture(GetNextNode(context, messages));
}

NThreading::TFuture<TTaxiSupportChatSuggestClient::TSuggestResponse> TTestNodeResolver::GetSuggest(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages, bool userOptionsSuggest) const {
    Y_UNUSED(context);
    Y_UNUSED(messages);
    Y_UNUSED(userOptionsSuggest);
    return NThreading::MakeFuture(TTaxiSupportChatSuggestClient::TSuggestResponse());
}

NJson::TJsonValue TTestNodeResolver::GetSuggestedChatOptions(const TTaxiSupportChatSuggestClient::TSuggestResponse& suggestResponse) const {
    Y_UNUSED(suggestResponse);
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    NJson::TJsonValue m1 = NJson::JSON_MAP;
    m1["text"] = "Option 1";
    m1["message_text"] = "node_class_1";
    m1["type"] = "message";
    NJson::TJsonValue m2 = NJson::JSON_MAP;
    m2["text"] = "";
    m2["message_text"] = "node_class_2";
    m2["type"] = "message";
    result.AppendValue(m1);
    result.AppendValue(m2);
    return result;
}

void TTestNodeResolver::AddClassificationResult(const TString& classification, i32 confidence) {
    MockClassificationResults.emplace_back(std::make_pair(confidence, classification));
}

TVector<std::pair<i32, TString>>  TTestNodeResolver::GetDefaultClassificationResults() const {
    TVector<std::pair<i32, TString>> result;
    result.emplace_back(std::make_pair(40, "class_1"));
    result.emplace_back(std::make_pair(60, "class_2"));
    return result;
}

TString TTestNodeResolver::GetType() const {
    return "test";
}

INodeResolver::TFactory::TRegistrator<TTestNodeResolver> TTestNodeResolver::Registrator("test");
