#pragma once

#include <drive/backend/base/server.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/data/evolution_policy.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/database/history/common.h>
#include <drive/backend/documents_verification/manager.h>
#include <drive/backend/fueling_manager/ut/library/tanker_emulator.h>
#include <drive/backend/localization/localization.h>
#include <drive/backend/server/library/config.h>
#include <drive/backend/server/library/server.h>
#include <drive/backend/support_center/telephony/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/tags/tags_search.h>

#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/async_proxy/async_delivery.h>
#include <rtline/library/async_proxy/ut/helper/helper.h>
#include <rtline/util/network/neh.h>
#include <rtline/util/types/accessor.h>

#include <util/random/fast.h>
#include <util/stream/output.h>
#include <util/string/builder.h>
#include <util/string/join.h>
#include <util/system/env.h>

namespace maps::local_postgres {
    class Database;
}

class TDriveAPI;

namespace NDrive {
    class TEntitySession;
}

namespace NRTLine {
    class TPostgresStorageOptionsImpl;
}

namespace NDrivematics {
    class TCarInfo;
}

class TValidationHelpers {
public:

    static bool ResponseListHasObject(const NJson::TJsonValue::TArray& objects, const TString& objectId, const TString& objectNumber = "") {
        for (auto object : objects) {
            DEBUG_LOG << object.GetStringRobust() << Endl;
            if (object.Has("id") && object.GetValueByPath("id")->GetStringRobust() == objectId) {
                return true;
            }
            if (!!objectNumber && object.Has("number") && object.GetValueByPath("number")->GetStringRobust() == objectNumber) {
                return true;
            }
        }
        return false;
    }

    static NJson::TJsonValue GetFlatJsonItem(const NJson::TJsonValue::TArray& objects, const TString& objectsKeyColumn, const TString& objectId) {
        for (auto object : objects) {
            ERROR_LOG << object << Endl;
            if (object.Has(objectsKeyColumn) && object[objectsKeyColumn].IsString() && object[objectsKeyColumn].GetStringRobust() == objectId) {
                return object;
            }
        }
        return NJson::JSON_NULL;
    }

    static TString DocumentTypeToString(const NUserDocument::EType& documentType) {
        return ToString(documentType);
    }

    static bool CheckIsActiveRole(const NJson::TJsonValue& report, const TString& roleId, const bool active) {
        const NJson::TJsonValue::TArray* arr;
        if (!report["report"].GetArrayPointer(&arr)) {
            ERROR_LOG << report << Endl;
            return false;
        }
        for (auto&& i : *arr) {
            if (!i["role_id"].IsString()) {
                ERROR_LOG << report << Endl;
                return false;
            }
            if (i["role_id"].GetString() == roleId) {
                if (!i["active"].IsString()) {
                    ERROR_LOG << report << Endl;
                    return false;
                }
                if (!active == (i["active"].GetString() == "1")) {
                    ERROR_LOG << report << Endl;
                    return false;
                }
                return true;
            }
        }
        ERROR_LOG << report << " / " << roleId << Endl;
        return false;
    }

    static bool WaitFor(std::function<bool()> condition, const TDuration timeout) {
        TInstant start = Now();
        while (Now() - start < timeout) {
            if (condition())
                return true;
            Sleep(TDuration::MilliSeconds(5));
        }
        return false;
    }
};

class TFakeHistoryContext: public ITagsHistoryContext {
private:
    NStorage::IDatabase::TPtr Database;
    THistoryContext HistoryContext;
    THolder<TTagsMeta> TagsMeta;

public:
    TFakeHistoryContext(NStorage::IDatabase::TPtr database = nullptr);

    virtual TAtomicSharedPtr<NStorage::IDatabase> GetDatabase() const override {
        return Database;
    }
    virtual const ITagsMeta& GetTagsManager() const override {
        return *TagsMeta;
    }
};

class TReplyContextMock : public IReplyContext {
private:
    TServerRequestData RequestData;

public:
    TReplyContextMock(const TString& queryString = "")
        : RequestData(queryString.c_str())
    {
    }

    const TServerRequestData& GetRequestData() const override {
        return RequestData;
    }

    TServerRequestData& MutableRequestData() override {
        return RequestData;
    }

    const TCgiParameters& GetCgiParameters() const override {
        return RequestData.CgiParam;
    }

    TCgiParameters& MutableCgiParameters() override {
        return RequestData.CgiParam;
    }

    TInstant GetRequestStartTime() const override {
        return TInstant::MicroSeconds(RequestData.RequestBeginTime());
    }

    TStringBuf GetUri() const override {
        return RequestData.ScriptName();
    }

    IOutputStream& Output() override {
        ythrow yexception() << "Not implemented";
    }

    bool IsHttp() const override {
        ythrow yexception() << "Not implemented";
    }

    const TBlob& DoGetBuf() const override {
        ythrow yexception() << "Not implemented";
    }

    void AddReplyInfo(const TString& /* key */, const TString& /* value */) override {
        ythrow yexception() << "Not implemented";
    }

    void MakeSimpleReply(const TBuffer& /* buf */, int /* code */) override {
        ythrow yexception() << "Not implemented";
    }
};

const TString USER_ID_NOPHONE = "953fc0ca-93b2-4323-80a1-071200708db9";
const TString USER_ID_DEFAULT = "97b933ed-4145-400c-98cc-f0c755a12590";
const TString USER_ID_TECH = "94c9653e-9f55-49f2-af8a-5156342bd9c8";
const TString USER_ID_BLOCKED = "c6ef6e9d-469b-4122-8535-3c721229bc21";
const TString USER_ID_DEFAULT2 = "b7b4b743-7b91-4da2-8c29-6c2fc20021ba";
const TString USER_ID_DEFAULT1 = "a6021e53-6a12-4261-92e8-b5468376336e";
const TString USER_ROOT_DEFAULT = "8b33b36b-ca2a-4f16-9af4-dc1598f02ec4";

const TString USER_PHONE_BLOCKED = "79865434234";
const TString USER_ID_DEFAULT_PHONE = "79006468203";
const TString USER_PHONE_ONBOARDING = "79651782340";
const TString USER_PHONE_BASE_CLASS = "79038144628";
const TString USER_ID_DEFAULT2_PHONE = "79043356296";

const TString USER_PHONE_BLOCKED_ALT = "89865434234";
const TString USER_ID_DEFAULT_PHONE_ALT = "89006468203";
const TString USER_PHONE_ONBOARDING_ALT = "89651782340";
const TString USER_PHONE_BASE_CLASS_ALT = "89038144628";
const TString USER_ID_DEFAULT2_PHONE_ALT = "89043356296";

const TString DEFAULT_PETROL_STATION = "46e2059185f44d61b6aa578b1415ec49"; // https://st.yandex-team.ru/DRIVEBACK-2143/attachments/22547452
const TString POST_PAY_PETROL_STATION = "1702571f75f597a20bacfcd221e6654c";
const TString DEFAULT_PETROL_COLUMN = "1";
const TString DEFAULT_POSTPAY_PETROL_COLUMN = "8";
const TString CANCEL_PETROL_COLUMN = "4";

const TString PROMO_CODE_GENERATOR = "tst_promo_generator";
const TString PROMO_CODE_GENERATOR_ACCOUNT = "tst_account_promo_generator";
const TString PROMO_CODE_GENERATOR_SECOND = "tst_promo_generator_second";

const TString OBJECT_ID_DEFAULT = "65e60f94-29f7-4aa6-97e0-35f9786829cc";
const TString HEAD_ID_DEFAULT = "aaaaaaaaaaaa";

const TString SIGNAL_DEVICE_SN_DEFAULT = "2120244E00055";
#define OBJECT_IMEI_DEFAULT "77678"

#define OBJECT_ID_TM "f125aee3-56b9-4025-b28b-87fe01b3c4fb"
#define OBJECT_IMEI_TM "12332155"

#define OBJECT_ID_DEFAULT1 "ae66e869-7b59-4d9b-bc1c-535aa5ac66a7"
#define OBJECT_IMEI_DEFAULT1 "776781"

#define OBJECT_ID_DEFAULT2 "79314805-ca87-436a-b48f-711d5d0fd81d"
#define OBJECT_IMEI_DEFAULT2 "7767812"

namespace NDrive {

class TServerGuard: public IMessageProcessor {
private:
    TAtomicSharedPtr<TServer> Server;

public:
    TServerGuard(TServerConfig& config);
    ~TServerGuard();

    virtual bool Process(IMessage* message) override {
        const TSessionCorruptedGlobalMessage* mess = dynamic_cast<const TSessionCorruptedGlobalMessage*>(message);
        if (mess) {
            UNIT_ASSERT(!NDrive::HasServer());
            return true;
        }
        return false;
    }

    virtual TString Name() const override {
        return "ServerGuard";
    }

    const TServer& operator*() const {
        return *Server;
    }

    TServer& operator*() {
        return *Server;
    }

    TServer* operator->() {
        return Server.Get();
    }

    const TServer* operator->() const {
        return Server.Get();
    }

    TServer* Get() {
        return Server.Get();
    }

    const TServer* Get() const {
        return Server.Get();
    }
};

} // namespace NDrive

enum class EEnvironmentFeatures: ui32 {
    Default = 1 << 0,
    Cleaner = 1 << 1,
    Root = 1 << 2,
    Lock = 1 << 3,
    Scanner = 1 << 4,
    Tech = 1 << 5,
    Landing = 1 << 6,
    InfoAccess = 1 << 7,
    CarModels = 1 << 8,
    OfferBuilder = 1 << 9,
    Filters = 1 << 10,
    PromoCodes = 1 << 11,
    TankerMock = 1 << 12,
};

enum EFuelPatchTags {
    SimplePatch     /* "unit_test_patch_100p" */,
    OverrideModel   /* "unit_test_patch_100p_only" */,
    Dut1Sensor      /* "unit_test_patch_100p_dut" */,
    Dut2Sensor      /* "unit_test_patch_100p_dut2" */,
};

class TEnvironmentGenerator {
private:
    const NDrive::IServer& Server;
    const TDriveAPI& DriveAPI;
    TAtomicSharedPtr<NAPHelper::TSlowNehServer> BillingMock;
    R_FIELD(TString, SurgeTag, "old_state_reservation");
    R_FIELD(bool, NeedTelematics, true);
public:
    struct TCar {
        TString Id;
        TString IMEI;
        TString Number;
        TString Vin;
    };

    struct TPromoCodeFilter {
        TMaybe<TString> Prefix;
        TMaybe<TString> GivenOut;
        TMaybe<TString> Generator;
        TMaybe<TInstant> Since;
        TMaybe<TInstant> Until;
        TMaybe<bool> ActiveOnly;
        TMaybe<ui32> Count;
        TSet<TString> Ids;
    };

public:
    TEnvironmentGenerator(const NDrive::IServer& server)
        : Server(server)
        , DriveAPI(*Server.GetDriveAPI())
    {
        Singleton<TConfigurableCheckers>()->SetHistoryParsingAlertsActive(true);
    }

    NAPHelper::TSlowNehServer& GetBillingMock() {
        UNIT_ASSERT(BillingMock);
        return *BillingMock;
    }

    static const ui32 DefaultTraits = Max<ui32>();

    void BuildEnvironment(const ui32 envTraits = DefaultTraits);
    void BuildScannerData(NDrive::TEntitySession& session);
    void BuildTechData(NDrive::TEntitySession& session);
    void BuildCleanerData(NDrive::TEntitySession& session);
    void BuildAdditionalOfferData(NDrive::TEntitySession& session);
    void BuildDefaultData(NDrive::TEntitySession& session);
    void BuildRootPermissions(NDrive::TEntitySession& session);
    void BuildAdmPermissions(NDrive::TEntitySession& session);
    void BuildLockPermissions(NDrive::TEntitySession& session);
    void BuildInfoAccessPermissions(NDrive::TEntitySession& session);
    void BuildLandingPermissions(NDrive::TEntitySession& session);
    void BuildCarModelsPermissions(NDrive::TEntitySession& session);
    void BuildPrioritySimpleData(NDrive::TEntitySession& session, const i32 priority, const TString& prefix, const TString& baseClass);
    void BuildRTBackgrounds(const IServerBase& server, const ui64 backgrounds);
    void BuildPromoCodePermissions(NDrive::TEntitySession& session);
    void BuildOfferBuildersPermissions(NDrive::TEntitySession& session);
    void BuildFilters(NDrive::TEntitySession& session);
    void BuildAreaTags(NDrive::TEntitySession& session);
    void BuildUserTags(NDrive::TEntitySession& session);
    void BuildFilterActions(NDrive::TEntitySession& session);
    void BuildCarTags(NDrive::TEntitySession& session);

    TDocumentsVerificationConfig BuildDefaultDocumentsVerificationConfig() const;

    TCar CreateCar(NDrive::TEntitySession& session, const TString& model = "porsche_carrera", const TString& carIdSalt = Default<TString>(), const TString& vin = "");
    TCar CreateCar(const TString& model = "porsche_carrera", const TString& vin = "");
    TString CreateUser(const TString& login, const bool withRoles = true, const TString& status = "active", const TString& phoneNumber = "", const bool phoneVerified = true);
    TString CreateUserDocumentPhoto(const TString& userId, const NUserDocument::EType& photoType, const NUserDocument::EVerificationStatus status = NUserDocument::EVerificationStatus::NotYetProcessed);
    bool CompleteYangAssignment(const TString& assignmentId, const TString& photoStatuses, const TYangDocumentVerificationAssignment::EFraudStatus& overallStatus, bool isExp = false, const TVector<TString>& assignmentIds = TVector<TString>());
    bool SetPhotoStatus(const TString& photoId, const char status);
    bool CreateNewYangAssignments(const NDrive::IServer* server) const;
    bool CreateNewSelfieYangAssignments(const NDrive::IServer* server) const;
};

class TPostgresConfigGenerator {
public:
    TPostgresConfigGenerator();

    TString GetTestingConnectionString() const;
    NRTLine::TPostgresStorageOptionsImpl GetPostgresConfigTesting();

    void SetHost(const TString& value) {
        Host = value;
    }
    void SetPort(ui16 value) {
        Port = value;
    }
    void SetDb(const TString& value) {
        Db = value;
    }
    void SetUser(const TString& value) {
        User = value;
    }
    void SetPassword(const TString& value) {
        Password = value;
    }
    void SetCertificate(const TString& value) {
        Certificate = value;
    }

private:
    TString Host = "pgaas.mail.yandex.net";
    ui16 Port = 12000;

    TString Db = "extmaps-carsharing-testing";
    TString User = "carsharing";
    TString Password = "cYKIbr1eRXyzhX2Z7quoC1FUQ67W9geDDQG1B7qlnv4HCMKgzvnXiQBXvjcjv0fVe9k2Rlb3ORxXma2IPNXMMvNCXEzZweZNjViNIXY1HktOYDRXL8bEkj8MkV8YLcu5";
    TString Certificate;
};

class TDriveAPIConfig;

class TDriveAPIConfigGenerator {
public:
    TDriveAPIConfigGenerator(const TString& databaseType = "SQLite", const TString& databasePath = {});

    void SetOffersStorageName(const TString& name);
    void SetPrivateDataClientType(const TString& value);

    void DatabaseToString(IOutputStream& os) const;
    void ToString(IOutputStream& os, bool isYtEnabled) const;

    NStorage::IDatabase::TPtr CreateDatabase() const;
    THolder<TAnyYandexConfig> GenerateConfig(bool isYtEnabled) const;
    THolder<TDriveAPIConfig> Generate() const;
    ui32 GetBillingPort() const;

    ~TDriveAPIConfigGenerator();

private:
    TPostgresConfigGenerator PostgresConfigGenerator;
    TPortManager PortManager;
    TString DatabaseType;
    TFsPath DatabasePath;
    TFsPath DatabaseCopy;
    TString OffersStorageName;
    TString PrivateDataClientType;
    ui32 BillingPort;
    THolder<maps::local_postgres::Database> LocalPostgres;
};

enum EServerBackgrounds: ui32 {
    RentPricing = 1 << 0,
    InsuranceTasks = 1 << 1,
    Radar = 1 << 2,
    SurgeConstructor = 1 << 3,
    FuelingService = 1 << 4,
    CarMarkers = 1 << 5,
    TakeoutRegular = 1 << 7,
    FuturesBooking = 1 << 8,
    DBService = 1 << 9
};

class TValidationManager {
    class TProcessorTemplates {
        class TProcessоrStateTemplates {
        public:
            bool Init(const TYandexConfig::Section* section);
            bool ValidateJson(const NJson::TJsonValue& json) const;
            bool DescribeJson(const NJson::TJsonValue& json, TString& description) const;
            bool CheckValidationCount() const;
            TString GetStateDescription() const {
                return Description;
            }

        private:
            TMaybe<NDrive::TScheme> DescriptionScheme;
            TVector<NDrive::TScheme> ValidationSchemes;
            ui32 RequiredChecksCount = 0;
            mutable ui32 ChecksCount;
            TString Description;
            TString Name;
        };

    public:
        bool Init(const TYandexConfig::Section* section);
        bool ValidateJson(const TString& state, const NJson::TJsonValue& json, bool required = false) const;
        bool DescribeJson(const TString& state, const NJson::TJsonValue& json, TString& description) const;
        bool GetStateDescription(const TString& state, TString& description) const;
        bool CheckValidationCount() const;

    private:
        TMap<TString, TProcessоrStateTemplates> States;
        TString ProcessorName;
    };

public:
    using TTemplates = TMap<TString, TProcessorTemplates>;

    enum class EProcessorAction {
        Request,
        Reply,
    };

public:
    ~TValidationManager();
    bool Init(const TYandexConfig::Section* section);
    bool ValidateAndDescribeJson(EProcessorAction action, const TString& uri, const TString& state, const NJson::TJsonValue& json, bool required = false) const;

private:
    bool ValidateJson(EProcessorAction action, const TString& uri, const TString& state, const NJson::TJsonValue& json, bool required = false) const;
    bool DescribeJson(EProcessorAction action, const TString& uri, const TString& state, const NJson::TJsonValue& json) const;
    bool AddStateTemplates(EProcessorAction action, const TYandexConfig::TSectionsMap& rootSections);

private:
    TMap<EProcessorAction, TTemplates> Templates;
    bool WikiLayout;
};

namespace NDrive {

class TServerConfigGenerator {
public:
    class TDisableLogging {
    public:
        TDisableLogging(TServerConfigGenerator& generator)
            : Generator(generator)
            , Previous(generator.LoggingEnabled)
        {
            Generator.LoggingEnabled = false;
        }
        ~TDisableLogging() {
            Generator.LoggingEnabled = Previous;
        }

    private:
        TServerConfigGenerator& Generator;
        bool Previous;
    };

    class TSessionStateGuard {
    public:
        TSessionStateGuard(TServerConfigGenerator& generator, const TMaybe<TString>& state)
            : Generator(generator)
            , Previous(generator.SessionState)
        {
            if (state) {
                Generator.SessionState = *state;
            }
        }

        ~TSessionStateGuard() {
            Generator.SessionState = Previous;
        }
    private:
        TServerConfigGenerator& Generator;
        TString Previous;
    };

public:
    R_READONLY(ui32, ControllerPort, 0);
    R_READONLY(ui32, ServerPort, 0);
    R_FIELD(ui32, LogLevel, 6);
    R_FIELD(ui32, NeedBackground, EServerBackgrounds::RentPricing);
    R_FIELD(TString, SensorApiName);
    R_FIELD(TString, SensorDump);
    R_FIELD(TString, SurgeTag, "simple1");
    R_FIELD(bool, IsYtEnabled, false);
    R_FIELD(TString, StateTemplatesPath);
    R_FIELD(TValidationManager, ValidationManager, {});
    R_FIELD(TString, SessionState, "undefined");
    R_OPTIONAL(TString, YdbEndpoint);
    R_OPTIONAL(TString, YdbDatabase);

private:
    TDriveAPIConfigGenerator DriveAPIGenerator;
    THolder<NNeh::THttpClient> NehAgent;
    NSimpleMeta::TConfig MetaConfig;

    mutable TMap<TString, TString> StringTokens;
    mutable TMap<i64, i64> IntegerTokens;
    bool LoggingEnabled = true;

    TAtomicSharedPtr<TTankerEmulator> TankerMock;

private:
    TString Canonize(const TString& value) const;
    TString Canonize(const NJson::TJsonValue& value) const;
    TString Canonize(const NUtil::THttpReply& reply) const;
    TString Canonize(const NNeh::THttpRequest& request) const;

    TString FormMultipleSearchParams(const TVector<TString>& params) {
        TString paramStr = "";
        for (auto token : params) {
            if (!!paramStr) {
                paramStr += ",";
            }
            paramStr += token;
        }
        return paramStr;
    }

    bool CommonSendReply(const NNeh::THttpRequest& request, const ui32 attCountOn5xx = 1, const TInstant deadline = TInstant::Max(), const TString& landingId = "") const {
        for (ui32 i = 0; i < attCountOn5xx; ++i) {
            ui32 code = GetSendReply(request, deadline, landingId).Code();
            if (code / 100 == 5) {
                continue;
            } else {
                return code == HTTP_OK;
            }
        }
        return false;
    }

    virtual void GetApiConfigString(IOutputStream& os) const;
    virtual void GetBackgroundConfigString(IOutputStream& os) const;

public:
    NUtil::THttpReply GetSendReply(const NNeh::THttpRequest& request, const TInstant deadline = TInstant::Max(), const TString& landingId = Default<TString>()) const;
    NJson::TJsonValue Request(const TString& userId, const TString& uri, const TString& cgi = {}, const NJson::TJsonValue& post = {}, bool checkReplyStatus = true, TMap<TString, TString> headers = {}) const;

    bool ValidateAndDescribeJson(TValidationManager::EProcessorAction action, const TString& uri, const NJson::TJsonValue& json, bool required = false) const {
        return ValidationManager.ValidateAndDescribeJson(action, uri, SessionState, json, required);
    }
    bool ValidateAndDescribeReply(const TString& uri, const NUtil::THttpReply& reply, bool required = false) const;
    bool ValidateAndDescribeRequest(const NNeh::THttpRequest& request, bool required = false) const;

    bool SetYDBConfig() {
        YdbDatabase = GetEnv("YDB_DATABASE");
        YdbEndpoint = GetEnv("YDB_ENDPOINT");
        if (!YdbEndpoint || !YdbDatabase) {
            return false;
        }
        return true;
    }

    TString GetString() const {
        TStringStream ss;
        ToString(ss);
        return ss.Str();
    }

    bool WaitPrice(const ui32 expectation, const ui32 precision = 1) {
        TDisableLogging disableLogging(*this);
        for (ui32 i = 0; i <= 100; ++i) {
            NJson::TJsonValue userSessionReport = GetCurrentSession(USER_ID_DEFAULT);
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << " / " << expectation << Endl;
            if (Abs((double)expectation - userSessionReport["segment"]["session"]["specials"]["total_price"].GetDouble()) < precision) {
                return true;
            }
            Sleep(TDuration::Seconds(1));
        }
        ERROR_LOG << expectation << Endl;
        return false;
    }

    bool AddTag(NDrive::ITag::TPtr tag, const TString& objectId, const TString& userActorId, const NEntityTagsManager::EEntityType type, const EUniquePolicy policy = EUniquePolicy::NoUnique) {
        NNeh::THttpRequest request;
        NJson::TJsonValue tagInfo;
        tagInfo["tag_name"] = tag->GetName();
        if (tag->HasTagPriority()) {
            tagInfo["priority"] = tag->GetTagPriority(0);
        }
        tag->SerializeSpecialDataToJson(tagInfo);
        tagInfo.InsertValue("object_id", objectId);
        request.SetUri("/api/staff/" + ::ToString(type) + "_tags/add").SetPostData(tagInfo.GetStringRobust()).SetRequestType("POST");
        request.SetCgiData("unique_policy=" + ::ToString(policy));
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    bool AddCarTag(NDrive::ITag::TPtr tag, const TString& carNumber, const TString& userActorId, const EUniquePolicy policy = EUniquePolicy::NoUnique) const;

    bool RegisterTag(TTagDescription::TPtr td, const TString& userActorId, bool force = false) {
        NNeh::THttpRequest request;
        NJson::TJsonValue tagInfo = td->SerializeToTableRecord().SerializeToJson();
        request.SetUri(TStringBuilder() << "/service_tags/add?force=" << force).SetPostData(tagInfo.GetStringRobust()).SetRequestType("POST");
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    bool ProposeTagDesc(TTagDescription::TConstPtr td, const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue tagInfo = td->SerializeToTableRecord().SerializeToJson();
        request.SetUri("/api/staff/tag/description/propose").SetPostData(tagInfo).SetRequestType("POST");
        request.AddHeader("Authorization", userActorId);
        request.SetCgiData("comment=propose-" + td->GetName());
        return CommonSendReply(request);
    }

    bool ConfirmTagDesc(TSet<TString> ids, const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue tagInfo;
        tagInfo["proposition_ids"] = NJson::ToJson(ids);
        request.SetUri("/api/staff/tag/description/confirm").SetPostData(tagInfo).SetRequestType("POST");
        request.AddHeader("Authorization", userActorId);
        request.SetCgiData("comment=confirm");
        return CommonSendReply(request);
    }

    bool RejectTagDesc(TSet<TString> ids, const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue tagInfo;
        tagInfo["proposition_ids"] = NJson::ToJson(ids);
        request.SetUri("/api/staff/tag/description/reject").SetPostData(tagInfo).SetRequestType("POST");
        request.AddHeader("Authorization", userActorId);
        request.SetCgiData("comment=reject");
        return CommonSendReply(request);
    }

    auto ListTagDesc(const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue tagInfo;
        request.SetUri("/api/staff/tag/description/list");
        request.AddHeader("Authorization", userActorId);
        return GetSendReply(request);
    }

    bool RemoveTag(const TVector<TString>& tagIds, const TString& userActorId, const NEntityTagsManager::EEntityType type) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/" + ::ToString(type) + "_tags/remove").SetCgiData("tag_id=" + JoinStrings(tagIds, ","));
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    bool RemoveTags(const TVector<TString>& tagNames, const TString& objectId, const TString& userActorId, const NEntityTagsManager::EEntityType type) {
        NNeh::THttpRequest request;
        NJson::TJsonValue jsonData = NJson::JSON_MAP;
        NJson::TJsonValue& jsonIds = jsonData.InsertValue("object_ids", NJson::JSON_ARRAY);
        jsonIds.AppendValue(objectId);
        request.SetUri("/api/staff/" + ::ToString(type) + "_tags/remove").SetCgiData("&tag_names=" + JoinSeq(",", tagNames)).SetPostData(jsonData);
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    bool ViewBillAfterDelegation(const TString& tagId, const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/sessions/state/drop").SetCgiData("tag_id=" + tagId);
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool ActualizeTags(const TVector<NDrive::ITag::TPtr>& tagsAdd, const TVector<TString>& tagNamesRemove, const TString& objectId, const TString& userActorId, const TString& sessionId = {}) {
        NNeh::THttpRequest request;
        NJson::TJsonValue tagInfo;
        NJson::TJsonValue jsonNamesRemove = NJson::JSON_ARRAY;
        for (auto&& i : tagNamesRemove) {
            jsonNamesRemove.AppendValue(i);
        }
        NJson::TJsonValue jsonAddTags = NJson::JSON_ARRAY;
        for (auto&& i : tagsAdd) {
            const IJsonSerializableTag* jsTag = dynamic_cast<const IJsonSerializableTag*>(i.Get());
            UNIT_ASSERT(jsTag);
            jsonAddTags.AppendValue(jsTag->SerializeToJson());
        }
        tagInfo["remove"]["names"] = jsonNamesRemove;
        tagInfo["add"] = jsonAddTags;
        if (sessionId) {
            tagInfo["session_id"] = sessionId;
        }

        request.SetPostData(tagInfo.GetStringRobust());
        request.SetRequestType("POST");
        request.SetUri("/api/yandex/car/actualization").SetCgiData("car_id=" + objectId);
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetLandingsBase(const TString& route, const TString& cgi, const TString& userActorId, const TGeoCoord* userCoord, const TString& platform = "") {
        NNeh::THttpRequest request;
        request.SetUri(route).SetCgiData(cgi);
        request.AddHeader("Authorization", userActorId);
        if (userCoord) {
            request.AddHeader("Lat", ::ToString(userCoord->Y));
            request.AddHeader("Lon", ::ToString(userCoord->X));
        }
        if (platform) {
            request.AddHeader(platform, 103901);
        }
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue GetLandings(const TString& userActorId, const TGeoCoord* userCoord) {
        return GetLandingsBase("/api/yandex/landing/info", "", userActorId, userCoord);
    }

    NJson::TJsonValue GetChatLandings(const TString& userActorId, const TGeoCoord* userCoord, const TString& platform = "") {
        return GetLandingsBase("/api/yandex/landing/info", "report_new_landings=true&format=array", userActorId, userCoord, platform);
    }

    bool AcceptLandings(const TVector<TString>& landings, const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue requestJson;
        {
            NJson::TJsonValue landingIds;
            for (auto&& i : landings) {
                NJson::TJsonValue landing;
                landing["id"] = i;
                landing["comment"] = "aaa";
                landingIds.AppendValue(landing);
            }
            requestJson.InsertValue("landings", landingIds);
        }
        request.SetUri("/api/yandex/landing/accept");
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    bool ClearLandings(const TVector<TString>& landings, const TString& userActorId) {
        NNeh::THttpRequest request;
        if (landings.size()) {
            request.SetUri("/api/staff/landing/clear?id=" + JoinVectorIntoString(landings, ","));
        } else {
            request.SetUri("/api/staff/landing/clear");
        }
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        return result.Code() == 200;
    }

    bool ClearLandingsById(const TVector<TString>& landings, const TString& userActorId, const TString& authId) {
        NNeh::THttpRequest request;
        if (landings.size()) {
            request.SetUri("/api/staff/landing/clear?id=" + JoinVectorIntoString(landings, ",") + "&user_id=" + userActorId);
        } else {
            request.SetUri("/api/staff/landing/clear?user_id=" + userActorId);
        }
        request.AddHeader("Authorization", authId);
        NUtil::THttpReply result = GetSendReply(request);
        return result.Code() == 200;
    }

    NJson::TJsonValue GetLandingsById(const TVector<TString>& landings, const TString& userActorId) {
        NNeh::THttpRequest request;
        if (landings.size()) {
            request.SetUri("/api/staff/landing/get?id=" + JoinVectorIntoString(landings, ","));
        } else {
            request.SetUri("/api/staff/landing/get");
        }
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool ModifyLanding(const TString& landing, const TString& text, bool enabled, const TString& userActorId, const TVector<TString>& geoTags = {},
                const TString& landingChatId = "", const TString& landingChatMessagesGroup = "", const NJson::TJsonValue& landingCondition = NJson::JSON_NULL,
                const bool chatEnabled = true, const TInstant chatDeadline = TInstant::Zero(), const TInstant timestampOverride = TInstant::Zero(), i32 priority = 0, bool checkListCondtions = false) {
        NNeh::THttpRequest request;
        NJson::TJsonValue requestJson;
        requestJson["landing_id"] = landing;
        requestJson["landing_geo_tags"] = JoinSeq(", ", geoTags);
        requestJson["landing_priority"] = priority;
        if (!NJson::ReadJsonFastTree(text, &requestJson["landing_json"]) || !requestJson["landing_json"].IsMap()) {
            return false;
        }
        requestJson["landing_enabled"] = enabled;
        requestJson["landing_check_auditory_condition_in_list"] = checkListCondtions;
        if (!!landingChatId) {
            requestJson["landing_chat_id"] = landingChatId;
            requestJson["landing_chat_messages_group"] = landingChatMessagesGroup;
            if (landingCondition != NJson::JSON_NULL) {
                requestJson["landing_auditory_condition"] = landingCondition;
            }
            requestJson["landing_chat_enabled"] = chatEnabled;
            if (chatDeadline) {
                requestJson["landing_chat_deadline"] = chatDeadline.Seconds();
            }
            if (timestampOverride) {
                requestJson["landing_timestamp_override"] = timestampOverride.Seconds();
            }
        }
        request.SetUri("/api/staff/landing/upsert");
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    NUtil::THttpReply UpsertCar(const TString& requestPayloadString, const TString& userActorId, bool isForce=true) {
        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/car/edit");
        if (isForce) {
            request.SetCgiData("force=true");
        }
        request.SetRequestType("POST").SetPostData(requestPayloadString);
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        return result;
    }

    bool RemoveLandings(const TVector<TString>& landings, const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue requestJson;
        for (const auto& id : landings) {
            requestJson["id"].AppendValue(id);
        }
        request.SetUri("/api/staff/landing/remove");
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    bool SetDefaultCard(const TString& cardId, const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue requestJson;
        requestJson["paymethod_id"] = cardId;
        request.SetUri("/api/yandex/card");
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetCarModels(const TVector<TString>& ids, const TString& userActorId) {
        NNeh::THttpRequest request;
        if (ids.size()) {
            request.SetUri("/api/staff/model/info?id=" + JoinVectorIntoString(ids, ","));
        } else {
            request.SetUri("/api/staff/model/info");
        }
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool RemoveCarModels(const TVector<TString>& ids, const TString& userActorId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/model/remove");
        NJson::TJsonValue requestJson(NJson::JSON_MAP);
        for (const auto& id : ids) {
            requestJson["id"].AppendValue(id);
        }
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        return result.Code() == 200;
    }

    bool ModifyCarModel(const TString& code, const TString& name, const TString& manufacturer, const TString& userActorId, NJson::TJsonValue requestJson = NJson::JSON_MAP) {
        NNeh::THttpRequest request;
        requestJson["code"] = code;
        requestJson["name"] = name;
        requestJson["manufacturer"] = manufacturer;
        request.SetUri("/api/staff/model/info");
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue AddCarModelSpecification(const TString& model_id, const TString& name, const TString& value, const ui32 position, const TString& userActorId, const TString& id = {}) {
        NNeh::THttpRequest request;
        NJson::TJsonValue requestJson;
        if (id) {
            requestJson["id"] = id;
        }
        requestJson["model_id"] = model_id;
        requestJson["name"] = name;
        requestJson["value"] = value;
        requestJson["position"] = position;
        request.SetUri("/api/staff/model_spec/add");
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool RemoveCarModelSpecification(const TVector<TString>& ids, const TString& userActorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue requestJson;
        for (const auto& id : ids) {
            requestJson["id"].AppendValue(id);
        }
        request.SetUri("/api/staff/model_spec/remove");
        request.SetRequestType("POST").SetPostData(requestJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue ListMajorQuery(const TString& userActorId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/major/get");
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool CancelMajorQuery(const TString& tagId, const TString& userActorId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/major/cancel?id=" + tagId);
        request.AddHeader("Authorization", userActorId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue ListTags(const TString& objectId, const TString& userActorId, const NEntityTagsManager::EEntityType type) {
        NNeh::THttpRequest request;
        TString entity;
        switch (type) {
        case NEntityTagsManager::EEntityType::Car:
            entity = "car_tags";
            break;
        case NEntityTagsManager::EEntityType::User:
            entity = "user_tags";
            break;
        case NEntityTagsManager::EEntityType::Trace:
            entity = "trace_tags";
            break;
        case NEntityTagsManager::EEntityType::Account:
            entity = "account_tags";
            break;
        default:
            ythrow yexception() << "type " << type << " is not supported";
        }
        request.SetUri("/api/staff/" + entity + "/list").SetCgiData("object_id=" + objectId);
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool ProposeTag(ITag::TPtr tag, const TString& objectId, const TString& userActorId, const NEntityTagsManager::EEntityType type) {
        NNeh::THttpRequest request;
        request.SetUri(::ToString(type) + "/tags/propositions/propose").SetCgiData("object_id=" + objectId);
        request.SetPostData(tag->SerializeToJson().GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return false;
        }
        return true;
    }

    bool RejectTag(const TString& propositionId, const TString& userActorId, const NEntityTagsManager::EEntityType type) {
        NNeh::THttpRequest request;
        request.SetUri(::ToString(type) + "/tags/propositions/reject");
        NJson::TJsonValue propositionsJson;
        propositionsJson["proposition_ids"].AppendValue(propositionId);
        request.SetPostData(propositionsJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return false;
        }
        return true;
    }

    bool ConfirmTag(const TString& propositionId, const TString& userActorId, const NEntityTagsManager::EEntityType type) {
        NNeh::THttpRequest request;
        request.SetUri(::ToString(type) + "/tags/propositions/confirm");
        NJson::TJsonValue propositionsJson;
        propositionsJson["proposition_ids"].AppendValue(propositionId);
        request.SetPostData(propositionsJson.GetStringRobust());
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return false;
        }
        return true;
    }

    bool GetTagId(const TString& objectId, const TString& tagName, const TString& userId, const NEntityTagsManager::EEntityType type, TString& result) {
        NJson::TJsonValue report = ListTags(objectId, userId, type);
        if (report.IsNull()) {
            return false;
        }
        for (auto&& i : report["records"].GetArraySafe()) {
            UNIT_ASSERT(i.IsMap());
            if (i["tag"].GetString() == tagName) {
                result = i["tag_id"].GetString();
                return true;
            }
        }
        return false;
    }

    bool CarRootControl(const TString& objectId, const TString& userId, const TString& command) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/car/control").SetCgiData("car_id=" + objectId + "&command=" + command);
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool StartTag(const TString& tagId, const TString& userId) const {
        return StartTag(TSet<TString>({ tagId }), userId);
    }

    bool StartTag(const TSet<TString>& tagIds, const TString& userId) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        for (auto&& i : tagIds) {
            post["tag_id"].AppendValue(i);
        }
        request.SetUri("/service_app/servicing/start").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool StartTag(const TString& carId, const TSet<TString>& tagNames, const TString& userId) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        for (auto&& i : tagNames) {
            post["tag_name"].AppendValue(i);
        }
        post["car_id"] = carId;
        request.SetUri("/service_app/servicing/start").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool FinishTag(const TString& tagId, const TString& userId, const bool removeOnFinish = true) const {
        NNeh::THttpRequest request;

        NJson::TJsonValue post;
        if (removeOnFinish) {
            post["tag_id"].AppendValue(tagId);
        } else {
            post["drop_tag_ids"].AppendValue(tagId);
        }
        request.SetUri("/service_app/servicing/finish").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");

        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue DoSearch(const TVector<TString> required, const TVector<TString> optional, const TVector<TString> what, int limit = 20, const TString& userId = USER_ROOT_DEFAULT) {
        TStringStream ss;
        if (required.size()) {
            ss << "has_all_of=" << FormMultipleSearchParams(required);
        }
        if (optional.size()) {
            ss << "&has_one_of=" << FormMultipleSearchParams(optional);
        }
        ss << "&what=" << FormMultipleSearchParams(what);
        ss << "&limit=" << limit;
        NNeh::THttpRequest request;
        request.SetUri("/service_app/search").SetCgiData(ss.Str());
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultJson;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue GetUserAdminInfo(const TString& userId, const TString& operatorId) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/users/info");
        request.SetCgiData("user_id=" + userId);
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool UpsertLocalization(const NLocalization::TResource& resource, const TString& userId) const {
        NJson::TJsonValue requestBody = resource.SerializeToJson();
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/localization/upsert");
        request.SetRequestType("POST").SetPostData(requestBody.GetStringRobust());
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool RemoveLocalization(const TString& id, const TString& userId) const {
        NJson::TJsonValue requestBody;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/localization/remove");
        auto& arr = requestBody.InsertValue("id", NJson::JSON_ARRAY);
        arr.AppendValue(id);
        request.SetRequestType("POST").SetPostData(requestBody.GetStringRobust());
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool GetLocalizations(const TString& userId, TVector<NLocalization::TResource>& resultResources, const TVector<TString>& ids = {}) const {
        resultResources.clear();
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/localization/info");
        if (!ids.empty()) {
            request.SetCgiData("id=" + JoinStrings(ids.begin(), ids.end(), ","));
        }
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        const NJson::TJsonValue::TArray* arr;
        if (!resultJson["resources"].GetArrayPointer(&arr)) {
            return false;
        }
        for (auto&& i : *arr) {
            NLocalization::TResource res;
            if (!res.DeserializeFromJson(i)) {
                return false;
            }
            resultResources.emplace_back(std::move(res));
        }
        return true;
    }

    bool UpdateUserInfo(const TString& userId, NJson::TJsonValue requestBody, const TString& operatorId) const {
        requestBody["id"] = userId;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/users/edit");
        request.SetRequestType("POST").SetPostData(requestBody.GetStringRobust());
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    bool UpsertRTBackground(const TRTBackgroundProcessContainer& container, const TString& userId) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue post = NJson::JSON_MAP;
        NJson::TJsonValue& bg = post.InsertValue("backgrounds", NJson::JSON_ARRAY);
        bg.AppendValue(container.SerializeToTableRecord().SerializeToJson());

        request.SetUri("/api/staff/bg/upsert").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool ForceUpsertRTBackground(const TRTBackgroundProcessContainer& container, const TString& userId) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue post = NJson::JSON_MAP;
        NJson::TJsonValue& bg = post.InsertValue("backgrounds", NJson::JSON_ARRAY);
        bg.AppendValue(container.SerializeToTableRecord().SerializeToJson());

        request.SetUri("/api/staff/bg/upsert").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST").SetCgiData("force=true");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool RemoveRTBackground(const TString& id, const TString& userId) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue post = NJson::JSON_MAP;
        NJson::TJsonValue& bg = post.InsertValue("ids", NJson::JSON_ARRAY);
        bg.AppendValue(id);

        request.SetUri("/api/staff/bg/remove").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetRTBackgrounds(const TString& userId, const TSet<TString>& robotIds) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/bg/info");
        if (!robotIds.empty()) {
            request.SetCgiData("ids=" + JoinStrings(robotIds.begin(), robotIds.end(), ","));
        }
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue GetUserEditHistory(const TString& userId, const TString& operatorId) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/users/history");
        request.SetCgiData("user_id=" + userId);
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    TString GetDocumentPhoto(const TString& photoId, const TString& userId) const {
        auto photoIds = SplitString(photoId, ",");

        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/user/document-photo");
        request.SetCgiData("photo_id=" + photoIds[0]);
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);

        INFO_LOG << "User " << userId << " document photo " << photoId << " content: " << result.Content() << Endl;

        return result.Content();
    }

    NJson::TJsonValue GetPromoCodes(const TString& prefix = "", const TString& userId = "") const {
        TEnvironmentGenerator::TPromoCodeFilter filter;
        if (prefix) {
            filter.Prefix = prefix;
        }
        if (userId) {
            filter.GivenOut = userId;
        }
        filter.ActiveOnly = true;
        return GetPromoCodes(filter);
    }

    NJson::TJsonValue GetPromoCodes(const TEnvironmentGenerator::TPromoCodeFilter& filter) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/promo/get");
        TStringBuilder cgiData;
        cgiData << "with_code=true";
        if (filter.ActiveOnly) {
            cgiData << "&active_only=true";
        }
        if (filter.Prefix) {
            cgiData << "&prefix=" << filter.Prefix.GetRef();
        }
        if (filter.GivenOut) {
            cgiData << "&given_out=" << filter.GivenOut.GetRef();
        }
        if (filter.Generator) {
            cgiData << "&generator=" << filter.Generator.GetRef();
        }
        if (filter.Since) {
            cgiData << "&since=" << filter.Since->Seconds();
        }
        if (filter.Until) {
            cgiData << "&until=" << filter.Until->Seconds();
        }
        if (!filter.Ids.empty()) {
            cgiData << "&ids=" << JoinSeq(",", filter.Ids);
        }
        if (filter.Count) {
            cgiData << "&count=" << filter.Count.GetRef();
        }
        request.SetCgiData(cgiData);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue GeneratePromoCodes(const TString& action, const ui32 count, const TString& prefix, const TString& userId = "", const TInstant activityDeadline = TInstant::Zero()) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/promo/generation");
        request.SetCgiData(TStringBuilder() << "generator=" << action << "&count=" << count);
        NJson::TJsonValue postData;
        NJson::InsertField(postData, "prefix", prefix);
        NJson::InsertNonNull(postData, "given_out", userId);
        if (activityDeadline) {
            NJson::InsertField(postData, "activity_deadline", activityDeadline.Seconds());
        }
        request.SetPostData(postData);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue CheckPromoCode(const TString& code) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/promo/check");
        request.SetCgiData("code=" + code);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool RemovePromoCodes(const TVector<TString>& ids) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/promo/remove");
        NJson::TJsonValue postData;
        NJson::InsertField(postData, "ids", ids);
        request.SetPostData(postData);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GiveOutPromoCode(const TString& userId, const TVector<TString>& ids, const TString& generator = "", const ui32 count = 0) const {
        UNIT_ASSERT(userId && (!ids.empty() || (generator && count)));
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/promo/give_out");
        NJson::TJsonValue postData;
        NJson::InsertField(postData, "given_out_info", userId);
        NJson::InsertNonNull(postData, "ids", ids);
        NJson::InsertNonNull(postData, "generator", generator);
        NJson::InsertNonNull(postData, "count", count);
        request.SetPostData(postData);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool AcceptPromoCode(const TString& code, const TString& userId, const TString& objectId = "") const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/user_app/promo/accept");
        request.SetCgiData("code=" + code);
        if (objectId) {
            request.AddCgiData("&object_id=" + objectId);
        }
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool CreateReferralCode(const TString& userId, const TString& code = "") const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/user_app/promo/create_referral_code");
        if (code) {
            request.SetPostData(NJson::TMapBuilder("custom_code", code));
        }
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetPromoCodeHistory(const TString& userId) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/promo/user_history");
        request.SetCgiData("user_id=" + userId);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue GetPromoInfo(const TString& userId) const {
        NJson::TJsonValue resultJson;
        NUtil::THttpReply result = GetSendReply(NNeh::THttpRequest().SetUri("/user_app/promo/referral").AddHeader("Authorization", userId));
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    TMaybe<TString> GetPersonalSetting(const TString& userId, const TString& key) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/user/settings/get");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return {};
        }
        UNIT_ASSERT_C(resultJson.IsDefined() && resultJson.Has("settings") && resultJson["settings"].IsArray(), "Fail to parse response: " + resultJson.GetStringRobust());
        for (auto&& item : resultJson["settings"].GetArray()) {
            if (item.Has("id") && item.Has("value") && item["id"].GetStringRobust() == key) {
                return item["value"].GetStringRobust();
            }
        }
        WARNING_LOG << "Fail to find '" << key << "' in " << resultJson << Endl;
        return {};
    }

    NJson::TJsonValue GetFuelingInfo(const TString& userId, const TString& cgiParams = "", const bool assert = true) const {
        NNeh::THttpRequest request;
        request.SetUri("/user_app/fueling/info");
        request.AddHeader("Authorization", userId);
        if (!!cgiParams) {
            request.SetCgiData(cgiParams);
        }
        NJson::TJsonValue resultJson;
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            resultJson = NJson::JSON_NULL;
        }
        if (assert) {
            UNIT_ASSERT_C(resultJson.IsDefined(), TStringBuilder() << result.Code() << " : " << result.Content() << Endl);
        }
        return resultJson;
    }

    NJson::TJsonValue GetFuelingMap(const TString& userId) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/user_app/fueling/map");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool StartFueling(const TString& userId, const TString& columnId, const bool tanker = false, const TString& patchName = "") const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        TString cgiData = "column_id=" + columnId;
        if (tanker) {
            cgiData += "&tanker=true";
        }
        if (patchName) {
            cgiData += "&tag_name=" + patchName;
        }
        request.SetUri("/user_app/fueling/start").SetCgiData(cgiData);
        request.AddHeader("Authorization", userId);
        TInstant start = Now();
        while (Now() - start < TDuration::Minutes(2)) {
            const ui32 code = GetSendReply(request).Code();
            if (code == 200) {
                return true;
            }
            if (code / 100 == 4) {
                return false;
            }
            WARNING_LOG << "repeat for start fueling" << Endl;
            Sleep(TDuration::Seconds(5));
        }
        return false;
    }

    bool CancelFueling(const TString& userId) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/user_app/fueling/cancel").SetCgiData("reason=отмена");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    struct CustomOrder {
        TString userId;
        TString carId;
        TString stationId;
        TString columnId;
        TString liters;
        TString fuelType;
    };

    bool SetCustomOrder(const CustomOrder& customOrder) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/support/fueling/custom/start").SetCgiData("reason=отмена");
        NJson::TJsonValue postData;
        postData.InsertValue("user_id", customOrder.userId);
        postData.InsertValue("car_id", customOrder.carId);
        postData.InsertValue("station_id", customOrder.stationId);
        postData.InsertValue("column_id", customOrder.columnId);
        postData.InsertValue("liters", customOrder.liters);
        postData.InsertValue("fuel_type", customOrder.fuelType);
        request.SetPostData(postData);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetUserSessions(const TString& userId, const TString& sessionId = {}) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/user_app/sessions/history");
        if (sessionId) {
            request.SetCgiData("session_id=" + sessionId);
        }
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue GetUserRoles(const TString& userId, const TString& actorId) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/user_roles/list").SetCgiData("user_id=" + userId);
        request.AddHeader("Authorization", actorId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool AddUserRole(const TUserRole& role, const TString& userId) const {
        const NJson::TJsonValue resultJson = role.SerializeToTableRecord().SerializeToJson();
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/user_roles/add");
        request.AddHeader("Authorization", userId).SetRequestType("POST").SetPostData(resultJson.GetStringRobust());
        return CommonSendReply(request);
    }

    bool RegisterRole(const TDriveRoleHeader& role, const TString& userId) const {
        const NJson::TJsonValue resultJson = role.SerializeToTableRecord().SerializeToJson();
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/roles/add");
        request.AddHeader("Authorization", userId).SetRequestType("POST").SetPostData(resultJson.GetStringRobust());
        return CommonSendReply(request);
    }

    bool RoleActivation(const TString& roleId, const bool enabled, const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/my/roles/activation").SetCgiData("role_id=" + roleId + "&enabled=" + ::ToString(enabled));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue DoTagsSearch(const TVector<TString>& hasAllOf, const TVector<TString>& hasNoneOf) const {
        TString cgiData;
        cgiData = "has_all_of=" + JoinStrings(hasAllOf, ",");
        if (hasNoneOf) {
            cgiData += "&has_none_of=" + JoinStrings(hasNoneOf, ",");
        }

        NNeh::THttpRequest request;
        request.SetUri("/api/staff/tags/search").SetCgiData(cgiData);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);

        NJson::TJsonValue resultJson;
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue GetUserLastSession(const TString& userId) const;

    NJson::TJsonValue GetUserSessionById(const TString& sessionId, const TString& userId) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/user_sessions").SetCgiData("session_id=" + sessionId);
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool RegisterPhoto(const TString& userId, const TString& tagId, const TString& id) const {
        NJson::TJsonValue postData;
        postData["marker"] = "test";
        postData["uuid"] = id;
        postData["tag_ids"].AppendValue(tagId);

        NNeh::THttpRequest request;
        request.SetUri("/service_app/tag/photos/register");
        request.SetRequestType("POST").SetPostData(postData.GetStringRobust());
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue BindEmail(const TString& userId, const TString& email) const {
        NJson::TJsonValue postData;
        postData["email"] = email;

        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/user/email");
        request.SetRequestType("POST").SetPostData(postData.GetStringRobust());
        request.AddHeader("Authorization", userId);

        NJson::TJsonValue resultJson;
        NUtil::THttpReply result = GetSendReply(request);
        NJson::ReadJsonFastTree(result.Content(), &resultJson);
        INFO_LOG << "bind email endpoint reply: " << result.Code() << " : " << result.Content() << Endl;
        return resultJson;
    }

    NJson::TJsonValue GetCurrentSession(const TString& userId, const TGeoCoord* c = nullptr, const TString* deviceId = nullptr, bool multiSession= false) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/sessions/current");
        request.AddHeader("Authorization", userId);
        if (multiSession) {
            request.SetCgiData("&need_bill=1&multi_sessions=1");
        } else {
            request.SetCgiData("&need_bill=1");
        }
        if (deviceId) {
            request.AddHeader("DeviceId", *deviceId);
        }
        if (c) {
            request.AddHeader("Lat", ::ToString(c->Y));
            request.AddHeader("Lon", ::ToString(c->X));
        }
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool DropSession(const TString& sessionId, const TString& userId) const;

    NJson::TJsonValue GetAdminCarsList(const TString& userId) const {
        return GetCarsList("/api/staff/car/list", {}, userId, "");
    }

    NJson::TJsonValue GetUserCarsListClient(const TVector<TString>& cars, const TString& userId, const TGeoCoord& c) const {
        return GetCarsList("/user_app/car/list", cars, userId, "", &c);
    }

    NJson::TJsonValue GetUserCarsList(const TVector<TString>& cars, const TString& userId, const TString& tagsFilter = "", const TString& additionalCgi = "") const {
        return GetCarsList("/user_app/car/list", cars, userId, tagsFilter, nullptr, additionalCgi);
    }

    NJson::TJsonValue GetServiceCarsList(const TVector<TString>& cars, const TString& userId, const TString& tagsFilter = "", const TGeoCoord* c = nullptr, const ui32 preLimit = Max<ui32>()) const {
        TStringStream ss;
        if (preLimit != Max<ui32>()) {
            ss << "&pre_limit=" << preLimit;
        }
        ss << "&no_clusters=1";
        return GetCarsList("/service_app/car/list", cars, userId, tagsFilter, c, ss.Str());
    }

    NJson::TJsonValue GetCachedServiceCarsList(const TVector<TString>& cars, const TString& userId, const TString& tagsFilter = "") const {
        return GetCarsList("/service_app/cached/car/list", cars, userId, tagsFilter);
    }

    NJson::TJsonValue GetServiceCarsDetails(const TVector<TString>& cars, const TString& userId, const TString& tagsFilter = "", const ui32 cluster = 0) const {
        return GetCarsList("/service_app/car/details", cars, userId, tagsFilter, nullptr, "", cluster);
    }

    NJson::TJsonValue GetCarsList(const TString& handler, const TVector<TString>& cars, const TString& userId, const TString& tagsFilter = "", const TGeoCoord* c = nullptr, const TString& additionalCgi = "", const ui32 cluster = 0) const {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri(handler);

        TString requestCgiData = "report=service_app_details&" + additionalCgi;
        if (cars.size()) {
            requestCgiData += "&car_id=" + JoinSeq(",", cars);
        }
        if (tagsFilter.size()) {
            requestCgiData += "&tags_filter=" + CGIEscapeRet(tagsFilter);
        }
        if (cluster) {
            requestCgiData += "&cluster=" + ::ToString(cluster);
        }
        if (requestCgiData.size()) {
           request.SetCgiData(requestCgiData);
        }

        request.AddHeader("Authorization", userId);
        if (c) {
            request.AddHeader("Lon", ::ToString(c->X));
            request.AddHeader("Lat", ::ToString(c->Y));
        }
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    NJson::TJsonValue GetCarModificationHistory(const TString& objectId, const TString& userActorId) {
        NJson::TJsonValue resultJson;
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/car/history");
        request.SetCgiData(TString("car_id=") + objectId);
        request.AddHeader("Authorization", userActorId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultJson)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return NJson::JSON_NULL;
        }
        return resultJson;
    }

    bool StartScanner(const TString& userId, const TGeoCoord& c, const TString& filters = "", const TDuration walkTime = TDuration::Seconds(1000), const TString& action = "order") const {
        NNeh::THttpRequest request;
        {
            NJson::TJsonValue scannerSettings;
            scannerSettings.InsertValue("walking_time", walkTime.Seconds());
            scannerSettings.InsertValue("livetime", 1000);
            scannerSettings.InsertValue("action", action);
            scannerSettings.InsertValue("lon", c.X);
            scannerSettings.InsertValue("lat", c.Y);
            if (!!filters) {
                scannerSettings.InsertValue("filter", filters);
            }

            request.AddHeader("Lon", ::ToString(c.X));
            request.AddHeader("Lat", ::ToString(c.Y));
            request.SetUri("/user_app/scanner/start").SetPostData(TBlob::FromString(scannerSettings.GetStringRobust())).SetRequestType("POST");
        }
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool StopScanner(const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/user_app/scanner/stop");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetActions(const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/actions/list");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool AddAction(const TUserAction& userAction, const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/actions/add");
        request.SetRequestType("POST").SetPostData(userAction.SerializeToTableRow().SerializeToJson().GetStringRobust());
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool GetConstants(const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/constants");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool ProposeAction(const TUserAction& userAction, const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/actions/propose");
        request.SetCgiData("comment=sdfsdfsd");
        auto jsonInfo = userAction.SerializeToTableRow().SerializeToJson();
        request.SetRequestType("POST").SetPostData(jsonInfo.GetStringRobust());
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool ConfirmActionProposition(const TString& propositionId, const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/actions/confirm");
        NJson::TJsonValue requestData;
        requestData["proposition_ids"].AppendValue(propositionId);
        request.SetRequestType("POST").SetPostData(requestData.GetStringRobust());
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool RejectActionProposition(const TString& propositionId, const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/actions/reject");
        NJson::TJsonValue requestData;
        requestData["proposition_ids"].AppendValue(propositionId);
        request.SetRequestType("POST").SetPostData(requestData.GetStringRobust());
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool EvolveTag(const TString& tagTargetName, const TString& userId, const TDuration timeout = TDuration::Seconds(60), const TString& landingId = Default<TString>(), const TMaybe<EEvolutionMode> evolutionMode = {}, NJson::TJsonValue post = NJson::JSON_MAP, const TString& sessionId = "") const;

    bool BookOffer(const TString& offerId, const TString& userId, const TString* deviceId, const TString& payloadPatch) const;
    bool BookOffer(const TString& offerId, const TString& userId, const TString* deviceId = nullptr, const NJson::TJsonValue& baseBody = NJson::JSON_NULL) const;

    bool SwitchOffer(const TString& offerId, const TString& userId, const TString* deviceId = nullptr) const;

    TString CreateOffer(const TString& carId, const TString& userId, NJson::TJsonValue* report = nullptr, const ui32 version = 0, TMaybe<TGeoCoord> userDestination = {}, TMaybe<TGeoCoord> userPosition = {}, const TString* deviceId = nullptr, const TString& offerName = "", class TInstant deadline = TInstant::Max(), const TString& defaultCard = "") const {
        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/offers/create");
        if (carId) {
            request.SetCgiData("car_id=" + carId);
        }
        if (!!offerName) {
            request.AddCgiData("&offer_name=" + offerName);
        }
        if (!!userDestination) {
            TString coord = userDestination->ToString();
            UrlEscape(coord);
            request.AddCgiData("&user_destination=" + coord);
        }
        if (!!userPosition) {
            request.AddHeader("Lon", userPosition->X);
            request.AddHeader("Lat", userPosition->Y);
        }
        if (deviceId) {
            request.AddHeader("DeviceId", *deviceId);
        }
        request.AddHeader("Authorization", userId);
        if (version) {
            request.AddHeader("TEST_Version", ::ToString(version));
        }

        NJson::TJsonValue paymentMethods;
        if (defaultCard) {
            NJson::TJsonValue paymethod;
            paymethod.InsertValue("account_id", "card");
            paymethod.InsertValue("card", defaultCard);

            paymentMethods.AppendValue(paymethod);
        }
        if (paymentMethods.GetArray().size()) {
            request.SetPostData(NJson::TMapBuilder("payment_methods", paymentMethods));
        }

        NUtil::THttpReply result = GetSendReply(request, deadline);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return "";
        }
        if (report) {
            *report = resultReport;
        }

        return resultReport["offers"][0]["offer_id"].GetString();
    }

    bool UpsertArea(const TString& areaId, const TString& userId, const TVector<TGeoCoord>& coords, const TVector<TString>& areaTags, const TString& type = {}, const TMaybe<i64> revision = {}) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        post["areas"][0]["area_id"] = areaId;
        post["areas"][0]["area_coords"] = TGeoCoord::SerializeVector(coords);
        post["areas"][0]["area_tags"] = JoinSeq(",", areaTags);
        if (revision) {
            post["areas"][0]["revision"] = *revision;
        }
        if (type) {
            post["areas"][0]["area_type"] = type;
        }
        request.SetUri("/api/staff/areas/upsert").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);

        return CommonSendReply(request);
    }

    bool RemoveArea(const TString& areaId, const TString& userId) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        post["areas"][0]["area_id"] = areaId;
        request.SetUri("/api/staff/areas/remove").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);

        return CommonSendReply(request);
    }

    TMaybe<i64> AreaRevision(const TString& areaId, const TString& userId) const {
        try {
            NNeh::THttpRequest request;
            NJson::TJsonValue post;
            request.SetUri("/api/staff/areas/info").SetCgiData("ids=" + areaId);
            request.AddHeader("Authorization", userId);

            NUtil::THttpReply reply = GetSendReply(request);
            NJson::TJsonValue reportJson;
            if (reply.Code() != 200 || !NJson::ReadJsonFastTree(reply.Content(), &reportJson) || reportJson["areas"].GetArraySafe().size() != 1) {
                return {};
            }
            return reportJson["areas"].GetArraySafe()[0]["revision"].GetUIntegerSafe();
        } catch (const std::exception& e) {
            ERROR_LOG << FormatExc(e) << Endl;
            return {};
        }
    }

    bool UpsertNotifier(NJson::TJsonValue& notifierDesc, const TString& userId) const {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/notifiers/upsert").SetPostData(
                TBlob::FromString(notifierDesc.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool ConfirmNotifier(const TString& propositionId, const TString& userId) const {
        NNeh::THttpRequest request;
        NJson::TJsonValue requestData;
        requestData["proposition_ids"] = {propositionId};
        request
            .SetUri("/api/staff/notifiers/confirm")
            .SetPostData(TBlob::FromString(requestData.GetStringRobust()))
            .SetCgiData("comment=foo")
            .SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool ProposeNotifier(NJson::TJsonValue& notifierDesc, const TString& userId) const {
        NNeh::THttpRequest request;
        request
            .SetUri("/api/staff/notifiers/propose")
            .SetPostData(TBlob::FromString(notifierDesc.GetStringRobust()))
            .SetRequestType("POST")
            .SetCgiData("comment=bar");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue ListNotifierPropositions(const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/notifiers/propositions").SetRequestType("GET");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue NotifiersInfo(const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/notifiers/info").SetRequestType("GET");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool RegisterComplaintPhoto(const TString& userId, const TString& car_id, const TString& chat_id, const TString& photoId, const TString& md5Str, const TString& marker, const TString& photoOrigin, ui64 complaintWeight, bool lostItem = false) {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;

        post["complaint_tag"] = "car_complaint";
        post["complaint_weight"] = complaintWeight;
        post["complaint_id"] = "car_complaint";
        post["origin"] = photoOrigin;

        post["tag_name"] = "car_complaint";
        post["car_id"] = car_id;
        post["comment"] = "";
        post["chat_id"] = chat_id;

        post["photos"][0]["marker"] = marker;
        post["photos"][0]["uuid"] = photoId;
        post["photos"][0]["md5"] = md5Str;
        post["photos"][0]["origin"] = photoOrigin;

        if (lostItem) {
            NJson::TJsonValue lostItem;
            lostItem["type"] = "forgotten";
            lostItem["description"] = "passport";
            lostItem["document"] = "true";
            post["lost_item"] = lostItem;
        }
        post["location"] = "somewhere";
        post["is_user_available"] = "true";

        auto string = post.GetStringRobust();

        request.SetUri("/api/service/photos/register").SetPostData(TBlob::FromString(string)).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetDocumentCheck(const TString& userId, const TString& documentType) {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;

        post["user_id"] = userId;
        post["types"] = NJson::EJsonValueType::JSON_ARRAY;
        post["types"][0] = documentType;

        auto string = post.GetStringRobust();

        request.SetUri("/api/support_staff/documents_checks/get").SetPostData(TBlob::FromString(string)).SetRequestType("POST");
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue GetDocumentChecksHistory(ui64 sinceEventId, const TString& userId, const TString& documentType) {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;

        post["user_id"] = userId;
        post["type"] = documentType;
        post["since_event_id"] = sinceEventId;

        auto string = post.GetStringRobust();

        request.SetUri("/api/support_staff/documents_checks/get_history").SetPostData(TBlob::FromString(string)).SetRequestType("POST");
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool RegisterPhotos(const TString& userId, const TString& photoId, const TString& tagId, const TString& md5Str, const TString& marker, NJson::TJsonValue post = NJson::JSON_MAP) {
        NNeh::THttpRequest request;
        post["tag_ids"].AppendValue(tagId);
        post["marker"] = marker;
        post["uuid"] = photoId;
        post["md5"] = md5Str;
        request.SetUri("/api/service/photos/register").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool UploadPhotos(const TString& userId, const TString& objectId, const TString& photoId, TBlob data) {
        NNeh::THttpRequest request;
        request.SetUri("/api/service/photos/upload").SetPostData(data).SetRequestType("POST");
        request.SetCgiData(TString("car_id=") + objectId + "&photo_id=" + photoId);
        request.AddHeader("Authorization", userId);
        request.AddHeader("Content-Type", "image/jpeg");
        return CommonSendReply(request);
    }

    NJson::TJsonValue ListAttachedHardware(const TString& carId, const TString& operatorId, bool viewAsAdmin=false) {
        NNeh::THttpRequest request;
        if (!viewAsAdmin) {
            request.SetUri("/service_app/attachments/list").SetCgiData("car_id=" + carId);
        } else {
            request.SetUri("/api/staff/car/attachments/list").SetCgiData("car_id=" + carId);
        }
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport["attachments"];
    }

    NJson::TJsonValue ServiceAppAssignDevice(const TString& carId, const TString& deviceCode, const bool isForce, const TString& operatorId) {
        NNeh::THttpRequest request;

        TString cgiData = "car_id=" + carId;
        if (isForce) {
            cgiData += "&force=da";
        }
        NJson::TJsonValue postData;
        postData["device_code"] = deviceCode;

        request.SetUri("/service_app/attachments/assign").SetCgiData(cgiData).SetPostData(TBlob::FromString(postData.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if ((result.Code() != 200 && result.Code() != 400) || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue ServiceAppAssignmentHistory(const TString& carId, const TString& operatorId) {
        NNeh::THttpRequest request;
        TString cgiData = "car_id=" + carId;
        request.SetUri("/service_app/attachments/history").SetCgiData(cgiData).SetRequestType("GET");
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool DetachAttachment(const TString& id, const TString& operatorId) {
        NNeh::THttpRequest request;
        TString cgiData = "attachment_id=" + id;
        request.SetUri("/api/staff/car/attachments/unassign").SetCgiData(cgiData).SetPostData(TString("{\"a\": \"b\"}")).SetRequestType("POST");
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return false;
        }
        return true;
    }

    bool BatchUploadCars(const NJson::TJsonValue& payload, const TString& operatorId) {
        NNeh::THttpRequest request;

        request.SetUri("/api/staff/car/batch-upload").SetPostData(TBlob::FromString(payload.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return false;
        }
        return true;
    }

    bool BatchUploadInsurance(const NJson::TJsonValue& payload, const TString& operatorId) {
        NNeh::THttpRequest request;

        request.SetUri("/api/staff/car/insurance/batch-upload").SetPostData(TBlob::FromString(payload.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return false;
        }
        return true;
    }

    NJson::TJsonValue GetChatMessages(const TString& userId, const TString& chatId, const TString operatorId = "") {
        NNeh::THttpRequest request;
        TString cgi = "chat_id=" + chatId;
        if (operatorId) {
            cgi += "&user_id=" + userId;
        }
        request.SetUri("/api/yandex/chat/history").SetCgiData(cgi);
        request.AddHeader("Authorization", operatorId ? operatorId : userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue GetChatsList(const TString& userId, const NDrive::IServer* server, const TString& operatorId = "", const TString& platform = "") {
        server->GetChatRobot("support")->Refresh(Now());
        UNIT_ASSERT(server->GetChatEngine()->GetChats().RefreshCache(Now()));
        NNeh::THttpRequest request;
        auto requesterId = operatorId ? operatorId : userId;
        if (operatorId) {
            request.SetUri("/api/yandex/chats/list").SetCgiData("user_id=" + userId);
        } else {
            request.SetUri("/api/yandex/chats/list");
        }
        if (platform) {
            request.AddHeader(platform, 103901);
        }
        request.AddHeader("Authorization", requesterId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool MarkMessagesRead(const TString& userId, const TString& chatId, ui64 messageId) {
        NNeh::THttpRequest request;
        request.SetUri("api/yandex/chats/read").SetCgiData("chat_id=" + chatId + "&message_id=" + ::ToString(messageId));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool PhoneBindSubmit(const TString& userId, const TString& phone) {
        NNeh::THttpRequest request;

        NJson::TJsonValue payload;
        payload["phone"] = phone;

        request.SetUri("/support_api/registration/user/phone/submit").SetPostData(payload.GetStringRobust());
        request.AddHeader("Authorization", userId);
        request.AddHeader("DeviceId", userId);
        return CommonSendReply(request);
    }

    bool PhoneBindCommit(const TString& userId, const TString& code) {
        NNeh::THttpRequest request;

        NJson::TJsonValue payload;
        payload["code"] = code;

        request.SetUri("/support_api/registration/user/phone/commit").SetPostData(payload.GetStringRobust());
        request.AddHeader("Authorization", userId);
        request.AddHeader("DeviceId", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetAdminChatsList(const TVector<std::pair<TString, TString>>& chats) {
        NNeh::THttpRequest request;

        NJson::TJsonValue chatRequestList = NJson::JSON_ARRAY;
        for (auto&& chat : chats) {
            NJson::TJsonValue req;
            req["user_id"] = chat.first;
            req["chat_id"] = chat.second;
            chatRequestList.AppendValue(std::move(req));
        }
        NJson::TJsonValue payload;
        payload["chats"] = std::move(chatRequestList);

        request.SetUri("/api/yandex/chats/list").SetPostData(TBlob::FromString(payload.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    static TString GetValidImageString() {
        TStringStream ss;
        ss << "/9j/4AAQSkZJRgABAQEAYABgAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRofHh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/";
        ss << "2wBDAQkJCQwLDBgNDRgyIRwhMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjL/";
        ss << "wAARCAABAAEDASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAA";
        ss << "F9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd";
        ss << "4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBA";
        ss << "QEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREAAgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMz";
        ss << "UvAVYnLRChYkNOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOEhYaHiImKkpO";
        ss << "UlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwD3+iiigD//2Q==";
        return ss.Str();
    }

    bool CommitChatAction(const TString& userId, const TString& chatId, const TString& message, const TVector<TString>& attachments, const double lat = 0.0, const double lon = 0.0, const TVector<NDrive::NChat::TMessage::EMessageTraits>& traits = {}, const TString& operatorId = "", const TMap<TString, TString>& headers = {}) {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        post["message"] = message;
        {
            NJson::TJsonValue attachmentsJson = NJson::JSON_ARRAY;
            for (auto&& attachment : attachments) {
                attachmentsJson.AppendValue(attachment);
            }
            post["attachments"] = std::move(attachmentsJson);
        }
        {
            NJson::TJsonValue traitsJson = NJson::JSON_ARRAY;
            for (auto&& trait : traits) {
                traitsJson.AppendValue(::ToString(trait));
            }
            post["traits"] = std::move(traitsJson);
        }

        request.SetUri("/api/yandex/chat/action").SetCgiData("chat_id=" + chatId + (operatorId ? "&user_id=" + userId : "")).SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", operatorId ? operatorId : userId);
        for (auto&& [name, header] : headers) {
            request.AddHeader(name, header);
        }

        if (lat != 0.0 || lon != 0.0) {
            request.AddHeader("Lat", ::ToString(lat));
            request.AddHeader("Lon", ::ToString(lon));
        }
        return CommonSendReply(request);
    }

    bool RateChatMessage(const TString& userId, const TString& chatId, const ui64 messageId, const TString& rating) {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        post["chat_id"] = chatId;
        post["user_id"] = userId;
        post["message_id"] = messageId;
        post["rating"] = rating;

        request.SetUri("/api/staff/chat/message/rate").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    bool CommitSpecialChatAction(const TString& userId, const TString& chatId, const NDrive::NChat::TMessage::EMessageType type, const TString& payload, const TString& operatorId = "") {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        post["message_type"] = ::ToString(type);
        post["message"] = payload;

        TString cgi = "chat_id=" + chatId;
        if (operatorId) {
            cgi += "&user_id=" + userId;
        }

        request.SetUri("/api/yandex/chat/action").SetCgiData(cgi).SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", operatorId ? operatorId : userId);
        return CommonSendReply(request);
    }

    bool SendBonusesToChat(const TString& userId, const TString& chatId, const ui32 amount, const TString& operatorId = "") {
        return CommitSpecialChatAction(userId, chatId, NDrive::NChat::TMessage::EMessageType::Bonus, ::ToString(amount), operatorId);
    }

    bool SendOrderToChat(const TString& userId, const TString& chatId, const TString& sessionId, const TString& operatorId = "") {
        return CommitSpecialChatAction(userId, chatId, NDrive::NChat::TMessage::EMessageType::Order, sessionId, operatorId);
    }

    bool SendStickerToChat(const TString& userId, const TString& chatId, const TString& stickerId, const TString& operatorId = "") {
        return CommitSpecialChatAction(userId, chatId, NDrive::NChat::TMessage::EMessageType::Sticker, stickerId, operatorId);
    }

    bool RemoveChat(const TString& userId, const TString& chatId, const bool dropState) {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        post["user_id"] = userId;
        post["chat_id"] = chatId;
        if (dropState) {
            post["drop_state"] = "da";
        }

        request.SetUri("/api/yandex/chats/remove").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool AlterChatFlag(const TString& userId, const TString& chatId, const NDrive::NChat::TChat::EChatFlags flag, const bool targetState) {
        NNeh::THttpRequest request;
        request.SetUri("/support_api/chat/flags/edit").SetCgiData("chat_id=" + chatId + "&flag=" + ::ToString(flag) + "&enabled=" + (targetState ? "true" : "false"));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    TString SubmitChatResource(const TString& userId, const TString& chatId, const TString& resource, const TString& resourceId = "", const TString& contentType = "", const bool shared = false) {
        NNeh::THttpRequest request;
        request.SetUri("support_api/chat/resource").SetCgiData("chat_id=" + chatId + (resourceId ? ("&resource_id=" + resourceId) : "") + "&shared=" + ::ToString(shared)).SetPostData(TBlob::FromString(resource));
        request.AddHeader("Authorization", userId);
        if (contentType) {
            request.AddHeader("Content-Type", contentType);
        }

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return "";
        }

        if (resourceId) {
            UNIT_ASSERT_VALUES_EQUAL(resultReport["resource_id"].GetString(), resourceId);
        }

        return resultReport["resource_id"].GetString();
    }

    TString GetChatResource(const TString& userId, const TString& chatId, const TString& resourceId) {
        NNeh::THttpRequest request;
        request.SetUri("support_api/chat/resource").SetCgiData("chat_id=" + chatId + "&resource_id=" + resourceId);
        request.AddHeader("Authorization", userId);

        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return "";
        }

        UNIT_ASSERT(result.GetHeaders().HasHeader("Content-Type"));

        return result.Content();
    }

    TString GetChatResourcePreview(const TString& userId, const TString& chatId, const TString& resourceId) {
        NNeh::THttpRequest request;
        request.SetUri("support_api/chat/resource/preview").SetCgiData("chat_id=" + chatId + "&resource_id=" + resourceId);
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        if (result.Code() != 200) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
            return "";
        }
        UNIT_ASSERT(result.GetHeaders().HasHeader("Content-Type"));
        return result.Content();
    }

    bool ResetChat(const TString& userId, const TString& chatId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/chats/reset").SetCgiData("chat_id=" + chatId);
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool ReaskUserDocuments(const TString& userId, const TVector<NUserDocument::EType>& resubmitDocuments) {
        TAtomicSharedPtr<TUserDocumentsReaskTag> tagPtr = new TUserDocumentsReaskTag("documents_reask_tag");
        auto docsSet = MakeSet(resubmitDocuments);
        tagPtr->SetLicenseBack(docsSet.contains(NUserDocument::EType::LicenseBack));
        tagPtr->SetLicenseFront(docsSet.contains(NUserDocument::EType::LicenseFront));
        tagPtr->SetPassportBiographical(docsSet.contains(NUserDocument::EType::PassportBiographical));
        tagPtr->SetPassportRegistration(docsSet.contains(NUserDocument::EType::PassportRegistration));
        tagPtr->SetPassportSelfie(docsSet.contains(NUserDocument::EType::PassportSelfie));
        return AddTag(tagPtr, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
    }

    bool TakeoutRequest(const TString& userPassportUid, const TString& jobId = "abc") {
        NNeh::THttpRequest request;
        TString payload = "job_id=" + jobId + "&uid=" + userPassportUid + "&unixtime=187678678";
        request.SetUri("/api/takeout/request").SetPostData(TBlob::FromString(payload)).SetRequestType("POST");
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);  // i.e. auth is not tested in the ut
        return CommonSendReply(request);
    }

    bool DeleteChatMessage(const TString& userId, const TString& chatId, const ui64 messageId) {
        NNeh::THttpRequest request;

        NJson::TJsonValue post;
        post["chat_id"] = chatId;
        post["user_id"] = userId;
        post["message_id"] = messageId;
        post["delete"] = "da";

        request.SetUri("/api/staff/chat/message/edit").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    bool SendArbitraryMessage(const TString& chatId, const TString& userId, const TString& messageText, const TString& operatorId = USER_ROOT_DEFAULT) {
        NNeh::THttpRequest request;

        NJson::TJsonValue post;
        post["message"] = messageText;

        request.SetUri("/api/staff/chat/message").SetCgiData("chat_id=" + chatId + "&user_id=" + userId).SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    bool ForceRegister(const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("/api/staff/user/register-force").SetCgiData("user_id=" + userId);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    bool ForceRemoveUser(const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/user/remove").SetCgiData("user_id=" + userId);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    bool AddExternalBlacklistEntry(const TString& fieldType, const TString& fieldValue, const TString& policy) {
        NNeh::THttpRequest request;

        NJson::TJsonValue post;
        post["field_type"] = fieldType;
        post["field_value"] = fieldValue;
        post["policy"] = policy;
        post["comment"] = "test";

        request.SetUri("/api/staff/blacklist-external/add").SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetYangAssignmentData(const TString& secretId, const TString& assignmentId) {
        NNeh::THttpRequest request;
        request.SetUri("/support_api/yang/assignment/data").SetCgiData("secretId=" + secretId + "&assignmentId=" + assignmentId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue GetFeed(const TString& tagNames, const TString& performer, const TString& operatorId, TTagsSearch* tagsSearch, const TVector<TString>& requestTypes, const TString& additionalCgi = "") {
        UNIT_ASSERT(tagsSearch);
        NNeh::THttpRequest request;
        TString cgi;
        if (!!performer) {
            cgi = "&performer_id=" + performer;
        }
        if (!!tagNames) {
            if (cgi) {
                cgi += "&";
            }
            cgi += "tag_names=" + tagNames;
        }
        if (!requestTypes.empty()) {
            cgi += "&request_types=" + JoinStrings(requestTypes, ",");
        }
        if (!additionalCgi.empty()) {
            if (cgi) {
                cgi += "&";
            }
            cgi += additionalCgi;
        }
        request.SetUri("/api/staff/chats/feed").SetCgiData(cgi);
        request.AddHeader("Authorization", operatorId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue GetChatsFeed(const TString& tagNames, const TString& performer, const TString& operatorId, TTagsSearch* tagsSearch) {
        return GetFeed(tagNames, performer, operatorId, tagsSearch, {"chat"});
    }

    NJson::TJsonValue GetFullFeed(const TString& tagNames, const TString& performer, const TString& operatorId, TTagsSearch* tagsSearch) {
        return GetFeed(tagNames, performer, operatorId, tagsSearch, {"chat", "call", "outgoing"});
    }

    NJson::TJsonValue GetHistoricalFillingData(const TString& secretId, const TString& assignmentId, const TString& reqAssignmentId) {
        NNeh::THttpRequest request;
        request.SetUri("/support_api/yang/assignment/historical").SetCgiData("secretId=" + secretId + "&assignmentId=" + assignmentId + "&requestAssignmentId=" + reqAssignmentId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue GetSupportRequestsWithCgi(const TString& cgi) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/support/requests").SetCgiData(cgi);
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool DeferRequest(const TString& tagId, const TString& operatorId, const TMaybe<TInstant> deferUntil = {}, const TString& message = "", const TString& comment = "") {
        NNeh::THttpRequest request;
        NJson::TJsonValue postData;
        if (deferUntil) {
            postData["defer_until"] = deferUntil->Seconds();
        }
        if (comment) {
            postData["comment"] = comment;
        }
        if (message) {
            postData["message"] = message;
        }
        request.SetUri("api/staff/support/request/defer").SetCgiData("tag_id=" + tagId).SetPostData(TBlob::FromString(postData.GetStringRobust()));
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    bool UndeferRequest(const TString& tagId, const TString& chatId, const TString& operatorId) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/support/request/undefer").SetCgiData("tag_id=" + tagId + "&chat_id=" + chatId);
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    bool CloseChat(const TString& tagId, const TString& operatorId, const TString& additionalParams = "") {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/support/request/defer").SetCgiData("tag_id=" + tagId + "&is_finishing=true" + additionalParams);
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetCategorizerTree(const TString& userId, const TString& filter = "") const {
        NNeh::THttpRequest request;
        request.SetUri("api/support_staff/categorizer/tree").SetCgiData(filter ? ("filter=" + filter) : "");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport["tree"];
    }

    NJson::TJsonValue GetChatStickersList(const TString& userId, const bool isAdmin) {
        NNeh::THttpRequest request;
        if (isAdmin) {
            request.SetUri("api/staff/chat/sticker/list");
        } else {
            request.SetUri("api/yandex/chat/sticker/list");
        }
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool AddCategorizerNode(const TSupportCategorizerTreeNode& node, const TString& parentId, const TString& userId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload;
        payload["action"] = "add";
        if (parentId) {
            payload["parent_id"] = parentId;
        } else {
            payload["parent_id"] = NJson::JSON_NULL;
        }
        payload["meta"] = node.SerializeMeta();
        request.SetUri("api/support_staff/categorizer/tree/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool AudioteleCallEvent(const TString& phoneNumber, const ISupportTelephonyIncomingEvent::EType type, const TString& performerId = "") {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload;
        payload["cc"] = "audiotele";
        payload["queue"] = "audiotele-incoming-default";
        payload["phone"] = phoneNumber;
        payload["type"] = ::ToString(type);
        if (performerId) {
            payload["performer"] = performerId;
        }
        request.SetUri("api/support_staff/telephony/event").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);
        return CommonSendReply(request);
    }

    bool MoveCategorizerNode(const TString& childId, const TString& oldParentId, const TString& newParentId, const TString& userId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload;
        payload["action"] = "move";
        if (oldParentId) {
            payload["old_parent_id"] = oldParentId;
        } else {
            payload["old_parent_id"] = NJson::JSON_NULL;
        }
        if (newParentId) {
            payload["new_parent_id"] = newParentId;
        } else {
            payload["new_parent_id"] = NJson::JSON_NULL;
        }
        payload["id"] = childId;
        request.SetUri("api/support_staff/categorizer/tree/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool RemoveCategorizerNode(const TString& nodeId, const TString& userId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload;
        payload["action"] = "remove_node";
        payload["id"] = nodeId;
        request.SetUri("api/support_staff/categorizer/tree/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool AddCategorizerEdge(const TString& from, const TString& to, const TString& userId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload;
        payload["action"] = "copy";
        payload["new_parent_id"] = from;
        payload["id"] = to;
        request.SetUri("api/support_staff/categorizer/tree/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool EditCategorizerNode(const TString& id, const TSupportCategorizerTreeNode& node, const TString& userId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload;
        payload["action"] = "modify";
        payload["id"] = id;
        payload["meta"] = node.SerializeMeta();
        request.SetUri("api/support_staff/categorizer/tree/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetStickersHistory(const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/chat/sticker/history");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool AddChatSticker(const NDrive::NChat::TSticker& sticker, const TString& operatorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload = sticker.SerializeToJson();
        payload["operation"] = "add";
        request.SetUri("api/staff/chat/sticker/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    bool EditChatSticker(const NDrive::NChat::TSticker& sticker, const TString& operatorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload = sticker.SerializeToJson();
        payload["operation"] = "update";
        request.SetUri("api/staff/chat/sticker/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    bool RemoveChatSticker(const TString& id, const TString& operatorId) {
        NNeh::THttpRequest request;
        NJson::TJsonValue payload;
        payload["id"] = id;
        payload["operation"] = "remove";
        request.SetUri("api/staff/chat/sticker/edit").SetPostData(TBlob::FromString(payload.GetStringRobust()));
        request.AddHeader("Authorization", operatorId);
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetSupportRequestArchiveByUser(const TString& userId) {
        return GetSupportRequestsWithCgi("user_id=" + userId);
    }

    NJson::TJsonValue GetSupportRequestArchiveByPerformer(const TString& performerId) {
        return GetSupportRequestsWithCgi("performer_id=" + performerId);
    }

    NJson::TJsonValue GetSupportRequestArchiveByTags(const TVector<TString>& tags, const TString& userId = "") {
        TString cgi = "tags=" + JoinStrings(tags, ",");
        if (userId) {
            cgi += "&user_id=" + userId;
        }
        return GetSupportRequestsWithCgi(cgi);
    }

    bool PostYangAssignmentData(const TString& secretId, const TString& assignmentId, const NJson::TJsonValue& jsonData, bool passIdsViaPost = false, bool intoQueue = false) {
        NNeh::THttpRequest request;
        TString uri = intoQueue ? "/support_api/yang/assignment/queue" : "/support_api/yang/assignment/data";
        if (!passIdsViaPost) {
            request.SetUri(uri).SetCgiData("secretId=" + secretId + "&assignmentId=" + assignmentId).SetPostData(TBlob::FromString(jsonData.GetStringRobust())).SetRequestType("POST");
        } else {
            auto jsonDataPost = jsonData;
            jsonDataPost["secretId"] = secretId;
            jsonDataPost["assignmentId"] = assignmentId;
            request.SetUri(uri).SetPostData(TBlob::FromString(jsonDataPost.GetStringRobust())).SetRequestType("POST");
        }
        return CommonSendReply(request);
    }

    bool SubmitSelfieVerificationResults(const TString& secretId, const TString& assignmentId, const NJson::TJsonValue& jsonData) {
        NNeh::THttpRequest request;
        request.SetUri("/support_api/yang/selfie-assignment/data").SetCgiData("secretId=" + secretId + "&assignmentId=" + assignmentId).SetPostData(TBlob::FromString(jsonData.GetStringRobust())).SetRequestType("POST");
        return CommonSendReply(request);
    }

    bool SubmitBv(const TString& userId, const NUserDocument::EType type, const TString& content) {
        NNeh::THttpRequest request;
        request.SetUri("support_api/registration/bv").SetCgiData("type=" + ::ToString(type)).SetPostData(TBlob::FromString(content));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    bool AddCategorization(const TString& tagId, const TString& category, const TString& comment, const TString& userId) {
        NNeh::THttpRequest request;

        NJson::TJsonValue cat;
        cat["tag_id"] = tagId;
        cat["category"] = category;
        cat["comment"] = comment;

        request.SetUri("api/support_staff/categorization/add").SetPostData(TBlob::FromString(cat.GetStringRobust()));
        request.AddHeader("Authorization", userId);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return result.Code() == 200;
    }

    bool RemoveCategorization(const TString& tagId, const ui64 id, const TString& userId) {
        NNeh::THttpRequest request;

        request.SetUri("api/support_staff/categorization/remove").SetCgiData("id=" + ::ToString(id) + "&tag_id=" + tagId);
        request.AddHeader("Authorization", userId);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return result.Code() == 200;
    }

    NJson::TJsonValue GetCategorizations(const TVector<TString>& tagIds, const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("api/support_staff/categorization/get").SetCgiData("tag_id=" + JoinStrings(tagIds, ","));
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    NJson::TJsonValue FilteredUserCount(NJson::TJsonValue& filtersJson, const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/support_calls/users_count");
        request.AddHeader("Authorization", userId);
        request.SetPostData(TBlob::FromString(filtersJson.GetStringRobust()));

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;

        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }
        return resultReport;
    }

    bool SubmitUserDocumentPhoto(const TString& userId, const NUserDocument::EType type, const TString& content) {
        NNeh::THttpRequest request;
        request.SetUri("api/yandex/user_document_photo/upload").SetCgiData("type=" + ::ToString(type)).SetPostData(TBlob::FromString(content));
        request.AddHeader("Authorization", userId);
        return CommonSendReply(request);
    }

    std::pair<size_t, TVector<TString>> GetTagOperators(const TString& tagName, const TTagAction::ETagAction tagAction) {
        NNeh::THttpRequest request;

        request.SetUri("api/staff/tag/operators").SetCgiData("tag_name=" + tagName + "&action=" + ::ToString(tagAction));
        request.AddHeader("Authorization", USER_ROOT_DEFAULT);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        UNIT_ASSERT_C(resultReport["count"].IsInteger(), TStringBuilder() << resultReport["count"].GetStringRobust());
        UNIT_ASSERT_C(resultReport["user_ids"].IsArray(), TStringBuilder() << resultReport["user_ids"].GetStringRobust());
        std::pair<size_t, TVector<TString>> output;

        output.first = resultReport["count"].GetInteger();
        for (auto&& it : resultReport["user_ids"].GetArray()) {
            UNIT_ASSERT_C(it.IsString(), TStringBuilder() << it.GetStringRobust());
            output.second.push_back(it.GetString());
        }

        return output;
    }

    NJson::TJsonValue CrmCreateLead(const TString& userId, NJson::TJsonValue payload) {
        NNeh::THttpRequest request;
        request.SetUri("crm/organization/application");
        request.AddHeader("Authorization", userId);
        request.SetRequestType("POST").SetPostData(payload.GetStringRobust());

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        return resultReport;
    }

    NJson::TJsonValue GetSessionReport(const TString& cgiStr, const TString& userId) {
        NNeh::THttpRequest request;
        request.SetUri("b2b/sessions/dedicated_fleet");
        request.AddHeader("Authorization", userId);
        request.SetRequestType("GET").SetCgiData(cgiStr);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        return resultReport;
    }

    TVector<TString> GetNewAssignmentIdsByUserId(const TDocumentPhotosManager& docPhotoManager, const TString& userId, const NUserDocument::EType docType,
                                                const TMaybe<TString> chatId = {}, const TMaybe<TInstant> verifiedAt = {}) const {
        auto allAssignments = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo();
        auto allPhotos = docPhotoManager.GetUserPhotosDB().FetchInfo();

        TVector<TString> newAssignmentIds;
        for (auto&& it : allAssignments) {
            TString photoId;
            switch (docType) {
            case NUserDocument::EType::LicenseBack:
                photoId = it.second.GetLicenseBackId();
                break;
            case NUserDocument::EType::LicenseFront:
                photoId = it.second.GetLicenseFrontId();
                break;
            case NUserDocument::EType::PassportBiographical:
                photoId = it.second.GetPassportBiographicalId();
                break;
            case NUserDocument::EType::PassportRegistration:
                photoId = it.second.GetPassportRegistrationId();
                break;
            case NUserDocument::EType::PassportSelfie:
                photoId = it.second.GetPassportSelfieId();
                break;
            case NUserDocument::EType::Selfie:
                photoId = it.second.GetSelfieId();
                break;
            case NUserDocument::EType::LicenseSelfie:
                photoId = it.second.GetLicenseSelfieId();
                break;
            default:
                ythrow yexception() << docType << " is not supported";
            }
            if (photoId) {
                auto photoPtr = allPhotos.GetResultPtr(photoId);
                UNIT_ASSERT(!!photoPtr);
                if (photoPtr->GetUserId() == userId && (!chatId || photoPtr->GetOriginChat() == chatId.GetRef()) && (!verifiedAt || photoPtr->GetVerifiedAt() == verifiedAt.GetRef())) {
                    newAssignmentIds.push_back(it.first);
                }
            }
        }
        return newAssignmentIds;
    }

    NJson::TJsonValue CreateB2bAccount(const NJson::TJsonValue& payload = NJson::TJsonValue()) {
        NNeh::THttpRequest request;
        request.SetUri("b2b/organization/create")
            .SetRequestType("POST")
            .AddHeader("Authorization", USER_ROOT_DEFAULT)
            .SetPostData(payload.GetStringRobust());

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        return std::move(resultReport);
    }

    bool UpdateB2bAccount(const NJson::TJsonValue& payload) {
        NNeh::THttpRequest request;
        request.SetUri("b2b/organization/update")
            .SetRequestType("POST")
            .AddHeader("Authorization", USER_ROOT_DEFAULT)
            .SetPostData(payload.GetStringRobust());

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        return result.Code() == 200;
    }

    bool CreateB2bWalletAccount(const NJson::TJsonValue& payload) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/billing/accounts/update")
            .SetRequestType("POST")
            .AddHeader("Authorization", USER_ROOT_DEFAULT)
            .SetPostData(payload.GetStringRobust());

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        return result.Code() == 200;
    }

    bool ActivateB2bAccount(const NJson::TJsonValue& payload) {
        NNeh::THttpRequest request;
        request.SetUri("b2b/accounts/activate")
            .SetRequestType("POST")
            .AddHeader("Authorization", USER_ROOT_DEFAULT)
            .SetRequestType("POST").SetPostData(payload.GetStringRobust());
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetB2bAccountsDescription(TStringStream& cgiStream) {
        NNeh::THttpRequest request;
        request.SetUri("b2b/wallets/get").SetCgiData(cgiStream.Str())
            .SetRequestType("GET")
            .AddHeader("Authorization",  USER_ROOT_DEFAULT);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        return std::move(resultReport);
    }

    NJson::TJsonValue GetB2bAccounts(const TString& limitName) {
        NNeh::THttpRequest request;
        request.SetUri("b2b/accounts/get")
            .SetRequestType("GET")
            .SetCgiData("&with_users=true&account_name=" + limitName + "&numdoc=99999")
            .AddHeader("Authorization",  USER_ROOT_DEFAULT);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << " : " << result.Content() << Endl;
        }

        return std::move(resultReport);
    }

    bool LinkB2bAccount(const NJson::TJsonValue& payload) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/billing/accounts/link")
            .SetRequestType("POST")
            .AddHeader("Authorization", USER_ROOT_DEFAULT)
            .SetPostData(payload.GetStringRobust());
        return CommonSendReply(request);
    }

    NJson::TJsonValue GetB2bSessionsList(const TString& cgiStr) {
        NNeh::THttpRequest request;
        request.SetUri("b2b/sessions/list")
            .SetRequestType("GET")
            .SetCgiData(cgiStr)
            .AddHeader("Authorization", USER_ROOT_DEFAULT);

        NUtil::THttpReply result = GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_NULL;
        if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
            ERROR_LOG << result.Code() << ":" << result.Content() << Endl;
        }

        return std::move(resultReport);
    }

    bool CreateCompiledBill(const NJson::TJsonValue& payload) {
        NNeh::THttpRequest request;
        request.SetUri("api/staff/compiled_bill/add")
            .SetRequestType("POST")
            .AddHeader("Authorization", USER_ROOT_DEFAULT)
            .SetPostData(payload.GetStringRobust());
        return CommonSendReply(request);
    }

    NJson::TJsonValue SignalqSessionsStart(const TString& serialNumber, const TString& operatorId);

    NJson::TJsonValue SignalqCreateJsonEvent(const TString& serialNumber, const TString& eventType, const TString& eventId, const TInstant timestamp);

    NJson::TJsonValue SignalqSignalCreate(const NJson::TJsonValue& postData, const TString& operatorId);

    NJson::TJsonValue SignalqTraceTagResolutionSet(const TString& tagId, const TString& resolution, const TString& operatorId);
    NJson::TJsonValue SignalqTraceTagResolutionSet(const TString& serialNumber, const TString& eventId, const TInstant eventAt, const TString& resolution, const TString& operatorId);

    NJson::TJsonValue SignalqNotifySupport(const NJson::TJsonValue& postData, const TString& operatorId);

    std::pair<ui32, NJson::TJsonValue> SignalqStatusHistory(const NJson::TJsonValue& postData, const TString& operatorId);

    NUtil::THttpReply UpsertExternalAccessToken(const TString& userId, const TString& token, const TVector<TString>& ipWhiteList, const TString& userActorId) const;

    bool InvalidateExternalAccessToken(const TString& token, const TString& userActorId) const;

    NUtil::THttpReply CreateMosgorpasOffer(const TString& carId, const TString& token, const TString& ipAddress) const;

    NUtil::THttpReply GetServiceRoute(const TString& userId, const TString& userActorId) const;
    void CompleteInitialRegistrationChatSteps(const NDrive::IServer* server, const TString& userId, TEnvironmentGenerator& eGenerator, bool bindCard = true);

    NUtil::THttpReply ShowDistributingBlock(const TString& userId, const TString& distributingBlockId) const;
    NUtil::THttpReply HideDistributingBlock(const TString& userId, const TString& distributingBlockId) const;
    NUtil::THttpReply AddLeasingCar(const TString& userId, const NDrivematics::TCarInfo& carInfo) const;
    NUtil::THttpReply GetLeasingPortfolio(const TString& userId, TMaybe<std::pair<ui32, ui32>> segment = Nothing()) const;
    NUtil::THttpReply GetLeasingTaxiCompanies(const TString& userId, TMaybe<std::pair<ui32, ui32>> segment = Nothing()) const;
    NUtil::THttpReply GetSignals(const TString& userId, const TVector<TString>& signalsExt, TMaybe<ui64> pageSize = {}, TMaybe<ui64> carTagsHistoryCursor = {}, TMaybe<ui64> sessionTagsHistoryCursor = {}) const;
    NUtil::THttpReply ZoneRequest(const TString& userId, const TString method, const NJson::TJsonValue& post, const TString& cgi = "") const;

    bool LeasingAddScore(const TString& userId, const TString& leasingCompanyName, double score, TMaybe<double> telematicsScore = Nothing()) const;
    bool WaitLocation(const TString& carId, TMaybe<TGeoCoord> coordinate = {}, TMaybe<TString> tag = {});
    bool WaitSensor(const TString& carId, TStringBuf sensor, TStringBuf expected = {});
    bool WaitSensorState(const TString& carId, TStringBuf sensor, TStringBuf expected);
    bool WaitStatus(const TString& carId, TStringBuf status, const NDrive::IServer& server, TDuration waitingDuration = TDuration::Seconds(60));
    bool WaitCar(const TString& carId);

    TServerConfigGenerator(const TString& databaseType = "SQLite", const TString& databasePath = {}, const TString& sensorApi = {}, const TString& sensorsDumpPath = {}, const TString& stateTemplatesPath = {});
    virtual ~TServerConfigGenerator();

    void SetOffersStorageName(const TString& name);
    void SetPrivateDataClientType(const TString& value);

    void ToString(IOutputStream& os) const;

    THolder<TAnyYandexConfig> Generate();
};

} // namespace NDrive

class TDocumentsHelper {
public:
    const static TVector<TString> UserProfileFields;

    const static TVector<TString> PassportStringFields;
    const static TVector<TString> PassportDateFields;
    const static TVector<TString> DrivingLicenseStringFields;
    const static TVector<TString> DrivingLicenseDateFields;

    static TString RandomString() {
        TReallyFastRng32 rand(Now().NanoSeconds());
        TString result = "";
        for (size_t i = 0; i < 10; ++i) {
            result += TString((char)('a' + rand.Uniform(26)));
        }
        return result;
    }

    static TString RandomPhoneNumber() {
        TReallyFastRng32 rand(Now().NanoSeconds());
        TString result = "+7";
        for (size_t i = 0; i < 10; ++i) {
            result += TString((char)('0' + rand.Uniform(10)));
        }
        return result;

    }

    static TString RandomDateIsoFormat() {
        TReallyFastRng32 rand(Now().NanoSeconds());
        ui32 randomMoment = rand.Uniform(1450080958) + 1000000;
        randomMoment -= randomMoment % (24 * 60 * 60);  // start of day
        return TInstant::Seconds(randomMoment).ToString();
    }

    static NJson::TJsonValue CreateRandomUserAccountData(bool withPassport = false, bool withDrivingLicense = false) {
        NJson::TJsonValue result = NJson::JSON_MAP;
        for (auto&& fieldName : UserProfileFields) {
            result[fieldName] = RandomString();
        }

        result["phone"] = RandomPhoneNumber();

        if (withPassport) {
            result["passport"] = CreateRandomPassportDataJson(false);
        }
        if (withDrivingLicense) {
            result["driving_license"] = CreateRandomDrivingLicenseDataJson();
            result["driving_license"]["categories_b_valid_to_date"] = TInstant::Seconds(Max<i32>()).ToString();
        }
        if (withPassport && withDrivingLicense) {
            result["driving_license"]["first_name"] = result["passport"]["first_name"];
            result["driving_license"]["last_name"] = result["passport"]["last_name"];
            result["driving_license"]["birth_date"] = result["passport"]["birth_date"];
        }

        return result;
    }

    static NJson::TJsonValue CreateRandomPassportDataJson(const bool withDocType = true) {
        NJson::TJsonValue result = NJson::JSON_MAP;
        if (withDocType) {
            result["doc_type"] = "id";
        }

        for (auto&& fieldName : PassportStringFields) {
            result[fieldName] = RandomString();
        }

        for (auto&& fieldName : PassportDateFields) {
            result[fieldName] = RandomDateIsoFormat();
        }
        result["birth_date"] = "1980-01-01T00:00:00.000Z";

        return result;
    }

    static NJson::TJsonValue CreateRandomDrivingLicenseDataJson() {
        NJson::TJsonValue result = NJson::JSON_MAP;

        for (auto&& fieldName : DrivingLicenseStringFields) {
            result[fieldName] = RandomString();
        }

        for (auto&& fieldName : DrivingLicenseDateFields) {
            result[fieldName] = RandomDateIsoFormat();
        }
        result["birth_date"] = "1980-01-01T00:00:00.000Z";

        return result;
    }

    static TUserPassportData CreateRandomPassportData() {
        auto json = CreateRandomPassportDataJson();
        TUserPassportData passport;
        UNIT_ASSERT(passport.ParseFromDatasync(json));
        return passport;
    }

    static TUserDrivingLicenseData CreateRandomDrivingLicenseData() {
        auto json = CreateRandomDrivingLicenseDataJson();
        TUserDrivingLicenseData drivingLicense;
        drivingLicense.ParseFromDatasync(json);
        return drivingLicense;
    }

    static NJson::TJsonValue ConstructConsistentYangPayload() {
        auto passport = CreateRandomPassportData();
        auto drivingLicense = CreateRandomDrivingLicenseData();

        NJson::TJsonValue payload;
        payload["passport_biographical"] = passport.SerializeBioToYang(false);
        payload["passport_registration"] = passport.SerializeRegToYang(false);
        payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
        payload["license_back"] = drivingLicense.SerializeBackToYang(false);

        payload["license_front"]["data"]["first_name"] = passport.GetFirstName();
        payload["license_front"]["data"]["last_name"] = passport.GetLastName();
        payload["license_front"]["data"]["birth_date"] = passport.GetBirthDate();

        payload["license_back"]["data"]["categories_b_valid_to_date"] = TInstant::Seconds(Max<i32>()).ToString();
        payload["passport_biographical"]["data"]["number"] = "HB2352296";

        payload["license_back"]["data"]["has_at_mark"] = true;
        payload["verification_statuses"]["is_fraud"] = "NOT_FRAUD";

        TString testComment = "'\"тест ТЕСТ test\"'\\";
        payload["verification_statuses"]["comment"] = testComment;

        payload["verification_statuses"]["passport_biographical_status"] = "OK";
        payload["verification_statuses"]["passport_registration_status"] = "OK";
        payload["verification_statuses"]["license_front_status"] = "OK";
        payload["verification_statuses"]["license_back_status"] = "OK";
        payload["verification_statuses"]["passport_selfie_status"] = "OK";

        return payload;
    }
};

class TTestCallback {
protected:
    TString DebugName;

    TCondVar CondVarResponseCount;
    TMutex ResponseActionMutex;

    ui32 CountResponsesReceived = 0;
    ui32 SuccessCount = 0;
    ui32 FailureCount = 0;

public:
    TTestCallback(const TString& debugName)
        : DebugName(debugName)
    {
    }

    void RegisterSuccess(const TString& additionalInfo = "") {
        TGuard<TMutex> g(ResponseActionMutex);
        SuccessCount += 1;
        CountResponsesReceived += 1;
        INFO_LOG << "Successful response received in callback " << DebugName << " : " << additionalInfo << Endl;
        CondVarResponseCount.Signal();
    }

    void RegisterFailure(const TString& additionalInfo = "") {
        TGuard<TMutex> g(ResponseActionMutex);
        FailureCount += 1;
        CountResponsesReceived += 1;
        INFO_LOG << "Unsuccessful response received in callback " << DebugName << " : " << additionalInfo << Endl;
        CondVarResponseCount.Signal();
    }

    ui32 GetCountResponsesReceived() const {
        return CountResponsesReceived;
    }

    ui32 GetSuccessCount() const {
        return SuccessCount;
    }

    ui32 GetFailureCount() const {
        return FailureCount;
    }

    void WaitForResponsesCount(ui32 requiredCount) {
        while (CountResponsesReceived < requiredCount) {
            TGuard<TMutex> g(ResponseActionMutex);
            CondVarResponseCount.Wait(ResponseActionMutex);
        }
    }
};

class TPrivateDataAcquisutionTestCallback : public TTestCallback, public IPrivateDataAcquisitionCallback {
    R_READONLY(TUserPassportData, LastPassport);
    R_READONLY(TUserDrivingLicenseData, LastDrivingLicense);

public:
    TPrivateDataAcquisutionTestCallback(const TString& debugName)
        : TTestCallback(debugName)
        , IPrivateDataAcquisitionCallback(2)
    {
    }

    virtual void DoOnUnsuccessfulResponse(EPrivateDataType /*documentType*/, const TString& /*revision*/) override final {
        RegisterFailure();
    }

    virtual void DoOnPassportReceipt(const TString& /*revision*/, const TUserPassportData& passport) override final {
        LastPassport = passport;
        RegisterSuccess(passport.SerializeToJson(NUserReport::ReportAll).GetStringRobust());
    }

    virtual void DoOnDrivingLicenseReceipt(const TString& /*revision*/, const TUserDrivingLicenseData& drivingLicense) override final {
        LastDrivingLicense = drivingLicense;
        RegisterSuccess(drivingLicense.SerializeToJson(NUserReport::ReportAll).GetStringRobust());
    }

    virtual void ProcessAllResponses() override final {
    }
};

class TUserDocumentPhotoUploadTestCallback : public IDocumentMediaUpdateCallback {
private:
    TMutex Mutex;
    ui32 TotalResponses = 0;
    ui32 TotalSuccesses = 0;
    TString LastSuccessId;

public:
    virtual void OnSuccess(const TString& id, const TString&) override {
        TGuard<TMutex> g(Mutex);
        INFO_LOG << "Success: " << id << Endl;
        ++TotalResponses;

        ++TotalSuccesses;
        LastSuccessId = id;
    }

    virtual void OnFailure(const TString&) override {
        TGuard<TMutex> g(Mutex);
        INFO_LOG << "Failure" << Endl;
        ++TotalResponses;
    }

    ui32 GetTotalResponses() const {
        TGuard<TMutex> g(Mutex);
        return TotalResponses;
    }

    TString GetLastSuccessId() const {
        TGuard<TMutex> g(Mutex);
        return LastSuccessId;
    }

    ui32 GetTotalSuccesses() const {
        return TotalSuccesses;
    }
};

class TTestNodeResolver: public INodeResolver {
public:
    virtual TString GetNextNode(const IChatUserContext::TPtr /*context*/, const NDrive::NChat::TMessageEvents& /*messages*/) const override;
    virtual NThreading::TFuture<TString> GetNextNodeFuture(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages) const override;
    virtual NThreading::TFuture<TTaxiSupportChatSuggestClient::TSuggestResponse> GetSuggest(const IChatUserContext::TPtr context, const NDrive::NChat::TMessageEvents& messages, bool userOptionsSuggest = false) const override;
    virtual NJson::TJsonValue GetSuggestedChatOptions(const TTaxiSupportChatSuggestClient::TSuggestResponse& suggestResponse) const override;
    void AddClassificationResult(const TString& classification, i32 confidence);
    TVector<std::pair<i32, TString>>  GetDefaultClassificationResults() const;
    virtual TString GetType() const override;

private:
    mutable TVector<std::pair<i32, TString>> MockClassificationResults;
    static TFactory::TRegistrator<TTestNodeResolver> Registrator;
};
