#include "helper2.h"

#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>

#include <rtline/library/storage/sqlite/structured.h>

TTestEnvironment::TTestEnvironment(const TString& databaseType)
    : ConfigGenerator(databaseType)
{
    ConfigGenerator.SetNeedBackground(0);
    Singleton<NSQLite::TUniqueFields>()->RegisterRowIdColumn("drive_payments", "payment_id");
    Singleton<NSQLite::TUniqueFields>()->RegisterRowIdColumn("drive_refunds", "payment_id");
}

void TTestEnvironment::Build() {
    GetEnvironmentGenerator().SetNeedTelematics(true);
    GetEnvironmentGenerator().BuildEnvironment();
    GetEnvironmentGenerator().BuildRTBackgrounds(*GetServer(), GetConfigGenerator().GetNeedBackground());
}

NDrive::TServerConfigGenerator& TTestEnvironment::GetConfigGenerator() {
    return ConfigGenerator;
}

TEnvironmentGenerator& TTestEnvironment::GetEnvironmentGenerator() {
    if (!EnvironmentGenerator) {
        EnvironmentGenerator = MakeHolder<TEnvironmentGenerator>(*GetServer());
    }
    return *EnvironmentGenerator;
}

NDrive::TServerConfig& TTestEnvironment::GetConfig() {
    if (!Config) {
        TServerConfigConstructorParams params(ConfigGenerator.GetString().data());
        Config = MakeHolder<NDrive::TServerConfig>(params);
    }
    return *Config;
}

NDrive::TServerGuard& TTestEnvironment::GetServer() {
    if (!Server) {
        Server = MakeHolder<NDrive::TServerGuard>(GetConfig());
    }
    return *Server;
}

NDrive::NTest::TRTContext& TTestEnvironment::GetContext() {
    if (!Context) {
        auto telematicServerBuilder = GetOrCreateTelematicServerBuilder();
        if (!Checked(telematicServerBuilder)->IsRunning()) {
            telematicServerBuilder->Run();
        }
        Context = MakeHolder<NDrive::NTest::TRTContext>(GetConfigGenerator(), GetServer(), telematicServerBuilder);
    }
    return *Context;
}

TTelematicServerBuilder& TTestEnvironment::GetTelematicServerBuilder() {
    auto result = GetOrCreateTelematicServerBuilder();
    return *Checked(result);
}

TAtomicSharedPtr<TTelematicServerBuilder> TTestEnvironment::GetOrCreateTelematicServerBuilder() {
    if (!TelematicServerBuilder) {
        TelematicServerBuilder = MakeAtomicShared<TTelematicServerBuilder>();
    }
    return TelematicServerBuilder;
}

void TTestEnvironment::Execute(NDrive::NTest::ITestAction&& action) {
    Execute(action);
}

void TTestEnvironment::Execute(NDrive::NTest::ITestAction& action) {
    INFO_LOG << "Executing TestAction " << TypeName(action) << Endl;
    TInstant start = Now();
    action.Execute(GetContext());
    TInstant finish = Now();
    TDuration duration = finish - start;
    INFO_LOG << "Executed TestAction " << TypeName(action) << ": " << duration << Endl;
}

void AddRoleToUser(const NDrive::IServer& server, const TString& userId, const TString& roleId) {
    auto driveApi = server.GetDriveAPI();
    UNIT_ASSERT(driveApi);
    auto session = driveApi->template BuildTx<NSQL::Writable>();
    {
        TUserRole userRole;
        userRole.SetRoleId(roleId).SetUserId(userId);
        UNIT_ASSERT(driveApi->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
    }
    UNIT_ASSERT(session.Commit());
}

void RegisterAction(const NDrive::IServer& server, TUserAction::TPtr action, const TString& roleId) {
    UNIT_ASSERT(action);
    auto driveApi = server.GetDriveAPI();
    UNIT_ASSERT(driveApi);
    auto session = driveApi->template BuildTx<NSQL::Writable>();
    UNIT_ASSERT(driveApi->GetRolesManager()->GetActionsDB().ForceUpsert(action, USER_ROOT_DEFAULT, session));
    if (roleId) {
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId(action->GetName()).SetRoleId(roleId);
        UNIT_ASSERT(driveApi->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
    }
    UNIT_ASSERT(session.Commit());
}

void RegisterTag(const NDrive::IServer& server, TTagDescription::TPtr tagDescription) {
    UNIT_ASSERT(tagDescription);
    auto driveApi = server.GetDriveAPI();
    UNIT_ASSERT(driveApi);
    auto session = driveApi->template BuildTx<NSQL::Writable>();
    UNIT_ASSERT_C(driveApi->GetTagsManager().GetTagsMeta().RegisterTag(tagDescription, USER_ROOT_DEFAULT, session), session.GetStringReport());
    UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
}

void SetSetting(const NDrive::IServer& server, const TString& key, const TString& value) {
    TVector<TSetting> settings;
    settings.emplace_back(key, value);
    UNIT_ASSERT(server.GetSettings().SetValues(settings, USER_ROOT_DEFAULT));
}

TBillingCompilation GetBillingCompilation(const NDrive::IServer& server, const TString& sessionId) {
    auto tx = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    auto optionalSession = server.GetDriveDatabase().GetSessionManager().GetSession(sessionId, tx);
    UNIT_ASSERT(optionalSession);
    auto session = *optionalSession;
    UNIT_ASSERT(session);
    auto compilation = session->GetCompilationAs<TBillingSession::TBillingCompilation>();
    UNIT_ASSERT(compilation);
    return *compilation;
}
