#include "script.h"

#include <drive/backend/data/delegation.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>

#include <drive/telematics/server/library/config.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NDrive::NTest {

    void TCheckLandingComment::DoExecute(TRTContext& context) {
        auto queryUserId = HasUserId() ? GetUserIdUnsafe() : context.GetUserId();
        auto landingsFR = context.GetServer()->GetDriveAPI()->GetUserLandingData()->FetchInfo(queryUserId);
        for (auto&& landingIt : landingsFR) {
            for (auto&& acc : landingIt.second.GetLandings()) {
                if (acc.GetUserId() != queryUserId) {
                    continue;
                }
                if (acc.GetId() == LandingId) {
                    bool isOK = acc.GetComment() == Comment;
                    UNIT_ASSERT_VALUES_EQUAL(isOK, GetExpectOK());
                    return;
                }
            }
        }
        UNIT_ASSERT(!GetExpectOK());
    }

    void TCheckUserSetting::DoExecute(TRTContext& context) {
        auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
        TDBTag settingsTag = context.GetDriveAPI().GetUserSettings(UserId ? *UserId : context.GetUserId(), session);
        TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
        UNIT_ASSERT(!!settingsData);
        auto value = settingsData->GetField(Key);
        UNIT_ASSERT(Checker(value));
    }

    TWarningScreenChecker::TWarningScreenChecker(const TInstant startInstant, const TString& event,
                                                 const TChecker checker,
                                                 const TCgiBuilder cgiBuilder)
        : TBase(startInstant)
        , Event(event)
        , Checker(checker)
        , CgiBuilder(cgiBuilder)
    {
    }

    TString TWarningScreenChecker::GetProcessorConfiguration() const {
        TStringStream ss;
        ss << "<user_app/warnings/get>" << Endl;
        ss << "    AuthModuleName: fake" << Endl;
        ss << "    ProcessorType: warning_screen" << Endl;
        ss << "</user_app/warnings/get>" << Endl;
        return ss.Str();
    }

    void TWarningScreenChecker::DoExecute(NDrive::NTest::TRTContext& context) {
        const TString request = "/user_app/warnings/get";
        TStringBuilder cgi;
        cgi << "event=" << Event;
        if (CgiBuilder) {
            cgi << "&" << CgiBuilder();
        }
        NJson::TJsonValue response = SendRequest(context, request, NJson::JSON_NULL, cgi);
        Checker(response);
    }

    void TAddLanding::DoExecute(TRTContext& context) {
        UNIT_ASSERT(context.GetConfigGenerator().ModifyLanding(LandingId, JsonLanding.GetStringRobust(), true, UserId));
    }

    NDrive::NTest::TAPIAction::TFactory::TRegistrator<TWarningScreenChecker>
            TWarningScreenChecker::Registrator("CherWarningScreenResponse");


    TString TSubmitDocumentPhoto::GetProcessorConfiguration() const {
        TStringStream ss;
        ss << "<api/yandex/user_document_photo/upload>" << Endl;
        ss << "    AuthModuleName: fake" << Endl;
        ss << "    ProcessorType: user_document_photo_upload" << Endl;
        ss << "</api/yandex/user_document_photo/upload>" << Endl;
        return ss.Str();
    }

    void TSubmitDocumentPhoto::DoExecute(TRTContext& context) {
        TString cgiData("type=" + ::ToString(Type));
        if (AdditionalLength) {
            cgiData += "&additional_length=" + ::ToString(AdditionalLength);
        }
        NJson::TJsonValue postJson;
        postJson["content"] = Content;
        NJson::TJsonValue report = SendRequest(context, "/api/yandex/user_document_photo/upload", postJson, cgiData);
        UNIT_ASSERT(report != NJson::JSON_NULL);
    }

    void TGetDocumentPhoto::DoExecute(TRTContext& context) {
        auto userPhotos = context.GetServer()->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetAllForUser(UserId);
        bool succeeded = false;
        for (auto&& p : userPhotos) {
            if (p.GetType() != Type) {
                continue;
            }
            TString content;
            UNIT_ASSERT(context.GetServer()->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentPhoto(p.GetId(), content, *context.GetServer()));
            if (content == Content) {
                succeeded = true;
                break;
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(succeeded, GetExpectOK());
    }

    void TGetDocumentVideo::DoExecute(TRTContext& context) {
        auto userPhotos = context.GetServer()->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetAllForUser(UserId);
        bool succeeded = false;
        for (auto&& p : userPhotos) {
            if (p.GetType() != Type) {
                continue;
            }
            TString content;
            UNIT_ASSERT(context.GetServer()->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentBackgroundVideo(p.GetId(), content, *context.GetServer()));
            if (content == Content) {
                succeeded = true;
                break;
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(succeeded, GetExpectOK());
    }

    TString TP2PCarDelegationCheck::GetProcessorConfiguration() const {
        TStringStream ss;
        ss << "<api/yandex/delegation/p2p>" << Endl;
        ss << "    AuthModuleName: fake" << Endl;
        ss << "    ProcessorType: user_p2p_delegation" << Endl;
        ss << "</api/yandex/delegation/p2p>" << Endl;
        return ss.Str();
    }

    void TP2PCarDelegationCheck::DoExecute(TRTContext& context) {
        TString cgi = "object_id=" + context.GetCar().Id;
        if (TargetUserPhone) {
            cgi += "&phone=" + TargetUserPhone;
        }

        auto reply = GetSendReply(context, "/api/yandex/delegation/p2p", NJson::JSON_NULL, cgi);
        INFO_LOG << reply.Content() << Endl;

        UNIT_ASSERT_C(reply.Code() / 100 != 5, TStringBuilder() << reply.GetDebugReply());
        UNIT_ASSERT_VALUES_EQUAL((reply.Code() / 100 == 2), GetExpectOK());

        NJson::TJsonValue replyJson = NJson::JSON_MAP;
        UNIT_ASSERT(NJson::ReadJsonFastTree(reply.Content(), &replyJson));

        if (ExpectedError != TDriveAPI::EDelegationAbility::Possible) {
            TString uiMessage;
            uiMessage = replyJson["error_details"]["special_info"]["error_code"].GetString();
            UNIT_ASSERT_VALUES_EQUAL(uiMessage , (::ToString(ExpectedError)));
        }

        if (HasTotalOptions()) {
            UNIT_ASSERT_VALUES_EQUAL(replyJson["types"].GetArray().size(), GetTotalOptionsUnsafe());
        }
    }
}
