#pragma once

#include "helper.h"

#include <drive/backend/ut/library/scripts/chat.h>
#include <drive/backend/ut/library/scripts/common.h>
#include <drive/backend/ut/library/scripts/roles.h>
#include <drive/backend/ut/library/scripts/session.h>
#include <drive/backend/ut/library/scripts/tag.h>

namespace NDrive::NTest {

    class TNotificationsProcessor: public NMessenger::IMessageProcessor {
    private:
        using TChecker = std::function<bool(const TString& uid, const TString& message)>;
        R_OPTIONAL(TChecker, Checker);
        R_READONLY(ui32, Counter, 0);
    public:

        TNotificationsProcessor() {
            RegisterGlobalMessageProcessor(this);
        }

        ~TNotificationsProcessor() {
            UnregisterGlobalMessageProcessor(this);
        }

        virtual bool Process(IMessage* message) override {
            const TInternalNotificationMessage* intMessage = dynamic_cast<const TInternalNotificationMessage*>(message);
            if (intMessage) {
                if (Checker) {
                    UNIT_ASSERT((*Checker)(intMessage->GetUID(), intMessage->GetMessage()));
                }
                ++Counter;
                return true;
            }
            return false;
        }
        virtual TString Name() const override {
            return "InternalNotificatonsChecker" + ::ToString((ui64)this);
        }
    };

    class TCreateArea: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, Type);
        R_FIELD(TString, Id);
        R_FIELD(TVector<TString>, AreaTags);
        R_FIELD(TVector<TGeoCoord>, Coords);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            auto revision = context.GetConfigGenerator().AreaRevision(Id, GetActionUserId());
            UNIT_ASSERT(context.GetConfigGenerator().UpsertArea(Id, GetActionUserId(), Coords, AreaTags, Type, revision));
        }
    public:

        TCreateArea& SetCoords(const TString& coordsStr) {
            UNIT_ASSERT(TGeoCoord::DeserializeVector(coordsStr, Coords));
            return *this;
        }

        using TBase::TBase;
    };

    class TAddLanding: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, LandingId);
        R_FIELD(NJson::TJsonValue, JsonLanding, NJson::JSON_NULL);
        R_FIELD(TString, UserId, USER_ROOT_DEFAULT);
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:

        TAddLanding(const TInstant startInstant, const TString& landingId, const TString& jsonLanding)
            : TBase(startInstant)
            , LandingId(landingId)
        {
            UNIT_ASSERT(NJson::ReadJsonFastTree(jsonLanding, &JsonLanding));
        }
    };

    class TP2PCarDelegationCheck : public TAPIAction {
        R_FIELD(TString, TargetUserPhone);
        R_FIELD(TDriveAPI::EDelegationAbility, ExpectedError, TDriveAPI::EDelegationAbility::Possible);
        R_OPTIONAL(size_t, TotalOptions);

    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TP2PCarDelegationCheck> Registrator;

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;

        virtual TString GetProcessorConfiguration() const override;
    };

    class TSubmitDocumentPhoto: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TSubmitDocumentPhoto> Registrator;

    public:
        R_FIELD(NUserDocument::EType, Type, NUserDocument::EType::Selfie);
        R_FIELD(TString, Content, "meow");
        R_FIELD(size_t, AdditionalLength, 0);

    protected:
        virtual void DoExecute(TRTContext& context) override;
        virtual TString GetProcessorConfiguration() const override;

    public:
        using TBase::TBase;
    };

    class TGetDocumentPhoto: public ITestAction {
    private:
        using TBase = ITestAction;

    public:
        R_FIELD(NUserDocument::EType, Type, NUserDocument::EType::Selfie);
        R_FIELD(TString, Content);
        R_FIELD(TString, UserId, USER_ID_DEFAULT);

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;
    };

    class TGetDocumentVideo: public ITestAction {
    private:
        using TBase = ITestAction;

    public:
        R_FIELD(NUserDocument::EType, Type, NUserDocument::EType::Selfie);
        R_FIELD(TString, Content);
        R_FIELD(TString, UserId, USER_ID_DEFAULT);

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;
    };

    class TCheckLandingComment: public ITestAction {
    private:
        using TBase = ITestAction;

    private:
        R_OPTIONAL(TString, UserId);
        R_FIELD(TString, LandingId);
        R_FIELD(TString, Comment);

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;
    };

    class TCheckUserSetting : public ITestAction {
    private:
        using TBase = ITestAction;
        using TChecker = std::function<bool(const TMaybe<TString>& setting)>;
        TChecker Checker;
        R_READONLY(TString, Key);
        R_OPTIONAL(TString, UserId);
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:
        TCheckUserSetting(const TInstant startInstant, const TString& key, const TChecker checker)
            : TBase(startInstant)
            , Checker(checker)
            , Key(key) {

        }
    };


    class TWarningScreenChecker: public NDrive::NTest::TAPIAction {
    public:
        using TChecker = std::function<void(const NJson::TJsonValue& response)>;
        using TCgiBuilder = std::function<TString()>;

    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TWarningScreenChecker> Registrator;
        TString Event;
        TChecker Checker;
        TCgiBuilder CgiBuilder;

    public:
        using TBase::TBase;

        TWarningScreenChecker(const TInstant startInstant, const TString& event,
                              const TChecker checker,
                              const TCgiBuilder cgiBuilder = TCgiBuilder());
        TString GetProcessorConfiguration() const override;

    protected:
        void DoExecute(NDrive::NTest::TRTContext& context) override;
    };


    template <class T>
    class TAddAndRunRTBackground : public ITestAction {
    private:
        using TBase = ITestAction;
        THolder<T> Backgound;
        R_FIELD(TString, Name, "rt_background_simple");

    protected:
        virtual void DoExecute(TRTContext& context) override {
            TRTBackgroundProcessContainer container(Backgound.Release());
            container.SetName(Name);
            UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        }

    public:
        TAddAndRunRTBackground(const TInstant startInstant, const TDuration period)
            : TBase(startInstant)
            , Backgound(MakeHolder<T>())
        {
            Backgound->SetPeriod(period);
            Backgound->SetRobotUserId(USER_ROOT_DEFAULT);
            Backgound->SetEnabled(true);
        }
    };

}
