#pragma once

#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NDrive::NTest {
    static const TString LogColorGreen = "\033[1;32m";
    static const TString LogColorRed = "\033[1;31m";
    static const TString LogColorNo = "\033[0m";

    class TRTContext {
    private:
        THolder<TInstantGuard> InstantGuard;
        NDrive::TServerConfigGenerator& ConfigGenerator;
        NDrive::TServerGuard& Server;

        TEnvironmentGenerator EGenerator;
        TAtomicSharedPtr<TTelematicServerBuilder> TelematicServerBuilder;
        NDrive::TCarDriver Driver;
        TMap<TString, TAtomicSharedPtr<NDrive::TCarEmulator>> Emulators;
        TMap<TString, TEnvironmentGenerator::TCar> NamedCars;
        TMap<TString, NJson::TJsonValue> NamedCarsOffers;

        R_READONLY(TEnvironmentGenerator::TCar, Car);
        R_FIELD(TVector<TString>, PropositionIds);
        R_FIELD(TString, UserId, USER_ID_DEFAULT);
        R_FIELD(bool, MultiSession, false);
        R_OPTIONAL(NJson::TJsonValue, OfferJson);
        R_OPTIONAL(TString, DeviceId);
        R_OPTIONAL(TString, CurrentSessionId);

    public:
        TEnvironmentGenerator::TCar& MutableCar() {
            return Car;
        }

        void SetOfferJson(const NJson::TJsonValue& offerJson, const TString& carName) {
            if (!carName) {
                SetOfferJson(offerJson);
            } else {
                NamedCarsOffers[carName] = offerJson;
            }
        }

        const TString& GetOfferId(const TString& carName = "") const {
            if (!carName) {
                return OfferJson.Defined() ?  (*OfferJson)["offer_id"].GetString() : Default<TString>();
            }
            auto it = NamedCarsOffers.find(carName);
            if (it != NamedCarsOffers.end()) {
                return it->second["offer_id"].GetString();
            }
            return Default<TString>();
        }

        TRTContext(NDrive::TServerConfigGenerator& configGenerator, NDrive::TServerGuard& server, TAtomicSharedPtr<TTelematicServerBuilder> telematicsServerBuilder = nullptr);

        void RestartServer() {
            Server->Stop(0);
            Server->Run();
        }

        const TEnvironmentGenerator::TCar* GetNamedCar(const TString& name) const {
            auto it = NamedCars.find(name);
            if (it == NamedCars.end()) {
                return nullptr;
            }
            return &(it->second);
        }

        TEnvironmentGenerator::TCar* MutableNamedCar(const TString& name) {
            auto it = NamedCars.find(name);
            if (it == NamedCars.end()) {
                return nullptr;
            }
            return &(it->second);
        }

        void SetMileage(const double value) {
            Checked(TelematicServerBuilder)->SetSensorDoubleValue(*GetEmulator(), CAN_ODOMETER_KM, value);
            UNIT_ASSERT(ConfigGenerator.WaitSensor(Car.Id, "mileage", ToString(value)));
        }

        void SetFuelLevel(const ui64 value) {
            Checked(TelematicServerBuilder)->SetSensorValue<ui64>(*GetEmulator(), CAN_FUEL_LEVEL_P, value);
            UNIT_ASSERT(ConfigGenerator.WaitSensor(Car.Id, "fuel_level", ToString(value)));
        }

        TRTContext& AddNamedCar(const TString& name, const TString& imei, const TString& carId) {
            TEnvironmentGenerator::TCar car;
            car.IMEI = imei;
            car.Id = carId;
            return AddNamedCar(name, car);
        }

        TRTContext& AddNamedCar(const TString& name, const TEnvironmentGenerator::TCar& car) {
            auto it = NamedCars.insert(std::make_pair(name, car));
            UNIT_ASSERT(it.second);
            InitEmulator(car.IMEI);
            return *this;
        }

        TRTContext& SetCar(const TString& imei, const TString& id) {
            Car.IMEI = imei;
            Car.Id = id;
            SetCar(Car);
            return *this;
        }

        TRTContext& SetCar(const TEnvironmentGenerator::TCar& car) {
            Car = car;
            InitEmulator(car.IMEI);
            UNIT_ASSERT(ConfigGenerator.WaitCar(car.Id));
            return *this;
        }

        TRTContext& InitEmulator(const TString& imei) {
            if (!Emulators.contains(imei)) {
                Emulators.emplace(imei, Checked(TelematicServerBuilder)->BuildEmulator(imei));
            }
            return *this;
        }

        TAtomicSharedPtr<NDrive::TCarEmulator> GetEmulator() {
            if (Emulators.contains(Car.IMEI)) {
                return Emulators.at(Car.IMEI);
            }
            return nullptr;
        }

        TRTContext& DropEmulator(const TString& imei) {
            if (Emulators.contains(imei)) {
                Emulators.erase(imei);
            }
            return *this;
        }

        TEnvironmentGenerator& GetEGenerator() {
            return EGenerator;
        }

        NDrive::TServerGuard& GetServer() {
            return Server;
        }

        NDrive::TServerConfigGenerator& GetConfigGenerator() {
            return ConfigGenerator;
        }

        TRTContext& SetInstantGuard(const TInstant value) {
            if (!InstantGuard) {
                InstantGuard.Reset(new TInstantGuard(value));
            }
            InstantGuard->Set(value);
            return *this;
        }

        void RelocateCar(const TGeoCoord& c, TEnvironmentGenerator::TCar& car) {
            Driver.RelocateCar(car.IMEI, c);
            UNIT_ASSERT(ConfigGenerator.WaitLocation(car.Id, c));
        }

        const TDriveAPI& GetDriveAPI() const {
            return *(Server->GetDriveAPI());
        }
        const NDrive::TServerGuard& GetServer() const {
            return Server;
        }
    };

    class ITestAction {
        R_FIELD(bool, ExpectOK, true);
        R_FIELD(TString, ActionUserId);
        R_FIELD(TString, AlertLandingId);
        R_FIELD(TString, NamedCar);
    private:
        TInstant StartInstant;
    protected:
        virtual void DoExecute(TRTContext& context) = 0;

        TEnvironmentGenerator::TCar& GetCar(TRTContext& context) const {
            if (NamedCar) {
                auto car = context.MutableNamedCar(NamedCar);
                UNIT_ASSERT(car);
                return *car;
            }
            return context.MutableCar();
        }

    public:

        const TString& GetUserId(const TRTContext& context) const {
            if (!!ActionUserId) {
                return ActionUserId;
            } else {
                return context.GetUserId();
            }
        }

        virtual ~ITestAction() = default;
        using TPtr = TAtomicSharedPtr<ITestAction>;
        ITestAction(TInstant startInstant = TInstant::Zero())
            : StartInstant(startInstant)
        {
        }

        virtual void Execute(TRTContext& context) {
            if (StartInstant) {
                context.SetInstantGuard(StartInstant);
            }
            ChangeState(context.GetConfigGenerator().MutableSessionState());
            DoExecute(context);
        }

    private:
        virtual void ChangeState(TString& /*state*/) const {

        }
    };

    class TAPIAction : public ITestAction {
        R_OPTIONAL(TGeoCoord, UserPosition);
        R_FIELD(ui32, Version, 200);
        R_OPTIONAL(TString, UserId);
    private:
        using TBase = ITestAction;
    public:
        using TFactory = NObjectFactory::TObjectFactory<TAPIAction, TString>;

        TAPIAction(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TAPIAction()
            : TBase(Now())
        {}

        virtual TString GetProcessorConfiguration() const = 0;

    protected:
        NUtil::THttpReply GetSendReply(TRTContext& context, const TString& uri, const NJson::TJsonValue& post = NJson::JSON_NULL, const TString& cgi = {}) const {
            NNeh::THttpRequest request;
            request.SetUri(uri).SetPostData(TBlob::FromString(post.GetStringRobust()));
            if (cgi) {
                request.SetCgiData(cgi);
            }
            if (!post.IsNull()) {
                INFO_LOG << post.GetStringRobust() << Endl;
                request.SetRequestType("POST");
                request.SetPostData(post.GetStringRobust());
            }
            if (UserId) {
                request.AddHeader("Authorization", *UserId);
            } else {
                request.AddHeader("Authorization", context.GetUserId());
            }

            if (UserPosition) {
                request.AddHeader("Lon", UserPosition->X);
                request.AddHeader("Lat", UserPosition->Y);
            }
            if (Version) {
                request.AddHeader("TEST_Version", ::ToString(Version));
            }

            auto deviceId = context.GetDeviceIdPtr();
            if (deviceId) {
                request.AddHeader("DeviceId", *deviceId);
            }

            auto reply = context.GetConfigGenerator().GetSendReply(request);

            if (reply.Code() % 100 != 2) {
                ERROR_LOG << request.GetDebugRequest() << Endl;
            }

            return reply;
        }

        NJson::TJsonValue SendRequest(TRTContext& context, const TString& uri, const NJson::TJsonValue& post = NJson::JSON_NULL, const TString& cgi = {}) const {
            NUtil::THttpReply result = GetSendReply(context, uri, post, cgi);
            NJson::TJsonValue resultReport = NJson::JSON_MAP;
            if (result.Code() != 200 || !NJson::ReadJsonFastTree(result.Content(), &resultReport)) {
                ERROR_LOG << result.GetDebugReply() << ": " << result.Code() << ": " << result.Content() << Endl;
                return NJson::JSON_NULL;
            }
            return resultReport;
        }
    };

    class TScript {
    private:
        TMap<TInstant, TVector<ITestAction::TPtr>> Actions;
        TInstant Start;
        TInstant CurrentTS;
        NDrive::TServerConfigGenerator& ConfigGenerator;
        THolder<NDrive::TServerConfig> Config;
        THolder<NDrive::TServerGuard> Server;
    public:
        TScript(NDrive::TServerConfigGenerator& configGenerator, const TInstant start = Now());

        const TDriveAPI& GetDriveAPI() const {
            return *(*Server)->GetDriveAPI();
        }

        template <class T, typename... Args>
        T& Add(Args... args) {
            THolder<T> result(new T(CurrentTS, args...));
            Actions[CurrentTS].emplace_back(result.Get());
            return *result.Release();
        }

        template <class T, typename... Args>
        T& Add(const TDuration tsDiff, Args... args) {
            CurrentTS = Start + tsDiff;
            return Add<T>(args...);
        }

        template <class T, typename... Args>
        T& AddMove(const TDuration tsDiff, Args... args) {
            CurrentTS = CurrentTS + tsDiff;
            return Add<T>(args...);
        }

        bool Execute();
    };
}
