#pragma once

#include "abstract.h"

namespace NDrive::NTest {
    class TCreateCar: public ITestAction {
    private:
        using TBase = ITestAction;
        R_OPTIONAL(TGeoCoord, Position);
        R_FIELD(TString, Model, "porsche_carrera");
        R_FIELD(TString, Vin);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            if (GetNamedCar()) {
                context.AddNamedCar(GetNamedCar(), context.GetEGenerator().CreateCar(Model, Vin));
            } else {
                context.SetCar(context.GetEGenerator().CreateCar(Model, Vin));
            }
            if (HasPosition()) {
                context.RelocateCar(GetPositionUnsafe(), GetCar(context));
            }
        }
    public:
        using TBase::TBase;
    };

    class TRelocateCar: public ITestAction {
    private:
        using TBase = ITestAction;
        R_OPTIONAL(TGeoCoord, Position);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            UNIT_ASSERT(Position);
            context.RelocateCar(GetPositionUnsafe(), GetCar(context));
        }
    public:
        using TBase::TBase;
    };

    class TCarEmulatorControl: public ITestAction {
    public:
        enum class EAction {
            Create,
            Drop
        };
    private:
        using TBase = ITestAction;
        R_FIELD(EAction, Action, EAction::Create);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            if (Action == EAction::Create) {
                context.InitEmulator(context.GetCar().IMEI);
            } else if (Action == EAction::Drop) {
                context.DropEmulator(context.GetCar().IMEI);
            } else {
                FAIL_LOG("incorrect Action");
            }
        }
    public:
        using TBase::TBase;
    };

    class TCheckCarsReport: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, ObjectId);
        using TChecker = std::function<void(const NJson::TJsonValue& json, TRTContext& context)>;
        R_OPTIONAL(TChecker, Checker);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            NJson::TJsonValue result = context.GetConfigGenerator().GetServiceCarsDetails({!!ObjectId ? ObjectId : context.GetCar().Id}, GetActionUserId());
            INFO_LOG << result << Endl;
            if (Checker) {
                (*Checker)(result, context);
            }

        }
    public:
        TCheckCarsReport(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };

    class TCheckCarVisibility: public ITestAction {
    private:
        using TBase = ITestAction;
        R_OPTIONAL(TString, ObjectId);
        R_FIELD(bool, Visibility, true);
        R_OPTIONAL(ui32, ClusterId);
        R_FIELD(TDuration, WaitingDuration, TDuration::Zero());
    protected:
        virtual void DoExecute(TRTContext& context) override {
            TInstant start = Now();
            do {
                NJson::TJsonValue result = context.GetConfigGenerator().GetServiceCarsDetails({!!ObjectId ? *ObjectId : context.GetCar().Id}, GetActionUserId(), "", ClusterId ? *ClusterId : 0);
                if ((result["cars"].GetArraySafe().size() == 1) == Visibility) {
                    return;
                }
                Sleep(TDuration::Seconds(1));
            } while (Now() - start < WaitingDuration);
            ythrow yexception() << "incorrect visibility for car " << (!!ObjectId ? *ObjectId : context.GetCar().Id) << " / " << Visibility << Endl;
        }
    public:
        TCheckCarVisibility(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };

    class TModifyCarModel: public NDrive::NTest::TAPIAction {
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TModifyCarModel> Registrator;

    private:
        R_FIELD(TString, Model, "porsche_carrera");
        R_FIELD(TString, Name, "Porsche 911 Carrera 4S");
        R_FIELD(TString, Manufacturer, "Porsche");
        R_OPTIONAL(ui64, FirstMaintenanceMileage);
        R_OPTIONAL(ui64, MaintenanceMileage);
        R_OPTIONAL(ui64, IntermediateMaintenanceMileage);
        R_OPTIONAL(TDuration, MaintenancePeriod);

    public:
        using TBase::TBase;

        TString GetProcessorConfiguration() const override;

    protected:
        void DoExecute(TRTContext& context) override;
    };

    class TModifyCarRegistryData: public NDrive::NTest::TAPIAction {
        using TBase = TAPIAction;
        using TValuesMap = TMap<TString, TString>;
        static TFactory::TRegistrator<TModifyCarRegistryData> Registrator;

    private:
        R_FIELD(TValuesMap, Values);

    public:
        using TBase::TBase;

        TString GetProcessorConfiguration() const override;

    protected:
        void DoExecute(TRTContext& context) override;
    };
}
