#pragma once

#include "abstract.h"

namespace NDrive::NTest {

    class TBuildEnv: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(ui32, Traits, TEnvironmentGenerator::DefaultTraits);
        R_FIELD(bool, NeedTelematics, true);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            context.GetEGenerator().SetNeedTelematics(NeedTelematics);
            context.GetEGenerator().BuildEnvironment(Traits);
            context.GetEGenerator().BuildRTBackgrounds(*context.GetServer().Get(), context.GetConfigGenerator().GetNeedBackground());
        }
    public:
        using TBase::TBase;
    };

    class TSetMileage: public ITestAction {
    private:
        using TBase = ITestAction;

    private:
        R_FIELD(double, Value, 0);

    public:
        using TBase::TBase;
        virtual void DoExecute(TRTContext& context) override {
            context.SetMileage(Value);
        }
    };

    class TSetSetting: public ITestAction {
    private:
        using TBase = ITestAction;
        TMap<TString, TString> KV;
    protected:
        virtual void DoExecute(TRTContext& /*context*/) override {
            const auto& settings = NDrive::GetServer().GetSettings();
            for (auto&& i : KV) {
                UNIT_ASSERT(settings.SetValue(i.first, i.second, USER_ROOT_DEFAULT));
            }
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }
    public:
        TSetSetting& Set(const TString& key, const TString& value) {
            KV[key] = value;
            return *this;
        }
        TSetSetting& Reset(const TMap<TString, TString>& map) {
            KV = map;
            return *this;
        }
        using TBase::TBase;
    };

    class TSetDeviceId: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, DeviceId);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            context.SetDeviceId(DeviceId);
        }
    public:
        using TBase::TBase;
    };

   class TSetScriptUser: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, UserId);
    protected:
        virtual void DoExecute(TRTContext& context) override {
            if (!!UserId) {
                context.SetUserId(UserId);
            }
        }
    public:
        TSetScriptUser(const TInstant instant, const TString& userId)
            : TBase(instant)
            , UserId(userId)
        {

        }
        using TBase::TBase;
    };

    class TSwitchObject: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, ObjectId);
    protected:
        virtual void DoExecute(TRTContext& /*context*/) override {
            Y_UNUSED(ObjectId);
            UNIT_ASSERT(false);
        }
    public:
        TSwitchObject(const TInstant instant, const TString& objectId)
            : TBase(instant)
            , ObjectId(objectId)
        {

        }
        using TBase::TBase;
    };

    class TSetMultiSession : public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(bool, Flag, true);
    public:
        TSetMultiSession(const TInstant startInstant, const bool flag = true)
            : TBase(startInstant)
            , Flag(flag)
        {}
    protected:
        virtual void DoExecute(TRTContext& context) override {
            context.SetMultiSession(Flag);
        }
    public:
        using TBase::TBase;
    };

    class TCommonChecker : public ITestAction {
    private:
        using TBase = ITestAction;
        std::function<void(NDrive::NTest::TRTContext&)> Checker;

    protected:
        virtual void DoExecute(TRTContext& context) override {
            Checker(context);
        }

    public:
        TCommonChecker(const TInstant startInstant, const std::function<void(NDrive::NTest::TRTContext&)>& checker)
            : TBase(startInstant)
            , Checker(checker)
        {}
    };

    class TSleepAction: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TDuration, WaitingDuration, TDuration::Seconds(30));
    protected:
        virtual void DoExecute(TRTContext& /*context*/) override {
            Sleep(WaitingDuration);
        }
    public:
        using TBase::TBase;
    };

    class TInitSessionId: public ITestAction {
    private:
        using TBase = ITestAction;
        R_OPTIONAL(TString, UserId);
    protected:

        virtual void DoExecute(TRTContext& context) override {
            const NJson::TJsonValue userSessionReport = context.GetConfigGenerator().GetCurrentSession(UserId ? *UserId : USER_ID_DEFAULT, nullptr, context.GetDeviceIdPtr());
            context.SetCurrentSessionId(userSessionReport["segment"]["meta"]["session_id"].GetStringSafe());
        }
    public:

        TInitSessionId(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };

    class TRestartServer: public ITestAction {
    public:
        using ITestAction::ITestAction;
        virtual void DoExecute(TRTContext& context) override {
            context.RestartServer();
        }
    };

    class TDropCache: public ITestAction {
    private:
        using TBase = ITestAction;
    protected:
        virtual void DoExecute(TRTContext& context) override {
            Y_UNUSED(context);
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }
    public:
        using TBase::TBase;
    };
}
