#include "delegation.h"

namespace NDrive::NTest {
    bool TDelegatedOfferAction::RejectOutgoingDelegation(TRTContext& context, const TString& delegatorId) {
        auto prevContextUser = GetUserIdUnsafe();
        SetUserId(delegatorId);

        NJson::TJsonValue currentSessionReport = SendRequest(context, "/api/yandex/sessions/current", NJson::JSON_NULL, {});
        UNIT_ASSERT(currentSessionReport != NJson::JSON_NULL);

        UNIT_ASSERT_C(currentSessionReport.Has("delegation"), TStringBuilder() << currentSessionReport.GetStringRobust());
        UNIT_ASSERT_C(currentSessionReport["delegation"].Has("tag_id"), TStringBuilder() << currentSessionReport["delegation"].GetStringRobust());
        UNIT_ASSERT_C(currentSessionReport["delegation"]["tag_id"].IsString(), TStringBuilder() << currentSessionReport["delegation"]["tag_id"].GetStringRobust());

        auto tagId = currentSessionReport["delegation"]["tag_id"].GetString();
        UNIT_ASSERT(!!tagId);
        NJson::TJsonValue cancellationReport = SendRequest(context, "/api/yandex/delegation/p2p/reject", NJson::JSON_NULL, "tag_id=" + tagId + "&by_delegator=true");
        UNIT_ASSERT(cancellationReport.Has("status"));
        UNIT_ASSERT(cancellationReport["status"].IsString());
        UNIT_ASSERT_VALUES_EQUAL(cancellationReport["status"].GetString(), "ok");

        SetUserId(prevContextUser);
        return true;
    }

    bool TDelegatedOfferAction::RejectServiceDelegation(TRTContext& context, const TString& carTagId) {
        NJson::TJsonValue currentSessionReport = SendRequest(context, "/api/yandex/delegation/service/reject", NJson::JSON_NULL, "car_tag_id=" + carTagId);
        return currentSessionReport != NJson::JSON_NULL;
    }

    TString TDelegatedOfferAction::GetDelegationTagId(TRTContext& context, const ECarDelegationType type, NJson::TJsonValue* delegationData) {
        NJson::TJsonValue currentSessionReport = SendRequest(context, "/api/yandex/sessions/current", NJson::JSON_NULL, "multi_sessions=1");
        UNIT_ASSERT(currentSessionReport != NJson::JSON_NULL);

        TSet<TString> reportedCarIds;
        for (auto&& carData : currentSessionReport["cars"].GetArray()) {
            UNIT_ASSERT(carData.IsMap());
            auto carId = carData["id"].GetString();
            reportedCarIds.insert(carId);
        }

        for (auto&& delegationReport : currentSessionReport["delegations"]["actual"].GetArray()) {
            if (delegationReport["type"].GetString() == ToString(type)) {
                if (delegationData) {
                    *delegationData = delegationReport;
                }

                if (delegationReport.Has("car_id")) {
                    auto carId = delegationReport["car_id"].GetString();
                    UNIT_ASSERT(reportedCarIds.contains(carId));
                }

                return delegationReport["tag_id"].GetString();
            }
        }
        return "";
    }

    TString TDelegatedOfferAction::GetProcessorConfiguration() const {
        return "";
    }

    void TBookDelegatedOffer::DoExecute(TRTContext& context) {
        auto&& [acceptCgi, acceptPost] = GetDelegatedOfferActionParams(context, true);
        auto offerId = acceptPost["offer_id"].GetString();
        if (!offerId && !GetExpectOK()) {
            return;
        }
        UNIT_ASSERT(!!offerId);
        UNIT_ASSERT(context.GetConfigGenerator().BookOffer(offerId, HasUserId() ? GetUserIdUnsafe() : context.GetUserId(), context.GetDeviceIdPtr(), acceptPost) == GetExpectOK());
    }

    void TRejectAndBookDelegatedOffer::DoExecute(TRTContext& context) {
        auto&& [acceptCgi, acceptPost] = GetDelegatedOfferActionParams(context, true);

        UNIT_ASSERT(RejectOutgoingDelegation(context, GetDelegatorId()));

        auto offerId = acceptPost["offer_id"].GetString();
        UNIT_ASSERT(!!offerId);
        UNIT_ASSERT(!context.GetConfigGenerator().BookOffer(offerId, HasUserId() ? GetUserIdUnsafe() : context.GetUserId(), context.GetDeviceIdPtr(), acceptPost));
    }

    void TRejectOutgoingDelegation::DoExecute(TRTContext& context) {
        UNIT_ASSERT(RejectOutgoingDelegation(context, HasUserId() ? GetUserIdUnsafe() : context.GetUserId()));
    }

    void TCheckDelegationRejected::DoExecute(TRTContext& context) {
        TVector<TDBTag> tags;
        auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();

        auto carId = CarId ? CarId : context.GetCar().Id;

        UNIT_ASSERT_C(context.GetDriveAPI().GetTagsManager().GetDeviceTags().RestoreTags({carId}, {TP2PDelegationTag::TypeName}, tags, session), TStringBuilder() << "Could not restore tags");
        UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);

        auto impl = tags.front().GetTagAs<TP2PDelegationTag>();
        UNIT_ASSERT_VALUES_EQUAL(impl->GetRejected(), Rejected);
        UNIT_ASSERT_VALUES_EQUAL(impl->GetReport(tags.front(), context.GetServer().Get())["p2p_expired"].GetBoolean(), Expired);
    }

    TString TRejectIncomingDelegation::GetProcessorConfiguration() const {
        TStringStream ss;
        ss << "<api/yandex/delegation/p2p/reject>" << Endl;
        ss << "    AuthModuleName: fake" << Endl;
        ss << "    ProcessorType: user_p2p_delegation_reject" << Endl;
        ss << "</api/yandex/delegation/p2p/reject>" << Endl;
        return ss.Str();
    }

    void TRejectIncomingDelegation::DoExecute(TRTContext& context) {
        NJson::TJsonValue currentSessionReport = SendRequest(context, "/api/yandex/sessions/current", NJson::JSON_NULL, {});
        UNIT_ASSERT(currentSessionReport != NJson::JSON_NULL);
        TString incomingDelegationId;
        {
            UNIT_ASSERT(currentSessionReport.Has("incoming_delegations") && currentSessionReport["incoming_delegations"].IsArray());
            auto incomingArray = currentSessionReport["incoming_delegations"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(incomingArray.size(), 1);
            auto incomingRequest = currentSessionReport["incoming_delegations"].GetArray()[0];
            UNIT_ASSERT(incomingRequest.Has("tag_id") && incomingRequest["tag_id"].IsString());
            incomingDelegationId = incomingRequest["tag_id"].GetString();
        }
        NJson::TJsonValue cancellationReport = SendRequest(context, "/api/yandex/delegation/p2p/reject", NJson::JSON_NULL, "tag_id=" + incomingDelegationId);
        UNIT_ASSERT(cancellationReport.Has("status"));
        UNIT_ASSERT(cancellationReport["status"].IsString());
        UNIT_ASSERT_VALUES_EQUAL(cancellationReport["status"].GetString(), "ok");
    }

    std::pair<TString, NJson::TJsonValue> TDelegatedOfferAction::GetDelegatedOfferActionParams(TRTContext& context, const bool accept) {
        TString incomingDelegationId;
        if (!UseQR) {
            NJson::TJsonValue currentSessionReport = SendRequest(context, "/api/yandex/sessions/current", NJson::JSON_NULL, {});
            if (currentSessionReport == NJson::JSON_NULL && !GetExpectOK()) {
                return {};
            }
            UNIT_ASSERT(currentSessionReport != NJson::JSON_NULL);
            {
                UNIT_ASSERT(currentSessionReport.Has("incoming_delegations") && currentSessionReport["incoming_delegations"].IsArray());
                auto incomingArray = currentSessionReport["incoming_delegations"].GetArray();
                UNIT_ASSERT_VALUES_EQUAL(incomingArray.size(), 1);
                auto incomingRequest = currentSessionReport["incoming_delegations"].GetArray()[0];
                UNIT_ASSERT(incomingRequest.Has("tag_id") && incomingRequest["tag_id"].IsString());
                incomingDelegationId = incomingRequest["tag_id"].GetString();
            }
        } else {
            auto tmpUserId = GetUserIdUnsafe();
            SetUserId(DelegatorId);
            NJson::TJsonValue currentSessionReport = SendRequest(context, "/api/yandex/sessions/current", NJson::JSON_NULL, {});
            if (currentSessionReport == NJson::JSON_NULL && !GetExpectOK()) {
                return {};
            }
            UNIT_ASSERT(currentSessionReport != NJson::JSON_NULL);
            UNIT_ASSERT(currentSessionReport.Has("delegation") && currentSessionReport["delegation"].IsMap());
            UNIT_ASSERT_C(currentSessionReport["delegation"].Has("p2p_qr_code"), TStringBuilder() << currentSessionReport["delegation"].GetStringRobust());
            incomingDelegationId = currentSessionReport["delegation"]["p2p_qr_code"].GetString();
            SetUserId(tmpUserId);
        }
        NJson::TJsonValue offerReport = SendRequest(context, "/api/yandex/offers/create", NJson::JSON_NULL, "delegation_payload=" + incomingDelegationId);
        if (offerReport == NJson::JSON_NULL && !GetExpectOK()) {
            return {};
        }
        INFO_LOG << "offer report: " << offerReport.GetStringRobust() << Endl;

        UNIT_ASSERT(offerReport.Has("car"));
        UNIT_ASSERT_C(offerReport["car"].Has("number"), TStringBuilder() << offerReport["car"].GetStringRobust());

        UNIT_ASSERT(offerReport.Has("accept_params") && offerReport["accept_params"].Has("post") && offerReport["accept_params"]["post"].Has("offer_id"));

        NJson::TJsonValue post;
        NJson::TJsonValue cgi;

        if (accept) {
            post = offerReport["accept_params"]["post"];
            cgi = offerReport["accept_params"]["cgi"];
        } else {
            post = offerReport["reject_params"]["post"];
            cgi = offerReport["reject_params"]["cgi"];
        }

        TString cgiStr = "";
        if (cgi.IsDefined()) {
            UNIT_ASSERT(cgi.IsMap());
            for (auto&& it : cgi.GetMap()) {
                cgiStr += it.first + "=" + it.second.GetStringRobust();
            }
        }

        return std::make_pair(cgiStr, post);
    }

    void TDropCarDelegation::DoExecute(TRTContext& context) {
        UNIT_ASSERT_VALUES_EQUAL(context.GetConfigGenerator().RemoveTags({TDelegationUserTag::TypeName}, GetUserId(context), GetUserId(context), NEntityTagsManager::EEntityType::User), GetExpectOK());
    }

    ITag::TPtr TFreeCarDelegation::CreateTag(TRTContext& context) const {
        THolder<TFreeDelegationTag> dTag(new TFreeDelegationTag(TFreeDelegationTag::TypeName));
        dTag->SetBaseUserId(GetUserId(context));
        return dTag.Release();
    }

    void TDelegateCar::DoExecute(TRTContext& context) {
        const TInstant start = Now();
        while (Now() - start < WaitingTime) {
            if (context.GetConfigGenerator().AddTag(CreateTag(context), GetObjectId(context), GetUserId(context), NEntityTagsManager::EEntityType::Car) != GetExpectOK()) {
                continue;
            } else {
                SendGlobalMessage<NDrive::TCacheRefreshMessage>();
                return;
            }
        }
        ythrow yexception() << "cannot delegate";
    }
}
