#pragma once

#include "session.h"

namespace NDrive::NTest {
    class TCheckCarDelegation: public TCheckCurrentSessionWaitingImpl {
    private:
        using TBase = TCheckCurrentSessionWaitingImpl;
        R_OPTIONAL(TString, DelegationTypeExpected);
        R_FIELD(bool, Finished, false);
        R_OPTIONAL(TString, RentStatus);
    protected:
        virtual bool Check(const TRTContext& /*context*/, const NJson::TJsonValue& json) const override {
            INFO_LOG << json << Endl;
            try {
                if (RentStatus.Defined()) {
                    auto status = json["segment"]["session"]["current_performing"].GetString();
                    if (status != *RentStatus) {
                        ERROR_LOG << "rent status mismatch: " << status << " instead of expected " << *RentStatus << Endl;
                        return false;
                    }
                }
                if (Finished) {
                    return json["delegation"].Has("finished") && json["delegation"]["finished"].GetBooleanSafe();
                } else {
                    if (DelegationTypeExpected) {
                        return json["delegation"]["type"].GetStringRobust() == *DelegationTypeExpected;
                    } else {
                        return !json["delegation"].IsDefined();
                    }
                    return !json["delegation"].Has("finished") || !json["delegation"]["finished"].GetBooleanSafe();
                }
            } catch (const std::exception& e) {
                ERROR_LOG << "CheckCarDelegation exception: " << FormatExc(e) << Endl;
                throw;
            }
            return false;
        }
    public:

        TCheckCarDelegation(const TInstant startInstant)
            : TBase(startInstant)
        {

        }

        TCheckCarDelegation(const TInstant startInstant, const TString& delegationTypeExpected)
            : TBase(startInstant)
            , DelegationTypeExpected(delegationTypeExpected) {

        }
    };

    class TCheckDelegationRejected: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, CarId);
        R_FIELD(bool, Rejected, false);
        R_FIELD(bool, Expired, false);

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        TCheckDelegationRejected(const TInstant startInstant)
            : TBase(startInstant)
        {
        }
    };

    class TCheckIncomingCarDelegation: public TCheckCurrentSessionWaitingImpl {
    private:
        using TBase = TCheckCurrentSessionWaitingImpl;
        R_FIELD(TString, CarId);
        R_FIELD(bool, Exists, false);
        R_FIELD(bool, Expired, false);
    protected:
        virtual bool Check(const TRTContext& context, const NJson::TJsonValue& json) const override {
            INFO_LOG << json << Endl;
            auto expectedCarId = CarId ? CarId : context.GetCar().Id;
            try {
                bool invitationPresent = false;
                if (json.Has("incoming_delegations") && json["incoming_delegations"].IsArray()) {
                    INFO_LOG << "has incoming delegations" << Endl;
                    for (auto&& elem : json["incoming_delegations"].GetArray()) {
                        INFO_LOG << "incoming delegation " << elem["object_id"].GetString() << " " << expectedCarId << Endl;
                        if (elem["object_id"].GetString() == expectedCarId) {
                            UNIT_ASSERT(elem.Has("expired") && elem["expired"].IsBoolean());
                            UNIT_ASSERT_VALUES_EQUAL(elem["expired"].GetBoolean(), Expired);
                            invitationPresent = true;
                            break;
                        }
                    }
                }
                INFO_LOG << invitationPresent << " " << Exists << Endl;
                return invitationPresent == Exists;
            } catch (const std::exception& e) {
                ERROR_LOG << "CheckIncomingCarDelegation exception: " << FormatExc(e) << Endl;
                throw;
            }
            return false;
        }
    public:
        using TBase::TBase;
    };

    class TDelegatedOfferAction: public TAPIAction {
    private:
        using TBase = TAPIAction;

        R_FIELD(bool, UseQR, false);
        R_FIELD(TString, DelegatorId);

    protected:
        virtual bool RejectOutgoingDelegation(TRTContext& context, const TString& delegatorId);
        virtual std::pair<TString, NJson::TJsonValue> GetDelegatedOfferActionParams(TRTContext& context, const bool accept);
        virtual TString GetDelegationTagId(TRTContext& context, const ECarDelegationType type, NJson::TJsonValue* delegationData = nullptr);
        virtual bool RejectServiceDelegation(TRTContext& context, const TString& carTagId);

        void DoExecute(TRTContext& context) override = 0;

    public:
        using TBase::TBase;

        virtual TString GetProcessorConfiguration() const override;
    };

    class TBookDelegatedOffer: public TDelegatedOfferAction {
    private:
        using TBase = TDelegatedOfferAction;
        static TFactory::TRegistrator<TBookDelegatedOffer> Registrator;

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;
    };

    class TRejectAndBookDelegatedOffer: public TDelegatedOfferAction {
    private:
        using TBase = TDelegatedOfferAction;
        static TFactory::TRegistrator<TRejectAndBookDelegatedOffer> Registrator;

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;
    };

    class TRejectOutgoingDelegation: public TDelegatedOfferAction {
    private:
        using TBase = TDelegatedOfferAction;
        static TFactory::TRegistrator<TRejectOutgoingDelegation> Registrator;

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;
    };

    class TRejectIncomingDelegation: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TRejectIncomingDelegation> Registrator;

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;

        virtual TString GetProcessorConfiguration() const override;
    };

    class TViewFinishedDelegation: public ITestAction {
    private:
        using TBase = ITestAction;

        R_FIELD(TDuration, WaitingDuration, TDuration::Minutes(1));
        R_OPTIONAL(TGeoCoord, Position);
        R_OPTIONAL(TString, UserId);
        R_OPTIONAL(TString, State);
        R_OPTIONAL(TString, P2PUserId);

    protected:
        virtual void DoExecute(TRTContext& context) override {
            bool success = false;
            for (TInstant start = Now(); Now() - start < WaitingDuration; Sleep(TDuration::Seconds(1))) {
                NDrive::TServerConfigGenerator::TDisableLogging disableLogging(context.GetConfigGenerator());
                const NJson::TJsonValue userSessionReport = context.GetConfigGenerator().GetCurrentSession(UserId ? *UserId : USER_ID_DEFAULT, Position.Get(), context.GetDeviceIdPtr());
                if (GetExpectOK() != userSessionReport.IsDefined()) {
                    continue;
                }
                if (!GetExpectOK()) {
                    success = true;
                    break;
                }
                if (Check(context, userSessionReport)) {
                    success = true;
                    break;
                }
            }
            UNIT_ASSERT(success);
            {
                NDrive::TServerConfigGenerator::TSessionStateGuard sg(context.GetConfigGenerator(), State);
                context.GetConfigGenerator().GetCurrentSession(UserId ? *UserId : USER_ID_DEFAULT, Position.Get(), context.GetDeviceIdPtr());
            }
        }

        virtual bool Check(TRTContext& context, const NJson::TJsonValue& json) {
            INFO_LOG << json << Endl;
            TString tagId;
            try {
                if (!json["delegation"].Has("finished") || !json["delegation"]["finished"].GetBooleanSafe()) {
                    return false;
                }
                if (!json["delegation"].Has("tag_id") || !json["delegation"]["tag_id"].IsString()) {
                    return false;
                }
                if (P2PUserId.Defined()) {
                    auto p2pUserId = json["delegation"]["p2p_user_id"].GetString();
                    UNIT_ASSERT_VALUES_EQUAL(*P2PUserId, p2pUserId);
                }
                tagId = json["delegation"]["tag_id"].GetString();
            } catch (const std::exception& e) {
                ERROR_LOG << "ViewFinishedDelegation exception: " << FormatExc(e) << Endl;
                throw;
            }
            return context.GetConfigGenerator().ViewBillAfterDelegation(tagId, HasUserId() ? GetUserIdUnsafe() : USER_ID_DEFAULT);
        }

    public:
        TViewFinishedDelegation(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };

    class TDropCarDelegation: public ITestAction {
    private:
        R_FIELD(TString, ObjectId);
        R_FIELD(TString, UserId);

    public:
        using ITestAction::ITestAction;

        const TString& GetObjectId(const TRTContext& context) const {
            if (!!ObjectId) {
                return ObjectId;
            } else {
                return context.GetCar().Id;
            }
        }

        const TString& GetUserId(const TRTContext& context) const {
            if (!!UserId) {
                return UserId;
            } else {
                return context.GetUserId();
            }
        }

        virtual void DoExecute(TRTContext& context) override;
    };

    class TDelegateCar: public ITestAction {
    private:
        R_FIELD(TString, ObjectId);
        R_FIELD(TString, UserId);
        R_FIELD(TDuration, WaitingTime, TDuration::Seconds(10));
    public:
        using ITestAction::ITestAction;
        const TString& GetObjectId(const TRTContext& context) const {
            if (!!ObjectId) {
                return ObjectId;
            } else {
                return context.GetCar().Id;
            }
        }

        const TString& GetUserId(const TRTContext& context) const {
            if (!!UserId) {
                return UserId;
            } else {
                return context.GetUserId();
            }
        }

        virtual ITag::TPtr CreateTag(TRTContext& context) const = 0;

        virtual void DoExecute(TRTContext& context) override;
    };

    class TFreeCarDelegation: public TDelegateCar {
    public:
        using TDelegateCar::TDelegateCar;
        virtual ITag::TPtr CreateTag(TRTContext& context) const override;
    };

    class TP2PCarDelegation: public TEvolution {
    private:
        using TBase = TEvolution;

    private:
        R_FIELD(TString, P2PUserId);
        R_FIELD(bool, PreserveOffer, false);

    protected:
        virtual TString GetState() const override {
            return "old_state_reservation";
        }

        virtual NJson::TJsonValue GetInitialPost() const override {
            NJson::TJsonValue result;
            result["delegation"]["type"] = PreserveOffer ? "p2p_pass_offer" : "p2p";
            result["delegation"]["user_id"] = P2PUserId;
            return result;
        }

    public:
        using TBase::TBase;
    };
}
