#pragma once

#include "abstract.h"
#include "car.h"

namespace NDrive::NTest {
    class TCreateAndBookOffer: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TCreateAndBookOffer> Registrator;

    private:
        R_OPTIONAL(TGeoCoord, UserDestination);
        R_OPTIONAL(TString, OfferType);
        R_OPTIONAL(TString, OfferName);
        R_FIELD(TString, AccountName);
        R_FIELD(bool, WithHints, false);
        R_FIELD(bool, IsFutures, false);
        R_FIELD(bool, IsOfferWaiting, true);
        R_OPTIONAL(ui32, OffersCount);
        R_FIELD(NJson::TJsonValue, PostJson, NJson::JSON_NULL);
        R_FIELD(TDuration, WaitingDuration, TDuration::Minutes(1));
        R_FIELD(TString, PayloadPatch);

    protected:
        virtual void DoExecute(TRTContext& context) override;

        TString GetProcessorConfiguration() const override;

    public:
        using TBase::TBase;
    };

    class TBookDefault: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TBookDefault> Registrator;

        R_FIELD(TString, AccountName);

    protected:
        virtual void DoExecute(TRTContext& context) override;
        TString GetProcessorConfiguration() const override;

    public:
        using TBase::TBase;
    };

    class TEvolution: public ITestAction {
    private:
        using TBase = ITestAction;

    private:
        R_FIELD(bool, CheckStatus, false);
        R_FIELD(bool, UseSessionId, true);
        R_OPTIONAL(TGeoCoord, CarPosition);
        R_OPTIONAL(EEvolutionMode, EvolutionMode);

    protected:
        virtual TString GetWaitingStatus() const {
            return GetState();
        }
        virtual TString GetState() const = 0;
        virtual void DoExecute(TRTContext& context) override;
        virtual void ChangeState(TString& state) const override;

        virtual NJson::TJsonValue GetInitialPost() const {
            return NJson::JSON_MAP;
        }

    public:
        TEvolution(const TInstant start)
            : ITestAction(start)
        {
            SetActionUserId("");
        }
    };

    class TAccept: public TEvolution {
    private:
        using TBase = TEvolution;

    protected:
        virtual TString GetState() const override {
            return "old_state_acceptance";
        }

    public:
        using TBase::TBase;
    };

    class TRide: public TEvolution {
    private:
        using TBase = TEvolution;

    protected:
        virtual TString GetState() const override {
            return "old_state_riding";
        }

    public:
        using TBase::TBase;
    };

    class TPark: public TEvolution {
    private:
        using TBase = TEvolution;

    protected:
        virtual TString GetState() const override {
            return "old_state_parking";
        }

    public:
        using TBase::TBase;
    };

    class TDrop: public TEvolution {
    private:
        using TBase = TEvolution;

    protected:
        virtual TString GetState() const override {
            return "old_state_reservation";
        }
        virtual TString GetWaitingStatus() const override {
            return "available";
        }

    public:
        using TBase::TBase;
    };

    class TServiceDrop: public TDrop {
    private:
        using TBase = TDrop;

    protected:
        virtual NJson::TJsonValue GetInitialPost() const {
            NJson::TJsonValue result;
            result["delegation"]["type"] = "service";
            return result;
        }

    public:
        using TBase::TBase;
    };

    class TCreateAndCheckOffer: public TAPIAction {
    private:
        using TBase = TAPIAction;

    private:
        R_OPTIONAL(TGeoCoord, UserDestination);
        R_OPTIONAL(TString, OfferType);
        R_OPTIONAL(TString, OfferName);
        R_FIELD(TString, AccountName);
        R_FIELD(bool, IsFutures, false);
        R_FIELD(bool, OnlyCheck, false);
        R_FIELD(TDuration, FuturesWaitingDuration, TDuration::Minutes(1));
        using TChecker = std::function<void(const NJson::TJsonValue& jsonOffer, TRTContext& context)>;
        R_OPTIONAL(TChecker, Checker);

    protected:
        virtual void DoExecute(TRTContext& context) override;
        TString GetProcessorConfiguration() const override {
            return "";
        }

    public:
        using TBase::TBase;
    };

    class TCreateAndSwitchOffer: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TCreateAndSwitchOffer> Registrator;

    private:
        R_OPTIONAL(TGeoCoord, UserDestination);
        R_OPTIONAL(TString, OfferType);
        R_OPTIONAL(TString, OfferName);
        R_FIELD(TString, AccountName);
        R_FIELD(TDuration, WaitingDuration, TDuration::Minutes(1));

    protected:
        virtual void DoExecute(TRTContext& context) override;

        TString GetProcessorConfiguration() const override;

    public:
        using TBase::TBase;
    };

    class TCheckPrice: public ITestAction {
    private:
        using TBase = ITestAction;
        using TCalcer = std::function<ui32(const NJson::TJsonValue& jsonOffer)>;
        TCalcer Calcer;
        R_FIELD(ui32, Precision, 1);

    protected:
        virtual void DoExecute(TRTContext& context) override {
            const ui32 price = Calcer(context.GetOfferJsonUnsafe());
            UNIT_ASSERT(context.GetConfigGenerator().WaitPrice(price, Precision));
        }

    public:
        TCheckPrice(const TInstant startInstant, const TCalcer calcer)
            : TBase(startInstant)
            , Calcer(calcer)
        {
        }
    };

    class TCheckCurrentSession: public ITestAction {
    private:
        using TBase = ITestAction;
        using TChecker = std::function<void(const NJson::TJsonValue& json)>;
        R_OPTIONAL(TChecker, Checker);
        R_OPTIONAL(TGeoCoord, Position);
        R_OPTIONAL(TString, UserId);
        R_OPTIONAL(TString, State);

    protected:
        virtual void DoExecute(TRTContext& context) override {
            NJson::TJsonValue userSessionReport;
            NDrive::TServerConfigGenerator::TSessionStateGuard sg(context.GetConfigGenerator(), State);
            userSessionReport = context.GetConfigGenerator().GetCurrentSession(UserId ? *UserId : USER_ID_DEFAULT, Position.Get(), context.GetDeviceIdPtr());
            UNIT_ASSERT_VALUES_EQUAL(GetExpectOK(), userSessionReport.IsDefined());
            if (!GetExpectOK()) {
                return;
            }
            if (Checker) {
                (*Checker)(userSessionReport);
            }
        }

    public:
        TCheckCurrentSession(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TCheckCurrentSession(const TInstant startInstant, const TChecker checker)
            : TBase(startInstant)
            , Checker(checker)
        {
        }
    };

    class TCheckCurrentSessionWaitingImpl: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TDuration, WaitingDuration, TDuration::Minutes(1));
        R_OPTIONAL(TGeoCoord, Position);
        R_OPTIONAL(TString, UserId);
        R_OPTIONAL(TString, State);

    protected:
        virtual bool Check(const TRTContext& context, const NJson::TJsonValue& json) const = 0;

        virtual void DoExecute(TRTContext& context) override {
            bool success = false;
            for (TInstant start = Now(); Now() - start < WaitingDuration; Sleep(TDuration::Seconds(1))) {
                NDrive::TServerConfigGenerator::TDisableLogging disableLogging(context.GetConfigGenerator());
                const NJson::TJsonValue userSessionReport = context.GetConfigGenerator().GetCurrentSession(UserId ? *UserId : USER_ID_DEFAULT, Position.Get(), context.GetDeviceIdPtr(), context.GetMultiSession());
                if (GetExpectOK() != userSessionReport.IsDefined()) {
                    continue;
                }
                if (!GetExpectOK()) {
                    success = true;
                    break;
                }
                if (Check(context, userSessionReport)) {
                    success = true;
                    break;
                }
            }
            UNIT_ASSERT(success);
            {
                NDrive::TServerConfigGenerator::TSessionStateGuard sg(context.GetConfigGenerator(), State);
                context.GetConfigGenerator().GetCurrentSession(UserId ? *UserId : USER_ID_DEFAULT, Position.Get(), context.GetDeviceIdPtr());
            }
        }

    public:
        TCheckCurrentSessionWaitingImpl(const TInstant startInstant)
            : TBase(startInstant)
        {
        }
    };

    class TCheckCurrentSessionWaiting: public TCheckCurrentSessionWaitingImpl {
    private:
        using TBase = TCheckCurrentSessionWaitingImpl;
        using TChecker = std::function<bool(const TRTContext& context, const NJson::TJsonValue& json)>;
        R_OPTIONAL(TChecker, Checker);

    protected:
        virtual bool Check(const TRTContext& context, const NJson::TJsonValue& json) const override {
            return !Checker || (*Checker)(context, json);
        }

    public:
        TCheckCurrentSessionWaiting(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TCheckCurrentSessionWaiting(const TInstant startInstant, const TChecker checker)
            : TBase(startInstant)
            , Checker(checker)
        {
        }
    };

    class TCheckSessionHistoryWaitingImpl: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TDuration, WaitingDuration, TDuration::Minutes(1));
        R_OPTIONAL(TString, UserId);

    protected:
        virtual bool Check(const TRTContext& context, const NJson::TJsonValue& json) const = 0;

        virtual void DoExecute(TRTContext& context) override {
            bool success = false;
            for (TInstant start = Now(); Now() - start < WaitingDuration; Sleep(TDuration::Seconds(1))) {
                NDrive::TServerConfigGenerator::TDisableLogging disableLogging(context.GetConfigGenerator());
                const NJson::TJsonValue userSessionReport = context.GetConfigGenerator().GetUserSessionById(context.GetCurrentSessionIdUnsafe(), UserId ? *UserId : USER_ID_DEFAULT);
                if (GetExpectOK() != userSessionReport.IsDefined()) {
                    continue;
                }
                if (!GetExpectOK()) {
                    success = true;
                    break;
                }
                if (Check(context, userSessionReport)) {
                    success = true;
                    break;
                }
            }
            UNIT_ASSERT(success);
        }

    public:
        TCheckSessionHistoryWaitingImpl(const TInstant startInstant)
            : TBase(startInstant)
        {
        }
    };

    class TCheckSessionHistoryWaiting: public TCheckSessionHistoryWaitingImpl {
    private:
        using TBase = TCheckSessionHistoryWaitingImpl;
        using TChecker = std::function<bool(const TRTContext& context, const NJson::TJsonValue& json)>;
        R_OPTIONAL(TChecker, Checker);

    protected:
        virtual bool Check(const TRTContext& context, const NJson::TJsonValue& json) const override {
            return !Checker || (*Checker)(context, json);
        }

    public:
        TCheckSessionHistoryWaiting(const TInstant startInstant)
            : TBase(startInstant)
        {
        }

        TCheckSessionHistoryWaiting(const TInstant startInstant, const TChecker checker)
            : TBase(startInstant)
            , Checker(checker)
        {
        }
    };

    class TSwitchOffer: public TAPIAction {
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TSwitchOffer> Registrator;

    private:
        R_OPTIONAL(TString, OfferId);

    protected:
        virtual void DoExecute(TRTContext& context) override;

        TString GetProcessorConfiguration() const override;

    public:
        using TBase::TBase;
    };
}
