#include "tag.h"

namespace NDrive::NTest {
    void TAddUserTag::DoExecute(TRTContext& context) {
        UNIT_ASSERT(context.GetConfigGenerator().AddTag(Tag, !!ObjectId ? ObjectId : GetUserId(context), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
    }

    void TAddTag::DoExecute(TRTContext& context) {
        UNIT_ASSERT(context.GetConfigGenerator().AddTag(Tag, !!ObjectId ? ObjectId : GetCar(context).Id, USER_ROOT_DEFAULT, EntityType));
    }

    void TCheckHasTag::DoExecute(TRTContext& context) {
        TString objectId = ObjectId;
        if (!objectId && EntityType == NEntityTagsManager::EEntityType::Car) {
            objectId = GetCar(context).Id;
        }
        if (!objectId && EntityType == NEntityTagsManager::EEntityType::User) {
            objectId = GetUserId(context);
        }
        TVector<TDBTag> tags;
        auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
        UNIT_ASSERT_C(context.GetDriveAPI().GetEntityTagsManager(EntityType).RestoreTags({ objectId }, { TagName }, tags, session), TStringBuilder() << "Could not restore tags");

        if (TargetCount.Defined()) {
            UNIT_ASSERT_VALUES_EQUAL(*TargetCount, tags.size());
        }

        UNIT_ASSERT_VALUES_EQUAL(!tags.empty(), GetExpectOK());
    }

    void TCheckTagPerformer::DoExecute(TRTContext& context) {
        TVector<TDBTag> tags;
        auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
        if (TagId) {
            UNIT_ASSERT_C(context.GetDriveAPI().GetEntityTagsManager(EntityType).RestoreTags({ TagId }, tags, session), "Could not restore tags");
            for (auto&& tag : tags) {
                UNIT_ASSERT_C((tag->GetPerformer() == Performer) == GetExpectOK(), "Tag performer: " + tag->GetPerformer() + " expected: " + Performer);
            }
        }
        if (TagName) {
            UNIT_ASSERT_C(context.GetDriveAPI().GetEntityTagsManager(EntityType).RestoreTags({ ObjectId }, { TagName }, tags, session), "Could not restore tags");
            for (auto&& tag : tags) {
                UNIT_ASSERT_C((tag->GetPerformer() == Performer) == GetExpectOK(), "Tag performer: " + tag->GetPerformer() + " expected: " + Performer);
            }
        }
    }

    void TPerformTag::DoExecute(TRTContext& context) {
        TString objectId = ObjectId ? : context.GetUserId();
        TVector<TDBTag> tags;
        {
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetEntityTagsManager(EntityType).RestoreTags({ objectId }, { TagName }, tags, session), "Could not restore tags");
        }
        auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
        UNIT_ASSERT(context.GetDriveAPI().GetEntityTagsManager(EntityType).SetTagsPerformer(tags, GetActionUserId(), false, session, context.GetServer().Get()));
        UNIT_ASSERT(session.Commit());
    }

    TAPIAction::TFactory::TRegistrator<TAddTagActions> TAddTagActions::Registrator("add_tag");

    void TAddTagActions::DoExecute(TRTContext& context) {
        NJson::TJsonValue post = CustomData;
        post["tag"] = TagName;
        post["object_id"] = ObjectId;
        post["comment"] = Comment;
        NJson::TJsonValue reply = SendRequest(context, "api/staff/" + ::ToString(EntityType) + "/tag/add", post);
        UNIT_ASSERT(!reply.IsNull());
    }

    TString TAddTagActions::GetProcessorConfiguration() const {
        TStringStream ss;
        ss << "<api/staff/car/tag/add>" << Endl;
        ss << "    AuthModuleName: fake" << Endl;
        ss << "    ProcessorType: car_tags/add" << Endl;
        ss << "</api/staff/car/tag/add>" << Endl;
        ss << "<api/staff/user/tag/add>" << Endl;
        ss << "    AuthModuleName: fake" << Endl;
        ss << "    ProcessorType: user_tags/add" << Endl;
        ss << "</api/staff/user/tag/add>" << Endl;
        ss << "<api/staff/area/tag/add>" << Endl;
        ss << "    AuthModuleName: fake" << Endl;
        ss << "    ProcessorType: area_tags/add" << Endl;
        ss << "</api/staff/area/tag/add>" << Endl;
        return ss.Str();
    }

    void TListTagPropositions::DoExecute(TRTContext& context) {
        NJson::TJsonValue report;
        if (!ObjectId) {
            report = context.GetConfigGenerator().ListTags(context.GetCar().Id, USER_ROOT_DEFAULT, EntityType);
        } else {
            report = context.GetConfigGenerator().ListTags(ObjectId, USER_ROOT_DEFAULT, EntityType);
        }
        context.MutablePropositionIds().clear();
        for (const auto& pr : report["propositions"].GetArray()) {
            if (pr.Has("proposition_id")) {
                context.MutablePropositionIds().emplace_back(pr["proposition_id"].GetStringSafe());
            }
        }
        INFO_LOG << report << Endl;
    }

    void TProposeTag::DoExecute(TRTContext& context) {
        UNIT_ASSERT(context.GetConfigGenerator().ProposeTag(Tag, !!ObjectId ? ObjectId : context.GetCar().Id, GetActionUserId(), EntityType));
    }

    void TRejectTag::DoExecute(TRTContext& context) {
        UNIT_ASSERT_VALUES_EQUAL(context.GetPropositionIds().size(), 1);
        UNIT_ASSERT(context.GetConfigGenerator().RejectTag(context.GetPropositionIds().front(), GetActionUserId(), EntityType));
    }

    void TConfirmTag::DoExecute(TRTContext& context) {
        UNIT_ASSERT_VALUES_EQUAL(context.GetPropositionIds().size(), 1);
        UNIT_ASSERT(context.GetConfigGenerator().ConfirmTag(context.GetPropositionIds().front(), GetActionUserId(), EntityType));
    }

    void TRegisterTag::DoExecute(TRTContext& context) {
        UNIT_ASSERT(TagDescription);
        auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
        UNIT_ASSERT(context.GetDriveAPI().GetTagsManager().GetTagsMeta().RegisterTag(TagDescription, USER_ROOT_DEFAULT, session));
        UNIT_ASSERT(session.Commit());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
    }

    void TAddSettingTag::DoExecute(TRTContext& context) {
        auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
        TUserSetting userSetting(ObjectId ? ObjectId : context.GetUserId());
        UNIT_ASSERT(userSetting.SetValue(SettingTagName, SettingTagField, SettingTagValue, GetActionUserId(), session));
        UNIT_ASSERT(session.Commit());
    }
}
