#pragma once

#include "abstract.h"

namespace NDrive::NTest {

    class TAddUserTag: public ITestAction {
    private:
        using TBase = ITestAction;
        ITag::TPtr Tag;
        TString ObjectId;
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:

        TAddUserTag(const TInstant startInstant, ITag::TPtr tag, const TString& objectId = "")
            : TBase(startInstant)
            , Tag(tag)
            , ObjectId(objectId) {

        }

        ITag::TPtr GetTag() const {
            return Tag;
        }

        const TString& GetObjectId() const {
            return ObjectId;
        }
    };

    class TAddTagActions : public TAPIAction {
        R_READONLY(TString, TagName);
        R_READONLY(TString, ObjectId);
        R_FIELD(NJson::TJsonValue, CustomData, {});
        R_FIELD(TString, Comment);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::Car);
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TAddTagActions> Registrator;
    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        using TBase::TBase;

        TAddTagActions(const TInstant startInstant, const TString& tagName, const TString& objectId)
            : TBase(startInstant)
            , TagName(tagName)
            , ObjectId(objectId) {}

        virtual TString GetProcessorConfiguration() const override;
    };

    class TAddTag: public ITestAction {
    private:
        using TBase = ITestAction;
        ITag::TPtr Tag;
        R_FIELD(TString, ObjectId);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::Car);
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:

        TAddTag(const TInstant startInstant, ITag::TPtr tag)
            : TBase(startInstant)
            , Tag(tag) {

        }

        ITag::TPtr GetTag() const {
            return Tag;
        }
    };

    class TStartPerform: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, ObjectId);
        R_FIELD(TString, TagName);
    protected:

        const TString& GetObjectId(const TRTContext& context) const {
            if (ObjectId) {
                return ObjectId;
            } else {
                return context.GetCar().Id;
            }
        }

        virtual void DoExecute(TRTContext& context) override {
            UNIT_ASSERT(!!TagName);
            UNIT_ASSERT_VALUES_EQUAL(context.GetConfigGenerator().StartTag(GetObjectId(context), {TagName}, GetUserId(context)), GetExpectOK());
        }
    public:
        TStartPerform(const TInstant startInstant)
            : TBase(startInstant) {
            SetActionUserId("");
        }
    };

    class TPerformTag: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, ObjectId);
        R_FIELD(TString, TagName);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::User);
    protected:

        virtual void DoExecute(TRTContext& context) override;

    public:
        TPerformTag(const TInstant startInstant)
            : TBase(startInstant) {
        }
    };

    class TDropPerform: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, ObjectId);
        R_FIELD(TString, TagName);
        R_FIELD(bool, RemoveOnFinish, false);
    protected:
        const TString& GetObjectId(const TRTContext& context) const {
            if (ObjectId) {
                return ObjectId;
            } else {
                return context.GetCar().Id;
            }
        }

        virtual void DoExecute(TRTContext& context) override {
            UNIT_ASSERT(!!TagName);
            TMaybe<TTaggedDevice> dev = context.GetServer().Get()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(GetObjectId(context), Now());
            UNIT_ASSERT(!!dev);
            TMaybe<TDBTag> dbTag = dev->GetTag(TagName);
            UNIT_ASSERT(!!dbTag);
            UNIT_ASSERT_VALUES_EQUAL(context.GetConfigGenerator().FinishTag({dbTag->GetTagId()}, GetUserId(context), RemoveOnFinish), GetExpectOK());
        }
    public:
        TDropPerform(const TInstant startInstant)
            : TBase(startInstant) {
            SetActionUserId("");
        }
    };

    class TCheckHasTag: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, ObjectId);
        R_FIELD(TString, TagName);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::User);
        R_OPTIONAL(size_t, TargetCount);

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        TCheckHasTag(const TInstant startInstant)
            : TBase(startInstant)
        {
        }
    };

    template <class TagType>
    class CheckTagsEqual: public NDrive::NTest::ITestAction {
    private:
        using TBase = NDrive::NTest::ITestAction;
        R_FIELD(TagType, CheckTag);

    protected:
        virtual void DoExecute(NDrive::NTest::TRTContext& context) override {
            TVector<TDBTag> tags;
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetTagsManager().GetDeviceTags().RestoreTags({ GetCar(context).Id }, { CheckTag.GetName() }, tags, session), TStringBuilder() << "Could not restore tags");
            UNIT_ASSERT(!tags.empty());
            UNIT_ASSERT_VALUES_EQUAL(tags.front()->SerializeToJson().GetStringRobust(), CheckTag.SerializeToJson().GetStringRobust());
        }

    public:
        using TBase::TBase;
    };


    class TCheckTagPerformer: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, ObjectId, USER_ID_DEFAULT);
        R_FIELD(TString, TagName);
        R_FIELD(TString, TagId);
        R_FIELD(TString, Performer);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::User);
        R_OPTIONAL(size_t, TargetCount);

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        TCheckTagPerformer(const TInstant startInstant)
            : TBase(startInstant)
        {
        }
    };

    class TProposeTag: public ITestAction {
    private:
        using TBase = ITestAction;
        ITag::TPtr Tag;
        R_FIELD(TString, ObjectId);
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::Car);
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:

        TProposeTag(const TInstant startInstant, ITag::TPtr tag)
            : TBase(startInstant)
            , Tag(tag) {

        }

        ITag::TPtr GetTag() const {
            return Tag;
        }
    };

    class TListTagPropositions: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::Car);
        R_FIELD(TString, ObjectId);
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:

        TListTagPropositions(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };

    class TRejectTag: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::Car);
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:

        TRejectTag(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };

    class TConfirmTag: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(NEntityTagsManager::EEntityType, EntityType, NEntityTagsManager::EEntityType::Car);
    protected:
        virtual void DoExecute(TRTContext& context) override;
    public:

        TConfirmTag(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };

    class TRegisterTag: public ITestAction {
    private:
        using TBase = ITestAction;
        TTagDescription::TPtr TagDescription;

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        TRegisterTag(const TInstant startInstant, TTagDescription::TPtr tagDescription)
            : TBase(startInstant)
            , TagDescription(tagDescription)
        {
        }

        TRegisterTag(const TInstant startInstant, const NJson::TJsonValue& descriptionJson)
            : TBase(startInstant)
        {
            NStorage::TTableRecord tRecord;
            UNIT_ASSERT(tRecord.DeserializeFromJson(descriptionJson));
            TagDescription = TTagDescription::ConstructFromTableRecord(tRecord);
            UNIT_ASSERT(TagDescription);
        }

        TTagDescription::TPtr GetTagDescription() const {
            return TagDescription;
        }
    };

    class TAddSettingTag: public ITestAction {
    private:
        using TBase = ITestAction;
        R_FIELD(TString, SettingTagName);
        R_FIELD(TString, SettingTagField);
        R_FIELD(TString, SettingTagValue);
        R_FIELD(TString, ObjectId);

    protected:
        virtual void DoExecute(TRTContext& context) override;

    public:
        TAddSettingTag(const TInstant startInstant, const TString& settingTagName, const TString& settingTagField, const TString& settingTagValue, const TString& objectId = "")
            : TBase(startInstant)
            , SettingTagName(settingTagName)
            , SettingTagField(settingTagField)
            , SettingTagValue(settingTagValue)
            , ObjectId(objectId)
        {
        }
    };
}
