#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/actions/filter.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/long_term.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/data/rental/rental_service_mode_tag.h>
#include <drive/backend/data/rental/timetable_builder.h>
#include <drive/backend/data/rental/timetable_builder_impl.h>
#include <drive/backend/offers/actions/long_term.h>
#include <drive/backend/sessions/manager/billing.h>

#include <library/cpp/iterator/enumerate.h>
#include <library/cpp/testing/unittest/registar.h>

inline const TString LongTermAutoAssignedTagName = "long_term_aa_preparation";

inline const TString SelectCarTagName = "long_term_select_car";
inline const TString PrepareCarTagName = "long_term_prepare_car";
inline const TString PrepareAaCarTagName = "long_term_prepare_aa_car";

inline const TString SimpleLongTermOfferName = "simple_long_term_offer";
inline const TGeoCoord LongTermDeliveryLocation = {37.640373, 55.736713};
inline const TGeoCoord LongTermDeliveryLocation2 = {37.648888, 55.731414};
inline const TString LongTermTier1TagName = "long_term_standards_tier_1";
inline const TString LongTermTier2TagName = "long_term_standards_tier_2";

const i64 EarlyReturnCost = 2000;

void RegisterLongTermTags(const NDrive::IServer& server) {
    auto basic = MakeAtomicShared<TLongTermOfferHolderTag::TDescription>();
    basic->SetName(TLongTermOfferHolderTag::Type());
    basic->SetType(TLongTermOfferHolderTag::Type());
    basic->SetCommunicationTag(SelectCarTagName);
    basic->SetStartBillingTask(true);
    RegisterTag(server, basic);

    auto tagDescription = MakeAtomicShared<TLongTermOfferHolderTag::TDescription>();
    tagDescription->SetName(TLongTermOfferHolderTag::Preparation);
    tagDescription->SetType(TLongTermOfferHolderTag::Type());
    tagDescription->MutableCarTags().push_back(TLongTermCarDeliveryTag::Type());
    tagDescription->MutableUserTags().push_back("test_push_smile");
    RegisterTag(server, tagDescription);

    auto serviceModeTagDescription = MakeAtomicShared<TRentalServiceModeTag::TDescription>();
    serviceModeTagDescription->SetName(TRentalServiceModeTag::Type());
    serviceModeTagDescription->SetType(TRentalServiceModeTag::Type());
    RegisterTag(server, serviceModeTagDescription);

    tagDescription->SetName(LongTermAutoAssignedTagName);
    RegisterTag(server, tagDescription);

    auto communicationLongTermSelectCar = MakeAtomicShared<TSupportOutgoingCommunicationTag::TDescription>();
    communicationLongTermSelectCar->SetName(SelectCarTagName);
    communicationLongTermSelectCar->SetType(TSupportOutgoingCommunicationTag::TypeName);
    communicationLongTermSelectCar->SetLinkedTagName(TLongTermOfferHolderTag::Type());
    RegisterTag(server, communicationLongTermSelectCar);

    auto communicationLongTermPrepareCar = MakeAtomicShared<TSupportOutgoingCommunicationTag::TDescription>();
    communicationLongTermPrepareCar->SetName(PrepareCarTagName);
    communicationLongTermPrepareCar->SetType(TSupportOutgoingCommunicationTag::TypeName);
    communicationLongTermPrepareCar->SetLinkedTagName(TLongTermOfferHolderTag::Preparation);
    RegisterTag(server, communicationLongTermPrepareCar);

    auto communicationLongTermPrepareAaCar = MakeAtomicShared<TSupportOutgoingCommunicationTag::TDescription>();
    communicationLongTermPrepareAaCar->SetName(PrepareAaCarTagName);
    communicationLongTermPrepareAaCar->SetType(TSupportOutgoingCommunicationTag::TypeName);
    communicationLongTermPrepareAaCar->SetLinkedTagName(LongTermAutoAssignedTagName);
    RegisterTag(server, communicationLongTermPrepareAaCar);

    auto tier = MakeAtomicShared<TServiceTagRecord::TDescription>();
    tier->SetName(LongTermTier1TagName);
    tier->SetType(TDeviceTagRecord::TypeName);
    RegisterTag(server, tier);
    tier->SetName(LongTermTier2TagName);
    RegisterTag(server, tier);

    auto replacing = MakeAtomicShared<TTagDescription>();
    replacing->SetName(TReplaceCarTag::TypeName);
    replacing->SetType(TReplaceCarTag::TypeName);
    RegisterTag(server, replacing);
}

void RegisterLongTermOffer(const NDrive::IServer& server, bool autohide = false) {
    auto action = MakeAtomicShared<TLongTermOfferBuilder>();
    action->SetName(SimpleLongTermOfferName);
    action->SetDescription(SimpleLongTermOfferName);
    action->SetDetailedDescription("this is a detailed description");
    action->SetChargableAccounts({ "card" });
    action->SetCarModel("audi_q3_quattro");
    action->SetCarTagsFilter(LongTermTier1TagName);
    action->SetAutohide(autohide);
    action->SetNew(true);
    action->SetOfferTags({ "model_qashqai" });
    action->SetTargetUserTags({ "test_push" });
    action->SetGrouppingTags({ SimpleLongTermOfferName });
    action->SetSwitchOfferTagsList({ SimpleLongTermOfferName });
    action->SetReplaceOfferBuildingAction("pack_offer_minutes");
    action->SetEarlyReturnCost(EarlyReturnCost);

    auto groupAction = MakeAtomicShared<TLongTermOfferBuilder>();
    groupAction->SetName("GroupOfferBuilder");
    groupAction->SetDescription("GroupOfferBuilder");
    groupAction->SetDetailedDescription("this is a detailed description");
    groupAction->SetOfferTags({ "model_qashqai" });
    groupAction->SetTargetUserTags({ "test_push" });;
    groupAction->SetEarlyReturnCost(EarlyReturnCost);
    groupAction->SetGroupOfferBuilder(true);

    auto subaction1 = MakeAtomicShared<TLongTermOfferBuilder>();
    subaction1->SetName("LongTermSuboffer1");
    subaction1->SetDescription("LongTermSuboffer1");
    subaction1->SetDetailedDescription("this is a detailed description");
    subaction1->SetChargableAccounts({ "card" });
    subaction1->SetCarModel("audi_q3_quattro");
    subaction1->SetCarTagsFilter(LongTermTier1TagName);
    subaction1->SetAutohide(autohide);
    subaction1->SetNew(true);
    subaction1->SetOfferTags({ "model_qashqai" });
    subaction1->SetTargetUserTags({ "test_push" });;
    subaction1->SetEarlyReturnCost(EarlyReturnCost);
    subaction1->SetActionParent("GroupOfferBuilder");

    auto subaction2 = MakeAtomicShared<TLongTermOfferBuilder>();
    subaction2->SetName("LongTermSuboffer2");
    subaction2->SetDescription("LongTermSuboffer2");
    subaction2->SetDetailedDescription("this is a detailed description");
    subaction2->SetChargableAccounts({ "card" });
    subaction2->SetCarModel("audi_q3_quattro");
    subaction2->SetCarTagsFilter(LongTermTier1TagName);
    subaction2->SetAutohide(autohide);
    subaction2->SetNew(true);
    subaction2->SetOfferTags({ "model_qashqai" });
    subaction2->SetTargetUserTags({ "test_push" });;
    subaction2->SetEarlyReturnCost(EarlyReturnCost);
    subaction2->SetActionParent("GroupOfferBuilder");

    auto tagAction = MakeAtomicShared<TTagAction>();
    tagAction->SetName("view_offer_holder_user_tag");
    tagAction->SetTagName(TLongTermOfferHolderTag::Type());
    tagAction->AddTagAction(NTagActions::ETagAction::Observe);

    auto viewPreparation = MakeAtomicShared<TTagAction>();
    viewPreparation->SetName("view_long_term_preparation");
    viewPreparation->SetTagName("$" + TLongTermOfferHolderTag::Preparation + "|" + LongTermAutoAssignedTagName);
    viewPreparation->AddTagAction(NTagActions::ETagAction::Observe);
    viewPreparation->AddTagAction(NTagActions::ETagAction::Update);

    auto manipulateCommunicationTags = MakeAtomicShared<TTagAction>();
    manipulateCommunicationTags->SetName("manipulate_communication_tags");
    manipulateCommunicationTags->SetTagName("$" + SelectCarTagName + "|" + PrepareCarTagName + "|" + PrepareAaCarTagName);
    manipulateCommunicationTags->AddTagAction(NTagActions::ETagAction::DropPerform);
    manipulateCommunicationTags->AddTagAction(NTagActions::ETagAction::ForcePerform);

    RegisterAction(server, action, "pack_access");
    RegisterAction(server, groupAction, "pack_access");
    RegisterAction(server, subaction1, "pack_access");
    RegisterAction(server, subaction2, "pack_access");
    RegisterAction(server, tagAction, "pack_access");
    RegisterAction(server, viewPreparation, "pack_access");
    RegisterAction(server, manipulateCommunicationTags, "pack_access");
    server.GetSettings().SetValue("warning_screens.events.long_term_replace.long_term_replace_check.enabled", "true", USER_ROOT_DEFAULT);
    server.GetSettings().SetValue("warning_screens.checkers.long_term_replace_check.landings.main", "{}", USER_ROOT_DEFAULT);
}

void RegisterLongTermFilter(const NDrive::IServer& server) {
    auto filter = MakeAtomicShared<TFilterAction>("lt_filter", "model_qashqai");
    filter->SetDescription("lt_filter");
    filter->SetExplicit(true);
    filter->SetText("Base");
    filter->SetIcon("Icon");
    RegisterAction(server, filter, "pack_access");
}

void RegisterLongTermEvolutions(const NDrive::IServer& server) {
    auto from = TLongTermOfferHolderTag::Type();
    auto to = TOfferBookingUserTag::Type();
    auto book = MakeAtomicShared<TTagEvolutionAction>(from + "->" + to);
    book->SetTagNameFrom(from).SetTagNameTo(to);
    book->SetOnlyPerformed(false);

    auto prepare = MakeAtomicShared<TTagEvolutionAction>(from + "->" + TLongTermOfferHolderTag::Preparation);
    prepare->SetTagNameFrom(from).SetTagNameTo(TLongTermOfferHolderTag::Preparation);
    prepare->SetOnlyPerformed(false);
    prepare->SetTwoSideEvolution(true);

    auto book2 = MakeAtomicShared<TTagEvolutionAction>(TLongTermOfferHolderTag::Preparation + "->" + to);
    book2->SetTagNameFrom(TLongTermOfferHolderTag::Preparation).SetTagNameTo(to);
    book2->SetOnlyPerformed(false);

    auto communicationPrepare = MakeAtomicShared<TTagEvolutionAction>(SelectCarTagName + "->" + PrepareCarTagName);
    communicationPrepare->SetTagNameFrom(SelectCarTagName).SetTagNameTo(PrepareCarTagName);
    communicationPrepare->SetOnlyPerformed(false);
    communicationPrepare->SetTwoSideEvolution(true);

    RegisterAction(server, book, "pack_access");
    RegisterAction(server, book2, "pack_access");
    RegisterAction(server, prepare, "pack_access");
    RegisterAction(server, communicationPrepare, "pack_access");
}

void RegisterLongTermAll(const NDrive::IServer& server) {
    RegisterLongTermOffer(server);
    RegisterLongTermFilter(server);
    RegisterLongTermTags(server);
    RegisterLongTermEvolutions(server);
}

TInstant Day(int days) {
    return TInstant::ParseIso8601("2010-01-01T00:00:00") + TDuration::Days(days);
}

void CreateOfferWithAssignedCarAndOptionalDates(TTestEnvironment& env, const auto carId, const auto userId, TMaybe<std::pair<TInstant, TInstant>> offerPeriod = {}) {
    NJson::TJsonValue createdOffers;
    if (offerPeriod) {
        createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(offerPeriod->first))
                ("until", NJson::ToJson(offerPeriod->second))
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
    } else {
        createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
    }
    auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
    UNIT_ASSERT(offerId);
    UNIT_ASSERT(env->BookOffer(offerId, userId));

    TString communicationTagId;
    UNIT_ASSERT(env->GetTagId(userId, SelectCarTagName, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, communicationTagId));
    UNIT_ASSERT(communicationTagId);
    UNIT_ASSERT(env->DeferRequest(communicationTagId, USER_ROOT_DEFAULT, Now() + TDuration::Days(1), "for tests"));

    TString offerHolderTagId;
    UNIT_ASSERT(env->GetTagId(userId, TLongTermOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
    UNIT_ASSERT(offerHolderTagId);
    {
        const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
        auto tx = userTagManager.BuildTx<NSQL::Writable>();
        auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
        UNIT_ASSERT_C(optionalTag, tx.GetStringReport());

        auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT);
        UNIT_ASSERT(permissions);

        bool assigned = TLongTermOfferHolderTag::AssignCarId(*optionalTag, carId, PrepareAaCarTagName, *permissions, *env.GetServer(), tx);
        UNIT_ASSERT_C(assigned, tx.GetStringReport());
        UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
    }
    {
        const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
        auto tx = userTagManager.BuildTx<NSQL::Writable>();
        auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
        UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
        auto longTermOfferHolder = optionalTag->GetTagAs<TLongTermOfferHolderTag>();
        UNIT_ASSERT(longTermOfferHolder);
        UNIT_ASSERT_VALUES_EQUAL(longTermOfferHolder->GetName(), LongTermAutoAssignedTagName);
        UNIT_ASSERT_VALUES_EQUAL(longTermOfferHolder->GetCarIdRef(), carId);
    }
}

Y_UNIT_TEST_SUITE(LongTermSuite) {
    Y_UNIT_TEST(Simple) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermOffer(*env.GetServer());
        RegisterLongTermFilter(*env.GetServer());
        RegisterLongTermEvolutions(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto api = Yensured(env.GetServer()->GetDriveAPI());
        auto car = env.GetEnvironmentGenerator().CreateCar();
        auto emulator = env.GetContext().SetCar(car).GetEmulator();

        auto userId = TString{USER_ID_DEFAULT};
        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto restored = env->Request(userId, "/api/yandex/offers/restore", "offer_id=" + offerId);
        UNIT_ASSERT(restored.IsDefined());

        {
            auto session = api->template BuildTx<NSQL::ReadOnly>();
            auto holders = TLongTermOfferHolderTag::RestoreOfferHolderTags(SimpleLongTermOfferName, *env.GetServer(), session);
            UNIT_ASSERT(holders);
            UNIT_ASSERT_VALUES_EQUAL(holders->size(), 0);
        }
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        {
            auto session = api->template BuildTx<NSQL::ReadOnly>();
            auto holders = TLongTermOfferHolderTag::RestoreOfferHolderTags(SimpleLongTermOfferName, *env.GetServer(), session);
            UNIT_ASSERT(holders);
            UNIT_ASSERT_VALUES_EQUAL(holders->size(), 1);
        }
        {
            auto session = api->template BuildTx<NSQL::ReadOnly>();
            auto holders = TLongTermOfferHolderTag::RestoreOfferHolderTags({}, *env.GetServer(), session);
            UNIT_ASSERT(holders);
            UNIT_ASSERT(holders->size() >= 1);
        }

        auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/true);

        TString offerHolderTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            auto userTags = env->ListTags(userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            for (auto&& record : userTags["records"].GetArray()) {
                auto tagId = NJson::FromJson<TString>(record["tag_id"]);
                auto tagName = NJson::FromJson<TString>(record["tag"]);
                if (tagName == TLongTermOfferHolderTag::Type()) {
                    offerHolderTagId = tagId;
                    break;
                }
            }
        }
        UNIT_ASSERT(offerHolderTagId);
        {
            auto tx = api->template BuildTx<NSQL::ReadOnly>();
            auto optionalChargableSessionTag = TChargableSessionTag::Get(offerId, api->GetTagsManager(), tx);
            UNIT_ASSERT(optionalChargableSessionTag);
            auto chargableSessionTag = optionalChargableSessionTag->GetTagAs<TChargableSessionTag>();
            UNIT_ASSERT(chargableSessionTag);
            UNIT_ASSERT_VALUES_EQUAL(chargableSessionTag->GetSessionTagId(), offerHolderTagId);
            UNIT_ASSERT_VALUES_EQUAL(chargableSessionTag->GetSessionTagEntityType(), NEntityTagsManager::EEntityType::User);

            auto optionalSession = api->GetSessionManager().GetSession(offerId, tx);
            UNIT_ASSERT(optionalSession);
            auto session = *optionalSession;
            UNIT_ASSERT(session);
            UNIT_ASSERT_VALUES_EQUAL(session->GetSessionId(), offerId);

            auto optionalUserSessions = api->GetSessionManager().GetUserSessions(userId, tx);
            UNIT_ASSERT(optionalUserSessions);
            UNIT_ASSERT_VALUES_EQUAL(optionalUserSessions->size(), 1);
            auto userSession = optionalUserSessions->front();
            UNIT_ASSERT(userSession);
            UNIT_ASSERT_VALUES_EQUAL(userSession->GetSessionId(), offerId);
        }
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag(TOfferBookingUserTag::Type(), userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", car.Id)
                ("tag_id", offerHolderTagId)
            ));
        }

        TString userPushTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            auto userTags = env->ListTags(userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            for (auto&& record : userTags["records"].GetArray()) {
                auto tagId = NJson::FromJson<TString>(record["tag_id"]);
                auto tagName = NJson::FromJson<TString>(record["tag"]);
                if (tagName == "test_push") {
                    userPushTagId = tagId;
                    break;
                }
            }
        }
        UNIT_ASSERT(userPushTagId);

        auto bookedSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
        UNIT_ASSERT_VALUES_EQUAL(bookedSession["segment"]["session"]["current_performing"].GetString(), "old_state_reservation");

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        bookedSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        bookedSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
        env.GetContext().SetMileage(10420);
        UNIT_ASSERT(env->EvolveTag("old_state_reservation", userId));
        bookedSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);

        auto sessionHistory = env->GetUserSessions(userId, offerId);
        UNIT_ASSERT(sessionHistory.IsDefined());
        auto sessions = sessionHistory["sessions"].GetArraySafe();
        UNIT_ASSERT_VALUES_EQUAL(sessions.size(), 1);
    }

    Y_UNIT_TEST(DeliveryArea) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        auto emulator = env.GetContext().GetEmulator();

        auto userId = TString{USER_ID_DEFAULT};

        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation)
            ))
        );
        UNIT_ASSERT(createdOffers.IsDefined());
        UNIT_ASSERT(createdOffers["offers"][0]["delivery_area"].IsArray());
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "prereservation");
        }
        TString communicationTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            auto userTags = env->ListTags(userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            for (auto&& record : userTags["records"].GetArray()) {
                auto tagId = NJson::FromJson<TString>(record["tag_id"]);
                auto tagName = NJson::FromJson<TString>(record["tag"]);
                if (tagName == SelectCarTagName) {
                    communicationTagId = tagId;
                    break;
                }
            }
        }
        UNIT_ASSERT(communicationTagId);
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag(PrepareCarTagName, userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("CarId", OBJECT_ID_DEFAULT)
                ("tag_id", communicationTagId)
            ));
        }
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "long_term_preparation");
        }

        TString deliveryTagId;
        UNIT_ASSERT(env->GetTagId(carId, TLongTermCarDeliveryTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, deliveryTagId));
        UNIT_ASSERT(deliveryTagId);
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag(SelectCarTagName, userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("CarId", "")
                ("tag_id", communicationTagId)
            ));
        }
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "prereservation");
        }
        UNIT_ASSERT(!env->GetTagId(carId, TLongTermCarDeliveryTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, deliveryTagId));
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag(PrepareCarTagName, userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("CarId", OBJECT_ID_DEFAULT)
                ("tag_id", communicationTagId)
            ));
        }
        UNIT_ASSERT(env->GetTagId(carId, TLongTermCarDeliveryTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, deliveryTagId));
        UNIT_ASSERT(deliveryTagId);
        {
            UNIT_ASSERT(env->StartTag(deliveryTagId, USER_ROOT_DEFAULT));
            UNIT_ASSERT(!env->FinishTag(deliveryTagId, USER_ROOT_DEFAULT));
            emulator->GetContext().SetCurrentPosition(LongTermDeliveryLocation);
            UNIT_ASSERT(env->WaitLocation(carId, LongTermDeliveryLocation));
            UNIT_ASSERT(env->FinishTag(deliveryTagId, USER_ROOT_DEFAULT));
        }
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            communicationTagId = {};
            UNIT_ASSERT(!env->GetTagId(userId, PrepareCarTagName, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, communicationTagId));
            UNIT_ASSERT(!communicationTagId);
        }
        auto bookedSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
        UNIT_ASSERT_VALUES_EQUAL(bookedSession["segment"]["session"]["current_performing"].GetString(), "old_state_reservation");
    }

    Y_UNIT_TEST(CancellationCost) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto userId = TString{USER_ID_DEFAULT};
        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
                ("franchise", 0)
            )
        );
        UNIT_ASSERT(createdOffers.IsDefined());
        UNIT_ASSERT(createdOffers["offers"][0]["delivery_area"].IsArray());
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        TString offerTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            auto userTags = env->ListTags(userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            for (auto&& record : userTags["records"].GetArray()) {
                auto tagId = NJson::FromJson<TString>(record["tag_id"]);
                auto tagName = NJson::FromJson<TString>(record["tag"]);
                if (tagName == TLongTermOfferHolderTag::Type()) {
                    offerTagId = tagId;
                    break;
                }
            }
        }
        UNIT_ASSERT(offerTagId);
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag(TLongTermOfferHolderTag::Preparation, userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("CarId", OBJECT_ID_DEFAULT)
                ("tag_id", offerTagId)
            ));
        }

        TString communicationTagId;
        TString userPushTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            auto userTags = env->ListTags(userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            for (auto&& record : userTags["records"].GetArray()) {
                auto tagId = NJson::FromJson<TString>(record["tag_id"]);
                auto tagName = NJson::FromJson<TString>(record["tag"]);
                if (tagName == SelectCarTagName) {
                    communicationTagId = tagId;
                    continue;
                }
                if (tagName == "test_push_smile") {
                    userPushTagId = tagId;
                    continue;
                }
            }
        }
        UNIT_ASSERT(communicationTagId);
        UNIT_ASSERT(userPushTagId);
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->StartTag(communicationTagId, USER_ROOT_DEFAULT));
        }

        UNIT_ASSERT(env->DropSession(offerId, userId));
        auto sessionHistory = env->GetUserSessions(userId, offerId);
        UNIT_ASSERT(sessionHistory.IsDefined());
        auto sessions = sessionHistory["sessions"].GetArraySafe();
        UNIT_ASSERT_VALUES_EQUAL(sessions.size(), 1);
    }

    Y_UNIT_TEST(EarlyReturn) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SetSetting(*env.GetServer(), "offer.long_term.use_mileage_for_schedule", "true");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto car = env.GetEnvironmentGenerator().CreateCar();
        auto emulator = env.GetContext().SetCar(car).GetEmulator();

        auto userId = TString{USER_ID_DEFAULT};

        TInstantGuard ig(Day(1));

        auto resultReport = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(Day(5)))
                ("until", NJson::ToJson(Day(45)))
                ("payment_plan", "Weekly")
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto offerReport = resultReport["offers"][0];
        auto totalMileage = offerReport["primary"][0]["value"];
        UNIT_ASSERT_VALUES_EQUAL(totalMileage, 3000);
        auto offerId = offerReport["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));

        ig.Set(Day(4));

        {
            TString offerHolderTagId;
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(userId, TLongTermOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
            UNIT_ASSERT(env->EvolveTag(TOfferBookingUserTag::Type(), userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", car.Id)
                ("tag_id", offerHolderTagId)
            ));
        }

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        env.GetContext().SetMileage(689);

        ig.Set(Day(6));

        UNIT_ASSERT(env->EvolveTag("old_state_reservation", userId));

        auto compilation = GetBillingCompilation(*env.GetServer(), offerId);
        UNIT_ASSERT_VALUES_EQUAL(compilation.GetReportSumPrice(), static_cast<ui64>(679.0 / 3000 * offerReport["pack_price_undiscounted"].GetInteger()) + EarlyReturnCost);
    }

    Y_UNIT_TEST(Drop) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermOffer(*env.GetServer());
        RegisterLongTermFilter(*env.GetServer());
        RegisterLongTermTags(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto userId = TString{USER_ID_DEFAULT};
        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            TString communicationTagId;
            UNIT_ASSERT(env->GetTagId(userId, SelectCarTagName, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, communicationTagId));
            UNIT_ASSERT(communicationTagId);
        }
        UNIT_ASSERT(env->DropSession(offerId, userId));
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            TString communicationTagId;
            UNIT_ASSERT(!env->GetTagId(userId, SelectCarTagName, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, communicationTagId));
            UNIT_ASSERT(!communicationTagId);
        }
        auto emptySession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/true);
    }

    Y_UNIT_TEST(Update) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation)
            ))
        );
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));

        auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
        auto offerHolderTagId = currentSession["segment"]["meta"]["instance_id"].GetString();
        UNIT_ASSERT(offerHolderTagId);

        TString communicationTagId;
        UNIT_ASSERT(env->GetTagId(userId, SelectCarTagName, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, communicationTagId));
        UNIT_ASSERT(communicationTagId);

        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag(PrepareCarTagName, userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", OBJECT_ID_DEFAULT)
                ("tag_id", communicationTagId)
            ));
        }

        auto newSince = Now() + TDuration::Days(42);
        auto updatedOffer = env->Request(userId, "/api/yandex/user/tag/update", "", NJson::TMapBuilder
            ("tag_id", offerHolderTagId)
            ("delivery_location", NJson::ToJson(LongTermDeliveryLocation2))
            ("since", NJson::ToJson(newSince))
        );

        currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
        auto deliveryLocation = NJson::FromJson<TGeoCoord>(currentSession["segment"]["session"]["specials"]["current_offer"]["delivery_location"]);
        auto since = NJson::FromJson<TInstant>(currentSession["segment"]["session"]["specials"]["current_offer"]["since"]);
        UNIT_ASSERT_DOUBLES_EQUAL(deliveryLocation.X, LongTermDeliveryLocation2.X, 0.001);
        UNIT_ASSERT_DOUBLES_EQUAL(deliveryLocation.Y, LongTermDeliveryLocation2.Y, 0.001);
        UNIT_ASSERT_VALUES_EQUAL(since.Seconds(), newSince.Seconds());
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            auto userTags = env->ListTags(carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
            for (auto&& record : userTags["records"].GetArray()) {
                auto tagName = NJson::FromJson<TString>(record["tag"]);
                if (tagName == SelectCarTagName) {
                    auto lat = record["latitude"].GetDouble();
                    auto lon = record["longitude"].GetDouble();
                    UNIT_ASSERT_DOUBLES_EQUAL(lat, LongTermDeliveryLocation2.X, 0.001);
                    UNIT_ASSERT_DOUBLES_EQUAL(lon, LongTermDeliveryLocation2.Y, 0.001);
                }
            }
        }
    }

    Y_UNIT_TEST(Prolong) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SetSetting(*env.GetServer(), "offer.long_term.use_mileage_for_schedule", "true");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        TInstantGuard ig(Day(1));

        auto since1 = Day(5);
        auto until1 = Day(45);
        auto resultReport = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(since1))
                ("until", NJson::ToJson(until1))
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto offerId = resultReport["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));

        ig.Set(Day(3));

        TString offerHolderTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(userId, TLongTermOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
            UNIT_ASSERT(env->EvolveTag(TOfferBookingUserTag::Type(), userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", carId)
                ("tag_id", offerHolderTagId)
            ));
        }

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        ui32 mileage = 679;
        env.GetContext().SetMileage(mileage + 10); // Default mileage is 10

        ig.Set(Day(42));

        auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);

        auto realSince1 = NJson::FromJson<TInstant>(currentSession["segment"]["session"]["specials"]["current_offer_state"]["since"]);
        auto realUntil1 = NJson::FromJson<TInstant>(currentSession["segment"]["session"]["specials"]["current_offer_state"]["until"]);
        UNIT_ASSERT_VALUES_EQUAL(realSince1.Seconds(), Day(3).Seconds());
        UNIT_ASSERT_VALUES_EQUAL(realUntil1.Seconds(), Day(43).Seconds());

        auto since2 = realUntil1 - TDuration::Hours(1);
        auto until2 = since2 + TDuration::Days(80);
        resultReport = env->Request(userId, "/api/yandex/offers/create", "car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(since2))
                ("until", NJson::ToJson(until2))
            )
        );
        auto nextOfferReport = resultReport["offers"][0];
        auto nextOfferId = nextOfferReport["offer_id"].GetString();
        UNIT_ASSERT(nextOfferId);
        auto updatedSince2 = NJson::FromJson<TInstant>(nextOfferReport["since"]);
        auto updatedUntil2 = NJson::FromJson<TInstant>(nextOfferReport["until"]);
        //UNIT_ASSERT_VALUES_EQUAL(updatedSince2.Seconds(), realUntil1.Seconds());
        UNIT_ASSERT_VALUES_EQUAL((updatedUntil2 - updatedSince2).Seconds(), (until2 - since2).Seconds());

        auto switchedOffer = env->Request(userId, "/api/yandex/offers/switch", "", NJson::TMapBuilder
            ("offer_id", nextOfferId)
        );
        currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);

        auto bookedNextOfferReport = currentSession["segment"]["session"]["specials"]["current_offer"];
        UNIT_ASSERT_VALUES_EQUAL(bookedNextOfferReport["pack_price_schedule"][0]["mileage"].GetInteger(), 2330); // roundUpTo10(3000 - 679)
    }

    Y_UNIT_TEST(CarReplacing) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId1 = TString{OBJECT_ID_DEFAULT};
        auto carId2 = TString{OBJECT_ID_DEFAULT1};
        auto userId = TString{USER_ID_DEFAULT};

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<TChargableTag>(TChargableTag::Reservation), carId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));

        TInstantGuard ig(Day(1));

        auto resultReport = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(Day(5)))
                ("until", NJson::ToJson(Day(45)))
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto offerReport = resultReport["offers"][0];
        auto offerId = offerReport["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));

        ig.Set(Day(4)); // actual since is 4, until is 44

        {
            TString offerHolderTagId;
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(userId, TLongTermOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
            UNIT_ASSERT(env->EvolveTag(TOfferBookingUserTag::Type(), userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", carId1)
                ("tag_id", offerHolderTagId)
            ));
        }

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));

        ig.Set(Day(34));
        double timeFractionUsed = (34.0 - 4.0) / 40.0;
        double timeFractionLeft = 1 - timeFractionUsed;

        NNeh::THttpRequest request;
        request.SetUri("/api/yandex/offers/replace_car_prepare");
        request.SetCgiData("&session_id=" + offerId);
        request.AddHeader("Authorization", userId);
        auto reply = env->GetSendReply(request);
        UNIT_ASSERT(!reply.IsSuccessReply());

        env->AddTag(MakeAtomicShared<TReplaceCarTag>(TReplaceCarTag::TypeName), carId1, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);

        resultReport = env->Request(userId, "/api/yandex/offers/replace_car_prepare", "session_id=" + offerId);
        resultReport = env->Request(userId, "/api/yandex/offers/create", "replacing_car=true&session_id=" + offerId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto nextOfferReport = resultReport["offers"][0];
        auto nextOfferId = nextOfferReport["offer_id"].GetString();
        UNIT_ASSERT(nextOfferId);
        UNIT_ASSERT(nextOfferId != offerId);

        // check replacing offer parameters
        UNIT_ASSERT_VALUES_EQUAL(nextOfferReport["duration"].GetInteger(), static_cast<ui32>(offerReport["duration"].GetInteger() * timeFractionLeft));

        env->Request(userId, "/api/yandex/offers/replace_car", "offer_id=" + nextOfferId + "&session_id=" + offerId);

        auto compilation = GetBillingCompilation(*env.GetServer(), offerId);
        UNIT_ASSERT_VALUES_EQUAL(compilation.GetReportSumPrice(), static_cast<ui32>(offerReport["pack_price_undiscounted"].GetInteger() * timeFractionUsed));

        {
            TString offerHolderTagId;
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(userId, TLongTermOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
            UNIT_ASSERT(env->EvolveTag(TOfferBookingUserTag::Type(), userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", carId2)
                ("tag_id", offerHolderTagId)
            ));
        }

        env->AddTag(MakeAtomicShared<TReplaceCarTag>(TReplaceCarTag::TypeName), carId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);

        resultReport = env->Request(userId, "/api/yandex/offers/replace_car_prepare", "session_id=" + nextOfferId);
        resultReport = env->Request(userId, "/api/yandex/offers/create", "replacing_car=true&session_id=" + nextOfferId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );

        auto nextOfferId2 = resultReport["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(nextOfferId2);
        UNIT_ASSERT(nextOfferId2 != nextOfferId);
        UNIT_ASSERT(nextOfferId2 != offerId);

        env->Request(userId, "/api/yandex/offers/replace_car", "offer_id=" + nextOfferId2 + "&session_id=" + nextOfferId);
    }

    Y_UNIT_TEST(CarReplacingScheduleWithMileage) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SetSetting(*env.GetServer(), "offer.long_term.use_mileage_for_schedule", "true");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId1 = TString{OBJECT_ID_DEFAULT};
        auto carId2 = TString{OBJECT_ID_DEFAULT1};
        auto userId = TString{USER_ID_DEFAULT};

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<TChargableTag>(TChargableTag::Reservation), carId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));

        TInstantGuard ig(Day(1));

        auto resultReport = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(Day(5)))
                ("until", NJson::ToJson(Day(45)))
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto offerReport = resultReport["offers"][0];
        auto totalMileage = offerReport["primary"][0]["value"].GetInteger();
        UNIT_ASSERT_VALUES_EQUAL(totalMileage, 3000);
        auto offerId = offerReport["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));

        ig.Set(Day(4)); // actual since is 4, until is 44

        {
            TString offerHolderTagId;
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(userId, TLongTermOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
            UNIT_ASSERT(env->EvolveTag(TOfferBookingUserTag::Type(), userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", carId1)
                ("tag_id", offerHolderTagId)
            ));
        }

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));

        ui32 mileage = 679;
        env.GetContext().SetMileage(mileage + 10); // Default mileage is 10
        double mileageFractionUsed = 1.0 * mileage / totalMileage;
        double mileageFractionLeft = 1 - mileageFractionUsed;

        ig.Set(Day(6));
        double timeFractionUsed = (6.0 - 4.0) / 40.0;
        double timeFractionLeft = 1 - timeFractionUsed;

        env->AddTag(MakeAtomicShared<TReplaceCarTag>(TReplaceCarTag::TypeName), carId1, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);

        resultReport = env->Request(userId, "/api/yandex/offers/replace_car_prepare", "session_id=" + offerId);
        resultReport = env->Request(userId, "/api/yandex/offers/create", "replacing_car=true&session_id=" + offerId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );

        auto nextOfferReport = resultReport["offers"][0];
        auto nextOfferId = nextOfferReport["offer_id"].GetString();
        UNIT_ASSERT(nextOfferId);
        UNIT_ASSERT(nextOfferId != offerId);

        // check replacing offer parameters
        UNIT_ASSERT_VALUES_EQUAL(nextOfferReport["duration"].GetInteger(), static_cast<ui32>(offerReport["duration"].GetInteger() * timeFractionLeft));
        UNIT_ASSERT_VALUES_EQUAL(nextOfferReport["mileage_limit"].GetInteger(), totalMileage - mileage);
        UNIT_ASSERT_VALUES_EQUAL(nextOfferReport["pack_price_undiscounted"].GetInteger(), static_cast<ui64>(ceil(mileageFractionLeft * offerReport["pack_price_undiscounted"].GetInteger())));

        env->Request(userId, "/api/yandex/offers/replace_car", "offer_id=" + nextOfferId + "&session_id=" + offerId);

        auto compilation = GetBillingCompilation(*env.GetServer(), offerId);
        UNIT_ASSERT_VALUES_EQUAL(compilation.GetReportSumPrice(), static_cast<ui64>(mileageFractionUsed * offerReport["pack_price_undiscounted"].GetInteger()));
    }

    Y_UNIT_TEST(CarReplacingTagRemovedOnSessionEnd) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        auto resultReport = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        auto offerReport = resultReport["offers"][0];
        auto offerId = offerReport["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));

        {
            TString offerHolderTagId;
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(userId, TLongTermOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
            UNIT_ASSERT(env->EvolveTag(TOfferBookingUserTag::Type(), userId, TDuration::Seconds(10), {}, {}, NJson::TMapBuilder
                ("car_id", carId)
                ("tag_id", offerHolderTagId)
            ));
        }

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));

        env->AddTag(MakeAtomicShared<TReplaceCarTag>(TReplaceCarTag::TypeName), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);

        UNIT_ASSERT(env->EvolveTag("old_state_reservation", userId));

        TString replacingTagId;
        UNIT_ASSERT(!env->GetTagId(carId, TReplaceCarTag::TypeName, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, replacingTagId));
        UNIT_ASSERT(!replacingTagId);
    }

    Y_UNIT_TEST(GetCarsharingTimetable) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        const auto startOfferDate = Now();

        CreateOfferWithAssignedCarAndOptionalDates(env, carId, userId, std::pair<TInstant, TInstant>{ startOfferDate + TDuration::Days(15), startOfferDate + TDuration::Days(50) });
        CreateOfferWithAssignedCarAndOptionalDates(env, carId, userId, std::pair<TInstant, TInstant>{ startOfferDate + TDuration::Days(60), startOfferDate + TDuration::Days(150) });

        {
            const auto sinceTime = startOfferDate;
            const auto untilTime = sinceTime + TDuration::Days(100);
            auto carsharingTimetable = env->Request(userId, "/api/yandex/offers/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

            UNIT_ASSERT(carsharingTimetable["offers_timetable"].Has(carId));
        }
    }

    Y_UNIT_TEST(ProcessTimetable) {
        struct TFakeEvent {
            TAtomicSharedPtr<TLongTermOfferHolderTag> GetData() const {
                return Offer;
            }

            EObjectHistoryAction GetHistoryAction() const {
                return Action;
            }

            void SetHistoryAction(EObjectHistoryAction action) {
                Action = action;
            }

            TString GetTagId() const {
                return "sample_tag";
            }

            TString GetObjectId() const {
                return "sample_object_id";
            }

            TAtomicSharedPtr<TLongTermOfferHolderTag> Offer = new TLongTermOfferHolderTag();
            EObjectHistoryAction Action = EObjectHistoryAction::Confirmation;
        };

        TMap<TString, TMap<TString, TTimetableEventMetadata>> carsTimetable;
        auto ClearVariables = [&]() {
            carsTimetable.clear();
        };

        TVector<TFakeEvent> userEvents, carEvents;
        TInstant sinceTimetable = TInstant::Now();
        TInstant untilTimetable = sinceTimetable + TDuration::Days(7);

        auto AddEvent = [&userEvents](auto since, auto until, auto carId, auto offerId, auto userId, EObjectHistoryAction action) {
            TFakeEvent event;
            auto pOffer = new TLongTermOffer;
            pOffer->SetSince(since);
            pOffer->SetUntil(until);
            pOffer->SetObjectId(carId);
            pOffer->SetOfferId(offerId);
            pOffer->SetUserId(userId);
            ICommonOffer::TPtr offer = pOffer;
            event.GetData()->SetOffer(offer);
            event.SetHistoryAction(action);
            userEvents.push_back(event);
        };

        // Add events
        AddEvent(sinceTimetable - TDuration::Days(7), untilTimetable + TDuration::Days(7), "car1", "offer1", "user1", EObjectHistoryAction::Confirmation);
        AddEvent(sinceTimetable + TDuration::Days(2), untilTimetable + TDuration::Days(3), "car2", "offer2", "user2", EObjectHistoryAction::Confirmation);
        AddEvent(sinceTimetable - TDuration::Days(7), untilTimetable + TDuration::Days(3), "car3", "offer3", "user3", EObjectHistoryAction::Confirmation);
        AddEvent(sinceTimetable + TDuration::Days(3), untilTimetable + TDuration::Days(3), "car4", "offer4", "user4", EObjectHistoryAction::Confirmation);
        AddEvent(sinceTimetable, untilTimetable, "car5", "offer5", "user5", EObjectHistoryAction::Confirmation);
        AddEvent(sinceTimetable + TDuration::Days(14), untilTimetable + TDuration::Days(21), "car6", "offer6", "user6", EObjectHistoryAction::Confirmation);
        AddEvent(sinceTimetable - TDuration::Days(14), untilTimetable - TDuration::Days(10), "car7", "offer7", "user7", EObjectHistoryAction::Confirmation);

        TSet<TString> fakeFetchedOfferIds;
        ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fakeFetchedOfferIds);
        UNIT_ASSERT(carsTimetable.size() == 5);
        UNIT_ASSERT(carsTimetable["car1"]["offer1"].Since == sinceTimetable - TDuration::Days(7));
        UNIT_ASSERT(carsTimetable["car2"]["offer2"].Since == sinceTimetable + TDuration::Days(2));
        UNIT_ASSERT(carsTimetable["car2"]["offer2"].Until == untilTimetable + TDuration::Days(3));
        ClearVariables();

        // Remove offer
        AddEvent(sinceTimetable + TDuration::Days(2), untilTimetable + TDuration::Days(3), "car2", "offer2", "user1", EObjectHistoryAction::Remove);
        ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fakeFetchedOfferIds);
        UNIT_ASSERT(carsTimetable.size() == 4);
        ClearVariables();

        // Change until
        AddEvent(sinceTimetable - TDuration::Days(7), sinceTimetable + TDuration::Days(3), "car1", "offer1", "user1", EObjectHistoryAction::Confirmation);
        ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fakeFetchedOfferIds);
        UNIT_ASSERT(carsTimetable["car1"]["offer1"].Until == sinceTimetable + TDuration::Days(3));
        ClearVariables();

        // Change since, until <-> remove offer from timetable
        AddEvent(sinceTimetable - TDuration::Days(7), sinceTimetable - TDuration::Days(3), "car1", "offer1", "user1", EObjectHistoryAction::Confirmation);
        ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fakeFetchedOfferIds);
        UNIT_ASSERT(carsTimetable.size() == 3);
        ClearVariables();

        // Change since, until <-> remove offer from timetable
        AddEvent(sinceTimetable + TDuration::Days(14), untilTimetable + TDuration::Days(18), "car3", "offer3", "user3", EObjectHistoryAction::Confirmation);
        ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable,  userEvents, carEvents, sinceTimetable, untilTimetable, fakeFetchedOfferIds);
        UNIT_ASSERT(carsTimetable.size() == 2);
        ClearVariables();

        // Change car
        AddEvent(sinceTimetable, sinceTimetable + TDuration::Days(3), "car4", "offer5", "user5", EObjectHistoryAction::Confirmation);
        ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fakeFetchedOfferIds);
        UNIT_ASSERT(carsTimetable["car4"]["offer5"].Until == sinceTimetable + TDuration::Days(3));
        UNIT_ASSERT(carsTimetable.size() == 1);
        UNIT_ASSERT(carsTimetable.begin()->second.size() == 2);
        ClearVariables();

        // Remove car
        AddEvent(sinceTimetable, sinceTimetable + TDuration::Days(3), "", "offer5", "user5", EObjectHistoryAction::Confirmation);
        ProcessTimetable<TLongTermOfferHolderTag, TLongTermOffer, TRentalServiceModeTag>(carsTimetable, userEvents, carEvents, sinceTimetable, untilTimetable, fakeFetchedOfferIds);
        UNIT_ASSERT(carsTimetable.size() == 1);
        UNIT_ASSERT(carsTimetable.begin()->second.size() == 1);
        ClearVariables();
    }

    Y_UNIT_TEST(AssignCarId) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        CreateOfferWithAssignedCarAndOptionalDates(env, carId, userId);
    }

    Y_UNIT_TEST(Autohide) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermOffer(*env.GetServer(), /*autohide=*/true);
        RegisterLongTermTags(*env.GetServer());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        auto offersReport = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName);
        INFO_LOG << offersReport.GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(offersReport["offers"].GetArraySafe().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(offersReport["offers"][0]["available"].GetBooleanSafe(), false);
        UNIT_ASSERT(offersReport["debug_session_info"]["long_term_autohide"].IsDefined());

        bool added = env->AddTag(MakeAtomicShared<TServiceTagRecord>(LongTermTier1TagName), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
        UNIT_ASSERT(added);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        offersReport = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleLongTermOfferName);
        INFO_LOG << offersReport.GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(offersReport["offers"].GetArraySafe().size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(offersReport["offers"][0]["available"].GetBooleanSafe(), true);
        UNIT_ASSERT(offersReport["debug_session_info"]["long_term_autohide"].IsDefined());

        auto offerId = env->CreateOffer(/*carId=*/{}, userId/*, nullptr, 0, {}, {}, nullptr, SimpleLongTermOfferName*/);
        UNIT_ASSERT(offerId);
    }

    Y_UNIT_TEST(UnderMileage) {
        TTestEnvironment env;
        auto action = MakeAtomicShared<TLongTermOfferBuilder>();
        action->SetName(SimpleLongTermOfferName);
        action->SetDescription(SimpleLongTermOfferName);
        action->SetDetailedDescription("this is a detailed description");
        action->SetChargableAccounts({ "card" });
        action->SetCarModel("audi_q3_quattro");
        action->SetCarTagsFilter(LongTermTier1TagName);
        action->SetAutohide(false);
        action->SetNew(true);
        action->SetOfferTags({ "model_qashqai" });
        action->SetTargetUserTags({ "test_push" });
        action->SetGrouppingTags({ SimpleLongTermOfferName });
        action->SetSwitchOfferTagsList({ SimpleLongTermOfferName });
        int mileageAlpha = 7;
        int extraMileageAlpha = 10;
        int underMileageAlpha = 5;
        action->SetMileageAlpha(mileageAlpha);
        action->SetExtraMileageAlpha(extraMileageAlpha);
        action->SetUnderMileageAlpha(underMileageAlpha);
        action->MutableMileage().SetDefaultValue(30000);
        action->MutableMileage().SetValues(TSet<ui64>{
            10000,
            30000,
            40000
        });

        auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT);
        auto session = env.GetServer()->GetDriveAPI()->BuildSession(true, false);

        TVector<IOfferReport::TPtr> offers;
        TUserOfferContext uoc(&*env.GetServer(), permissions);
        TOffersBuildingContext offersBuildingContext{&*env.GetServer()};
        offersBuildingContext.SetUserHistoryContext(std::move(uoc));

        offersBuildingContext.MutableVariables()["mileage"] = NJson::ToJson(3000);
        action->BuildOffersClean(*permissions, offers, offersBuildingContext, &*env.GetServer(), session);
        auto offer = dynamic_cast<TLongTermOffer*>(offers[0]->GetOffer().Get());
        auto priceDefault = offer->GetPackPrice();
        offers.clear();

        offersBuildingContext.MutableVariables()["mileage"] = NJson::ToJson(1000);
        action->BuildOffersClean(*permissions, offers, offersBuildingContext, &*env.GetServer(), session);
        offer = dynamic_cast<TLongTermOffer*>(offers[0]->GetOffer().Get());
        auto priceUnder = offer->GetPackPrice();
        UNIT_ASSERT_VALUES_EQUAL(priceDefault - priceUnder, 2000 * underMileageAlpha + 2000 * mileageAlpha);
        offers.clear();

        offersBuildingContext.MutableVariables()["mileage"] = NJson::ToJson(4000);
        action->BuildOffersClean(*permissions, offers, offersBuildingContext, &*env.GetServer(), session);
        offer = dynamic_cast<TLongTermOffer*>(offers[0]->GetOffer().Get());
        auto priceExtra = offer->GetPackPrice();
        UNIT_ASSERT_VALUES_EQUAL(priceExtra - priceDefault, 1000 * extraMileageAlpha + 1000 * mileageAlpha);
        offers.clear();
    }

    Y_UNIT_TEST(BuildPaymentScheduleWeekly) {
        TLongTermOffer offer;
        TInstant bookingTimestamp = Day(1);
        offer.SetPackPrice(1000);
        offer.SetPaymentPlan(ELongTermPaymentPlan::Weekly);
        offer.SetWeeklyCost(220);
        offer.SetMileageLimit(150);
        offer.SetBookingTimestamp(bookingTimestamp);
        offer.SetUseMileageForSchedule(true);
        auto schedule = offer.BuildPaymentSchedule(Day(5));
        auto quanta = schedule.GetQuanta();
        auto regularValue = 220;
        auto lastValue = 120;
        TVector<TInstant> timestamps{bookingTimestamp, Day(12), Day(19), Day(26), Day(33)};
        TVector<ui32> mileages{0, 40, 70, 100, 140};
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), timestamps.size());
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), mileages.size());
        for (auto [i, p] : Enumerate(quanta)) {
            UNIT_ASSERT_VALUES_EQUAL(p.Timestamp, timestamps[i]);
            UNIT_ASSERT(p.Mileage);
            UNIT_ASSERT_VALUES_EQUAL(*p.Mileage, mileages[i]);
            UNIT_ASSERT_VALUES_EQUAL(p.Value, i == quanta.size() - 1 ? lastValue : regularValue);
        }
    }

    Y_UNIT_TEST(BuildPaymentScheduleMonthly) {
        TLongTermOffer offer;
        TInstant bookingTimestamp = Day(1);
        offer.SetPackPrice(1000);
        offer.SetPaymentPlan(ELongTermPaymentPlan::Monthly);
        offer.SetMonthlyCost(220);
        offer.SetMileageLimit(150);
        offer.SetBookingTimestamp(bookingTimestamp);
        offer.SetUseMileageForSchedule(true);
        auto schedule = offer.BuildPaymentSchedule(Day(5));
        auto quanta = schedule.GetQuanta();
        auto regularValue = 220;
        auto lastValue = 120;
        TVector<TInstant> timestamps{bookingTimestamp, Day(36), Day(64), Day(95), Day(125)};
        TVector<ui32> mileages{0, 40, 70, 100, 140};
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), timestamps.size());
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), mileages.size());
        for (auto [i, p] : Enumerate(quanta)) {
            UNIT_ASSERT_VALUES_EQUAL(p.Timestamp, timestamps[i]);
            UNIT_ASSERT(p.Mileage);
            UNIT_ASSERT_VALUES_EQUAL(*p.Mileage, mileages[i]);
            UNIT_ASSERT_VALUES_EQUAL(p.Value, i == quanta.size() - 1 ? lastValue : regularValue);
        }
    }

    Y_UNIT_TEST(BuildPaymentScheduleOneTime) {
        TLongTermOffer offer;
        TInstant bookingTimestamp = Day(1);
        offer.SetPackPrice(1000);
        offer.SetPaymentPlan(ELongTermPaymentPlan::OneTime);
        offer.SetMileageLimit(150);
        offer.SetBookingTimestamp(bookingTimestamp);
        offer.SetUseMileageForSchedule(true);
        auto schedule = offer.BuildPaymentSchedule(Day(5));
        auto quanta = schedule.GetQuanta();
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(quanta.begin()->Timestamp, bookingTimestamp);
        UNIT_ASSERT(quanta.begin()->Mileage);
        UNIT_ASSERT_VALUES_EQUAL(*quanta.begin()->Mileage, 0);
        UNIT_ASSERT_VALUES_EQUAL(quanta.begin()->Value, 1000);
    }

    Y_UNIT_TEST(BuildPaymentScheduleCustom) {
        TLongTermOffer offer;
        TInstant bookingTimestamp = Day(1);
        offer.SetPackPrice(1000);
        offer.SetPaymentPlan(ELongTermPaymentPlan::Custom);
        offer.SetPackPriceQuantum(220);
        offer.SetPackPriceQuantumPeriod(TDuration::Days(3));
        offer.SetMileageLimit(150);
        offer.SetBookingTimestamp(bookingTimestamp);
        offer.SetUseMileageForSchedule(true);
        auto schedule = offer.BuildPaymentSchedule(Day(5));
        auto quanta = schedule.GetQuanta();
        auto regularValue = 220;
        auto lastValue = 120;
        TVector<TInstant> timestamps{bookingTimestamp, Day(8), Day(11), Day(14), Day(17)};
        TVector<ui32> mileages{0, 40, 70, 100, 140};
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), timestamps.size());
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), mileages.size());
        for (auto [i, p] : Enumerate(quanta)) {
            UNIT_ASSERT_VALUES_EQUAL(p.Timestamp, timestamps[i]);
            UNIT_ASSERT_VALUES_EQUAL(quanta.size(), mileages.size());
            UNIT_ASSERT(p.Mileage);
            UNIT_ASSERT_VALUES_EQUAL(*p.Mileage, mileages[i]);
            UNIT_ASSERT_VALUES_EQUAL(p.Value, i == quanta.size() - 1 ? lastValue : regularValue);
        }
    }

    Y_UNIT_TEST(BuildPaymentScheduleWeekly2) {
        TLongTermOffer offer;
        TInstant bookingTimestamp = Day(1);
        offer.SetPackPrice(1000);
        offer.SetPaymentPlan(ELongTermPaymentPlan::Weekly);
        offer.SetWeeklyCost(334);
        offer.SetMileageLimit(3000);
        offer.SetBookingTimestamp(bookingTimestamp);
        offer.SetUseMileageForSchedule(true);
        auto schedule = offer.BuildPaymentSchedule(Day(5));
        auto quanta = schedule.GetQuanta();
        auto regularValue = 334;
        auto lastValue = 332;
        TVector<TInstant> timestamps{bookingTimestamp, Day(12), Day(19)};
        TVector<ui32> mileages{0, 1000, 2000};
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), timestamps.size());
        UNIT_ASSERT_VALUES_EQUAL(quanta.size(), mileages.size());
        for (auto [i, p] : Enumerate(quanta)) {
            UNIT_ASSERT_VALUES_EQUAL(p.Timestamp, timestamps[i]);
            UNIT_ASSERT(p.Mileage);
            UNIT_ASSERT_VALUES_EQUAL(*p.Mileage, mileages[i]);
            UNIT_ASSERT_VALUES_EQUAL(p.Value, i == quanta.size() - 1 ? lastValue : regularValue);
        }
    }

    Y_UNIT_TEST(LongTermGroupOffer) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        auto resultReport = env->Request(userId, "/api/yandex/offers/create", "", NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        TString groupOfferReportId;
        int suboffer1count = 0;
        int suboffer2count = 0;
        TString suboffer1groupOffer;
        TString suboffer2groupOffer;
        for (auto r: resultReport["offers"].GetArray()) {
            if (r["is_group_offer"].GetBoolean()) {
                UNIT_ASSERT(!groupOfferReportId);
                groupOfferReportId = r["offer_id"].GetString();
                // TODO: remove me when client does not need this
                // https://st.yandex-team.ru/DRIVEBACK-4596
                continue;
            }
            if (r["group_offer_id"].GetString()) {
                if (r["name"].GetString() == "LongTermSuboffer1") {
                    ++suboffer1count;
                    suboffer1groupOffer = r["group_offer_id"].GetString();
                }
                if (r["name"].GetString() == "LongTermSuboffer2") {
                    ++suboffer2count;
                    suboffer2groupOffer = r["group_offer_id"].GetString();
                }
            }
        }
        UNIT_ASSERT(groupOfferReportId);
        UNIT_ASSERT_VALUES_EQUAL(suboffer1count, 1);
        UNIT_ASSERT_VALUES_EQUAL(suboffer2count, 1);
        UNIT_ASSERT_VALUES_EQUAL(suboffer1groupOffer, groupOfferReportId);
        UNIT_ASSERT_VALUES_EQUAL(suboffer2groupOffer, groupOfferReportId);
    }

    Y_UNIT_TEST(LongTermGroupOfferDisabled) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterLongTermAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("offers.long_term.group_offers.enabled", "false", USER_ROOT_DEFAULT);

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};

        auto resultReport = env->Request(userId, "/api/yandex/offers/create", "", NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("delivery_location", NJson::ToJson(LongTermDeliveryLocation))
            )
        );
        int suboffer1count = 0;
        int suboffer2count = 0;
        int groupOfferCount = 0;
        for (auto r: resultReport["offers"].GetArray()) {
            UNIT_ASSERT(!r["is_group_offer"].IsDefined());
            UNIT_ASSERT(!r["group_offer_id"].IsDefined());
            if (r["name"].GetString() == "LongTermSuboffer1") {
                ++suboffer1count;
            }
            if (r["name"].GetString() == "LongTermSuboffer2") {
                ++suboffer2count;
            }
            if (r["name"].GetString() == "GroupOfferBuilder") {
                ++groupOfferCount;
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(suboffer1count, 1);
        UNIT_ASSERT_VALUES_EQUAL(suboffer2count, 1);
        UNIT_ASSERT_VALUES_EQUAL(groupOfferCount, 0);
    }
}

