#include <drive/backend/ut/library/mock_notifier.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/rt_background/user_push_sender/config.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/async_proxy/ut/helper/fixed_response_server.h>
#include <rtline/library/json/builder.h>

namespace {

    const TString MOCK_NOTIFIER = "mail_mock_notifier";
    const TString TEST_MAIL_TAG_NAME = "test_mail_notification_tag";
    const ui32 SENDER_PACK_SIZE = 10;

    void AddTestMailTag(NDrive::NTest::TRTContext& context) {
        const IDriveTagsManager& tagsManager = context.GetDriveAPI().GetTagsManager();

        NJson::TJsonValue descJson;
        descJson["type"] = TUserMailTag::TypeName;
        descJson["name"] = TEST_MAIL_TAG_NAME;
        descJson["meta"] = NJson::TMapBuilder("template_id", "test_template_id")
            ("template_args", NJson::TArrayBuilder(
                NJson::TMapBuilder("id", "arg1")("name", "test arg")
                    ("default", "arg1_value_def")));

        NStorage::TTableRecord tRecord;
        UNIT_ASSERT(tRecord.DeserializeFromJson(descJson));
        auto tag = TTagDescription::ConstructFromTableRecord(tRecord);
        UNIT_ASSERT(tag);
        auto session = tagsManager.GetDeviceTags().BuildTx<NSQL::Writable>();
        UNIT_ASSERT(tagsManager.GetTagsMeta().RegisterTag(
            tag, USER_ROOT_DEFAULT, session));
        UNIT_ASSERT(session.Commit());
        UNIT_ASSERT(TValidationHelpers::WaitFor([&tagsManager](){
            return !!tagsManager.GetTagsMeta().GetDescriptionByName(TEST_MAIL_TAG_NAME);
        }, TDuration::Seconds(20)));
        Sleep(TDuration::Seconds(10));
    }

    void AddNotifier(NDrive::NTest::TRTContext& context) {
        NJson::TJsonValue mailNotifierDesc = NJson::TMapBuilder("type", "mock")
            ("name", MOCK_NOTIFIER)("meta", NJson::JSON_MAP);
        NJson::TJsonValue requestData = NJson::TMapBuilder(
            "objects", NJson::TArrayBuilder(mailNotifierDesc));
        UNIT_ASSERT(context.GetConfigGenerator().UpsertNotifier(
            requestData, USER_ROOT_DEFAULT));
    }

    void RemoveDBMailTags(NDrive::NTest::TRTContext& context) {
        const auto& tagsManager = context.GetDriveAPI().GetTagsManager();
        const auto& userTags = tagsManager.GetUserTags();
        auto session = userTags.BuildTx<NSQL::Writable>();
        ITagsMeta::TTagDescriptions td = tagsManager.GetTagsMeta().GetTagsByType(TUserMailTag::TypeName);
        TSet<TString> tagNames;
        for (auto&& i : td) {
            tagNames.emplace(i->GetName());
        }
        TVector<TDBTag> tagsForRemove;
        UNIT_ASSERT(userTags.RestoreTags(TSet<TString>(), MakeVector(tagNames), tagsForRemove,
                                         session));
        UNIT_ASSERT(userTags.RemoveTagsSimple(tagsForRemove, USER_ROOT_DEFAULT, session, true));
        UNIT_ASSERT(session.Commit());
        UNIT_ASSERT(TValidationHelpers::WaitFor(
                [&tagsManager, tagNames]() {
                    auto session = tagsManager.GetUserTags().BuildTx<NSQL::ReadOnly>();
                    TVector<TDBTag> userTags;
                    UNIT_ASSERT(tagsManager.GetUserTags().RestoreTags(
                            TSet<TString>(), MakeVector(tagNames), userTags, session));
                    return userTags.empty();
                },
                TDuration::Seconds(10)));
    }

    void StartMailSender(NDrive::NTest::TRTContext& context) {
        auto mailSender = MakeHolder<TRTUserPushSenderCounter>();

        NJson::TJsonValue config = NJson::JSON_MAP;
        config["period"] = "10m";
        config["bp_enabled"] = true;
        config["period"] = "100ms";
        config["notifier"] = MOCK_NOTIFIER;
        config["pack_size"] = SENDER_PACK_SIZE;
        config["message_tag_type"] = TUserMailTag::TypeName;
        UNIT_ASSERT(mailSender->DeserializeFromJson(config));

        TRTBackgroundProcessContainer container(mailSender.Release());
        container.SetName("unittest_mail_notifications");
        UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
    }

    bool CheckMailTagsRemoved(NDrive::NTest::TRTContext& context, size_t leftCount=0) {
        const IDriveTagsManager& tagsManager = context.GetDriveAPI().GetTagsManager();
        return TValidationHelpers::WaitFor(
                [&tagsManager, leftCount]() {
                    auto session = tagsManager.GetUserTags().BuildTx<NSQL::ReadOnly>();
                    TVector<TDBTag> userTags;
                    UNIT_ASSERT(tagsManager.GetUserTags().RestoreTags(
                            TSet<TString>(), {TEST_MAIL_TAG_NAME}, userTags, session));
                    return userTags.size() == leftCount;
                },
                TDuration::Seconds(20));
    }

    void SetUserMailTags(NDrive::NTest::TScript& script, const NJson::TJsonValue& tagBody,
                         const ui32 count) {
        for (ui32 i = 0; i < count; ++i) {
            TString userId = i % 2 ? USER_ID_DEFAULT : USER_ID_DEFAULT1;
            script.Add<NDrive::NTest::TAddTagActions>(TEST_MAIL_TAG_NAME, userId)
                    .SetCustomData(tagBody)
                    .SetEntityType(NEntityTagsManager::EEntityType::User);
        }
    }
}  // namespace

Y_UNIT_TEST_SUITE(RTMailSenderTestSuite) {
    Y_UNIT_TEST(Smoke) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TCommonChecker>(RemoveDBMailTags);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>(AddNotifier);
        script.Add<NDrive::NTest::TCommonChecker>(AddTestMailTag);

        const NJson::TJsonValue tagJson =
                NJson::TMapBuilder("template_args", NJson::TMapBuilder("arg1", "1"));
        SetUserMailTags(script, tagJson, 30);

        script.Add<NDrive::NTest::TCommonChecker>(StartMailSender);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(CheckMailTagsRemoved(context, 20));
        });
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto notifier = context.GetServer()->GetNotifier(MOCK_NOTIFIER);
            TMockNotifier* mockNotifier = dynamic_cast<TMockNotifier*>(notifier.Get());
            UNIT_ASSERT(mockNotifier);
            UNIT_ASSERT_VALUES_EQUAL(mockNotifier->GetProcessedMessages().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(mockNotifier->GetCallsCount(), 1);
        });

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(Attachments) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TCommonChecker>(RemoveDBMailTags);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>(AddNotifier);
        script.Add<NDrive::NTest::TCommonChecker>(AddTestMailTag);

        auto serverPort = Singleton<TPortManager>()->GetPort();
        auto fileServerMock = TFixedResponseServer::BuildAndRun(serverPort, {{200, "test file"}});
        NJson::TJsonValue tagJson = NJson::JSON_MAP;
        tagJson["template_args"] = NJson::TMapBuilder("arg1", "test");
        tagJson["attachments"] = NJson::TArrayBuilder(
                NJson::TMapBuilder("url", TStringBuilder() << "http://localhost:" << serverPort << "/file.txt")
                                  ("mime_type", "text/plain")
                                  ("filename", "Attachment"));
        SetUserMailTags(script, tagJson, 1);

        script.Add<NDrive::NTest::TCommonChecker>(StartMailSender);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(CheckMailTagsRemoved(context));
        });
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto notifier = context.GetServer()->GetNotifier(MOCK_NOTIFIER);
            TMockNotifier* mockNotifier = dynamic_cast<TMockNotifier*>(notifier.Get());
            UNIT_ASSERT(mockNotifier);
            UNIT_ASSERT_VALUES_EQUAL(mockNotifier->GetProcessedMessages().size(), 1);
            const auto& messages = mockNotifier->GetProcessedMessages();
            UNIT_ASSERT(!messages.empty());
            UNIT_ASSERT(!messages.begin()->second.empty());
            auto info = messages.begin()->second[0].GetAdditionalInfo();
            UNIT_ASSERT(info.IsDefined());
            const auto* attachmentData = info.GetValueByPath("attachments.[0].data");
            UNIT_ASSERT_C(attachmentData, info.GetStringRobust());
            UNIT_ASSERT_C(attachmentData->GetString() == "dGVzdCBmaWxl", info.GetStringRobust());
        });

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(TagsNotRemovedOnError) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TCommonChecker>(RemoveDBMailTags);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>(AddNotifier);
        script.Add<NDrive::NTest::TCommonChecker>(AddTestMailTag);

        NJson::TJsonValue tagJson =
                NJson::TMapBuilder("template_args", NJson::TMapBuilder("arg1", "1"));
        tagJson["attachments"] = NJson::TArrayBuilder(
                NJson::TMapBuilder("url", "fake_url")
                                  ("mime_type", "text/plain")
                                  ("filename", "Attachment"));
        SetUserMailTags(script, tagJson, 1);
        script.Add<NDrive::NTest::TCommonChecker>(StartMailSender);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT(!CheckMailTagsRemoved(context));
        });

        UNIT_ASSERT(script.Execute());
    }
}
