#include <drive/backend/cars/car_model.h>
#include <drive/backend/car_attachments/registry/registry.h>
#include <drive/backend/rt_background/sensor_tags/maintenance.h>
#include <drive/backend/maintenance/manager.h>
#include <drive/backend/ut/library/script.h>

#include <rtline/library/json/builder.h>

#include <library/cpp/testing/unittest/registar.h>


namespace {
    void StartWatcher(NDrive::NTest::TRTContext& context) {
        const TString configStr = R"({
            "precision": 20,
            "critical_period": "5d",
            "critical_value": 100,
            "period": "100ms",
            "critical_priority": 1000,
            "blocks_count_limit": 100,
            "modify_performed": true,
            "tag_name": "maintenance_tag",
            "dry_run_mode": false,
            "sensor_id": 2103,
            "bp_enabled": true,
            "remove_tags": true
        })";
        NJson::TJsonValue jsonConfig;
        UNIT_ASSERT(ReadJsonFastTree(configStr, &jsonConfig));
        jsonConfig["filter"] = NJson::TMapBuilder("include_cars", context.GetCar().Id);
        auto watcher = MakeHolder<TRTMaintenanceWatcher>();
        UNIT_ASSERT(watcher->DeserializeFromJson(jsonConfig));
        TRTBackgroundProcessContainer container(watcher.Release());
        container.SetName("test_maintenance_watcher");
        UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
    }

    class TUpdateMaintenanceInfo: public NDrive::NTest::ITestAction {
    private:
        using TBase = NDrive::NTest::ITestAction;
        R_FIELD(TMaintenanceInfo, MaintenanceInfo);

    protected:
        virtual void DoExecute(NDrive::NTest::TRTContext& context) override {
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            MaintenanceInfo.SetVIN(context.GetCar().Vin);
            UNIT_ASSERT_C(context.GetServer()->GetDriveAPI()->GetMaintenanceDB().UpsertMaintenanceInfo(MaintenanceInfo, USER_ROOT_DEFAULT, /* force = */ true, session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }

    public:
        using TBase::TBase;
    };
}

Y_UNIT_TEST_SUITE(RTMaintenanceTestSuite) {
    Y_UNIT_TEST(SimpleFirstMaintenance) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetFirstMaintenanceMileage(10500).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>();
        script.Add<NDrive::NTest::TSetMileage>().SetValue(10);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        script.Add<NDrive::NTest::TSetMileage>().SetValue(11000);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag");
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FirstMaintenancePriority) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetIntermediateMaintenanceMileage(1050).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>();
        script.Add<NDrive::NTest::TSetMileage>().SetValue(1100);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Intermediate)
            .SetCurrentMileage(1100)
            .SetRequiredMileage(1050)
            .SetCriticalMileage(1150)
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<NDrive::NTest::TModifyCarModel>().SetIntermediateMaintenanceMileage(950).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Intermediate)
            .SetCurrentMileage(1100)
            .SetRequiredMileage(950)
            .SetCriticalMileage(1050)
            .SetName("maintenance_tag")
            .SetTagPriority(1000);
        script.Add<NDrive::NTest::TModifyCarModel>().SetIntermediateMaintenanceMileage(2000).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        script.Add<NDrive::NTest::TModifyCarModel>().SetFirstMaintenanceMileage(1050).SetIntermediateMaintenanceMileage(1050).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::First)
            .SetCurrentMileage(1100)
            .SetRequiredMileage(1050)
            .SetCriticalMileage(1150)
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<NDrive::NTest::TModifyCarModel>().SetFirstMaintenanceMileage(950).SetIntermediateMaintenanceMileage(950).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::First)
            .SetCurrentMileage(1100)
            .SetRequiredMileage(950)
            .SetCriticalMileage(1050)
            .SetName("maintenance_tag")
            .SetTagPriority(1000);
        script.Add<NDrive::NTest::TModifyCarModel>().SetFirstMaintenanceMileage(2000).SetIntermediateMaintenanceMileage(2000).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        script.Add<NDrive::NTest::TModifyCarModel>().SetFirstMaintenanceMileage(1050).SetIntermediateMaintenanceMileage(1050).SetMaintenanceMileage(1050).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Plan)
            .SetCurrentMileage(1100)
            .SetRequiredMileage(1050)
            .SetCriticalMileage(1150)
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<NDrive::NTest::TModifyCarModel>().SetFirstMaintenanceMileage(950).SetIntermediateMaintenanceMileage(950).SetMaintenanceMileage(950).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Plan)
            .SetCurrentMileage(1100)
            .SetRequiredMileage(950)
            .SetCriticalMileage(1050)
            .SetName("maintenance_tag")
            .SetTagPriority(1000);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ModelSettings) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        TInstantGuard ig(Now());
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenancePeriod(TDuration::Days(15)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto mgr = context.GetServer()->GetDriveAPI()->GetModelsData();
            UNIT_ASSERT(mgr);
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            auto models = mgr->FetchInfo(session);
            UNIT_ASSERT(models);
            auto model = models.GetResultPtr("porsche_carrera");
            UNIT_ASSERT(model);
            UNIT_ASSERT(model->HasMaintenancePeriod());
            UNIT_ASSERT_VALUES_EQUAL(model->GetMaintenancePeriodRef(), TDuration::Days(15));
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(FirstTimeoutedMaintenance) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        TInstantGuard ig(Now());
        auto now = TInstant::Days(ModelingNow().Days());
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenancePeriod(TDuration::Days(15)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetVin("00000000000000000");
        script.Add<NDrive::NTest::TModifyCarRegistryData>()
            .SetValues({{ ToString(TCarRegistryDocument::EFields::TransferDate), (now - TDuration::Days(16)).ToStringUpToSeconds() }})
            .SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Timeout)
            .SetRequiredTimestamp(now - TDuration::Days(16) + TDuration::Days(15))
            .SetCriticalTimestamp(now - TDuration::Days(16) + TDuration::Days(20))
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetReadyDate(now - TDuration::Days(10))
            .SetStartDate(now - TDuration::Days(9))
            .SetMileage(1000);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(TimeoutedMaintenance) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        TInstantGuard ig(Now());
        auto now = TInstant::Seconds(ModelingNow().Seconds());
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenancePeriod(TDuration::Days(15)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetVin("00000000000000000");
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetReadyDate(now - TDuration::Days(10))
            .SetStartDate(now - TDuration::Days(9))
            .SetMileage(1000);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetReadyDate(now - TDuration::Days(16))
            .SetStartDate(now - TDuration::Days(17))
            .SetMileage(1000);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Timeout)
            .SetRequiredTimestamp(now - TDuration::Days(16) + TDuration::Days(15))
            .SetCriticalTimestamp(now - TDuration::Days(16) + TDuration::Days(20))
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetReadyDate(now - TDuration::Days(21))
            .SetStartDate(now - TDuration::Days(23))
            .SetMileage(1000);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Timeout)
            .SetRequiredTimestamp(now - TDuration::Days(21) + TDuration::Days(15))
            .SetCriticalTimestamp(now - TDuration::Days(21) + TDuration::Days(20))
            .SetName("maintenance_tag")
            .SetTagPriority(1000);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetReadyDate(now)
            .SetStartDate(now - TDuration::Days(1))
            .SetMileage(1000);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(PlanMaintenanceSkip) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        TInstantGuard ig(Now());
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenanceMileage(1000).SetIntermediateMaintenanceMileage(500).SetMaintenancePeriod(TDuration::Days(365)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetVin("00000000000000000");
        script.Add<NDrive::NTest::TSetMileage>().SetValue(2050);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(10))
            .SetStartDate(ModelingNow() - TDuration::Days(9))
            .SetMileage(2000);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(PlanMaintenanceTagOperations) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenanceMileage(1500).SetIntermediateMaintenanceMileage(1000).SetMaintenancePeriod(TDuration::Days(365)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetVin("00000000000000000");
        script.Add<NDrive::NTest::TSetMileage>().SetValue(1050);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Intermediate)
            .SetCurrentMileage(1050)
            .SetRequiredMileage(1000)
            .SetCriticalMileage(1100)
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(1))
            .SetStartDate(ModelingNow() - TDuration::Days(1))
            .SetIntermediate(true)
            .SetMileage(1050);
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenanceMileage(1000).SetIntermediateMaintenanceMileage(1000).SetMaintenancePeriod(TDuration::Days(365)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Plan)
            .SetCurrentMileage(1050)
            .SetRequiredMileage(1000)
            .SetCriticalMileage(1100)
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow())
            .SetStartDate(ModelingNow())
            .SetIntermediate(false)
            .SetMileage(1050);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(MultiMaintenanceInfo) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        TInstantGuard ig(Now());
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenanceMileage(1000).SetIntermediateMaintenanceMileage(500).SetMaintenancePeriod(TDuration::Days(365)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetVin("00000000000000000");
        script.Add<NDrive::NTest::TSetMileage>().SetValue(2050);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(3))
            .SetStartDate(ModelingNow() - TDuration::Days(3))
            .SetIntermediate(false)
            .SetMileage(1000);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(2))
            .SetStartDate(ModelingNow() - TDuration::Days(2))
            .SetIntermediate(true)
            .SetMileage(1500);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::CheckTagsEqual<TMaintenanceTag>>().MutableCheckTag()
            .SetReason(TMaintenanceTag::EReason::Plan)
            .SetCurrentMileage(2050)
            .SetRequiredMileage(2000)
            .SetCriticalMileage(2100)
            .SetName("maintenance_tag")
            .SetTagPriority(0);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(1))
            .SetStartDate(ModelingNow() - TDuration::Days(1))
            .SetIntermediate(false)
            .SetMileage(2050);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(5));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(PlanMaintenanceHistory) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TModifyCarModel>().SetMaintenanceMileage(1000).SetIntermediateMaintenanceMileage(1000).SetMaintenancePeriod(TDuration::Days(365)).SetUserId(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetVin("00000000000000000");
        script.Add<NDrive::NTest::TSetMileage>().SetValue(1050);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(3))
            .SetStartDate(ModelingNow() - TDuration::Days(3))
            .SetIntermediate(false)
            .SetMileage(1);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(2))
            .SetStartDate(ModelingNow() - TDuration::Days(2))
            .SetIntermediate(false)
            .SetMileage(1000);
        script.Add<TUpdateMaintenanceInfo>().MutableMaintenanceInfo()
            .SetSource("Test")
            .SetReadyDate(ModelingNow() - TDuration::Days(1))
            .SetStartDate(ModelingNow() - TDuration::Days(1))
            .SetIntermediate(true)
            .SetMileage(1000);
        script.Add<NDrive::NTest::TCommonChecker>(StartWatcher);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(3));
        script.Add<NDrive::NTest::TCheckHasTag>().SetTargetCount(0).SetEntityType(NEntityTagsManager::EEntityType::Car).SetTagName("maintenance_tag").SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }
}
