#include <drive/backend/ut/library/helper.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/device_snapshot/snapshot.h>
#include <drive/backend/models/storage.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/actions/ut/library/helper.h>
#include <drive/backend/offers/ranking/model.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/util/algorithm/ptr.h>

template <class T>
TAtomicSharedPtr<T> CreateAction(TStringBuf jsonDescription) {
    auto d0 = NJson::ReadJsonFastTree(jsonDescription);
    NStorage::TTableRecord d1;
    UNIT_ASSERT(d1.DeserializeFromJson(d0));

    auto action = TUserAction::BuildFromTableRow(d1);
    UNIT_ASSERT(action);
    auto result = std::dynamic_pointer_cast<T>(std::move(action));
    UNIT_ASSERT(result);
    return result;
}

Y_UNIT_TEST_SUITE(ModelsSuite) {
    TString ModelCorrectorDescription = R"(
{
    "id": "1230",
    "parent": "",
    "action_revision": "12312",
    "action_description": "fake",
    "action_id": "my_little_experiment",
    "action_type": "offer_corrector_price_model",
    "enabled": true,
    "deprecated": false,
    "action_meta": {
        "model": "fakenews",
        "bucket_salt": "heart_shaped_box",
        "buckets": [
            0,
            42
        ],
        "complementary_buckets": [
            1,
            43
        ]
    }
}
    )";

    Y_UNIT_TEST(SimpleOfferModels) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        auto models = server->GetModelsStorage();
        UNIT_ASSERT(models);
        UNIT_ASSERT(server->GetSettings().SetValue("offers.fix_point.min_price", "0", USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSettings().SetValue("offers.fix_point.min_price_difference", "0", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto constantModelName = TString("fakenews");
        auto constantModelValue = double(42);
        auto constantModel = MakeAtomicShared<NDrive::TConstantOfferModel>(constantModelName, constantModelValue);

        UNIT_ASSERT(models->AddOfferModel(constantModel));

        auto allModels = models->ListOfferModels();
        UNIT_ASSERT_VALUES_EQUAL(allModels.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(allModels[0], constantModelName);

        auto model = models->GetOfferModel(constantModelName);
        UNIT_ASSERT(model);
        UNIT_ASSERT_VALUES_EQUAL(model->GetName(), constantModelName);

        auto corrector = CreateAction<TPriceModelOfferCorrector>(ModelCorrectorDescription);
        TUserPermissions::TPtr userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);

        TUserOfferContext uoc(server.Get(), userPermissions);
        TOffersBuildingContext ctx(std::move(uoc));
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            offer.SetTimestamp(TInstant::Zero());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto result = corrector->ApplyForOffer(&sor, {}, ctx, "donald_j_trump", server.Get(), session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offer.GetRiding().GetPrice(), ui32(constantModelValue * 100));
        }
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            offer.SetTimestamp(TInstant::Minutes(24));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto result = corrector->ApplyForOffer(&sor, {}, ctx, "donald_j_trump", server.Get(), session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Unimplemented);
            UNIT_ASSERT_VALUES_EQUAL(offer.GetRiding().GetPrice(), 1000);
        }
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            offer.SetTimestamp(TInstant::Minutes(42));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto result = corrector->ApplyForOffer(&sor, {}, ctx, "donald_j_trump", server.Get(), session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offer.GetRiding().GetPrice(), ui32(constantModelValue * 100));
        }
        {
            TStandartOfferReport sor(BuildOfferPtr(1000, 200, 2000), nullptr);
            TStandartOffer& offer = *sor.GetOfferAs<TStandartOffer>();
            offer.SetTimestamp(TInstant::Zero() + TDuration::Hours(43));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto result = corrector->ApplyForOffer(&sor, {}, ctx, "donald_j_trump", server.Get(), session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Unimplemented);
        }
        {
            TFixPointOfferReport fpor(BuildOfferPtr<TFixPointOffer>(1000, 200, 2000), nullptr);
            TFixPointOffer& offer = *fpor.GetOfferAs<TFixPointOffer>();
            offer.SetAreasDiscount(0.14);
            offer.SetPackDiscount(0.9);
            offer.SetRouteDuration(TDuration::Hours(1));
            offer.SetTimestamp(TInstant::Zero());
            fpor.RecalcPrices(&*server);
            UNIT_ASSERT_DOUBLES_EQUAL(offer.GetPackPrice(), 60 * 1000 * 0.86 * 0.1, 1);

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto result = corrector->ApplyForOffer(&fpor, {}, ctx, "donald_j_trump", server.Get(), session);
            UNIT_ASSERT(result == EOfferCorrectorResult::Success);

            UNIT_ASSERT_DOUBLES_EQUAL(offer.GetPackPrice(), 60 * constantModelValue * 100 * 0.86 * 0.1, 1);
        }
    }

    TString AlwaysModelCorrectorDescription = R"(
{
    "id": "1230",
    "parent": "",
    "action_revision": "12312",
    "action_description": "fake",
    "action_id": "my_little_experiment",
    "action_type": "offer_corrector_price_model",
    "enabled": true,
    "deprecated": false,
    "action_meta": {
        "model": "fakenews"
    }
}
    )";

    TString DiscountOfferCorrectorDescription = R"(
{
    "id": "234232",
    "action_revision": "111",
    "action_description": "fake",
    "action_id": "my_little_discount",
    "action_type": "offer_corrector_discounts",
    "enabled": true,
    "deprecated": false,
    "parent" : "",
    "action_meta": {
        "discount": 0.1,
        "visible": false
    }
}
    )";

    Y_UNIT_TEST(FixPoint) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        auto models = server->GetModelsStorage();
        UNIT_ASSERT(models);

        UNIT_ASSERT(server->GetSettings().SetValue("offers.fix_point.min_price", "0", USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSettings().SetValue("offers.fix_point.min_price_difference", "0", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto constantModelName = TString("fakenews");
        auto constantModelValue = double(42);
        auto constantModel = MakeAtomicShared<NDrive::TConstantOfferModel>(constantModelName, constantModelValue);

        UNIT_ASSERT(models->AddOfferModel(constantModel));

        auto discountCorrector = CreateAction<IOfferCorrectorAction>(DiscountOfferCorrectorDescription);
        auto modelCorrector = CreateAction<IOfferCorrectorAction>(AlwaysModelCorrectorDescription);

        TUserPermissions::TPtr userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);

        TUserOfferContext uoc(server.Get(), userPermissions);
        TOffersBuildingContext ctx(std::move(uoc));
        TBillingSession billingSession;
        {
            TFixPointOfferReport fpor(BuildOfferPtr<TFixPointOffer>(1000, 200, 2000), nullptr);
            TFixPointOffer& offer = *fpor.GetOfferAs<TFixPointOffer>();
            offer.SetRouteDuration(TDuration::Hours(1));
            TGeoCoord c1(37.49, 55.39);
            TGeoCoord c2(37.49, 55.41);
            TGeoCoord c3(37.51, 55.41);
            TGeoCoord c4(37.51, 55.39);
            offer.SetFinishArea({ c1, c2, c3, c4, c1 });
            fpor.RecalcPrices(&*server);

            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                THolder<TChargableTag> tag(new TChargableTag("old_state_reservation"));
                tag->SetOffer(fpor.GetOfferPtrAs<IOffer>());
                AddEvent(billingSession, eventBase, std::move(tag), TInstant::Seconds(30000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(billingSession, eventBase, "old_state_acceptance", TInstant::Seconds(30000), EObjectHistoryAction::TagEvolve);
            AddEvent(billingSession, eventBase, "old_state_riding", TInstant::Seconds(30000), EObjectHistoryAction::TagEvolve);

            auto snapshot = MakeHolder<THistoryDeviceSnapshot>();
            NDrive::TLocation location;
            location.Latitude = 55.4;
            location.Longitude = 37.5;
            location.Timestamp = Now();
            snapshot->SetLocation(location);
            NDrive::IObjectSnapshot::TPtr aSnapshot = std::move(snapshot);

            AddEvent(billingSession, eventBase, "old_state_reservation", TInstant::Seconds(35001), EObjectHistoryAction::TagEvolve, aSnapshot);
            AddEvent(billingSession, eventBase, "old_state_reservation", TInstant::Seconds(35001), EObjectHistoryAction::DropTagPerformer, aSnapshot);

            {
                CheckSum(billingSession, 60 * 1000);
            }
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto result = discountCorrector->ApplyForOffer(&fpor, {}, ctx, "donald_j_trump", server.Get(), session);
                UNIT_ASSERT(result == EOfferCorrectorResult::Success);
                CheckSum(billingSession, 60 * 1000 * 0.9);
            }
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                auto result = modelCorrector->ApplyForOffer(&fpor, {}, ctx, "donald_j_trump", server.Get(), session);
                UNIT_ASSERT(result == EOfferCorrectorResult::Success);
                CheckSum(billingSession, 60 * constantModelValue * 100);
            }
        }
    }
}
