#include <drive/backend/offers/actions/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/sharing.h>
#include <drive/backend/data/state.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/manager.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/testing/unittest/registar.h>

inline TString BookingTag = "s_s_booked";
inline TString InterruptionTag = "s_s_interrupted";
inline TString RejectionTag = "s_s_rejected";
inline TString RevocationTag = "s_s_revoked";
inline TString VisibilityTag = "s_s_visibility";

void RegisterServicingActions(const NDrive::IServer& server) {
    auto servicingSimple1 = MakeAtomicShared<TTagAction>("servicing_add_simple1");
    servicingSimple1->AddTagAction(TTagAction::ETagAction::Add);
    servicingSimple1->SetTagName("simple1");
    RegisterAction(server, servicingSimple1, "default_user");
}

void RegisterServicingTags(const NDrive::IServer& server) {
    auto servicing = MakeAtomicShared<TChargableTag::TDescription>();
    servicing->SetName("servicing");
    servicing->SetType(TChargableTag::TypeName);
    RegisterTag(server, servicing);

    auto hardServicing = MakeAtomicShared<TServicingTag::TDescription>();
    hardServicing->SetDeviceTags({ "simple1" });
    hardServicing->SetCancellable(false);
    hardServicing->SetName("hard_servicing");
    hardServicing->SetType(TServicingTag::Type());
    RegisterTag(server, hardServicing);
}

void RegisterServicingEvolutions(const NDrive::IServer& server) {
    auto toServicing = MakeAtomicShared<TTagEvolutionAction>("to_servicing");
    toServicing->SetTagNameFrom("old_state_parking").SetTagNameTo("servicing");

    auto fromServicing = MakeAtomicShared<TTagEvolutionAction>("from_servicing");
    fromServicing->SetTagNameFrom("servicing").SetTagNameTo("old_state_acceptance");
    fromServicing->SetCommandCode(NDrive::NVega::ECommandCode::OPEN_DOORS);

    RegisterAction(server, toServicing, "default_user");
    RegisterAction(server, fromServicing, "default_user");
}

void RegisterServicingMetadata(const NDrive::IServer& server) {
    RegisterServicingActions(server);
    RegisterServicingTags(server);
    RegisterServicingEvolutions(server);
}

void RegisterSharingActions(const NDrive::IServer& server) {
    auto sessionSharingActions = MakeAtomicShared<TTagAction>("session_sharing_tag_actions");
    sessionSharingActions->AddTagAction(TTagAction::ETagAction::Observe);
    sessionSharingActions->AddTagAction(TTagAction::ETagAction::Remove);
    sessionSharingActions->AddTagAction(TTagAction::ETagAction::RemovePerform);
    sessionSharingActions->SetTagName(TSessionSharingTag::Type());
    RegisterAction(server, sessionSharingActions, "default_user");

    auto stateObservation = MakeAtomicShared<TTagAction>("state_observation");
    stateObservation->AddTagAction(TTagAction::ETagAction::Observe);
    stateObservation->SetTagName("$s_s_*");
    RegisterAction(server, stateObservation, "default_user");

    auto visibilityPerfrom = MakeAtomicShared<TTagAction>("visibility_tag_perform");
    visibilityPerfrom->AddTagAction(TTagAction::ETagAction::Perform);
    visibilityPerfrom->SetTagName(VisibilityTag);
    RegisterAction(server, visibilityPerfrom, "default_user");
}

void RegisterSharingTags(const NDrive::IServer& server) {
    TTagDescription::TPtr simpleTag = new TTagDescription();
    simpleTag->SetType(TGenericUserStateTag::Type());
    simpleTag->SetName(BookingTag);
    RegisterTag(server, simpleTag);
    simpleTag->SetName(InterruptionTag);
    RegisterTag(server, simpleTag);
    simpleTag->SetName(RejectionTag);
    RegisterTag(server, simpleTag);
    simpleTag->SetName(RevocationTag);
    RegisterTag(server, simpleTag);
    simpleTag->SetType(TDeviceTagRecord::TypeName);
    simpleTag->SetName(VisibilityTag);
    simpleTag->SetDefaultPriority(100);
    RegisterTag(server, simpleTag);

    auto sessionSharing = MakeAtomicShared<TSessionSharingTag::TDescription>();
    sessionSharing->SetOfferBuilderId("second_driver");
    sessionSharing->SetBookingTag(BookingTag);
    sessionSharing->SetInterruptionTag(InterruptionTag);
    sessionSharing->SetRejectionTag(RejectionTag);
    sessionSharing->SetRevocationTag(RevocationTag);
    sessionSharing->SetVisibilityTag(VisibilityTag);
    sessionSharing->SetName(TSessionSharingTag::Type());
    sessionSharing->SetType(TSessionSharingTag::Type());
    RegisterTag(server, sessionSharing);

    auto sharing = MakeAtomicShared<TChargableTag::TDescription>();
    sharing->SetName("sharing");
    sharing->SetType(TChargableTag::TypeName);
    RegisterTag(server, sharing);
}

void RegisterSharingEvolutions(const NDrive::IServer& server) {
    auto fromSharing = MakeAtomicShared<TTagEvolutionAction>("from_sharing");
    fromSharing->SetTagNameFrom("sharing").SetTagNameTo("old_state_acceptance");
    RegisterAction(server, fromSharing, "default_user");
}

void RegisterSharingMetadata(const NDrive::IServer& server) {
    RegisterSharingActions(server);
    RegisterSharingTags(server);
    RegisterSharingEvolutions(server);
}

Y_UNIT_TEST_SUITE(MultiPerforming) {
    Y_UNIT_TEST(PerformServiceTag) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        AddRoleToUser(*env.GetServer(), USER_ID_TECH, "default_user");
        AddRoleToUser(*env.GetServer(), USER_ID_TECH, "full_tag_access");
        AddRoleToUser(*env.GetServer(), USER_ID_TECH, "tech_access");

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto offerId = env->CreateOffer(carId, userId);
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        auto firstSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
        UNIT_ASSERT_VALUES_EQUAL(firstSession["segment"]["session"]["current_performing"].GetString(), "old_state_riding");
        {
            auto sessionBuilder = env.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing", Now());
            UNIT_ASSERT(sessionBuilder);
            auto session = sessionBuilder->GetLastObjectSession(carId);
            UNIT_ASSERT(session);
            auto sessionId = session->GetSessionId();
            UNIT_ASSERT_VALUES_EQUAL(sessionId, offerId);
        }

        auto tag = MakeAtomicShared<TDeviceTagRecord>("simple1");
        tag->SetTagPriority(1);
        UNIT_ASSERT(env->AddTag(tag, carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));

        auto tagId = TString();
        UNIT_ASSERT(env->GetTagId(carId, tag->GetName(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, tagId));
        UNIT_ASSERT(tagId);

        UNIT_ASSERT(env->StartTag(tagId, USER_ID_TECH));

        auto chargableTag = MakeAtomicShared<TChargableTag>(TChargableTag::Reservation);
        UNIT_ASSERT(env->AddTag(chargableTag, carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(env.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().RefreshCache(Now()));

        auto secondOfferId = env->CreateOffer(carId, USER_ID_TECH);
        UNIT_ASSERT(secondOfferId);
        UNIT_ASSERT(env->BookOffer(secondOfferId, USER_ID_TECH));
        {
            auto sessionBuilder = env.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing", Now());
            UNIT_ASSERT(sessionBuilder);
            auto session = sessionBuilder->GetLastObjectSession(carId);
            UNIT_ASSERT(session);
            auto sessionId = session->GetSessionId();
            UNIT_ASSERT_VALUES_EQUAL(sessionId, secondOfferId);
        }
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", USER_ID_TECH));
        UNIT_ASSERT(!env->EvolveTag("old_state_riding", USER_ID_TECH));
        auto secondSession = env->GetCurrentSession(USER_ID_TECH, nullptr, nullptr, /*multisession=*/false);
        UNIT_ASSERT_VALUES_EQUAL(secondSession["segment"]["session"]["current_performing"].GetString(), "old_state_acceptance");

        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));
        {
            auto sessionBuilder = env.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetSessionsBuilder("billing", Now());
            UNIT_ASSERT(sessionBuilder);
            auto session = sessionBuilder->GetLastObjectSession(carId);
            UNIT_ASSERT(session);
            auto sessionId = session->GetSessionId();
            UNIT_ASSERT_VALUES_EQUAL(sessionId, offerId);
        }
    }

    Y_UNIT_TEST(ServicingSimple) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterServicingMetadata(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto offerId = env->CreateOffer(carId, userId, nullptr, /*version=*/1000, {}, {}, nullptr, "pack_offer_constructor_quanted");
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(!env->EvolveTag("servicing", userId));
        }
        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag("servicing", userId, TDuration::Seconds(10), {}, {}, NJson::ReadJsonFastTree(R"({
                "subtags": [
                    {
                        "tag": "simple1"
                    }
                ]
            })")));
        }
        Sleep(TDuration::Seconds(10));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "servicing_expected");
            auto bill = currentSession["segment"]["session"]["specials"]["bill"];
            INFO_LOG << bill.GetStringRobust() << Endl;
        }
        TString serviceTagId;
        TString reservationTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(carId, "simple1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, serviceTagId));
            UNIT_ASSERT(serviceTagId);
        }
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(carId, "old_state_reservation", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, reservationTagId));
            UNIT_ASSERT(reservationTagId);
        }
        UNIT_ASSERT(env->StartTag(serviceTagId, USER_ROOT_DEFAULT));
        Sleep(TDuration::Seconds(10));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "servicing_in_progress");
            auto bill = currentSession["segment"]["session"]["specials"]["bill"];
            INFO_LOG << bill.GetStringRobust() << Endl;
        }
        UNIT_ASSERT(!env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->FinishTag(serviceTagId, USER_ROOT_DEFAULT));
        Sleep(TDuration::Seconds(10));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "servicing_done");
            auto bill = currentSession["segment"]["session"]["specials"]["bill"];
            INFO_LOG << bill.GetStringRobust() << Endl;
        }
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "old_state_acceptance");
        }
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(!env->GetTagId(carId, "simple1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, serviceTagId));
            UNIT_ASSERT(!env->GetTagId(carId, "old_state_reservation", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, reservationTagId));
        }
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        Sleep(TDuration::Minutes(1));
        UNIT_ASSERT(env->EvolveTag("old_state_reservation", userId));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            auto bill = currentSession["segment"]["session"]["specials"]["bill"];
            INFO_LOG << bill.GetStringRobust() << Endl;
            for (auto&& element : bill.GetArraySafe()) {
                auto cost = element["cost"].GetDoubleSafe();
                auto type = element["type"].GetStringSafe();
                if (type == "overtime") {
                    UNIT_ASSERT(cost > 0);
                }
            }
        }
    }

    Y_UNIT_TEST(ServicingMileage) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterServicingMetadata(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto offerId = env->CreateOffer(carId, userId, nullptr, /*version=*/1000, {}, {}, nullptr, "pack_offer_constructor");

        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));

        double totalMileage = 0;
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            totalMileage = currentSession["segment"]["session"]["specials"]["current_offer_state"]["remaining_distance"].GetDouble();
        }

        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));

        env.GetContext().SetMileage(50);

        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->EvolveTag("servicing", userId, TDuration::Seconds(10), {}, {}, NJson::ReadJsonFastTree(R"({
                "subtags": [
                    {
                        "tag": "simple1"
                    }
                ]
            })")));
        }

        TString serviceTagId;
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            UNIT_ASSERT(env->GetTagId(carId, "simple1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, serviceTagId));
            UNIT_ASSERT(serviceTagId);
        }

        UNIT_ASSERT(env->StartTag(serviceTagId, USER_ROOT_DEFAULT));

        env.GetContext().SetMileage(52);

        UNIT_ASSERT(env->FinishTag(serviceTagId, USER_ROOT_DEFAULT));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));

        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        env.GetContext().SetMileage(53);

        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));

        double remainingMileage = 0;
        double servicingMileage = 0;
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            remainingMileage = currentSession["segment"]["session"]["specials"]["current_offer_state"]["remaining_distance"].GetDouble();
            servicingMileage = currentSession["segment"]["session"]["specials"]["current_offer_state"]["servicing_mileage"].GetDouble();
        }

        UNIT_ASSERT_VALUES_EQUAL(servicingMileage, 2);
        UNIT_ASSERT_VALUES_EQUAL(totalMileage - remainingMileage, 53 - 2 - 10);
    }

    Y_UNIT_TEST(ServicingHard) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterServicingMetadata(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto offerId = env->CreateOffer(carId, userId, nullptr, /*version=*/1000, {}, {}, nullptr, "pack_offer_constructor_quanted");
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));

        auto hardServicing = MakeAtomicShared<TServicingTag>("hard_servicing");
        hardServicing->SetSessionId(offerId);
        UNIT_ASSERT(env->AddTag(hardServicing, carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/false);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["segment"]["session"]["current_performing"].GetString(), "servicing_expected");
        }
        UNIT_ASSERT(!env->EvolveTag("old_state_acceptance", userId));
    }

    Y_UNIT_TEST(SharingSimple) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSharingMetadata(*env.GetServer());

        const auto& server = *env.GetServer();
        const auto api = server.GetDriveAPI();
        UNIT_ASSERT(api);

        auto serviceTag = MakeAtomicShared<TDeviceTagRecord>("simple_car_tag");
        serviceTag->SetTagPriority(1);
        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto userId2 = TString{USER_ID_DEFAULT2};
        auto offerId = env->CreateOffer(carId, userId);
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        UNIT_ASSERT(env->AddTag(serviceTag, carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));

        env->Request(userId, "/api/yandex/sessions/sharing/invite", "", NJson::TMapBuilder
            ("session_id", offerId)
            ("user_id", userId2)
        );
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession["states"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["states"][0]["session_id"].GetStringRobust(), offerId);
            UNIT_ASSERT(currentSession["states"][0]["user"]["username"].IsString());
        }
        {
            auto currentSession2 = env->GetCurrentSession(userId2, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession2.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["states"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["states"][0]["session_id"].GetStringRobust(), offerId);
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["shared_sessions"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["shared_sessions"][0]["segment"]["meta"]["session_id"].GetStringRobust(), offerId);
            UNIT_ASSERT(currentSession2["shared_sessions"][0]["user"]["username"].IsString());
        }
        env->Request(userId2, "/api/yandex/sessions/sharing/book", "", NJson::TMapBuilder
            ("session_id", offerId)
        );
        TString bookingTagId;
        UNIT_ASSERT(env->GetTagId(userId, BookingTag, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, bookingTagId));
        UNIT_ASSERT(bookingTagId);
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession["states"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["states"][0]["session_id"].GetStringRobust(), offerId);
            UNIT_ASSERT(currentSession["states"][0]["user"]["username"].IsString());
        }
        {
            auto currentSession2 = env->GetCurrentSession(userId2, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession2.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["states"].GetArraySafe().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["shared_sessions"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["sessions"][0]["segment"]["session"]["current_performing"].GetStringSafe(), "old_state_reservation");
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["sessions"][0]["segment"]["session"]["specials"]["current_offer"]["offer_id"].GetStringSafe(), offerId);
        }

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId2));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId2));
        UNIT_ASSERT(!env->EvolveTag("old_state_acceptance", userId));
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession["sessions"][0]["segment"]["session"]["current_stage_info"]["busy"].GetBooleanSafe(), true);
        }

        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId2));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        {
            auto currentSession = env->GetCurrentSession(userId2, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession["sessions"].GetArraySafe().size(), 0);
        }
        TString interruptionTagId;
        UNIT_ASSERT(env->GetTagId(userId2, InterruptionTag, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, interruptionTagId));
        UNIT_ASSERT(interruptionTagId);
        {
            auto currentSession2 = env->GetCurrentSession(userId2, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession2.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["states"].GetArraySafe().size(), 1);
            UNIT_ASSERT(currentSession2["states"][0]["car"]["number"].IsString());
            UNIT_ASSERT(currentSession2["states"][0]["user"]["username"].IsString());
        }
    }

    template<typename T1, typename T2, typename T3>
    void TestSharingWithDebts(T1 callbackBeforeDebt, T2 callbackAfterDebt, T3 callbackAfterNoDebt) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSharingMetadata(*env.GetServer());

        const auto& server = *env.GetServer();
        const auto api = server.GetDriveAPI();
        UNIT_ASSERT(api);
        auto& eGenerator = env.GetContext().GetEGenerator();
        TString cardsReply = eGenerator.GetBillingMock().GetReply();
        const TString emptyCardsReply = "{\"status\": \"success\", \"bound_payment_methods\" : []}";

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto userId2 = TString{USER_ID_DEFAULT2};
        TString offerId;
        {
            auto offer = BuildOfferPtr(200, 100, 102400);
            offer->SetObjectId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
            offer->SetChargableAccounts({ "card", "bonus" });
            offer->SetDepositAmount(27182);
            offerId = offer->GetOfferId();
            UNIT_ASSERT(server.GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
        }
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));

        env->Request(userId, "/api/yandex/sessions/sharing/invite", "", NJson::TMapBuilder
            ("session_id", offerId)
            ("user_id", userId2)
        );
        env->Request(userId2, "/api/yandex/sessions/sharing/book", "", NJson::TMapBuilder
            ("session_id", offerId)
        );

        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId2));

        callbackBeforeDebt(env);

        eGenerator.GetBillingMock().SetReply(emptyCardsReply);
        api->GetBillingManager().WaitBillingCycle(15, 1);
        api->GetBillingManager().WaitBillingCycle(15, 1);

        callbackAfterDebt(env);

        eGenerator.GetBillingMock().SetReply(cardsReply);
        api->GetBillingManager().WaitBillingCycle(15, 1);
        api->GetBillingManager().WaitBillingCycle(15, 1);

        callbackAfterNoDebt(env);
    }

    Y_UNIT_TEST(SharingNoRidingWithDebts) {
        TestSharingWithDebts([](auto& env) {
            UNIT_ASSERT(env->EvolveTag("old_state_riding", USER_ID_DEFAULT2));
            UNIT_ASSERT(env->EvolveTag("old_state_parking", USER_ID_DEFAULT2));
            UNIT_ASSERT(env->EvolveTag("old_state_riding", USER_ID_DEFAULT2));
        }, [](auto& env) {
            UNIT_ASSERT(env->EvolveTag("old_state_parking", USER_ID_DEFAULT2));
            UNIT_ASSERT(!env->EvolveTag("old_state_riding", USER_ID_DEFAULT2));
        }, [](auto& env) {
            UNIT_ASSERT(env->EvolveTag("old_state_riding", USER_ID_DEFAULT2));
            UNIT_ASSERT(!env->EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
            UNIT_ASSERT(env->EvolveTag("old_state_parking", USER_ID_DEFAULT2));
            UNIT_ASSERT(env->EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
        });
    }

    Y_UNIT_TEST(SharingCanEndWithDebts) {
        TestSharingWithDebts([](auto& env) {
            UNIT_ASSERT(env->EvolveTag("old_state_riding", USER_ID_DEFAULT2));
        }, [](auto& env) {
            UNIT_ASSERT(!env->EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
            UNIT_ASSERT(env->EvolveTag("old_state_parking", USER_ID_DEFAULT2));
            UNIT_ASSERT(env->EvolveTag("old_state_acceptance", USER_ID_DEFAULT));
        }, [](auto& env) {
            Y_UNUSED(env);
        });
    }

    Y_UNIT_TEST(SharingReject) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSharingMetadata(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto userId2 = TString{USER_ID_DEFAULT2};
        auto offerId = env->CreateOffer(carId, userId);
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_parking", userId));

        env->Request(userId, "/api/yandex/sessions/sharing/invite", "", NJson::TMapBuilder
            ("session_id", offerId)
            ("user_id", userId2)
        );
        TString sessionSharingTagId;
        {
            auto currentSession2 = env->GetCurrentSession(userId2, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession2.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["states"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(currentSession2["states"][0]["session_id"].GetStringRobust(), offerId);
            sessionSharingTagId = currentSession2["states"][0]["tag_id"].GetStringRobust();
        }
        UNIT_ASSERT(sessionSharingTagId);
        env->Request(userId2, "/api/yandex/sessions/state/drop", "", NJson::TMapBuilder
            ("tag_id", sessionSharingTagId)
        );
        TString rejectionTagId;
        UNIT_ASSERT(env->GetTagId(userId, RejectionTag, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, rejectionTagId));
        UNIT_ASSERT(rejectionTagId);
    }

    Y_UNIT_TEST(SharingRevoke) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSharingMetadata(*env.GetServer());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        auto userId2 = TString{USER_ID_DEFAULT2};
        auto offerId = env->CreateOffer(carId, userId);
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));

        env->Request(userId, "/api/yandex/sessions/sharing/invite", "", NJson::TMapBuilder
            ("session_id", offerId)
            ("user_id", userId2)
        );
        TString sessionSharingTagId;
        {
            auto currentSession = env->GetCurrentSession(userId, nullptr, nullptr, /*multisession=*/true);
            INFO_LOG << currentSession.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(currentSession["states"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["states"][0]["session_id"].GetStringRobust(), offerId);
            sessionSharingTagId = currentSession["states"][0]["tag_id"].GetStringRobust();
        }
        UNIT_ASSERT(sessionSharingTagId);
        env->Request(userId, "/api/yandex/sessions/state/drop", "", NJson::TMapBuilder
            ("tag_id", sessionSharingTagId)
        );
        TString revocationTagId;
        UNIT_ASSERT(env->GetTagId(userId2, RevocationTag, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, revocationTagId));
        UNIT_ASSERT(revocationTagId);
    }
}
