#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/chat.h>

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/config.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/actions/correctors.h>
#include <drive/backend/offers/actions/distributing_block.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/actions/ut/library/helper.h>
#include <drive/backend/offers/context.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/ranking/calcer.h>
#include <drive/backend/offers/ranking/model.h>
#include <drive/backend/roles/manager.h>

#include <drive/telematics/server/library/config.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/system/env.h>

class TMutableCarTagHistoryEvent : public TCarTagHistoryEvent {
public:
    TMutableCarTagHistoryEvent& SetObjectId(const TString& value) {
        ObjectId = value;
        return *this;
    }

    TMutableCarTagHistoryEvent& SetTagId(const TString& value) {
        TagId = value;
        return *this;
    }

    TMutableCarTagHistoryEvent& SetData(ITag::TPtr value) {
        TagData = value;
        return *this;
    }
};

template <class T>
TAtomicSharedPtr<T> CreateAction(TStringBuf descriptionString) {
    auto d0 = NJson::ReadJsonFastTree(descriptionString);
    NStorage::TTableRecord d1;
    UNIT_ASSERT(d1.DeserializeFromJson(d0));

    auto action = TUserAction::BuildFromTableRow(d1);
    UNIT_ASSERT(action);
    auto result = std::dynamic_pointer_cast<T>(std::move(action));
    UNIT_ASSERT(result);
    return result;
}

Y_UNIT_TEST_SUITE(OffersSuite) {
    TString DiscountOfferCorrectorDescription = R"(
{
    "id": "234232",
    "action_revision": "1343",
    "action_description": "fake",
    "action_id": "my_little_discount",
    "action_type": "offer_corrector_discounts",
    "enabled": true,
    "deprecated": false,
    "parent": "",
    "action_meta": {
        "discount": 0.42,
        "visible": false
    }
}
    )";
    using namespace NDrive::NTest;
    Y_UNIT_TEST(HiddenDiscount) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TUserPermissions::TPtr userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);

        TUserOfferContext uoc(server.Get(), userPermissions);
        TOffersBuildingContext context(std::move(uoc));

        TAtomicSharedPtr<TStandartOffer> offer = BuildOfferPtr(1000, 200, 2000);
        TStandartOfferReport sor(offer, nullptr);
        sor.CalculateFeatures(context.GetFeatures());
        UNIT_ASSERT_DOUBLES_EQUAL(offer->GetDiscountMultiplier(""), 1, 0.001);
        UNIT_ASSERT_DOUBLES_EQUAL(offer->GetFeatures().Floats[NDriveOfferFactors::FI_PRICE], 1000 * 0.01, 0.01);
        UNIT_ASSERT_DOUBLES_EQUAL(offer->GetFeatures().Floats[NDriveOfferFactors::FI_WAITING_PRICE], 200 * 0.01, 0.01);

        auto corrector = CreateAction<TDiscountOfferCorrector>(DiscountOfferCorrectorDescription);

        auto tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        auto correctionResult = corrector->ApplyForOffer(&sor, {}, context, "fake_user_id", server.Get(), tx);
        UNIT_ASSERT(tx.Commit());
        UNIT_ASSERT(correctionResult == EOfferCorrectorResult::Success);
        UNIT_ASSERT_DOUBLES_EQUAL(offer->GetFeatures().Floats[NDriveOfferFactors::FI_PRICE], 1000 * (1 - 0.42) * 0.01, 0.01);
        UNIT_ASSERT_DOUBLES_EQUAL(offer->GetFeatures().Floats[NDriveOfferFactors::FI_WAITING_PRICE], 200 * (1 - 0.42) * 0.01, 0.01);

        TBillingSession session;
        TVector<TCarTagHistoryEvent> events;
        {
            TCarTagHistoryEventConstructor eventBase;
            eventBase.SetObjectId("object").SetTagId("1").SetHistoryUserId("user");
            {
                auto tag = MakeHolder<TChargableTag>("old_state_reservation");
                tag->SetOffer(offer);
                AddEvent(session, eventBase, std::move(tag), TInstant::Seconds(1000), EObjectHistoryAction::SetTagPerformer);
            }
            AddEvent(session, eventBase, "old_state_acceptance", TInstant::Seconds(1020), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_riding", TInstant::Seconds(1040), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(1340), EObjectHistoryAction::TagEvolve);
            AddEvent(session, eventBase, "old_state_reservation", TInstant::Seconds(1341), EObjectHistoryAction::DropTagPerformer);
        }
        TBillingSession::TBillingCompilation compilation;
        UNIT_ASSERT(session.FillCompilation(compilation));

        TBill bill = compilation.GetBill(DefaultLocale, server.Get(), nullptr);
        const TVector<TBillRecord>& records = bill.GetRecords();
        for (auto&& i : records) {
            INFO_LOG << i.SerializeToJson().GetStringRobust() << Endl;
        }
        UNIT_ASSERT_VALUES_EQUAL(records.size(), 4);
        UNIT_ASSERT_VALUES_EQUAL(records[0].GetCost() + records[1].GetCost() + records[2].GetCost(), records[3].GetCost());
    }

    TString StandardOfferConstructorDescription = R"(
{
    "id": "234232",
    "parent": "",
    "action_revision": "1111",
    "action_description": "fake",
    "action_id": "my_little_offer",
    "action_type": "standart_offer_builder",
    "deprecated": true,
    "enabled": true,
    "action_meta": {
        "chargable_accounts": [ "fake_account", "card" ],
        "filter_tag_name": "old_state_reservation",
        "riding_price": {
            "pricing_type": "constant",
            "price_details": { "price": 7 }
        },
        "parking_price": {
            "pricing_type": "constant",
            "price_details": { "price": 7 }
        }
    }
}
    )";

    Y_UNIT_TEST(Features) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetSensorApiName({});
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        auto coordinate = TGeoCoord(37.589446, 55.733566);
        emulator->GetContext().SetCurrentPosition(coordinate);
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id, coordinate));

        auto constructor = CreateAction<TStandartOfferConstructor>(StandardOfferConstructorDescription);

        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);

        TUserOfferContext uoc(server.Get(), userPermissions);
        TOffersBuildingContext context(std::move(uoc));
        context.SetCarId(car.Id);
        NDrive::CalcDeviceFeatures(context.MutableFeatures(), *server, car.Id);

        TVector<IOfferReport::TPtr> offers;
        auto session = driveApi.BuildTx<NSQL::Writable>();
        auto constructionResult = constructor->BuildOffers(*userPermissions, userPermissions->GetOfferCorrections(), offers, context, server.Get(), session);
        UNIT_ASSERT(session.Commit());
        UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
        UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
        auto offer = offers[0]->GetOffer();
        UNIT_ASSERT(offer);
        auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
        UNIT_ASSERT(standardOffer);
        const NDrive::TOfferFeatures& features = standardOffer->GetFeatures();
        UNIT_ASSERT(features.Floats[NDriveOfferFactors::FI_PRICE] > 0);
        UNIT_ASSERT_VALUES_EQUAL(features.Categories2[NDriveOfferCatFactors2::FI_MODEL], "porsche_carrera");

        UNIT_ASSERT_DOUBLES_EQUAL(features.Floats[NDriveOfferFactors::EVENING_POPULAR_FINISH_LOCATIONS_COUNT], -2, 0.001);

        auto externalUserFeatures = NDrive::GetExternalUserOfferFactors();
        UNIT_ASSERT(externalUserFeatures.size() > 10);
        auto externalUserCatFeatures = NDrive::GetExternalUserCatOfferFactors2();
        UNIT_ASSERT(externalUserCatFeatures.size() > 5);
    }

    Y_UNIT_TEST(CorrectedFeatures) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetSensorApiName({});
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        auto coordinate = TGeoCoord(37.589446, 55.733566);
        emulator->GetContext().SetCurrentPosition(coordinate);
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id, coordinate));

        auto builder = CreateAction<IOfferBuilderAction>(StandardOfferConstructorDescription);

        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);

        TUserOfferContext uoc(server.Get(), userPermissions);

        TOffersBuildingContext context(server.Get());
        context.SetUserHistoryContext(std::move(uoc));
        context.SetCarId(car.Id);
        NDrive::CalcDeviceFeatures(context.MutableFeatures(), *server, car.Id);

        TVector<IOfferReport::TPtr> offers;
        auto session = driveApi.BuildTx<NSQL::Writable>();
        auto constructionResult = builder->BuildOffers(*userPermissions, userPermissions->GetOfferCorrections(), offers, context, server.Get(), session);
        UNIT_ASSERT(session.Commit());
        UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
        UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
        auto offer = offers[0]->GetOffer();
        UNIT_ASSERT(offer);
        auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
        UNIT_ASSERT(standardOffer);
        {
            const NDrive::TOfferFeatures& features = standardOffer->GetFeatures();
            const NJson::TJsonValue& report = offer->BuildJsonReport({}, *server);
            auto visibleRidingPrice = report["prices"]["riding_discounted"].GetDoubleSafe() / 100;
            auto visibleParkingPrice = report["prices"]["parking_discounted"].GetDoubleSafe() / 100;
            UNIT_ASSERT_DOUBLES_EQUAL(visibleRidingPrice, features.Floats[NDriveOfferFactors::FI_PRICE], 0.01);
            UNIT_ASSERT_DOUBLES_EQUAL(visibleParkingPrice, features.Floats[NDriveOfferFactors::FI_WAITING_PRICE], 0.01);
        }

        auto hiddenDiscountCorrector = CreateAction<TDiscountOfferCorrector>(DiscountOfferCorrectorDescription);
        TStandartOfferReport sor(offer, nullptr);
        {
            auto tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto correctionResult = hiddenDiscountCorrector->ApplyForOffer(&sor, {}, context, "fake_user_id", server.Get(), tx);
            UNIT_ASSERT(tx.Commit());
            UNIT_ASSERT(correctionResult == EOfferCorrectorResult::Success);
        }
        {
            const NDrive::TOfferFeatures& features = standardOffer->GetFeatures();
            const NJson::TJsonValue& report = offer->BuildJsonReport({}, *server);
            auto visibleRidingPrice = report["prices"]["riding_discounted"].GetDoubleSafe() / 100;
            auto visibleParkingPrice = report["prices"]["parking_discounted"].GetDoubleSafe() / 100;
            UNIT_ASSERT_DOUBLES_EQUAL(visibleRidingPrice, features.Floats[NDriveOfferFactors::FI_PRICE], 0.01);
            UNIT_ASSERT_DOUBLES_EQUAL(visibleParkingPrice, features.Floats[NDriveOfferFactors::FI_WAITING_PRICE], 0.01);
        }
        {
            auto model = MakeAtomicShared<NDrive::TJitterOfferModel>("jitter", "jitter", 0.5, 1.5);
            sor.ApplyRidingPriceModel(*model);
            sor.ApplyParkingPriceModel(*model);
            sor.RecalcPrices(&*server);
        }
        {
            const NDrive::TOfferFeatures& features = standardOffer->GetFeatures();
            const NJson::TJsonValue& report = offer->BuildJsonReport({}, *server);
            auto visibleRidingPrice = report["prices"]["riding_discounted"].GetDoubleSafe() / 100;
            auto visibleParkingPrice = report["prices"]["parking_discounted"].GetDoubleSafe() / 100;
            UNIT_ASSERT_DOUBLES_EQUAL(visibleRidingPrice, features.Floats[NDriveOfferFactors::FI_PRICE], 0.01);
            UNIT_ASSERT_DOUBLES_EQUAL(visibleRidingPrice, features.Floats[NDriveOfferFactors::FI_MODEL_PRICE], 0.01);
            UNIT_ASSERT_DOUBLES_EQUAL(visibleParkingPrice, features.Floats[NDriveOfferFactors::FI_WAITING_PRICE], 0.01);
            UNIT_ASSERT_VALUES_EQUAL(features.Categories2[NDriveOfferCatFactors2::FI_GEOBASE_ID_A], "120542");
        }
    }

    Y_UNIT_TEST(OffersCount) {
        NDrive::TServerConfigGenerator configGenerator;
        if (const TString& offersStorageName = GetEnv("OffersStorageName")) {
            configGenerator.SetOffersStorageName(offersStorageName);
        }
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto car = eGenerator.CreateCar();
        const TString& objectId = car.Id;
        const TString& userId = USER_ID_DEFAULT;
        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(userId, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto count = server->GetOffersStorage()->GetOffersCount(objectId, userPermissions->GetUserId(), session);
            UNIT_ASSERT(count.Initialized());
            UNIT_ASSERT_VALUES_EQUAL(count.GetValueSync(), 0);
        }
        {
            auto offerId = configGenerator.CreateOffer(objectId, userId);
            UNIT_ASSERT(offerId);
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto count = server->GetOffersStorage()->GetOffersCount(objectId, userPermissions->GetUserId(), session);
            UNIT_ASSERT(count.Initialized());
            UNIT_ASSERT_VALUES_EQUAL(count.GetValueSync(), 1);
        }
        {
            auto offerId = configGenerator.CreateOffer(objectId, userId);
            UNIT_ASSERT(offerId);
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto count = server->GetOffersStorage()->GetOffersCount(objectId, userPermissions->GetUserId(), session);
            UNIT_ASSERT(count.Initialized());
            UNIT_ASSERT_VALUES_EQUAL(count.GetValueSync(), 2);
        }
    }

    Y_UNIT_TEST(PromoDiscounts) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        auto coordinate = TGeoCoord(37.589446, 55.733566);
        emulator->GetContext().SetCurrentPosition(coordinate);
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id, coordinate));

        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);

        TUserOfferContext uoc(server.Get(), userPermissions);

        TOffersBuildingContext context(server.Get());
        context.SetUserHistoryContext(std::move(uoc));
        context.SetCarId(car.Id);
        NDrive::CalcDeviceFeatures(context.MutableFeatures(), *server, car.Id);

        auto builder = CreateAction<IOfferBuilderAction>(StandardOfferConstructorDescription);

        {
            TVector<IOfferReport::TPtr> offers;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto constructionResult = builder->BuildOffers(*userPermissions, {}, offers, context, server.Get(), session);
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
            auto offer = offers[0]->GetOffer();
            UNIT_ASSERT(offer);
            auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
            UNIT_ASSERT(standardOffer);

            UNIT_ASSERT_DOUBLES_EQUAL(standardOffer->GetDiscountMultiplier(""), 1, 0.001);
        }

        auto corrector = CreateAction<TDiscountOfferCorrector>(DiscountOfferCorrectorDescription);
        corrector->MutableSourceContext().SetPromoCode(true);

        {
            TVector<IOfferReport::TPtr> offers;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto constructionResult = builder->BuildOffers(*userPermissions, { corrector, corrector }, offers, context, server.Get(), session);
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
            auto offer = offers[0]->GetOffer();
            UNIT_ASSERT(offer);
            auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
            UNIT_ASSERT(standardOffer);

            UNIT_ASSERT_DOUBLES_EQUAL(standardOffer->GetDiscountMultiplier(""), 1 - 0.42 * 2, 0.001);
        }

        UNIT_ASSERT(server->GetSettings().SetValue("promo.use_one_promo_discount", "1", USER_ID_TECH));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            TVector<IOfferReport::TPtr> offers;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto constructionResult = builder->BuildOffers(*userPermissions, { corrector, corrector }, offers, context, server.Get(), session);
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
            auto offer = offers[0]->GetOffer();
            UNIT_ASSERT(offer);
            auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
            UNIT_ASSERT(standardOffer);

            UNIT_ASSERT_DOUBLES_EQUAL(standardOffer->GetDiscountMultiplier(""), 1 - 0.42, 0.001);
        }
    }

    Y_UNIT_TEST(CorrectorChargableAccounts) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session);
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        auto coordinate = TGeoCoord(37.589446, 55.733566);
        emulator->GetContext().SetCurrentPosition(coordinate);
        UNIT_ASSERT(configGenerator.WaitLocation(car.Id, coordinate));

        TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);

        TUserOfferContext uoc(server.Get(), userPermissions);
        uoc.SetAccountName("fake_account");

        TOffersBuildingContext context(server.Get());
        context.SetUserHistoryContext(std::move(uoc));
        context.SetCarId(car.Id);
        NDrive::CalcDeviceFeatures(context.MutableFeatures(), *server, car.Id);

        auto builder = CreateAction<IOfferBuilderAction>(StandardOfferConstructorDescription);

        {
            TVector<IOfferReport::TPtr> offers;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto constructionResult = builder->BuildOffers(*userPermissions, {}, offers, context, server.Get(), session);
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
            auto offer = offers[0]->GetOffer();
            UNIT_ASSERT(offer);
            auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
            UNIT_ASSERT(standardOffer);

            UNIT_ASSERT_DOUBLES_EQUAL(standardOffer->GetDiscountMultiplier(""), 1, 0.001);
        }

        auto corrector = CreateAction<TDiscountOfferCorrector>(DiscountOfferCorrectorDescription);

        {
            TVector<IOfferReport::TPtr> offers;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto constructionResult = builder->BuildOffers(*userPermissions, { corrector }, offers, context, server.Get(), session);
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
            auto offer = offers[0]->GetOffer();
            UNIT_ASSERT(offer);
            auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
            UNIT_ASSERT(standardOffer);

            UNIT_ASSERT_DOUBLES_EQUAL(standardOffer->GetDiscountMultiplier(""), 1 - 0.42, 0.001);
        }

        corrector->SetChargableAccounts({ "card" });

        {
            TVector<IOfferReport::TPtr> offers;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto constructionResult = builder->BuildOffers(*userPermissions, { corrector }, offers, context, server.Get(), session);
            UNIT_ASSERT(session.Commit());
            UNIT_ASSERT(constructionResult == EOfferCorrectorResult::Success);
            UNIT_ASSERT_VALUES_EQUAL(offers.size(), 1);
            auto offer = offers[0]->GetOffer();
            UNIT_ASSERT(offer);
            auto standardOffer = dynamic_cast<TStandartOffer*>(offer.Get());
            UNIT_ASSERT(standardOffer);

            UNIT_ASSERT_DOUBLES_EQUAL(standardOffer->GetDiscountMultiplier(""), 1, 0.001);
        }
    }

    Y_UNIT_TEST(PromoLanding) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord outside(35.917933, 56.855074);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);

        NJson::TJsonValue payloadPatch;
        payloadPatch["user_choice"] = "accept";

        {
            NJson::TJsonValue jsonLanding;
            jsonLanding["text"] = "some alert text";
            jsonLanding["payload_patch"] = payloadPatch.GetStringRobust();
            script.Add<TAddLanding>("porsche_carrera_landing", jsonLanding.GetStringRobust());
        }

        script.Add<TSetSetting>().Set("model_promo.porsche_carrera.landing_id", "porsche_carrera_landing");
        script.Add<TSetSetting>().Set("model_promo.porsche_carrera.accept_tag_id", "unique_user_tag");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetExpectOK(false);
        script.Add<TCheckHasTag>().SetTagName("unique_user_tag").SetExpectOK(false);
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetPayloadPatch(payloadPatch.GetStringRobust()).SetUserPosition(from).SetExpectOK(true);
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TDrop>();
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from).SetExpectOK(true);
        script.Add<TCheckLandingComment>().SetLandingId("porsche_carrera_landing").SetComment("accept");
        script.Add<TCheckHasTag>().SetTagName("unique_user_tag").SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ChatOnBooking) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord outside(35.917933, 56.855074);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>(TDuration::Zero());
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TSetSetting>().Set("offers.fix.after_book_chat_id", "support_modern");
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetExpectOK(true);
        script.Add<THasChat>().SetChatId("support_modern");
        script.Add<TAccept>();
        script.Add<TRide>();
        script.Add<TDrop>();
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ExternalOffer) {
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TEnvironmentGenerator::TCar car;
        TString newUserId;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session);
            newUserId = server->GetDriveAPI()->GetUsersData()->RegisterNewUser("unittest", "424242", "onboarding_login", session, "911", "onboarding@mind_your_effing_business.com")->GetUserId();
            UNIT_ASSERT(session.Commit());
        }

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        auto coordinate = TGeoCoord(37.639780, 55.737314);
        emulator->GetContext().SetCurrentPosition(coordinate);
        UNIT_ASSERT(cg.WaitLocation(car.Id, coordinate));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        bool found = false;
        auto rect = TGeoRect(coordinate);
        rect.GrowDistance(100);
        for (size_t i = 0; i < 20; ++i) {
            auto geoHash = server->GetSnapshotsManager().GetGeoHash();
            UNIT_ASSERT(geoHash);

            auto actor = [&](const TDevicesSnapshotManager::THashedDevice& d) {
                if (d.CarId == car.Id) {
                    found = true;
                }
                return true;
            };
            geoHash->FindObjects(rect, actor);
            if (found) {
                break;
            }
            Sleep(TDuration::Seconds(10));
            WARNING_LOG << car.Id << " not found just yet" << Endl;
        }
        UNIT_ASSERT(found);

        auto drive = cg.Request(USER_ID_DEFAULT, "api/yandex/offers/fixpoint", "", NJson::ReadJsonFastTree(R"(
            {"destinations": [{"name": "home", "coordinate": [37.6824, 55.7365]}]}
        )"));
        INFO_LOG << drive.GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(drive["offers"].GetArray().size(), 1);

        auto taxi = cg.Request(USER_ID_DEFAULT, "api/taxi/offers/standard", "");
        INFO_LOG << taxi.GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(taxi["offers"].GetArray().size(), 1);

        auto maps = cg.Request(USER_ID_DEFAULT, "api/maps/offers/fixpoint?src=37.639680+55.737214&dst=37.6824+55.7365");
        INFO_LOG << maps.GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(maps["is_registred"].GetBooleanSafe(), true);
        UNIT_ASSERT_VALUES_EQUAL(maps["offers"].GetArray().size(), 1);

        auto mapsForOnboarding = cg.Request(newUserId, "api/maps/offers/fixpoint?src=37.639680+55.737214&dst=37.6824+55.7365");
        INFO_LOG << mapsForOnboarding.GetStringRobust() << Endl;
        UNIT_ASSERT_VALUES_EQUAL(mapsForOnboarding["is_registred"].GetBooleanSafe(), false);
        UNIT_ASSERT_VALUES_EQUAL(mapsForOnboarding["offers"].GetArray().size(), 1);
    }

    Y_UNIT_TEST(UserGeoFeatures) {
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        TString userId = "636bad06-05e7-4f7c-8eda-08cd972aafd4";

        auto user = server->GetDriveDatabase().GetUserManager().GetCachedObject(USER_ID_DEFAULT);
        UNIT_ASSERT(user);
        TUserPermissions::TMutablePtr permissions = server->GetDriveDatabase().GetUserPermissionManager().BuildUsersPermissions(
            *user,
            {},
            {},
            server->GetDriveDatabase().GetTagsManager(),
            server->GetDriveAPI()->GetConfig().GetRolesFeaturesConfig()
        );
        UNIT_ASSERT(permissions);
        permissions->SetUserId(userId);
        TUserOfferContext userOfferContext(server.Get(), permissions, nullptr);
        TOffersBuildingContext offersBuildingContext(std::move(userOfferContext));
        offersBuildingContext.SetCarId(OBJECT_ID_DEFAULT);
        offersBuildingContext.SetNeedRouteFeatures(true);

        auto userDestinationSuggest = offersBuildingContext.GetUserDestinationSuggest();
        UNIT_ASSERT(userDestinationSuggest);
        for (auto&& element : userDestinationSuggest->Elements) {
            UNIT_ASSERT(element.Latitude > 0);
            UNIT_ASSERT(element.Longitude > 0);
            UNIT_ASSERT(element.GeoFeatures.Initialized());
            UNIT_ASSERT(element.UserGeoFeatures.Initialized());
            UNIT_ASSERT(element.Route.Initialized());
        }
    }

    Y_UNIT_TEST(DistributingBlockShowsRestriction) {
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const ISettings& settings = server->GetSettings();
        UNIT_ASSERT(settings.SetValue("offers.distributing_block.shows_period", "3", USER_ID_DEFAULT));
        UNIT_ASSERT(settings.SetValue("offers.distributing_block.non_shows_period", "5", USER_ID_DEFAULT));
        UNIT_ASSERT(settings.SetValue("offers.distributing_block.enable_shows_restriction", "true", USER_ID_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_settings_history");
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto car = eGenerator.CreateCar();
        const TString& objectId = car.Id;
        const TString userId = eGenerator.CreateUser("distributing_block_user", true, "active");;
        NJson::TJsonValue report;
        //these values are set in GVars
        const auto expectedDistributingBlockShowsCount = 3;
        const auto expectedDistributingBlockNonShowsCount = 5;
        TUserRole userRole;
        userRole.SetRoleId("test_distro_external_link_role").SetUserId(userId);
        auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        {
            THolder<TExternalLinkDistributingBlockAction> action(new TExternalLinkDistributingBlockAction("external_link_distributing_block"));
            action->SetExternalLink("");
            action->SetPlaceReportTraits(TDistributingBlockAction::ReportInOffers);
            action->SetEnablePeriodicalShowsRestrictions(true);
            UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
        TLinkedRoleActionHeader actionHeader;
        actionHeader.SetSlaveObjectId("external_link_distributing_block").SetRoleId("test_distro_external_link_role");
        UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
        UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session) && session.Commit(), session.GetStringReport());
        for (int iter = 0; iter < 2; ++iter) {
            for (int i = 0; i < expectedDistributingBlockShowsCount; ++i) {
                auto offerId = cg.CreateOffer(objectId, userId, &report);
                UNIT_ASSERT(offerId);
                UNIT_ASSERT(report.Has("distributing_block"));
                UNIT_ASSERT_VALUES_EQUAL(report["distributing_block"]["context"]["type"].GetString(), "external_link_distributing_block");
                cg.ShowDistributingBlock(userId, "external_link_distributing_block");
            }
            for (int i = 0; i < expectedDistributingBlockNonShowsCount; ++i) {
                auto offerId = cg.CreateOffer(objectId, userId, &report);
                UNIT_ASSERT(offerId);
                UNIT_ASSERT(!report.Has("distributing_block"));
                cg.ShowDistributingBlock(userId, "");
            }
        }
    }

    Y_UNIT_TEST(SingleDistributingBlockShowsRestriction) {
        NDrive::TServerConfigGenerator cg;
        cg.SetNeedBackground(0);
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        const ISettings& settings = server->GetSettings();
        UNIT_ASSERT(settings.SetValue("offers.distributing_block.enable_shows_restriction", "true", USER_ID_DEFAULT));
        eGenerator.BuildEnvironment();

        auto car = eGenerator.CreateCar();
        const TString& objectId = car.Id;
        const TString userId = eGenerator.CreateUser("distributing_block_user", true, "active");
        const TString hideUserId = eGenerator.CreateUser("hide_distributing_block_user", true, "active");
        ui32 expectedDistributingBlockShowsCount = 3;
        {
            auto session = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            {
                THolder<TExternalLinkDistributingBlockAction> action(new TExternalLinkDistributingBlockAction("external_link_distro_with_show_restrictions"));
                action->SetExternalLink("");
                action->SetPlaceReportTraits(TDistributingBlockAction::ReportInOffers);
                action->SetShowTimes(expectedDistributingBlockShowsCount);
                UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
            }
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("external_link_distro_with_show_restrictions").SetRoleId("test_distro_external_link_role");
            UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            TUserRole userRole;

            userRole.SetRoleId("test_distro_external_link_role").SetUserId(userId);

            TUserRole hideUserRole;
            hideUserRole.SetRoleId("test_distro_external_link_role").SetUserId(hideUserId);
            UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session)
                        && server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(hideUserRole, USER_ROOT_DEFAULT, session)
                        && session.Commit(), session.GetStringReport());
        }

        NJson::TJsonValue report;
        for (ui32 i = 0; i < expectedDistributingBlockShowsCount; ++i) {
            for (const auto& u : {userId, hideUserId}) {
                auto offerId = cg.CreateOffer(objectId, u, &report);
                UNIT_ASSERT(offerId);
                bool shouldShowDistro = (u == userId) || (i == 0);
                if (shouldShowDistro) {
                    UNIT_ASSERT(report.Has("distributing_block"));
                    UNIT_ASSERT_VALUES_EQUAL(report["distributing_block"]["context"]["type"].GetString(), "external_link_distro_with_show_restrictions");
                } else {
                    UNIT_ASSERT(!report.Has("distributing_block"));
                }
                TString type = "external_link_distro_with_show_restrictions";
                if (i != 0) {
                    type += ":metadata";
                }
                if (u == userId) {
                    auto reply = cg.ShowDistributingBlock(u, type);
                    UNIT_ASSERT_VALUES_EQUAL(reply.Code(), HTTP_OK);
                } else {
                    auto reply = cg.HideDistributingBlock(u, type);
                    UNIT_ASSERT_VALUES_EQUAL(reply.Code(), HTTP_OK);
                }
            }
        }
        // do not show anymore
        for (int i = 0; i < 10; ++i) {
            auto offerId = cg.CreateOffer(objectId, userId, &report);
            UNIT_ASSERT(offerId);
            UNIT_ASSERT(!report.Has("distributing_block"));
            auto reply = cg.ShowDistributingBlock(userId, "");
            UNIT_ASSERT_VALUES_EQUAL(reply.Code(), HTTP_OK);
        }
    }
}
