#include <drive/backend/ut/library/script.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NDrive::NTest {
    class TCheckParkingResponse: public TAPIAction {
    public:
        using TChecker = std::function<void(const NJson::TJsonValue& response, TRTContext& context)>;

    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TCheckParkingResponse> Registrator;
        TChecker Checker;
        TGeoCoord Location;

    public:
        using TBase::TBase;

        TCheckParkingResponse(const TInstant startInstant, const TChecker checker,
                              const TGeoCoord& location)
            : TBase(startInstant)
            , Checker(checker)
            , Location(location)
        {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<api/yandex/parking>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: parking" << Endl;
            ss << "</api/yandex/parking>" << Endl;
            return ss.Str();
        }

    protected:
        void DoExecute(TRTContext& context) override {
            TString request = "/api/yandex/parking?position=" + CGIEscapeRet(Location.ToString());
            auto response = SendRequest(context, request);
            Checker(response, context);
        }
    };
    TAPIAction::TFactory::TRegistrator<TCheckParkingResponse> TCheckParkingResponse::Registrator("CheckParkingResponse");

    void PrepareParkingTest(NDrive::NTest::TScript& script,
                            const TGeoCoord& parkingPos,
                            const TCheckParkingResponse::TChecker checker,
                            const TGeoCoord& requestLocation,
                            bool enableRawLocation = true) {
        script.Add<TBuildEnv>();
        script.Add<TSetSetting>().Set("parking_use_raw_location", ToString(enableRawLocation));
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        TGeoCoord from(37.5848674, 55.7352435);
        TVector<TGeoCoord> coordinates;
        UNIT_ASSERT(TGeoCoord::DeserializeVector("37.49676 55.82376 37.49676 55.82378 37.49678 55.82378 37.49678 55.82376 37.49676 55.82376", coordinates));
        script.Add<TCreateArea>().SetId("deny_area").SetCoords(coordinates).SetAreaTags({"force_deny_parking"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferType("standart_offer");
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());
        script.Add<TRide>(TDuration::Zero()).SetCarPosition(parkingPos);
        script.Add<TCheckParkingResponse>(checker, requestLocation);
    }

    void ExpectParkingAllow(const NJson::TJsonValue& response, TRTContext& /*context*/) {
        auto jsonValue = response.GetValueByPath("data.parking_ability");
        UNIT_ASSERT(jsonValue);
        UNIT_ASSERT_STRINGS_EQUAL(jsonValue->GetStringSafe(), "Allow");
    }

    void ExpectParkingDeny(const NJson::TJsonValue& response, TRTContext& /*context*/) {
        auto jsonValue = response.GetValueByPath("data.parking_ability");
        UNIT_ASSERT(jsonValue);
        UNIT_ASSERT_STRINGS_EQUAL(jsonValue->GetStringSafe(), "Deny");
    }

    Y_UNIT_TEST_SUITE(ParkingZones) {
        const TGeoCoord DENIED_LOCATION(37.49677512432857, 55.823775592821185);
        const TGeoCoord ALLOWED_LOCATION(37.41781235, 55.90244293);

        Y_UNIT_TEST(CgiLocation) {
            NDrive::TServerConfigGenerator configGenerator;
            NDrive::NTest::TScript script(configGenerator);
            PrepareParkingTest(script, ALLOWED_LOCATION, ExpectParkingDeny, DENIED_LOCATION, false);
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(AllowOnRawLocation) {
            NDrive::TServerConfigGenerator configGenerator;
            NDrive::NTest::TScript script(configGenerator);
            PrepareParkingTest(script, ALLOWED_LOCATION, ExpectParkingAllow, DENIED_LOCATION);
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(DenyOnRawLocation) {
            NDrive::TServerConfigGenerator configGenerator;
            NDrive::NTest::TScript script(configGenerator);
            PrepareParkingTest(script, DENIED_LOCATION, ExpectParkingDeny, DENIED_LOCATION);
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(ForceAllowParking) {
            NDrive::TServerConfigGenerator configGenerator;
            NDrive::NTest::TScript script(configGenerator);
            PrepareParkingTest(script, DENIED_LOCATION, ExpectParkingAllow, DENIED_LOCATION);

            TVector<TGeoCoord> coords;
            UNIT_ASSERT(TGeoCoord::DeserializeVector("37.49614909805299 55.82493514541471 37.49935702003481 55.8234614906341 37.495172773971575 55.823238021694536", coords));
            UNIT_ASSERT(configGenerator.UpsertArea(
                "force_allow_parking_zone", USER_ROOT_DEFAULT, coords,
                {"force_allow_parking"}));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(script.Execute());
        }
    } //  Y_UNIT_TEST_SUITE(ParkingZones)
}
