#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>

#include <drive/backend/billing/ut/library/offer.h>
#include <drive/backend/data/billing_tags.h>

#include <library/cpp/testing/unittest/registar.h>


Y_UNIT_TEST_SUITE(DrivePaymethods) {
    using namespace NDrive::NTest;

    Y_UNIT_TEST(AccountsFilter) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Set("billing.filter_accounts", "true");
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "fix" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        TSet<TString> expectedOffers;
        const auto checker = [&expectedOffers](const NJson::TJsonValue& offerJson, const TRTContext& /*context*/) {
            INFO_LOG << offerJson << Endl;
            auto offerName = offerJson["name"].GetString();
            expectedOffers.insert(offerName);
        };

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TCreateAndCheckOffer>().SetAccountName("y.drive").SetOnlyCheck(true).SetUserDestination(to).SetChecker(checker).SetUserPosition(from);
        script.Add<NDrive::NTest::TCommonChecker>([&expectedOffers](NDrive::NTest::TRTContext& /*context*/) {
            UNIT_ASSERT_VALUES_EQUAL(expectedOffers.size(), 1);
            UNIT_ASSERT(expectedOffers.contains("fixpoint_offer_constructor"));
            expectedOffers.clear();
        });
        script.Add<NDrive::NTest::TCreateAndCheckOffer>().SetAccountName("card").SetOnlyCheck(true).SetUserDestination(to).SetChecker(checker).SetUserPosition(from);
        script.Add<NDrive::NTest::TCommonChecker>([&expectedOffers](NDrive::NTest::TRTContext& /*context*/) {
            UNIT_ASSERT_VALUES_EQUAL(expectedOffers.size(), 4);
            expectedOffers.clear();
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ChargableAccountOnSwitch) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Set("billing.filter_accounts", "true");

        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();

        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetAccountName("y.drive").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>();
        script.AddMove<NDrive::NTest::TRide>(TDuration::Seconds(5));

        TString firstSessionId;
        script.Add<NDrive::NTest::TCommonChecker>([&firstSessionId](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            billingManager.WaitBillingCycle(10000, 1);
            firstSessionId = context.GetOfferJsonUnsafe()["offer_id"].GetString();
            auto session = billingManager.BuildSession();
            auto bTasks = billingManager.GetActiveTasksManager().GetSessionsTasks({ firstSessionId }, session);
            UNIT_ASSERT(bTasks);
            UNIT_ASSERT_VALUES_EQUAL(bTasks->size(), 1);
            UNIT_ASSERT(bTasks->front().GetChargableAccounts().contains("y.drive"));
        });

        script.Add<NDrive::NTest::TSwitchOffer>();

        const auto sessionChecker = [&firstSessionId](const NDrive::NTest::TRTContext& context, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            TString sessionId = json["segment"]["session"]["specials"]["current_offer"]["offer_id"].GetString();
            UNIT_ASSERT_VALUES_UNEQUAL(firstSessionId, sessionId);
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession();
            auto bTasks = billingManager.GetActiveTasksManager().GetSessionsTasks({ sessionId }, session);
            UNIT_ASSERT(bTasks);
            UNIT_ASSERT_VALUES_EQUAL(bTasks->size(), 1);
            UNIT_ASSERT(bTasks->front().GetChargableAccounts().contains("y.drive"));
            return true;
        };
        script.Add<NDrive::NTest::TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);
        script.AddMove<NDrive::NTest::TDrop>(TDuration::Seconds(20)).SetCarPosition(to);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(DifferentAccountsOnSwitch) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Set("billing.filter_accounts", "true");

        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TCreateAccount>("auto").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TLinkAccount>("auto").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();

        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetAccountName("y.drive").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>();
        script.AddMove<NDrive::NTest::TRide>(TDuration::Seconds(5));

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto sessionId = context.GetOfferJsonUnsafe()["offer_id"].GetString();
            auto session = billingManager.BuildSession();
            auto bTasks = billingManager.GetActiveTasksManager().GetSessionsTasks({ sessionId }, session);
            UNIT_ASSERT(bTasks);
            UNIT_ASSERT_VALUES_EQUAL(bTasks->size(), 1);
            UNIT_ASSERT(bTasks->front().GetChargableAccounts().contains("y.drive"));
        });

        script.Add<NDrive::NTest::TCreateAndSwitchOffer>().SetAccountName("auto");

        const auto sessionChecker = [](const NDrive::NTest::TRTContext& context, const NJson::TJsonValue& json)->bool {
            TString sessionId = json["segment"]["session"]["specials"]["current_offer"]["offer_id"].GetString();
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession();
            auto bTasks = billingManager.GetActiveTasksManager().GetSessionsTasks({ sessionId }, session);
            UNIT_ASSERT(bTasks);
            UNIT_ASSERT_VALUES_EQUAL(bTasks->size(), 1);
            UNIT_ASSERT(!bTasks->front().GetChargableAccounts().contains("y.drive"));
            UNIT_ASSERT(bTasks->front().GetChargableAccounts().contains("auto"));
            return true;
        };
        script.Add<NDrive::NTest::TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);
        script.AddMove<NDrive::NTest::TDrop>(TDuration::Seconds(20)).SetCarPosition(to);
        UNIT_ASSERT(script.Execute());
    }

     Y_UNIT_TEST(ServiceBookDefault) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Set("billing.filter_accounts", "true");
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TBookDefault>().SetAccountName("y.drive");

        const auto sessionChecker = [](const NDrive::NTest::TRTContext& context, const NJson::TJsonValue& json)->bool {
            UNIT_ASSERT(!json["segment"]["meta"]["finished"].GetBoolean());
            TString sessionId = json["segment"]["session"]["specials"]["current_offer"]["offer_id"].GetString();
            TString constructorId = json["segment"]["session"]["specials"]["current_offer"]["constructor_id"].GetString();
            UNIT_ASSERT_VALUES_EQUAL(constructorId, "pack_offer_constructor");

            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession();
            auto bTasks = billingManager.GetActiveTasksManager().GetSessionsTasks({ sessionId }, session);
            UNIT_ASSERT(bTasks);
            UNIT_ASSERT_VALUES_EQUAL(bTasks->size(), 1);
            UNIT_ASSERT(bTasks->front().GetChargableAccounts().contains("y.drive"));
            return true;
        };
        script.Add<NDrive::NTest::TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ServiceBookDefaultPrefix) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        TGeoCoord from(37.5848674, 55.7352435);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Set("billing.filter_accounts", "true");
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "pack" });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TBookDefault>();

        const auto sessionChecker = [](const NDrive::NTest::TRTContext& context, const NJson::TJsonValue& json)->bool {
            UNIT_ASSERT(!json["segment"]["meta"]["finished"].GetBoolean());
            TString sessionId = json["segment"]["session"]["specials"]["current_offer"]["offer_id"].GetString();
            TString constructorId = json["segment"]["session"]["specials"]["current_offer"]["constructor_id"].GetString();
            UNIT_ASSERT_VALUES_EQUAL(constructorId, "standart_offer_constructor");

            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession();
            TMaybe<TVector<TBillingTask>> bTasks = billingManager.GetActiveTasksManager().GetSessionsTasks({ sessionId }, session);
            UNIT_ASSERT(bTasks);
            UNIT_ASSERT_VALUES_EQUAL(bTasks->size(), 1);
            UNIT_ASSERT(!bTasks->front().GetChargableAccounts().contains("y.drive"));
            return true;
        };
        script.Add<NDrive::NTest::TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(WalletPayFullScenario) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive").SetTags({ "fix" });
        script.Add<NDrive::NTest::TDropCache>();

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto& accountsManager = context.GetDriveAPI().GetBillingManager().GetAccountsManager();
            auto registeredAccounts = accountsManager.GetRegisteredAccounts(TInstant::Now());
            ui32 walletType = 0;
            for (auto&& d : registeredAccounts) {
                if (d.GetName() == "y.drive") {
                    walletType = d.GetId();
                    break;
                }
            }
            UNIT_ASSERT(walletType > 0);
        });

        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().GetBillingManager().BuildSession(true);
            auto& accountsManager = context.GetDriveAPI().GetBillingManager().GetAccountsManager();
            auto userAccounts = accountsManager.GetUserAccounts(USER_ID_DEFAULT, session);
            UNIT_ASSERT(userAccounts);
            NDrive::NBilling::IBillingAccount::TPtr account;
            for (auto&& acc : *userAccounts) {
                if (acc->GetName() == "y.drive") {
                    account = acc;
                }
            }
            UNIT_ASSERT(account);
            UNIT_ASSERT_VALUES_EQUAL(account->GetBalance(), NDrive::NBilling::AccountLimit);
        });

        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetAccountName("y.drive").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());
        script.Add<NDrive::NTest::TRide>(TDuration::Zero());
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(2)).SetCarPosition(to);
        script.Add<NDrive::NTest::TDropCache>();

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            TString sessionId = context.GetOfferJsonUnsafe()["offer_id"].GetString();
            UNIT_ASSERT(sessionId);
            TCachedPayments snapshot;
            auto session = billingManager.BuildSession(true);
            UNIT_ASSERT(billingManager.GetPaymentsManager().GetPayments(snapshot, sessionId, session));
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(snapshot.GetTimeline().front().GetPaymentType(), NDrive::NBilling::EAccount::Wallet);
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(PromocodeAccountLink) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive");
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT).SetPromocode("driveWallet");
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildSession(false, false);
            auto allEvents = context.GetDriveAPI().GetBillingManager().GetPromocodeAccountLinks().GetEvents(0, session);
            UNIT_ASSERT(allEvents);
            UNIT_ASSERT(!allEvents->empty());
            auto link = allEvents->back();
            UNIT_ASSERT_VALUES_EQUAL(link.GetUserId(), USER_ID_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(link.GetPromocode(), "driveWallet");

            auto accounts = context.GetDriveAPI().GetBillingManager().GetAccountsManager().GetUserAccounts(USER_ID_DEFAULT, session);
            UNIT_ASSERT(accounts);
            bool found = false;
            for (const auto& account : *accounts) {
                if (account->GetName() == "y.drive") {
                    found = true;
                    UNIT_ASSERT_VALUES_EQUAL(link.GetAccountId(), account->GetId());
                    break;
                }
            }
            UNIT_ASSERT(found);
        });
        UNIT_ASSERT(script.Execute());
    }
}
