#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

#include <util/random/fast.h>


// zxqfd-test4
const auto DefaultUserInfo = TUserContacts("4009159420");


Y_UNIT_TEST_SUITE(PrivateData) {

    Y_UNIT_TEST(UserPassportDatasyncClientFullCycle) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        // Request absent revision, get error.
        {
            auto callback = TAtomicSharedPtr<TPrivateDataAcquisutionTestCallback>(new TPrivateDataAcquisutionTestCallback("get_passport_error"));
            driveApi.GetPrivateDataClient().GetPassport(DefaultUserInfo, "absent_revision", callback);
            callback->WaitForResponsesCount(1);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetFailureCount(), 1);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetSuccessCount(), 0);
        }

        // Update revision, get success.
        {
            TUserPassportData passport;
            auto future = driveApi.GetPrivateDataClient().UpdatePassport(DefaultUserInfo, "updatetest", passport);
            future.Wait();
            future.GetValue();
        }

        // Request just updated revision, get success.
        {
            auto callback = TAtomicSharedPtr<TPrivateDataAcquisutionTestCallback>(new TPrivateDataAcquisutionTestCallback("get_passport_ok"));
            driveApi.GetPrivateDataClient().GetPassport(DefaultUserInfo, "updatetest", callback);
            callback->WaitForResponsesCount(1);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetFailureCount(), 0);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetSuccessCount(), 1);
        }

        // Full cycle: create a revision with all fields filled, submit and then check back.
        {
            NJson::TJsonValue originalPassportJson;
            {
                // Create JSON with passport and the load it into TUserPassportData
                TUserPassportData passport;
                originalPassportJson = TDocumentsHelper::CreateRandomPassportDataJson();
                passport.ParseFromDatasync(originalPassportJson);

                // Update the contents of the revision
                auto future = driveApi.GetPrivateDataClient().UpdatePassport(DefaultUserInfo, "updatetest2", passport);
                future.Wait();
                future.GetValue();
            }

            {
                auto callback = TAtomicSharedPtr<TPrivateDataAcquisutionTestCallback>(new TPrivateDataAcquisutionTestCallback("get_passport_complicated_ok"));
                driveApi.GetPrivateDataClient().GetPassport(DefaultUserInfo, "updatetest2", callback);
                callback->WaitForResponsesCount(1);
                UNIT_ASSERT_VALUES_EQUAL(callback->GetFailureCount(), 0);
                UNIT_ASSERT_VALUES_EQUAL(callback->GetSuccessCount(), 1);

                auto receivedPassportJson = callback->GetLastPassport().SerializeToDatasyncJson();

                for(auto&& field : TDocumentsHelper::PassportStringFields) {
                    UNIT_ASSERT_VALUES_EQUAL(originalPassportJson[field], receivedPassportJson[field]);
                }
                for (auto&& field : TDocumentsHelper::PassportDateFields) {
                    UNIT_ASSERT_VALUES_EQUAL(originalPassportJson[field], receivedPassportJson[field]);
                }
            }
        }
    }

    Y_UNIT_TEST(UserDrivingLicenseDatasyncClientFullCycle) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        // Request absent revision, get error.
        {
            auto callback = TAtomicSharedPtr<TPrivateDataAcquisutionTestCallback>(new TPrivateDataAcquisutionTestCallback("get_license_error"));
            driveApi.GetPrivateDataClient().GetDrivingLicense(DefaultUserInfo, "absent_revision", callback);
            callback->WaitForResponsesCount(1);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetFailureCount(), 1);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetSuccessCount(), 0);
        }

        // Update revision, get success.
        {
            TUserDrivingLicenseData drivingLicense;
            auto future = driveApi.GetPrivateDataClient().UpdateDrivingLicense(DefaultUserInfo, "updatetest", drivingLicense);
            future.Wait();
            future.GetValue();
        }

        // Request just updated revision, get success.
        {
            auto callback = TAtomicSharedPtr<TPrivateDataAcquisutionTestCallback>(new TPrivateDataAcquisutionTestCallback("get_driving_license_ok"));
            driveApi.GetPrivateDataClient().GetDrivingLicense(DefaultUserInfo, "updatetest", callback);
            callback->WaitForResponsesCount(1);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetFailureCount(), 0);
            UNIT_ASSERT_VALUES_EQUAL(callback->GetSuccessCount(), 1);
        }

        // Full cycle: create a revision with all fields filled, submit and then check back.
        {
            NJson::TJsonValue originalDrivingLicenseJson;
            {
                // Create JSON with drivingLicense and the load it into TUserDrivingLicenseData
                TUserDrivingLicenseData drivingLicense;
                originalDrivingLicenseJson = TDocumentsHelper::CreateRandomDrivingLicenseDataJson();
                drivingLicense.ParseFromDatasync(originalDrivingLicenseJson);

                // Update the contents of the revision
                auto future = driveApi.GetPrivateDataClient().UpdateDrivingLicense(DefaultUserInfo, "updatetest2", drivingLicense);
                future.Wait();
                future.GetValue();
            }

            {
                Sleep(TDuration::Seconds(5));  // wait for datasync changes to reflect
                TUserDrivingLicenseData dL;
                UNIT_ASSERT(driveApi.GetPrivateDataClient().GetDrivingLicenseSync(DefaultUserInfo, "updatetest2", dL));

                auto receivedDrivingLicenseJson = dL.SerializeToDatasyncJson();

                for(auto&& field : TDocumentsHelper::DrivingLicenseStringFields) {
                    UNIT_ASSERT_VALUES_EQUAL(originalDrivingLicenseJson[field], receivedDrivingLicenseJson[field]);
                }
                for (auto&& field : TDocumentsHelper::DrivingLicenseDateFields) {
                    UNIT_ASSERT_VALUES_EQUAL(originalDrivingLicenseJson[field], receivedDrivingLicenseJson[field]);
                }
            }
        }
    }
}
