#include <drive/backend/ut/library/helper.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/background/billing/config.h>
#include <drive/backend/background/billing/processor.h>
#include <drive/backend/background/major/config.h>
#include <drive/backend/background/major/processor.h>
#include <drive/backend/background/transformation/config.h>
#include <drive/backend/background/transformation/processor.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(BackgroundProcessesSuite) {
    Y_UNIT_TEST(TransformationWatcher) {
        TStringStream ss;
        NDrive::TServerConfigGenerator().ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const IDriveTagsManager& tagsManager = driveApi.GetTagsManager();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        IOffer::TPtr offer;
        {
            auto standartOffer = MakeHolder<TStandartOffer>();
            standartOffer->SetObjectId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
            standartOffer->SetChargableAccounts({ "bonus", "card" });
            offer = standartOffer.Release();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
        }
        {
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                auto userPermissions = driveApi.GetUserPermissions(USER_ID_DEFAULT, TUserPermissionsFeatures());
                UNIT_ASSERT_C(tagsManager.GetDeviceTags().AddTag(new TChargableTag("old_state_reservation"), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session), session.GetStringReport());

                TVector<TDBTag> tags;
                UNIT_ASSERT_C(tagsManager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "old_state_reservation" }, tags, session), session.GetStringReport());
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
                TDBTag tag = tags[0];
                tag.MutableTagAs<TChargableTag>()->SetOffer(offer);
                UNIT_ASSERT_C(driveApi.GetTagsManager().GetDeviceTags().InitPerformer({tag}, *userPermissions, server.Get(), session), session.GetStringReport());

                TVector<TDBTag> userTags;
                UNIT_ASSERT_C(tagsManager.GetDeviceTags().RestorePerformerTags({ USER_ID_DEFAULT }, userTags, session), session.GetStringReport());
                UNIT_ASSERT_VALUES_EQUAL(userTags.size(), 1);
                tag = userTags[0];

                auto transformation = MakeAtomicShared<TTransformationTag>("old_state_reservation", "old_state_acceptance");
                UNIT_ASSERT(transformation->CopyOnEvolve(*tag, nullptr, *server.Get()));
                transformation->SetRollbackTimestamp(TInstant::Zero());
                UNIT_ASSERT_C(driveApi.GetTagsManager().GetDeviceTags().EvolveTag(tag, transformation, *userPermissions, server.Get(), session), session.GetStringReport());

                {
                    TVector<TDBTag> after;
                    UNIT_ASSERT_C(tagsManager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "old_state_reservation" }, after, session), session.GetStringReport());
                    UNIT_ASSERT_VALUES_EQUAL(after.size(), 0);
                }

                UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
            }
            {
                TTransformationWatcherConfig processConfig("transformation_watcher", "fake");
                auto process = MakeAtomicShared<TTransformationWatcher>(processConfig);
                process->Start();
                process->RestoreTransformations(nullptr, process, server.Get());
                process->Stop();
            }

            {
                auto session = driveApi.BuildTx<NSQL::ReadOnly>();
                TVector<TDBTag> after;
                UNIT_ASSERT_C(tagsManager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "old_state_reservation" }, after, session), session.GetStringReport());
                UNIT_ASSERT_VALUES_EQUAL(after.size(), 1);
            }
        }
    }

    TCarsFilter GetCarFilter(const TStringStream& ss) {
        THolder<TAnyYandexConfig> config(new TAnyYandexConfig);
        UNIT_ASSERT(config->ParseMemory(ss.Str()));

        TCarsFilter carFilter;
        carFilter.Init(config->GetRootSection());
        return carFilter;
    }

    void CheckFilter(const TString& carId, const TSet<TString>& allCarsExt, const TString& name, const TString& value, NDrive::IServer* server) {
        ui64 count = 0;
        TSet<TString> allCars = allCarsExt;
        const ui32 all = allCars.size();
        {
            TStringStream ss;
            ss << "Include" << name << ": " << value << Endl;
            TCarsFilter carFilter = GetCarFilter(ss);

            TSet<TString> cars;
            carFilter.GetAllowedCarIds(cars, server);
            count = cars.size();
            UNIT_ASSERT(count > 0);
            UNIT_ASSERT(cars.contains(carId));
            INFO_LOG << name << " " << count << "/" << all << Endl;
            for (auto&& i : cars) {
                allCars.erase(i);
            }
        }
        {
            TStringStream ss;
            ss << "Exclude" << name << ": " << value << Endl;
            TCarsFilter carFilter = GetCarFilter(ss);

            TSet<TString> cars;
            carFilter.GetAllowedCarIds(cars, server);
            for (auto&& i : cars) {
                allCars.erase(i);
            }
            UNIT_ASSERT_VALUES_EQUAL(allCars.size(), 0);
            UNIT_ASSERT(!cars.contains(carId));
        }
    }
    Y_UNIT_TEST(CarFilter) {
        TStringStream ss;
        NDrive::TServerConfigGenerator().ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const IDriveTagsManager& tagsManager = driveApi.GetTagsManager();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        {
            TSet<TString> allCar;
            {
                TStringStream ss;
                TCarsFilter carFilter = GetCarFilter(ss);
                carFilter.GetAllowedCarIds(allCar, server.Get());
            }
            UNIT_ASSERT(allCar.size() > 0);
            const TString& carId = *allCar.begin();
            TCarsDB::TFetchResult result = driveApi.GetCarsData()->FetchInfo(carId);
            UNIT_ASSERT_VALUES_EQUAL(result.size(), 1);
            TDriveCarInfo car = result.begin()->second;
            CheckFilter(carId, allCar, "ModelCodes", car.GetModel(), server.Get());
//            CheckFilter(carId, allCar.size(), "CarStatus", car.GetStatus(), server.Get());
            CheckFilter(carId, allCar, "Cars", carId, server.Get());
            {
                auto session = driveApi.BuildTx<NSQL::ReadOnly>();
                TVector<TDBTag> tagsAll;

                UNIT_ASSERT(tagsManager.GetDeviceTags().RestoreTags(TSet<TString>({carId}), {}, tagsAll, session));
                UNIT_ASSERT(tagsAll.size() > 0);
                const TString& tagName = tagsAll[0]->GetName();
                CheckFilter(carId, allCar, "TagsFilter", tagName, server.Get());
            }
        }
    }

    Y_UNIT_TEST(RepairTag) {
        if (!GetEnv("MAJOR_PASSWORD_PATH", "")) {
            return;
        }
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.GetConfig()->SetPusherOptions({});
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const auto& manager = driveApi.GetTagsManager();

        TRepairTagRecord tag("maintenance");
        tag.SetComment("Comment for tag");
        const TString carId("50c0c78d-f546-4a26-b1ea-ae577d7b0bf6");

        auto emulator = tmBuilder.BuildEmulator(server->GetDriveAPI()->GetIMEI(carId));
        tmBuilder.SetSensorDoubleValue(*emulator, CAN_ODOMETER_KM, 1000.0);
        UNIT_ASSERT(gServer.WaitLocation(carId));
        UNIT_ASSERT(gServer.WaitSensor(carId, "mileage"));

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> tagsAll;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreTags(TSet<TString>({carId}), {tag.GetName()}, tagsAll, session));
            UNIT_ASSERT_C(manager.GetDeviceTags().RemoveTagsSimple(tagsAll, USER_ROOT_DEFAULT, session, true), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT(session.Commit());
        }
        {
            TVector<TDBTag> tagsAll;
            auto session = driveApi.BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(manager.GetDeviceTags().RestoreTags(TSet<TString>({carId}), {tag.GetName()}, tagsAll, session));
            UNIT_ASSERT_VALUES_EQUAL(tagsAll.size(), 0);
        }
        ui64 queryCount = gServer.ListMajorQuery(USER_ROOT_DEFAULT)["queries"].GetArray().size();
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&tag, USER_ID_DEFAULT, carId, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        {
            TMajorRequesterConfig processConfig("major_requester", "fake_major", { "maintenance" });
            TMajorRequester process(&processConfig);
            UNIT_ASSERT(process.SendRequests(server.Get()));
        }
        UNIT_ASSERT_VALUES_EQUAL(gServer.ListMajorQuery(USER_ROOT_DEFAULT)["queries"].GetArray().size(), (queryCount + 1));
        INFO_LOG << queryCount << Endl;
        {
            TVector<TDBTag> tagsAll;
            {
                auto session = driveApi.BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(manager.GetDeviceTags().RestoreTags(TSet<TString>({carId}), {"maintenance"}, tagsAll, session));
            }
            UNIT_ASSERT_VALUES_EQUAL(tagsAll.size(), 1);
            for (auto&& tag : tagsAll) {
                UNIT_ASSERT_VALUES_EQUAL(tag.GetObjectId(), carId);
                TRepairTagRecord* repairTag = dynamic_cast<TRepairTagRecord*>(tag.GetData().Get());
                UNIT_ASSERT(!!repairTag && !repairTag->GetQueryId().empty());
                UNIT_ASSERT(gServer.CancelMajorQuery(tag.GetTagId(), USER_ROOT_DEFAULT));
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(gServer.ListMajorQuery(USER_ROOT_DEFAULT)["queries"].GetArray().size(), queryCount);
    }

    Y_UNIT_TEST(SoftQueueSwitcher) {
        TStringStream ss;
        NDrive::TServerConfigGenerator().ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto& billingManager = driveApi.GetBillingManager();
        {
            auto session = billingManager.BuildSession();
            TBillingTask task;
            task.SetId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetBillingType(EBillingType::CarUsage);
            task.SetQueue(EBillingQueue::Prestable);
            task.SetNextQueue(EBillingQueue::Prestable);
            UNIT_ASSERT(billingManager.CreateBillingTask(task, USER_ID_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        const TBillingQueueSwitcherConfig processConfig("billing_queue_switcher", "fake");
        TBillingQueueSwitcher process(processConfig);
        {
            const auto queues = process.GetQueues(server.Get(), "billing.queue_switcher.soft_update_queue");
            process.SoftSwitchQueue(billingManager, queues, USER_ID_DEFAULT, /* tasksAtOnce = */ 1, (ui32)TBillingLogic::ETaskPriority::RegularDebts);
            auto session = billingManager.BuildSession();
            auto optionalActiveTasks = billingManager.GetActiveTasksManager().GetTasks(session);
            UNIT_ASSERT(!!optionalActiveTasks);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->front().GetQueue(), EBillingQueue::Prestable);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->front().GetNextQueue(), EBillingQueue::Prestable);
        }

        server->GetSettings().SetValue("billing.queue_switcher.soft_update_queue", ToString(EBillingQueue::Prestable), USER_ROOT_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            const auto queues = process.GetQueues(server.Get(), "billing.queue_switcher.soft_update_queue");
            process.SoftSwitchQueue(billingManager, queues, USER_ID_DEFAULT, /* tasksAtOnce = */ 1, (ui32)TBillingLogic::ETaskPriority::RegularDebts);
            auto session = billingManager.BuildSession();
            auto optionalActiveTasks = billingManager.GetActiveTasksManager().GetTasks(session);
            UNIT_ASSERT(!!optionalActiveTasks);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->front().GetQueue(), EBillingQueue::Prestable);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->front().GetNextQueue(), billingManager.GetConfig().GetActiveQueue());
        }
    }

    Y_UNIT_TEST(HardQueueSwitcher) {
        TStringStream ss;
        NDrive::TServerConfigGenerator().ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto& billingManager = driveApi.GetBillingManager();
        {
            auto session = billingManager.BuildSession();
            TBillingTask task;
            task.SetId(OBJECT_ID_DEFAULT).SetUserId(USER_ID_DEFAULT).SetBillingType(EBillingType::CarUsage);
            task.SetQueue(EBillingQueue::Prestable);
            task.SetNextQueue(EBillingQueue::Prestable);
            UNIT_ASSERT(billingManager.CreateBillingTask(task, USER_ID_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        const TBillingQueueSwitcherConfig processConfig("billing_queue_switcher", "fake");
        TBillingQueueSwitcher process(processConfig);
        {
            const auto queues = process.GetQueues(server.Get(), "billing.queue_switcher.hard_update_queue");
            process.HardSwitchQueue(billingManager, queues, USER_ID_DEFAULT, /* tasksAtOnce = */ 1, (ui32)TBillingLogic::ETaskPriority::RegularDebts);
            auto session = billingManager.BuildSession();
            auto optionalActiveTasks = billingManager.GetActiveTasksManager().GetTasks(session);
            UNIT_ASSERT(!!optionalActiveTasks);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->size(), 1);
            UNIT_ASSERT_VALUES_UNEQUAL(optionalActiveTasks->front().GetQueue(), billingManager.GetConfig().GetActiveQueue());
        }

        server->GetSettings().SetValue("billing.queue_switcher.hard_update_queue", ToString(EBillingQueue::Prestable), USER_ROOT_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            const auto queues = process.GetQueues(server.Get(), "billing.queue_switcher.hard_update_queue");
            process.HardSwitchQueue(billingManager, queues, USER_ID_DEFAULT, /* tasksAtOnce = */ 1, (ui32)TBillingLogic::ETaskPriority::RegularDebts);
            auto session = billingManager.BuildSession();
            auto optionalActiveTasks = billingManager.GetActiveTasksManager().GetTasks(session);
            UNIT_ASSERT(!!optionalActiveTasks);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->front().GetQueue(), billingManager.GetConfig().GetActiveQueue());
            UNIT_ASSERT_VALUES_EQUAL(optionalActiveTasks->front().GetNextQueue(), billingManager.GetConfig().GetActiveQueue());
        }
    }
}
