#include <drive/backend/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>
#include <drive/backend/billing/accounts/limited.h>

using TPromoFilter = TEnvironmentGenerator::TPromoCodeFilter;

Y_UNIT_TEST_SUITE(PromoCodes) {
    Y_UNIT_TEST(SimpleOperations) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        {
            auto report = cg.GetPromoCodes();
            UNIT_ASSERT_C(report.IsDefined() && report.Has("codes") && report["codes"].IsArray(), TStringBuilder() << report << Endl);
            INFO_LOG << "Already exist codes count: " << report["codes"].GetArray().size() << Endl;
        }
        const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
        {
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT);
            INFO_LOG << "Generated codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
        TString code;
        {
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix, USER_ID_DEFAULT);
            INFO_LOG << "Generated codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") || report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            UNIT_ASSERT(report["codes"].GetArray().front().IsString());
            code = report["codes"].GetArray().front().GetStringRobust();
        }
        TString codeId;
        {
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            auto& jsonItem = report["codes"].GetArray().front();
            UNIT_ASSERT(NJson::ParseField(jsonItem, "id", codeId, true));
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["prefix"].GetStringRobust(), prefix);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["given_out"].GetStringRobust(), USER_ID_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["generator"].GetStringRobust(), PROMO_CODE_GENERATOR);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["usage_limit"].GetInteger(), 1);
        }
        {
            auto report = cg.CheckPromoCode(code);
            INFO_LOG << "Got code: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            auto& jsonItem = report["codes"].GetArray().front();
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["id"].GetStringRobust(), codeId);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["prefix"].GetStringRobust(), prefix);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["given_out"].GetStringRobust(), USER_ID_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["generator"].GetStringRobust(), PROMO_CODE_GENERATOR);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["usage_limit"].GetInteger(), 1);
        }
        {
            UNIT_ASSERT(cg.RemovePromoCodes({codeId}));
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
    }

    Y_UNIT_TEST(UserOperations) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
        {
            auto report = cg.GetPromoCodes(prefix);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
        TString codeId;
        {
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix);
            INFO_LOG << "Generated codes: " << report << Endl;
            UNIT_ASSERT(!report.IsDefined());
            report = cg.GetPromoCodes(prefix);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            UNIT_ASSERT(NJson::ParseField(report["codes"].GetArray().front(), "id", codeId, true));
        }
        {
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
        TString code;
        {
            auto report = cg.GiveOutPromoCode(USER_ID_DEFAULT, {codeId});
            INFO_LOG << "Given out codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") || report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            UNIT_ASSERT(report["codes"].GetArray().front().IsString());
            code = report["codes"].GetArray().front().GetStringRobust();
        }
        {
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
        }
        {
            UNIT_ASSERT(cg.AcceptPromoCode(code, USER_ID_DEFAULT));
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
        {
            auto report = cg.GetPromoCodeHistory(USER_ID_DEFAULT);
            INFO_LOG << "Got codes history: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") || report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            auto& jsonItem = report["codes"].GetArray().front();
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["id"].GetStringRobust(), codeId);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["prefix"].GetStringRobust(), prefix);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["given_out"].GetStringRobust(), USER_ID_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["generator"].GetStringRobust(), PROMO_CODE_GENERATOR);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["usage_limit"].GetInteger(), 1);
            UNIT_ASSERT(jsonItem["history"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["history"].GetArray().size(), 1);
            auto& historyItem = jsonItem["history"].GetArray().front();
            UNIT_ASSERT_VALUES_EQUAL(historyItem["user_id"].GetStringRobust(), USER_ID_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(historyItem["promo_id"].GetStringRobust(), codeId);
            UNIT_ASSERT_VALUES_EQUAL(historyItem["action"].GetStringRobust(), "remove");
        }
    }

    Y_UNIT_TEST(AccountOperations) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
        {
            auto report = cg.GetPromoCodes(prefix);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
        TString code;
        {
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR_ACCOUNT, 1, prefix, USER_ID_DEFAULT);
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            code = report["codes"].GetArray().front().GetStringRobust();
        }
        auto& billingManager = server->GetDriveAPI()->GetBillingManager();
        auto& accountsManager = billingManager.GetAccountsManager();;
        {
            auto session = billingManager.BuildSession();
            NJson::TJsonValue descJson;
            descJson["type"] = ::ToString(NDrive::NBilling::EAccount::Wallet);
            descJson["hard_limit"] = NDrive::NBilling::AccountLimit;
            descJson["soft_limit"] = NDrive::NBilling::AccountLimit;
            descJson["name"] = "y.drive";
            descJson["meta"]["hr_name"] = "y.drive";
            descJson["meta"]["refresh_policy"] = ToString(NDrive::NBilling::TRefreshSchedule::EPolicy::None);
            NDrive::NBilling::TAccountDescriptionRecord description;
            UNIT_ASSERT(description.FromJson(descJson, nullptr));
            UNIT_ASSERT(accountsManager.UpsertAccountDescription(description, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        auto registeredAccounts = accountsManager.GetRegisteredAccounts(TInstant::Now());
        ui32 walletType = 0;
        for (auto&& d : registeredAccounts) {
            if (d.GetName() == "y.drive") {
                walletType = d.GetId();
                break;
            }
        }
        UNIT_ASSERT(walletType > 0);
        ui32 accountId = 0;
        {
            auto session = billingManager.BuildSession();
            auto accountRecord = MakeAtomicShared<NDrive::NBilling::TLimitedAccountRecord>();
            accountRecord->SetTypeId(walletType);
            accountRecord->SetActive(true);
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, USER_ROOT_DEFAULT, session, &accountId));
            UNIT_ASSERT(accountsManager.LinkAccount(USER_ROOT_DEFAULT, accountId, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            UNIT_ASSERT(cg.AcceptPromoCode(code, USER_ID_DEFAULT, ::ToString(accountId)));
            TString tagId;
            UNIT_ASSERT(cg.GetTagId(::ToString(accountId), "promo_account", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Account, tagId));
        }
    }

    Y_UNIT_TEST(SimpleGiveOut) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
        {
            auto report = cg.GetPromoCodes(prefix);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
        TString codeId;
        {
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix);
            INFO_LOG << "Generated codes: " << report << Endl;
            UNIT_ASSERT(!report.IsDefined());
            report = cg.GetPromoCodes(prefix);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            UNIT_ASSERT(NJson::ParseField(report["codes"].GetArray().front(), "id", codeId, true));
        }
        {
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT(report["codes"].GetArray().empty());
        }
        TString code;
        {
            auto report = cg.GiveOutPromoCode(USER_ID_DEFAULT, {}, PROMO_CODE_GENERATOR, 1);
            INFO_LOG << "Given out codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") || report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            UNIT_ASSERT(report["codes"].GetArray().front().IsString());
            code = report["codes"].GetArray().front().GetStringRobust();
        }
        {
            auto report = cg.CheckPromoCode(code);
            INFO_LOG << "Got code: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            auto& jsonItem = report["codes"].GetArray().front();
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["id"].GetStringRobust(), codeId);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["prefix"].GetStringRobust(), prefix);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["generator"].GetStringRobust(), PROMO_CODE_GENERATOR);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["usage_limit"].GetInteger(), 1);
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["given_out"].GetStringRobust(), USER_ID_DEFAULT);
        }
    }

    Y_UNIT_TEST(SimpleSelect) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        auto genCode = [&cg] (const TInstant activityDeadline = TInstant::Zero()) {
            const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix, USER_ID_DEFAULT, activityDeadline);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            UNIT_ASSERT(report["codes"].GetArray().front().IsString());
            return report["codes"].GetArray().front().GetStringRobust();
        };
        {
            TString codeId;
            {
                auto report = cg.CheckPromoCode(genCode());
                INFO_LOG << "Got code: " << report << Endl;
                UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
                UNIT_ASSERT(NJson::ParseField(report["codes"].GetArray().front(), "id", codeId, true));
            }
            TPromoFilter filter;
            filter.Ids = {codeId};
            auto report = cg.GetPromoCodes(filter);
            UNIT_ASSERT_C(report.IsDefined() && report.Has("codes") && report["codes"].IsArray(), TStringBuilder() << report << Endl);
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            TString testCodeId;
            UNIT_ASSERT(NJson::ParseField(report["codes"].GetArray().front(), "id", testCodeId, true));
            UNIT_ASSERT_VALUES_EQUAL(testCodeId, codeId);
        }
        auto getCodes = [&cg] (const TPromoFilter& filter) {
            auto report = cg.GetPromoCodes(filter);
            UNIT_ASSERT_C(report.IsDefined() && report.Has("codes") && report["codes"].IsArray(), TStringBuilder() << report << Endl);
            return report["codes"].GetArray().size();
        };
        {
            const auto now = Now() + TDuration().Hours(1) + TDuration().Days(5);
            const auto since = now - TDuration().Days(1);
            const auto until = now + TDuration().Days(1);
            const auto old = now - TDuration().Days(2);
            const auto young = now + TDuration().Days(2);
            TPromoFilter filter;
            filter.Since = since;
            filter.Until = until;
            {
                const auto before = getCodes(filter);
                genCode(old);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before);
                genCode(young);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before);
                genCode(now);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before + 1);
            }
            {
                filter.Until = {};
                const auto before = getCodes(filter);
                genCode(old);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before);
                genCode(now);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before + 1);
                genCode(young);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before + 2);
            }
            {
                filter.Since = {};
                filter.Until = until;
                const auto before = getCodes(filter);
                genCode(young);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before);
                genCode(old);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before + 1);
                genCode(now);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before + 2);
            }
        }
        {
            TPromoFilter filter;
            {
                filter.GivenOut = "__FILLED";
                const auto before = getCodes(filter);
                {
                    const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
                    auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix);
                    INFO_LOG << "Got codes: " << report << Endl;
                    UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before);
                }
                {
                    genCode();
                    UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before + 1);
                }
            }
            {
                filter.GivenOut = "__EMPTY";
                const auto before = getCodes(filter);
                {
                    genCode();
                    UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before);
                }
                {
                    const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
                    auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix);
                    INFO_LOG << "Got codes: " << report << Endl;
                    UNIT_ASSERT_VALUES_EQUAL(getCodes(filter), before + 1);
                }
            }
        }
    }

    Y_UNIT_TEST(SelectLimit) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        auto getCodes = [&cg] (const TString& prefix, const ui32 count = 1000) {
            TPromoFilter filter;
            filter.ActiveOnly = true;
            filter.Count = count;
            filter.Prefix = prefix;
            auto report = cg.GetPromoCodes(filter);
            UNIT_ASSERT_C(report.IsDefined() && report.Has("codes") && report["codes"].IsArray(), TStringBuilder() << report << Endl);
            return report["codes"].GetArray().size();
        };
        const TString prefix1 = TStringBuilder() << "tst1_" << Now().Seconds() << "_";
        {
            const ui32 count = 50;
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, count, prefix1, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), count);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix1), count);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix1, count / 2), count / 2);
        }
        const TString prefix2 = TStringBuilder() << "tst2_" << Now().Seconds() << "_";
        {
            const ui32 count = 1001;
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, count, prefix2, USER_ID_DEFAULT);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), count);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix2, 1000), 1000);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix2, count), count);
        }
        const TString prefix3 = TStringBuilder() << "tst3_" << Now().Seconds() << "_";
        {
            {
                const ui32 count = 301;
                auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, count, prefix3, USER_ID_DEFAULT);
                INFO_LOG << "Got codes: " << report << Endl;
                UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), count);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix3, count), count);
                UNIT_ASSERT(cg.AcceptPromoCode(report["codes"].GetArray().front().GetStringRobust(), USER_ID_DEFAULT));
                UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix3), count - 1);
            }
            {
                const ui32 count = 500;
                auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, count, prefix3, USER_ID_DEFAULT);
                INFO_LOG << "Got codes: " << report << Endl;
                UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
                UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), count);
                UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix3), 800);
            }
        }
        UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix1), 50);
        UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix2, 2000), 1001);
        UNIT_ASSERT_VALUES_EQUAL(getCodes(prefix3), 800);
    }

    Y_UNIT_TEST(GeneratorSelect) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        auto getCodes = [&cg] (const TString& generator = "") {
            TPromoFilter filter;
            if (generator) {
                filter.Generator = generator;
            }
            filter.ActiveOnly = true;
            auto report = cg.GetPromoCodes(filter);
            UNIT_ASSERT_C(report.IsDefined() && report.Has("codes") && report["codes"].IsArray(), TStringBuilder() << report << Endl);
            return report["codes"].GetArray().size();
        };
        const auto before = getCodes();
        const auto beforeFirst = getCodes(PROMO_CODE_GENERATOR);
        const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
        {
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(PROMO_CODE_GENERATOR), beforeFirst + 1);
        }
        const auto beforeSecond = getCodes(PROMO_CODE_GENERATOR_SECOND);
        {
            auto report = cg.GeneratePromoCodes(PROMO_CODE_GENERATOR_SECOND, 1, prefix);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(PROMO_CODE_GENERATOR_SECOND), beforeSecond + 1);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(PROMO_CODE_GENERATOR), beforeFirst + 1);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 2);
        }
    }

    void SetCreateReferralCodeSettings(NDrive::TServerGuard& server, const TString prefix) {
        const TString handlerSetting = "handlers.user_app/promo/create_referral_code.";
        UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "generator", PROMO_CODE_GENERATOR, USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "settings_key", "referral_code", USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "chat_topic_link", "referral", USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "min_custom_code_length", "16", USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "max_custom_code_length", "20", USER_ROOT_DEFAULT));
        NJson::TJsonValue context = NJson::TMapBuilder
            ("prefix", prefix)
            ("usage_limit", 1)
            ("upper_literals_usage", true)
            ("lower_literals_usage", true)
            ("digits_usage", true);
        UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + PROMO_CODE_GENERATOR + "_standart_context", context.GetStringRobust(), USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSettings().SetValue("enable_referral", "true", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
    }

    Y_UNIT_TEST(CreateReferralCode) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TString prefix = TStringBuilder() << "tst_" << Now().Seconds() << "_";
        SetCreateReferralCodeSettings(server, prefix);
        auto getCodes = [&cg] () {
            TPromoFilter filter;
            filter.Generator = PROMO_CODE_GENERATOR;
            filter.ActiveOnly = true;
            auto report = cg.GetPromoCodes(filter);
            UNIT_ASSERT_C(report.IsDefined() && report.Has("codes") && report["codes"].IsArray(), TStringBuilder() << report << Endl);
            return report["codes"].GetArray().size();
        };
        const TString referralCodeParam = "referral_code";
        const auto before = getCodes();
        {
            {
                auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
                UNIT_ASSERT(param);
                UNIT_ASSERT(param->empty());
            }
            UNIT_ASSERT(cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam));
            cg.GeneratePromoCodes(PROMO_CODE_GENERATOR, 1, prefix);
            UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 1);
            UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT));
            UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 2);
            {
                auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
                UNIT_ASSERT(param && !param->empty());
                INFO_LOG << "New code: " << *param << Endl;
            }
        }
        const TString ownedCode = "TestCode" + ToString(Now().Seconds());
        const TString customReferralCodeParam = "custom_referral_code";
        {
            const TString& code = ownedCode;
            UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT1, code));
            UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 3);
            auto report = cg.GetPromoCodes(prefix, USER_ID_DEFAULT1);
            INFO_LOG << "Got codes: " << report << Endl;
            UNIT_ASSERT(report.IsDefined() && report.Has("codes") && report["codes"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(report["codes"].GetArray().size(), 1);
            auto& jsonItem = report["codes"].GetArray().front();
            UNIT_ASSERT(jsonItem.Has("code"));
            UNIT_ASSERT_VALUES_EQUAL(jsonItem["code"].GetStringRobust(), code);
            auto param = cg.GetPersonalSetting(USER_ID_DEFAULT1, customReferralCodeParam);
            UNIT_ASSERT(param && !param->empty());
            UNIT_ASSERT_VALUES_EQUAL(*param, code);
        }
        {
            const TString code = "TooLongTestCode" + ToString(Now().Seconds());
            UNIT_ASSERT(!cg.CreateReferralCode(USER_ID_DEFAULT2, code));
        }
        {
            const TString code = "short" + ToString(Now().Seconds());
            UNIT_ASSERT(!cg.CreateReferralCode(USER_ID_DEFAULT2, code));
        }
        {
            const TString code = "кириллица" + ToString(Now().Seconds());
            UNIT_ASSERT(!cg.CreateReferralCode(USER_ID_DEFAULT2, code));
        }
        {
            const TString code = "with space" + ToString(Now().Seconds());
            UNIT_ASSERT(!cg.CreateReferralCode(USER_ID_DEFAULT2, code));
        }
        UNIT_ASSERT(!cg.CreateReferralCode(USER_ID_DEFAULT2, ownedCode));
        UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 3);
        {
            const TString handlerSetting = "handlers.user_app/promo/create_referral_code.";
            const TString customError = "{\"custom_test_error\": \"[a-z]+\"}";
            UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "custom_errors", customError, USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            const TString code = "customerr" + ToString(Now().Seconds());
            UNIT_ASSERT(!cg.CreateReferralCode(USER_ID_DEFAULT2, code));
            UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "custom_errors", "{}", USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }
        UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT2));
        UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 4);
        UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT1));
        UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 5);
        TString recentlCode;
        {
            auto param = cg.GetPersonalSetting(USER_ID_DEFAULT1, referralCodeParam);
            UNIT_ASSERT(param && !param->empty());
            recentlCode = *param;
        }
        UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT1));
        UNIT_ASSERT_VALUES_EQUAL(getCodes(), before + 5);
        auto param = cg.GetPersonalSetting(USER_ID_DEFAULT1, referralCodeParam);
        UNIT_ASSERT(param);
        UNIT_ASSERT_VALUES_EQUAL(recentlCode, *param);
    }

    Y_UNIT_TEST(ReferralProgramParticipation) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        SetCreateReferralCodeSettings(server, TStringBuilder() << "tst_" << Now().Seconds() << "_");
        const TString referralCodeParam = "referral_code";
        {
            UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT));
            {
                auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
                UNIT_ASSERT(param && !param->empty());
            }
            auto permissions = server->GetDriveAPI()->GetUserPermissions(USER_ID_DEFAULT, {});
            UNIT_ASSERT(permissions);
            {
                TString code;
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->CheckReferralProgramParticipation(code, *permissions, session));
                auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
                UNIT_ASSERT(param);
                UNIT_ASSERT_VALUES_EQUAL(code, *param);
            }
            const TString customCode = "custom" + ToString(Now().Seconds());
            UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT, customCode));
            {
                TString code;
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->CheckReferralProgramParticipation(code, *permissions, session));
                UNIT_ASSERT_VALUES_EQUAL(code, customCode);
            }
            UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT));
            {
                TString code;
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->CheckReferralProgramParticipation(code, *permissions, session));
                auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
                UNIT_ASSERT(param);
                UNIT_ASSERT_VALUES_EQUAL(code, *param);
            }
        }
        {
            const TString customCode = "custom" + ToString(Now().Seconds());
            UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT1, customCode));
            TString code;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto permissions = server->GetDriveAPI()->GetUserPermissions(USER_ID_DEFAULT1, {});
            UNIT_ASSERT(permissions);
            UNIT_ASSERT(server->GetDriveAPI()->CheckReferralProgramParticipation(code, *permissions, session));
            UNIT_ASSERT_VALUES_EQUAL(code, customCode);
        }
    }

    Y_UNIT_TEST(ReferralProgramParticipationWithCustomCode) {
        NDrive::TServerConfigGenerator cg;
        TServerConfigConstructorParams params(cg.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TString handlerSetting = "handlers.user_app/promo/referral.";
        const TString reportTemplate = "{\"code\": \"_PROMOCODE_\", \"is_custom\": _IS_CUSTOM_REFERRAL_CODE_}";
        UNIT_ASSERT(server->GetSettings().SetValue(handlerSetting + "report_template", reportTemplate, USER_ROOT_DEFAULT));
        SetCreateReferralCodeSettings(server, TStringBuilder() << "tst_" << Now().Seconds() << "_");
        UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT));
        TString defCode;
        const TString referralCodeParam = "referral_code";
        {
            auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
            UNIT_ASSERT(param && !param->empty());
            defCode = *param;
        }
        {
            auto report = cg.GetPromoInfo(USER_ID_DEFAULT);
            UNIT_ASSERT(report.Has("code"));
            UNIT_ASSERT_VALUES_EQUAL(report["code"], defCode);
            UNIT_ASSERT(report.Has("is_custom") && report["is_custom"].IsBoolean() && !report["is_custom"].GetBooleanRobust());
        }
        const TString customCode = "custom" + ToString(Now().Seconds());
        UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT, customCode));
        {
            auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
            UNIT_ASSERT(param);
            UNIT_ASSERT_VALUES_EQUAL(*param, defCode);
        }
        {
            auto report = cg.GetPromoInfo(USER_ID_DEFAULT);
            UNIT_ASSERT(report.Has("code"));
            UNIT_ASSERT_VALUES_EQUAL(report["code"], customCode);
            UNIT_ASSERT(report.Has("is_custom") && report["is_custom"].IsBoolean() && report["is_custom"].GetBooleanRobust());
        }
        UNIT_ASSERT(cg.CreateReferralCode(USER_ID_DEFAULT));
        {
            auto report = cg.GetPromoInfo(USER_ID_DEFAULT);
            UNIT_ASSERT(report.Has("code"));
            UNIT_ASSERT_VALUES_EQUAL(report["code"], defCode);
            UNIT_ASSERT(report.Has("is_custom") && report["is_custom"].IsBoolean() && !report["is_custom"].GetBooleanRobust());
        }
        {
            auto param = cg.GetPersonalSetting(USER_ID_DEFAULT, referralCodeParam);
            UNIT_ASSERT(param);
            UNIT_ASSERT_VALUES_EQUAL(*param, defCode);
        }
    }
}
