#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/data/radar.h>
#include <drive/backend/radar/radar_geohash.h>
#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

Y_UNIT_TEST_SUITE(RadarGeohash) {
    THolder<TRadarUserTag> GetRadarTag(TDuration walkDuration, TGeoCoord coord) {
        TDuration delay = TDuration::Seconds(2);
        TInstant startTime = ModelingNow() + delay;
        TVector<TString> filters = {"filter_porsche_macan"};

        THolder<TRadarUserTag> tag(new TRadarUserTag(TRadarUserTag::TypeName));
        TGeoRect polygon = {coord};
        polygon.GrowDistance(walkDuration.Seconds());

        tag->SetSearchArea(polygon.GetCoords());
        tag->SetSLAInstant(startTime + TDuration::Seconds(5));
        tag->SetWalkDuration(walkDuration);
        tag->SetFilterIdentifiers(filters);
        tag->SetStartTimestamp(startTime);
        tag->SetAction(TRadarUserTag::EAction::Order);
        tag->SetRadarPosition(coord);

        return tag;
    }
    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        server->GetSettings().SetValue("radar.use_geohash_add", "true", USER_ROOT_DEFAULT);
        server->GetSettings().SetValue("radar.use_geohash_remove", "true", USER_ROOT_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto radarGeohashManager = server->GetRadarGeohashManager();
        UNIT_ASSERT(radarGeohashManager);
        auto userId1 = eGenerator.CreateUser("sultan_with_radar1", false, "active");
        auto userId2 = eGenerator.CreateUser("sultan_with_radar2", false, "active");
        auto userId3 = eGenerator.CreateUser("sultan_with_radar3", false, "active");

        TDuration walkDuration1 = TDuration::Minutes(25);
        TGeoCoord coord1(37.52175528, 55.71331785);
        TDuration walkDuration2 = TDuration::Minutes(15);
        TGeoCoord coord2(37.51566102, 55.71237919);
        TDuration walkDuration3 = TDuration::Minutes(10);
        TGeoCoord coord3(37.78372899, 55.74443722);

        auto tag1 = GetRadarTag(walkDuration1, coord1);
        auto tag2 = GetRadarTag(walkDuration2, coord2);
        auto tag3 = GetRadarTag(walkDuration3, coord3);

        ui8 precision = 7;
        TGeoRect polygon = {coord1};
        polygon.GrowDistance(walkDuration1.Seconds());

        TDBTag dbTag1;
        TDBTag dbTag2;
        TDBTag dbTag3;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto op_tag = server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag1.Release(), userId1, userId1, server.Get(), session);
            UNIT_ASSERT_C(op_tag, session.GetStringReport());
            dbTag1 = op_tag->front();
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = radarGeohashManager->GetTagIds(coord1, precision, session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(*result->begin(), dbTag1.GetTagId());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto op_tag = server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag2.Release(), userId2, userId2, server.Get(), session);
            UNIT_ASSERT_C(op_tag, session.GetStringReport());
            dbTag2 = op_tag->front();
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = radarGeohashManager->GetTagIds(coord1, precision, session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result->size(), 2);
            UNIT_ASSERT_VALUES_UNEQUAL(*result->begin(), *result->begin() + 1);
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto op_tag = server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag3.Release(), userId3, userId3, server.Get(), session);
            UNIT_ASSERT_C(op_tag, session.GetStringReport());
            dbTag3 = op_tag->front();
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = radarGeohashManager->GetTagIds(coord1, precision, session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result->size(), 2);
            UNIT_ASSERT_VALUES_UNEQUAL(*result->begin(), *result->begin() + 1);
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(dbTag1, userId1, server.Get(), session, true), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = radarGeohashManager->GetTagIds(coord1, precision, session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result->size(), 1);
        }
        {
            TSet<TString> result1 = radarGeohashManager->GetPolygonTiles(polygon.GetCoords(), precision);
            UNIT_ASSERT_VALUES_EQUAL(result1.size(), 735);
            precision = 6;
            TSet<TString> result2 = radarGeohashManager->GetPolygonTiles(polygon.GetCoords(), precision);
            UNIT_ASSERT_VALUES_EQUAL(result2.size(), 30);
            TSet<TString> precision6GeoHashes{"ucftj2", "ucftj8", "ucftjb", "ucftn0", "ucftn2",
                                              "ucfsvr", "ucfsvx", "ucfsvz", "ucfsyp", "ucfsyr",
                                              "ucfsvq", "ucfsvw", "ucfsvy", "ucfsyn", "ucfsyq",
                                              "ucfsvm", "ucfsvt", "ucfsvv", "ucfsyj", "ucfsym",
                                              "ucfsvk", "ucfsvs", "ucfsvu", "ucfsyh", "ucfsyk",
                                              "ucfsv7", "ucfsve", "ucfsvg", "ucfsy5", "ucfsy7"};
            UNIT_ASSERT_VALUES_EQUAL(result2, precision6GeoHashes);
            TSet<TString> result3 = radarGeohashManager->GetPolygonTiles({}, precision);
            UNIT_ASSERT_VALUES_EQUAL(result3.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(*result3.begin(), "");
        }
    }

    Y_UNIT_TEST(Update) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        server->GetSettings().SetValue("radar.use_geohash_add", "true", USER_ROOT_DEFAULT);
        server->GetSettings().SetValue("radar.use_geohash_remove", "true", USER_ROOT_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto radarGeohashManager = server->GetRadarGeohashManager();
        UNIT_ASSERT(radarGeohashManager);
        UNIT_ASSERT(config.GetRadarGeohashConfig());
        auto radarDB = server->GetDatabase(config.GetRadarGeohashConfig()->GetDBName());
        UNIT_ASSERT(radarDB);
        TRadarGeohashDB radarGeohashDB{THistoryContext(radarDB)};
        auto userId1 = eGenerator.CreateUser("sultan_with_radar1", false, "active");
        TDuration walkDuration1 = TDuration::Minutes(25);
        TGeoCoord coord1(37.52175528, 55.71331785);
        TDuration walkDuration2 = TDuration::Minutes(15);
        TGeoCoord coord2(30.3350986, 59.9342802);
        TDuration walkDuration3 = TDuration::Minutes(10);
        TGeoCoord coord3(27.567444, 53.893009);
        auto tag1 = GetRadarTag(walkDuration1, coord1);
        auto tag2 = GetRadarTag(walkDuration2, coord2);
        auto tag3 = GetRadarTag(walkDuration3, coord3);
        ui8 precision = 6;
        TSet<TString> tiles1 = radarGeohashManager->GetPolygonTiles(tag1->GetSearchArea(), precision);
        TSet<TString> tiles2 = radarGeohashManager->GetPolygonTiles(tag2->GetSearchArea(), precision);
        TSet<TString> tiles3 = radarGeohashManager->GetPolygonTiles(tag3->GetSearchArea(), precision);
        UNIT_ASSERT_VALUES_UNEQUAL(tiles1, tiles2);
        UNIT_ASSERT_VALUES_UNEQUAL(tiles2, tiles3);
        UNIT_ASSERT_VALUES_UNEQUAL(tiles1, tiles3);
        TDBTag dbTag1;
        // Create tag.
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto optionalTag = server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag1.Release(), userId1, userId1, server.Get(), session);
            UNIT_ASSERT_C(optionalTag, session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
            dbTag1 = optionalTag->front();
        }
        // Check geohash index.
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto fetchResult = radarGeohashDB.FetchInfo(session, NSQL::TQueryOptions().AddGenericCondition("tag_id", dbTag1.GetTagId()));
            UNIT_ASSERT_C(fetchResult, session.GetStringReport());
            TSet<TString> result;
            for (auto&& [_, entity] : fetchResult) {
                result.insert(std::move(entity.GetTileId()));
            }
            UNIT_ASSERT_VALUES_EQUAL(result, tiles1);
        }
        // Update tag.
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(dbTag1, tag2.Release(), userId1, server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        // Check geohash index.
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto fetchResult = radarGeohashDB.FetchInfo(session, NSQL::TQueryOptions().AddGenericCondition("tag_id", dbTag1.GetTagId()));
            UNIT_ASSERT_C(fetchResult, session.GetStringReport());
            TSet<TString> result;
            for (auto&& [_, entity] : fetchResult) {
                result.insert(std::move(entity.GetTileId()));
            }
            UNIT_ASSERT_VALUES_EQUAL(result, tiles2);
        }
        // Update tag.
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(dbTag1, tag3.Release(), userId1, server.Get(), session), session.GetStringReport());
            UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
        }
        // Check geohash index.
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto fetchResult = radarGeohashDB.FetchInfo(session, NSQL::TQueryOptions().AddGenericCondition("tag_id", dbTag1.GetTagId()));
            UNIT_ASSERT_C(fetchResult, session.GetStringReport());
            TSet<TString> result;
            for (auto&& [_, entity] : fetchResult) {
                result.insert(std::move(entity.GetTileId()));
            }
            UNIT_ASSERT_VALUES_EQUAL(result, tiles3);
        }
    }
}
