#include <drive/backend/data/radar.h>
#include <drive/backend/processors/car_scanner/processor.h>
#include <drive/backend/ut/library/script.h>

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(Radar) {

    TMap<TString, TString> radarSettings = {
        {"handlers.default.radar.create_tag.enabled", "true"},
        {"radar.close_session.enabled", "true"},
        {"radar.use_geohash_add", "true"},
        {"radar.use_geohash_remove", "true"},
        {"radar.use_geohash_find", "true"},
    };

    TGeoCoord userCoordinates(37.52175528, 55.71331785);
    TGeoCoord carCoordinates(37.5238721, 55.71331785);

    auto assertWithSession = [](const NJson::TJsonValue& report) {
        UNIT_ASSERT(report["segment"]["meta"]["finished"].IsBoolean());
        UNIT_ASSERT(!report["segment"]["meta"]["finished"].GetBoolean());
    };

    auto assertWithoutSession = [](const NJson::TJsonValue& report) {
        UNIT_ASSERT(!report.Has("segment"));
    };

    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator gen;
        gen.SetNeedBackground(0);
        NDrive::NTest::TScript script(gen);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Reset(radarSettings);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(carCoordinates);
        // Book offer for user 1.
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetOfferType("standart_offer");
        // Check session for user 1.
        script.Add<NDrive::NTest::TCheckCurrentSession>(assertWithSession).SetUserId(USER_ID_DEFAULT).SetExpectOK(true);
        // Start scanner for user 2.
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& /*context*/) {
            UNIT_ASSERT(gen.StartScanner(USER_ID_DEFAULT2, userCoordinates, "filter_porsche_carrera", TDuration::Seconds(2000), "order"));
        });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(15));
        // Check session for user 2.
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            auto radarTags = context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT2, { TRadarUserTag::TypeName }, session);
            UNIT_ASSERT(radarTags);
            UNIT_ASSERT_EQUAL(radarTags->size(), 1);
        });
        script.Add<NDrive::NTest::TCheckCurrentSession>(assertWithoutSession).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        // Drop session for user 1.
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(1)).SetExpectOK(true);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(15));
        // Check session for user 2.
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            auto radarTags = context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT2, { TRadarUserTag::TypeName }, session);
            UNIT_ASSERT(radarTags);
            UNIT_ASSERT_EQUAL(radarTags->size(), 0);
        });
        script.Add<NDrive::NTest::TCheckCurrentSession>(assertWithSession).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        // Check push notifications for user 2.
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            auto radarPushTags = context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT2, { "radarpush_booked" }, session);
            UNIT_ASSERT(radarPushTags);
            UNIT_ASSERT_EQUAL(radarPushTags->size(), 1);
        });
        // // Drop session for user 2.
        // script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT2);
        // script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(1)).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(WrongFilters) {
        NDrive::TServerConfigGenerator gen;
        gen.SetNeedBackground(0);
        NDrive::NTest::TScript script(gen);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Reset(radarSettings);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(carCoordinates);
        // Book offer for user 1.
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetOfferType("standart_offer");
        // Start scanner for user 2.
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& /*context*/) {
            UNIT_ASSERT(gen.StartScanner(USER_ID_DEFAULT2, userCoordinates, "filter_porsche_macan", TDuration::Seconds(2000), "order"));
        });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(15));
        // Check session for user 2.
        script.Add<NDrive::NTest::TCheckCurrentSession>(assertWithoutSession).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        // Drop session for user 1.
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(1)).SetExpectOK(true);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(10));
        // Check session for user 2.
        script.Add<NDrive::NTest::TCheckCurrentSession>(assertWithoutSession).SetUserId(USER_ID_DEFAULT2).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(SortUsers) {
        NDrive::TServerConfigGenerator gen;
        gen.SetNeedBackground(0);
        NDrive::NTest::TScript script(gen);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetSetting>().Reset(radarSettings);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(carCoordinates);
        // Book offer for special user.
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            TString userId = context.GetEGenerator().CreateUser("sultan-was-there", true, "active");
            UNIT_ASSERT(userId);
            context.SetUserId(userId);
        });
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetOfferType("standart_offer");
        // Start scanner for users.
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& /*context*/) {
            UNIT_ASSERT(gen.StartScanner(USER_ID_DEFAULT, userCoordinates, "filter_porsche_carrera", TDuration::Seconds(2000), "order"));
            Sleep(TDuration::Seconds(2));
            UNIT_ASSERT(gen.StartScanner(USER_ID_DEFAULT2, userCoordinates, "filter_porsche_carrera", TDuration::Seconds(2000), "order"));
        });
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(15));
        // Drop session or special user.
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(1)).SetExpectOK(true);
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(10));
        script.Add<NDrive::NTest::TCheckCurrentSession>(assertWithoutSession).SetUserId(MakeMaybe(USER_ID_DEFAULT2)).SetExpectOK(true);
        script.Add<NDrive::NTest::TCheckCurrentSession>(assertWithSession).SetUserId(MakeMaybe(USER_ID_DEFAULT)).SetExpectOK(true);
        script.Add<NDrive::NTest::TCommonChecker>([&](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            auto radarPushTag = context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, { "radarpush_booked" }, session);
            UNIT_ASSERT(radarPushTag);
            UNIT_ASSERT_EQUAL(radarPushTag->size(), 1);
        });
        UNIT_ASSERT(script.Execute());
    }
}
