#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/registrar/hasher.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>


#define NEW_CAR_ID_DEFAULT "c64943b5-b3a1-422d-8878-9c9ccf82c34d"


class TDocumentAcquisitionTestCallback : public IDocumentMediaAcquisitionCallback {
private:
    TMutex Mutex;
    ui32 TotalResponses = 0;
    ui32 TotalSuccesses = 0;

    TString LastSuccessContent;

public:
    void OnSuccess(const TString& photoId, const TString& content) override {
        TGuard<TMutex> g(Mutex);
        INFO_LOG << "Success: " << photoId << Endl;
        ++TotalResponses;

        ++TotalSuccesses;
        LastSuccessContent = content;
    }

    void OnFailure(const TString& photoId) override {
        TGuard<TMutex> g(Mutex);
        INFO_LOG << "Failure: " << photoId << Endl;
        ++TotalResponses;
    }

    TString GetLastSuccessContent() const {
        TGuard<TMutex> g(Mutex);
        return LastSuccessContent;
    }

    ui32 GetTotalResponses() const {
        TGuard<TMutex> g(Mutex);
        return TotalResponses;
    }

    ui32 GetTotalSuccesses() const {
        return TotalSuccesses;
    }
};

Y_UNIT_TEST_SUITE(RegistrationChat) {

    Y_UNIT_TEST(Util) {
        {
            TString a = "сергей";
            auto result = NRegistrarUtil::ToTitle(a);
            UNIT_ASSERT_VALUES_EQUAL(result, "Сергей");
        }

        {
            TString a = "abcdef";
            auto result = NRegistrarUtil::ToTitle(a);
            UNIT_ASSERT_VALUES_EQUAL(result, "Abcdef");
        }
    }

    Y_UNIT_TEST(BindEmail) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId, "zxczxczxc")["error_details"]["special_info"]["error_code"].GetString(), "email.invalid");
        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId, "zxczxczxc@akjsdhkjasd")["error_details"]["special_info"]["error_code"].GetString(), "email.invalid");
        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId, "zx.c.z.x.c.z.xc@akjsdhkjasd")["error_details"]["special_info"]["error_code"].GetString(), "email.invalid");
        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId, "zxsdasdxc@akjsdhkjasd.ru")["status"].GetString(), "success");

        auto userId2 = eGenerator.CreateUser("zxqfd555-was-there");
        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId2, "zxsdasdxc@akjsdhkjasd.ru")["error_details"]["special_info"]["error_code"].GetString(), "email.exists");
        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId2, "zxsdasdxc@akjsdhkjasd2.ru")["status"].GetString(), "success");

        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId, "zxsdasdxc@akjsdhkjasd3.ru")["status"].GetString(), "success");
        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId2, "zxsdasdxc@akjsdhkjasd.ru")["status"].GetString(), "success");

        {
            auto permissionsPtr = server->GetDriveAPI()->GetUserPermissions(userId2, TUserPermissionsFeatures());
            UNIT_ASSERT(!!permissionsPtr && permissionsPtr->HasAction("standart_offer_constructor"));
        }
        UNIT_ASSERT(server->GetSettings().SetValue("handlers.api/yandex/user/email.send_confirmation_email_action", "standart_offer_constructor", USER_ROOT_DEFAULT));
        UNIT_ASSERT_VALUES_EQUAL(configGenerator.BindEmail(userId2, "with_binding_zxsdasdxc@akjsdhkjasd.ru")["status"].GetString(), "success");
    }

    Y_UNIT_TEST(PhotoUpload) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);
        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
        auto testUserId = eGenerator.CreateUser("zxqfd-was-there");

        TString photoContent;
        try {
            photoContent = Base64Decode(configGenerator.GetValidImageString());
        } catch (const std::exception& e) {
            UNIT_ASSERT_C(false, FormatExc(e));
        }
        TString videoContent = "";
        for (size_t i = 0; i < 1024; ++i) {
            videoContent += "b";
        }

        TString photoId;
        {
            auto addFuture = docPhotoManager.AddDocumentPhoto(testUserId, "registration", NUserDocument::EType::PassportSelfie, photoContent, "", *server, true);
            addFuture.Wait();
            UNIT_ASSERT_C(!addFuture.HasException(), NThreading::GetExceptionMessage(addFuture));
            UNIT_ASSERT_C(addFuture.HasValue(), "add document future has no value");
            photoId = addFuture.GetValue().GetPhoto().GetId();
            UNIT_ASSERT_VALUES_EQUAL(docPhotoManager.GetUserPhotosDB().FetchInfo(photoId).GetResult().begin()->second.GetUserId(), testUserId);
            UNIT_ASSERT_VALUES_EQUAL(docPhotoManager.GetUserBackgroundVideosDB().FetchInfo(photoId).GetResult().size(), 1);
        }

        {
            auto photoFuture = docPhotoManager.GetDocumentPhoto(photoId, *server);
            photoFuture.Wait();
            UNIT_ASSERT_C(!photoFuture.HasException(), NThreading::GetExceptionMessage(photoFuture));
            UNIT_ASSERT_C(photoFuture.HasValue(), "get document future has no value");
            UNIT_ASSERT_VALUES_EQUAL(photoFuture.GetValue().GetContent(), photoContent);
        }

        UNIT_ASSERT(configGenerator.SubmitBv(testUserId, NUserDocument::EType::PassportSelfie, videoContent));

        {
            auto videoFuture = docPhotoManager.GetDocumentBackgroundVideo(photoId, *server);
            videoFuture.Wait();
            UNIT_ASSERT_C(!videoFuture.HasException(), NThreading::GetExceptionMessage(videoFuture));
            UNIT_ASSERT_C(videoFuture.HasValue(), "get document future has no value");
            UNIT_ASSERT_VALUES_EQUAL(videoFuture.GetValue().GetContent(), videoContent);
        }
    }

    Y_UNIT_TEST(CipherBackwardsCompatibility) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);
        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        {
            TAtomicSharedPtr<TDocumentAcquisitionTestCallback> callbackAcquire = new TDocumentAcquisitionTestCallback();
            docPhotoManager.GetDocumentPhoto("0b78b0e4-98fd-4dbb-98dd-a2d804be26e4", callbackAcquire, *server);
            while (callbackAcquire->GetTotalResponses() < 1) ;
            UNIT_ASSERT_VALUES_EQUAL(callbackAcquire->GetTotalSuccesses(), 1);
        }
        {
            TAtomicSharedPtr<TDocumentAcquisitionTestCallback> callbackAcquire = new TDocumentAcquisitionTestCallback();
            docPhotoManager.GetDocumentBackgroundVideo("0b78b0e4-98fd-4dbb-98dd-a2d804be26e4", callbackAcquire, *server);
            while (callbackAcquire->GetTotalResponses() < 1) ;
            UNIT_ASSERT_VALUES_EQUAL(callbackAcquire->GetTotalSuccesses(), 1);
            TString videoContent = callbackAcquire->GetLastSuccessContent();
            unsigned char firstChars[32] = {0, 0, 0, 28, 102, 116, 121, 112, 109, 112, 52, 50, 0, 0, 0, 1, 109, 112, 52, 49, 109, 112, 52, 50, 105, 115, 111, 109, 0, 0, 0, 1};
            for (size_t i = 0; i < 32; ++i) {
                UNIT_ASSERT_VALUES_EQUAL((unsigned char)videoContent[i], firstChars[i]);
            }
        }
    }

    Y_UNIT_TEST(Hasher) {
        // Ensure that both backends do passport and driving license number hashing in the same way
        TSensitiveDataHasher hasher("90d8eb23f8eb14ebd95c49edc81dfcc8");
        UNIT_ASSERT_VALUES_EQUAL(hasher.GetHash("HB2352296"), "f3c266214dfa332bd6386c215c34c3043bb03291");
        UNIT_ASSERT_VALUES_EQUAL(hasher.GetHash("HB2352297"), "a98b892c32faf66b410d7725aee102fe625a6ce5");
    }

    Y_UNIT_TEST(ChatScriptParsing) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        NJson::TJsonValue json;
        TFileInput in(JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_correct.json"));
        ReadJsonTree(&in, &json);

        TChatRobotScript s;
        TMessagesCollector errors;
        UNIT_ASSERT_C(s.Parse(json, errors), TStringBuilder() << errors.GetStringReport());
        UNIT_ASSERT_VALUES_EQUAL(s.GetFaqUrl(), "https://yandex.ru/support/drive/joining/joining-faq.html");
    }

    Y_UNIT_TEST(ArbitraryMessage) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        // Intro
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        {
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("registration", userId, "hello!"));
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 3);
        }

        {
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("registration", userId, "hello!"));
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
        }
    }

    Y_UNIT_TEST(ChatsList) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            auto result = configGenerator.GetChatsList(userId, server.Get(), "")["chats"];
            UNIT_ASSERT_VALUES_EQUAL(result.GetArray().size(), 0);
        }

        // Intro
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        {
            auto result = configGenerator.GetChatsList(userId, server.Get(), "")["chats"];
            UNIT_ASSERT_VALUES_EQUAL(result.GetArray().size(), 1);
        }
    }

    Y_UNIT_TEST(RegistrationProcess) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        // And now check that Yang document verification assignment was created for this user
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
        TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
        UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
        TString newAssignmentId = newAssignmentIds.front();
        UNIT_ASSERT(!!newAssignmentId);

        // Check resubmit. The message should be created.
        {
            UNIT_ASSERT(configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 27);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
        }

        // Resubmit request should not work during active resubmit process. Send LicenseFront
        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 29);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_back");
        }

        // Checked that now LicenseBack is required. Send it.
        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 31);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["messages"].GetArray().size());
        }

        // Now, resubmit should be working again
        {
            UNIT_ASSERT(configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::PassportSelfie}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 32);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_selfie");
        }

    }

    Y_UNIT_TEST(PhotoResubmitRequestBeforeCardBinding) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, false);

        // Request resubmit of some photos. Nothing shall change.
        {
            auto session = server->GetChatEngine()->BuildSession();

            auto robotPtr = server->GetChatRobot("registration");
            auto robotImpl = dynamic_cast<TRegistrationChatBot*>(robotPtr.Get());
            UNIT_ASSERT(!!robotImpl);
            UNIT_ASSERT(robotImpl->AskToResubmit(userId, "", 1, session, USER_ROOT_DEFAULT, false));
            UNIT_ASSERT(session.Commit());

            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 22);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "credit_card");
        }

        // Bind credit card.
        {
            eGenerator.GetBillingMock().SetReply("{\"status\": \"success\", \"bound_payment_methods\" : ["
                                                "{"
                                                "\"region_id\": 225, \"payment_method\" : \"card\", \"expiration_month\" : \"07\", \"binding_ts\" : \"1536324485.656\","
                                                "\"id\" : \"card-x12345\", \"expired\" : false, \"card_bank\" : \"TINKOFF BANK\", \"system\" : \"VISA\","
                                                "\"account\" : \"510000****0257\", \"expiration_year\" : \"2021\""
                                                "}"
                                                "]}");

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "card-x12345", {}));

            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 24);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
        }
    }

    Y_UNIT_TEST(ManualApproval) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        {
            auto status = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "onboarding");
        }

        configGenerator.ForceRegister(userId);

        {
            auto status = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();


            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["last_name"] = "Kulik";
            updateJson["driving_license"]["last_name"] = "NotKulik";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto status = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
        TVector<TDBTag> dbTags;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"user_registered_manually"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
    }

    Y_UNIT_TEST(RegistrationManagerInteraction) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
        TString newAssignmentId;
        // And now check that Yang document verification assignment was created for this user
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();


            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);
            eGenerator.CompleteYangAssignment(newAssignmentId, "ooiii", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto session = configGenerator.GetCurrentSession(userId);
            UNIT_ASSERT_VALUES_EQUAL(session["user"]["show_chat"]["id"].GetString(), "registration");
            UNIT_ASSERT(!session["user"]["show_chat"]["is_closable"].GetBoolean());
        }

        // Now we should do 3 steps of resubmit.
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 27);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_biographical");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 29);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_registration");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 31);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_selfie");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 33);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["messages"].GetArray().size());
        }

        // Check that another Yang assignment is created. Complete it then.
        TString secretId = "";
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::PassportBiographical, {}, TInstant::Zero());
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId2 = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId2);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId2, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            UNIT_ASSERT(!!driveApi.GetUsersData());
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId2).GetResult().begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
            secretId = assignment.GetId();
        }

        // You made it!
        {
            auto session = configGenerator.GetCurrentSession(userId);
            UNIT_ASSERT(!session["user"].Has("show_chat"));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 35);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "enter_map");

            auto hash = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetPassportNumberHash();
            UNIT_ASSERT_VALUES_EQUAL(hash, "f3c266214dfa332bd6386c215c34c3043bb03291");

            auto isFirstRiding = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.IsFirstRiding();
            UNIT_ASSERT(isFirstRiding);

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 35);
        }

        {
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(secretId).GetResult().begin()->second;
            assignment.SetFinalizedAt(Now());
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(docPhotoManager.GetDocumentVerificationAssignments().Upsert(assignment, session) && session.Commit());
            assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(secretId).GetResult().begin()->second;
            UNIT_ASSERT(assignment.GetFinalizedAt() != TInstant::Zero());
        }

        // Now post-check happens
        {
            auto payload = TDocumentsHelper::ConstructConsistentYangPayload();
            payload["verification_statuses"]["license_front_status"] = "NEED_INFO";
            payload["verification_statuses"]["license_back_status"] = "NEED_INFO";
            UNIT_ASSERT(configGenerator.PostYangAssignmentData(secretId, "1", payload));

            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(secretId).GetResult().begin()->second;
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetFinalizedAt(), TInstant::Zero());
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        // And you are banned until you resubmit some stuff
        {
            auto status = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "blocked");
        }
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 37);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));

            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 39);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_back");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 41);
        }

        {
            auto payload = TDocumentsHelper::ConstructConsistentYangPayload();
            UNIT_ASSERT(configGenerator.PostYangAssignmentData(secretId, "1", payload));

            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(secretId).GetResult().begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }
        {
            auto status = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
    }

    Y_UNIT_TEST(DuplicateProfileByPassport) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        {
            auto pretenderId = eGenerator.CreateUser("pretender-was-there", true, "active");
            auto userData = driveApi.GetUsersData()->FetchInfo(pretenderId).begin()->second;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            userData.SetPassportNumberHash("f3c266214dfa332bd6386c215c34c3043bb03291");
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(userData, "robot-frontend", session));
            UNIT_ASSERT(session.Commit());
            driveApi.GetUsersData()->FetchInfo(pretenderId);
            {
                UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), pretenderId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
                auto user = driveApi.GetUsersData()->FetchInfo(pretenderId).GetResult().begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
            }
            {
                UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), pretenderId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
                auto user = driveApi.GetUsersData()->FetchInfo(pretenderId).GetResult().begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "blocked");
            }
        }

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        // And now check that Yang document verification assignment was created for this user
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);
            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["first_name"] = "сергей";
            updateJson["driving_license"]["first_name"] = "сергей";
            updateJson["first_name"] = "";
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        // You made it, but no...
        {
            auto name = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetFirstName();
            UNIT_ASSERT_VALUES_EQUAL(name, "Сергей");

            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "blocked");

            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 29);
        }

        UNIT_ASSERT(configGenerator.ResetChat(userId, "registration"));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 31);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
        }
    }

    Y_UNIT_TEST(BlacklistExternal) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        UNIT_ASSERT(configGenerator.AddExternalBlacklistEntry("passport_number", "HB2352296", "block"));

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        // And now check that Yang document verification assignment was created for this user
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
        UNIT_ASSERT_VALUES_EQUAL(status, "blocked");
    }

    Y_UNIT_TEST(BlacklistByNames) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, {"unique_user_tag", "user_alert_test"}, dbTags, session));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(dbTags, "robot-frontend", server.Get(), session) && session.Commit());
        }

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        {
            auto userFetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
            auto user = userFetchResult.begin()->second;

            user.SetFirstName("sajfsadfksdhfk");
            user.SetLastName("hagfhgsdfdf");
            user.SetPhone("+79170001122");

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->UpdateUser(user, "robot-frontend", session));

            auto tag = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag("unique_user_tag");
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, userId, userId, server.Get(), session) && session.Commit());
        }

        auto twinId = eGenerator.CreateUser("twin-was-there", true, "onboarding");
        {
            auto userFetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(twinId);
            auto user = userFetchResult.begin()->second;

            user.SetFirstName("sajfsadfksdhfk");
            user.SetLastName("hagfhgsdfdf");
            user.SetPhone("+79170001122");

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->UpdateUser(user, "robot-frontend", session) && session.Commit());
        }

        IUserRegistrationManager::TBlacklistOptions options;
        options.Traits = NUserConnections::ESimilarityTraits::UserFields;
        UNIT_ASSERT(server->GetUserRegistrationManager()->ActualizeBlacklistBans("unittest", {"unique_user_tag"}, "user_alert_test", options));

        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, {"user_alert_test"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0].GetObjectId(), twinId);
        }
    }

    Y_UNIT_TEST(ResumeAfterRejected) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "rejected");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 1);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
        }
    }

    Y_UNIT_TEST(ResumeAfterActive) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "enter_map");
        }
    }

    Y_UNIT_TEST(ReaskAfterActive) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        UNIT_ASSERT(configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));

            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_back");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString()}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
        }
    }

    Y_UNIT_TEST(ChatReset) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        size_t messagesShift = 0;
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            messagesShift = messagesJson["messages"].GetArray().size();
        }

        UNIT_ASSERT(configGenerator.ResetChat(userId, "registration"));

        // Intro
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), messagesShift + 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["faq_url"].GetString(), "https://yandex.ru/support/drive/joining/joining-faq.html");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["support_url"].GetString(), "https://forms.yandex.ru/surveys/6149/");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Support email
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), messagesShift + 6);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Agreements
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), messagesShift + 10);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Driving license
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), messagesShift + 13);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString(), configGenerator.GetValidImageString()}));
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }
    }

    Y_UNIT_TEST(ResumeAfterBlockedGeneral) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "blocked");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 3);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[0]["text"].GetString().StartsWith("Ваш доступ в сервис отозван по решению службы безопасности"), TStringBuilder() << messagesJson["messages"].GetArray()[0]["text"].GetString());
        }
    }

    Y_UNIT_TEST(ResumeAfterBlockedOldLicense) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "blocked");
        UNIT_ASSERT(configGenerator.AddTag(new TUserProblemTag("blocked_old_license"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {configGenerator.GetValidImageString(), configGenerator.GetValidImageString()}));
            messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
        }
    }

    Y_UNIT_TEST(PassportSoonExpires) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        {
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["first_name"] = "сергей";
            updateJson["driving_license"]["first_name"] = "сергей";
            updateJson["first_name"] = "";
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 27);
        }

        {
            UNIT_ASSERT(configGenerator.AddTag(new TRegistrationUserTag("soon_license_expires"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(configGenerator.AddTag(new TRegistrationUserTag("soon_passport_expires"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }
        Sleep(TDuration::Seconds(5));

        {
            auto result = configGenerator.GetChatsList(userId, server.Get(), "")["chats"];
            UNIT_ASSERT_VALUES_EQUAL(result.GetArray().size(), 3);
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "expiring_license_notify");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "expiring_passport_notify");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport");
        }
    }

    Y_UNIT_TEST(ReaskRightAfterApproval) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        {
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["first_name"] = "сергей";
            updateJson["driving_license"]["first_name"] = "сергей";
            updateJson["first_name"] = "";
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 27);
        }

        {
            UNIT_ASSERT(configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::PassportSelfie}));
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 28);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_selfie");
        }
    }

    Y_UNIT_TEST(ResourceBasedRegistration) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["faq_url"].GetString(), "https://yandex.ru/support/drive/joining/joining-faq.html");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["support_url"].GetString(), "https://forms.yandex.ru/surveys/6149/");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Support email
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Agreements
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 10);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "button");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}));
            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Driving license
        TString lfId;
        TString lbId;
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 13);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");

            lfId = configGenerator.SubmitChatResource(userId, "registration", "some sample lf");
            lbId = configGenerator.SubmitChatResource(userId, "registration", "some sample lb");
            UNIT_ASSERT(lfId && lbId);
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {"!r " + lfId, "!r " + lbId}));

            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetDocumentPhoto(lfId, userId), "some sample lf");
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetDocumentPhoto(lbId, userId), "some sample lb");

            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Passport
        TString pbId;
        TString prId;
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 15);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport");

            pbId = configGenerator.SubmitChatResource(userId, "registration", "some sample pb");
            prId = configGenerator.SubmitChatResource(userId, "registration", "some sample pr");
            UNIT_ASSERT(pbId && prId);
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {"!r " + pbId, "!r " + prId}));

            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetDocumentPhoto(pbId, userId), "some sample pb");
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetDocumentPhoto(prId, userId), "some sample pr");

            UNIT_ASSERT(!configGenerator.ReaskUserDocuments(userId, {NUserDocument::EType::LicenseFront, NUserDocument::EType::LicenseBack}));
        }

        // Passport selfie
        TString psId;
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 20);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_selfie");

            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[15]["type"].GetString(), "user_document_photos");
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetDocumentPhoto(messagesJson["messages"].GetArray()[15]["text"].GetString(), userId), "some sample pb");

            psId = configGenerator.SubmitChatResource(userId, "registration", "some sample ps");
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {"!r " + psId}));

            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetDocumentPhoto(psId, userId), "some sample ps");
        }

        // Card
        {
            eGenerator.GetBillingMock().SetReply("{\"status\": \"success\", \"bound_payment_methods\" : ["
                                                "{"
                                                "\"region_id\": 225, \"payment_method\" : \"card\", \"expiration_month\" : \"07\", \"binding_ts\" : \"1536324485.656\","
                                                "\"id\" : \"card-x12345\", \"expired\" : false, \"card_bank\" : \"TINKOFF BANK\", \"system\" : \"VISA\","
                                                "\"account\" : \"510000****0257\", \"expiration_year\" : \"2021\""
                                                "}"
                                                "]}");

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "card-x12345", {}));
        }

        {
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["first_name"] = "сергей";
            updateJson["driving_license"]["first_name"] = "сергей";
            updateJson["first_name"] = "";
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;

            UNIT_ASSERT_VALUES_EQUAL(assignment.GetLicenseBackId(), lbId);
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetLicenseFrontId(), lfId);
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetPassportBiographicalId(), pbId);
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetPassportRegistrationId(), prId);
            UNIT_ASSERT_VALUES_EQUAL(assignment.GetPassportSelfieId(), psId);

            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "enter_map");
        }
    }

    Y_UNIT_TEST(RegistrationGeo) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}, 55.78111, 49.17812));
            auto user = *(driveApi.GetUsersData()->FetchInfo(userId).GetResultPtr(userId));
            UNIT_ASSERT_VALUES_EQUAL(user.GetRegistrationGeo(), "kazan");
        }

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}, 60.12345, 30.31111));
            auto user = *(driveApi.GetUsersData()->FetchInfo(userId).GetResultPtr(userId));
            UNIT_ASSERT_VALUES_EQUAL(user.GetRegistrationGeo(), "spb");
        }

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}, 55.75222, 37.12345));
            auto user = *(driveApi.GetUsersData()->FetchInfo(userId).GetResultPtr(userId));
            UNIT_ASSERT_VALUES_EQUAL(user.GetRegistrationGeo(), "moscow");
        }
    }

    Y_UNIT_TEST(RegistrationGeoSochi) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "registration", "", {}, 43.643453, 39.673628));
            auto user = *(driveApi.GetUsersData()->FetchInfo(userId).GetResultPtr(userId));
            UNIT_ASSERT_VALUES_EQUAL(user.GetRegistrationGeo(), "sochi");
        }
    }

    Y_UNIT_TEST(RemoveChat) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        auto pretenderId = eGenerator.CreateUser("pretender-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), pretenderId, eGenerator, true);
        UNIT_ASSERT(configGenerator.RemoveChat(userId, "registration", true));
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            session->Exec("DELETE FROM user_document_photo where user_id='" + userId + "'");
            UNIT_ASSERT(session.Commit());
        }
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
    }

    Y_UNIT_TEST(ArchiveChat) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        {
            auto result = configGenerator.GetChatsList(userId, server.Get(), "")["chats"];
            UNIT_ASSERT_VALUES_EQUAL(result.GetArray().size(), 1);
        }

        UNIT_ASSERT(configGenerator.RemoveChat(userId, "registration", false));

        Sleep(TDuration::Seconds(2));
        {
            auto result = configGenerator.GetChatsList(userId, server.Get(), "")["chats"];
            UNIT_ASSERT_VALUES_EQUAL(result.GetArray().size(), 0);
        }
    }

    Y_UNIT_TEST(FirstRidingByPhone) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", false, "onboarding");
        auto pretenderId = eGenerator.CreateUser("pretender-was-there", false, "onboarding");

        {
            const auto& settings = NDrive::GetServer().GetSettings();
            UNIT_ASSERT(settings.SetValue("handlers.api/yandex/sessions/current.new_device.policy", "verification", USER_ROOT_DEFAULT));
        }

        {
            auto userFR = server->GetDriveAPI()->GetUsersData()->FetchInfo(pretenderId);
            UNIT_ASSERT_VALUES_EQUAL(userFR.size(), 1);
            auto user = userFR.begin()->second;
            user.SetPhone("+79167790100");
            user.SetFirstRiding(false);

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->UpdateUser(user, "robot-frontend", session) && session.Commit());
        }

        {
            auto currentSession = configGenerator.GetCurrentSession(userId, nullptr, &userId);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["device_status"].GetString(), "Verified");
        }

        {
            auto userFR = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
            UNIT_ASSERT_VALUES_EQUAL(userFR.size(), 1);
            auto user = userFR.begin()->second;
            UNIT_ASSERT(user.IsFirstRiding());
        }

        {
            UNIT_ASSERT(configGenerator.PhoneBindSubmit(userId, "+79167790100"));

            bool confirmed = false;
            for (auto&& [token, code] : TFakeUserDevicesManager::GetCodes()) {
                if (configGenerator.PhoneBindCommit(userId, code)) {
                    confirmed = true;
                    break;
                }
            }

            UNIT_ASSERT(confirmed);
        }

        {
            auto currentSession = configGenerator.GetCurrentSession(userId, nullptr, &userId);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["device_status"].GetString(), "Verified");
        }

        {
            TInstantGuard g(Now() + TDuration::Hours(1));
            UNIT_ASSERT(configGenerator.PhoneBindSubmit(userId, "+79167790101"));

            TVector<TObjectEvent<TUserDevice>> events;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetUserDevicesManager()->GetFullHistoryByUser(userId, events, session));

            auto currentSession = configGenerator.GetCurrentSession(userId, nullptr, &userId);
            UNIT_ASSERT_VALUES_EQUAL(currentSession["device_status"].GetString(), "Verified");
        }

        {
            auto userFR = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
            UNIT_ASSERT_VALUES_EQUAL(userFR.size(), 1);
            auto user = userFR.begin()->second;
            UNIT_ASSERT(!user.IsFirstRiding());
        }
    }

    Y_UNIT_TEST(MultipleAccountTransition) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", false, "onboarding");
        auto pretenderId = eGenerator.CreateUser("pretender-was-there", false, "active");

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();

        {
            auto userData = driveApi.GetUsersData()->FetchInfo(pretenderId).begin()->second;
            auto session = driveApi.BuildTx<NSQL::Writable>();
            userData.SetFirstRiding(false);
            userData.SetPassportNumberHash("f3c266214dfa332bd6386c215c34c3043bb03291");
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(userData, "robot-frontend", session));
            UNIT_ASSERT(session.Commit());
            driveApi.GetUsersData()->FetchInfo(pretenderId);
            {
                auto dTag = new TTemporaryActionTag("additional_offer_tag");
                dTag->SetAttempts(2);
                UNIT_ASSERT(configGenerator.AddTag(dTag, pretenderId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
                auto user = driveApi.GetUsersData()->FetchInfo(pretenderId).GetResult().begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
            }
            {
                TDriveRoleHeader roleHeader;
                roleHeader.SetDescription("GR_default_user_test").SetName("GR_default_user_test").SetGroup(1).SetIsPublic(true).SetOptional(true);
                UNIT_ASSERT(configGenerator.RegisterRole(roleHeader, USER_ROOT_DEFAULT));
            }
            {
                TUserRole uRole;
                uRole.SetUserId(pretenderId).SetRoleId("GR_default_user_test").SetActive(true);
                UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));

                auto report = configGenerator.GetUserRoles(pretenderId, USER_ROOT_DEFAULT);
                UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "GR_default_user_test", true));

                report = configGenerator.GetUserRoles(userId, USER_ROOT_DEFAULT);
                UNIT_ASSERT(!TValidationHelpers::CheckIsActiveRole(report, "GR_default_user_test", true));
            }
            {
                auto session = driveApi.GetBillingManager().BuildSession();
                auto baDescription = driveApi.GetBillingManager().GetAccountsManager().GetDescriptionByName("bonus");
                UNIT_ASSERT(baDescription.Defined());
                auto bonusAccount = driveApi.GetBillingManager().GetAccountsManager().GetOrCreateAccount(pretenderId, baDescription.GetRef(), USER_ROOT_DEFAULT, session);
                UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);
                UNIT_ASSERT_VALUES_EQUAL(bonusAccount->Add(100, session), EDriveOpResult::Ok);
                UNIT_ASSERT(session.Commit());
            }
        }

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        // And now check that Yang document verification assignment was created for this user
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["first_name"] = "сергей";
            updateJson["driving_license"]["first_name"] = "сергей";
            updateJson["first_name"] = "";
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto name = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetFirstName();
            UNIT_ASSERT_VALUES_EQUAL(name, "Сергей");

            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");

            auto isFirstRiding = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.IsFirstRiding();
            UNIT_ASSERT(!isFirstRiding);

            auto messagesJson = configGenerator.GetChatMessages(userId, "registration");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 27);

            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"additional_offer_tag"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ pretenderId }, {"additional_offer_tag"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            }
            {
                auto report = configGenerator.GetUserRoles(userId, USER_ROOT_DEFAULT);
                UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "GR_default_user_test", true));
            }
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"same_person"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

                auto tag = dbTags.front();
                auto connTag = tag.GetTagAs<TConnectionUserTag>();
                UNIT_ASSERT(connTag);
                UNIT_ASSERT_VALUES_EQUAL(connTag->GetConnectedUserId(), pretenderId);
            }
            {
                auto session = driveApi.GetBillingManager().BuildSession();
                auto baDescription = driveApi.GetBillingManager().GetAccountsManager().GetDescriptionByName("bonus");
                UNIT_ASSERT(baDescription.Defined());
                auto bonusAccount = driveApi.GetBillingManager().GetAccountsManager().GetOrCreateAccount(pretenderId, baDescription.GetRef(), USER_ROOT_DEFAULT, session);
                UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);
            }
            {
                auto session = driveApi.GetBillingManager().BuildSession();
                auto baDescription = driveApi.GetBillingManager().GetAccountsManager().GetDescriptionByName("bonus");
                UNIT_ASSERT(baDescription.Defined());
                auto bonusAccount = driveApi.GetBillingManager().GetAccountsManager().GetOrCreateAccount(userId, baDescription.GetRef(), USER_ROOT_DEFAULT, session);
                UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 100);
            }
        }
    }

    //
    Y_UNIT_TEST(RegistrationVideoScreencap) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());

        // And now check that Yang document verification assignment was created for this user
        TString newAssignmentId = "";
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();


            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "uuuuu", TYangDocumentVerificationAssignment::EFraudStatus::DefinitelyFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto passport = TDocumentsHelper::CreateRandomPassportData();
            auto drivingLicense = TDocumentsHelper::CreateRandomDrivingLicenseData();

            NJson::TJsonValue payload;
            payload["passport_biographical"] = passport.SerializeBioToYang(false);
            payload["passport_registration"] = passport.SerializeRegToYang(false);
            payload["license_front"] = drivingLicense.SerializeFrontToYang(false);
            payload["license_back"] = drivingLicense.SerializeBackToYang(false);
            payload["license_back"]["data"]["has_at_mark"] = true;
            payload["verification_statuses"]["is_fraud"] = "MAYBE_FRAUD";

            TString testComment = "'\"тест ТЕСТ test\"'\\";
            payload["verification_statuses"]["comment"] = testComment;

            payload["verification_statuses"]["passport_biographical_status"] = "UNRECOGNIZABLE";
            payload["verification_statuses"]["passport_registration_status"] = "UNRECOGNIZABLE";
            payload["verification_statuses"]["license_front_status"] = "UNRECOGNIZABLE";
            payload["verification_statuses"]["license_back_status"] = "UNRECOGNIZABLE";
            payload["verification_statuses"]["passport_selfie_status"] = "VIDEO_SCREENCAP";

            UNIT_ASSERT(configGenerator.PostYangAssignmentData(newAssignmentId, "0", payload));
        }

        {
            auto status = driveApi.GetUsersData()->FetchInfo(userId).GetResult().begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "blocked");
        }

        {
            auto session = configGenerator.GetCurrentSession(userId);
            UNIT_ASSERT_VALUES_EQUAL(session["user"]["show_chat"]["id"].GetString(), "registration");
            UNIT_ASSERT(!session["user"]["show_chat"]["is_closable"].GetBoolean());
        }
    }

    void AddLicenseExpirationIgnoreJson(NJson::TJsonValue& ranges, const TInstant ignoreFrom, const TInstant ignoreUntil, const TInstant ignoreEnd) {
        if (!ranges["ranges"].IsArray()) {
            ranges["ranges"] = NJson::JSON_ARRAY;
        }
        NJson::TJsonValue range = NJson::JSON_MAP;
        if (ignoreFrom) {
            range["expiration_from"] = ignoreFrom.ToString();
        }
        if (ignoreUntil) {
            range["expiration_until"] = ignoreUntil.ToString();
        }
        if (ignoreEnd) {
            range["ignore_end"] = ignoreEnd.ToString();
        }
        ranges["ranges"].AppendValue(range);
    }

    bool CreateAndHandleAssignment(const TString& userId, const TDriveAPI& driveApi, TEnvironmentGenerator& eGenerator, const NDrive::TServerConfigGenerator& configGenerator, const NDrive::TServerGuard& server, NJson::TJsonValue& ranges, const TInstant licenseEnd) {
        const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
        eGenerator.CreateNewYangAssignments(server.Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        TVector<TString> newAssignmentIds = configGenerator.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
        if (newAssignmentIds.size() != 1) {
            return false;
        }
        TString newAssignmentId = newAssignmentIds.front();
        if (!newAssignmentId) {
            return false;
        }
        if (!server->GetSettings().SetValue("documents_expiration.driving_license.ignore_ranges", ranges.GetStringRobust(), USER_ROOT_DEFAULT)) {
            return false;
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
        updateJson["driving_license"]["categories_b_valid_to_date"] = licenseEnd.ToString();
        if (!configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT)) {
            ERROR_LOG << "couldn't update user info";
            return false;
        }
        driveApi.GetUsersData()->FetchInfo(userId);

        eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
        auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
        if (!server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now())) {
            return false;
        }
        return true;
    }

    Y_UNIT_TEST(LicenseExpirationIgnored) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());

        // in ignore range, no ignore_expiration_from var
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there0", true, "onboarding");
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, TInstant::Zero(), Now() + TDuration::Days(10), Now() + TDuration::Days(20));
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(2)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "screening");
        }
        // no vars
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there1", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, TInstant::Zero(), TInstant::Zero(), TInstant::Zero());
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(2)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "screening");
        }
        // in ignore range, before end date
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there2", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() + TDuration::Days(20));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(1)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
        // in ignore range, after end date
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there3", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() - TDuration::Days(2));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(1)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "screening");
        }
        // not expired
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there4", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range,  Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() - TDuration::Days(2));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() + TDuration::Days(30)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
        // in ignore range, no ignore_end var
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there5", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range,  Now() - TDuration::Days(10), Now() + TDuration::Days(10), TInstant::Zero());
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(2)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "screening");
        }
        // none of
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there2", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() - TDuration::Days(9), Now() + TDuration::Days(20));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() - TDuration::Days(5), Now() - TDuration::Days(4));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() - TDuration::Days(5));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(1)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "screening");
        }
        // one of, first
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there2", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() + TDuration::Days(20));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() - TDuration::Days(5), Now() + TDuration::Days(20));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() - TDuration::Days(5));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(1)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
        // one of, middle
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there2", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() - TDuration::Days(9), Now() + TDuration::Days(20));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(5), Now() + TDuration::Days(20));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() - TDuration::Days(5));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(1)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
        // one of, last
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there2", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() - TDuration::Days(9), Now() + TDuration::Days(20));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(5), Now() - TDuration::Days(4));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() + TDuration::Days(5));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(1)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
        // all of
        {
            auto userId = eGenerator.CreateUser("skulik-was-not-there2", true, "onboarding");
            configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
            NJson::TJsonValue range = NJson::JSON_MAP;
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(9), Now() + TDuration::Days(20));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(5), Now() + TDuration::Days(4));
            AddLicenseExpirationIgnoreJson(range, Now() - TDuration::Days(10), Now() + TDuration::Days(10), Now() + TDuration::Days(5));
            UNIT_ASSERT(CreateAndHandleAssignment(userId, driveApi, eGenerator, configGenerator, server, range, Now() - TDuration::Days(1)));
            auto status = driveApi.GetUsersData()->FetchInfo(userId).begin()->second.GetStatus();
            UNIT_ASSERT_VALUES_EQUAL(status, "active");
        }
    }
}
