#include <drive/library/cpp/tracks/track.h>

#include <library/cpp/testing/unittest/registar.h>


Y_UNIT_TEST_SUITE(InsuranceSuite) {
    Y_UNIT_TEST(SimplePartition) {
        TGeoCoord coordinate = { 37.330680, 55.889670 };
        TInstant timestamp = TInstant::Seconds(1606218863);
        NDrive::TTrackPartitionOptions options;
        {
            NDrive::TTrack track;
            for (ui32 i = 0; i < 10; ++i) {
                track.Coordinates.emplace_back(coordinate, timestamp + i * TDuration::Seconds(10), 0);
            }
            auto interval = TInterval<TInstant>(timestamp, timestamp + TDuration::Seconds(105));
            auto partition = NDrive::BuildSimplePartition(interval, track, options);
            UNIT_ASSERT(partition.Intervals.empty());

            track.Coordinates.at(5).Speed = 1;
            track.Coordinates.at(7).Speed = 1;
            partition = NDrive::BuildSimplePartition(interval, track, options);
            UNIT_ASSERT_VALUES_EQUAL(partition.Intervals.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(NDrive::GetDuration(partition).Seconds(), 20);
        }
    }

    Y_UNIT_TEST(TrackPartition) {
        TGeoCoord cTmp(1, 1);
        auto dt = TDuration::Seconds(3);
        TInstant trackStart = Now();

        NDrive::TTrackPartitionOptions options;
        options.Threshold = TDuration::Minutes(5);
        {
            // Full track with fast speed
            NDrive::TTrack track;
            for (ui32 i = 0; i < 1000; ++i) {
                NGraph::TRouter::TTimedGeoCoordinate point(cTmp, trackStart + TDuration::Seconds(i * dt.Seconds()), 20);
                track.Coordinates.push_back(point);
            }
            {
                TVector<TInterval<TInstant>> intervals;
                NDrive::BuildTrackPartition({ trackStart - TDuration::Seconds(1), trackStart + TDuration::Seconds(1000 * dt.Seconds()) }, track, options, intervals);
                UNIT_ASSERT_VALUES_EQUAL(intervals.size(), 1);
            }
            {
                TVector<TInterval<TInstant>> intervals;
                NDrive::BuildTrackPartition({ trackStart - TDuration::Seconds(20), trackStart + TDuration::Seconds(1000 * dt.Seconds()) }, track, options, intervals);
                UNIT_ASSERT_VALUES_EQUAL(intervals.size(), 2);
            }
        }

        {
            // Full track without speed
            NDrive::TTrack track;
            for (ui32 i = 0; i < 1000; ++i) {
                NGraph::TRouter::TTimedGeoCoordinate point(cTmp, trackStart + TDuration::Seconds(i * dt.Seconds()), 0);
                track.Coordinates.push_back(point);
            }
            {
                TVector<TInterval<TInstant>> intervals;
                NDrive::BuildTrackPartition({ trackStart - TDuration::Seconds(5), trackStart + TDuration::Seconds(1000 * dt.Seconds()) }, track, options, intervals);
                UNIT_ASSERT_VALUES_EQUAL(intervals.size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(intervals.front().GetMin(), trackStart);
                UNIT_ASSERT_VALUES_EQUAL(intervals.front().GetMax(), track.Coordinates.front().Timestamp);
            }
            {
                auto skipEmptyOptions = options;
                skipEmptyOptions.SkipEmpty = true;
                TVector<TInterval<TInstant>> intervals;
                NDrive::BuildTrackPartition({ trackStart - TDuration::Seconds(5), trackStart + TDuration::Seconds(1000 * dt.Seconds()) }, track, skipEmptyOptions, intervals);
                UNIT_ASSERT_VALUES_EQUAL(intervals.size(), 0);
            }
        }

        {
            NDrive::TTrack track;
            for (ui32 i = 0; i < 60; ++i) {
                if (i % 6 < 3) {
                    NGraph::TRouter::TTimedGeoCoordinate point(cTmp, trackStart + TDuration::Seconds(i * dt.Seconds()), 0);
                    track.Coordinates.push_back(point);
                } else {
                    NGraph::TRouter::TTimedGeoCoordinate point(cTmp, trackStart + TDuration::Seconds(i * dt.Seconds()), 20);
                    track.Coordinates.push_back(point);
                }
            }
            {
                TVector<TInterval<TInstant>> intervals;
                NDrive::BuildTrackPartition({ trackStart - TDuration::Seconds(5), trackStart + TDuration::Seconds(60 * dt.Seconds()) }, track, options, intervals);
                UNIT_ASSERT_VALUES_EQUAL(intervals.size(), 1);
            }
            options.Fault = TDuration::Seconds(5);
            options.Threshold = TDuration::Seconds(6);
            {
                TVector<TInterval<TInstant>> intervals;
                NDrive::BuildTrackPartition({ trackStart - TDuration::Seconds(5), trackStart + TDuration::Seconds(60 * dt.Seconds()) }, track, options, intervals);
                UNIT_ASSERT_VALUES_EQUAL(intervals.size(), 10);
                TInstant lastResult = intervals.back().GetMax();
                TInstant lastReal = trackStart + TDuration::Seconds(59 * dt.Seconds());
                UNIT_ASSERT_VALUES_EQUAL(lastResult.Seconds(), lastReal.Seconds());
            }
        }
    }

}
