#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/actions/email_after_acceptance.h>
#include <drive/backend/actions/email_after_finish_ride.h>
#include <drive/backend/actions/filter.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/offer.h>
#include <drive/backend/data/rental/rental_offer_holder_tag.h>
#include <drive/backend/data/rental/rental_service_mode_tag.h>
#include <drive/backend/data/rental/timetable_builder.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/rt_background/rental/process.h>

#include <library/cpp/iterator/enumerate.h>
#include <library/cpp/testing/unittest/registar.h>


inline const TString SimpleRentalOfferName = "simple_rental_offer";
inline const TGeoCoord RentalOfferDeliveryLocation = {37.640373, 55.736713};

inline TRentalOfferLocations GetRentalReturnLocations() {
    TRentalOfferLocations rentalOfferReturnLocations;
    {
        rentalOfferReturnLocations.push_back({});
        rentalOfferReturnLocations.back().MutableCoords().push_back({38.862048, 53.880882});
        rentalOfferReturnLocations.back().SetLocationName("Prague");
    }
    return rentalOfferReturnLocations;
}

const TString confirmedSmsTagName = "rental_confirmed_sms";
const TString cancelledSmsTagName = "rental_cancelled_sms";
const TString confirmedEmailTagName = "rental_confirmed_email";
const TString cancelledEmailTagName = "rental_cancelled_email";

const TString afterAcceptanceEmailTagName = "rental_after_acceptance_email";
const TString afterFinishRideEmailTagName = "rental_after_finish_ride_email";

TRentalOffer::TOption CreateOption(bool value, ui64 cost, const TString& id, int order) {
    TRentalOffer::TOption option;
    option.SetValue(value);
    option.SetCost(cost);
    option.MutableConstants().ConstructInPlace();
    option.MutableConstants()->SetId(id);
    option.MutableConstants()->SetOrder(order);
    option.MutableConstants()->SetCalculatePolicy(NDedicatedFleet::ECostCalculatePolicy::None);
    option.MutableConstants()->SetDefaultValue(true);

    option.MutableStringConstants().ConstructInPlace();
    option.MutableStringConstants()->SetTitle(TStringBuilder() << "rental.option." << id << ".title");
    option.MutableStringConstants()->SetSubtitle(TStringBuilder() << "rental.option." << id << ".subtitle");
    return option;
}

void RegisterRentalOfferTags(const NDrive::IServer& server) {
    {
        auto tagDescription = MakeAtomicShared<TRentalOfferHolderTag::TDescription>();
        tagDescription->SetName(TRentalOfferHolderTag::Type());
        tagDescription->SetType(TRentalOfferHolderTag::Type());
        RegisterTag(server, tagDescription);
    }

    {
        auto tagDescription = MakeAtomicShared<TRentalServiceModeTag::TDescription>();
        tagDescription->SetName(TRentalServiceModeTag::Type());
        tagDescription->SetType(TRentalServiceModeTag::Type());
        RegisterTag(server, tagDescription);
    }


    const TString messageBase = " delivery_date, delivery_time, return_date, return_time, company_name,"
                                "  delivery_location, car_model, company_phone, company_mail, _CUSTOM1_,"
                                " total_payment, deposit, first_name, last_name, client_phone, insurance_type,"
                                " currency";
    {
        auto tagSms = MakeAtomicShared<TUserSmsTag::TDescription>();
        tagSms->SetName(confirmedSmsTagName);
        tagSms->SetType(TUserSmsTag::TypeName);
        tagSms->SetTitle("Confirmed");
        tagSms->SetMessageText("Confirmed rental: " + messageBase);
        RegisterTag(server, tagSms);
    }
    {
        auto tagSms = MakeAtomicShared<TUserSmsTag::TDescription>();
        tagSms->SetName(cancelledSmsTagName);
        tagSms->SetType(TUserSmsTag::TypeName);
        tagSms->SetTitle("Cancelled");
        tagSms->SetMessageText("Cancelled rental: " + messageBase);
        RegisterTag(server, tagSms);
    }

    {
        auto tagEmail= MakeAtomicShared<TUserMailTag::TDescription>();
        tagEmail->SetName(confirmedEmailTagName);
        tagEmail->SetType(TUserMailTag::TypeName);
        tagEmail->SetTemplateId("template_id1");
        RegisterTag(server, tagEmail);
    }
    {
        auto tagEmail= MakeAtomicShared<TUserMailTag::TDescription>();
        tagEmail->SetName(cancelledEmailTagName);
        tagEmail->SetType(TUserMailTag::TypeName);
        tagEmail->SetTemplateId("template_id2");
        RegisterTag(server, tagEmail);
    }

    {
        auto tagEmail= MakeAtomicShared<TUserMailTag::TDescription>();
        tagEmail->SetName(afterFinishRideEmailTagName);
        tagEmail->SetType(TUserMailTag::TypeName);
        tagEmail->SetTemplateId("template_id3");
        RegisterTag(server, tagEmail);
    }

    {
        auto tagEmail= MakeAtomicShared<TUserMailTag::TDescription>();
        tagEmail->SetName(afterAcceptanceEmailTagName);
        tagEmail->SetType(TUserMailTag::TypeName);
        tagEmail->SetTemplateId("template_id4");
        RegisterTag(server, tagEmail);
    }

    auto tagOrganization = MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag::TDescription>();
    tagOrganization->SetName(NDrivematics::TUserOrganizationAffiliationTag::TypeName);
    tagOrganization->SetType(NDrivematics::TUserOrganizationAffiliationTag::TypeName);
    tagOrganization->SetCompanyName("test 1");
    tagOrganization->SetOwningCarTagName("test 2");
    tagOrganization->SetBookingConfirmedSmsTagName(confirmedSmsTagName);
    tagOrganization->SetBookingCancelledSmsTagName(cancelledSmsTagName);
    tagOrganization->SetBookingConfirmedEmailTagName(confirmedEmailTagName);
    tagOrganization->SetBookingCancelledEmailTagName(cancelledEmailTagName);
    tagOrganization->SetAfterAcceptanceEmailTagName(afterAcceptanceEmailTagName);
    tagOrganization->SetAfterFinishRideEmailTagName(afterFinishRideEmailTagName);
    tagOrganization->SetPhone("+79053333333");
    tagOrganization->SetEmail("rental@yandex.ru");
    tagOrganization->SetShortDescription("some description");
    tagOrganization->SetIconMobileLogin("login_icon.png");
    TMap<TString, TString> companyInformation;
    companyInformation["_CUSTOM1_"] = "best regards";
    tagOrganization->SetCompanyInformation(companyInformation);

    RegisterTag(server, tagOrganization);
}

void RegisterRentalOffer(const NDrive::IServer& server) {
    auto action = MakeAtomicShared<TRentalOfferBuilder>();
    action->SetName(SimpleRentalOfferName);
    action->SetDescription(SimpleRentalOfferName);
    action->SetAllowChildSeat(true);
    action->SetAllowComment(true);
    action->SetAllowSnowChains(true);
    action->SetAllowRoofRack(true);
    action->SetAllowGPS(true);
    action->SetAllowEntryToEcoZonesInGermany(true);
    auto insurances = action->GetAllowedInsurances();
    for (auto& insurance: insurances) {
        insurance.SetTitle("rental_custom." + insurance.GetId() + ".title");
    }
    action->SetAllowedInsurances(insurances);

    {
        TRentalOffer::TOptions options;
        auto option = CreateOption(true, 100, "new_option1", 2);
        options.push_back(std::move(option));
        option = CreateOption(true, 200, "new_option2", 55);
        options.push_back(std::move(option));
        action->SetOptions(options);
    }

    action->SetGrouppingTags({ SimpleRentalOfferName });
    action->SetSwitchOfferTagsList({ SimpleRentalOfferName });
    TRentalOffer::TRenterOfferCommonInfo renterCommonInfo;
    renterCommonInfo["{{renter.custom1}}"] = "Glasseles Etermelles s.r.o";
    renterCommonInfo["{{renter.custom2}}"] = "Politických vězňů, 912/10, Praha 1";
    renterCommonInfo["{{renter.custom3}}"] = "+375 33 618 83 94";
    renterCommonInfo["{{renter.petrol.fine_for_a_partial_tank}}"] = "123";

    action->SetRenterOfferCommonInfo(renterCommonInfo);
    action->SetOfferTimezone("Europe/Moscow");
    action->SetAgreement("document_manager." + SimpleRentalOfferName);

    TVector<TRentalOfferLocation> locations;
    TRentalOfferLocation location;
    location.SetId("id1");
    location.SetLocationName("Prague");
    location.MutableCoords().push_back(TGeoCoord(0., 0.));
    locations.push_back(location);
    action->SetAllowedDeliveryLocations(locations);
    action->SetAllowedReturnLocations(locations);

    TVector<TRentalOfferCurrency> currencies;
    TRentalOfferCurrency currency;
    currency.SetId("czk");
    currencies.push_back(currency);
    action->SetAllowedCurrencies(currencies);

    RegisterAction(server, action, "pack_access");

    auto rentalAction = MakeAtomicShared<TTagAction>();
    rentalAction->SetName(TRentalOfferHolderTag::Type());
    rentalAction->SetTagName(TRentalOfferHolderTag::Type());
    rentalAction->AddTagAction(NTagActions::ETagAction::Observe);
    rentalAction->AddTagAction(NTagActions::ETagAction::Update);
    rentalAction->AddTagAction(NTagActions::ETagAction::Remove);

    RegisterAction(server, rentalAction, "pack_access");

    auto rentalServiceAction = MakeAtomicShared<TTagAction>();
    rentalServiceAction->SetName(TRentalServiceModeTag::Type());
    rentalServiceAction->SetTagName(TRentalServiceModeTag::Type());
    rentalServiceAction->AddTagAction(NTagActions::ETagAction::Observe);
    rentalServiceAction->AddTagAction(NTagActions::ETagAction::Update);
    rentalServiceAction->AddTagAction(NTagActions::ETagAction::Remove);

    RegisterAction(server, rentalServiceAction, "pack_access");

    auto organizationAction = MakeAtomicShared<TTagAction>();
    organizationAction->SetName(NDrivematics::TUserOrganizationAffiliationTag::TypeName);
    organizationAction->SetTagName(NDrivematics::TUserOrganizationAffiliationTag::TypeName);
    organizationAction->AddTagAction(NTagActions::ETagAction::Observe);
    organizationAction->AddTagAction(NTagActions::ETagAction::Update);
    organizationAction->AddTagAction(NTagActions::ETagAction::Remove);

    RegisterAction(server, organizationAction, "pack_access");
}

void RegisterRentalEvolutions(const NDrive::IServer& server) {
    {
        auto from = TRentalOfferHolderTag::Type();
        auto to = TOfferBookingUserTag::Type();
        auto evolution = MakeAtomicShared<TTagEvolutionAction>(from + "->" + to);
        evolution->SetTagNameFrom(from).SetTagNameTo(to);
        evolution->SetOnlyPerformed(false);
        RegisterAction(server, evolution, "pack_access");
    }
    {
        auto from = TRentalServiceModeTag::Type();
        auto to = TRentalServiceModeTag::Type();

        auto evolution = MakeAtomicShared<TTagEvolutionAction>(from + "->" + to);
        evolution->SetTagNameFrom(from).SetTagNameTo(to);
        evolution->SetOnlyPerformed(false);
        RegisterAction(server, evolution, "pack_access");
    }
}

void RegisterRentalOfferAll(const NDrive::IServer& server) {
    RegisterRentalOffer(server);
    RegisterRentalOfferTags(server);
    RegisterRentalEvolutions(server);
}

void BookRental(TTestEnvironment& env, TInstant sinceRental, TInstant untilRental, TString status) {
    auto carId = TString{OBJECT_ID_DEFAULT};
    auto userId = TString{USER_ID_DEFAULT};

    auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
        ("variables", NJson::TMapBuilder
            ("since", NJson::ToJson(sinceRental))
            ("until", NJson::ToJson(untilRental))
            ("comment", "test_comment")
            ("total_payment", 1000)
            ("deposit", 200)
            ("status", status)
            ("currency", "czk")
        )
    );
    auto createdOffer = createdOffers["offers"][0];
    auto offerId = createdOffer["offer_id"].GetString();
    UNIT_ASSERT(offerId);

    auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
        ("offer_id", offerId)
        ("client_id", userId)
    );
}

void CreateModel(TTestEnvironment& env) {
    auto userId = TString{USER_ROOT_DEFAULT};

    NJson::TJsonValue payload = NJson::TMapBuilder("name", "some_name")("manufacturer", "some_manufacturer")("code", "code123");
    NJson::TJsonArray jSpecifications;
    NJson::TJsonValue jSpec;

    jSpec["name"] = "Transmission";
    jSpec["value"] = "Robotic";
    jSpec["id"] = "123";
    jSpec["position"] = 0;
    jSpec["icon"] = "icon";
    jSpecifications.AppendValue(jSpec);

    jSpec["name"] = "FuelType";
    jSpec["value"] = "some_fuel_type";
    jSpec["id"] = "124";
    jSpec["position"] = 1;
    jSpec["icon"] = "icon";
    jSpecifications.AppendValue(jSpec);

    jSpec["name"] = "Category";
    jSpec["value"] = "some_category";
    jSpec["id"] = "125";
    jSpec["position"] = 2;
    jSpec["icon"] = "icon";
    jSpecifications.AppendValue(jSpec);

    jSpec["name"] = "ExteriorType";
    jSpec["value"] = "some_exterior_type";
    jSpec["id"] = "126";
    jSpec["position"] = 3;
    jSpec["icon"] = "icon";
    jSpecifications.AppendValue(jSpec);

    jSpec["name"] = "AirConditioning";
    jSpec["value"] = "some_conditioning";
    jSpec["id"] = "127";
    jSpec["position"] = 4;
    jSpec["icon"] = "icon";
    jSpecifications.AppendValue(jSpec);

    payload["model_specifications"] = jSpecifications;

    NJson::TJsonValue jBody;
    jBody["models"].AppendValue(payload);

    UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

    auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/models/create", {}, jBody);
}

Y_UNIT_TEST_SUITE(RentalOfferSuite) {
    Y_UNIT_TEST(Simple) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOffer(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);
        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("offer_options",  NJson::TMapBuilder
                    ("child_seat", true)
                    ("roof_rack", true)
                    ("gps", true)
                    ("snow_chains", true)
                    ("entry_to_eco_zones_in_germany", true)
                )
                ("insurance_type", "id_gold")
                ("delivery_location", NJson::ToJson(RentalOfferDeliveryLocation))
                ("delivery_location_name", "Prague")
                ("return_locations", NJson::ToJson(GetRentalReturnLocations()))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
                ("limit_km_per_day", 100)
                ("overrun_cost_per_km", 10)
                ("currency", "czk")
            )
        );

        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto createdOffersTemplate = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TJsonMap())
        );
        offerId = createdOffersTemplate["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
    }

    Y_UNIT_TEST(Book) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }
        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TRentalOfferHolderTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag != nullptr);
            auto pOffer = std::dynamic_pointer_cast<TRentalOffer>(pTag->GetOffer());
            UNIT_ASSERT(pOffer != nullptr);
            UNIT_ASSERT("test_comment" == pOffer->GetCommentRef());
            UNIT_ASSERT(1000 == pOffer->GetTotalPaymentRef());
            UNIT_ASSERT(200 == pOffer->GetDepositRef());
        }
    }

    Y_UNIT_TEST(UpdateOffer) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "draft")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["comment"] = "edited_comment";
        {
            NJson::TJsonValue options;
            options["child_seat"] = true;
            options["roof_rack"] = true;
            options["gps"] = false;
            options["snow_chains"] = false;
            options["entry_to_eco_zones_in_germany"] = true;
            createdOffer["offer_options"] = options;
        }
        {
            NJson::TJsonArray options;
            {
                NJson::TJsonValue option;
                option["id"] = "new_option1";
                option["value"] = false;
                option["cost"] = 333;
                options.AppendValue(option);
            }
            createdOffer["options"] = options;
        }
        createdOffer["insurance_type"] = "id_basic";
        createdOffer["delivery_location_name"] = "Moscow";
        createdOffer["return_locations"] = NJson::ToJson(GetRentalReturnLocations());
        createdOffer["total_payment"] = 500;
        createdOffer["deposit"] = 300;
        createdOffer["status"] = "confirmed";
        createdOffer["limit_km_per_day"] = 500;
        createdOffer["overrun_cost_per_km"] = 300;
        createdOffer["currency"] = "usd";
        createdOffer["car_id"] = "newid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TRentalOfferHolderTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag != nullptr);
            auto pOffer = std::dynamic_pointer_cast<TRentalOffer>(pTag->GetOffer());
            UNIT_ASSERT(pOffer != nullptr);
            UNIT_ASSERT("edited_comment" == pOffer->GetCommentRef());
            for (auto& option: pOffer->GetOptions()) {
                if (option.GetConstantsRef().GetId() == "roof_rack") {
                    UNIT_ASSERT(option.GetValueRef());
                } else if (option.GetConstantsRef().GetId() == "child_seat") {
                    UNIT_ASSERT(option.GetValueRef());
                } else if (option.GetConstantsRef().GetId() == "gps") {
                    UNIT_ASSERT(!option.GetValueRef());
                } else if (option.GetConstantsRef().GetId() == "snow_chains") {
                    UNIT_ASSERT(!option.GetValueRef());
                } else if (option.GetConstantsRef().GetId() == "entry_to_eco_zones_in_germany") {
                    UNIT_ASSERT(option.GetValueRef());
                } else if (option.GetConstantsRef().GetId() == "new_option1") {
                    UNIT_ASSERT(!option.GetValueRef());
                    UNIT_ASSERT(option.GetCostRef() == 333);
                } else if (option.GetConstantsRef().GetId() == "new_option2") {
                    UNIT_ASSERT(!option.HasValue());
                    UNIT_ASSERT(!option.HasCost());
                }
            }
            UNIT_ASSERT("id_basic" == pOffer->GetInsuranceRef().GetId());
            UNIT_ASSERT("Moscow" == pOffer->GetDeliveryLocationNameRef());
            UNIT_ASSERT("Prague" == pOffer->GetReturnLocationsRef()[0].GetLocationName());
            UNIT_ASSERT(500 == pOffer->GetTotalPaymentRef());
            UNIT_ASSERT(300 == pOffer->GetDepositRef());
            UNIT_ASSERT("confirmed" == pOffer->GetStatusRef());
            UNIT_ASSERT(500 == pOffer->GetLimitKmPerDayRef());
            UNIT_ASSERT(300 == pOffer->GetOverrunCostPerKmRef());
            UNIT_ASSERT("usd" == pOffer->GetCurrencyRef());
            UNIT_ASSERT("newid" == pOffer->GetObjectId());
        }
    }

    Y_UNIT_TEST(GetCarsharingTimetable) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Days(7);
        BookRental(env, sinceRental, untilRental, "draft");

        sinceRental += TDuration::Days(7);
        untilRental += TDuration::Days(7);
        BookRental(env, sinceRental, untilRental, "confirmed");

        sinceRental += TDuration::Days(7);
        untilRental += TDuration::Days(7);
        BookRental(env, sinceRental, untilRental, "confirmed");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        UNIT_ASSERT(carsharingTimetable["offers_timetable"].Has(carId));
        UNIT_ASSERT(carsharingTimetable["offers_timetable"][carId].GetArray().size() == 3);

        NJson::TJsonArray jTagIdArr;
        jTagIdArr.AppendValue(carsharingTimetable["offers_timetable"][carId].GetArray()[0]["tag_id"].GetString());
        jTagIdArr.AppendValue(carsharingTimetable["offers_timetable"][carId].GetArray()[1]["tag_id"].GetString());

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/user_tags/remove", "",  NJson::TMapBuilder("tag_id", jTagIdArr));

        carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));
        UNIT_ASSERT(carsharingTimetable["offers_timetable"].Has(carId));
        UNIT_ASSERT(carsharingTimetable["offers_timetable"][carId].GetArray().size() == 1);

        {
            auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray()[0];
            auto offerId = jObj["offer_id"].GetString();
            auto tagId = jObj["tag_id"].GetString();
            auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
            createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

            createdOffer["tag_id"] = tagId;
            createdOffer["status"] = "paid";

            auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
            updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

            carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));
            TString carsharingTimetableStr = TStringBuilder() << carsharingTimetable;
            UNIT_ASSERT(carsharingTimetableStr.find("actual_since") == std::string::npos);
            TString chargableTagId;
            UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Reservation, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));
            carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));
            carsharingTimetableStr = TStringBuilder() << carsharingTimetable;
            UNIT_ASSERT(carsharingTimetableStr.find("actual_since") != std::string::npos);
            UNIT_ASSERT(carsharingTimetableStr.find("actual_until") == std::string::npos);
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, userId));
        }

        carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));
        TString carsharingTimetableStr = TStringBuilder() << carsharingTimetable;
        UNIT_ASSERT(carsharingTimetableStr.find("actual_since") != std::string::npos);
        UNIT_ASSERT(carsharingTimetableStr.find("actual_until") != std::string::npos);
    }

    Y_UNIT_TEST(CreateUser) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(res["client_id"].GetString(), TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }
        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TRentalOfferHolderTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag != nullptr);
            auto pOffer = std::dynamic_pointer_cast<TRentalOffer>(pTag->GetOffer());
            UNIT_ASSERT(pOffer != nullptr);
            UNIT_ASSERT("test_comment" == pOffer->GetCommentRef());
            UNIT_ASSERT(1000 == pOffer->GetTotalPaymentRef());
            UNIT_ASSERT(200 == pOffer->GetDepositRef());
        }
    }

    Y_UNIT_TEST(ConflictBooking) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru")
        );
        auto createUserId =  res["client_id"];

        sinceRental += TDuration::Days(2);
        createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        const auto clientId = res["client_id"];
        offerId = createdOffers["offers"][0]["offer_id"].GetString();

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "anotheruser")
            ("phone_number", "+79055553388"),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == createUserId);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "anotheruser")
            ("phone_number", "+7905999999")
            ("email", "test@yandex.ru"),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == createUserId);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "someuser")
            ("phone_number", "+7905000000"),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == "rental.book.car_is_already_used_by_another_booking");

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", clientId),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == "rental.book.user_already_has_booking");
    }

    Y_UNIT_TEST(ConflictUpdate) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "draft")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["comment"] = "edited_comment";
        NJson::TJsonValue options;
        options["child_seat"] = true;
        options["roof_rack"] = true;
        options["gps"] = false;
        options["snow_chains"] = false;
        options["entry_to_eco_zones_in_germany"] = true;
        createdOffer["offer_options"] = options;

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer, false);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer, false);
        UNIT_ASSERT(updateRes["error_details"]["special_info"]["session_info"]["RentalOffer::PatchFromJson"] == "rental.book.revision_is_invalid");

        sinceRental += TDuration::Days(7);
        untilRental += TDuration::Days(7);
        createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "draft")
            )
        );
        auto createdOfferNew = createdOffers["offers"][0];
        auto offerIdNew = createdOfferNew["offer_id"].GetString();

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerIdNew)
            ("client_id", userId)
        );

        TString newOfferHolderTagId;
        {
            NJson::TJsonValue report = env->ListTags(userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            for (auto&& i : report["records"].GetArraySafe()) {
                UNIT_ASSERT(i.IsMap());
                if (i["tag"].GetString() == TRentalOfferHolderTag::Type() && offerHolderTagId != i["tag_id"].GetString()) {
                    newOfferHolderTagId = i["tag_id"].GetString();
                    break;
                }
            }
            UNIT_ASSERT(newOfferHolderTagId);
        }

        createdOfferNew = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerIdNew + "&traits=ReportOffer", {});
        createdOfferNew = createdOfferNew["sessions"][0]["segment"]["offer"];

        createdOfferNew["tag_id"] = newOfferHolderTagId;
        createdOfferNew["since"] =  NJson::ToJson(sinceRental - TDuration::Days(3));
        createdOfferNew["until"] =  NJson::ToJson(untilRental - TDuration::Days(3));

        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOfferNew, false);
        UNIT_ASSERT(updateRes["error_details"]["special_info"]["session_info"]["RentalOffer::PatchOffer"] == "rental.book.user_already_has_booking");

        createdOffer["since"] =  NJson::ToJson(sinceRental - TDuration::Days(3));
        createdOffer["until"] =  NJson::ToJson(untilRental - TDuration::Days(3));
        createdOffer["revision"] = 1;
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer, false);
        UNIT_ASSERT(updateRes["error_details"]["special_info"]["session_info"]["RentalOffer::PatchOffer"] == "rental.book.user_already_has_booking");
    }

    Y_UNIT_TEST(SearchUser) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createOfferAndUser = [&](TInstant since, TInstant until, TString&& firstName, TString&& lastName, TString&& phone, TString email) {
            auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
                ("variables", NJson::TMapBuilder
                    ("since", NJson::ToJson(since))
                    ("until", NJson::ToJson(until))
                    ("comment", "test_comment")
                    ("total_payment", 1000)
                    ("deposit", 200)
                    ("status", "confirmed")
                )
            );
            auto createdOffer = createdOffers["offers"][0];
            auto offerId = createdOffer["offer_id"].GetString();
            UNIT_ASSERT(offerId);

            auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
                ("offer_id", offerId)
                ("first_name", firstName)
                ("last_name", lastName)
                ("phone_number", phone)
                ("email", email)
            );
        };

        createOfferAndUser(sinceRental, untilRental, "rental_firstname", "rental_lastname", "+79055553388", "rental_someemail@yandex.ru");
        createOfferAndUser(sinceRental + TDuration::Days(7), untilRental + TDuration::Days(7), "rental_aaaaa", "rental_bbbb", "+79055551234", "rental_someemail2@yandex.ru");

        auto searchResult = env->DoSearch(
            TVector<TString>({"rental_fi", "rental_la"}),
            TVector<TString>(),
            TVector<TString>({"users"})
        );
        UNIT_ASSERT(searchResult["objects"]["users"].GetArray().size() == 1);

        searchResult = env->DoSearch(
            TVector<TString>({"rental_firstname"}),
            TVector<TString>(),
            TVector<TString>({"users"})
        );
        UNIT_ASSERT(searchResult["objects"]["users"].GetArray().size() == 1);

        searchResult = env->DoSearch(
            TVector<TString>({"rental_lastname"}),
            TVector<TString>(),
            TVector<TString>({"users"})
        );
        UNIT_ASSERT(searchResult["objects"]["users"].GetArray().size() == 1);

        searchResult = env->DoSearch(
            TVector<TString>({"rental_fi", "rental_lab"}),
            TVector<TString>(),
            TVector<TString>({"users"})
        );
        UNIT_ASSERT(searchResult["objects"]["users"].GetArray().size() == 0);

        searchResult = env->DoSearch(
            TVector<TString>({"rental_someem"}),
            TVector<TString>(),
            TVector<TString>({"users"})
        );
        UNIT_ASSERT(searchResult["objects"]["users"].GetArray().size() == 2);

        searchResult = env->DoSearch(
            TVector<TString>({"7905555"}),
            TVector<TString>(),
            TVector<TString>({"users"})
        );
        UNIT_ASSERT(searchResult["objects"]["users"].GetArray().size() == 2);
    }

    Y_UNIT_TEST(StartRentalAfterUpdate) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + offerHolderTagId, {});

        {
            TString chargableTagId;
            UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Reservation, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));

            UNIT_ASSERT(chargableTagId);
            const auto& deviceTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
            auto tx = deviceTagManager.BuildTx<NSQL::Writable>();
            auto optionalTag = deviceTagManager.RestoreTag(chargableTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TChargableTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag->GetPerformer() == userId);
        }

        sinceRental += TDuration::Days(7);
        untilRental += TDuration::Days(7);

        createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );

        createdOffer = createdOffers["offers"][0];
        offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }
    }

    Y_UNIT_TEST(GetAgreement) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        env.GetServer()->GetSettings().SetValue("document_manager.simple_rental_offer",
                                                "Customer\n"
                                                "Name       {{rental.customer.first_name}}\n"
                                                "Last name  {{rental.customer.last_name}}\n"
                                                "Address    {{rental.customer.address}}\n"
                                                "Phone      {{rental.customer.phone}}\n"
                                                "Email      {{rental.customer.email}}\n"
                                                "Passport#  {{rental.customer.passport_number}}\n"
                                                "License#   {{rental.customer.license_number}}\n"
                                                "\n"
                                                "Car\n"
                                                "{{rental.car.model}}\n"
                                                "{{rental.car.number}}\n"
                                                "{{rental.car.vin}}\n"
                                                "\n"
                                                "Picking up\n"
                                                "{{rental.delivery.date}}\n"
                                                "{{rental.delivery.time}}\n"
                                                "{{rental.delivery.address}}\n"
                                                "\n"
                                                "Returning back\n"
                                                "{{rental.return.duration}}\n"
                                                "{{rental.return.date}}\n"
                                                "{{rental.return.time}}\n"
                                                "{{#loop rental.returns}}"
                                                "{{rental.return.address}}\n"
                                                "{{/loop rental.returns}}"
                                                "\n"
                                                "Petrol\n"
                                                "{{rental.petrol.tank_fullness}}\n"
                                                "{{renter.petrol.fine_for_a_partial_tank}}\n"
                                                "\n"
                                                "Mileage&limit\n"
                                                "{{rental.mileage_and_limit.mileage}}\n"
                                                "{{rental.mileage_and_limit.limit_km_per_day}}\n"
                                                "{{rental.mileage_and_limit.overrun_cost_per_km}}\n"
                                                "\n"
                                                "Payment\n"
                                                "{{rental.payment.total}}\n"
                                                "{{rental.payment.deposit}}\n"
                                                "\n"
                                                "Signature\n"
                                                "{{rental.signature.date}}\n"
                                                "Renter\n"
                                                "{{renter.custom1}}\n"
                                                "{{renter.custom2}}\n"
                                                "{{renter.custom3}}\n"
                                                "Options\n"
                                                "{{#loop rental.options}}"
                                                "<span>{{rental.option.name}} {{rental.option.cost}} {{rental.option.currency}}</span><span class=\"icon\" />"
                                                "{{/loop rental.options}}"
                                                "Damages\n"
                                                "{{#loop rental.car.damages}}"
                                                "<img src=\"{{rental.car.damage.url}}\" />"
                                                "<span>{{rental.car.damage.title}}</span>"
                                                "<span>{{rental.car.damage.level}}</span>"
                                                "{{/loop rental.car.damages}}"
                                                "Insurance\n"
                                                "{{rental.insurance_type}}"
                                                "{{rental.insurance_cost}}"
                                                "Currency\n"
                                                "{{rental.currency}}",
                                                USER_ID_DEFAULT);

        auto car = env.GetEnvironmentGenerator().CreateCar();
        auto emulator = env.GetContext().SetCar(car).GetEmulator();
        UNIT_ASSERT(emulator);
        auto carId = car.Id;

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));


        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("offer_options",  NJson::TMapBuilder
                    ("child_seat", true)
                    ("roof_rack", true)
                    ("gps", false)
                    ("snow_chains", false)
                )

                ("delivery_location", NJson::ToJson(RentalOfferDeliveryLocation))
                ("delivery_location_name", "Prague")
                ("return_locations", NJson::ToJson(GetRentalReturnLocations()))

                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("limit_km_per_day", 100)
                ("overrun_cost_per_km", 10)
                ("status", "confirmed")
                ("insurance", NJson::TMapBuilder
                    ("id", "id_gold")
                    ("cost", 3300)
                )
                ("currency", "czk")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );


        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + offerHolderTagId, {});

        TString chargableTagId;
        UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Reservation, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/agreement", "session_id=" + offerId, {});
        TString resStr = TStringBuilder() << res;

        UNIT_ASSERT(resStr.find("{{rental.customer.first_name}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.customer.last_name}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.customer.address}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.customer.phone}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.customer.email}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.customer.passport_number}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.customer.license_number}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.car.model}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.car.number}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.car.vin}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.delivery.date}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.delivery.time}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.return.duration}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.return.date}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.return.time}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.petrol.tank_fullness}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{renter.petrol.fine_for_a_partial_tank}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.mileage_and_limit.mileage}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.mileage_and_limit.limit_km_per_day}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.mileage_and_limit.overrun_cost_per_km}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.payment.total}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.payment.deposit}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.signature.date}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{renter.custom1}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{renter.custom2}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{renter.custom3}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.option.name}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.insurance_type}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.insurance_cost}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.currency}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.option.cost}}") == std::string::npos);
        UNIT_ASSERT(resStr.find("{{rental.option.currency}}") == std::string::npos);
    }

    Y_UNIT_TEST(WarningsBeforeFinishRental) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("warning_screens.events.finish_rental.rental_finish_fuel_checker.enabled", "true", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("warning_screens.events.finish_rental.rental_finish_mileage_checker.enabled", "true", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("warning_screens.events.finish_rental.rental_finish_location_checker.enabled", "true", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue(
            "warning_screens.checkers.rental_finish_fuel_checker.landings.fuel_limit_exceed",
            R"({ "points":
                [
                  "Petrol: _FuelVolumeStart_, _FuelVolumeFinish_"
                ]
            })",
            USER_ID_DEFAULT
        );

        env.GetServer()->GetSettings().SetValue(
            "warning_screens.checkers.rental_finish_mileage_checker.landings.mileage_limit_exceed",
            R"({ "points":
                [
                  "Mileage: _RentalMileage_, _RentalOverrun_"
                ]
            })",
            USER_ID_DEFAULT
        );

        env.GetServer()->GetSettings().SetValue(
            "warning_screens.checkers.rental_finish_location_checker.landings.return_location_zones_not_found",
            R"({ "points":
                [
                  "Zones_to_allow_drop_not_found"
                ]
            })",
            USER_ID_DEFAULT
        );
        env.GetServer()->GetSettings().SetValue(
            "warning_screens.checkers.rental_finish_location_checker.landings.return_location_not_allowed",
            R"({ "points":
                [
                  "Location: _NearestAdress_, _NearestDistance_"
                ]
            })",
            USER_ID_DEFAULT
        );
        env.GetServer()->GetSettings().SetValue(
            "warning_screens.checkers.rental_finish_location_checker.landings.return_locations_not_found",
            R"({ "points":
                [
                  "Offer_return_locations_not_present"
                ]
            })",
            USER_ID_DEFAULT
        );
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto car = env.GetEnvironmentGenerator().CreateCar();
        auto emulator = env.GetContext().SetCar(car).GetEmulator();
        UNIT_ASSERT(emulator);
        auto carId = car.Id;

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        const TGeoCoord returnLocation = {38.862048, 53.880882};
        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("offer_options",  NJson::TMapBuilder
                    ("child_seat", true)
                    ("roof_rack", true)
                    ("gps", false)
                    ("snow_chains", false)
                )

                ("delivery_location", NJson::ToJson(RentalOfferDeliveryLocation))
                ("delivery_location_name", "Prague")
                ("return_location", NJson::ToJson(returnLocation))
                ("return_location_name", "Prague")

                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("limit_km_per_day", 100)
                ("overrun_cost_per_km", 10)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + offerHolderTagId, {});

        TString chargableTagId;
        UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Reservation, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));
        env.GetContext().SetFuelLevel(30);
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));

        auto warningRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/warnings/get", "event=finish_rental&session_id=" + offerId, {});
        TString resStr = TStringBuilder() << warningRes;

        UNIT_ASSERT(resStr.find("Petrol") == std::string::npos);
        UNIT_ASSERT(resStr.find("Mileage") == std::string::npos);

        env.GetContext().SetFuelLevel(20);
        env.GetContext().SetMileage(1000);

        warningRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/warnings/get", "event=finish_rental&session_id=" + offerId, {});
        resStr = TStringBuilder() << warningRes;
        UNIT_ASSERT(resStr.find("Petrol") != std::string::npos);
        env.GetContext().SetFuelLevel(30);

        warningRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/warnings/get", "event=finish_rental&session_id=" + offerId, {});
        resStr = TStringBuilder() << warningRes;
        UNIT_ASSERT(resStr.find("Mileage") != std::string::npos);

        UNIT_ASSERT(resStr.find("Zones_to_allow_drop_not_found") != std::string::npos);

        NJson::TJsonValue jsonZone;
        UNIT_ASSERT(NJson::ReadJsonFastTree(
            R"({
                "object": [
                    {
                        "area_title": "title_zone_allow_drop",
                        "area_coords": [
                            [
                                38.80394376370033,
                                53.9203403411307
                            ],
                            [
                                38.956712419639224,
                                53.91407334063559
                            ],
                            [
                                38.93383955917397,
                                53.818194269414676
                            ],
                            [
                                38.77162282508495,
                                53.83119267419434
                            ],
                            [
                                38.80394376370033,
                                53.9203403411307
                            ]
                        ],
                        "area_type": "zone",
                        "revision": 100,
                        "area_details": {
                            "group": "zone allow drop",
                            "attributed_entity": {
                                "groupping_attributes": []
                            }
                        },
                        "area_tags": "allow_drop_car"
                    }
                ],
                "for_mobile": true,
                "revision": 1
            })"
        , &jsonZone));

        auto createdZone = env->Request(USER_ROOT_DEFAULT, "api/leasing/zone/add", "", jsonZone);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(5));

        warningRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/warnings/get", "event=finish_rental&session_id=" + offerId, {});
        resStr = TStringBuilder() << warningRes;
        INFO_LOG << resStr << Endl;
        UNIT_ASSERT(resStr.find("Zones_to_allow_drop_not_found") == std::string::npos);
        UNIT_ASSERT(resStr.find("Location: Prague") != std::string::npos);

        const TGeoCoord returnLocationInZoneAllow = {38.862047, 53.880881};
        env.GetContext().RelocateCar(returnLocationInZoneAllow, car);

        warningRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/warnings/get", "event=finish_rental&session_id=" + offerId, {});
        resStr = TStringBuilder() << warningRes;
        INFO_LOG << resStr << Endl;
        UNIT_ASSERT(resStr.find("Zones_to_allow_drop_not_found") == std::string::npos);
        UNIT_ASSERT(resStr.find("Location") == std::string::npos);

        const TGeoCoord returnLocationDeny = {48.862047, 63.880881};
        env.GetContext().RelocateCar(returnLocationDeny, car);

        warningRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/warnings/get", "event=finish_rental&session_id=" + offerId, {});
        resStr = TStringBuilder() << warningRes;
        INFO_LOG << resStr << Endl;
        UNIT_ASSERT(resStr.find("Zones_to_allow_drop_not_found") == std::string::npos);
        UNIT_ASSERT(resStr.find("Location") != std::string::npos);
    }

    Y_UNIT_TEST(RentalNotificationAfterAdd) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
                ("delivery_location_name", "Prague")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru")
        );
        auto createUserId =  res["client_id"].GetString();

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(createUserId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();
            auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TRentalOfferHolderTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag != nullptr);

            auto taggedUser = userTagManager.RestoreObject(createUserId, tx);
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserSmsTag>();
                auto smsTag = tag.GetTagAs<TUserSmsTag>();
                UNIT_ASSERT(smsTag);
                UNIT_ASSERT(smsTag->GetName() == confirmedSmsTagName);
            }
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserMailTag>();
                auto emailTag = tag.GetTagAs<TUserMailTag>();
                UNIT_ASSERT(emailTag);
                UNIT_ASSERT(emailTag->GetName() == confirmedEmailTagName);
            }
        }
    }

    Y_UNIT_TEST(RentalNotificationAfterUpdate) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "draft")
                ("delivery_location_name", "Prague")
                ("offer_options",  NJson::TMapBuilder
                    ("child_seat", true)
                    ("roof_rack", true)
                    ("gps", true)
                    ("snow_chains", true)
                    ("entry_to_eco_zones_in_germany", true)
                )
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru")
        );
        auto createUserId =  res["client_id"].GetString();

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(createUserId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["status"] = "confirmed";


        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();
            auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TRentalOfferHolderTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag != nullptr);
            auto pOffer = std::dynamic_pointer_cast<TRentalOffer>(pTag->GetOffer());
            UNIT_ASSERT("confirmed" == pOffer->GetStatusRef());

            auto taggedUser = userTagManager.RestoreObject(createUserId, tx);
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserSmsTag>();
                auto smsTag = tag.GetTagAs<TUserSmsTag>();
                UNIT_ASSERT(smsTag);
                UNIT_ASSERT(smsTag->GetName() == confirmedSmsTagName);
            }
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserMailTag>();
                auto emailTag = tag.GetTagAs<TUserMailTag>();
                UNIT_ASSERT(emailTag);
                UNIT_ASSERT(emailTag->GetName() == confirmedEmailTagName);
            }
        }
    }

    Y_UNIT_TEST(RentalNotificationAfterRemove) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
                ("delivery_location_name", "Prague")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru")
        );
        auto createUserId =  res["client_id"].GetString();

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(createUserId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            auto optionalTag = userTagManager.RestoreTag(offerHolderTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TRentalOfferHolderTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag != nullptr);

            auto taggedUser = userTagManager.RestoreObject(createUserId, tx);
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserSmsTag>();
                auto smsTag = tag.GetTagAs<TUserSmsTag>();
                UNIT_ASSERT(smsTag);
                UNIT_ASSERT(userTagManager.RemoveTag(tag, USER_ROOT_DEFAULT, env.GetServer().Get(), tx));
            }
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserMailTag>();
                auto emailTag = tag.GetTagAs<TUserMailTag>();
                UNIT_ASSERT(emailTag);
                UNIT_ASSERT(userTagManager.RemoveTag(tag, USER_ROOT_DEFAULT, env.GetServer().Get(), tx));
            }
            UNIT_ASSERT(tx.Commit());
        }

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/user_tags/remove", "",  NJson::TMapBuilder("tag_id", offerHolderTagId));
        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();

            auto taggedUser = userTagManager.RestoreObject(createUserId, tx);
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserSmsTag>();

                auto smsTag = tag.GetTagAs<TUserSmsTag>();
                UNIT_ASSERT(smsTag);
                UNIT_ASSERT(smsTag->GetName() == cancelledSmsTagName);
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("_CUSTOM1_"));
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("company_name"));
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("delivery_date"));
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("delivery_time"));
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("return_date"));
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("return_time"));
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("delivery_location"));
                UNIT_ASSERT(!smsTag->GetMessageText().Contains("car_model"));
            }
            {
                auto tag = taggedUser->GetFirstTagByClass<TUserMailTag>();
                auto emailTag = tag.GetTagAs<TUserMailTag>();
                UNIT_ASSERT(emailTag);
                UNIT_ASSERT(emailTag->GetName() == cancelledEmailTagName);
                auto& templateArgs = emailTag->GetTemplateArgs();
                UNIT_ASSERT(templateArgs.contains("_CUSTOM1_"));
                UNIT_ASSERT(templateArgs.contains("company_name"));
                UNIT_ASSERT(templateArgs.contains("delivery_date"));
                UNIT_ASSERT(templateArgs.contains("delivery_time"));
                UNIT_ASSERT(templateArgs.contains("return_date"));
                UNIT_ASSERT(templateArgs.contains("return_time"));
                UNIT_ASSERT(templateArgs.contains("delivery_location"));
                UNIT_ASSERT(templateArgs.contains("car_model"));

                UNIT_ASSERT(templateArgs.contains("company_phone"));
                UNIT_ASSERT(templateArgs.contains("company_mail"));
                UNIT_ASSERT(templateArgs.contains("_CUSTOM1_"));
                UNIT_ASSERT(templateArgs.contains("total_payment"));
                UNIT_ASSERT(templateArgs.contains("first_name"));
                UNIT_ASSERT(templateArgs.contains("client_phone"));
                UNIT_ASSERT(templateArgs.contains("insurance_type"));
                UNIT_ASSERT(templateArgs.contains("currency"));
                UNIT_ASSERT(templateArgs.contains("json_data"));
            }
        }
    }

    Y_UNIT_TEST(ServiceMode) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Days(7);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/tag/add", "tag_name=" + TRentalServiceModeTag::Type(),  NJson::TMapBuilder
            ("since", NJson::ToJson(sinceRental))
            ("car_id", carId)
            ("job_type", "some job")
            ("job_comment", "Prague")
            ("mileage", 100.0)
        );

        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            const auto& deviceTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
            auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly>();

            auto taggedUser = deviceTagManager.RestoreObject(carId, tx);
            {
                auto tag = taggedUser->GetFirstTagByClass<TRentalServiceModeTag>();
                auto serviceTag = tag.GetTagAs<TRentalServiceModeTag>();
                UNIT_ASSERT(serviceTag);

                res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/tag/details", "tag_id=" + tag.GetTagId(), {});
                auto& tagDetails = res["tag"];
                {
                    UNIT_ASSERT(!tagDetails.Has("until"));
                    UNIT_ASSERT(tagDetails["job_type"] == "some job");
                    UNIT_ASSERT(tagDetails["job_comment"] == "Prague");
                    UNIT_ASSERT(std::abs(tagDetails["mileage"].GetDouble() - 100.) < std::numeric_limits<double>::epsilon());
                }

                sinceRental += TDuration::Days(1);
                untilRental -= TDuration::Days(3);
                tagDetails["since"] = NJson::ToJson(sinceRental);
                tagDetails["until"] = NJson::ToJson(untilRental);
                tagDetails["job_type"] = "another job";
                tagDetails["job_comment"] = "Moscow";
                tagDetails["mileage"] = 200.;
                tagDetails["tag_name"] = TRentalServiceModeTag::Type();
                res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tag/evolve", "tag_id=" + tag.GetTagId(), tagDetails);
            }
        }


        {
            const auto& deviceTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
            auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly>();
            auto taggedUser = deviceTagManager.RestoreObject(carId, tx);

            auto tag = taggedUser->GetFirstTagByClass<TRentalServiceModeTag>();
            auto serviceTag = tag.GetTagAs<TRentalServiceModeTag>();
            UNIT_ASSERT(serviceTag->GetSince() == sinceRental);
            UNIT_ASSERT(serviceTag->GetUntilUnsafe() == untilRental);
            UNIT_ASSERT(serviceTag->GetJobType() == "another job");
            UNIT_ASSERT(serviceTag->GetJobComment() == "Moscow");
            UNIT_ASSERT(std::abs(serviceTag->GetMileage() - 200.) < std::numeric_limits<double>::epsilon());
        }
        sinceRental += TDuration::Days(7);
        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/tag/add", "tag_name=" + TRentalServiceModeTag::Type(),  NJson::TMapBuilder
            ("since", NJson::ToJson(sinceRental))
            ("car_id", carId)
            ("job_type", "some job")
            ("job_comment", "Prague")
            ("mileage", 100.0)
        );
        auto unlimitedServiceTagId = res["tagged_objects"].GetArray().front()["tag_id"].GetArray().front().GetString();

        sinceRental += TDuration::Days(30);
        untilRental = sinceRental +  TDuration::Days(30);
        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == "rental.book.car_is_already_used_by_another_booking");
        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/car_tags/remove", "tag_id=" + unlimitedServiceTagId, {}, false);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );
        UNIT_ASSERT(res["client_id"] == userId);

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        UNIT_ASSERT(carsharingTimetable["offers_timetable"].Has(carId));
        UNIT_ASSERT(carsharingTimetable["offers_timetable"][carId].GetArray().size() == 2);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/tag/add", "tag_name=" + TRentalServiceModeTag::Type(),  NJson::TMapBuilder
            ("since", NJson::ToJson(sinceRental))
            ("until", NJson::ToJson(untilRental))
            ("car_id", carId)
            ("job_type", "some job")
            ("job_comment", "Prague")
            ("mileage", 100.0),
            false
        );
        UNIT_ASSERT(res["error_details"]["special_info"]["error_code"] == "rental.book.car_is_already_used_by_another_booking");
    }

    Y_UNIT_TEST(EarlyFinishRiding) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Days(7);
        BookRental(env, sinceRental, untilRental, "draft");

        sinceRental += TDuration::Days(1);
        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto conflictOfferId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(conflictOfferId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("client_id", userId),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == "rental.book.user_already_has_booking");

        auto finishRiding = [&]() {
            auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

            auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
            auto offerId = jObj["offer_id"].GetString();
            auto tagId = jObj["tag_id"].GetString();
            auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
            createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

            createdOffer["tag_id"] = tagId;
            createdOffer["status"] = "paid";

            auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
            updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

            TString chargableTagId;
            UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Reservation, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, userId));
        };

        finishRiding();

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("client_id", userId),
            false
        );
        UNIT_ASSERT(res.Has("client_id"));

        sinceRental += TDuration::Days(1);

        auto planService = [&]() {
            res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/tag/add", "tag_name=" + TRentalServiceModeTag::Type(),  NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("car_id", carId)
                ("job_type", "some job")
                ("job_comment", "Prague")
                ("mileage", 100.0),
                false
            );
        };
        planService();
        UNIT_ASSERT(res["error_details"]["special_info"]["error_code"] == "rental.book.car_is_already_used_by_another_booking");
        finishRiding();
        planService();
        UNIT_ASSERT(res.Has("tagged_objects"));
        planService();
        UNIT_ASSERT(res["error_details"]["special_info"]["error_code"] == "rental.book.car_is_already_used_by_another_booking");
    }

    Y_UNIT_TEST(LateFinishRiding) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Seconds(20);
        BookRental(env, sinceRental, untilRental, "draft");

        sinceRental += TDuration::Seconds(30);
        untilRental += TDuration::Seconds(30);
        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto conflictOfferId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(conflictOfferId);

        auto startRiding = [&]() {
            auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

            auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
            auto offerId = jObj["offer_id"].GetString();
            auto tagId = jObj["tag_id"].GetString();
            auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
            createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

            createdOffer["tag_id"] = tagId;
            createdOffer["status"] = "paid";

            auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
            updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);
        };

        startRiding();
        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("client_id", userId),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == "rental.book.car_is_already_used_by_another_booking");

        auto finishRiding = [&]() {
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));
            UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, userId));
        };

        finishRiding();

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("client_id", userId),
            false
        );

        UNIT_ASSERT(res.Has("client_id"));
    }

    Y_UNIT_TEST(LateFinishRidingBeforeService) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Seconds(20);
        BookRental(env, sinceRental, untilRental, "draft");

        sinceRental += TDuration::Seconds(25);
        untilRental += TDuration::Seconds(30);

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        auto planService = [&]() {
            res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/tag/add", "tag_name=" + TRentalServiceModeTag::Type(),  NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("car_id", carId)
                ("job_type", "some job")
                ("job_comment", "Prague")
                ("mileage", 100.0),
                false
            );
        };

        planService();
        UNIT_ASSERT(res["error_details"]["special_info"]["error_code"] == "rental.book.car_is_already_used_by_another_booking");

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, userId));

        planService();
        UNIT_ASSERT(res.Has("tagged_objects"));
    }

    Y_UNIT_TEST(RentalBill) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Seconds(20);
        BookRental(env, sinceRental, untilRental, "draft");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, userId));

        auto billRes = env->Request(USER_ROOT_DEFAULT, "/user_app/sessions/history", "session_id=" + offerId + "&report=user_app&need_bill=false", createdOffer);

        auto& billArr = billRes["sessions"][0]["segment"]["bill"].GetArray();
        INFO_LOG << billRes["sessions"][0]["segment"]["bill"].GetStringRobust() << Endl;
        TSet<TString> requiredFiled{
              "mileage"
            , "duration"
            , "deposit"
            , "total"
            , "section"
            , "insurance_type"
            , "mileage_start"
            , "mileage_finish"
            , "fuel_tank_level_start"
            , "fuel_tank_level_finish"
        };
        for (const auto& el : billArr) {
            UNIT_ASSERT_C(el["type"].GetString().empty() || requiredFiled.contains(el["type"].GetString()), "type: " << el["type"].GetString() << " not found");
        }
    }

    Y_UNIT_TEST(UpdateBookingDuringRiding) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Seconds(20);
        BookRental(env, sinceRental, untilRental, "draft");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        ui64 newCost = 999;
        ui64 newDailyCost = 520;
        createdOffer["total_payment"] = newCost;
        createdOffer["total_daily_cost"] = newDailyCost;
        createdOffer["offer_id"] = offerId;
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", {}, createdOffer);

        TString chargableTagId;
        UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Riding, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));

        {
            const auto& deviceTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
            auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly>();
            auto optionalTag = deviceTagManager.RestoreTag(chargableTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TChargableTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag->GetPerformer() == userId);

            auto offer = pTag->GetOffer();
            UNIT_ASSERT(offer);

            auto rentalOffer = std::dynamic_pointer_cast<TRentalOffer>(offer);
            UNIT_ASSERT(rentalOffer);
            UNIT_ASSERT(rentalOffer->GetTotalPaymentRef() == newCost);
        }

        {
            auto driveApi =  env.GetServer()->GetDriveAPI();
            TAtomicSharedPtr<const ISession> offerSessison;
            UNIT_ASSERT(driveApi->GetUserSession(userId, offerSessison, offerId, Now()));
            R_ENSURE(offerSessison, HTTP_NOT_FOUND, "no session found");
            auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(offerSessison);
            R_ENSURE(billingSession, HTTP_INTERNAL_SERVER_ERROR, "cannot cast session " << offerSessison->GetSessionId() << " to BillingSession");
            auto offer = billingSession->GetCurrentOffer();
            UNIT_ASSERT(offer);

            auto rentalOffer = std::dynamic_pointer_cast<TRentalOffer>(offer);
            UNIT_ASSERT(rentalOffer);

            UNIT_ASSERT(rentalOffer->GetTotalPaymentRef() == newCost);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        auto offerSecondUpdate = createdOffer["sessions"][0]["segment"]["offer"];
        UNIT_ASSERT(createdOffer["sessions"][0]["segment"]["offer"]["total_payment"] == newCost);

        createdOffer = env->Request(userId, "/api/yandex/sessions/current", "multi_sessions=1&report=user_app", {});
        UNIT_ASSERT(createdOffer["sessions"][0]["segment"]["session"]["specials"]["current_offer"]["total_payment"] == newCost);
        UNIT_ASSERT(createdOffer["sessions"][0]["segment"]["session"]["specials"]["current_offer"]["total_daily_cost"] == newDailyCost);

        newCost = 555;
        offerSecondUpdate["total_payment"] = newCost;
        offerSecondUpdate["offer_id"] = offerId;
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", {}, offerSecondUpdate);

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer");
        UNIT_ASSERT(createdOffer["sessions"][0]["segment"]["offer"]["total_payment"] == newCost);

        createdOffer = env->Request(userId, "/api/yandex/sessions/current", "multi_sessions=1&report=user_app", {});
        UNIT_ASSERT(createdOffer["sessions"][0]["segment"]["session"]["specials"]["current_offer"]["total_payment"] == newCost);
    }

    Y_UNIT_TEST(BuildTimetableFromSessions) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Seconds(20);
        BookRental(env, sinceRental, untilRental, "draft");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];


        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            TTimetableBuilder::TCarsTimetable carsTimetable;
            TSet<TString> fetchedOfferIds;
            auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT);
            UNIT_ASSERT(TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable, {}, {}, sinceTime, untilTime, *permissions, env.GetServer().Get(), tx, false));
            UNIT_ASSERT(carsTimetable.size() == 0);
        }

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));

        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            TTimetableBuilder::TCarsTimetable carsTimetable;
            TSet<TString> fetchedOfferIds;
            auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT);
            UNIT_ASSERT(TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable, {}, {}, sinceTime, untilTime, *permissions, env.GetServer().Get(), tx, false));
            UNIT_ASSERT(carsTimetable.size() == 1);
        }
    }

    Y_UNIT_TEST(BuildTimetableFromSessionsEditUntil) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Seconds(20);
        BookRental(env, sinceRental, untilRental, "draft");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            TTimetableBuilder::TCarsTimetable carsTimetable;
            auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT);
            UNIT_ASSERT(TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable, {}, {}, sinceTime, untilTime, *permissions, env.GetServer().Get(), tx, false));
            UNIT_ASSERT(carsTimetable.size() == 0);
        }

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));

        untilRental += TDuration::Days(3);
        createdOffer["until"] = untilRental.MicroSeconds();
        createdOffer["revision"] = 100;
        createdOffer["offer_id"] = offerId;
        auto billRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", {}, createdOffer);

        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            TTimetableBuilder::TCarsTimetable carsTimetable;
            auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT);
            UNIT_ASSERT(TTimetableBuilder::Instance().BuildTimetable<TTimetableBuilder::ETimetableType::Rental>(carsTimetable, {}, {}, sinceTime, untilTime, *permissions, env.GetServer().Get(), tx, false));
            UNIT_ASSERT(carsTimetable.size() == 1);
            UNIT_ASSERT(carsTimetable.begin()->second.begin()->second.Until == untilRental);
        }
    }

    Y_UNIT_TEST(TryUpdateFinishedRiding) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Seconds(20);
        BookRental(env, sinceRental, untilRental, "draft");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, userId));

        ui64 newCost = 999;
        createdOffer["tag_id"] = tagId;
        createdOffer["total_payment"] = newCost;
        createdOffer["offer_id"] = offerId;
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", {}, createdOffer, false);
        UNIT_ASSERT(updateRes["error_details"]["special_info"]["error_code"] == "offer.update.active_riding_not_found");
    }

    Y_UNIT_TEST(ConflictBookingEditRiding) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Days(5);
        BookRental(env, sinceRental, untilRental, "draft");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        sinceRental += TDuration::Days(1);
        auto offers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto conflictOfferId = offers["offers"][0]["offer_id"].GetString();

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru"),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == "rental.book.car_is_already_used_by_another_booking");

        createdOffer["tag_id"] = tagId;
        createdOffer["offer_id"] = offerId;
        createdOffer["until"] = NJson::ToJson(sinceRental);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", {}, createdOffer);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru")
        );
    }

    Y_UNIT_TEST(ConflictBookingEarlyFinishedRiding) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);
        NJson::TJsonValue res;

        auto userId = TString{USER_ID_DEFAULT};

        const auto sinceTime = Now();
        const auto untilTime = sinceTime + TDuration::Days(100);

        auto sinceRental = sinceTime;
        auto untilRental = sinceTime + TDuration::Days(5);
        BookRental(env, sinceRental, untilRental, "draft");

        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceTime.Seconds()) + "&until=" + ToString(untilTime.Seconds()));

        auto jObj = carsharingTimetable["offers_timetable"][carId].GetArray().back();
        auto offerId = jObj["offer_id"].GetString();
        auto tagId = jObj["tag_id"].GetString();
        auto createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = tagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + tagId);

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, userId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, userId));

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        sinceRental += TDuration::Days(1);
        auto offers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto conflictOfferId = offers["offers"][0]["offer_id"].GetString();

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru"),
            false
        );
        UNIT_ASSERT(res["error_details"]["debug_message"] == "rental.book.car_is_already_used_by_another_booking");

        UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, userId));

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", conflictOfferId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru")
        );
    }

    Y_UNIT_TEST(OrganizationList) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("first_name", "username")
            ("phone_number", "+79055553388")
            ("email", "test@yandex.ru")
        );
        auto createUserId =  res["client_id"].GetString();

        auto companyList = env->Request(createUserId, "/api/yandex/organization/list");
        auto& company = companyList[0];
        UNIT_ASSERT(company["short_description"] == "some description");
        UNIT_ASSERT(company["icon"] == "login_icon.png");
        UNIT_ASSERT(company["company_name"] == "test 1");
        UNIT_ASSERT(company["user_id"] == createUserId);
    }

    Y_UNIT_TEST(EmailTest) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto driveApi = env.GetServer()->GetDriveAPI();
        {
            THolder<TEmailAfterAcceptanceEvolutionPolicyAction> action(new TEmailAfterAcceptanceEvolutionPolicyAction("email_after_acceptance"));
            action->SetNotificationType(TRentalOffer::AcceptanceFinishedNotificationType);
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, tx));
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("email_after_acceptance").SetRoleId("root");
            UNIT_ASSERT(driveApi->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, tx));
            UNIT_ASSERT(tx.Commit());
        }

        {
            THolder<TEmailAfterFinishRideEvolutionPolicyAction> action(new TEmailAfterFinishRideEvolutionPolicyAction("email_after_finish_ride"));
            action->SetNotificationType(TRentalOffer::RideFinishedNotificationType);
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, tx));
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("email_after_finish_ride").SetRoleId("root");
            UNIT_ASSERT(driveApi->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, tx));
            UNIT_ASSERT(tx.Commit());
        }

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", USER_ROOT_DEFAULT)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(USER_ROOT_DEFAULT, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        TStartRentalProcess::Execute((*env.GetServer()), USER_ROOT_DEFAULT, TDuration::Hours(1));

        TString chargableTagId;
        UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Reservation, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Acceptance, USER_ROOT_DEFAULT));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Riding, USER_ROOT_DEFAULT));
        UNIT_ASSERT(env->EvolveTag(TChargableTag::Reservation, USER_ROOT_DEFAULT));

        {
            const auto& userTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetUserTags();
            auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();

            auto taggedUser = userTagManager.RestoreObject(USER_ROOT_DEFAULT, tx);
            bool acceptanceFound = false;
            bool rideFound = false;
            {
                auto tags = taggedUser->GetTagsByClass<TUserMailTag>();
                for (auto& tag: tags) {
                    auto emailTag = tag.GetTagAs<TUserMailTag>();
                    if (emailTag->GetName() == afterAcceptanceEmailTagName) {
                        acceptanceFound = true;
                    } else if (emailTag->GetName() == afterFinishRideEmailTagName) {
                        rideFound = true;
                    }
                }
            }
            UNIT_ASSERT(acceptanceFound);
            UNIT_ASSERT(rideFound);
        }
    }

    Y_UNIT_TEST(EconomyWriteRead) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        auto actionReport = res["report"];
        auto actionMeta = actionReport["action_meta"];
        NJson::TJsonValue jEconomy;

        {
            NJson::TJsonValue jMigration;
            NJson::TJsonArray jTarrifs, jRanges;
            {
                NJson::TJsonValue jRange;
                jRange["id"] = "id1";
                jRange["end"] = 3;
                jRanges.AppendValue(jRange);
                jRange["id"] = "id2";
                jRange["end"] = 6;
                jRanges.AppendValue(jRange);
                jMigration["ranges"] = jRanges;
            }

            {
                NJson::TJsonValue jTariff;
                jTariff["name"] = "econom";
                jTariff["id"] = "id1";
                jTariff["model"]["id"] = "model_id1";
                jTariff["category"]["id"] = "category_id1";
                jTariff["exterior_type"]["id"] = "exteriortype_id1";
                jTariff["transmission"]["id"] = GetEnumNames<NDriveModelSpecification::ETransmissionType>().begin()->second;
                jTariff["fuel_type"]["id"] = "fuel_type1";
                jTariff["air_conditioning"]["id"] = "air1";
                jTarrifs.AppendValue(jTariff);

                jTariff["name"] = "business";
                jTariff["id"] = "id2";
                jTariff["model"]["id"] = "model_id2";
                jTariff["category"]["id"] = "category_id2";
                jTariff["exterior_type"]["id"] = "exteriortype_id2";
                jTariff["transmission"]["id"] = GetEnumNames<NDriveModelSpecification::ETransmissionType>().begin()->second;
                jTariff["fuel_type"]["id"] = "fuel_type2";
                jTariff["air_conditioning"]["id"] = "air2";
                jTarrifs.AppendValue(jTariff);

                jMigration["tariffs"] = jTarrifs;
            }
            jEconomy["migration"] = jMigration;
        }

        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;

        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);
        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["ranges"].GetArray().size() == 2);
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["tariffs"].GetArray().size() == 2);

        actionReport = res["report"];
        actionMeta = actionReport["action_meta"];
        jEconomy = actionMeta["economy"];

        {
            NJson::TJsonValue jMigration;
            auto jGridData = jEconomy["grid"];
            {
                jGridData[0]["tariff_costs"][0]["daily_cost"] = 500;
            }
            jMigration["grid"] = jGridData;
            jEconomy["migration"] = jMigration;
        }

        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;

        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);
        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["grid"][0]["tariff_costs"][0]["daily_cost"].GetInteger() == 500);

        actionReport = res["report"];
        actionMeta = actionReport["action_meta"];
        jEconomy = actionMeta["economy"];

        {
            NJson::TJsonValue jMigration;
            auto jGridData = jEconomy["grid"];
            {
                jGridData[0]["tariff_costs"][0]["daily_cost"] = NJson::JSON_NULL;
            }
            jMigration["grid"] = jGridData;
            jEconomy["migration"] = jMigration;
        }

        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;

        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);
        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});

        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["grid"][0]["tariff_costs"][0]["daily_cost"].IsNull());

        actionReport = res["report"];
        actionMeta = actionReport["action_meta"];
        jEconomy = actionMeta["economy"];

        {
            NJson::TJsonValue jMigration;
            NJson::TJsonArray jTarrifs, jRanges;
            {
                NJson::TJsonValue jRange;
                jRange["id"] = "id2";
                jRange["end"] = 9;
                jRanges.AppendValue(jRange);
                jMigration["ranges"] = jRanges;
            }

            {
                NJson::TJsonValue jTariff;
                jTariff["name"] = "econom_modified";
                jTariff["id"] = "id1";
                jTariff["model"]["id"] = "model_id1";
                jTariff["category"]["id"] = "category_id1";
                jTariff["exterior_type"]["id"] = "exteriortype_id1";
                jTariff["transmission"]["id"] = GetEnumNames<NDriveModelSpecification::ETransmissionType>().begin()->second;
                jTariff["fuel_type"]["id"] = "fuel_type1";
                jTariff["air_conditioning"]["id"] = "air1";
                jTariff["deposit"] = 900;
                jTarrifs.AppendValue(jTariff);

                jMigration["tariffs"] = jTarrifs;
            }

            {
                NJson::TJsonArray jOptions;
                {
                    NJson::TJsonValue jOption;
                    jOption["id"] = "child_seat";
                    jOption["cost"] = 300;
                    jOption["is_cost_per_day"] = true;
                    jOption["max_total_cost"] = 500;
                    jOptions.AppendValue(jOption);
                }
                jMigration["options"] = jOptions;
            }

            {
                NJson::TJsonArray jInsurances;
                {
                    NJson::TJsonValue jInsurance;
                    jInsurance["id"] = "id_gold";
                    jInsurances.AppendValue(jInsurance);
                }
                {
                    NJson::TJsonValue jInsurance;
                    jInsurance["id"] = "id_basic";
                    jInsurances.AppendValue(jInsurance);
                }
                jMigration["insurances"] = jInsurances;
            }

            jEconomy["migration"] = jMigration;
        }

        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;

        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);
        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["ranges"].GetArray().size() == 1);
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["tariffs"].GetArray().size() == 1);
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["tariffs"][0]["deposit"].GetUInteger() == 900);
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["options"].GetArray().size() == 1);
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["options"][0]["id"].GetString() == "child_seat");
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["options"][0]["cost"].GetUInteger() == 300);
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["options"][0]["is_cost_per_day"].GetBoolean());
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["options"][0]["max_total_cost"].GetUInteger() == 500);
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["insurances"][0]["id"].GetString() == "id_gold");
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["insurances"][1]["id"].GetString() == "id_basic");
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["insurances_grid"][0]["id"].GetString() == "id_gold");
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["insurances_grid"][0]["insurances_costs"][0]["cost"].IsNull());
    }


    Y_UNIT_TEST(TariffFilters) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        CreateModel(env);
        auto reply = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/filters/list", {}, {});
        auto filters = reply["filters"];
        UNIT_ASSERT(filters["model"].GetArray()[0]["name"].GetString() == "some_name");
        UNIT_ASSERT(filters["category"].GetArray()[0]["id"].GetString() == "some_category");
        UNIT_ASSERT(filters["air_conditioning"].GetArray()[0]["id"].GetString() == "some_conditioning");
        UNIT_ASSERT(filters["exterior_type"].GetArray()[0]["id"].GetString() == "some_exterior_type");
        UNIT_ASSERT(filters["fuel_type"].GetArray()[0]["id"].GetString() == "some_fuel_type");
        UNIT_ASSERT(filters["transmission"].GetArray()[0]["id"].GetString() == "Robotic");
    }

    Y_UNIT_TEST(RentalActionRead) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        CreateModel(env);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        auto actionReport = res["report"];
        auto actionMeta = actionReport["action_meta"];
        NJson::TJsonValue jEconomy;

        {
            NJson::TJsonValue jMigration;
            NJson::TJsonArray jTarrifs, jRanges;
            {
                NJson::TJsonValue jRange;
                jRange["id"] = "id1";
                jRange["end"] = 3;
                jRanges.AppendValue(jRange);
                jRange["id"] = "id2";
                jRange["end"] = 6;
                jRanges.AppendValue(jRange);
                jMigration["ranges"] = jRanges;
            }

            {
                NJson::TJsonValue jTariff;
                jTariff["name"] = "econom";
                jTariff["id"] = "id1";
                jTariff["model"]["id"] = "code123";
                jTariff["category"]["id"] = "category_id1";
                jTariff["exterior_type"]["id"] = "exteriortype_id1";
                jTariff["transmission"]["id"] = GetEnumNames<NDriveModelSpecification::ETransmissionType>().begin()->second;
                jTariff["fuel_type"]["id"] = "fuel_type1";
                jTariff["air_conditioning"]["id"] = "air1";
                jTarrifs.AppendValue(jTariff);

                jTariff["name"] = "business";
                jTariff["id"] = "id2";
                jTariff["model"]["id"] = "model_id2";
                jTariff["category"]["id"] = "category_id2";
                jTariff["exterior_type"]["id"] = "exteriortype_id2";
                jTariff["transmission"]["id"] = GetEnumNames<NDriveModelSpecification::ETransmissionType>().begin()->second;
                jTariff["fuel_type"]["id"] = "fuel_type2";
                jTariff["air_conditioning"]["id"] = "air2";
                jTarrifs.AppendValue(jTariff);

                jMigration["tariffs"] = jTarrifs;
            }
            jEconomy["migration"] = jMigration;
        }

        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;

        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);

        {
            NJson::TJsonValue jEditCar;
            jEditCar["id"] =  TString{OBJECT_ID_DEFAULT};
            jEditCar["model_code"] = "code123";
            jEditCar["force"] = true;
            res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/edit", {}, jEditCar);
        }

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        UNIT_ASSERT(res["report"]["action_meta"]["economy"]["tariffs"].GetArray()[1]["cars_count"].GetInteger() == 1);
    }

    Y_UNIT_TEST(TariffRecommendation) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        CreateModel(env);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        auto actionReport = res["report"];
        auto actionMeta = actionReport["action_meta"];
        NJson::TJsonValue jEconomy;
        NJson::TJsonValue jSavedTariff, jSavedInsurance;
        {
            NJson::TJsonValue jMigration;
            NJson::TJsonArray jTarrifs, jRanges;
            {
                NJson::TJsonValue jRange;
                jRange["id"] = "id1";
                jRange["end"] = 3;
                jRanges.AppendValue(jRange);
                jRange["id"] = "id2";
                jRange["end"] = 6;
                jRanges.AppendValue(jRange);
                jMigration["ranges"] = jRanges;
            }

            {
                NJson::TJsonValue jTariff;
                jTariff["name"] = "econom";
                jTariff["id"] = "id1";
                jTariff["model"]["id"] = "code123";
                jTariff["category"]["id"] = "category_id1";
                jTariff["exterior_type"]["id"] = "exteriortype_id1";
                jTariff["transmission"]["id"] = GetEnumNames<NDriveModelSpecification::ETransmissionType>().begin()->second;
                jTariff["fuel_type"]["id"] = "fuel_type1";
                jTariff["air_conditioning"]["id"] = "air1";
                jTariff["deposit"] = 333;
                jTariff["overrun_cost_per_km"] = 999;
                jSavedTariff = jTariff;
                jTarrifs.AppendValue(jTariff);

                jTariff["name"] = "business";
                jTariff["id"] = "id2";
                jTariff["model"]["id"] = "model_id2";
                jTariff["category"]["id"] = "category_id2";
                jTariff["exterior_type"]["id"] = "exteriortype_id2";
                jTariff["transmission"]["id"] = GetEnumNames<NDriveModelSpecification::ETransmissionType>().begin()->second;
                jTariff["fuel_type"]["id"] = "fuel_type2";
                jTariff["air_conditioning"]["id"] = "air2";
                jTarrifs.AppendValue(jTariff);

                jMigration["tariffs"] = jTarrifs;
            }

            {
                NJson::TJsonArray jInsurances;
                NJson::TJsonValue jInsurance;
                {
                    jInsurance["id"] = "id_gold";
                    jSavedInsurance = jInsurance;
                    jInsurances.AppendValue(jInsurance);
                }
                {
                    jInsurance["id"] = "id_basic";
                    jInsurances.AppendValue(jInsurance);
                }
                jMigration["insurances"] = jInsurances;
            }
            jEconomy["migration"] = jMigration;
        }

        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;

        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);

        {
            NJson::TJsonValue jEditCar;
            jEditCar["id"] =  TString{OBJECT_ID_DEFAULT};
            jEditCar["model_code"] = "code123";
            jEditCar["force"] = true;
            res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/car/edit", {}, jEditCar);
        }

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(4);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
            )
        );
        UNIT_ASSERT(createdOffers["offers"].GetArray()[0]["recommendations"]["tariff"].Has("daily_cost"));
        UNIT_ASSERT(createdOffers["offers"].GetArray()[0]["recommendations"]["tariff"]["id"] == "id1");
        UNIT_ASSERT(createdOffers["offers"].GetArray()[0]["recommendations"]["tariff"]["name"] == "econom");

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        actionReport = res["report"];
        actionMeta = actionReport["action_meta"];
        jEconomy = actionMeta["economy"];

        {
            NJson::TJsonValue jMigration;
            auto jGridData = jEconomy["grid"];
            {
                jGridData[0]["tariff_costs"][0]["daily_cost"] = 33;
                jGridData[0]["tariff_costs"][1]["daily_cost"] = 31;
                jGridData[1]["tariff_costs"][0]["daily_cost"] = 34;
                jGridData[1]["tariff_costs"][1]["daily_cost"] = 32;
            }
            jMigration["grid"] = jGridData;

            {
                NJson::TJsonArray jOptions;
                {
                    NJson::TJsonValue jOption;
                    jOption["id"] = "new_option1";
                    jOption["cost"] = 300;
                    jOption["is_cost_per_day"] = true;
                    jOption["max_total_cost"] = 500;
                    jOptions.AppendValue(jOption);
                }
                jMigration["options"] = jOptions;
            }

            auto jInsurancesGridData = jEconomy["insurances_grid"];
            {
                jInsurancesGridData[0]["insurances_costs"][0]["cost"] = 33;
                jInsurancesGridData[0]["insurances_costs"][1]["cost"] = 31;
                jInsurancesGridData[1]["insurances_costs"][0]["cost"] = 34;
                jInsurancesGridData[1]["insurances_costs"][1]["cost"] = 32;
            }
            jMigration["insurances_grid"] = jInsurancesGridData;

            auto jLimitsGridData = jEconomy["limits_grid"];
            {
                jLimitsGridData[0]["limits"][0]["mileage"] = 200;
                jLimitsGridData[0]["limits"][1]["mileage"] = 300;
            }
            jMigration["limits_grid"] = jLimitsGridData;

            jEconomy["migration"] = jMigration;
        }

        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;
        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);
        NJson::TJsonArray options;
        {
            NJson::TJsonValue jval;
            jval["id"] = "snow_chains";
            jval["value"] = true;
            jval["cost"] = 100;
            options.AppendValue(jval);
        }

        {
            NJson::TJsonValue jval;
            jval["id"] = "entry_to_eco_zones_in_germany";
            jval["value"] = true;
            jval["cost"] = 50;
            options.AppendValue(jval);
        }
        {
            NJson::TJsonValue jval;
            jval["id"] = "new_option1";
            jval["value"] = true;
            jval["cost"] = 77;
            options.AppendValue(jval);
        }
        {
            NJson::TJsonValue jval;
            jval["id"] = "new_option2";
            jval["value"] = false;
            jval["cost"] = 92;
            options.AppendValue(jval);
        }

        createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
                ("options", options)
            )
        );
        auto& recommendations = createdOffers["offers"].GetArray()[0]["recommendations"];
        UNIT_ASSERT(recommendations["tariff"]["daily_cost"] == 32);
        UNIT_ASSERT(recommendations["tariff"]["deposit"] == 333);
        UNIT_ASSERT(recommendations["tariff"]["overrun_cost_per_km"] == 999);
        UNIT_ASSERT(recommendations["tariff"]["total_daily_cost"] == 32 * (untilRental - sinceRental).Days());
        UNIT_ASSERT(recommendations["options"][0]["cost"].GetUInteger() == 500);
        UNIT_ASSERT(recommendations["options"][0]["id"].GetString() == "new_option1");
        UNIT_ASSERT(recommendations["total_payment"].GetUInteger() == 32 * (untilRental - sinceRental).Days() + 500);
        UNIT_ASSERT(recommendations["insurances"][0]["id"].GetString() == "id_gold");
        UNIT_ASSERT(recommendations["insurances"][0]["cost"].GetUInteger() == 31);
        UNIT_ASSERT(recommendations["insurances"][1]["id"].GetString() == "id_basic");
        UNIT_ASSERT(recommendations["insurances"][1]["cost"].GetUInteger() == 32);
        UNIT_ASSERT(recommendations["limits"]["mileage"].GetUInteger() == 300);

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        actionReport = res["report"];
        actionMeta = actionReport["action_meta"];
        jEconomy = actionMeta["economy"];
        {
            NJson::TJsonValue jMigration;
            {
                NJson::TJsonArray jTarrifs;
                jTarrifs.AppendValue(jSavedTariff);
                jMigration["tariffs"] = jTarrifs;
            }
            {
                NJson::TJsonArray jInsurances;
                jInsurances.AppendValue(jSavedInsurance);
                jMigration["insurances"] = jInsurances;
            }
            jEconomy["migration"] = jMigration;
        }
        actionMeta["economy"] = jEconomy;
        actionReport["action_meta"] = actionMeta;

        res = env->Request(USER_ROOT_DEFAULT, "/api/staff/actions/add", {}, actionReport);
        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tariffs/list", "action_id=" + SimpleRentalOfferName, {});
        actionReport = res["report"];
        actionMeta = actionReport["action_meta"];
        jEconomy = actionMeta["economy"];
        UNIT_ASSERT(jEconomy["grid"][0]["tariff_costs"].GetArray().size() == 1);
        UNIT_ASSERT(jEconomy["insurances_grid"].GetArray().size() == 1);
        UNIT_ASSERT(jEconomy["insurances_grid"][0]["insurances_costs"].GetArray().size() == 1);
        UNIT_ASSERT(jEconomy["limits_grid"][0]["limits"].GetArray().size() == 1);
    }

    Y_UNIT_TEST(Options) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOffer(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        NJson::TJsonArray options;
        {
            NJson::TJsonValue jval;
            jval["id"] = "snow_chains";
            jval["value"] = true;
            jval["cost"] = 100;
            options.AppendValue(jval);
        }

        {
            NJson::TJsonValue jval;
            jval["id"] = "entry_to_eco_zones_in_germany";
            jval["value"] = true;
            jval["cost"] = 50;
            options.AppendValue(jval);
        }
        {
            NJson::TJsonValue jval;
            jval["id"] = "new_option1";
            jval["value"] = true;
            jval["cost"] = 77;
            options.AppendValue(jval);
        }

        {
            NJson::TJsonValue jval;
            jval["id"] = "new_option2";
            jval["value"] = false;
            jval["cost"] = 92;
            options.AppendValue(jval);
        }
        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("offer_options",  NJson::TMapBuilder
                    ("child_seat", true)
                    ("roof_rack", true)
                    ("gps", true)
                )
                ("options", options)
                ("insurance_type", "id_gold")
                ("delivery_location", NJson::ToJson(RentalOfferDeliveryLocation))
                ("delivery_location_name", "Prague")
                ("return_locations", NJson::ToJson(GetRentalReturnLocations()))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "confirmed")
                ("limit_km_per_day", 100)
                ("overrun_cost_per_km", 10)
                ("currency", "czk")
            )
        );

        auto offerId = createdOffers["offers"][0]["offer_id"].GetString();
        UNIT_ASSERT(offerId);
        auto& offer =  createdOffers["offers"][0];
        UNIT_ASSERT(offer["offer_options"]["child_seat"].GetBoolean());
        UNIT_ASSERT(offer["offer_options"]["new_option1"].GetBoolean());
        UNIT_ASSERT(offer["offer_options"]["roof_rack"].GetBoolean());
        UNIT_ASSERT(offer["offer_options"]["snow_chains"].GetBoolean());
        UNIT_ASSERT(offer["offer_options"]["gps"].GetBoolean());
        UNIT_ASSERT(offer["offer_options"]["entry_to_eco_zones_in_germany"].GetBoolean());
        UNIT_ASSERT(!offer["offer_options"]["new_option2"].GetBoolean());

        UNIT_ASSERT(offer["options"][0]["value"].GetBoolean());
        UNIT_ASSERT(offer["options"][1]["value"].GetBoolean());
        UNIT_ASSERT(offer["options"][2]["value"].GetBoolean());
        UNIT_ASSERT(offer["options"][3]["value"].GetBoolean());
        UNIT_ASSERT(offer["options"][4]["value"].GetBoolean());
        UNIT_ASSERT(offer["options"][5]["value"].GetBoolean());
        UNIT_ASSERT(!offer["options"][6]["value"].GetBoolean());

        UNIT_ASSERT(offer["options"][2]["cost"].GetInteger() == 77);
        UNIT_ASSERT(offer["options"][4]["cost"].GetInteger() == 100);
        UNIT_ASSERT(offer["options"][5]["cost"].GetInteger() == 50);
        UNIT_ASSERT(offer["options"][6]["cost"].GetInteger() == 92);
    }

    Y_UNIT_TEST(InsuranceDetails) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOffer(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);
        auto createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("insurance",  NJson::TMapBuilder
                    ("id", "id_gold")
                    ("cost", 20)
                )
                ("status", "confirmed")
                ("currency", "czk")
            )
        );
        UNIT_ASSERT(createdOffers["offers"][0]["insurance"]["id"].GetString() == "id_gold");
        UNIT_ASSERT(createdOffers["offers"][0]["insurance"]["cost"].GetUInteger() == 20);
        createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("insurance",  NJson::TMapBuilder
                    ("id", "id_gold")
                )
                ("status", "confirmed")
                ("currency", "czk")
            )
        );

        UNIT_ASSERT(createdOffers["offers"][0]["insurance"]["cost"].IsNull());
        createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("insurance_type", "id_gold")
                ("status", "confirmed")
                ("currency", "czk")
            )
        );
        UNIT_ASSERT(createdOffers["offers"][0]["insurance"]["id"].GetString() == "id_gold");
        UNIT_ASSERT(createdOffers["offers"][0]["insurance"]["cost"].IsNull());
        createdOffers = env->Request(userId, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("status", "confirmed")
                ("currency", "czk")
            )
        );
        UNIT_ASSERT(!createdOffers["offers"][0].Has("insurance"));
        UNIT_ASSERT(!createdOffers["offers"][0].Has("insurance_type"));
    }

    Y_UNIT_TEST(CancelTripDuringTrip) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "paid")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        auto startRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + offerHolderTagId, {});
        TMap<TString, TString> headers;
        headers["UserIdDelegation"] = userId;
        NJson::TJsonValue jBody;
        jBody["session_id"] = offerId;
        jBody["tag"] = "old_state_reservation";
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tag/evolve", "", jBody, true, headers);
        }
        auto carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceRental.Seconds()) + "&until=" + ToString(untilRental.Seconds()));
        auto carsharingTimetableStr = ToString(carsharingTimetable);
        UNIT_ASSERT(carsharingTimetableStr.find("actual_since") == std::string::npos);
        UNIT_ASSERT(carsharingTimetableStr.find("actual_until") != std::string::npos);

        sinceRental = Now();

        createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("comment", "test_comment")
                ("total_payment", 1000)
                ("deposit", 200)
                ("status", "paid")
            )
        );
        offerId = createdOffers["offers"][0]["offer_id"].GetString();

        res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
        UNIT_ASSERT(offerHolderTagId);

        startRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/start", "tag_id=" + offerHolderTagId, {});
        UNIT_ASSERT(env->EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(env->EvolveTag("old_state_riding", userId));
        {
            NDrive::TServerConfigGenerator::TDisableLogging disableLogging(env.GetConfigGenerator());
            res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/tag/evolve", "", jBody, true, headers);
        }

        carsharingTimetable = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/timetable", "since=" + ToString(sinceRental.Seconds()) + "&until=" + ToString(untilRental.Seconds()));
        carsharingTimetableStr = ToString(carsharingTimetable);
        UNIT_ASSERT(carsharingTimetableStr.find("actual_since") != std::string::npos);
        UNIT_ASSERT(carsharingTimetableStr.find("actual_until") != std::string::npos);
    }

    // will be deleted after frontend update
    Y_UNIT_TEST(StartRentalOldPipeline) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterRentalOfferAll(*env.GetServer());
        env.GetServer()->GetSettings().SetValue("billing.enabled", "false", USER_ID_DEFAULT);
        env.GetServer()->GetSettings().SetValue("rental.is_old_start_riding_flow", "true", USER_ID_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<NDrivematics::TUserOrganizationAffiliationTag>(NDrivematics::TUserOrganizationAffiliationTag::TypeName), USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        auto carId = TString{OBJECT_ID_DEFAULT};
        env.GetContext().SetCar(OBJECT_IMEI_DEFAULT, carId);
        env.GetContext().InitEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = TString{USER_ID_DEFAULT};
        auto sinceRental = Now();
        auto untilRental = sinceRental + TDuration::Days(7);

        auto createdOffers = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/create", "offer_name=" + SimpleRentalOfferName + "&car_id=" + carId, NJson::TMapBuilder
            ("variables", NJson::TMapBuilder
                ("since", NJson::ToJson(sinceRental))
                ("until", NJson::ToJson(untilRental))
                ("status", "confirmed")
            )
        );
        auto createdOffer = createdOffers["offers"][0];
        auto offerId = createdOffer["offer_id"].GetString();
        UNIT_ASSERT(offerId);

        auto res = env->Request(USER_ROOT_DEFAULT, "/api/yandex/rental/book", "", NJson::TMapBuilder
            ("offer_id", offerId)
            ("client_id", userId)
        );

        TString offerHolderTagId;
        {
            UNIT_ASSERT(env->GetTagId(userId, TRentalOfferHolderTag::Type(), USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, offerHolderTagId));
            UNIT_ASSERT(offerHolderTagId);
        }

        createdOffer = env->Request(USER_ROOT_DEFAULT, "/api/yandex/sessions/history", "session_id=" + offerId + "&traits=ReportOffer", {});
        createdOffer = createdOffer["sessions"][0]["segment"]["offer"];

        createdOffer["tag_id"] = offerHolderTagId;
        createdOffer["status"] = "paid";

        auto updateRes = env->Request(USER_ROOT_DEFAULT, "/api/yandex/offers/update", "", createdOffer);
        {
            TString chargableTagId;
            UNIT_ASSERT(env->GetTagId(carId, TChargableTag::Reservation, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, chargableTagId));

            UNIT_ASSERT(chargableTagId);
            const auto& deviceTagManager = env.GetServer()->GetDriveDatabase().GetTagsManager().GetDeviceTags();
            auto tx = deviceTagManager.BuildTx<NSQL::Writable>();
            auto optionalTag = deviceTagManager.RestoreTag(chargableTagId, tx);
            UNIT_ASSERT_C(optionalTag, tx.GetStringReport());
            auto pTag = dynamic_cast<const TChargableTag*>(optionalTag->GetData().Get());
            UNIT_ASSERT(pTag->GetPerformer() == userId);
        }
    }
}
