#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/actions/disable.h>
#include <drive/backend/base/client.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/config.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <library/cpp/testing/unittest/registar.h>
#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(DriveRoles) {
    TDBAction GetActionSafe(const TVector<TDBAction>& actions, const TString& actionName) {
        for (auto&& a : actions) {
            if (a->GetName() == actionName) {
                return a;
            }
        }
        ythrow yexception() << "cannot find action " << actionName;
    }

    template <class TRole>
    TRole GetRoleSafe(const TVector<TRole>& roles, const TString& roleName) {
        for (auto&& r : roles) {
            if (r.GetRoleId() == roleName) {
                return r;
            }
        }
        ythrow yexception() << "cannot find role " << roleName;
    }

    Y_UNIT_TEST(LoginsSimple) {
        TDriveAPIConfigGenerator generator;
        TLoginsManager mLogin(generator.CreateDatabase());
        TUserLogin login(ELoginType::Telegram, "aaa");
        {
            auto session = mLogin.BuildSession();
            mLogin.RemoveLogin(login, session);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = mLogin.BuildSession();
            UNIT_ASSERT(!mLogin.AddLogin(login, session));
        }
        {
            auto session = mLogin.BuildSession();
            login.SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(mLogin.AddLogin(login, session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(mLogin.GetUserIdSafe(ELoginType::Telegram, "aaa", "asdasd", TDuration::Zero()), USER_ID_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(mLogin.GetUserIdSafe(ELoginType::Telegram, "aaa", "", TDuration::Zero()), USER_ID_DEFAULT);
        {
            auto session = mLogin.BuildSession();
            mLogin.RemoveLogin(login, session);
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(!mLogin.GetUserIdSafe(ELoginType::Telegram, "aaa", "", TDuration::Zero()));
        login.SetDeviceId("ccccc");
        {
            auto session = mLogin.BuildSession();
            UNIT_ASSERT(mLogin.AddLogin(login, session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(!mLogin.GetUserIdSafe(ELoginType::Telegram, "aaa", "asdasd", TDuration::Zero()));
        UNIT_ASSERT(!!mLogin.GetUserIdSafe(ELoginType::Telegram, "aaa", "ccccc", TDuration::Zero()));
        {
            auto session = mLogin.BuildSession();
            mLogin.RemoveLogin(login, session);
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(!mLogin.GetUserIdSafe(ELoginType::Telegram, "aaa", "ccccc", TDuration::Zero()));
    }

    void BuildActions(const TRolesManager& rm) {
        {
            auto session = rm.BuildSession();
            auto act1 = MakeHolder<TTagAction>("named_action1");
            act1->AddTagAction(TTagAction::ETagAction::Observe).SetTagName("low_voltage").SetDescription("d1");
            UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(act1), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport() << Endl);
            UNIT_ASSERT(session.Commit());
            TVector<TDBAction> actions = rm.GetActions(Now());
            auto action = GetActionSafe(actions, "named_action1");
            UNIT_ASSERT_VALUES_EQUAL(action->GetDescription(), "d1");
        }
        {
            auto session = rm.BuildSession();
            auto act1 = MakeHolder<TTagAction>("named_action1");
            act1->AddTagAction(TTagAction::ETagAction::Observe).SetTagName("low_voltage").SetDescription("d1");
            UNIT_ASSERT(rm.GetActionsDB().ForceUpsert(std::move(act1), USER_ID_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
            {
                TVector<TDBAction> actions = rm.GetActions(Now());
                auto action = GetActionSafe(actions, "named_action1");
                UNIT_ASSERT_VALUES_EQUAL(action->GetDescription(), "d1");
                UNIT_ASSERT_VALUES_EQUAL(Yensured(action.GetAs<TTagAction>())->GetTagName(), "low_voltage");
            }
        }
        {
            auto session = rm.BuildSession();
            auto act2 = MakeHolder<TTagAction>("named_action2");
            act2->AddTagAction(TTagAction::ETagAction::Observe).SetTagName("no_signal").SetDescription("d2");
            UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(act2), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport() << Endl);
            UNIT_ASSERT(session.Commit());
            {
                TVector<TDBAction> actions = rm.GetActions(Now());
                auto action0 = GetActionSafe(actions, "named_action1");
                auto action1 = GetActionSafe(actions, "named_action2");
                UNIT_ASSERT_VALUES_EQUAL(action0->GetDescription(), "d1");
                UNIT_ASSERT_VALUES_EQUAL(action1->GetDescription(), "d2");
                UNIT_ASSERT_VALUES_EQUAL(Yensured(action1.GetAs<TTagAction>())->GetTagName(), "no_signal");
            }
        }
        {
            auto session = rm.BuildSession();
            auto act2 = MakeHolder<TTagAction>("named_action2");
            act2->AddTagAction(TTagAction::ETagAction::Observe).SetTagName("$no_signal|low_voltage").SetDescription("d3");
            UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(act2), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport() << Endl);
            UNIT_ASSERT(session.Commit());
            {
                TVector<TDBAction> actions = rm.GetActions(Now());
                auto action0 = GetActionSafe(actions, "named_action1");
                auto action1 = GetActionSafe(actions, "named_action2");
                UNIT_ASSERT_VALUES_EQUAL(action0->GetDescription(), "d1");
                UNIT_ASSERT_VALUES_EQUAL(action1->GetDescription(), "d3");
            }
        }
        {
            auto session = rm.BuildSession();
            auto act2 = MakeHolder<TTagAction>("named_action3");
            act2->AddTagAction(TTagAction::ETagAction::Observe).SetTagName("no_signal").SetDescription("d2");
            UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(act2), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport() << Endl);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = rm.BuildSession();
            auto act2 = MakeHolder<TTagAction>("named_action4");
            act2->AddTagAction(TTagAction::ETagAction::Observe).SetTagName("no_signal").SetDescription("d2");
            UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(act2), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport() << Endl);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = rm.BuildSession();
            auto act2 = MakeHolder<TTagAction>("named_action5");
            act2->AddTagAction(TTagAction::ETagAction::Observe).SetTagName("no_signal").SetDescription("d2");
            UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(act2), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport() << Endl);
            UNIT_ASSERT(session.Commit());
        }
    }

    void BuildRoles(const TRolesManager& rm) {
        {
            {
                {
                    auto session = rm.BuildSession();
                    TDriveRoleHeader role;
                    role.SetDescription("r1").SetName("role1");
                    UNIT_ASSERT(rm.UpsertRole(role, USER_ROOT_DEFAULT, session));
                    UNIT_ASSERT(session.Commit());
                }
                {
                    TVector<TDriveRoleHeader> roles = rm.GetRoles(Now());
                    auto role0 = GetRoleSafe(roles, "role1");
                    UNIT_ASSERT_VALUES_EQUAL(role0.GetDescription(), "r1");
                }
            }
            {
                {
                    auto session = rm.BuildSession();
                    {
                        TDriveRoleHeader role;
                        role.SetDescription("r2").SetName("role1");
                        UNIT_ASSERT(rm.UpsertRole(role, USER_ROOT_DEFAULT, session));
                    }
                    {
                        TDriveRoleHeader role;
                        role.SetDescription("r3").SetName("role3");
                        UNIT_ASSERT(rm.UpsertRole(role, USER_ROOT_DEFAULT, session));
                    }
                    UNIT_ASSERT(session.Commit());
                }
                {
                    TVector<TDriveRoleHeader> roles = rm.GetRoles(Now());
                    auto role0 = GetRoleSafe(roles, "role1");
                    auto role1 = GetRoleSafe(roles, "role3");
                    UNIT_ASSERT_VALUES_EQUAL(role0.GetDescription(), "r2");
                    UNIT_ASSERT_VALUES_EQUAL(role1.GetDescription(), "r3");
                }
            }

            {
                {
                    auto session = rm.BuildSession();
                    TDriveRoleHeader role;
                    role.SetDescription("r2").SetName("role1");
                    UNIT_ASSERT(rm.UpsertRole(role, USER_ROOT_DEFAULT, session));
                    UNIT_ASSERT(session.Commit());
                }
                {
                    TVector<TDriveRoleHeader> roles = rm.GetRoles(Now());
                    auto role0 = GetRoleSafe(roles, "role1");
                    auto role1 = GetRoleSafe(roles, "role3");
                    UNIT_ASSERT_VALUES_EQUAL(role0.GetDescription(), "r2");
                    UNIT_ASSERT_VALUES_EQUAL(role1.GetDescription(), "r3");
                }
            }

        }
        for (ui32 i = 1; i < 5; ++i) {
            if (i == 2) {
                continue;
            }
            auto session = rm.BuildSession();
            TDriveRoleHeader role;
            role.SetDescription("r" + ToString(i)).SetName("role" + ToString(i));
            UNIT_ASSERT(rm.UpsertRole(role, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
    }

    void ClearActions(const TRolesManager& rm) {
        {
            auto session = rm.BuildSession();
            UNIT_ASSERT_C(rm.RemoveAction("named_action1", USER_ID_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.RemoveAction("named_action2", USER_ID_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT(session.Commit());
        }
        TVector<TDBAction> actions = rm.GetActions(Now());
        for (auto&& i : actions) {
            UNIT_ASSERT(i->GetName() != "named_action1" && i->GetName() != "named_action2");
        }
    }

    void ClearRoles(const TRolesManager& rm) {
        {
            auto session = rm.BuildSession();
            UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT1, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT2, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_BLOCKED, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.GetRolesInfoDB().UnlinkAllFromRole("role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.GetRolesInfoDB().UnlinkAllFromRole("role2", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.GetRolesInfoDB().UnlinkAllFromRole("role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.RemoveRole("role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.RemoveRole("role2", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT_C(rm.RemoveRole("role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetStringReport() << Endl);
            UNIT_ASSERT(session.Commit());
        }
        TVector<TDriveRoleHeader> roles = rm.GetRoles(Now());
        for (auto&& i : roles) {
            UNIT_ASSERT(i.GetName() != "role1" && i.GetName() != "role2" && i.GetName() != "role3");
        }
    }

    void ClearLandings(const TDriveAPI& api) {
        {
            auto session = api.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(session->Exec("DELETE FROM user_landings"));
            UNIT_ASSERT(session->Exec("DELETE FROM drive_landings"));
            UNIT_ASSERT(session.Commit());
        }
    }

    Y_UNIT_TEST(CheckConstantsScheme) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        UNIT_ASSERT(gServer.GetConstants(USER_ID_DEFAULT));
    }

    Y_UNIT_TEST(PropositionsReject) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TTagAction ta;
        ta.SetTagName("bbb").SetName("aaa");
        UNIT_ASSERT(!gServer.AddAction(ta, USER_ID_DEFAULT));

        {
            NJson::TJsonValue reportActions = gServer.GetActions(USER_ID_DEFAULT);
            for (auto&& i : reportActions["report"].GetArraySafe()) {
                UNIT_ASSERT(i["action_id"].GetString() != "aaa");
            }
        }

        UNIT_ASSERT(gServer.ProposeAction(ta, USER_ID_DEFAULT));
        TString propositionId;
        {
            NJson::TJsonValue reportActions = gServer.GetActions(USER_ID_DEFAULT);
            INFO_LOG << reportActions << Endl;
            for (auto&& i : reportActions["report"].GetArraySafe()) {
                if (i["action_id"].GetString() == "aaa") {
                    UNIT_ASSERT(!propositionId);
                    UNIT_ASSERT(i["is_proposition_only"].IsBoolean() && i["is_proposition_only"].GetBoolean());
                    UNIT_ASSERT(i["propositions"].IsArray());
                    UNIT_ASSERT_VALUES_EQUAL(i["propositions"].GetArraySafe().size(), 1);
                    propositionId = i["propositions"].GetArraySafe()[0]["proposition_id"].GetString();
                }
            }
            UNIT_ASSERT(!!propositionId);
        }

        UNIT_ASSERT(gServer.RejectActionProposition(propositionId, USER_ID_DEFAULT));
        {
            NJson::TJsonValue reportActions = gServer.GetActions(USER_ID_DEFAULT);
            for (auto&& i : reportActions["report"].GetArraySafe()) {
                UNIT_ASSERT(i["action_id"].GetString() != "aaa");
            }
        }
    }

    Y_UNIT_TEST(PropositionsConfirm) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TTagAction ta;
        ta.SetTagName("bbb").SetName("aaa");
        UNIT_ASSERT(!gServer.AddAction(ta, USER_ID_DEFAULT));

        {
            NJson::TJsonValue reportActions = gServer.GetActions(USER_ID_DEFAULT);
            for (auto&& i : reportActions["report"].GetArraySafe()) {
                UNIT_ASSERT(i["action_id"].GetString() != "aaa");
            }
        }

        UNIT_ASSERT(gServer.ProposeAction(ta, USER_ID_DEFAULT));
        TString propositionId;
        {
            NJson::TJsonValue reportActions = gServer.GetActions(USER_ID_DEFAULT);
            INFO_LOG << reportActions << Endl;
            for (auto&& i : reportActions["report"].GetArraySafe()) {
                if (i["action_id"].GetString() == "aaa") {
                    UNIT_ASSERT(!propositionId);
                    UNIT_ASSERT(i["is_proposition_only"].IsBoolean() && i["is_proposition_only"].GetBoolean());
                    UNIT_ASSERT(i["propositions"].IsArray());
                    UNIT_ASSERT_VALUES_EQUAL(i["propositions"].GetArraySafe().size(), 1);
                    propositionId = i["propositions"].GetArraySafe()[0]["proposition_id"].GetString();
                }
            }
            UNIT_ASSERT(!!propositionId);
        }

        UNIT_ASSERT(!gServer.ConfirmActionProposition(propositionId, USER_ID_DEFAULT));
        UNIT_ASSERT(gServer.ConfirmActionProposition(propositionId, USER_ID_DEFAULT1));
        {
            NJson::TJsonValue reportActions = gServer.GetActions(USER_ID_DEFAULT);
            INFO_LOG << reportActions << Endl;
            bool found = false;
            for (auto&& i : reportActions["report"].GetArraySafe()) {
                if (i["action_id"].GetString() == "aaa") {
                    UNIT_ASSERT(!i.Has("is_proposition_only"));
                    UNIT_ASSERT(i["propositions"].IsArray());
                    UNIT_ASSERT_VALUES_EQUAL(i["propositions"].GetArraySafe().size(), 0);
                    found = true;
                }
            }
            UNIT_ASSERT(found);
        }
    }

    Y_UNIT_TEST(RevisionsUsage) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        const TString actionName = "uniq_simple1_observe_" + ::ToString(RandomNumber<ui32>());
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            THolder<TTagAction> action(new TTagAction(actionName));
            action->SetTagName("uniq_simple1").AddTagAction(TTagAction::ETagAction::ObserveObject);
            action->SetRevision(555);
            UNIT_ASSERT(driveApi.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            THolder<TTagAction> action(new TTagAction(actionName));
            action->SetTagName("uniq_simple1").AddTagAction(TTagAction::ETagAction::ObserveObject);
            UNIT_ASSERT(!driveApi.GetRolesManager()->GetActionsDB().Upsert(action.Release(), USER_ROOT_DEFAULT, session));
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            THolder<TTagAction> action(new TTagAction(actionName));
            action->SetTagName("uniq_simple1").AddTagAction(TTagAction::ETagAction::ObserveObject);
            action->SetRevision(22223324);
            UNIT_ASSERT(!driveApi.GetRolesManager()->GetActionsDB().Upsert(action.Release(), USER_ROOT_DEFAULT, session));
        }

        {
            ui32 revision;
            {
                TMaybe<TDBAction> action = driveApi.GetRolesManager()->GetActionsDB().GetObject(actionName, Now());
                UNIT_ASSERT(action);
                revision = (*action)->GetRevision();
            }

            auto session = driveApi.BuildTx<NSQL::Writable>();
            THolder<TTagAction> action(new TTagAction(actionName));
            action->SetTagName("uniq_simple1").AddTagAction(TTagAction::ETagAction::ObserveObject);
            action->SetRevision(revision);
            UNIT_ASSERT(driveApi.GetRolesManager()->GetActionsDB().Upsert(action.Release(), USER_ROOT_DEFAULT, session));
        }
    }

    Y_UNIT_TEST(Landings) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        ClearLandings(driveApi);

        UNIT_ASSERT(gServer.ModifyLanding("landing1", "{'text':'landing1'}", true, USER_ROOT_DEFAULT));
        UNIT_ASSERT(gServer.ModifyLanding("landing2", "{'text':'<name>landing2'}", true, USER_ROOT_DEFAULT));
        UNIT_ASSERT(gServer.ModifyLanding("landing3", "{'text':'landing3'}", false, USER_ROOT_DEFAULT));
        UNIT_ASSERT(gServer.ModifyLanding("landing4", "{'text':'landing4'}", true, USER_ROOT_DEFAULT));
        UNIT_ASSERT(gServer.ModifyLanding("landing5", "{'text':'landing5'}", true, USER_ROOT_DEFAULT, {"undefined"}));
        UNIT_ASSERT(gServer.ModifyLanding("landing_tag", "{'text':'landing_tag'}", true, USER_ROOT_DEFAULT));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");

        TGeoCoord gc(37.5, 55.5);
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 2);
            for (const auto& landing : landings["landings"].GetMapSafe()) {
                UNIT_ASSERT(landing.second.GetMapSafe().contains("id") && landing.second.GetMapSafe().contains("text"));
            }
        }
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_BLOCKED, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 0);
        }
        UNIT_ASSERT(gServer.AcceptLandings({ "landing1" }, USER_ID_DEFAULT));
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 1);
            UNIT_ASSERT(landings["landings"].Has("landing2"));
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetStringRobust().find("name"), TString::npos);
        }
        UNIT_ASSERT(gServer.ClearLandings({}, USER_ID_DEFAULT));
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 2);
        }
        UNIT_ASSERT(gServer.AcceptLandings({ "landing1" }, USER_ID_DEFAULT));
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 1);
            UNIT_ASSERT(landings["landings"].Has("landing2"));
        }
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, nullptr);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 2);
            UNIT_ASSERT(landings["landings"].Has("landing2"));
            UNIT_ASSERT(landings["landings"].Has("landing5"));
        }

        THolder<TLandingUserTag> userTag(new TLandingUserTag("landing_tag"));

        UNIT_ASSERT(gServer.AddTag(userTag.Release(), USER_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, EUniquePolicy::NoUnique));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("user_tags_history");
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, nullptr);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 3);
            UNIT_ASSERT(landings["landings"].Has("landing2"));
            UNIT_ASSERT(landings["landings"].Has("landing5"));
            UNIT_ASSERT(landings["landings"].Has("landing_tag"));
        }
        UNIT_ASSERT(gServer.AcceptLandings({"landing_tag"}, USER_ID_DEFAULT));
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, nullptr);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 2);
            UNIT_ASSERT(landings["landings"].Has("landing2"));
            UNIT_ASSERT(landings["landings"].Has("landing5"));
        }

        UNIT_ASSERT(!gServer.RemoveLandings({ "landing1" }, USER_ROOT_DEFAULT));
        UNIT_ASSERT(gServer.ClearLandingsById({}, USER_ID_DEFAULT, USER_ROOT_DEFAULT));
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, &gc);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetMapSafe().size(), 2);
        }
        UNIT_ASSERT(gServer.RemoveLandings({ "landing1" }, USER_ROOT_DEFAULT));
        {
            NJson::TJsonValue landings = gServer.GetLandingsById({}, USER_ROOT_DEFAULT);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetArraySafe().size(), 5);
        }
        UNIT_ASSERT(gServer.ModifyLanding("landing6", "{'text':'landing6'}", true, USER_ROOT_DEFAULT));
        {
            NJson::TJsonValue landings = gServer.GetLandingsById({}, USER_ROOT_DEFAULT);
            INFO_LOG << landings << Endl;
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"].GetArraySafe().size(), 6);
        }
    }

    Y_UNIT_TEST(RolesCleaning) {
        TFakeHistoryContext context;
        TUsersDB userDB(context);
        TRolesManager rm(context, userDB, TPropositionsManagerConfig(), THistoryConfig());
        rm.Start();
        ClearRoles(rm);
        BuildRoles(rm);
        {
            auto session = rm.BuildSession();
            {
                TUserRole roleInfo;
                roleInfo.SetUserId(USER_ID_DEFAULT);
                roleInfo.SetRoleId("role1");
                roleInfo.SetDeadline(TInstant::Seconds(10000000));
                UNIT_ASSERT(rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session));
            }
            UNIT_ASSERT_C(!rm.RemoveRole("role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
        }
        {
            auto session = rm.BuildSession();
            {
                TUserRole roleInfo;
                roleInfo.SetUserId(USER_ID_DEFAULT);
                roleInfo.SetRoleId("role1");
                roleInfo.SetDeadline(TInstant::Seconds(10000000));
                UNIT_ASSERT_C(rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetReport());
            }
            {
                TUserRole roleInfo;
                roleInfo.SetUserId(USER_ID_DEFAULT1);
                roleInfo.SetRoleId("role1");
                roleInfo.SetDeadline(TInstant::Seconds(3000000000));
                UNIT_ASSERT(rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session));
            }
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = rm.BuildSession();
            UNIT_ASSERT_C(!rm.RemoveRole("role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
        }

        {
            auto session = rm.BuildSession();
            TInstantGuard ig(TInstant::Seconds(3000000001));
            UNIT_ASSERT(rm.RemoveRoleForUser(USER_ID_DEFAULT1, "role1", USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(rm.RemoveRoleForUser(USER_ID_DEFAULT, "role1", USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(rm.RemoveRole("role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT(session.Commit());
        }
        ClearRoles(rm);
    };

    Y_UNIT_TEST(PermissionsSimple) {
        TDriveAPIConfigGenerator generator;
        TRolesConfig rolesCfg;
        TFakeHistoryContext context;
        TUsersDB userDB(context);
        TRolesManager rm(context, userDB, TPropositionsManagerConfig(), THistoryConfig());
        rm.Start();
        auto config = generator.Generate();
        TDriveTagsManager tagsManager(userDB.GetDatabase(), *config);
        {
            {
                auto session = rm.BuildSession();
                UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT1, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT1, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action2", "role2", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action2", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            ClearRoles(rm);
            ClearActions(rm);
        }
        BuildActions(rm);
        BuildRoles(rm);

        {
            auto session = rm.BuildSession();
            UNIT_ASSERT_C(!rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action2", "role2"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
        }

        {
            auto session = rm.BuildSession();
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action1", "role1"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action1", "role3"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action2", "role3"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT(session.Commit());
            TVector<TDriveRoleActions> roles = rm.GetRolesInfoDB().GetRoleActions().GetInfo(Now());
            TDriveRoleActions role1 = GetRoleSafe(roles, "role1");
            TDriveRoleActions role3 = GetRoleSafe(roles, "role3");
            UNIT_ASSERT_VALUES_EQUAL(role1.GetSlaves().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(role3.GetSlaves().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(role1.GetRoleId(), "role1");
            UNIT_ASSERT_VALUES_EQUAL(role3.GetRoleId(), "role3");
            UNIT_ASSERT_VALUES_EQUAL(role1.GetSlaves()[0].GetSlaveObjectId(), "named_action1");
            UNIT_ASSERT_VALUES_EQUAL(role3.GetSlaves()[0].GetSlaveObjectId(), "named_action1");
            UNIT_ASSERT_VALUES_EQUAL(role3.GetSlaves()[1].GetSlaveObjectId(), "named_action2");
        }

        {
            auto session = rm.BuildSession();
            TUserRole roleInfo;
            roleInfo.SetUserId(USER_ID_DEFAULT1);
            roleInfo.SetRoleId("role3");
            roleInfo.SetDeadline(TInstant::Seconds(2000000000));
            rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT1, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
            UNIT_ASSERT(!!perm);
            auto tags = perm->GetTagNamesByAction(TTagAction::ETagAction::Observe);
            INFO_LOG << JoinSeq(", ", tags) << Endl;
            UNIT_ASSERT(tags.contains("low_voltage"));
            UNIT_ASSERT(tags.contains("no_signal"));
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action1");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[1]->GetName(), "named_action2");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetDeadline(), TInstant::Seconds(2000000000));
        }

        {
            auto session = rm.BuildSession();
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action2", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT(session.Commit());
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT1, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
            UNIT_ASSERT(!!perm);
            {
                auto tags = perm->GetTagNamesByAction(TTagAction::ETagAction::Perform);
                INFO_LOG << JoinSeq(", ", tags) << Endl;
                UNIT_ASSERT(tags.empty());
            }
            {
                auto tags = perm->GetTagNamesByAction(TTagAction::ETagAction::Observe);
                INFO_LOG << JoinSeq(", ", tags) << Endl;
                UNIT_ASSERT(tags.contains("low_voltage"));
                UNIT_ASSERT(!tags.contains("no_signal"));
            }
            {
                auto session = rm.BuildSession();
                auto act2 = MakeHolder<TTagAction>("named_action1");
                act2->AddTagAction(TTagAction::ETagAction::Perform).SetTagName("$no_signal|low_voltage").SetDescription("d3");
                UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(act2), USER_ID_DEFAULT1, session), TStringBuilder() << session.GetReport() << Endl);
                UNIT_ASSERT(session.Commit());
            }
            perm = rm.GetUserPermissions(USER_ID_DEFAULT1, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
            UNIT_ASSERT(!!perm);
            {
                auto tags = perm->GetTagNamesByAction(TTagAction::ETagAction::Perform);
                INFO_LOG << JoinSeq(", ", tags) << Endl;
                UNIT_ASSERT(tags.contains("low_voltage"));
                UNIT_ASSERT(tags.contains("no_signal"));
            }
            {
                auto tags = perm->GetTagNamesByAction(TTagAction::ETagAction::Observe);
                INFO_LOG << JoinSeq(", ", tags) << Endl;
                UNIT_ASSERT(tags.empty());
            }
        }

        {
            {
                auto session = rm.BuildSession();
                UNIT_ASSERT_C(rm.RemoveRoleForUser(USER_ID_DEFAULT1, "role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.RemoveRoleForUser(USER_ID_DEFAULT1, "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action2", "role2", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action2", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT(session.Commit());
            }

            ClearRoles(rm);
            ClearActions(rm);
        }
    }

    Y_UNIT_TEST(PermissionsByTime) {
        TDriveAPIConfigGenerator generator;
        TRolesConfig rolesCfg;
        TFakeHistoryContext context;
        TUsersDB userDB(context);
        TRolesManager rm(context, userDB, TPropositionsManagerConfig(), THistoryConfig());
        rm.Start();
        auto config = generator.Generate();
        TDriveTagsManager tagsManager(context.GetDatabase(), *config);
        TInstantGuard ig(TInstant::Zero());
        ClearRoles(rm);
        ClearActions(rm);
        BuildActions(rm);
        BuildRoles(rm);
        {
            {
                auto session = rm.BuildSession();
                TTimeRestriction restriction;
                restriction.SetTimeRestriction(2300, 400);
                {
                    TLinkedRoleActionHeader linkInfo("named_action1", "role1");
                    linkInfo.SetTimeRestriction(restriction);
                    UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(linkInfo, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                }
                {
                    TLinkedRoleActionHeader linkInfo("named_action1", "role3");
                    linkInfo.SetTimeRestriction(restriction);
                    UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(linkInfo, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                }
                {
                    TLinkedRoleActionHeader linkInfo("named_action2", "role3");
                    linkInfo.SetTimeRestriction(restriction);
                    UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(linkInfo, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                }
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = rm.BuildSession();
                TUserRole roleInfo;
                roleInfo.SetUserId(USER_ID_DEFAULT2);
                roleInfo.SetRoleId("role3");
                roleInfo.SetDeadline(TInstant::Seconds(2000000000));
                rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session);
                UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            }
            {
                ig.Set(TInstant::Zero() + TDuration::Days(1));
                TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT2, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
                UNIT_ASSERT(!!perm);
                auto tags = perm->GetTagNamesByAction(TTagAction::ETagAction::Observe);
                INFO_LOG << JoinSeq(", ", tags) << Endl;
                UNIT_ASSERT(tags.contains("low_voltage"));
                UNIT_ASSERT(tags.contains("no_signal"));
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 2);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action1");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[1]->GetName(), "named_action2");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetDeadline(), TInstant::Seconds(2000000000));
            }
            {
                ig.Set(TInstant::Seconds(1) + TDuration::Days(1) + TDuration::Hours(4));
                TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT2, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
                UNIT_ASSERT(!!perm);
                auto tags = perm->GetTagNamesByAction(TTagAction::ETagAction::Observe);
                INFO_LOG << JoinSeq(", ", tags) << Endl;
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
                UNIT_ASSERT(perm->GetActionsActual().empty());
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetDeadline(), TInstant::Seconds(2000000000));
            }
        }
        {
            {
                auto session = rm.BuildSession();
                UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT2, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT2, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action2", "role2", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action2", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT(session.Commit());
            }

            ClearActions(rm);
            ClearRoles(rm);
        }
    }

    Y_UNIT_TEST(PermissionsByHierarchyRoles) {
        TDriveAPIConfigGenerator generator;
        TFakeHistoryContext context;
        TUsersDB userDB(context);
        TRolesManager rm(context, userDB, TPropositionsManagerConfig(), THistoryConfig());
        rm.Start();
        auto config = generator.Generate();
        TDriveTagsManager tagsManager(context.GetDatabase(), *config);
        {
            {
                auto session = rm.BuildSession();
                UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT1, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action4", "role4", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action3", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Unlink("role4", "role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Unlink("role3", "role4", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Unlink("role4", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            ClearActions(rm);
            ClearRoles(rm);
        }
        BuildActions(rm);
        BuildRoles(rm);

        {
            auto session = rm.BuildSession();
            TUserRole roleInfo;
            roleInfo.SetUserId(USER_ID_DEFAULT1);
            roleInfo.SetRoleId("role1");
            roleInfo.SetDeadline(TInstant::Seconds(2000000000));
            rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action1", "role1"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action4", "role4"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action3", "role3"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("role4", "role1"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("role3", "role4"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("role4", "role3"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
        }
        bool found = false;
        const TInstant startInstant = Now();
        while (Now() - startInstant < TDuration::Seconds(10) && !found) {
            TRolesConfig rolesCfg;
            rolesCfg.SetDefaultRoles({"role1"});
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT2, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now() - TDuration::Minutes(10));
            UNIT_ASSERT(!!perm);
            if (perm->GetActionsActual().size() == 3 && perm->GetRolesByUser().size() == 1) {
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 3);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action1");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[1]->GetName(), "named_action3");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[2]->GetName(), "named_action4");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetRoleId(), "role1");
                found = true;
            } else {
                Sleep(TDuration::Seconds(1));
            }
        }
        UNIT_ASSERT(found);
        {
            TRolesConfig rolesCfg;
            rolesCfg.SetDefaultRoles({"role4"});
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT2, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now() - TDuration::Minutes(10));
            UNIT_ASSERT(!!perm);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action3");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[1]->GetName(), "named_action4");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetRoleId(), "role4");
        }
        {
            TRolesConfig rolesCfg;
            rolesCfg.SetDefaultRoles({"role3"});
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT2, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now() - TDuration::Minutes(10));
            UNIT_ASSERT(!!perm);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action3");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[1]->GetName(), "named_action4");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetRoleId(), "role3");
        }
    }

    Y_UNIT_TEST(PermissionsByExternalRoles) {
        TDriveAPIConfigGenerator generator;
        TFakeHistoryContext context;
        TUsersDB userDB(context);
        TRolesManager rm(context, userDB, TPropositionsManagerConfig(), THistoryConfig());
        rm.Start();
        auto config = generator.Generate();
        TDriveTagsManager tagsManager(context.GetDatabase(), *config);
        {
            {
                auto session = rm.BuildSession();
                UNIT_ASSERT_C(rm.RemoveAllRolesForUser(USER_ID_DEFAULT1, USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action1", "role1", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Unlink("named_action3", "role3", USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
                UNIT_ASSERT(session.Commit());
            }
            ClearActions(rm);
            ClearRoles(rm);
        }
        BuildActions(rm);
        BuildRoles(rm);

        {
            auto session = rm.BuildSession();
            TUserRole roleInfo;
            roleInfo.SetUserId(USER_ID_DEFAULT1);
            roleInfo.SetRoleId("role1");
            roleInfo.SetDeadline(TInstant::Seconds(2000000000));
            rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action1", "role1"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action3", "role3"), USER_ROOT_DEFAULT, session), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
        }

        {
            auto session = rm.BuildSession();
            TUserRole roleInfo;
            roleInfo.SetUserId(USER_ID_BLOCKED);
            roleInfo.SetRoleId("role1");
            roleInfo.SetDeadline(TInstant::Seconds(2000000000));
            rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session);
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetTransaction()->GetErrors().GetStringReport());
        }
        TInstant startInstant = Now();
        bool found = false;
        while (Now() - startInstant < TDuration::Seconds(10) && !found) {
            // Test default role apply ( role2 not exists)
            TRolesConfig rolesCfg;
            rolesCfg.SetDefaultRoles({"role3", "role2"});
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT2, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now() - TDuration::Minutes(10));
            UNIT_ASSERT(!!perm);
            for (auto&& i : perm->GetActionsActual()) {
                INFO_LOG << i->GetName() << Endl;
            }
            if (perm->GetActionsActual().size() == 1 && perm->GetRolesByUser().size() == 1) {
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action3");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetRoleId(), "role3");
                found = true;
            } else {
                Sleep(TDuration::MilliSeconds(200));
            }
        }
        UNIT_ASSERT(found);
        {
            auto session = rm.BuildSession();
            auto onboardingUserId = userDB.RegisterNewUser("unittest", "8623622312", "skulik-was-there", session)->GetUserId();
            UNIT_ASSERT(session.Commit());

            // Test base role apply
            TRolesConfig rolesCfg;
            TUserPermissions::TPtr perm = rm.GetUserPermissions(onboardingUserId, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now() - TDuration::Minutes(0));
            UNIT_ASSERT(!!perm);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 0);

            rolesCfg.SetBaseRoles({ "role3" });
            perm = rm.GetUserPermissions(onboardingUserId, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
            UNIT_ASSERT(!!perm);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser()[0].GetRoleId(), "role3");
        }
        {
            // Test default role ignore
            TRolesConfig rolesCfg;
            rolesCfg.SetDefaultRoles({"role3"});
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT1, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
            UNIT_ASSERT(!!perm);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action1");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
        }
        {
            // Test external role apply
            TRolesConfig rolesCfg;
            rolesCfg.SetPlusRoles({"role3"});
            rolesCfg.SetDefaultRoles({"role1"});
            TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_DEFAULT1, tagsManager, rolesCfg, TUserPermissionsFeatures().SetIsPlusUser(true), Now() - TDuration::Minutes(10));
            UNIT_ASSERT(!!perm);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action1");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[1]->GetName(), "named_action3");
            UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 2);
        }
        {
            // Test blocked role apply
            TRolesConfig rolesCfg;
            rolesCfg.SetBlockedRoles({"role3"});
            {
                auto session = rm.BuildSession();
                auto user = userDB.FetchInfo(USER_ID_BLOCKED).GetResult().begin()->second;
                user.SetStatus("blocked");
                userDB.UpdateUser(user, "unittest", session);
                UNIT_ASSERT(session.Commit());
                TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_BLOCKED, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
                UNIT_ASSERT(!!perm);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action3");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
            }

            {
                auto session = rm.BuildSession();
                auto user = userDB.FetchInfo(USER_ID_BLOCKED).GetResult().begin()->second;
                user.SetStatus("active");
                userDB.UpdateUser(user, "unittest", session);
                UNIT_ASSERT(session.Commit());
                TUserPermissions::TPtr perm = rm.GetUserPermissions(USER_ID_BLOCKED, tagsManager, rolesCfg, TUserPermissionsFeatures(), Now());
                UNIT_ASSERT(!!perm);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(perm->GetActionsActual()[0]->GetName(), "named_action1");
                UNIT_ASSERT_VALUES_EQUAL(perm->GetRolesByUser().size(), 1);
            }
        }

    }

    Y_UNIT_TEST(ActionsHierarchy) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        const TRolesManager& rm = *driveApi.GetRolesManager();
        {
            auto mbActionParent = driveApi.GetRolesManager()->GetAction("fixpoint_offer_constructor", Now());
            auto mbActionChild = driveApi.GetRolesManager()->GetAction("fixpoint_offer_constructor_child", Now());
            TDBAction actionParent = *mbActionParent;
            TDBAction actionChild = *mbActionChild;
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionParent.GetAs<TFixPointOfferConstructor>())->GetRerunPriceKM(), 800);
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionChild.GetAs<TFixPointOfferConstructor>())->GetRerunPriceKM(), 900);
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionParent.GetAs<TFixPointOfferConstructor>())->GetReturningDuration(), TDuration::Days(100));
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionChild.GetAs<TFixPointOfferConstructor>())->GetReturningDuration(), TDuration::Days(100));
            {
                auto session = rm.BuildSession();
                THolder<TFixPointOfferConstructor> actionParentNew(new TFixPointOfferConstructor(*Yensured(actionParent.GetAs<TFixPointOfferConstructor>())));
                actionParentNew->SetRerunPriceKM(1000);
                actionParentNew->SetReturningDuration(TDuration::Days(200));
                UNIT_ASSERT(rm.GetActionsDB().ForceUpsert(actionParentNew.Release(), USER_ROOT_DEFAULT, session) && session.Commit());
            }
        }

        {
            auto mbActionParent = driveApi.GetRolesManager()->GetAction("fixpoint_offer_constructor", Now());
            auto mbActionChild = driveApi.GetRolesManager()->GetAction("fixpoint_offer_constructor_child", Now());
            TDBAction actionParent = *mbActionParent;
            TDBAction actionChild = *mbActionChild;
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionParent.GetAs<TFixPointOfferConstructor>())->GetRerunPriceKM(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionChild.GetAs<TFixPointOfferConstructor>())->GetRerunPriceKM(), 900);
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionParent.GetAs<TFixPointOfferConstructor>())->GetReturningDuration(), TDuration::Days(200));
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionChild.GetAs<TFixPointOfferConstructor>())->GetReturningDuration(), TDuration::Days(200));
            {
                auto session = rm.BuildSession();
                THolder<TFixPointOfferConstructor> actionChildNew(new TFixPointOfferConstructor(*Yensured(actionChild.GetAs<TFixPointOfferConstructor>())));
                actionChildNew->SetRerunPriceKM(1000);
                actionChildNew->SetReturningDuration(TDuration::Days(300));
                UNIT_ASSERT(rm.GetActionsDB().ForceUpsert(actionChildNew.Release(), USER_ROOT_DEFAULT, session) && session.Commit());
            }
        }

        {
            auto mbActionParent = driveApi.GetRolesManager()->GetAction("fixpoint_offer_constructor", Now());
            auto mbActionChild = driveApi.GetRolesManager()->GetAction("fixpoint_offer_constructor_child", Now());
            TDBAction actionParent = *mbActionParent;
            TDBAction actionChild = *mbActionChild;
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionParent.GetAs<TFixPointOfferConstructor>())->GetRerunPriceKM(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionChild.GetAs<TFixPointOfferConstructor>())->GetRerunPriceKM(), 1000);
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionParent.GetAs<TFixPointOfferConstructor>())->GetReturningDuration(), TDuration::Days(200));
            UNIT_ASSERT_VALUES_EQUAL(Yensured(actionChild.GetAs<TFixPointOfferConstructor>())->GetReturningDuration(), TDuration::Days(300));
        }
    }

    Y_UNIT_TEST(RolesInGroup) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        {
            TDriveRoleHeader role;
            role.SetDescription("r_A").SetName("role_A").SetGroup(1).SetIsPublic(true).SetOptional(true);
            UNIT_ASSERT(configGenerator.RegisterRole(role, USER_ROOT_DEFAULT));

            role.SetDescription("r_B").SetName("role_B").SetGroup(1).SetIsPublic(true).SetOptional(true);
            UNIT_ASSERT(configGenerator.RegisterRole(role, USER_ROOT_DEFAULT));

            role.SetDescription("r_C").SetName("role_C").SetGroup(1).SetIsPublic(true).SetOptional(true);
            UNIT_ASSERT(configGenerator.RegisterRole(role, USER_ROOT_DEFAULT));
        }
        NJson::TJsonValue report;

        TUserRole uRole;
        uRole.SetUserId(USER_ID_DEFAULT).SetRoleId("role_A").SetActive(true);
        UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", true));

        uRole.SetUserId(USER_ID_DEFAULT).SetRoleId("role_B").SetActive(true);
        UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", true));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", false));

        uRole.SetUserId(USER_ID_DEFAULT).SetRoleId("role_C").SetActive(true);
        UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", true));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_C", false));

        UNIT_ASSERT(configGenerator.RoleActivation("role_A", true, USER_ID_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", true));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_C", false));

        UNIT_ASSERT(configGenerator.RoleActivation("role_B", true, USER_ID_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", true));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_C", false));

        UNIT_ASSERT(configGenerator.RoleActivation("role_B", false, USER_ID_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_C", false));

        uRole.SetUserId(USER_ID_DEFAULT).SetRoleId("role_C").SetActive(true);
        UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_C", true));

        uRole.SetUserId(USER_ID_DEFAULT).SetRoleId("role_C").SetActive(false);
        UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_C", false));

        UNIT_ASSERT(configGenerator.RoleActivation("role_A", true, USER_ID_DEFAULT));
        report = configGenerator.GetUserRoles(USER_ID_DEFAULT, USER_ROOT_DEFAULT);
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_A", true));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_B", false));
        UNIT_ASSERT(TValidationHelpers::CheckIsActiveRole(report, "role_C", false));
    }

    Y_UNIT_TEST(StaffOnboardingRoles) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        {
            TUserRole userRole;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            userRole.SetRoleId("root").SetUserId(userId);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }

        UNIT_ASSERT_VALUES_EQUAL(gServer.GetUserAdminInfo(USER_ROOT_DEFAULT, userId), NJson::JSON_NULL);

        {
            auto userFetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
            UNIT_ASSERT_VALUES_EQUAL(userFetchResult.size(), 1);
            auto user = userFetchResult.begin()->second;

            TString newUid = "1120000000" + ToString(RandomNumber<ui64>(899999) + 100000);
            user.SetUid(newUid);
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->UpdateUser(user, "robot-frontend", session) && session.Commit());
            server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
        }
        UNIT_ASSERT(gServer.GetUserAdminInfo(USER_ROOT_DEFAULT, userId) != NJson::JSON_NULL);
    }

    Y_UNIT_TEST(PossibleAssigneesSearch) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        const ui32 countStartMajor = driveApi.GetRolesManager()->GetPotentialTagAssignees("user_problem_tag_major", Now()).size();
        const ui32 countStartMinor = driveApi.GetRolesManager()->GetPotentialTagAssignees("user_problem_tag_minor", Now()).size();

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }

        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("user_problem_tag_minor", Now()).size(), countStartMinor + 1);
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("user_problem_tag_major", Now()).size(), countStartMajor + 1);

        auto usersReport = gServer.GetTagOperators("user_problem_tag_major", TTagAction::ETagAction::Perform);
        UNIT_ASSERT_C(usersReport.first >= 1, TStringBuilder() << usersReport.first);
        UNIT_ASSERT_VALUES_EQUAL(usersReport.second.size(), usersReport.first);

        bool isUserIdFound = false;
        for (auto&& operatorId : usersReport.second) {
            if (userId == operatorId) {
                isUserIdFound = true;
                break;
            }
        }

        UNIT_ASSERT(isUserIdFound);
    }

    Y_UNIT_TEST(DisabledActions) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const TRolesManager& rm = *driveApi.GetRolesManager();
        BuildActions(rm);
        BuildRoles(rm);
        {
            TLinkedRoleRoleHeader link;
            link.SetRoleId("default_user");
            link.SetSlaveObjectId("role3");
            auto session = driveApi.BuildSession(false, false);
            UNIT_ASSERT(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("named_action2", "role3"), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(rm.GetRolesInfoDB().GetRoleRoles().Link(link, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        auto userId = eGenerator.CreateUser("svshevtsov-was-here", true, "active");
        {
            auto permissions = driveApi.GetUserPermissions(userId, {}, Now());
            UNIT_ASSERT(permissions);
            TSet<TString> actionIds;
            for (auto&& action : permissions->GetActionsActual()) {
                actionIds.insert(action->GetName());
            }
            UNIT_ASSERT(actionIds.contains("named_action2"));
        }
        {
            auto session = driveApi.BuildSession(false, false);
            auto action = MakeAtomicShared<TDisableAction>("disable_named_action2");
            action->SetActions({ "named_action2" });
            UNIT_ASSERT(rm.GetActionsDB().ForceUpsert(action, USER_ROOT_DEFAULT, session));

            TDriveRoleHeader role;
            role.SetDescription("disable_named_action2").SetName("disable_named_action2");
            UNIT_ASSERT(rm.UpsertRole(role, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("disable_named_action2", "disable_named_action2"), USER_ROOT_DEFAULT, session));

            TUserRole roleInfo;
            roleInfo.SetUserId(userId);
            roleInfo.SetRoleId("disable_named_action2");
            UNIT_ASSERT(rm.UpsertRoleForUser(roleInfo, USER_ROOT_DEFAULT, session));

            UNIT_ASSERT(session.Commit());
        }
        {
            auto permissions = driveApi.GetUserPermissions(userId, {}, Now());
            UNIT_ASSERT(permissions);
            TSet<TString> actionIds;
            for (auto&& action : permissions->GetActionsActual()) {
                actionIds.insert(action->GetName());
            }
            UNIT_ASSERT(!actionIds.contains("named_action2"));
        }
    }

    Y_UNIT_TEST(UsersWithRoles) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        ui32 oldResult1Size;
        {
            TVector<TString> oldResult1;
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user"}, oldResult1, true));
            oldResult1Size = oldResult1.size();
        }
        ui32 oldResult2Size;
        {
            TVector<TString> oldResult2;
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user", "standart_access"}, oldResult2, true));
            oldResult2Size = oldResult2.size();
        }
        ui32 oldResult3Size;
        {
            TVector<TString> oldResult3;
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user"}, oldResult3, false));
            oldResult3Size = oldResult3.size();
        }
        ui32 oldResult4Size;
        {
            TVector<TString> oldResult4;
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user", "standart_access"}, oldResult4, false));
            oldResult4Size = oldResult4.size();
        }
        ui32 oldResult5Size;
        {
            TVector<TString> oldResult5;
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"root"}, oldResult5, false));
            oldResult5Size = oldResult5.size();
        }

        auto userId = eGenerator.CreateUser("user-roletest-1", false, "onboarding");
        auto userId2 = eGenerator.CreateUser("user-roletest-2", false, "active");
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

        TUserRole userRole;
        userRole.SetRoleId("default_user").SetUserId(userId).SetActive(true);
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session));

        TUserRole userRole2;
        userRole2.SetRoleId("standart_access").SetUserId(userId).SetActive(true);
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole2, userId, session));

        TUserRole userRole3;
        userRole3.SetRoleId("default_user").SetUserId(userId2).SetActive(true);
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole3, userId2, session));

        TUserRole userRole4;
        userRole4.SetRoleId("standart_access").SetUserId(userId2).SetActive(false);
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole4, userId2, session));

        UNIT_ASSERT(session.Commit());

        TVector<TString> newResult1;
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user"}, newResult1, true));
        TVector<TString> newResult2;
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user", "standart_access"}, newResult2, true));
        TVector<TString> newResult3;
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user"}, newResult3, false));
        TVector<TString> newResult4;
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"default_user", "standart_access"}, newResult4, false));
        TVector<TString> newResult5;
        UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().GetUsersWithRoles({"root"}, newResult5, false));

        UNIT_ASSERT(newResult1.size() - oldResult1Size == 2
            && std::find(newResult1.begin(), newResult1.end(), userId) != newResult1.end()
            && std::find(newResult1.begin(), newResult1.end(), userId2) != newResult1.end());

        UNIT_ASSERT(newResult2.size() - oldResult2Size == 1
                        && std::find(newResult2.begin(), newResult2.end(), userId) != newResult2.end()
                        && std::find(newResult2.begin(), newResult2.end(), userId2) == newResult2.end());

        UNIT_ASSERT(newResult3.size() - oldResult3Size == 2
                        && std::find(newResult3.begin(), newResult3.end(), userId) != newResult3.end()
                        && std::find(newResult3.begin(), newResult3.end(), userId2) != newResult3.end());

        UNIT_ASSERT(newResult4.size() - oldResult4Size == 2
                        && std::find(newResult4.begin(), newResult4.end(), userId) != newResult4.end()
                        && std::find(newResult4.begin(), newResult4.end(), userId2) != newResult4.end());

        UNIT_ASSERT(newResult5.size() - oldResult5Size == 0
                        && std::find(newResult5.begin(), newResult5.end(), userId) == newResult5.end()
                        && std::find(newResult5.begin(), newResult5.end(), userId2) == newResult5.end());
    }

    // SetupRoleTree setups following role tree:
    // 1) test_role_1 -> test_role_2 -> test_role_3
    //                -> test_role_4
    //
    // 2) test_role_5
    //
    // 3) test_role_6 -> test_role_1
    //                -> test_role_2
    //                -> test_role_3
    //                -> test_role_4
    //                -> test_role_5
    void SetupRoleTree(const TRolesManager& rm) {
        // Create actions.
        {
            auto session = rm.BuildSession();
            for (int i = 1; i <= 6; i++) {
                auto action = MakeHolder<TTagAction>("test_action_" + ToString(i));
                action->AddTagAction(TTagAction::ETagAction::Observe);
                action->SetTagName("test_tag_" + ToString(i));
                action->SetDescription("test_action_" + ToString(i));
                UNIT_ASSERT_C(rm.GetActionsDB().ForceUpsert(std::move(action), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            }
            UNIT_ASSERT(session.Commit());
        }
        // Create roles.
        {
            auto session = rm.BuildSession();
            for (int i = 1; i <= 6; i++) {
                TDriveRoleHeader role;
                role.SetName("test_role_" + ToString(i)).SetDescription("test_role_" + ToString(i));
                UNIT_ASSERT_C(rm.UpsertRole(role, USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            }
            UNIT_ASSERT(session.Commit());
        }
        // Attach actions to roles.
        {
            auto session = rm.BuildSession();
            for (int i = 1; i <= 6; i++) {
                UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleActions().Link(TLinkedRoleActionHeader("test_action_" + ToString(i), "test_role_" + ToString(i)), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            }
            UNIT_ASSERT(session.Commit());
        }
        // Attach roles to roles.
        {
            auto session = rm.BuildSession();
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_2", "test_role_1"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_3", "test_role_2"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_4", "test_role_1"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_1", "test_role_6"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_2", "test_role_6"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_3", "test_role_6"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_4", "test_role_6"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT_C(rm.GetRolesInfoDB().GetRoleRoles().Link(TLinkedRoleRoleHeader("test_role_5", "test_role_6"), USER_ROOT_DEFAULT, session), session.GetTransaction()->GetErrors().GetStringReport());
            UNIT_ASSERT(session.Commit());
        }
        // Check all roles and actions.
        {
            TVector<TDBAction> actions = rm.GetActions(Now());
            for (int i = 1; i <= 6; i++) {
                auto action = GetActionSafe(actions, "test_action_" + ToString(i));
                UNIT_ASSERT_VALUES_EQUAL(action->GetDescription(), "test_action_" + ToString(i));
            }
        }
        {
            TVector<TDriveRoleHeader> roles = rm.GetRoles(Now());
            for (int i = 1; i <= 6; i++) {
                auto role = GetRoleSafe(roles, "test_role_" + ToString(i));
                UNIT_ASSERT_VALUES_EQUAL(role.GetDescription(), "test_role_" + ToString(i));
            }
        }
    }

    Y_UNIT_TEST(RecursiveDisabledRoles) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetSensorApiName({});
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const TRolesManager& rm = *driveApi.GetRolesManager();
        const IDriveTagsManager& tm = driveApi.GetTagsManager();
        SetupRoleTree(rm);
        {
            auto userId = eGenerator.CreateUser("test-user-1", true, "active");
            TUserRole uRole;
            uRole.SetUserId(userId).SetRoleId("test_role_1").SetActive(true);
            UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
            TRolesConfig rolesCfg;
            {
                auto replyCtx = MakeAtomicShared<TReplyContextMock>();
                replyCtx->MutableRequestData().AddHeader(
                    "X-Yandex-ExpFlags",
                    // "recursive_disabled_roles": [ "test_role_1" ].
                    "W3siSEFORExFUiI6IkRSSVZFX01PQklMRSIsIkNPTlRFWFQiOnsiRFJJVkVfTU9CSUxFIjp7InJlY3Vyc2l2ZV9kaXNhYmxlZF9y"
                    "b2xlcyI6WyJ0ZXN0X3JvbGVfMSJdfX19XQ=="
                );
                TUserPermissions::TPtr perm = rm.GetUserPermissions(userId, tm, rolesCfg, TUserPermissionsFeatures(), Now(), replyCtx);
                UNIT_ASSERT(perm);
                UNIT_ASSERT(!perm->HasAction("test_action_1"));
                UNIT_ASSERT(!perm->HasAction("test_action_2"));
                UNIT_ASSERT(!perm->HasAction("test_action_3"));
                UNIT_ASSERT(!perm->HasAction("test_action_4"));
                UNIT_ASSERT(!perm->HasAction("test_action_5"));
                UNIT_ASSERT(!perm->HasAction("test_action_6"));
            }
            {
                auto replyCtx = MakeAtomicShared<TReplyContextMock>();
                replyCtx->MutableRequestData().AddHeader(
                    "X-Yandex-ExpFlags",
                    // "recursive_disabled_roles": [ "test_role_2" ].
                    "W3siSEFORExFUiI6IkRSSVZFX01PQklMRSIsIkNPTlRFWFQiOnsiRFJJVkVfTU9CSUxFIjp7InJlY3Vyc2l2ZV9kaXNhYmxlZF9y"
                    "b2xlcyI6WyJ0ZXN0X3JvbGVfMiJdfX19XQ=="
                );
                TUserPermissions::TPtr perm = rm.GetUserPermissions(userId, tm, rolesCfg, TUserPermissionsFeatures(), Now(), replyCtx);
                UNIT_ASSERT(perm);
                UNIT_ASSERT(perm->HasAction("test_action_1"));
                UNIT_ASSERT(!perm->HasAction("test_action_2"));
                UNIT_ASSERT(!perm->HasAction("test_action_3"));
                UNIT_ASSERT(perm->HasAction("test_action_4"));
                UNIT_ASSERT(!perm->HasAction("test_action_5"));
                UNIT_ASSERT(!perm->HasAction("test_action_6"));
            }
            {
                auto replyCtx = MakeAtomicShared<TReplyContextMock>();
                replyCtx->MutableRequestData().AddHeader(
                    "X-Yandex-ExpFlags",
                    // "recursive_disabled_roles": [ "test_role_5" ].
                    "W3siSEFORExFUiI6IkRSSVZFX01PQklMRSIsIkNPTlRFWFQiOnsiRFJJVkVfTU9CSUxFIjp7InJlY3Vyc2l2ZV9kaXNhYmxlZF9y"
                    "b2xlcyI6WyJ0ZXN0X3JvbGVfNSJdfX19XQ=="
                );
                TUserPermissions::TPtr perm = rm.GetUserPermissions(userId, tm, rolesCfg, TUserPermissionsFeatures(), Now(), replyCtx);
                UNIT_ASSERT(perm);
                UNIT_ASSERT(perm->HasAction("test_action_1"));
                UNIT_ASSERT(perm->HasAction("test_action_2"));
                UNIT_ASSERT(perm->HasAction("test_action_3"));
                UNIT_ASSERT(perm->HasAction("test_action_4"));
                UNIT_ASSERT(!perm->HasAction("test_action_5"));
                UNIT_ASSERT(!perm->HasAction("test_action_6"));
            }
        }
        {
            auto userId = eGenerator.CreateUser("test-user-1", true, "active");
            TUserRole uRole;
            uRole.SetUserId(userId).SetRoleId("test_role_1").SetActive(true);
            UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
            uRole.SetUserId(userId).SetRoleId("test_role_2").SetActive(true);
            UNIT_ASSERT(configGenerator.AddUserRole(uRole, USER_ROOT_DEFAULT));
            TRolesConfig rolesCfg;
            {
                auto replyCtx = MakeAtomicShared<TReplyContextMock>();
                replyCtx->MutableRequestData().AddHeader(
                    "X-Yandex-ExpFlags",
                    // "recursive_disabled_roles": [ "test_role_1" ].
                    "W3siSEFORExFUiI6IkRSSVZFX01PQklMRSIsIkNPTlRFWFQiOnsiRFJJVkVfTU9CSUxFIjp7InJlY3Vyc2l2ZV9kaXNhYmxlZF9y"
                    "b2xlcyI6WyJ0ZXN0X3JvbGVfMSJdfX19XQ=="
                );
                TUserPermissions::TPtr perm = rm.GetUserPermissions(userId, tm, rolesCfg, TUserPermissionsFeatures(), Now(), replyCtx);
                UNIT_ASSERT(perm);
                UNIT_ASSERT(!perm->HasAction("test_action_1"));
                UNIT_ASSERT(perm->HasAction("test_action_2"));
                UNIT_ASSERT(perm->HasAction("test_action_3"));
                UNIT_ASSERT(!perm->HasAction("test_action_4"));
                UNIT_ASSERT(!perm->HasAction("test_action_5"));
                UNIT_ASSERT(!perm->HasAction("test_action_6"));
            }
            {
                auto replyCtx = MakeAtomicShared<TReplyContextMock>();
                replyCtx->MutableRequestData().AddHeader(
                    "X-Yandex-ExpFlags",
                    // "recursive_disabled_roles": [ "test_role_2" ].
                    "W3siSEFORExFUiI6IkRSSVZFX01PQklMRSIsIkNPTlRFWFQiOnsiRFJJVkVfTU9CSUxFIjp7InJlY3Vyc2l2ZV9kaXNhYmxlZF9y"
                    "b2xlcyI6WyJ0ZXN0X3JvbGVfMiJdfX19XQ=="
                );
                TUserPermissions::TPtr perm = rm.GetUserPermissions(userId, tm, rolesCfg, TUserPermissionsFeatures(), Now(), replyCtx);
                UNIT_ASSERT(perm);
                UNIT_ASSERT(perm->HasAction("test_action_1"));
                UNIT_ASSERT(!perm->HasAction("test_action_2"));
                UNIT_ASSERT(!perm->HasAction("test_action_3"));
                UNIT_ASSERT(perm->HasAction("test_action_4"));
                UNIT_ASSERT(!perm->HasAction("test_action_5"));
                UNIT_ASSERT(!perm->HasAction("test_action_6"));
            }
            {
                auto replyCtx = MakeAtomicShared<TReplyContextMock>();
                replyCtx->MutableRequestData().AddHeader(
                    "X-Yandex-ExpFlags",
                    // "recursive_disabled_roles": [ "test_role_5" ].
                    "W3siSEFORExFUiI6IkRSSVZFX01PQklMRSIsIkNPTlRFWFQiOnsiRFJJVkVfTU9CSUxFIjp7InJlY3Vyc2l2ZV9kaXNhYmxlZF9y"
                    "b2xlcyI6WyJ0ZXN0X3JvbGVfNSJdfX19XQ=="
                );
                TUserPermissions::TPtr perm = rm.GetUserPermissions(userId, tm, rolesCfg, TUserPermissionsFeatures(), Now(), replyCtx);
                UNIT_ASSERT(perm);
                UNIT_ASSERT(perm->HasAction("test_action_1"));
                UNIT_ASSERT(perm->HasAction("test_action_2"));
                UNIT_ASSERT(perm->HasAction("test_action_3"));
                UNIT_ASSERT(perm->HasAction("test_action_4"));
                UNIT_ASSERT(!perm->HasAction("test_action_5"));
                UNIT_ASSERT(!perm->HasAction("test_action_6"));
            }
        }
    }

    Y_UNIT_TEST(ActionUserTagsFilter) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        {
            auto tagAction = MakeAtomicShared<TTagAction>();
            tagAction->SetName("view_fake_news");
            tagAction->SetTagName("fake_news");
            tagAction->AddTagAction(NTagActions::ETagAction::Observe);
            tagAction->SetUserTagsFilter(TTagsFilter::BuildFromString("test_push_smile"));
            RegisterAction(*env.GetServer(), tagAction, "pack_access");
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto userId = TString{USER_ID_DEFAULT};
        {
            auto permissions = env.GetServer()->GetDriveDatabase().GetUserPermissions(userId, {});
            UNIT_ASSERT(permissions);
            UNIT_ASSERT(!permissions->HasAction("view_fake_news"));
        }
        auto tag = MakeAtomicShared<TLandingUserTag>("test_push_smile");
        UNIT_ASSERT(env->AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        {
            auto permissions = env.GetServer()->GetDriveDatabase().GetUserPermissions(userId, {});
            UNIT_ASSERT(permissions);
            UNIT_ASSERT(permissions->HasAction("view_fake_news"));
        }
    }
}
