#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>
#include <drive/backend/ut/rt_background/test/config.h>

#include <drive/backend/background/major/config.h>
#include <drive/backend/background/major/processor.h>
#include <drive/backend/background/transformation/config.h>
#include <drive/backend/background/transformation/processor.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/billing/accounts/limited.h>
#include <drive/backend/billing/ut/library/offer.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/transformation.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/rt_background/billing/accounts_refresh.h>
#include <drive/backend/rt_background/billing/bonus_tags.h>
#include <drive/backend/rt_background/billing/refunds.h>
#include <drive/backend/rt_background/toll_roads/process.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>


Y_UNIT_TEST_SUITE(RTBackgroundProcessesSuite) {

    class TTestStateAcceptor: public IMessageProcessor {
        R_OPTIONAL(i32, Value);
    public:

        TTestStateAcceptor() {
            RegisterGlobalMessageProcessor(this);
        }

        ~TTestStateAcceptor() {
            UnregisterGlobalMessageProcessor(this);
        }

        bool Process(IMessage* message) override {
            const TRTBackgroundMessageOut* outMessage = dynamic_cast<const TRTBackgroundMessageOut*>(message);
            if (outMessage) {
                Value = outMessage->GetValue();
                return true;
            }
            return false;
        }

        TString Name() const override {
            return ToString(ui64(this));
        }
    };

    Y_UNIT_TEST(Simple) {
        TStringStream ss;
        NDrive::TServerConfigGenerator gen;
        gen.SetNeedBackground(0);
        gen.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        TTestStateAcceptor baseAcceptor;
        {
            TRTBackgroundMessageIncoming mess;
            SendGlobalMessage(mess);
            UNIT_ASSERT(mess.GetPingValues().empty());
            Sleep(TDuration::Seconds(2));
            UNIT_ASSERT(!baseAcceptor.HasValue());
        }

        {
            THolder<TRTBackgroundTester> actor(new TRTBackgroundTester);
            actor->SetPingValue(1);
            actor->SetPeriod(TDuration::Seconds(1));
            actor->SetEnabled(true);
            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("tester");
            UNIT_ASSERT(gen.ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        }
        for (ui32 i = 0; i <= 100; ++i) {
            TRTBackgroundMessageIncoming mess;
            SendGlobalMessage(mess);
            if (!mess.GetPingValues().empty() && mess.GetPingValues().size() == 1 && mess.GetPingValues().front() == 1) {
                break;
            } else {
                UNIT_ASSERT(i < 100);
            }
            Sleep(TDuration::Seconds(1));
        }
        for (ui32 i = 0; i <= 100; ++i) {
            if (baseAcceptor.HasValue() && baseAcceptor.GetValueUnsafe() > 0) {
                break;
            } else {
                UNIT_ASSERT(i < 100);
            }
            Sleep(TDuration::Seconds(1));
        }

        const i32 value1 = baseAcceptor.GetValueUnsafe();
        for (ui32 i = 0; i <= 100; ++i) {
            if (baseAcceptor.GetValueUnsafe() > value1 + 5) {
                break;
            } else {
                UNIT_ASSERT(i < 100);
            }
            Sleep(TDuration::Seconds(1));
        }

        {
            THolder<TRTBackgroundTester> actor(new TRTBackgroundTester);
            actor->SetPingValue(-10);
            actor->SetPeriod(TDuration::Seconds(1));
            actor->SetEnabled(true);
            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("tester");
            UNIT_ASSERT(gen.ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        }

        for (ui32 i = 0; i <= 100; ++i) {
            TRTBackgroundMessageIncoming mess;
            SendGlobalMessage(mess);
            if (mess.GetPingValues().size() == 1 && mess.GetPingValues().front() == -10) {
                break;
            } else {
                UNIT_ASSERT(i < 100);
            }
            Sleep(TDuration::Seconds(1));
        }
        for (ui32 i = 0; i <= 100; ++i) {
            if (baseAcceptor.GetValueUnsafe() < -100) {
                break;
            } else {
                UNIT_ASSERT(i < 100);
            }
            Sleep(TDuration::Seconds(1));
        }

        {
            THolder<TRTBackgroundTester> actor(new TRTBackgroundTester);
            actor->SetPingValue(1);
            actor->SetPeriod(TDuration::Seconds(1));
            actor->SetEnabled(false);
            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("tester");
            UNIT_ASSERT(gen.ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        }
        Sleep(TDuration::Seconds(5));
        TTestStateAcceptor newAcceptor;
        for (ui32 i = 0; i <= 5; ++i) {
            UNIT_ASSERT(!newAcceptor.HasValue());
            Sleep(TDuration::Seconds(1));
        }
        {
            THolder<TRTBackgroundTester> actor(new TRTBackgroundTester);
            actor->SetPingValue(1);
            actor->SetPeriod(TDuration::Seconds(1));
            actor->SetEnabled(true);
            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("tester");
            UNIT_ASSERT(gen.ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        }
        for (ui32 i = 0; i <= 100; ++i) {
            if (newAcceptor.HasValue()) {
                break;
            } else {
                UNIT_ASSERT(i < 100);
            }
            Sleep(TDuration::Seconds(1));
        }

        UNIT_ASSERT(gen.RemoveRTBackground("tester", USER_ROOT_DEFAULT));
        Sleep(TDuration::Seconds(5));
        TTestStateAcceptor lastAcceptor;
        for (ui32 i = 0; i <= 7; ++i) {
            UNIT_ASSERT(!lastAcceptor.HasValue());
            Sleep(TDuration::Seconds(1));
        }

    }

    Y_UNIT_TEST(BillingProcess) {
        TStringStream ss;
        NDrive::TServerConfigGenerator gen;
        gen.SetNeedBackground(0);
        gen.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto registeredAccounts = driveApi.GetBillingManager().GetAccountsManager().GetRegisteredAccounts(TInstant::Now());
        ui32 bonusTypeId = 0;
        for (auto&& d : registeredAccounts) {
            if (d.GetName() == "bonus") {
                bonusTypeId = d.GetId();
                break;
            }
        }
        UNIT_ASSERT(bonusTypeId > 0);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        auto session = driveApi.GetBillingManager().BuildSession();
        auto baDescription = driveApi.GetBillingManager().GetAccountsManager().GetDescriptionByName("bonus");
        UNIT_ASSERT(baDescription.Defined());
        auto bonusAccount = driveApi.GetBillingManager().GetAccountsManager().GetOrCreateAccount(USER_ROOT_DEFAULT, baDescription.GetRef(), USER_ROOT_DEFAULT, session);
        UNIT_ASSERT(session.Commit());
        UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 0);
        const IDriveTagsManager& tagsManager = driveApi.GetTagsManager();
        {
            NJson::TJsonValue descJson;
            descJson["type"] = "fixed_sum_tag";
            descJson["name"] = "add_bonus_tag";
            NJson::TJsonValue metaJson;
            metaJson["amount"] = 10000;
            metaJson["action"] = "credit";
            metaJson["accounts"].AppendValue("bonus");
            descJson["meta"] = metaJson;

            NStorage::TTableRecord tRecord;
            UNIT_ASSERT(tRecord.DeserializeFromJson(descJson));
            auto tag = TTagDescription::ConstructFromTableRecord(tRecord);
            UNIT_ASSERT(tag);
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(tagsManager.GetTagsMeta().RegisterTag(tag, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            NJson::TJsonValue tagJson;
            tagJson["tag_name"] = "add_bonus_tag";
            tagJson["comment"] = "test add_bonus_tag";
            ITag::TPtr tag = IJsonSerializableTag::BuildFromJson(tagsManager, tagJson, nullptr);
            UNIT_ASSERT(tag);
            UNIT_ASSERT(gen.AddTag(tag, USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> userTags;
            UNIT_ASSERT(tagsManager.GetUserTags().RestoreEntityTags(USER_ROOT_DEFAULT, {}, userTags, session));
            bool hasBonusTag = false;
            for (auto&& tag : userTags) {
                if (tag->GetName() == "add_bonus_tag") {
                    hasBonusTag = true;
                }
            }
            UNIT_ASSERT(hasBonusTag);
        }

        {
            THolder<TRTBillingTagsWatcher> actor(new TRTBillingTagsWatcher);
            actor->SetPeriod(TDuration::Seconds(1));
            actor->SetEnabled(true);
            actor->SetRobotUserId(USER_ROOT_DEFAULT);
            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("billing");
            UNIT_ASSERT(gen.ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        }

        {
            TInstant deadline = Now() + TDuration::Seconds(30);
            bool accepted = false;
            while (Now() < deadline) {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                TVector<TDBTag> userTags;
                UNIT_ASSERT(tagsManager.GetUserTags().RestoreEntityTags(USER_ROOT_DEFAULT, {}, userTags, session));
                accepted = true;
                for (auto&& tag : userTags) {
                    if (tag->GetName() == "add_bonus_tag") {
                        accepted = false;
                        break;
                    }
                }
                if (accepted) {
                    break;
                }
                Sleep(TDuration::Seconds(3));
            }
            UNIT_ASSERT(accepted);
            auto bonusAccountSession = driveApi.GetBillingManager().BuildSession();
            auto bonusAccount = driveApi.GetBillingManager().GetAccountsManager().GetOrCreateAccount(USER_ROOT_DEFAULT, baDescription.GetRef(), USER_ROOT_DEFAULT, bonusAccountSession);
            UNIT_ASSERT(bonusAccountSession.Commit());
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), 10000);
        }
    }

    Y_UNIT_TEST(RefreshAccountProcess) {
        TStringStream ss;
        NDrive::TServerConfigGenerator gen;
        gen.SetNeedBackground(0);
        gen.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto& billingManager = driveApi.GetBillingManager();
        auto& accountsManager = billingManager.GetAccountsManager();;

        {
            auto session = billingManager.BuildSession();
            NJson::TJsonValue descJson;
            descJson["type"] = ::ToString(NDrive::NBilling::EAccount::Wallet);
            descJson["hard_limit"] = NDrive::NBilling::AccountLimit;
            descJson["soft_limit"] = NDrive::NBilling::AccountLimit;
            descJson["name"] = "y.drive";
            descJson["meta"]["hr_name"] = "y.drive";
            descJson["meta"]["refresh_policy"] = ToString(NDrive::NBilling::TRefreshSchedule::EPolicy::Month);
            NDrive::NBilling::TAccountDescriptionRecord description;
            UNIT_ASSERT(description.FromJson(descJson, nullptr));
            UNIT_ASSERT(accountsManager.UpsertAccountDescription(description, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        auto registeredAccounts = accountsManager.GetRegisteredAccounts(TInstant::Now());
        ui32 walletType = 0;
        for (auto&& d : registeredAccounts) {
            if (d.GetName() == "y.drive") {
                walletType = d.GetId();
                break;
            }
        }
        UNIT_ASSERT(walletType > 0);
        {
            auto session = billingManager.BuildSession();
            auto accountRecord = MakeAtomicShared<NDrive::NBilling::TLimitedAccountRecord>();
            accountRecord->SetTypeId(walletType);
            accountRecord->SetActive(true);
            accountRecord->SetExpenditure(NDrive::NBilling::AccountLimit);
            accountRecord->SetNextRefresh(TInstant::Now() - TDuration::Minutes(10));
            ui32 accountId = 0;
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, USER_ROOT_DEFAULT, session, &accountId));
            UNIT_ASSERT(accountsManager.LinkAccount(USER_ROOT_DEFAULT, accountId, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = billingManager.BuildSession(true);
            auto userAccounts = accountsManager.GetUserAccounts(USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(userAccounts);
            NDrive::NBilling::IBillingAccount::TPtr account;
            for (auto&& acc : *userAccounts) {
                if (acc->GetName() == "y.drive") {
                    account = acc;
                }
            }
            UNIT_ASSERT(account);
            UNIT_ASSERT_VALUES_EQUAL(account->GetBalance(), 0);
        }
        {
            auto actor = MakeHolder<TRTBillingAccountsWatcher>();
            actor->SetPeriod(TDuration::Seconds(1));
            actor->SetEnabled(true);
            actor->SetRobotUserId(USER_ROOT_DEFAULT);
            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("billing");
            UNIT_ASSERT(gen.ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
        }

        {
            TInstant deadline = Now() + TDuration::Seconds(30);
            bool accepted = false;
            while (Now() < deadline) {
                auto session = billingManager.BuildSession(true);
                auto userAccounts = accountsManager.GetUserAccounts(USER_ROOT_DEFAULT, session);
                UNIT_ASSERT(userAccounts);
                for (auto&& acc : *userAccounts) {
                    if (acc->GetName() == "y.drive" && acc->GetBalance() == NDrive::NBilling::AccountLimit) {
                        accepted = true;
                        break;
                    }
                }
            }
            UNIT_ASSERT(accepted);
        }
    }

    Y_UNIT_TEST(RefreshAccountWithCustomRefreshLogic) {
        TStringStream ss;
        NDrive::TServerConfigGenerator gen;
        gen.SetNeedBackground(0);
        gen.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto& billingManager = driveApi.GetBillingManager();
        auto& accountsManager = billingManager.GetAccountsManager();;

        {
            auto session = billingManager.BuildSession();
            NJson::TJsonValue descJson;
            descJson["type"] = ::ToString(NDrive::NBilling::EAccount::Wallet);
            descJson["hard_limit"] = NDrive::NBilling::AccountLimit;
            descJson["soft_limit"] = NDrive::NBilling::AccountLimit;
            descJson["name"] = "y.drive";
            descJson["meta"]["hr_name"] = "y.drive";
            descJson["meta"]["refresh_policy"] = ToString(NDrive::NBilling::TRefreshSchedule::EPolicy::Month);
            NDrive::NBilling::TAccountDescriptionRecord description;
            UNIT_ASSERT(description.FromJson(descJson, nullptr));
            UNIT_ASSERT(accountsManager.UpsertAccountDescription(description, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        auto registeredAccounts = accountsManager.GetRegisteredAccounts(TInstant::Now());
        ui32 walletType = 0;
        for (auto&& d : registeredAccounts) {
            if (d.GetName() == "y.drive") {
                walletType = d.GetId();
                break;
            }
        }
        UNIT_ASSERT(walletType > 0);
        {
            auto session = billingManager.BuildSession();
            auto accountRecord = MakeAtomicShared<NDrive::NBilling::TLimitedAccountRecord>();
            accountRecord->SetTypeId(walletType);
            accountRecord->SetActive(true);
            accountRecord->SetExpenditure(NDrive::NBilling::AccountLimit);
            accountRecord->SetRefreshPolicy(NDrive::NBilling::TRefreshSchedule::EPolicy::None);
            accountRecord->SetNextRefresh(TInstant::Now() - TDuration::Minutes(10));
            ui32 accountId = 0;
            UNIT_ASSERT(accountsManager.RegisterAccount(accountRecord, USER_ROOT_DEFAULT, session, &accountId));
            UNIT_ASSERT(accountsManager.LinkAccount(USER_ROOT_DEFAULT, accountId, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto session = billingManager.BuildSession(true);
            auto userAccounts = accountsManager.GetUserAccounts(USER_ROOT_DEFAULT, session);
            UNIT_ASSERT(userAccounts);
            NDrive::NBilling::IBillingAccount::TPtr account;
            for (auto&& acc : *userAccounts) {
                if (acc->GetName() == "y.drive") {
                    account = acc;
                }
            }
            UNIT_ASSERT(account);
            UNIT_ASSERT_VALUES_EQUAL(account->GetBalance(), 0);
            UNIT_ASSERT_VALUES_EQUAL(account->GetNextRefresh(), TInstant::Max());
        }
    }

    Y_UNIT_TEST(SimpleRefund) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive");
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();

        TString offerId = ICommonOffer::CreateOfferId();
        TCachedPayments payments;
        script.Add<NDrive::NTest::TCommonChecker>([&offerId, &payments](NDrive::NTest::TRTContext& context) {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();

            auto offer = MakeAtomicShared<TFakeOffer>((ui32)0);
            offer->SetOfferId(offerId).SetPaymentDiscretization(500);
            TVector<TString> accounts;
            accounts.push_back("y.drive");
            offer->SetChargableAccounts(accounts);
            {
                auto session = billingManager.BuildSession(false);
                UNIT_ASSERT(billingManager.CreateBillingTask(USER_ID_DEFAULT, offer, session, EBillingType::Ticket));
                UNIT_ASSERT(session.Commit());
            }
            {
                TMap<TString, double> money;
                money[offerId] = 10000;
                auto session = billingManager.BuildSession(false);
                UNIT_ASSERT(billingManager.SetBillingInfo(money, session));
                UNIT_ASSERT(session.Commit());
            }
            {
                auto session = billingManager.BuildSession(false);
                UNIT_ASSERT(billingManager.FinishingBillingTask(offer->GetOfferId(), session));
                UNIT_ASSERT(billingManager.AddClosedBillingInfo(offer->GetOfferId(), USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(billingManager.GetPaymentsManager().GetPayments(payments, offerId, session));
                UNIT_ASSERT(session.Commit());
            }
            UNIT_ASSERT_VALUES_EQUAL(payments.GetPayments().size(), 1);
            billingManager.WaitBillingCycle();
        });
        script.Add<NDrive::NTest::TDropCache>();

        NJson::TJsonValue tagData;
        tagData["amount"] = 5000;
        tagData["session_id"] = offerId;
        tagData["compensation"] = 0;
        script.Add<NDrive::NTest::TAddTagActions>("refund_tag", USER_ID_DEFAULT).SetCustomData(tagData).SetEntityType(NEntityTagsManager::EEntityType::User).SetComment("test");

        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto actor = MakeHolder<TRTRefundsWatcher>();
            actor->SetPeriod(TDuration::Seconds(1));
            actor->SetEnabled(true);
            actor->SetRobotUserId(USER_ROOT_DEFAULT);
            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("billing");
            UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
            Sleep(TDuration::Seconds(20));
        });

        script.Add<NDrive::NTest::TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
            TCachedPayments paymentsAfter;
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto session = billingManager.BuildSession(false);
            UNIT_ASSERT(billingManager.GetPaymentsManager().GetPayments(paymentsAfter, offerId, session));
            UNIT_ASSERT_VALUES_EQUAL(paymentsAfter.GetPayments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(paymentsAfter.GetTimeline().front().GetCleared(), 5000);
        });

        UNIT_ASSERT(script.Execute());
    }

    class TFakeRoadConfig : public ITollRoadConfig {
        R_FIELD(TVector<TTransponderEvent>, ToReturn);
    private:
        static TFactory::TRegistrator<TFakeRoadConfig> Registrator;
    public:
        bool Init(const TYandexConfig::Section* /*section*/) override {
            return true;
        }

        void ToString(IOutputStream& /*os*/) const override {
        }

        bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo) override{
            if (jsonInfo.IsArray()) {
                for (auto&& json : jsonInfo.GetArray()) {
                    TString carId, transponderId;
                    ui32 amount;
                    TInstant date;
                    if (!NJson::ParseField(json["car_id"], carId, true)
                        || !NJson::ParseField(json["transponder_id"], transponderId, true)
                        || !NJson::ParseField(json["amount"], amount, true)
                        || !NJson::ParseField(json["date"], date, true)) {
                        return false;
                    }
                    ToReturn.push_back(TTransponderEvent(carId, transponderId, amount, date));
                }
            }
            return true;
        }

        NJson::TJsonValue SerializeToJson() const override{
            NJson::TJsonValue result = NJson::JSON_ARRAY;
            for (auto&& event : ToReturn) {
                NJson::TJsonValue jsonEvent = NJson::JSON_MAP;
                jsonEvent.InsertValue("car_id", event.GetCarId());
                jsonEvent.InsertValue("transponder_id", event.GetTransponderId());
                jsonEvent.InsertValue("amount", event.GetAmount());
                jsonEvent.InsertValue("date", event.GetDateTime().Seconds());
                result.AppendValue(jsonEvent);
            }
            return result;
        }

        THolder<ITollRoad> Construct() const override;
    };

    class TFakeRoad : public ITollRoad {
    public:
        TFakeRoad(const TFakeRoadConfig& roadConfig)
        : RoadConfig(roadConfig)
        {
        }

        TString GetType() const override {
            return GetTypeName();
        }

        static TString GetTypeName() {
            return "fake";
        }

        TExpected<TVector<TTransponderEvent>, TString> GetTransponderEvents(TInstant /*from*/, TInstant /*to*/, const NDrive::IServer& /*server*/, const TSet<TString>& filter) const override {
            TVector<TTransponderEvent> result;
            for (auto&& event : RoadConfig.GetToReturn()) {
                if (filter.contains(event.GetCarId())) {
                    result.push_back(event);
                }
            }
            return result;
        }
    private:
        const TFakeRoadConfig& RoadConfig;
    };

    THolder<ITollRoad> TFakeRoadConfig::Construct() const {
        return MakeHolder<TFakeRoad>(*this);
    }

    TFakeRoadConfig::TFactory::TRegistrator<TFakeRoadConfig> TFakeRoadConfig::Registrator("fake");

    void RegisterBillingTag(NDrive::NTest::TScript& script) {
        NJson::TJsonValue descJson;
        descJson["type"] = "operation_tag";
        descJson["name"] = "road_operation_tag";
        descJson["display_name"] = "test";
        descJson["comment"] = "";
        NJson::TJsonValue metaJson;
        metaJson["action"] = "debit";
        metaJson["terminal"] = "toll_road";
        metaJson["accounts"].AppendValue("card");
        descJson["meta"] = metaJson;
        script.Add<NDrive::NTest::TRegisterTag>(descJson);
    }

    Y_UNIT_TEST(WhsdTransponderCharges) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        TInstant start = Now();
        TInstant end = start + TDuration::Minutes(30);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            context.GetConfigGenerator().ServiceAppAssignDevice(context.GetCar().Id, "6666666000001111111", true, USER_ROOT_DEFAULT);
            UNIT_ASSERT(context.GetDriveAPI().GetCarGenericAttachments().RefreshCache(Now()));
        });
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());
        script.Add<NDrive::NTest::TRide>(TDuration::Minutes(1));
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(20)).SetCarPosition(to);
        script.Add<NDrive::NTest::TDropCache>();
        RegisterBillingTag(script);

        script.Add<NDrive::NTest::TCommonChecker>([start, end](NDrive::NTest::TRTContext& context) {
            context.SetMileage(100);
            NJson::TJsonValue descJson;
            TJsonProcessor::Write(descJson, "first_instant", start);
            TJsonProcessor::Write(descJson, "last_instant", end);
            TJsonProcessor::Write(descJson, "bp_enabled", true);
            TJsonProcessor::Write(descJson, "period", TDuration::Seconds(5));
            descJson["robot_user_id"] = USER_ROOT_DEFAULT;
            descJson["offer_attribute_filter"] = "standart";

            TAtomicSharedPtr<TFakeRoadConfig> roadConfig = dynamic_cast<TFakeRoadConfig*>(ITollRoadConfig::TFactory::Construct("fake"));
            UNIT_ASSERT(roadConfig);
            roadConfig->SetToReturn({
                TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(5)),
                TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(6))
                });

            NJson::TJsonValue roadJson;
            roadJson["road_type"] = "fake";
            roadJson["landing_datetime_format"] = "%Y-%m-%d %H:%M:%S";
            roadJson["config"] = roadConfig->SerializeToJson();
            TVector<TString> tags = {"road_operation_tag"};
            TJsonProcessor::WriteContainerArray(roadJson, "billing_tags", tags);
            NJson::TJsonValue& roadsJson = descJson.InsertValue("road_types", NJson::JSON_ARRAY);
            roadsJson.AppendValue(roadJson);

            auto actor = MakeHolder<TRTRoadChargesWatcher>();
            UNIT_ASSERT(actor->DeserializeFromJson(descJson));

            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("charges");
            UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
            Sleep(TDuration::Seconds(5));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        });
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::User).SetObjectId(USER_ID_DEFAULT).SetTagName("road_operation_tag").SetExpectOK(true);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(WhsdTransponderFreeRides) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        TInstant start = Now();
        TInstant end = start + TDuration::Minutes(30);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            context.GetConfigGenerator().ServiceAppAssignDevice(context.GetCar().Id, "6666666000001111111", true, USER_ROOT_DEFAULT);
            UNIT_ASSERT(context.GetDriveAPI().GetCarGenericAttachments().RefreshCache(Now()));
        });
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>(TDuration::Zero());
        script.Add<NDrive::NTest::TRide>(TDuration::Minutes(1));
        script.Add<NDrive::NTest::TDrop>(TDuration::Minutes(20)).SetCarPosition(to);
        script.Add<NDrive::NTest::TDropCache>();
        RegisterBillingTag(script);
        {
            TTagDescription description;
            description.SetName("landing_tag_freerides").SetType(TLandingUserTag::TypeName).SetDefaultPriority(0);
            TTagDescription::TPtr tag = new TTagDescription(description);
            script.Add<NDrive::NTest::TRegisterTag>(tag);
            TTagDescription description2;
            description2.SetName("skipped_billing_tag_freerides").SetType(TLandingUserTag::TypeName).SetDefaultPriority(0);
            TTagDescription::TPtr tag2 = new TTagDescription(description2);
            script.Add<NDrive::NTest::TRegisterTag>(tag2);
        }

        script.Add<NDrive::NTest::TCommonChecker>([start, end](NDrive::NTest::TRTContext& context) {
            context.SetMileage(100);
            NJson::TJsonValue descJson;
            TJsonProcessor::Write(descJson, "first_instant", start);
            TJsonProcessor::Write(descJson, "last_instant", end);
            TJsonProcessor::Write(descJson, "bp_enabled", true);
            TJsonProcessor::Write(descJson, "period", TDuration::Seconds(10));
            descJson["robot_user_id"] = USER_ROOT_DEFAULT;
            descJson["offer_attribute_filter"] = "standart";

            TAtomicSharedPtr<TFakeRoadConfig> roadConfig = dynamic_cast<TFakeRoadConfig*>(ITollRoadConfig::TFactory::Construct("fake"));
            UNIT_ASSERT(roadConfig);
            roadConfig->SetToReturn({TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(5)),
                TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(6))});

            NJson::TJsonValue roadJson;
            roadJson["road_type"] = "fake";
            roadJson["landing_datetime_format"] = "%Y-%m-%d %H:%M:%S";
            roadJson["config"] = roadConfig->SerializeToJson();
            TVector<TString> tags = {"road_operation_tag", "landing_tag_freerides"};
            TJsonProcessor::WriteContainerArray(roadJson, "billing_tags", tags);
            TVector<TString> skippedTags = {"skipped_billing_tag_freerides"};
            TJsonProcessor::WriteContainerArray(roadJson, "billing_skipped_tags", skippedTags);
            roadJson["billing_allowed_setting"] = "billing_setting";
            roadJson["free_rides_number_setting"] = "free_rides_setting";
            roadJson["free_rides_number"] = 2;
            NJson::TJsonValue& roadsJson = descJson.InsertValue("road_types", NJson::JSON_ARRAY);
            roadsJson.AppendValue(roadJson);

            auto actor = MakeHolder<TRTRoadChargesWatcher>();
            UNIT_ASSERT(actor->DeserializeFromJson(descJson));

            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("charges");
            UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        });
        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(1));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::User).SetObjectId(USER_ID_DEFAULT).SetTagName("road_operation_tag").SetExpectOK(false);
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::User).SetObjectId(USER_ID_DEFAULT).SetTagName("skipped_billing_tag_freerides").SetExpectOK(true);
        std::function<bool(const TMaybe<TString>&)> settingsNoBillingChecker = [](const TMaybe<TString>& setting) {
            return setting.Empty();
        };
        std::function<bool(const TMaybe<TString>&)> settingsOneRideChecker = [](const TMaybe<TString>& setting) {
            return setting.Defined() && setting.GetRef() == "1";
        };
        script.Add<NDrive::NTest::TCheckUserSetting>("free_rides_setting", settingsOneRideChecker);
        script.Add<NDrive::NTest::TCheckUserSetting>("billing_setting", settingsNoBillingChecker);

        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(10));
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::User).SetObjectId(USER_ID_DEFAULT).SetTagName("road_operation_tag").SetExpectOK(false);
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::User).SetObjectId(USER_ID_DEFAULT).SetTagName("landing_tag_freerides").SetExpectOK(false);
        std::function<bool(const TMaybe<TString>&)> settingsTwoRidesChecker = [](const TMaybe<TString>& setting) {
            return setting.Defined() && setting.GetRef() == "2";
        };
        script.Add<NDrive::NTest::TCheckUserSetting>("free_rides_setting", settingsTwoRidesChecker);
        script.Add<NDrive::NTest::TCheckUserSetting>("billing_setting", settingsNoBillingChecker);

        script.Add<NDrive::NTest::TSleepAction>().SetWaitingDuration(TDuration::Seconds(10));

        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::User).SetObjectId(USER_ID_DEFAULT).SetTagName("road_operation_tag").SetExpectOK(true);
        script.Add<NDrive::NTest::TCheckHasTag>().SetEntityType(NEntityTagsManager::EEntityType::User).SetObjectId(USER_ID_DEFAULT).SetTagName("landing_tag_freerides").SetExpectOK(true);

        std::function<bool(const TMaybe<TString>&)> settingsBillingAllowedChecker = [](const TMaybe<TString>& setting) {
            return setting.Defined();
        };

        script.Add<NDrive::NTest::TCheckUserSetting>("billing_setting", settingsBillingAllowedChecker);

        script.Add<NDrive::NTest::TCommonChecker>([start](NDrive::NTest::TRTContext& context) {
            TVector<TDBTag> tags;
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetEntityTagsManager(NEntityTagsManager::EEntityType::User).RestoreTags({ USER_ID_DEFAULT }, { "landing_tag_freerides" }, tags, session), TStringBuilder() << "Could not restore tags");
            UNIT_ASSERT(tags.size() > 0);
            auto tag = tags[0];
            auto landingTag = tag.GetTagAs<TLandingUserTag>();
            UNIT_ASSERT(landingTag);
            auto map = landingTag->GetContextParameters();
            UNIT_ASSERT_VALUES_EQUAL(map[ToString(TRTRoadChargesWatcher::EContextParameterKey::BillingAmount)], "40");
            TInstant ts = start + TDuration::Minutes(5);
            UNIT_ASSERT_VALUES_EQUAL(map[ToString(TRTRoadChargesWatcher::EContextParameterKey::EventTimestamp)], ts.FormatLocalTime("%Y-%m-%d %H:%M:%S"));
        });
        script.Add<NDrive::NTest::TCommonChecker>([start](NDrive::NTest::TRTContext& context) {
            TVector<TDBTag> tags;
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetEntityTagsManager(NEntityTagsManager::EEntityType::User).RestoreTags({ USER_ID_DEFAULT }, { "skipped_billing_tag_freerides" }, tags, session), TStringBuilder() << "Could not restore tags");
            UNIT_ASSERT(tags.size() > 0);
            auto tag = tags[0];
            auto landingTag = tag.GetTagAs<TLandingUserTag>();
            UNIT_ASSERT(landingTag);
            auto map = landingTag->GetContextParameters();
            UNIT_ASSERT_VALUES_EQUAL(map[ToString(TRTRoadChargesWatcher::EContextParameterKey::BillingAmount)], "40");
            TInstant ts = start + TDuration::Minutes(5);
            UNIT_ASSERT_VALUES_EQUAL(map[ToString(TRTRoadChargesWatcher::EContextParameterKey::EventTimestamp)], ts.FormatLocalTime("%Y-%m-%d %H:%M:%S"));
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(WhsdTransponderLastEventState) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        TInstant start = Now();
        TInstant end = start + TDuration::Minutes(30);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        TGeoCoord from(37.5848674, 55.7352435);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            context.GetConfigGenerator().ServiceAppAssignDevice(context.GetCar().Id, "6666666000001111111", true, USER_ROOT_DEFAULT);
            UNIT_ASSERT(context.GetDriveAPI().GetCarGenericAttachments().RefreshCache(Now()));
        });

        script.Add<NDrive::NTest::TCommonChecker>([start, end](NDrive::NTest::TRTContext& context) {
            context.SetMileage(100);
            NJson::TJsonValue descJson;
            TJsonProcessor::Write(descJson, "first_instant", start);
            TJsonProcessor::Write(descJson, "last_instant", end);
            TJsonProcessor::Write(descJson, "bp_enabled", true);
            TJsonProcessor::Write(descJson, "period", TDuration::Seconds(5));
            descJson["robot_user_id"] = USER_ROOT_DEFAULT;
            descJson["offer_attribute_filter"] = "standart";

            TAtomicSharedPtr<TFakeRoadConfig> roadConfig = dynamic_cast<TFakeRoadConfig*>(ITollRoadConfig::TFactory::Construct("fake"));
            UNIT_ASSERT(roadConfig);
            roadConfig->SetToReturn({
                TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(1)),
                TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(7)),
                TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(3)),
                TTransponderEvent(context.GetCar().Id, "6666666000001111111", 4000, start + TDuration::Minutes(5))
                });

            NJson::TJsonValue roadJson;
            roadJson["road_type"] = "fake";
            roadJson["config"] = roadConfig->SerializeToJson();
            TVector<TString> tags = {"road_operation_tag"};
            TJsonProcessor::WriteContainerArray(roadJson, "billing_tags", tags);
            NJson::TJsonValue& roadsJson = descJson.InsertValue("road_types", NJson::JSON_ARRAY);
            roadsJson.AppendValue(roadJson);

            auto actor = MakeHolder<TRTRoadChargesWatcher>();
            UNIT_ASSERT(actor->DeserializeFromJson(descJson));

            TRTBackgroundProcessContainer container(actor.Release());
            container.SetName("charges");
            UNIT_ASSERT(context.GetConfigGenerator().ForceUpsertRTBackground(container, USER_ROOT_DEFAULT));
            Sleep(TDuration::Seconds(5));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        });

        script.Add<NDrive::NTest::TCommonChecker>([start](NDrive::NTest::TRTContext& context) {
            auto robotJson = context.GetConfigGenerator().GetRTBackgrounds(USER_ROOT_DEFAULT, {"charges"});
            TInstant last = start + TDuration::Minutes(7);
            UNIT_ASSERT(robotJson["rt_backgrounds"].IsArray() && robotJson["rt_backgrounds"].GetArray()[0]["background_process_state"].IsMap());
            UNIT_ASSERT_VALUES_EQUAL(robotJson["rt_backgrounds"].GetArray()[0]["background_process_state"]["last_instant"].GetUInteger(), last.Seconds());
        });

        UNIT_ASSERT(script.Execute());
    }
}
