#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/actions/info_access.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

void RegisterSearchTraits(const NDrive::IServer& server) {
    TVector<TSetting> settingValues;
    settingValues.emplace_back("handlers.default.use_permitted_search_traits", "true");
    UNIT_ASSERT(server.GetSettings().SetValues(settingValues, USER_ROOT_DEFAULT));

    auto searchFlagAction = MakeAtomicShared<TInfoAccessAction>("search_traits");
    searchFlagAction->SetDeviceSearchTraits(NDeviceReport::ReportCarId);
    RegisterAction(server, searchFlagAction, "view_permissions_all");
}

Y_UNIT_TEST_SUITE(DriveDBSearch) {

    class TSearchActor : public IObjectInQueue {
    private:
        NDrive::TServerConfigGenerator& GServer;

        struct TSearchPrefix {
            TString Value;
            bool IsCarPrefix;
            bool IsUserPrefix;

            TSearchPrefix() = default;

            TSearchPrefix(TString value, bool isCarPrefix, bool isUserPrefix)
                : Value(value)
                , IsCarPrefix(isCarPrefix)
                , IsUserPrefix(isUserPrefix)
            {
            }
        };

        void AddAllPrefixes (TVector<TSearchPrefix>& queries, const TString& maxString, bool isCarPrefix, bool isUserPrefix) {
            TString prefix = "";
            for (char c : maxString) {
                prefix += c;
                queries.push_back(TSearchPrefix(prefix, isCarPrefix, isUserPrefix));
            }
        }

    public:

        TSearchActor(NDrive::TServerConfigGenerator& gServer)
            : GServer(gServer)
        {
        }

        void Process(void*) override {
            TVector<TSearchPrefix> queries;
            AddAllPrefixes(queries, "ttt", true, false);
            AddAllPrefixes(queries, "zxqfd-test4", false, true);
            AddAllPrefixes(queries, "accman", false, true);
            AddAllPrefixes(queries, "c", true, true);

            for (auto prefix : queries) {
                auto searchResult = GServer.DoSearch(
                    TVector<TString>({prefix.Value}),
                    TVector<TString>(),
                    TVector<TString>({"cars", "users"})
                );
                if (prefix.IsCarPrefix) {
                    NJson::TJsonValue result;
                    UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
                    UNIT_ASSERT_C(result.GetArray().size() > 0, TStringBuilder() << prefix.Value << Endl);
                }
                if (prefix.IsUserPrefix) {
                    NJson::TJsonValue result;
                    UNIT_ASSERT(searchResult.GetValueByPath("objects.users", result));
                    UNIT_ASSERT_C(result.GetArray().size() > 0, TStringBuilder() << prefix.Value << Endl);
                }
            }
        }
    };

    Y_UNIT_TEST(SearchFullCycleTest) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        auto car = eGenerator.CreateCar("renault_kaptur");
        UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        ui32 threads = 1;
        TThreadPool searchThreadsQueue;
        searchThreadsQueue.Start(threads);

        for (ui32 index = 0; index < threads; ++index) {
            auto actor = MakeHolder<TSearchActor>(gServer);
            searchThreadsQueue.SafeAddAndOwn(std::move(actor));
        }

        searchThreadsQueue.Stop();
    }

    Y_UNIT_TEST(SearchCarVisibilityTest) {
        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        NJson::TJsonValue requestData;

        auto car = eGenerator.CreateCar();
        auto carId = car.Id;
        auto carNumber = car.Number;

        TGeoCoord someCoord1(12.58643489401084, 45.73761541977004);
        auto emulator1 = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(gServer.WaitCar(car.Id));

        auto objectIds = TVector<TString>({carId});
        for (auto objectId : objectIds) {
            UNIT_ASSERT(gServer.WaitCar(objectId));
            NJson::TJsonValue report = gServer.ListTags(objectId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
            NJson::TJsonValue::TArray arr;
            if (!report["records"].GetArray(&arr)) {
                continue;
            }
            for (auto&& i : arr) {
                UNIT_ASSERT(i.IsMap());
                gServer.RemoveTag({ i["tag_id"].GetString() }, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
            }
        }

        auto searchResult = gServer.GetServiceCarsList({}, USER_ID_DEFAULT);
        NJson::TJsonValue result;
        UNIT_ASSERT(searchResult.GetValueByPath("cars", result));
        UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), carId), TStringBuilder() << searchResult << Endl);

        UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), carId, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({carNumber}),
                TVector<TString>(),
                TVector<TString>({"cars"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), carId), TStringBuilder() << result);
        }
    }

    Y_UNIT_TEST(SearchCarSimpleQueries) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        NJson::TJsonValue requestData;

        auto car = eGenerator.CreateCar("renault_kaptur");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        UNIT_ASSERT(driveApi.GetCarsData()->RefreshCache(Now()));

        for (size_t i = 0; i < 2; ++i) {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({TString(SubstrUTF8(car.Number, i, car.Number.length() - i))}),
                TVector<TString>(),
                TVector<TString>({"cars"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), car.Id), TStringBuilder() << result << i);
        }
    }

    Y_UNIT_TEST(SearchTraits) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSearchTraits(*env.GetServer());

        auto car = env.GetEnvironmentGenerator().CreateCar("renault_kaptur");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto searchResult = env->DoSearch({ car.Id }, {} , {"cars"});
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), car.Id), TStringBuilder() << result);
        }
        {
            auto searchResult = env->DoSearch({ car.IMEI }, {} , {"cars"});
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), car.Id), TStringBuilder() << result);
        }
        {
            auto searchResult = env->DoSearch({ car.Number }, {} , {"cars"});
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), car.Id), TStringBuilder() << result);
        }
    }

    Y_UNIT_TEST(SearchUserByPhone) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        NJson::TJsonValue requestData;

        // standard format
        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({"79778180939"}),
                TVector<TString>(),
                TVector<TString>({"users"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.users", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), "a0ed1346-0a06-4fae-9d4a-96e5bc8bb4f6"), TStringBuilder() << result);
        }

        // shortened format
        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({"9778180939"}),
                TVector<TString>(),
                TVector<TString>({"users"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.users", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), "a0ed1346-0a06-4fae-9d4a-96e5bc8bb4f6"), TStringBuilder() << result);
        }

        // non-international format
        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({"89778180939"}),
                TVector<TString>(),
                TVector<TString>({"users"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.users", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), "a0ed1346-0a06-4fae-9d4a-96e5bc8bb4f6"), TStringBuilder() << result);
        }
    }

    Y_UNIT_TEST(SearchCarByChangedNumber) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        NJson::TJsonValue requestData;

        TString carId;
        {
            auto car = eGenerator.CreateCar("renault_kaptur");
            UNIT_ASSERT(server->GetDriveAPI()->GetCarsData()->RefreshCache(Now()));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            carId = car.Id;
        }

        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({"uniquenumber1"}),
                TVector<TString>(),
                TVector<TString>({"cars"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), carId), TStringBuilder() << result);
        }

        {
            auto car = *server->GetDriveAPI()->GetCarsData()->GetObject(carId);
            car.SetNumber("uniquenumber1");
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetCarsData()->UpdateCar(car, "robot-frontend", session) && session.Commit());
            UNIT_ASSERT(server->GetDriveAPI()->GetCarsData()->RefreshCache(Now()));
        }

        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({"uniquenumber1"}),
                TVector<TString>(),
                TVector<TString>({"cars"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), carId), TStringBuilder() << result);
        }

        {
            auto car = *server->GetDriveAPI()->GetCarsData()->GetObject(carId);
            car.SetNumber("uniquenumber2");
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetCarsData()->UpdateCar(car, "robot-frontend", session) && session.Commit());
            UNIT_ASSERT(server->GetDriveAPI()->GetCarsData()->RefreshCache(Now()));
        }

        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({"uniquenumber2"}),
                TVector<TString>(),
                TVector<TString>({"cars"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), carId), TStringBuilder() << result);
        }

        {
            auto searchResult = gServer.DoSearch(
                TVector<TString>({"uniquenumber1"}),
                TVector<TString>(),
                TVector<TString>({"cars"})
            );
            NJson::TJsonValue result;
            UNIT_ASSERT(searchResult.GetValueByPath("objects.cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), carId), TStringBuilder() << result);
        }
    }

}
