#include <drive/backend/cars/car.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/rt_background/service_routes/common.h>
#include <drive/backend/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>

Y_UNIT_TEST_SUITE(ServiceRoutes) {
    Y_UNIT_TEST(RouteExists) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        const auto& deviceTagManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
        const auto& userTagManager = server->GetDriveAPI()->GetTagsManager().GetUserTags();
        TVector<TEnvironmentGenerator::TCar> cars;
        {
            auto session = deviceTagManager.BuildTx<NSQL::Writable>();
            cars.push_back(eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD"));
            cars.push_back(eGenerator.CreateCar(session, "porsche_carrera", "", "ZXQFDZXQFDZXQFD2"));
            UNIT_ASSERT(session.Commit());
        }
        TSet<TString> carVins;
        for (const auto& car : cars) {
            carVins.insert(car.Vin);
        }
        UNIT_ASSERT(server->GetDriveAPI()->GetCarsData()->RefreshCache(Now()));
        UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RefreshCache(Now()));
        {
            auto session = deviceTagManager.BuildTx<NSQL::Writable>();
            auto tag = MakeAtomicShared<TDeviceTagRecord>("priority_simple1");
            tag->SetComment("dirty0");
            UNIT_ASSERT(deviceTagManager.AddTag(tag, USER_ROOT_DEFAULT, cars[0].Id, server.Get(), session));

            tag = MakeAtomicShared<TDeviceTagRecord>("priority_simple1");
            tag->SetComment("dirty1");
            UNIT_ASSERT(deviceTagManager.AddTag(tag, USER_ROOT_DEFAULT, cars[1].Id, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        auto tag = MakeAtomicShared<TServiceRouteTag>("service_route_tag");
        TVector<TString> tagIds;
        TVector<TInstant> times;
        {
            TVector<TServiceTask> tasks(2);
            TVector<TDBTag> tagsAll;
            for (int i = 0; i < 2; ++i) {
                auto session = deviceTagManager.BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(deviceTagManager.RestoreEntityTags(cars[i].Id, {"priority_simple1"}, tagsAll, session));
                UNIT_ASSERT_VALUES_EQUAL(tagsAll.size(), 1);
                tagIds.push_back(tagsAll[0].GetTagId());
                tasks[i].SetTagId(tagsAll[0].GetTagId());
                times.push_back(Now());
                tasks[i].SetStartTime(times.back());
                times.push_back(Now() + TDuration::Seconds(1));
                tasks[i].SetEndTime(times.back());
            }
            TServiceRoute route;
            route.SetTasks(tasks);
            route.SetRoutingTaskId("aaaabbbbffffcccc");
            tag->SetRoute(route);
        }
        {
            auto session = deviceTagManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(userTagManager.AddTag(tag, USER_ROOT_DEFAULT, USER_ID_TECH, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }

        auto reply = configGenerator.GetServiceRoute(USER_ID_TECH, USER_ID_TECH);
        UNIT_ASSERT_VALUES_EQUAL(reply.Code(), 200);
        NJson::TJsonValue report = NJson::JSON_NULL;
        if (!NJson::ReadJsonFastTree(reply.Content(), &report)) {
            ERROR_LOG << reply.Code() << " : " << reply.Content() << Endl;
        }

        UNIT_ASSERT(report["tasks"].IsArray());
        UNIT_ASSERT_VALUES_EQUAL(report["tasks"].GetArray().size(), 2);
        const auto& arr = report["tasks"].GetArray();
        for (int i = 0; i < 2; ++i) {
            UNIT_ASSERT_VALUES_EQUAL(arr[i]["object_id"].GetStringRobust(), cars[i].Id);
            UNIT_ASSERT_VALUES_EQUAL(arr[i]["tag_id"].GetStringRobust(), tagIds[i]);
            UNIT_ASSERT(arr[i]["start_time"].IsInteger() && arr[i]["end_time"].IsInteger());
            UNIT_ASSERT_VALUES_EQUAL(arr[i]["start_time"].GetInteger(), times[i * 2].Seconds());
            UNIT_ASSERT_VALUES_EQUAL(arr[i]["end_time"].GetInteger(), times[i * 2 + 1].Seconds());
            UNIT_ASSERT_VALUES_EQUAL(arr[i]["tag_details"]["comment"], "dirty" + ToString(i));
        }
        UNIT_ASSERT(report["cars"].IsArray());
        UNIT_ASSERT_VALUES_EQUAL(report["cars"].GetArray().size(), 2);
        const auto& carsArray = report["cars"].GetArray();
        TSet<TString> reportVins;
        for (int i = 0; i < 2; ++i) {
            reportVins.insert(carsArray[i]["vin"].GetString());
        }
        UNIT_ASSERT_VALUES_EQUAL(reportVins, carVins);
        UNIT_ASSERT_VALUES_EQUAL(report["routing_task_id"].GetString(), "aaaabbbbffffcccc");
    }
    Y_UNIT_TEST(RouteDoesNotExist) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits - (ui32)EEnvironmentFeatures::Default);
        auto reply = configGenerator.GetServiceRoute(USER_ID_TECH, USER_ID_TECH);
        UNIT_ASSERT_VALUES_EQUAL(reply.Code(), 200);
        NJson::TJsonValue report = NJson::JSON_NULL;
        if (!NJson::ReadJsonFastTree(reply.Content(), &report)) {
            ERROR_LOG << reply.Code() << " : " << reply.Content() << Endl;
        }
        UNIT_ASSERT_VALUES_EQUAL(report["started_at"].GetInteger(), 0);
        UNIT_ASSERT(!report["tasks"].IsDefined());
    }
    Y_UNIT_TEST(ShiftDescription) {
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(2300, 100);
            description.MutableShiftBoundaries().SetTimezoneShift(3);
            description.MutableCalculateBoundaries().SetTimeRestriction(2250, 100);
            description.MutableCalculateBoundaries().SetTimezoneShift(3);
            UNIT_ASSERT(description.MutableShiftBoundaries().Compile());
            {
                auto timestamp = TInstant::Days(1000) - TDuration::Hours(3) + TDuration::Minutes(10); // 00:10 MSK
                UNIT_ASSERT(description.IsActual(timestamp));
            }
            {
                auto timestamp = TInstant::Days(1000) - TDuration::Hours(3) - TDuration::Minutes(70);      // 22:50 MSK
                UNIT_ASSERT(description.IsActual(timestamp));
                auto [leftBound, rightBound] = description.GetShiftBoundaries(timestamp);
                UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) - TDuration::Hours(3) - TDuration::Hours(1));
                UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) - TDuration::Hours(3) + TDuration::Hours(1));
            }
            {
                auto timestamp = TInstant::Days(1000) - TDuration::Hours(3) - TDuration::Minutes(71);      // 22:49 MSK
                UNIT_ASSERT(!description.IsActual(timestamp));
            }
            {
                auto timestamp = TInstant::Days(1000) - TDuration::Hours(3) + TDuration::Minutes(59);      // 0:59 MSK
                UNIT_ASSERT(description.IsActual(timestamp));
            }
            {
                auto timestamp = TInstant::Days(1000) - TDuration::Hours(3) + TDuration::Hours(1);      // 1:00 MSK
                UNIT_ASSERT(!description.IsActual(timestamp));
            }
        }
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(10, 110);
            description.MutableShiftBoundaries().SetTimezoneShift(3);
            description.MutableCalculateBoundaries().SetTimeRestriction(2350, 110);
            description.MutableCalculateBoundaries().SetTimezoneShift(3);
            auto timestamp = TInstant::Days(1000) - TDuration::Hours(3) - TDuration::Minutes(10); // 23:50 MSK
            UNIT_ASSERT(description.IsActual(timestamp));
        }
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(2350, 110);
            description.MutableShiftBoundaries().SetTimezoneShift(3);
            description.MutableCalculateBoundaries().SetTimeRestriction(2350, 110);
            description.MutableCalculateBoundaries().SetTimezoneShift(3);
            auto timestamp = TInstant::Days(1000) - TDuration::Hours(3) + TDuration::Minutes(10); // 0:10 MSK
            UNIT_ASSERT(description.IsActual(timestamp));
            auto [leftBound, rightBound] = description.GetShiftBoundaries(timestamp);
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) - TDuration::Hours(3) - TDuration::Minutes(10));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) - TDuration::Hours(3) + TDuration::Minutes(70));
        }
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(2350, 20);
            description.MutableShiftBoundaries().SetTimezoneShift(0);
            description.MutableCalculateBoundaries().SetTimeRestriction(2350, 20);
            description.MutableCalculateBoundaries().SetTimezoneShift(0);
            auto timestamp = TInstant::Days(1000) + TDuration::Minutes(10); // 0:10 UTC
            UNIT_ASSERT(description.IsActual(timestamp));
            auto [leftBound, rightBound] = description.GetShiftBoundaries(timestamp);
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) - TDuration::Minutes(10));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Minutes(20));
            std::tie(leftBound, rightBound) = description.GetAdjustedShiftBoundaries(timestamp);
            UNIT_ASSERT_VALUES_EQUAL(leftBound, timestamp);
            UNIT_ASSERT_VALUES_EQUAL(rightBound, rightBound);
        }
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(2350, 10);
            description.MutableShiftBoundaries().SetTimezoneShift(0);
            description.MutableCalculateBoundaries().SetTimeRestriction(2350, 10);
            description.MutableCalculateBoundaries().SetTimezoneShift(0);
            auto timestamp = TInstant::Days(1000) + TDuration::Minutes(20); // 0:20 UTC
            UNIT_ASSERT(!description.IsActual(timestamp));
            auto [leftBound, rightBound] = description.GetShiftBoundaries(timestamp);
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1001) - TDuration::Minutes(10));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1001) + TDuration::Minutes(10));
        }
    }
    Y_UNIT_TEST(ShiftDescriptionTimeWindowRules) {
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(0, 100); // 0:00 - 1:00
            description.MutableShiftBoundaries().SetTimezoneShift(0);
            description.MutableCalculateBoundaries().SetTimeRestriction(0, 100); // 0:00 - 1:00
            description.MutableCalculateBoundaries().SetTimezoneShift(0);
            description.MutableTimeWindowRules().emplace_back().SetAreaId("area1")
                                                            .SetArrangement(NServiceRouting::TTimeWindowRule::EArrangement::Inside)
                                                            .MutableTimeWindow().SetTimeRestriction(30, 50); // 0:30 - 0:50

            auto timestamp = TInstant::Days(1000) + TDuration::Minutes(20); // 0:20 UTC
            UNIT_ASSERT(description.IsActual(timestamp));
            auto [leftBound, rightBound] = description.GetRulesBasedTimeWindow(timestamp, { "area1" });
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Minutes(30));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Minutes(50));

            std::tie(leftBound, rightBound) = description.GetAdjustedRulesBasedTimeWindow(timestamp, { "area1" });
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Minutes(30));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Minutes(50));

            std::tie(leftBound, rightBound) = description.GetRulesBasedTimeWindow(timestamp, { "area2" });
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Hours(1));

            std::tie(leftBound, rightBound) = description.GetAdjustedRulesBasedTimeWindow(timestamp, { "area2" });
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Minutes(20));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Hours(1));
        }
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(0, 100); // 0:00 - 1:00
            description.MutableShiftBoundaries().SetTimezoneShift(0);
            description.MutableCalculateBoundaries().SetTimeRestriction(0, 100); // 0:00 - 1:00
            description.MutableCalculateBoundaries().SetTimezoneShift(0);
            description.MutableTimeWindowRules().emplace_back().SetAreaId("area1")
                                                            .SetArrangement(NServiceRouting::TTimeWindowRule::EArrangement::Outside)
                                                            .MutableTimeWindow().SetTimeRestriction(10, 50); // 0:10 - 0:50

            auto timestamp = TInstant::Days(1000) + TDuration::Minutes(20); // 0:20 UTC
            UNIT_ASSERT(description.IsActual(timestamp));
            auto [leftBound, rightBound] = description.GetRulesBasedTimeWindow(timestamp, {});
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Minutes(10));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Minutes(50));

            std::tie(leftBound, rightBound) = description.GetAdjustedRulesBasedTimeWindow(timestamp, {});
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Minutes(20));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Minutes(50));

            std::tie(leftBound, rightBound) = description.GetRulesBasedTimeWindow(timestamp, { "area1" });
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Hours(1));

            std::tie(leftBound, rightBound) = description.GetAdjustedRulesBasedTimeWindow(timestamp, { "area1" });
            UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Minutes(20));
            UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1000) + TDuration::Hours(1));
        }
        {
            NServiceRouting::TShiftDescription description;
            description.MutableShiftBoundaries().SetTimeRestriction(2200, 300); // 22:00 - 3:00
            description.MutableShiftBoundaries().SetTimezoneShift(0);
            description.MutableCalculateBoundaries().SetTimeRestriction(2200, 300); // 22:00 - 3:00
            description.MutableCalculateBoundaries().SetTimezoneShift(0);
            description.MutableTimeWindowRules().emplace_back().SetAreaId("area1")
                                                            .SetArrangement(NServiceRouting::TTimeWindowRule::EArrangement::Outside)
                                                            .MutableTimeWindow().SetTimeRestriction(2300, 100); // 23:00 - 1:00
            description.MutableTimeWindowRules().emplace_back().SetAreaId("area2")
                                                            .SetArrangement(NServiceRouting::TTimeWindowRule::EArrangement::Inside)
                                                            .MutableTimeWindow().SetTimeRestriction(200, 230); // 2:00 - 2:30

            auto timestamp = TInstant::Days(1000) + TDuration::Hours(22) + TDuration::Minutes(20); // 22:20 UTC
            UNIT_ASSERT(description.IsActual(timestamp));
            // 1st rule
            {
                auto [leftBound, rightBound] = description.GetRulesBasedTimeWindow(timestamp, {});
                UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Hours(23));
                UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1001) + TDuration::Hours(1));

                std::tie(leftBound, rightBound) = description.GetAdjustedRulesBasedTimeWindow(timestamp, {});
                UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Hours(23));
                UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1001) + TDuration::Hours(1));
            }

            // 2nd rule
            {
                auto [leftBound, rightBound] = description.GetRulesBasedTimeWindow(timestamp, { "area1", "area2" });
                UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1001) + TDuration::Hours(2));
                UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1001) + TDuration::Hours(2) + TDuration::Minutes(30));

                std::tie(leftBound, rightBound) = description.GetAdjustedRulesBasedTimeWindow(timestamp, { "area1", "area2" });
                UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1001) + TDuration::Hours(2));
                UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1001) + TDuration::Hours(2) + TDuration::Minutes(30));
            }

            // default
            {
                auto [leftBound, rightBound] = description.GetRulesBasedTimeWindow(timestamp, { "area1" });
                UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Hours(22));
                UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1001) + TDuration::Hours(3));

                std::tie(leftBound, rightBound) = description.GetAdjustedRulesBasedTimeWindow(timestamp, { "area1" });
                UNIT_ASSERT_VALUES_EQUAL(leftBound, TInstant::Days(1000) + TDuration::Hours(22) + TDuration::Minutes(20));
                UNIT_ASSERT_VALUES_EQUAL(rightBound, TInstant::Days(1001) + TDuration::Hours(3));
            }
        }
    }
}
