#include <drive/backend/ut/library/signalq.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/drivematics/signal/manager.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/processors/common_app/fetcher.h>
#include <drive/backend/offers/actions/ut/library/helper.h>
#include <drive/backend/processors/leasing/get_signals.h>
#include <drive/backend/processors/signalq/processor.h>
#include <drive/backend/sessions/manager/billing.h>
#include <drive/backend/signalq/signals/tag.h>

#include <drive/telematics/server/back/pusher.h>
#include <drive/telematics/server/library/config.h>

namespace {

void RegisterEngineSessionMetadata(const NDrive::IServer& server) {
    auto enableEngineSessionsTag = MakeAtomicShared<TTagDescription>();
    enableEngineSessionsTag->SetName("enable_engine_sessions");
    enableEngineSessionsTag->SetType(TDeviceTagRecord::TypeName);
    RegisterTag(server, enableEngineSessionsTag);
    SetSetting(server, "telematics.engine_session.enable", "true");
    SetSetting(server, "telematics.engine_session.evolve_to_riding", "true");
    SetSetting(server, "telematics.engine_session.tag", enableEngineSessionsTag->GetName());
    SetSetting(server, "telematics.engine_session.offer_name", "engine_sessions");
    SetSetting(server, "telematics.engine_session.time_to_finish_session", "30s");
}

bool CheckSessionStarted(TTestEnvironment& env, const size_t secondsToSleepFor, const TString& operatorId) {
    TString stage;
    for (size_t i = 0; i < 10; ++i) {
        auto currentSession = env->GetCurrentSession(operatorId);
        UNIT_ASSERT(currentSession.IsDefined());
        stage = currentSession["segment"]["session"]["current_performing"].GetStringRobust();
        if (stage == TChargableTag::Riding) {
            break;
        }
        Sleep(TDuration::Seconds(secondsToSleepFor));
    }
    return stage == TChargableTag::Riding;
}

TRTDeviceSnapshot BuildSnapshot(const TInstant& statusAt, bool badCameraPose = false, bool cameraClosed = false, bool shutdown = false) {
    NJson::TJsonValue states(NJson::EJsonValueType::JSON_MAP);
    if (badCameraPose) {
        states.InsertValue("dms.analytics.CameraPose", NJson::TJsonValue(true));
    }
    if (cameraClosed) {
        states.InsertValue("dms.analytics.TrashFrames", NJson::TJsonValue(true));
    }
    NDrive::NSignalq::TStatus status;
    status.SetStatusAt(std::move(statusAt));
    status.SetStates(std::move(states));
    TRTDeviceSnapshot result;
    if (shutdown) {
        NDrive::NSignalq::TEvent event;
        event.SetId("some_id");
        event.SetAt(Now());
        event.SetType("shutdown");
        result.SetLastSignalqEvent(std::move(event));
    }
    result.SetSignalqStatus(std::move(status));
    return result;
}

}

Y_UNIT_TEST_SUITE(Signalq) {

    Y_UNIT_TEST(GetSignalqStatus) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSignalqSessionMetadata(*env.GetServer().Get());

        const auto carId = OBJECT_ID_DEFAULT;
        const auto userId = USER_ID_DEFAULT;
        const auto signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;

        env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT);
        env->AddTag(new TDeviceTagRecord("enable_signalq_events"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
        env->AddTag(new TDeviceTagRecord("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        NJson::TJsonValue postData1 = NJson::JSON_MAP;
        NJson::TJsonValue& events1 =  postData1.InsertValue("events", NJson::JSON_ARRAY);
        events1.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "smoking", "id1", Now()));
        events1.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "tired", "id2", Now()));
        env->SignalqSignalCreate(postData1, USER_ROOT_DEFAULT);

        auto permissions = env.GetServer()->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
        TCarsFetcher fetcher1(*env.GetServer().Get(), NDeviceReport::EReportTraits::ReportSignalq);
        UNIT_ASSERT(fetcher1.FetchData(permissions, {carId}));

        const auto snapshot1 = BuildSnapshot(Now() - TDuration::Days(1000));
        const auto signalqStatus1 = fetcher1.MakeSignalqStatus(&snapshot1);
        UNIT_ASSERT(signalqStatus1 == TCarsFetcher::ESignalqStatus::Offline);

        const auto snapshot2 = BuildSnapshot(Now(), true);
        const auto signalqStatus2 = fetcher1.MakeSignalqStatus(&snapshot2);
        UNIT_ASSERT(signalqStatus2 == TCarsFetcher::ESignalqStatus::FacedAway);

        const auto snapshot3 = BuildSnapshot(Now(), false, true);
        const auto signalqStatus3 = fetcher1.MakeSignalqStatus(&snapshot3);
        UNIT_ASSERT(signalqStatus3 == TCarsFetcher::ESignalqStatus::CameraClosed);

        const auto snapshot4 = BuildSnapshot(Now());
        const auto signalqStatus4 = fetcher1.MakeSignalqStatus(&snapshot4);
        UNIT_ASSERT(signalqStatus4 == TCarsFetcher::ESignalqStatus::TurnedOn);

        NJson::TJsonValue postData2 = NJson::JSON_MAP;
        NJson::TJsonValue& events2 =  postData2.InsertValue("events", NJson::JSON_ARRAY);
        events2.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "shutdown", "id3", Now()));
        env->SignalqSignalCreate(postData2, USER_ROOT_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        TCarsFetcher fetcher2(*env.GetServer().Get(), NDeviceReport::EReportTraits::ReportSignalq);
        UNIT_ASSERT(fetcher2.FetchData(permissions, {carId}));

        const auto snapshot5 = BuildSnapshot(Now(), false, false, true);
        const auto signalqStatus5 = fetcher2.MakeSignalqStatus(&snapshot5);
        UNIT_ASSERT(signalqStatus5 == TCarsFetcher::ESignalqStatus::TurnedOff);
    }

    Y_UNIT_TEST(StartSession) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        RegisterSignalqSessionMetadata(*env.GetServer());

        const auto carId = OBJECT_ID_DEFAULT;
        const auto signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;

        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));

        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(!CheckSessionStarted(env, 0, USER_ROOT_DEFAULT));

        UNIT_ASSERT(!env->SignalqSessionsStart(signalDeviceSerialNumber, USER_ROOT_DEFAULT).Has("error_details"));

        UNIT_ASSERT(CheckSessionStarted(env, 10, USER_ROOT_DEFAULT));
    }

    Y_UNIT_TEST(DoNotStartSession) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        RegisterSignalqSessionMetadata(*env.GetServer());
        RegisterEngineSessionMetadata(*env.GetServer());

        const auto carId = OBJECT_ID_DEFAULT;
        const auto signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;

        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));

        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_engine_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        UNIT_ASSERT(!CheckSessionStarted(env, 0, USER_ROOT_DEFAULT));

        UNIT_ASSERT(!env->SignalqSessionsStart(signalDeviceSerialNumber, USER_ROOT_DEFAULT).Has("error_details"));

        UNIT_ASSERT(!CheckSessionStarted(env, 1, USER_ROOT_DEFAULT));
    }

    Y_UNIT_TEST(DropEngineSessionAndStartSignalqSession) {
        TTestEnvironment env;
        {
            auto backendPusherOptions = MakeHolder<NDrive::TBackendPusherOptions>();
            backendPusherOptions->Endpoint = TStringBuilder() << "http://localhost:" << env.GetConfig().GetHttpServerOptions().Port << "/api/telematics/push";
            backendPusherOptions->SensorIds.insert(CAN_ENGINE_IS_ON);
            env.GetTelematicServerBuilder().GetConfig()->AddPusher(std::move(backendPusherOptions));
        }
        env.Execute(NDrive::NTest::TBuildEnv());

        const auto& server = *env.GetServer();
        RegisterEngineSessionMetadata(server);
        RegisterSignalqSessionMetadata(server);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto carId = OBJECT_ID_DEFAULT;

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<TDeviceTagRecord>("enable_engine_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        auto emulator = env.GetContext().GetEmulator();
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(true));

        UNIT_ASSERT(CheckSessionStarted(env, 10, USER_ID_DEFAULT));

        UNIT_ASSERT(env->RemoveTags(TVector<TString>{"enable_engine_sessions"}, carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));

        const auto signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;

        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<TDeviceTagRecord>("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto driveApi = server.GetDriveAPI();

        auto offer = driveApi->GetCurrentCarOffer(carId);
        TString offerName;
        if (offer) {
            offerName = offer->GetName();
        }

        UNIT_ASSERT(offerName == "engine_sessions");

        UNIT_ASSERT(!env->SignalqSessionsStart(signalDeviceSerialNumber, USER_ROOT_DEFAULT).Has("error_details"));

        UNIT_ASSERT(CheckSessionStarted(env, 10, USER_ROOT_DEFAULT));

        offer = driveApi->GetCurrentCarOffer(carId);
        if (offer) {
            offerName = offer->GetName();
        }

        UNIT_ASSERT(offerName == "signalq_sessions");
    }

    Y_UNIT_TEST(SignalCreatePastSession) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSignalqSessionMetadata(*env.GetServer().Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& api = *Yensured(env.GetServer()->GetDriveAPI());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        const TString signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;
        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_events"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& sessionManager = api.GetSessionManager();
        const auto& traceTagManager = api.GetTagsManager().GetTraceTags();
        auto offerId = env->CreateOffer(carId, userId);
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        const auto timestampEvent1 = Now();
        Sleep(TDuration::Seconds(1));
        const auto timestampEvent2 = Now();
        Sleep(TDuration::Seconds(1));
        const auto timestampEvent3 = Now();
        Sleep(TDuration::Seconds(1));
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            auto session = sessionManager.GetSession(offerId, tx);
            UNIT_ASSERT(!(*session)->GetClosed());
        }
        env->EvolveTag("old_state_reservation", userId);
        const auto endSessionTimestamp = Now();
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            auto session = sessionManager.GetSession(offerId, tx);
            UNIT_ASSERT((*session)->GetClosed());
        }

        NJson::TJsonValue postData = NJson::JSON_MAP;
        NJson::TJsonValue& events =  postData.InsertValue("events", NJson::JSON_ARRAY);
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "id1", timestampEvent1));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "id2", timestampEvent2));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "tired", "id3", timestampEvent3));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "event_after_ride", endSessionTimestamp + TDuration::Seconds(1)));
        UNIT_ASSERT(!env->SignalqSignalCreate(postData, USER_ROOT_DEFAULT).Has("error_details"));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));

        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            const auto taggedSession = traceTagManager.RestoreObject(offerId, tx);
            UNIT_ASSERT(taggedSession);
            const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 4);
        }
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            NDrivematics::TSignalManager signalManager;
            auto signals = signalManager.Fetch(tx, NDrivematics::TSignalManager::TQueryOptions());
            UNIT_ASSERT(signals);
            UNIT_ASSERT_VALUES_EQUAL(signals->size(), 4);
        }
    }

    Y_UNIT_TEST(SignalCreateActualSession) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSignalqSessionMetadata(*env.GetServer().Get());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& api = *Yensured(env.GetServer()->GetDriveAPI());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        const TString signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;
        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_events"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& traceTagManager = api.GetTagsManager().GetTraceTags();

        auto offerId = env->CreateOffer(carId, userId);
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        const auto bookedTimestamp = Now();
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        const auto timestamp = Now();
        Sleep(TDuration::Seconds(1));
        NJson::TJsonValue postData = NJson::JSON_MAP;
        NJson::TJsonValue& events =  postData.InsertValue("events", NJson::JSON_ARRAY);
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "event_id_with_actual_session_gap", bookedTimestamp - TDuration::Seconds(20)));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "tired", "id2", timestamp));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "id3", timestamp));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bestshot", "id4", timestamp));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bestshot", "id4", timestamp)); //duplicate
        UNIT_ASSERT(!env->SignalqSignalCreate(postData, USER_ROOT_DEFAULT).Has("error_details"));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            const auto taggedSession = traceTagManager.RestoreObject(offerId, tx);
            UNIT_ASSERT(taggedSession);
            const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 4);
        }
        env->EvolveTag("old_state_reservation", userId);
    }

    Y_UNIT_TEST(SignalCreateNewSession) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSignalqSessionMetadata(*env.GetServer().Get());

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& api = *Yensured(env.GetServer()->GetDriveAPI());

        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        const TString signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;
        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_events"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& traceTagManager = api.GetTagsManager().GetTraceTags();
        const auto offer = env.GetServer()->GetDriveAPI()->GetCurrentCarOffer(carId);
        UNIT_ASSERT(!offer);
        NJson::TJsonValue postData = NJson::JSON_MAP;
        NJson::TJsonValue& events =  postData.InsertValue("events", NJson::JSON_ARRAY);
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "event_id_with_actual_session_gap", Now()));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "tired", "id2", Now()));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", "id3", Now()));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bestshot", "id4", Now()));
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bestshot", "id4", Now())); //duplicate
        UNIT_ASSERT(!env->SignalqSignalCreate(postData, USER_ROOT_DEFAULT).Has("error_details"));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            const auto offer = env.GetServer()->GetDriveAPI()->GetCurrentCarOffer(carId);
            UNIT_ASSERT(offer);
            const auto& offerId = offer->GetOfferId();
            const auto taggedSession = traceTagManager.RestoreObject(offerId, tx);
            UNIT_ASSERT(taggedSession);
            const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 4);
        }
        env->EvolveTag("old_state_reservation", userId);
    }

    Y_UNIT_TEST(NotifySupport) {
        InitGlobalLog2Console(TLOG_DEBUG);
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);

        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        {
            const TString coordsArea = "36.50571688 57.96862233 38.89798983 59.28069963 37.73069247 59.82167053 41.37266024 60.48976579 46.98667391 57.62840898 45.115325 53.61686361 38.14631985 51.43302596 35.14705227 52.77246007 31.93355129 55.45020355 31.26887844 55.89103786 36.50571688 57.96862233";
            TVector<TGeoCoord> area;
            UNIT_ASSERT(TGeoCoord::DeserializeVector(coordsArea, area));
            UNIT_ASSERT(configGenerator.UpsertArea("test_msk_area", USER_ROOT_DEFAULT, area, {}));

            THolder<TAreaInfoForUser> tagUserInfo(new TAreaInfoForUser("common_area_user_info"));
            tagUserInfo->MutableInfos().emplace("phone", "phone");
            tagUserInfo->MutableInfos().emplace("support_feedback_form", "form");
            tagUserInfo->MutableInfos().emplace("support_phone", "phone");
            tagUserInfo->MutableInfos().emplace("support_telegram", "phone");
            tagUserInfo->MutableInfos().emplace("support_whatsapp", "phone");
            UNIT_ASSERT(configGenerator.AddTag(tagUserInfo.Release(), "test_msk_area", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Area));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }

        TEnvironmentGenerator::TCar car;
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            car = eGenerator.CreateCar(session, "porsche_carrera", "", "z94c251ABJR008606");
            UNIT_ASSERT(session.Commit());
        }

        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        NDrive::TTelematicsTestClient::TPtr client = emulator->GetClient();
        UNIT_ASSERT(configGenerator.WaitCar(car.Id));
        TString offerId;
        {
            {
                auto offer = BuildOfferPtr(200, 100, 102400);
                offer->SetObjectId(car.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
                offer->SetChargableAccounts({ "card", "bonus" });
                offerId = offer->GetOfferId();
                UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}));
            }

            {
                UNIT_ASSERT(configGenerator.BookOffer(offerId, USER_ID_DEFAULT));
                NDrive::TServerConfigGenerator::TSessionStateGuard sg(configGenerator, "book_offer");
                NJson::TJsonValue currentSession = configGenerator.GetCurrentSession(USER_ID_DEFAULT);
                UNIT_ASSERT(!currentSession["notification"].IsDefined());
            }

        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        NJson::TJsonValue postData = NJson::JSON_MAP;
        NJson::TJsonValue& items =  postData.InsertValue("items", NJson::JSON_ARRAY);
        NJson::TJsonValue item = NJson::JSON_MAP;
        item["vin"] = car.Vin;
        const auto timestamp = Now();
        NJson::TJsonMap eventJson ({
            {"at", timestamp.Seconds()},
            {"type", "smoking"},
            {"video_url", "some_url"},
            {"photo_url", "some_url"},
            {"external_video_url", "some_url"},
            {"external_photo_url", "some_url"},
            {"alarm_raise_url", "some_url"},
        });
        item["event"] = std::move(eventJson);
        items.AppendValue(item);

        item["event"]["type"] = "eyeclose";
        items.AppendValue(item);

        item["vin"] = "unregistred_vin";
        items.AppendValue(std::move(item));

        const auto response = configGenerator.SignalqNotifySupport(postData, USER_ROOT_DEFAULT);
        UNIT_ASSERT(!response.Has("error_details"));
        UNIT_ASSERT_VALUES_EQUAL(response["added_tags_amount"].GetUInteger(), 2);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            auto taggedUser = driveApi.GetTagsManager().GetUserTags().RestoreObject(USER_ID_DEFAULT,tx);
            UNIT_ASSERT(taggedUser);
            const auto tags = taggedUser->GetTagsByClass<TSupportOutgoingCommunicationTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 2);
        }
    }

    Y_UNIT_TEST(SignalqSetResolutionFromFleet) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        RegisterSignalqSessionMetadata(*env.GetServer().Get());

        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& api = *Yensured(env.GetServer()->GetDriveAPI());
        const auto carId = TString{OBJECT_ID_DEFAULT};
        const auto userId = TString{USER_ID_DEFAULT};
        const TString signalDeviceSerialNumber = SIGNAL_DEVICE_SN_DEFAULT;
        UNIT_ASSERT(!env->ServiceAppAssignDevice(carId, signalDeviceSerialNumber, true, USER_ROOT_DEFAULT).Has("error_details"));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_events"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        UNIT_ASSERT(env->AddTag(new TDeviceTagRecord("enable_signalq_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        const auto& traceTagManager = api.GetTagsManager().GetTraceTags();
        const auto offerId = env->CreateOffer(carId, userId);
        UNIT_ASSERT(offerId);
        UNIT_ASSERT(env->BookOffer(offerId, userId));
        NJson::TJsonValue postData = NJson::JSON_MAP;
        NJson::TJsonValue& events =  postData.InsertValue("events", NJson::JSON_ARRAY);
        const auto eventAt = Now();
        const TString eventId = "event_id_with_actual_session_gap";
        events.AppendValue(env->SignalqCreateJsonEvent(signalDeviceSerialNumber, "bad_camera_pose", eventId, eventAt));
        env->SignalqSignalCreate(postData, USER_ROOT_DEFAULT);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        TString tagId;
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            const auto taggedSession = traceTagManager.RestoreObject(offerId, tx);
            UNIT_ASSERT(taggedSession);
            const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();
            auto signalqTraceTag = tags.front().GetTagAs<TSignalqEventTraceTag>();
            UNIT_ASSERT(!signalqTraceTag->GetResolution());
        }
        env->EvolveTag("old_state_reservation", userId);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        UNIT_ASSERT(!env->SignalqTraceTagResolutionSet(signalDeviceSerialNumber, eventId, eventAt, "wrong_event", USER_ROOT_DEFAULT).Has("error_details"));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        Sleep(TDuration::Seconds(1));
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            const auto taggedSession = traceTagManager.RestoreObject(offerId, tx);
            UNIT_ASSERT(taggedSession);
            const auto tags = taggedSession->GetTagsByClass<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();
            auto signalqTraceTag = tags.front().GetTagAs<TSignalqEventTraceTag>();
            UNIT_ASSERT_VALUES_EQUAL(signalqTraceTag->GetResolution(), "wrong_event");
        }
        {
            auto tx = api.BuildTx<NSQL::ReadOnly>();
            NDrivematics::TSignalManager signalManager;
            auto queryOptions = NDrivematics::TSignalManager::TQueryOptions()
                .SetIds({ tagId })
            ;
            auto signals = signalManager.Fetch(tx, std::move(queryOptions));
            UNIT_ASSERT(signals);
            UNIT_ASSERT_VALUES_EQUAL(signals->size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(signals->at(0).Resolution, "wrong_event");
        }
    }
}
