#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>

#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/rt_background/billing/bonus_tags.h>

#include <rtline/library/storage/structured.h>


namespace {

    class TCheckCoins: public NDrive::NTest::ITestAction {
        using TBase = ITestAction;

    private:
        R_FIELD(ui32, Amount, 0);

    protected:
        virtual void DoExecute(NDrive::NTest::TRTContext& context) override {
            const auto& billingManager = context.GetDriveAPI().GetBillingManager();
            const auto& accountsManager = billingManager.GetAccountsManager();
            auto coinsDescription = accountsManager.GetDescriptionByName(ToString(NDrive::NBilling::EWalletDataType::Coins));
            auto tx = billingManager.BuildSession(/* readOnly = */ true);
            auto accounts = accountsManager.GetUserAccounts(USER_ID_DEFAULT, tx);
            UNIT_ASSERT(accounts);
            UNIT_ASSERT(accounts->size() > 1);
            NDrive::NBilling::IBillingAccount::TPtr bonusAccount = accountsManager.GetOrCreateAccount(USER_ID_DEFAULT, coinsDescription.GetRef(), USER_ID_DEFAULT, tx);
            UNIT_ASSERT(!!bonusAccount);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetType(), NDrive::NBilling::EAccount::Coins);
            UNIT_ASSERT_VALUES_EQUAL(bonusAccount->GetBalance(), Amount);
        }

    public:
        TCheckCoins(const TInstant instant, ui32 amount)
            : TBase(instant)
            , Amount(amount)
        {
        }
    };

    class TCheckBillTasks: public NDrive::NTest::ITestAction {
        using TBase = ITestAction;

    private:
        R_FIELD(size_t, Count, 0);

    protected:
        virtual void DoExecute(NDrive::NTest::TRTContext& context) override {
            auto& billingManager = context.GetDriveAPI().GetBillingManager();
            auto tx = billingManager.BuildSession(/* readOnly = */ true);
            auto dbTasks = billingManager.GetActiveTasksManager().GetUsersTasks({ USER_ID_DEFAULT }, tx);
            UNIT_ASSERT(dbTasks);
            UNIT_ASSERT_VALUES_EQUAL(dbTasks->size(), Count);
        }

    public:
        TCheckBillTasks(const TInstant instant, ui32 count)
            : TBase(instant)
            , Count(count)
        {
        }
    };

    void ProcessPayment(NDrive::NTest::TRTContext& context, const TString& subscriptionTagName, TMaybe<TUserSubscriptionTag::EPaymentStatus> expected) {
        const auto& userTags = context.GetDriveAPI().GetTagsManager().GetUserTags();
        auto tx = userTags.BuildTx<NSQL::Writable>();
        auto dbTags = userTags.RestoreTagsRobust({ USER_ID_DEFAULT }, { subscriptionTagName }, tx);
        UNIT_ASSERT(dbTags);
        UNIT_ASSERT(!dbTags->empty());
        auto subsTag = dbTags->front().MutableTagAs<TUserSubscriptionTag>();
        UNIT_ASSERT(subsTag);
        if (expected) {
            UNIT_ASSERT_VALUES_EQUAL(subsTag->CheckPayment(dbTags->front(), USER_ID_DEFAULT, context.GetServer().Get(), tx), *expected);
        } else {
            UNIT_ASSERT(subsTag->ApplyPayment(dbTags->front(), USER_ID_DEFAULT, context.GetServer().Get(), tx));
        }
        UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
    }

    class TPaymentApplier: public NDrive::NTest::ITestAction {
        using TBase = ITestAction;

    private:
        R_FIELD(TString, SubscriptionTagName);

    protected:
        virtual void DoExecute(NDrive::NTest::TRTContext& context) override {
            ProcessPayment(context, SubscriptionTagName, {});
        }

    public:
        TPaymentApplier(const TInstant instant, const TString& subscriptionTagName)
            : TBase(instant)
            , SubscriptionTagName(subscriptionTagName)
        {
        }
    };

    class TPaymentChecker: public NDrive::NTest::ITestAction {
        using TBase = ITestAction;

    private:
        R_FIELD(TString, SubscriptionTagName);
        R_FIELD(TUserSubscriptionTag::EPaymentStatus, ExpectedStatus, TUserSubscriptionTag::EPaymentStatus::Unknown);

    protected:
        virtual void DoExecute(NDrive::NTest::TRTContext& context) override {
            ProcessPayment(context, SubscriptionTagName, ExpectedStatus);
        }

    public:
        TPaymentChecker(
            const TInstant instant
            , const TString& subscriptionTagName
            , TUserSubscriptionTag::EPaymentStatus expected
        )
            : TBase(instant)
            , SubscriptionTagName(subscriptionTagName)
            , ExpectedStatus(expected)
        {
        }
    };

    class TProcessBillingTags: public NDrive::NTest::ITestAction {
        using TBase = ITestAction;

    private:
        R_FIELD(ui32, Iterations, 0);

        const TRTBackgroundProcessContainer& Container;

    protected:
        virtual void DoExecute(NDrive::NTest::TRTContext& context) override {
            for (ui32 i = 0; i < Iterations; ++i) {
                UNIT_ASSERT(Container->Execute(nullptr, {*context.GetServer().Get()}));
            }
        }

    public:
        TProcessBillingTags(const TInstant instant, ui32 iterations, const TRTBackgroundProcessContainer& container)
            : TBase(instant)
            , Iterations(iterations)
            , Container(container)
        {
        }
    };

    TRTBackgroundProcessContainer BuildRTBackground() {
        TRTBackgroundProcessContainer container;
        container.SetName("billing");
        TAtomicSharedPtr<TRTBillingTagsWatcher> actor(new TRTBillingTagsWatcher);
        actor->SetPeriod(TDuration::Seconds(1));
        actor->SetEnabled(true);
        actor->SetRobotUserId(USER_ROOT_DEFAULT);
        container.SetProcessSettings(actor);
        return container;
    }
}

Y_UNIT_TEST_SUITE(SubscriptionTags) {

    struct TRegisteredTags {
        const TString DebitTagName;
        const TString CreditTagName;
        const TString SubscriptionTagName;
    };

    const ui32 CoinsAmount = 100000;

    TRegisteredTags PrepareTags(NDrive::NTest::TScript& script) {
        TRegisteredTags result = {
            "debit_tag_operation",
            "credit_tag_operation",
            "subscription_tag_flow",
        };
        auto registerBillingTag = [](const TString& tagName,
            TBillingTagDescription::EAction action,
            EBillingType terminal,
            const TString& account,
            NDrive::NTest::TScript& script)
        {
            NJson::TJsonValue descJson;
            descJson["type"] = TOperationTag::TypeName;
            descJson["name"] = tagName;
            descJson["display_name"] = "test";
            descJson["comment"] = "";
            NJson::TJsonValue metaJson;
            metaJson["billing_queue"] = ToString(EBillingQueue::Tests);
            metaJson["action"] = ToString(action);
            metaJson["terminal"] = ToString(terminal);
            metaJson["accounts"].AppendValue(account);
            descJson["meta"] = metaJson;
            script.Add<NDrive::NTest::TRegisterTag>(descJson);
        };
        registerBillingTag(
            result.DebitTagName,
            TBillingTagDescription::EAction::Debit,
            EBillingType::CarUsagePrepayment,
            "y.drive",
            script
        );
        registerBillingTag(
            result.CreditTagName,
            TBillingTagDescription::EAction::Credit,
            EBillingType::CarUsage,
            ToString(NDrive::NBilling::EWalletDataType::Coins),
            script
        );
        {
            NJson::TJsonValue descJson;
            descJson["type"] = TUserSubscriptionTag::TypeName;
            descJson["name"] = result.SubscriptionTagName;
            descJson["display_name"] = "test";
            descJson["comment"] = "";
            NJson::TJsonValue metaJson;
            metaJson["priority"] = 0;
            metaJson["debit_tag"] = result.DebitTagName;
            metaJson["credit_tag"] = result.CreditTagName;
            metaJson["description"]["price"] = CoinsAmount;
            descJson["meta"] = metaJson;
            script.Add<NDrive::NTest::TRegisterTag>(descJson);
        }
        return result;
    }

    enum ESubsState {
        Empty,
        DebitOnly,
        CreditOnly,
    };

    void AddEmptyTagsChecks(NDrive::NTest::TScript& script, const TString& debitTagName, const TString& creditTagName, ESubsState state) {
        switch (state) {
            case ESubsState::Empty:
                script.Add<NDrive::NTest::TCheckHasTag>()
                    .SetTagName(debitTagName).SetObjectId(USER_ID_DEFAULT)
                    .SetEntityType(NEntityTagsManager::EEntityType::User)
                    .SetExpectOK(false);
                script.Add<NDrive::NTest::TCheckHasTag>()
                    .SetTagName(creditTagName)
                    .SetObjectId(USER_ID_DEFAULT)
                    .SetEntityType(NEntityTagsManager::EEntityType::User)
                    .SetExpectOK(false);
                break;
            case ESubsState::DebitOnly:
                script.Add<NDrive::NTest::TCheckHasTag>()
                    .SetTagName(debitTagName).SetObjectId(USER_ID_DEFAULT)
                    .SetEntityType(NEntityTagsManager::EEntityType::User)
                    .SetTargetCount(1);
                script.Add<NDrive::NTest::TCheckHasTag>()
                    .SetTagName(creditTagName)
                    .SetObjectId(USER_ID_DEFAULT)
                    .SetEntityType(NEntityTagsManager::EEntityType::User)
                    .SetExpectOK(false);
                break;
            case ESubsState::CreditOnly:
                script.Add<NDrive::NTest::TCheckHasTag>()
                    .SetTagName(debitTagName).SetObjectId(USER_ID_DEFAULT)
                    .SetEntityType(NEntityTagsManager::EEntityType::User)
                    .SetExpectOK(false);
                script.Add<NDrive::NTest::TCheckHasTag>()
                    .SetTagName(creditTagName)
                    .SetObjectId(USER_ID_DEFAULT)
                    .SetEntityType(NEntityTagsManager::EEntityType::User)
                    .SetTargetCount(1);
                break;
        }
    }

    TRegisteredTags CommonSetUp(NDrive::NTest::TScript& script) {
        script.Add<NDrive::NTest::TBuildEnv>(TDuration::Zero());

        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>("y.drive");
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>("y.drive").SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);

        const auto registeredTags = PrepareTags(script);
        const auto& [debitTagName, creditTagName, subscriptionTagName] = registeredTags;
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);
        script.Add<NDrive::NTest::TAddTag>(new TUserSubscriptionTag(subscriptionTagName)).SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User);
        script.Add<NDrive::NTest::TCheckHasTag>().SetTagName(subscriptionTagName).SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User);
        script.Add<TCheckCoins>(0);
        script.Add<TCheckBillTasks>(0);
        return registeredTags;
    }

    Y_UNIT_TEST(SimpleFlow) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator, Now());
        const auto& [debitTagName, creditTagName, subscriptionTagName] = CommonSetUp(script);

        script.Add<TPaymentApplier>(subscriptionTagName);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::DebitOnly);

        auto container = BuildRTBackground();
        script.Add<TProcessBillingTags>(1, container);
        script.Add<TCheckBillTasks>(1);
        script.Add<TProcessBillingTags>(1, container);
        script.Add<NDrive::NTest::TRunBillingCycle>();
        script.Add<TCheckBillTasks>(0);
        script.Add<TProcessBillingTags>(3, container);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);

        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Processing);
        script.Add<TCheckBillTasks>(0);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::CreditOnly);
        script.Add<TProcessBillingTags>(2, container);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);

        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Completed);
        script.Add<TCheckCoins>(CoinsAmount);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(RepeatableFlow) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator, Now());
        const auto& [debitTagName, creditTagName, subscriptionTagName] = CommonSetUp(script);

        script.Add<TPaymentApplier>(subscriptionTagName);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::DebitOnly);
        script.Add<TPaymentApplier>(subscriptionTagName);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::DebitOnly);
        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Processing);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::DebitOnly);

        auto container = BuildRTBackground();
        script.Add<TProcessBillingTags>(1, container);
        script.Add<TCheckBillTasks>(1);
        script.Add<TProcessBillingTags>(1, container);
        script.Add<NDrive::NTest::TRunBillingCycle>();
        script.Add<TCheckBillTasks>(0);
        script.Add<TProcessBillingTags>(3, container);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);

        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Processing);
        script.Add<TCheckBillTasks>(0);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::CreditOnly);

        script.Add<TPaymentApplier>(subscriptionTagName);
        script.Add<TCheckBillTasks>(0);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::CreditOnly);

        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Processing);
        script.Add<TCheckBillTasks>(0);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::CreditOnly);

        script.Add<TProcessBillingTags>(2, container);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);
        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Completed);
        script.Add<TPaymentApplier>(subscriptionTagName);
        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Completed);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);
        script.Add<TCheckCoins>(CoinsAmount);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(ProlongationFlow) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator, Now());
        const auto& [debitTagName, creditTagName, subscriptionTagName] = CommonSetUp(script);

        script.Add<TPaymentApplier>(subscriptionTagName);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::DebitOnly);

        auto container = BuildRTBackground();
        script.Add<TProcessBillingTags>(2, container);
        script.Add<NDrive::NTest::TRunBillingCycle>();
        script.Add<TProcessBillingTags>(3, container);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);
        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Processing);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::CreditOnly);
        script.Add<TProcessBillingTags>(2, container);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);
        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Completed);
        script.Add<TCheckCoins>(CoinsAmount);

        script.Add<NDrive::NTest::TCommonChecker>([&subscriptionTagName = subscriptionTagName](NDrive::NTest::TRTContext& context) {
            const auto& userTags = context.GetDriveAPI().GetTagsManager().GetUserTags();
            auto tx = userTags.BuildTx<NSQL::Writable>();
            auto dbTags = userTags.RestoreTagsRobust({ USER_ID_DEFAULT }, { subscriptionTagName }, tx);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            auto subsTag = dbTags->front().MutableTagAs<TUserSubscriptionTag>();
            UNIT_ASSERT(subsTag);
            subsTag->SetSLAInstant(Now());
            UNIT_ASSERT_C(userTags.UpdateTagsData(*dbTags, USER_ROOT_DEFAULT, tx), tx.GetStringReport());
            UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
        });

        script.Add<TPaymentApplier>(subscriptionTagName);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::DebitOnly);
        script.Add<TProcessBillingTags>(2, container);
        script.Add<NDrive::NTest::TRunBillingCycle>();
        script.Add<TProcessBillingTags>(3, container);
        script.Add<TCheckBillTasks>(0);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);
        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Processing);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::CreditOnly);
        script.Add<TProcessBillingTags>(2, container);
        AddEmptyTagsChecks(script, debitTagName, creditTagName, ESubsState::Empty);
        script.Add<TPaymentChecker>(subscriptionTagName, TUserSubscriptionTag::EPaymentStatus::Completed);
        script.Add<TCheckCoins>(CoinsAmount * 2);

        UNIT_ASSERT(script.Execute());
    }
}
