#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/state/robot_state.pb.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/registrar/hasher.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/support_center/request_filter.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/uuid.h>

Y_UNIT_TEST_SUITE(SupportCenter) {

    Y_UNIT_TEST(DeferredOutgoingCommunication) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("zxqfd-was-there");
        TString tagId;
        UNIT_ASSERT(configGenerator.AddTag(new TSupportOutgoingCommunicationTag("test_outgoing_communication"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"test_outgoing_communication"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();
        }

        UNIT_ASSERT(configGenerator.DeferRequest(tagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10)));

        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"test_outgoing_communication"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TUserContainerTag::TypeName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);

            auto tagImpl = tags.front().MutableTagAs<TUserContainerTag>();
            auto permissions = server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
            UNIT_ASSERT(tagImpl->OnSLAExpired(tags.front(), *permissions, &server->GetDriveAPI()->GetTagsManager().GetUserTags(), server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }

        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"test_outgoing_communication"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }

        UNIT_ASSERT(configGenerator.DeferRequest(tagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10), "", "test comment 123"));
        {
            auto messages = configGenerator.GetChatMessages(userId, "outgoing_communication." + tagId, USER_ROOT_DEFAULT);
            UNIT_ASSERT(messages["messages"].IsArray());
            UNIT_ASSERT(messages["messages"].GetArray().size() > 0);
            auto lastMessage = messages["messages"].GetArray()[messages["messages"].GetArray().size() - 1];
            UNIT_ASSERT_VALUES_EQUAL(lastMessage["text"].GetString(), "test comment 123");
        }
    }

    Y_UNIT_TEST(PretermUndefer) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("zxqfd-was-there");
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {}));

        TString tagId;
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();

            tags.clear();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }

        UNIT_ASSERT(configGenerator.DeferRequest(tagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10)));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {}));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            auto tag = tags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), "");
            tags.clear();

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }
    }

    Y_UNIT_TEST(UndeferTimedRequest) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        TString chatId = "support.chatundeferrequest123";
        auto userId = eGenerator.CreateUser("zxqfd-was-there");
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, chatId, "Hello, Yandex.Drive!", {}));

        TString tagId;
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();

            tags.clear();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }

        UNIT_ASSERT(configGenerator.DeferRequest(tagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10)));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }

        UNIT_ASSERT(configGenerator.UndeferRequest(tagId, chatId, USER_ROOT_DEFAULT));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            auto tag = tags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), "");
            tags.clear();

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }
    }

    Y_UNIT_TEST(UndeferUntilMessageRequest) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        TString chatId = "support.chatundefermessagerequest123";
        auto userId = eGenerator.CreateUser("zxqfd-was-there");
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, chatId, "Hello, Yandex.Drive!", {}));
        TString tagId;
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();
        }
        UNIT_ASSERT(configGenerator.DeferRequest(tagId, USER_ROOT_DEFAULT));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }
        UNIT_ASSERT(configGenerator.UndeferRequest(tagId, chatId, USER_ROOT_DEFAULT));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }
    }

    Y_UNIT_TEST(UndeferUserChatSLA) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("zxqfd-was-there");
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {}));
        TString tagId;
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            tagId = tags.front().GetTagId();
        }
        UNIT_ASSERT(configGenerator.DeferRequest(tagId, USER_ROOT_DEFAULT, Now() + TDuration::Minutes(30)));
        TDBTag containerTag;
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            containerTag = tags.front();
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto cTag = containerTag.MutableTagAs<TUserContainerTag>();
            UNIT_ASSERT(cTag);
            auto userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ID_DEFAULT, TUserPermissionsFeatures());
            UNIT_ASSERT(cTag->OnSLAExpired(containerTag, *userPermissions, &server->GetDriveAPI()->GetTagsManager().GetUserTags(), server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"support_chat_2_line"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            auto tag = tags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), "");
            tags.clear();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }
    }

    Y_UNIT_TEST(CategorizationSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto requestId = NUtil::CreateUUID();
        {
            auto existing = configGenerator.GetCategorizations({requestId}, USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray().size(), 0);
        }

        UNIT_ASSERT(configGenerator.AddCategorization(requestId, "category 1", "comment 1", USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->RefreshCache(Now()));

        ui64 id;
        {
            auto existing = configGenerator.GetCategorizations({requestId}, USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["tag_id"].GetString(), requestId);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["items"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["items"].GetArray()[0]["category"]["id"].GetString(), "category 1" );
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["items"].GetArray()[0]["comment"].GetString(), "comment 1");

            UNIT_ASSERT(existing["categorizations"].GetArray()[0]["items"].GetArray()[0]["id"].IsUInteger());
            id = existing["categorizations"].GetArray()[0]["items"].GetArray()[0]["id"].GetUInteger();
        }

        UNIT_ASSERT(configGenerator.RemoveCategorization(requestId, id, USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->RefreshCache(Now()));

        {
            auto existing = configGenerator.GetCategorizations({requestId}, USER_ROOT_DEFAULT);
            ERROR_LOG << existing.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray().size(), 0);
        }
    }

    Y_UNIT_TEST(CategorizerTreeAddNode) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        NJson::TJsonValue treeBefore = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        {
            TSupportCategorizerTreeNode node;
            node.SetLabel("test label 123");
            node.SetOrder(Max<ui32>());
            UNIT_ASSERT(configGenerator.AddCategorizerNode(node, "", USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
        }
        NJson::TJsonValue treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), treeBefore["children"].GetArray().size() + 1);
    }

    Y_UNIT_TEST(CategorizerTreeEditNode) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        NJson::TJsonValue treeBefore = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        {
            TSupportCategorizerTreeNode node;
            node.SetLabel("test label 123");
            node.SetOrder(Max<ui32>());
            UNIT_ASSERT(configGenerator.AddCategorizerNode(node, "", USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
        }
        NJson::TJsonValue treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        {
            auto label = treeAfter["children"].GetArray()[treeAfter["children"].GetArray().size() - 1]["meta"]["label"].GetString();
            UNIT_ASSERT_VALUES_EQUAL(label, "test label 123");
        }
        UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), treeBefore["children"].GetArray().size() + 1);
        auto nodeId = treeAfter["children"].GetArray()[treeAfter["children"].GetArray().size() - 1]["id"].GetString();
        {
            TSupportCategorizerTreeNode node;
            node.SetLabel("test label 234");
            node.SetOrder(Max<ui32>());
            UNIT_ASSERT(configGenerator.EditCategorizerNode(nodeId, node, USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
        }
        treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), treeBefore["children"].GetArray().size() + 1);
        {
            auto label = treeAfter["children"].GetArray()[treeAfter["children"].GetArray().size() - 1]["meta"]["label"].GetString();
            UNIT_ASSERT_VALUES_EQUAL(label, "test label 234");
        }
    }

    Y_UNIT_TEST(CategorizerTreeCycle) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        NJson::TJsonValue treeBefore = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        {
            TSupportCategorizerTreeNode node;
            node.SetLabel("test_label_123");
            node.SetOrder(Max<ui32>());
            UNIT_ASSERT(configGenerator.AddCategorizerNode(node, "", USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
        }
        NJson::TJsonValue treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), treeBefore["children"].GetArray().size() + 1);
        auto parentId = treeAfter["children"].GetArray()[treeAfter["children"].GetArray().size() - 1]["id"].GetString();
        {
            TSupportCategorizerTreeNode node;
            node.SetLabel("test_label_234");
            node.SetOrder(Max<ui32>());
            UNIT_ASSERT(configGenerator.AddCategorizerNode(node, parentId, USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
        }
        treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), treeBefore["children"].GetArray().size() + 1);
        auto deepParentId = treeAfter["children"].GetArray()[treeAfter["children"].GetArray().size() - 1]["children"][0]["id"].GetString();
        {
            TSupportCategorizerTreeNode node;
            node.SetLabel("test_label_345");
            node.SetOrder(Max<ui32>());
            UNIT_ASSERT(configGenerator.AddCategorizerNode(node, deepParentId, USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
        }
        auto smallerTree = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT, "test_label_345");
        UNIT_ASSERT_VALUES_EQUAL(smallerTree["children"].GetArray().size(), 1);
        TString treeTop = smallerTree["children"][0]["id"].GetString();
        TString treeBottom = smallerTree["children"][0]["children"][0]["children"][0]["id"].GetString();
        UNIT_ASSERT_C(!!treeTop && !!treeBottom, TStringBuilder() << treeTop << " ttt " << treeBottom);
        UNIT_ASSERT(!configGenerator.AddCategorizerEdge(treeBottom, treeTop, USER_ROOT_DEFAULT));
        UNIT_ASSERT(configGenerator.AddCategorizerEdge(treeTop, treeBottom, USER_ROOT_DEFAULT));
    }

    Y_UNIT_TEST(CategorizerRemoveSubtree) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        NJson::TJsonValue treeBefore = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        size_t targetSize = treeBefore["children"].GetArray().size();
        for (auto&& ch : treeBefore["children"].GetArray()) {
            auto id = ch["id"].GetString();
            UNIT_ASSERT(!!id);
            if (!configGenerator.RemoveCategorizerNode(id, USER_ROOT_DEFAULT)) {
                for (auto&& deepCh : ch["children"].GetArray()) {
                    auto chId = deepCh["id"].GetString();
                    UNIT_ASSERT(configGenerator.RemoveCategorizerNode(chId, USER_ROOT_DEFAULT));
                    UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
                }
                UNIT_ASSERT(configGenerator.RemoveCategorizerNode(id, USER_ROOT_DEFAULT));
            }
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
            NJson::TJsonValue treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), (--targetSize));
        }
    }

    Y_UNIT_TEST(CategorizerMoveGroup) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        NJson::TJsonValue treeBefore = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        TString newParentId;
        {
            TSupportCategorizerTreeNode node;
            node.SetLabel("test label 123");
            node.SetOrder(Max<ui32>());
            UNIT_ASSERT(configGenerator.AddCategorizerNode(node, "", USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
            NJson::TJsonValue treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
            newParentId = treeAfter["children"].GetArray()[treeAfter["children"].GetArray().size() - 1]["id"].GetString();
        }
        size_t targetChildrenSize = 0;
        for (auto&& ch : treeBefore["children"].GetArray()) {
            auto oldParentId = ch["id"].GetString();
            for (auto&& deepCh : ch["children"].GetArray()) {
                auto childId = deepCh["id"].GetString();
                UNIT_ASSERT(configGenerator.MoveCategorizerNode(childId, oldParentId, newParentId, USER_ROOT_DEFAULT));
                UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
                NJson::TJsonValue treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
                UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray()[treeAfter["children"].GetArray().size() - 1]["children"].GetArray().size(), ++targetChildrenSize);
            }
        }
        treeBefore = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        size_t targetRootSize = treeBefore["children"].GetArray().size();
        for (size_t i = 0; i < treeBefore["children"].GetArray().size() - 1; ++i) {
            auto childId = treeBefore["children"].GetArray()[i]["id"].GetString();
            UNIT_ASSERT(configGenerator.MoveCategorizerNode(childId, "", newParentId, USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
            NJson::TJsonValue treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), --targetRootSize);
        }
        treeBefore = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
        UNIT_ASSERT_VALUES_EQUAL(treeBefore["children"].GetArray().size(), 1);
        {
            auto rootChildId = treeBefore["children"].GetArray()[0]["children"].GetArray()[0]["id"].GetString();
            UNIT_ASSERT(configGenerator.MoveCategorizerNode(rootChildId, treeBefore["children"][0]["id"].GetString(), "", USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->GetTreeManager()->RefreshAll(Now()));
            auto treeAfter = configGenerator.GetCategorizerTree(USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(treeAfter["children"].GetArray().size(), 2);
        }
    }

    Y_UNIT_TEST(AudioteleHookDriveUser) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(server->GetSettings().SetValue("support.cc.call_tags.use_tag_name_by_cc_fallback", ::ToString(true), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        TString phoneNumber = "+";
        for (size_t i = 0; i < 11; ++i) {
            phoneNumber += ('0' + rand() % 10);
        }

        auto userId = eGenerator.CreateUser("skulik-was-there222", true, "active");
        {
            auto userData = server->GetDriveAPI()->GetUsersData()->GetCachedObject(userId);
            UNIT_ASSERT(userData.Defined());
            auto data = *userData;
            data.SetPhone(phoneNumber);
            auto session = server->GetDriveAPI()->GetUsersData()->BuildSession(false);
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->UpdateUser(data, "robot-frontend", session) && session.Commit());
        }

        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"cc_audiotele_incoming"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }

        {
            UNIT_ASSERT(configGenerator.AudioteleCallEvent(phoneNumber, ISupportTelephonyIncomingEvent::EType::Start, ""));
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"cc_audiotele_incoming"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }

        {
            UNIT_ASSERT(configGenerator.AudioteleCallEvent(phoneNumber, ISupportTelephonyIncomingEvent::EType::SetPerformer, USER_ROOT_DEFAULT));
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"cc_audiotele_incoming"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> tags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestorePerformerTags({USER_ROOT_DEFAULT}, tags, session));
            bool tagFound = false;
            for (auto&& tag : tags) {
                if (tag->GetName() == "cc_audiotele_incoming" && tag.GetObjectId() == userId) {
                    tagFound = true;
                    break;
                }
            }
            UNIT_ASSERT(tagFound);
        }

        {
            UNIT_ASSERT(configGenerator.AudioteleCallEvent(phoneNumber, ISupportTelephonyIncomingEvent::EType::Finish));
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"cc_audiotele_incoming"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }
    }

    Y_UNIT_TEST(AudioteleHookExternalUser) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(server->GetSettings().SetValue("support.cc.call_tags.use_tag_name_by_cc_fallback", ::ToString(true), USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        TString phoneNumber;
        {
            TSet<TString> usedPhoneNumbers;
            auto users = server->GetDriveAPI()->GetUsersData()->FetchInfo();
            for (auto&& [id, user] : users) {
                usedPhoneNumbers.insert(user.GetPhone());
            }
            do {
                phoneNumber = "+";
                for (size_t i = 0; i < 11; ++i) {
                    phoneNumber += ('0' + rand() % 10);
                }
            } while (usedPhoneNumbers.contains(phoneNumber));
        }

        size_t usersBeforeCnt = server->GetDriveAPI()->GetUsersData()->FetchInfo().size();

        TString userId = "";
        {
            UNIT_ASSERT(configGenerator.AudioteleCallEvent(phoneNumber, ISupportTelephonyIncomingEvent::EType::Start));
            auto users = server->GetDriveAPI()->GetUsersData()->FetchInfo();
            for (auto&& [id, user] : users) {
                if (user.GetPhone() == phoneNumber) {
                    userId = user.GetUserId();
                    break;
                }
            }
            UNIT_ASSERT(!!userId);

            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"cc_audiotele_incoming"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }

        size_t usersAfterCnt = server->GetDriveAPI()->GetUsersData()->FetchInfo().size();
        UNIT_ASSERT_VALUES_EQUAL(usersAfterCnt, usersBeforeCnt + 1);

        {
            UNIT_ASSERT(configGenerator.AudioteleCallEvent(phoneNumber, ISupportTelephonyIncomingEvent::EType::Finish));
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"cc_audiotele_incoming"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 0);
        }

        {
            UNIT_ASSERT(configGenerator.AudioteleCallEvent(phoneNumber, ISupportTelephonyIncomingEvent::EType::Start));
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"cc_audiotele_incoming"}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }

        size_t usersFinalCnt = server->GetDriveAPI()->GetUsersData()->FetchInfo().size();
        UNIT_ASSERT_VALUES_EQUAL(usersFinalCnt, usersAfterCnt);
    }

    Y_UNIT_TEST(OutgoingCommunicationSimple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there222", true, "active");

        // Clear feed
        TVector<TString> tagNames;
        {
            auto td = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TSupportOutgoingCommunicationTag::TypeName);
            for (auto&& i : td) {
                tagNames.emplace_back(i->GetName());
            }
        }
        {
            auto td = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TFleetSupportOutgoingCommunicationTag::TypeName);
            for (auto&& i : td) {
                tagNames.emplace_back(i->GetName());
            }
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, tagNames, dbTags, session));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(dbTags, USER_ROOT_DEFAULT, server.Get(), session, true) && session.Commit());
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            auto feed = configGenerator.GetFullFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feed["outgoing"].GetArray().size(), 0);
        }

        {
            THolder<TSupportOutgoingCommunicationTag> tag = MakeHolder<TSupportOutgoingCommunicationTag>(TSupportOutgoingCommunicationTag::TypeName);
            tag->SetComment("hello");
            UNIT_ASSERT(configGenerator.AddTag(tag.Release(), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({userId}, {TSupportOutgoingCommunicationTag::TypeName}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags[0];

            auto chatMessages = configGenerator.GetChatMessages(userId, "outgoing_communication." + tag.GetTagId(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatMessages["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(chatMessages["messages"].GetArray()[4]["text"].GetString(), "hello");
        }

        {
            auto feed = configGenerator.GetFullFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feed["outgoing"].GetArray().size(), 1);
        }
    }

   Y_UNIT_TEST(FilteredUserCountRolesAndTags) {
        // roles: priority_simple simple standart_access default_user view_permissions_all scanner
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId1 = eGenerator.CreateUser("user_tagrolesfilter_1", false, "active");
        auto userId2 = eGenerator.CreateUser("user_tagrolesfilter_2", false, "active");
        auto userId3 = eGenerator.CreateUser("user_tagrolesfilter_3", false, "active");
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();

            TDriveRoleHeader roleHeader1("role_tagrolesfilter_1");
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesDB().Upsert(roleHeader1, userId1, session));
            TDriveRoleHeader roleHeader2("role_tagrolesfilter_2");
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesDB().Upsert(roleHeader2, userId1, session));
            TDriveRoleHeader roleHeader3("role_tagrolesfilter_3");
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesDB().Upsert(roleHeader3, userId1, session));

            TUserRole userRole1;
            userRole1.SetRoleId("role_tagrolesfilter_1").SetUserId(userId1).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole1, userId1, session));

            TUserRole userRole2;
            userRole2.SetRoleId("role_tagrolesfilter_2").SetUserId(userId2).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole2, userId2, session));

            TUserRole userRole3;
            userRole3.SetRoleId("role_tagrolesfilter_3").SetUserId(userId3).SetActive(false);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole3, userId3, session));

            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }
        { // roles
            {
                NJson::TJsonValue request;
                NJson::TJsonValue roles = NJson::JSON_ARRAY;
                roles.AppendValue("incorrect_role");
                roles.AppendValue("role_tagrolesfilter_1");
                roles.AppendValue("role_tagrolesfilter_2");
                roles.AppendValue("role_tagrolesfilter_3");
                request["roles"]["list"] = std::move(roles);
                request["roles"]["filter_mode"] = {"one_of"};
                request["roles"]["check_active"] = {true};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 2, result["users_count"].GetInteger());
                }
                request["roles"]["check_active"] = {false};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 3, result["users_count"].GetInteger());
                }
                request["roles"]["filter_mode"] = {"all_of"};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 0, result["users_count"].GetInteger());
                }
                {
                    auto session = driveApi.BuildTx<NSQL::Writable>();
                    TUserRole userRole4;
                    userRole4.SetRoleId("role_tagrolesfilter_1").SetUserId(userId2).SetActive(false);
                    UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole4, userId2, session));
                    TUserRole userRole5;
                    userRole5.SetRoleId("role_tagrolesfilter_2").SetUserId(userId1).SetActive(true);
                    UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole5, userId1, session));
                    UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
                }
            }
            {
                NJson::TJsonValue request;
                NJson::TJsonValue roles = NJson::JSON_ARRAY;
                roles.AppendValue("role_tagrolesfilter_1");
                roles.AppendValue("role_tagrolesfilter_2");
                request["roles"]["list"] = std::move(roles);
                request["roles"]["filter_mode"] = {"all_of"};
                request["roles"]["check_active"] = {false};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 2, result["users_count"].GetInteger());
                }
                request["roles"]["check_active"] = {true};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 1, result["users_count"].GetInteger());
                }
            }
        }
        { // assigned tags
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                TSupportChatTag::TDescription description1;
                description1.SetName("tag_tagrolesfilter_1").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
                UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description1), USER_ROOT_DEFAULT, session));
                TSupportChatTag::TDescription description2;
                description2.SetName("tag_tagrolesfilter_2").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
                UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description2), USER_ROOT_DEFAULT, session));
                TSupportChatTag::TDescription description3;
                description3.SetName("tag_tagrolesfilter_3").SetType(TSupportChatTag::TypeName).SetDefaultPriority(0);
                UNIT_ASSERT(driveApi.GetTagsManager().GetTagsMeta().RegisterTag(new TSupportChatTag::TDescription(description3), USER_ROOT_DEFAULT, session));
                UNIT_ASSERT_C(session.Commit(), session.GetStringReport());
                SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            }
            configGenerator.AddTag(new TSupportChatTag("tag_tagrolesfilter_1"), userId1, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            configGenerator.AddTag(new TSupportChatTag("tag_tagrolesfilter_2"), userId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            configGenerator.AddTag(new TSupportChatTag("tag_tagrolesfilter_3"), userId3, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            {
                NJson::TJsonValue request;
                NJson::TJsonValue tags = NJson::JSON_ARRAY;
                tags.AppendValue("incorrect_tag");
                tags.AppendValue("tag_tagrolesfilter_1");
                tags.AppendValue("tag_tagrolesfilter_2");
                tags.AppendValue("tag_tagrolesfilter_3");
                request["assigned_tags"]["list"] = std::move(tags);
                request["assigned_tags"]["filter_mode"] = {"one_of"};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 3, result["users_count"].GetInteger());
                }
                request["assigned_tags"]["filter_mode"] = {"all_of"};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 0, result["users_count"].GetInteger());
                }
            }
            configGenerator.AddTag(new TSupportChatTag("tag_tagrolesfilter_1"), userId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            configGenerator.AddTag(new TSupportChatTag("tag_tagrolesfilter_2"), userId1, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            {
                NJson::TJsonValue request;
                NJson::TJsonValue tags = NJson::JSON_ARRAY;
                tags.AppendValue("tag_tagrolesfilter_1");
                tags.AppendValue("tag_tagrolesfilter_2");
                request["assigned_tags"]["list"] = std::move(tags);
                request["assigned_tags"]["filter_mode"] = {"all_of"};
                {
                    NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                    UNIT_ASSERT(result.Has("users_count"));
                    UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 2, result["users_count"].GetInteger());
                }
            }
        }
        { //performing
            {
                auto session = driveApi.BuildTx<NSQL::Writable>();
                TUserRole userRole1;
                userRole1.SetRoleId("support_assignee").SetUserId(userId1).SetActive(true);
                UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole1, userId1, session));
                TUserRole userRole2;
                userRole2.SetRoleId("support_assignee").SetUserId(userId3).SetActive(true);
                UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole2, userId3, session));
                UNIT_ASSERT(session.Commit());
            }
            TString tagId3;
            configGenerator.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            UNIT_ASSERT(configGenerator.GetTagId(userId2, "user_problem_tag_minor", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId3));
            TString tagId4;
            configGenerator.AddTag(new TUserProblemTag("user_problem_tag_major"), userId3, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User);
            UNIT_ASSERT(configGenerator.GetTagId(userId3, "user_problem_tag_major", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId4));
            UNIT_ASSERT(configGenerator.StartTag(tagId3, userId1));
            UNIT_ASSERT(configGenerator.StartTag(tagId4, userId3));
            int twoPerformingResult;
            {
                NJson::TJsonValue request;
                NJson::TJsonValue tags = NJson::JSON_ARRAY;
                tags.AppendValue("user_problem_tag_major");
                tags.AppendValue("user_problem_tag_minor");
                request["performing_tags"]["list"] = std::move(tags);
                NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                UNIT_ASSERT(result.Has("users_count"));
                UNIT_ASSERT_C(result["users_count"].GetInteger() >= 2, result["users_count"].GetInteger());
                twoPerformingResult = result["users_count"].GetInteger();
            }
            {
                NJson::TJsonValue request;
                NJson::TJsonValue tags = NJson::JSON_ARRAY;
                tags.AppendValue("user_problem_tag_minor");
                request["performing_tags"]["list"] = std::move(tags);
                NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
                UNIT_ASSERT(result.Has("users_count"));
                UNIT_ASSERT_C(result["users_count"].GetInteger() >= 1 && result["users_count"].GetInteger() < twoPerformingResult, result["users_count"].GetInteger());
                twoPerformingResult = result["users_count"].GetInteger();
            }
        }
        { // roles and assigned
            NJson::TJsonValue request;
            NJson::TJsonValue roles = NJson::JSON_ARRAY;
            roles.AppendValue("role_tagrolesfilter_1");
            roles.AppendValue("role_tagrolesfilter_2");
            roles.AppendValue("role_tagrolesfilter_3");
            request["roles"]["list"] = std::move(roles);
            request["roles"]["filter_mode"] = {"one_of"};
            request["roles"]["check_active"] = {false};
            NJson::TJsonValue tags = NJson::JSON_ARRAY;
            tags.AppendValue("tag_tagrolesfilter_1");
            request["assigned_tags"]["list"] = std::move(tags);
            request["assigned_tags"]["filter_mode"] = {"one_of"};
            NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
            UNIT_ASSERT(result.Has("users_count"));
            UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 2, result["users_count"].GetInteger());
        }
        { // roles and performing
            NJson::TJsonValue request;
            NJson::TJsonValue roles = NJson::JSON_ARRAY;
            roles.AppendValue("role_tagrolesfilter_1");
            roles.AppendValue("role_tagrolesfilter_2");
            roles.AppendValue("role_tagrolesfilter_3");
            request["roles"]["list"] = std::move(roles);
            request["roles"]["filter_mode"] = {"one_of"};
            request["roles"]["check_active"] = {false};
            NJson::TJsonValue tags = NJson::JSON_ARRAY;
            tags.AppendValue("user_problem_tag_minor");
            tags.AppendValue("user_problem_tag_major");
            request["performing_tags"]["list"] = std::move(tags);
            NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
            UNIT_ASSERT(result.Has("users_count"));
            UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 2, result["users_count"].GetInteger());
        }
        { // assigned and performing
            NJson::TJsonValue request;
            NJson::TJsonValue tagsA = NJson::JSON_ARRAY;
            tagsA.AppendValue("tag_tagrolesfilter_1");
            tagsA.AppendValue("tag_tagrolesfilter_2");
            request["assigned_tags"]["list"] = std::move(tagsA);
            request["assigned_tags"]["filter_mode"] = {"one_of"};
            NJson::TJsonValue tags = NJson::JSON_ARRAY;
            tags.AppendValue("user_problem_tag_minor");
            tags.AppendValue("user_problem_tag_major");
            request["performing_tags"]["list"] = std::move(tags);
            NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
            UNIT_ASSERT(result.Has("users_count"));
            UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 1, result["users_count"].GetInteger());
        }
        { // roles, assigned and performing
            NJson::TJsonValue request;
            NJson::TJsonValue roles = NJson::JSON_ARRAY;
            roles.AppendValue("role_tagrolesfilter_1");
            roles.AppendValue("role_tagrolesfilter_2");
            roles.AppendValue("role_tagrolesfilter_3");
            request["roles"]["list"] = std::move(roles);
            request["roles"]["filter_mode"] = {"one_of"};
            request["roles"]["check_active"] = {false};
            NJson::TJsonValue tags = NJson::JSON_ARRAY;
            tags.AppendValue("tag_tagrolesfilter_2");
            request["assigned_tags"]["list"] = std::move(tags);
            request["assigned_tags"]["filter_mode"] = {"one_of"};
            NJson::TJsonValue tagsP = NJson::JSON_ARRAY;
            tagsP.AppendValue("user_problem_tag_major");
            tagsP.AppendValue("user_problem_tag_minor");
            request["performing_tags"]["list"] = std::move(tagsP);
            NJson::TJsonValue result = configGenerator.FilteredUserCount(request, USER_ID_DEFAULT);
            UNIT_ASSERT(result.Has("users_count"));
            UNIT_ASSERT_VALUES_EQUAL_C(result["users_count"].GetInteger(), 1, result["users_count"].GetInteger());
        }
   }

    Y_UNIT_TEST(LastPerformerMessageFilter) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("operator_lastperf_321", true, "active");
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }

        auto clientId = eGenerator.CreateUser("user_lastperf_123", true, "active");
        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.123", "Hello, Yandex.Drive!", {}));
            server->GetChatRobot("support")->Refresh(Now());
            UNIT_ASSERT(server->GetChatEngine()->GetChats().RefreshCache(Now()));
        }
        TChatSupportRequestsFilter requestFilter;
        requestFilter.SetPermissions(server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures()));
        {
            NJson::TJsonValue params = NJson::JSON_MAP;
            NJson::TJsonValue metricParams = NJson::JSON_MAP;
            metricParams["metric_type"] = "since_last_performer_message";
            metricParams["min_value"] = 0;
            metricParams["max_value"] = 100;
            params["chat_robot_id"] = "support";
            metricParams["split_by_operator"] = false;
            params["metric_filter"] = metricParams;
            params["chat_status"] = "any";
            params["perform_status"] = "any";
            params["muted_status"] = "any";
            requestFilter.DeserializeFromJson(params);
        }
        {
            TVector<TFilteredSupportChat> result;
            TMessagesCollector errors;
            UNIT_ASSERT_C(requestFilter.GetSupportRequests(server.Get(), result, TInstant::Now(), USER_ROOT_DEFAULT, errors), errors.GetStringReport());
            UNIT_ASSERT_C(result.size() == 0, result.size());
        }
        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_2_line" });
            UNIT_ASSERT(gServer.SendArbitraryMessage("support.123", userId, "Hello, client!"));
        }
        {
            TString tagId;
            UNIT_ASSERT(gServer.GetTagId(clientId, "support_chat_2_line", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, tagId));
            UNIT_ASSERT(gServer.StartTag(tagId, userId));
            server->GetChatRobot("support")->Refresh(Now());
            UNIT_ASSERT(server->GetChatEngine()->GetChats().RefreshCache(Now()));
        }
        {
            TMessagesCollector errors;
            TVector<TFilteredSupportChat> result;
            UNIT_ASSERT_C(requestFilter.GetSupportRequests(server.Get(), result, TInstant::Now(), USER_ROOT_DEFAULT, errors), errors.GetStringReport());
            UNIT_ASSERT_C(result.size() == 1, result.size());
        }
        {
            TInstantGuard ig(Now() + TDuration::Minutes(5));
            TVector<TFilteredSupportChat> result;
            TMessagesCollector errors;
            UNIT_ASSERT_C(requestFilter.GetSupportRequests(server.Get(), result, TInstant::Now(), USER_ROOT_DEFAULT, errors), errors.GetStringReport());
            UNIT_ASSERT_C(result.size() == 0, result.size());
        }
    }
}
