#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/state/robot_state.pb.h>
#include <drive/backend/data/user_tags.h>


Y_UNIT_TEST_SUITE(SupportChatContext) {

    void SendMessageAndCheckReply(const NDrive::TServerConfigGenerator& gServer, const TString& chatId, const TString& userId, const TString& message, const TString& messageType, const TString& expectedMessage, const TString& expectedType) {
        NNeh::THttpRequest request;
        NJson::TJsonValue post;
        post["message"] = message;
        post["message_type"] = messageType;
        request.SetUri("/api/staff/chat/message").SetCgiData("chat_id=" + chatId + "&user_id=" + userId).SetPostData(TBlob::FromString(post.GetStringRobust())).SetRequestType("POST");
        request.AddHeader("Authorization", userId);
        NUtil::THttpReply reply = gServer.GetSendReply(request);
        NJson::TJsonValue resultReport = NJson::JSON_MAP;
        UNIT_ASSERT_VALUES_EQUAL(reply.Code(), 200);
        UNIT_ASSERT_C(NJson::ReadJsonFastTree(reply.Content(), &resultReport), reply.Content());
        UNIT_ASSERT_VALUES_EQUAL(resultReport["type"].GetString(), expectedType);
        UNIT_ASSERT_VALUES_EQUAL(resultReport["text"].GetString(), expectedMessage);
    }

    Y_UNIT_TEST(ClassificationScriptParsing) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());

        NJson::TJsonValue json;
        TFileInput in(JoinFsPaths(ArcadiaSourceRoot(), "drive/backend/chat_robots/test_data/script_context_map.json"));
        ReadJsonTree(&in, &json);

        TChatRobotScript s;
        TMessagesCollector errors;
        UNIT_ASSERT_C(s.Parse(json, errors), TStringBuilder() << errors.GetStringReport());
    }

    Y_UNIT_TEST(WriteToNode) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-not-there", false, "active");

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!node_1", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!node_3", {}));
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.11111");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["text"].GetString(), "Node 4");
        }
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!back", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!node_2", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!node_3", {}));
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.11111");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 14);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[13]["text"].GetString(), "Visited both");
        }
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!ok", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!node_1", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.11111", "!node_3", {}));
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.11111");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 22);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[21]["text"].GetString(), "Visited both");
        }
    }

    Y_UNIT_TEST(PhoneNumber) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-not-there", false, "active");
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.22222", "!number", {}));

        auto message = "{\"phone\":\"+79998885544\",\"login\":\"qwerty\",\"user_id\":\"123d1200-55f5-406b-5555-aaaaaf0153c8\",\"name\":\"default_user\"}";
        SendMessageAndCheckReply(gServer, "support_context_map.22222", userId, message, "phone_number", "+79998885544, default_user, 123d1200-55f5-406b-5555-aaaaaf0153c8",  "phone_number");
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.22222");
            UNIT_ASSERT(messagesJson["messages"].GetArray().size() > 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"].GetString(), "+79998885544, default_user, 123d1200-55f5-406b-5555-aaaaaf0153c8");
        }
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.22222", "!node_3", {}));
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.22222");
            ERROR_LOG << "Context debug " << messagesJson["messages"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[8]["text"].GetString(), "You've entered phone");
        }
    }

    Y_UNIT_TEST(EndpointCall) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        UNIT_ASSERT(gServer.ModifyLanding("context_map_text_landing", "{'text':'landing1'}", true, USER_ROOT_DEFAULT, {}));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");
        auto userId = eGenerator.CreateUser("skulik-was-not-there", false, "active");

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.33333", "!endpoint", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.33333", "!next", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.33333", "!node_1", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.33333", "!node_3", {}));
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.33333");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 11);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["text"].GetString(), "context_map_text_landing");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[8]["text"].GetString(), "context_map_text_landing");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[10]["text"].GetString(), "You have landing");
        }
    }

    Y_UNIT_TEST(GiftBonus) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-not-there", false, "active");
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.44444", "!bonuses", {}));

        auto message = "{\"phone\":\"+79998885544\",\"login\":\"qwerty\",\"user_id\":\"123d1200-55f5-406b-5555-aaaaaf0153c8\",\"name\":\"default_user\"}";
        SendMessageAndCheckReply(gServer, "support_context_map.44444", userId, message, "phone_number", "Дарим бонусы другу default_user с номером +79998885544.",  "phone_number");
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.44444");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[3]["text"].GetString(), "Дарим бонусы другу default_user с номером +79998885544.");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"].GetString(), "Сколько хотите подарить?");
        }

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.44444", "!5000", {}));
        SendMessageAndCheckReply(gServer, "support_context_map.44444", userId, "card-x12345", "CreditCardBinding", "510000****0257",  "CreditCardBinding");
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.44444");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[5]["text"].GetString(), "5000 ₽");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[7]["text"].GetString(), "510000****0257");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[8]["text"].GetString(), "Отлично. Списываем деньги с этой карты?");
        }
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.44444", "!add_tag", {}));
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, { "simple_user_tag" }, dbTags, session));
                UNIT_ASSERT_EQUAL(dbTags.size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetComment(), "123d1200-55f5-406b-5555-aaaaaf0153c8 card-x12345 500000");
            }
        }
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.44444", "!ok", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_context_map.44444", "!3000", {}));
        SendMessageAndCheckReply(gServer, "support_context_map.44444", userId, "card-x321", "CreditCardBinding", "",  "CreditCardBinding");
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_context_map.44444");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 18);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[14]["text"].GetString(), "3000 ₽");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[16]["text"].GetString(), "");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[17]["text"].GetString(), "Хм, не получится использовать эту карту. Давайте лучше другую.");
        }
    }
}
